/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

// Need to display debug lines.
#include <Common/Visualize/hkDebugDisplay.h>
#include <Physics2012/Utilities/Actions/AngularDashpot/hkpAngularDashpotAction.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>

#include <Demos/DemoCommon/DemoFramework/hkDefaultPhysics2012Demo.h>

class AngularDashpotActionDemo : public hkDefaultPhysics2012Demo
{
public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
	AngularDashpotActionDemo(hkDemoEnvironment* env);
	~AngularDashpotActionDemo();

	// Step the demo
	virtual Result stepDemo();

private:	
	hkpRigidBody* m_boxRigidBodyTrunk;
	hkpRigidBody* m_boxRigidBodyBranch1;
	hkpRigidBody* m_boxRigidBodyBranch2;

	hkpAngularDashpotAction* m_springAction;
};

AngularDashpotActionDemo::AngularDashpotActionDemo(hkDemoEnvironment* env)
	: hkDefaultPhysics2012Demo(env)
{
	// Define gravity as zero for this demo.
	hkVector4 gravity( 0.0f, -9.8f, 0.0f );

	//
	// Set up the camera.
	//
	// We are using the simple demo framework so we first we set up the camera 
	// with the camera looking approximately at the origin.
	{ 
		hkVector4 from(0.0f, 5.0f, 20.0f);
		hkVector4 to  (0.0f, 0.0f, 0.0f);
		hkVector4 up  (0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}
	
	//
	// Create the world.
	//
	// We create our world with broadphase extents set to +/- 100 units and specify
	// hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM which is a reasonably all purpose solver type.
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.m_gravity = gravity;	
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_collisionFilter.setAndDontIncrementRefCount(new hkpGroupFilter);
		m_world = new hkpWorld( info );
		m_world->lock();
		setupGraphics();
	}

	// Register all collision agents
	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

	//
	// Create Trunk Rigid Body.
	//
	{
		/// Here we construct a simple cube with sides 2 units and mass 1.
		hkVector4 boxHalfExtents( 1.0f, 3.0f, 1.0f );  
		hkpBoxShape* geom = new hkpBoxShape( boxHalfExtents , 0 );

		const hkReal mass = 1.0f;
		hkVector4 pos( -0.0f, 0.0f, 0.0f );

		hkpRigidBodyCinfo boxInfo;
		hkMassProperties massProperties;
		hkpInertiaTensorComputer::computeBoxVolumeMassProperties(boxHalfExtents, mass, massProperties);
		boxInfo.m_mass = massProperties.m_mass;
		boxInfo.m_centerOfMass = massProperties.m_centerOfMass;
		boxInfo.m_inertiaTensor = massProperties.m_inertiaTensor;
		boxInfo.m_shape = geom;
		boxInfo.m_motionType = hkpMotion::MOTION_FIXED;
		boxInfo.m_position = pos;
		boxInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(0, 1, 1, 1);

		m_boxRigidBodyTrunk = new hkpRigidBody(boxInfo);

		// As the rigid bodies now references our shape, we no longer need to.
		geom->removeReference();

		// and add it to the world.
		m_world->addEntity( m_boxRigidBodyTrunk );
	}

	//
	// Create Branch Piece 1 Rigid Body.
	//
	{
		// Here we construct a simple cube with sides 2 units and mass 1.
		hkVector4 boxHalfExtents( 0.2f, 2.0f, 0.2f );  
		hkpBoxShape* geom = new hkpBoxShape( boxHalfExtents , 0 );

		hkVector4 pos( 0.0f, 5.0f, 0.0f );

		hkpRigidBodyCinfo boxInfo;
		hkMassProperties massProperties;
		hkpInertiaTensorComputer::computeBoxVolumeMassProperties(boxHalfExtents, 1.0f, massProperties);
		boxInfo.m_mass = massProperties.m_mass;
		boxInfo.m_centerOfMass = massProperties.m_centerOfMass;
		boxInfo.m_inertiaTensor.setIdentity(); //= massProperties.m_inertiaTensor;
		boxInfo.m_shape = geom;
		boxInfo.m_motionType = hkpMotion::MOTION_THIN_BOX_INERTIA;
		boxInfo.m_position = pos;
		boxInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(0, 1, 1, 1);

		m_boxRigidBodyBranch1 = new hkpRigidBody(boxInfo);

		// As the rigid bodies now references our shape, we no longer need to.
		geom->removeReference();

		// and add it to the world. 
		m_world->addEntity( m_boxRigidBodyBranch1 );
	}
	
	//
	// Create Branch Piece 2 Rigid Body.
	//
	{
		// Here we construct a simple cube with sides 2 units and mass 1.
		hkVector4 boxHalfExtents( 0.15f, 1.5f, 0.15f );  
		hkpBoxShape* geom = new hkpBoxShape( boxHalfExtents , 0 );

		hkVector4 pos( 0.0f, 8.5f, 0.0f );

		hkpRigidBodyCinfo boxInfo;
		hkMassProperties massProperties;
		hkpInertiaTensorComputer::computeBoxVolumeMassProperties(boxHalfExtents, 0.5f, massProperties);
		boxInfo.m_mass = massProperties.m_mass;
		boxInfo.m_centerOfMass = massProperties.m_centerOfMass;
		boxInfo.m_inertiaTensor.setIdentity(); //= massProperties.m_inertiaTensor;
		boxInfo.m_shape = geom;
		boxInfo.m_motionType = hkpMotion::MOTION_THIN_BOX_INERTIA;
		boxInfo.m_position = pos;
		boxInfo.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(0, 1, 1, 1);

		m_boxRigidBodyBranch2 = new hkpRigidBody(boxInfo);

		// As the rigid bodies now references our shape, we no longer need to.
		geom->removeReference();

		// and add it to the world.
		m_world->addEntity( m_boxRigidBodyBranch2 );
	}

	// Create ball and socket constraints.
	{
		hkpBallAndSocketConstraintData* bs = new hkpBallAndSocketConstraintData;
		bs->setInWorldSpace(m_boxRigidBodyTrunk->getTransform(), m_boxRigidBodyBranch1->getTransform(), hkVector4(0.0f, 3.0f, 0.0f));
		m_world->createAndAddConstraintInstance(m_boxRigidBodyTrunk, m_boxRigidBodyBranch1, bs)->removeReference();
		bs->removeReference();
	}
	{
		hkpBallAndSocketConstraintData* bs = new hkpBallAndSocketConstraintData;
		bs->setInWorldSpace(m_boxRigidBodyBranch1->getTransform(), m_boxRigidBodyBranch2->getTransform(), hkVector4(0.0f, 7.0f, 0.0f));
		m_world->createAndAddConstraintInstance(m_boxRigidBodyBranch1, m_boxRigidBodyBranch2, bs)->removeReference();
		bs->removeReference();
	}

	// Add dashpot actions
	{
		hkpAngularDashpotAction* action = new hkpAngularDashpotAction(m_boxRigidBodyTrunk, m_boxRigidBodyBranch1);
		action->setStrength(3.5f);
		action->setDamping(0.25f);
		hkQuaternion quat;
		quat.setAxisAngle(hkVector4(0.0f, 0.0f, 1.0f), HK_REAL_PI/8);
		action->setRotation(quat);
		m_world->addAction(action);
		action->removeReference();
	}
	{
		hkpAngularDashpotAction* action = new hkpAngularDashpotAction(m_boxRigidBodyBranch1, m_boxRigidBodyBranch2);
		action->setStrength(1.0f);
		action->setDamping(0.10f);
		hkQuaternion quat;
		quat.setAxisAngle(hkVector4(0.0f, 0.0f, 1.0f), HK_REAL_PI/6);
		action->setRotation(quat);
		m_world->addAction(action);
		action->removeReference();
	}

	m_world->unlock();
}

AngularDashpotActionDemo::~AngularDashpotActionDemo()
{
	m_world->lock();
	m_boxRigidBodyTrunk->removeReference();
	m_boxRigidBodyBranch1->removeReference();
	m_boxRigidBodyBranch2->removeReference();
	m_world->unlock();
}

hkDemo::Result AngularDashpotActionDemo::stepDemo()
{
	return hkDefaultPhysics2012Demo::stepDemo();
}



static const char helpString[] =					\
"A demo which shows creation and addition of an "	\
"hkpAngularDashpotAction to some rigid bodies, both "	\
"of which are boxes.";

HK_DECLARE_DEMO(AngularDashpotActionDemo, HK_DEMO_TYPE_PHYSICS_2012, "Demonstrates an hkpAngularDashpotAction on rigid bodies", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
