/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Common/Base/DebugUtil/MemoryExceptionTestingUtil/hkMemoryExceptionTestingUtil.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Math/Quaternion/hkQuaternionUtil.h>

#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTransform/hkpConvexTransformShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>

#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics/Constraint/Data/Hinge/hkpHingeConstraintData.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics/Constraint/Data/Prismatic/hkpPrismaticConstraintData.h>
#include <Physics/Constraint/Motor/hkpConstraintMotor.h>
#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>

#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpRigidBodyInertiaViewer.h>
#include <Physics2012/Utilities/Weapons/hkpBallGun.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayObject/hkgInstancedDisplayObject.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>

#include <Demos/Physics2012/Api/Constraints/Solver/SolverWorld.h>
#include <Demos/Physics2012/Api/Constraints/Solver/SolverDemo.h>

//
//	Global constants

#define MAX_RANDOM_TRANSLATION			100.0f

//
//	Body types

 #define USE_BOX_BODIES
// #define USE_SPHERE_BODIES

//
//	Constructor

SolverDemo::SolverDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
,	m_rng(11)
,	m_showTexts(false)
,	m_stableSolverOn(true)
{
	// Setup the camera
	{
		hkVector4 from(0.0f, 5.0f, 70.0f);
		hkVector4 to  (0.0f,  0.0f,  0.0f);
		hkVector4 up  (0.0f,  1.0f,  0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	// Create the world
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.m_solverIterations = 4;
		info.m_solverMicrosteps = 1;
		info.m_solverTau = 0.6f;
		info.setBroadPhaseWorldSize( 1000.0f );
		info.m_enableDeactivation = false;
		info.m_gravity.set(0.0f, -50.0f, 0.0f);
		m_world = new hkpWorld( info );
		m_world->lock();

		// Register the single agent required (a hkpBoxBoxAgent)
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

		// Create constraint viewer
		m_debugViewerNames.pushBack(hkpConstraintViewer::getName());
		hkpConstraintViewer::m_scale = 3.0f;

		// Create inertia tensor viewer
//		m_debugViewerNames.pushBack(hkpRigidBodyInertiaViewer::getName());

		setupGraphics();
	}

	// Create a group filter to avoid intra-collision for the rope
	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		m_world->setCollisionFilter( filter );
		filter->removeReference();
	}

	// Init motor angle & position
	{
		m_motorAngleVel = 1.0f;
		m_currentMotorAngle = 0.0f;

		m_currentMotorPos = 0.0f;
		m_motorPosVel = 1.0f;
	}

	// Populate demo world
	{
		// Create ground
			{
				hkVector4 vGroundSize(100.0f, 2.0f, 100.0f);
				hkVector4 vGroundPos(0.0f, -40.0f, 0.0f);
				createGround(vGroundSize, vGroundPos);
			}

		for (int k = 0; k < 5; k++)
		{
			hkVector4 vChainStart(10.0f * (hkReal)k, 0.0f, 0.0f);
			hkVector4 vChainDir(0.0f, -1.0f, 0.0f);

			hkQuaternion deltaRot;
			{
				hkVector4 vA(1.0f, 0.0f, 1.0f);
				vA.normalize3();
				deltaRot.setAxisAngle(vA, 0.5f);
			}

			createBallSocketChain(30 + 3 * k/*100*/, 2.0f, vChainStart, vChainDir, deltaRot, 1.0f, 100.0f, 10.0f, 80.0f);
		}
	}

	// Enable stabilized solver and pre-stabilize inertias
	hkpConstraintStabilizationUtil::setConstraintsSolvingMethod(m_world, hkpConstraintAtom::METHOD_STABILIZED);
	hkpConstraintStabilizationUtil::stabilizePhysicsWorldInertias(m_world);

	m_world->unlock();

	// Bind keys
	bindKeyPressed('T', "Toggle solver", KeyPressCallback::BoundMethod(&SolverDemo::keyHandler, this, 'T'));
	bindKeyPressed('C', "Toggle text display", KeyPressCallback::BoundMethod(&SolverDemo::keyHandler, this, 'C'));
	bindKeyPressed('X', "Randomize all", KeyPressCallback::BoundMethod(&SolverDemo::keyHandler, this, 'X'));
}

//
//	Destructor

SolverDemo::~SolverDemo()
{
}

//
//	Creates the ground

void SolverDemo::createGround(const hkVector4& vSizes, const hkVector4& vPos)
{
	// Create shape
	hkpShape* shape = new hkpBoxShape(vSizes);

	// Create rigid body
	hkpRigidBodyCinfo info;
	info.m_shape			= shape;
	info.m_position			= vPos;
	info.m_rotation			= hkQuaternion::getIdentity();
	info.m_angularDamping	= 0.1f;
	info.m_linearDamping	= 0.1f;
	hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 10.0f, info);
	info.m_motionType		= hkpMotion::MOTION_FIXED;	
	hkpRigidBody* body = new hkpRigidBody(info);

	// Add to world
	m_world->addEntity(body);
	body->removeReference();
	shape->removeReference();
}

//
//	Creates a chain of rigid bodies

void SolverDemo::createBodiesChain(	const int numLinks, const hkReal jointSize,
									const hkVector4& rootPos, const hkVector4& chainDir,
									const hkQuaternion& deltaLinkRot, 
									const hkReal linkMass, const hkReal endEffectorMass, const hkReal materialDensity, hkArray<hkpRigidBody*>& bodiesOut)
{
	hkVector4 vPos = rootPos;

	// COmpute the link and end effector size
	const hkReal cubicRoot	= 1.0f / 3.0f;
	const hkReal lSize		= hkMath::pow(4.0f * linkMass / materialDensity, cubicRoot);
	const hkReal eeSize		= hkMath::pow(4.0f * endEffectorMass / materialDensity, cubicRoot);
	const hkReal lSizeHalf	= lSize * 0.5f;
	const hkReal eeSizeHalf	= eeSize * 0.5f;

	// Create the shape for a link (link will have Z axis oriented along the chain)
	hkVector4 vHalfExtentsL(lSizeHalf, lSizeHalf, lSize);
	hkVector4 vHalfExtentsEe(eeSizeHalf, eeSizeHalf, eeSize);
	vHalfExtentsL.mul4(0.5f);
	vHalfExtentsEe.mul4(0.5f);

	// Create shapes
	hkpShape* bodyShapeL = HK_NULL;
	hkpShape* bodyShapeEe = HK_NULL;
#if defined(USE_BOX_BODIES)
	{
		bodyShapeL = new hkpBoxShape(vHalfExtentsL);
		bodyShapeEe = new hkpBoxShape(vHalfExtentsEe);
	}
#elif defined(USE_SPHERE_BODIES)
	{
		bodyShapeL = new hkpSphereShape(lSizeHalf);
		bodyShapeEe = new hkpSphereShape(eeSizeHalf);
	}
#endif

	// Compute the shape orientation
	hkVector4 vZ(0.0f, 0.0f, 1.0f);
	hkQuaternion qRot;
	hkQuaternionUtil::_computeShortestRotation(vZ, chainDir, qRot);

	hkpRigidBodyCinfo info;
	info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(1, 1);
	info.m_shape = bodyShapeL;

	// Create the bodies
	for (int bi = 0; bi < numLinks; bi++)
	{
		// Create a new body
		info.m_position = vPos;
		info.m_rotation = qRot;
		info.m_angularDamping = 0.1f;
		info.m_linearDamping = 0.1f;

		// Compute the box inertia tensor
		hkReal mass = linkMass;
		if ( numLinks - 1 == bi )
		{
			mass = endEffectorMass;
			info.m_shape = bodyShapeEe;
		}
		hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, mass, info);

		// Set motion type
		info.m_motionType = (bi) ? hkpMotion::MOTION_DYNAMIC : hkpMotion::MOTION_FIXED;

		// Create rigid body
		hkpRigidBody* thisBody = new hkpRigidBody(info);

		hkStringBuf strb;
		strb.printf("Body_%d", bi);
		thisBody->setName(strb.cString());
		bodiesOut.pushBack(thisBody);

		// Prepare for next iteration
		vPos.addMul4(jointSize, chainDir);
		qRot.mul(deltaLinkRot);
	}

	// Clean-up
	bodyShapeL->removeReference();
	bodyShapeEe->removeReference();
}

//
//	Creates a chain of bodies connected through ball & socket constraints

hkpRigidBody* SolverDemo::createBallSocketChain(const int numLinks, const hkReal linkSize,
												const hkVector4& rootPos, const hkVector4& chainDir,
												const hkQuaternion& deltaLinkRot, 
												const hkReal linkMass, const hkReal endEffectorMass, const hkReal materialDensity, const hkReal initialHeight)

{
	// Create the chain of bodies
	hkArray<hkpRigidBody*> bodies;
	createBodiesChain(numLinks, linkSize, rootPos, chainDir, deltaLinkRot, linkMass, endEffectorMass, materialDensity, bodies);

	// Create the joints
	hkpRigidBody* prevBody = HK_NULL;
	for (int bi = 0; bi < numLinks; bi++)
	{
		// Create rigid body
		hkpRigidBody* thisBody = bodies[bi];
		m_world->addEntity(thisBody);
		thisBody->removeReference();

		// Join the body with the previous
		if ( prevBody )
		{
			// Compute constraint pivots in local space
			hkVector4 vPivot;
			vPivot.setAddMul4(rootPos, chainDir, linkSize * ((hkReal)bi - 0.5f));

			const hkTransform& transformBodyA = prevBody->getTransform();
			const hkTransform& transformBodyB = thisBody->getTransform();

			// Create constraint data
			hkpConstraintData* data = HK_NULL;
			{
				hkpBallAndSocketConstraintData* bsd = new hkpBallAndSocketConstraintData();
				bsd->setInWorldSpace(transformBodyA, transformBodyB, vPivot);
				data = bsd;
			}

			hkpConstraintInstance* constraint = new hkpConstraintInstance(prevBody, thisBody, data);

			hkStringBuf strb;
			strb.printf("BallSocket_%d_%d", bi - 1, bi);
			constraint->setName(strb.cString());

			data->removeReference();
			m_world->addConstraint(constraint);
			constraint->removeReference();
		}

		// Prepare for next iteration
		prevBody = thisBody;
	}

	// Move the bodies
	for (int bi = 1; bi < numLinks; bi++)
	{
		hkpRigidBody* body = bodies[bi];
		hkVector4 pos = body->getPosition();
		pos.addMul4(-initialHeight, chainDir);
		body->setPosition(pos);
	}

	// Return the end-effector body
	return prevBody;
}

//
//	Key handler

int SolverDemo::keyHandler(char key)
{
	switch (key)
	{
	case 'T':
		{
			toggleSolver();
		}
		break;

	case 'C':
		{
			m_showTexts = !m_showTexts;
		}
		break;

	case 'X':
		{
			m_world->lock();

			// Get all rigid bodies
			hkArray<hkpRigidBody*> bodies;
			hkpSolverWorld::collectRigidBodies(m_world, bodies);

			// Randomize their transforms
			for (int bi = bodies.getSize() - 1; bi >= 0; bi--)
			{
				randomizeBody(bodies[bi]);
			}

			m_world->unlock();
		}
		break;

	default:
		break;
	}

	return 0;
}

//
//	Advances the simulation time

hkDemo::Result SolverDemo::stepDemo()
{
	// Xbox pad A - 'X'
	// Xbox pad X - 'T'
	// Pad handler
	if ( m_env->m_window )
	{
		HKG_KEYBOARD_VKEY keyFromPad[HKG_PAD_NUM_BUTTONS] =
		{
			0, // HKG_PAD_BUTTON_0    (1)
			0, // HKG_PAD_BUTTON_1    (1<<1)
			0, // HKG_PAD_BUTTON_2    (1<<2)
			0, // HKG_PAD_BUTTON_3    (1<<3)
			'T', // HKG_PAD_DPAD_UP     (1<<4)
			'X', // HKG_PAD_DPAD_DOWN   (1<<5)
			0, // HKG_PAD_DPAD_LEFT   (1<<6)
			0, // HKG_PAD_DPAD_RIGHT  (1<<7)
			0, // HKG_PAD_SELECT	    (1<<8)	
			0, // HKG_PAD_START       (1<<9)
			0, // HKG_PAD_BUTTON_L1   (1<<10)
			0, // HKG_PAD_BUTTON_R1   (1<<11)
			0, // HKG_PAD_BUTTON_L2   (1<<12)
			0, // HKG_PAD_BUTTON_R2   (1<<13)
			0, // HKG_PAD_BUTTON_LSTICK  (1<<14)
			0, // HKG_PAD_BUTTON_RSTICK  (1<<15)
		};

		for ( int ci = 0 ; ci < 2; ci++)
		{
			for( int i = 0; i < HKG_PAD_NUM_BUTTONS; ++i )
			{
				if( m_env->m_window->getGamePad(ci).wasButtonPressed(HKG_PAD_BUTTON(1 << i)) )
				{
					HKG_KEYBOARD_VKEY key = keyFromPad[i];
					if( key )
					{
						keyHandler(key);
					}
				}
			}
		}
	}

	// Call base class
	hkDemo::Result ret = hkDefaultPhysics2012Demo::stepDemo();

	// Draw texts
	if ( m_showTexts )
	{
		for (int i = m_texts.getSize() - 1; i >= 0; i--)
		{
			TextInfo& ti = m_texts[i];
			HK_DISPLAY_3D_TEXT(ti.m_text.cString(), ti.m_pos, ti.m_color);
		}
	}

	return ret;
}

//
//	Randomizes a rigid body

void SolverDemo::randomizeBody(hkpRigidBody* rb)
{
	if( !rb || rb->isFixed() )
	{
		return;
	}

	m_world->lock();
	{
		const hkReal maxVal = MAX_RANDOM_TRANSLATION;
		hkVector4 vMin, vMax;
		vMax.setAll(maxVal);
		vMin.setNeg4(vMax);

		hkVector4 vRandomPos;
		hkQuaternion vRandomRot;
		m_rng.getRandomVectorRange(vMin, vMax, vRandomPos);
		m_rng.getRandomVector11(vRandomRot.m_vec);

		vRandomPos.add4(rb->getPosition());
		vRandomRot.m_vec.add4(rb->getRotation().m_vec);
		vRandomRot.normalize();

		rb->setPositionAndRotation(vRandomPos, vRandomRot);
	}
	m_world->unlock();
}

//
//	Toggles the solver from/to stable to/from fast 

void SolverDemo::toggleSolver()
{
	m_stableSolverOn = !m_stableSolverOn;
	hkpConstraintAtom::SolvingMethod method = m_stableSolverOn ? hkpConstraintAtom::METHOD_STABILIZED : hkpConstraintAtom::METHOD_OLD;
	hkpConstraintStabilizationUtil::setConstraintsSolvingMethod(m_world, method);
}

//
//	Adds a text for display

void SolverDemo::addText(const char* text, const hkVector4& vPos, const hkColor::Argb color)
{
	TextInfo& ti = m_texts.expandOne();
	ti.m_text = text;
	ti.m_pos = vPos;
	ti.m_color = color;
}


static const char helpString[] = \
"Test bed for constraint solvers";

HK_DECLARE_DEMO(SolverDemo, HK_DEMO_TYPE_PHYSICS_2012, helpString, helpString); 

//
//	END!
//

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
