/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Demo includes
#include <Demos/demos.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Demos/Physics2012/Api/Constraints/EaseConstraintsAction/EaseConstraintsActionDemo.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>

// Base includes
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Serialize/Util/hkLoader.h>

// Physics includes
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics2012/Collide/Filter/Group/hkpGroupFilter.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
#include <Physics2012/Utilities/Collide/Filter/GroupFilter/hkpGroupFilterUtil.h>

// Animation includes
#include <Animation/Physics2012Bridge/Instance/hkaRagdollInstance.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapper.h>

// Ease Constraints Utils
#include <Physics2012/Utilities/Actions/EaseConstraints/hkpEaseConstraintsAction.h>
#include <Physics/Constraint/Data/hkpConstraintDataUtils.h>

// Asset File Paths
static const char* RIG_FILE = "Resources/Animation/EaseConstraints/Firefighter_Rig.hkt";
static const char* SKELETON_FILE = "Resources/Animation/EaseConstraints/hkInvalidPose.hkt";

// Some Demo Config Options
#define EASE_CONSTRAINTS_DEMO_SLOW_MO_FACTOR			4.0f
#define EASE_CONSTRAINTS_DEMO_RESTORE_TIME_IN_SECONDS	5.0f

static const char helpString[] = \
"This demo illustrates the use of the hkpEaseConstraintsAction. ";

static const char detailsString[] = \
"The pose on the far left violates the ragdolls' constraints.\
 The ragdolls exhibit various artifacts of matching this\
 violating pose. LimHinge signifies if the ragdoll has a limited hinge\
 constraint. Loosened signifies the ragdoll will have its constraint limits\
 loosened. Restored signifies it will have its constraint limits restored\
 over time (see code for more details).\n\n\
 Press \x10\\\x11 to enable constraints (without\\with gravity).\n\
 Press \x12\\\x13 to execute the above options in slow motion.";

EaseConstraintsActionDemo::EaseConstraintsActionDemo( hkDemoEnvironment* env )
:	hkDefaultPhysics2012Demo( env )
,	m_loader(HK_NULL)
,	m_lowAnimationPose(HK_NULL)
,	m_ragdollsEnabled(false)
{
	for (int i = 0; i < 4; i++)
	{
		m_ragdolls[i] = HK_NULL;
	}
	
	//
	// Disable some warnings that are not pertinent to this demo
	//
	{
		hkError& error = hkError::getInstance();

		// Warnings when a deprecated object is loaded.
		// Fires because Firefighter_Rig.hkx contains some
		// deprecated objects (which aren't used)
		error.setEnabled( 0x9fe65234, false );
		error.setEnabled( 0x651f7aa5, false );

		// Warning when setPoseModelSpace is called as
		// this method should not be used to drive
		// ragdolls.
		error.setEnabled( 0x71c72fe7, false );
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(0.2f,0,9.5f);
		hkVector4 to(0.3f,0,0);
		hkVector4 up(0,1,0);

		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create world
	//
	{
		hkpWorldCinfo info;
		info.m_gravity = hkVector4::getZero();
		info.m_enableDeactivation = false;

		m_world = new hkpWorld( info );
	}

	m_world->lock();

	// set constraint scale to something reasonable
	// for better viewing in Visual Debugger
	{
		hkpConstraintViewer::m_scale = 0.4f;
		m_debugViewerNames.pushBack(hkpConstraintViewer::getName());
	}

	setupGraphics( );

	//
	// Set group collision filter
	//
	{
		hkpGroupFilter* filter  = new hkpGroupFilter();
		m_world->setCollisionFilter( filter, true);
		filter->removeReference();
	}

	// Register ALL agents (though some may not be necessary)
	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

	m_loader = new hkLoader();

	//
	// Load the following from RIG_FILE:
	//	- rig skeleton
	//  - ragdoll instance
	//  - high to low skeleton mapper
	//
	hkaSkeleton* highSkeleton = HK_NULL;
	hkaSkeletonMapper* highToLowMapper = HK_NULL;
	{
		hkStringBuf assetFile( HK_GET_DEMOS_ASSET_FILENAME(RIG_FILE) );
		hkAssetManagementUtil::getFilePath( assetFile );
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset" );
		if ( container )
		{
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			// Grab the skeleton
			HK_ASSERT2( 0x27343435, ac && ( ac->m_skeletons.getSize() > 0 ), "Couldn't load high-res skeleton" );
			highSkeleton = ac->m_skeletons[0];

			// Grab the ragdoll instance which has limited hinge constraints
			m_ragdolls[0] = reinterpret_cast<hkaRagdollInstance*>( container->findObjectByType( hkaRagdollInstanceClass.getName() ) );
			HK_ASSERT2( 0, m_ragdolls[0], "Couldn't load ragdoll setup" );
			setupRagdoll( m_ragdolls[0], hkColor::RED );

			// Create clone 1, leave limited hinge constraints
			m_ragdolls[1] = cloneRagdoll( m_ragdolls[0], false, hkColor::ORANGE );

			// Create clone 2, replace limited hinge constraints with ragdoll constraints
			m_ragdolls[2] = cloneRagdoll( m_ragdolls[0], true, hkColor::GREEN );

			// Create clone 3, replace limited hinge constraints with ragdoll constraints
			m_ragdolls[3] = cloneRagdoll( m_ragdolls[0], true, hkColor::BLUE );

			const hkaSkeleton* tempLowSkeleton = m_ragdolls[0]->getSkeleton();

			// Grab the skeleton mappers
			// We iterate through all the skeleton mappers contained in the file,
			// and compare them against the ragdoll's skeleton to find out which one is the
			// low-resolution skeleton
			highToLowMapper = HK_NULL;
			void *objectFound = container->findObjectByType( hkaSkeletonMapperClass.getName() );
			while( objectFound )
			{
				hkaSkeletonMapper* mapperFound = reinterpret_cast<hkaSkeletonMapper*>( objectFound );

				if( mapperFound->m_mapping.m_skeletonB == tempLowSkeleton )
				{
					highToLowMapper = mapperFound;
				}

				objectFound = container->findObjectByType( hkaSkeletonMapperClass.getName(), objectFound );
			}

			HK_ASSERT2( 0, highToLowMapper, "Couldn't load high-to-ragdoll mapping" );
		}
	}

	//
	// Load the following from SKELETON_FILE:
	//	- skeleton with invalid pose
	//
	hkaSkeleton* invalidPoseSkeleton;
	{
		hkStringBuf assetFile( HK_GET_DEMOS_ASSET_FILENAME(SKELETON_FILE) );
		hkAssetManagementUtil::getFilePath( assetFile );
		hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
		HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset" );
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		// Grab the skeleton
		HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0), "Couldn't load skeleton." );
		invalidPoseSkeleton = ac->m_skeletons[0];
	}

	// Lock translations on all the bones, then unlock them only on the pelvis
	if ( highToLowMapper )
	{
		hkaSkeleton& skeletonA = *const_cast<hkaSkeleton*>(highToLowMapper->m_mapping.m_skeletonA.val());
		hkaSkeletonUtils::lockTranslations(skeletonA);

		const hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName(skeletonA, "Root");
		if (pelvis >= 0)
		{
			skeletonA.m_bones[pelvis].m_lockTranslation = false;
		}
	}

	// Lock translations on all the bones, then unlock them only on the pelvis
	if ( highToLowMapper )
	{
		hkaSkeleton& skeletonB = *const_cast<hkaSkeleton*>(highToLowMapper->m_mapping.m_skeletonB.val());
		hkaSkeletonUtils::lockTranslations(skeletonB);

		hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName(skeletonB, "Biped_Root");
		if (pelvis >=0 )
		{
			skeletonB.m_bones[pelvis].m_lockTranslation = false;
		}
	}

	//
	// Get violating pose from invalidPoseSkeleton for the ragdolls
	//
	if ( highToLowMapper )
	{
		hkaPose poseHigh( invalidPoseSkeleton );
		poseHigh.setToReferencePose();

		// Construct a low-res pose in the reference pose
		const hkaSkeleton* lowSkeleton = m_ragdolls[0]->getSkeleton();
		m_lowAnimationPose = new hkaPose( lowSkeleton );
		m_lowAnimationPose->setToReferencePose();

		// Map the high-res pos to the low-res pose
		highToLowMapper->mapPose( poseHigh, *m_lowAnimationPose, hkaSkeletonMapper::CURRENT_POSE );
	}

	//
	// Create floor
	//
	{
		hkVector4 halfExtents( 10.0f, 1.0f, 10.0f );
		hkpBoxShape* shape = new hkpBoxShape( halfExtents, 0 );

		hkpRigidBodyCinfo rigidBodyInfo;

		rigidBodyInfo.m_shape = shape;
		rigidBodyInfo.m_position.set( 0.0f, -2.8f, 0.0f );
		rigidBodyInfo.m_motionType = hkpMotion::MOTION_FIXED;

		hkpRigidBody* rigidBody = new hkpRigidBody( rigidBodyInfo );
		shape->removeReference();

		m_world->addEntity( rigidBody );
		rigidBody->removeReference();
	}

	// Move ragdolls into position
	hkQsTransform t = hkQsTransform::getIdentity();
	hkQuaternion r;
	r.setAxisAngle( hkVector4(0,1,0), HK_REAL_PI / 2 );
	t.setRotation(r);
	hkReal ragdollSpacing = 2;
	for (int i = 0; i < 4; i++)
	{
		t.setTranslation( hkVector4( -4.0f + (i+1) * ragdollSpacing, 0.0f, 0.0f, 0.0f ) );
		if ( m_ragdolls[i] )
		{
			m_ragdolls[i]->setPoseModelSpace( m_lowAnimationPose->accessSyncedPoseModelSpace().begin(), t );
		}
	}

	m_world->unlock();
}

EaseConstraintsActionDemo::~EaseConstraintsActionDemo()
{
	m_world->lock();
	
	if ( m_lowAnimationPose )
	{
		delete m_lowAnimationPose;
	}
	

	for (int i = 0; i < 4; i++)
	{
		if ( m_ragdolls[i] )
		{
			m_ragdolls[i]->removeFromWorld();
		}
	}

	// m_ragdolls[0]->removeReference(); // not cloned
	for (int i = 1; i < 4; i++)
	{
		if ( m_ragdolls[i] )
		{
			m_ragdolls[i]->removeReference();
		}
	}
	
	if ( m_loader )
	{
		delete m_loader;
	}

	m_world->unlock();
}

hkDemo::Result EaseConstraintsActionDemo::stepDemo()
{
	//
	// Draw invalid pose
	//
	if ( m_lowAnimationPose )
	{
		hkQsTransform t = hkQsTransform::getIdentity();
		t.setTranslation( hkVector4( -4.0f, 0.0f, 0.0f ) );
		hkQuaternion r;
		r.setAxisAngle( hkVector4(0,1,0), HK_REAL_PI / 2 );
		t.setRotation( r );

		AnimationUtils::drawPose( *m_lowAnimationPose, t );
	}

	//
	// Draw captions
	//
	{
		HK_DISPLAY_3D_TEXT( "Invalid\nPose", hkVector4( -4.4f, 2.4f, 0.0f ), hkColor::WHITE );

		HK_DISPLAY_3D_TEXT( "LimHinges:Y\nNot modified", hkVector4( -2.6f, 2.6f, 0.0f ), hkColor::RED );

		HK_DISPLAY_3D_TEXT( "LimHinges:Y\nLoosened:Y\nRestored:N", hkVector4( -0.6f, 2.6f, 0.0f ), hkColor::ORANGE );

		HK_DISPLAY_3D_TEXT( "LimHinges:N\nLoosened:Y\nRestored:N", hkVector4( 1.4f, 2.6f, 0.0f ), hkColor::GREEN );

		HK_DISPLAY_3D_TEXT( "LimHinges:N\nLoosened:Y\nRestored:Y", hkVector4( 3.4f, 2.6f, 0.0f ), hkColor::BLUE );
	}

	//
	// Enable the ragdolls and allow them to resolve
	//
	if( !m_ragdollsEnabled && 
	  ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_0 ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) ||
	    m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_2 ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_3 ) ) )
	{
		m_world->markForWrite();

		//
		// First ragdoll has limited hinge constraints.
		// We will not modify the ragdoll in any way.
		// This ragdoll will have it's constraints breached in both its
		// limited axes and it's principle axes.  It will react violently
		// to it's invalid pose.
		//
		if ( m_ragdolls[0] )
		{
			enableRagdoll( m_ragdolls[0] );
		}

		//
		// This ragdoll also has limited hinge constraints.
		// But we will loosen its constraint limits.
		// This ragdoll will only have it's constraints breached in its
		// principle axes.  This ragdoll illustrates that easing the
		// constraint limits isn't sufficient in all cases.
		// If the animation pose breaches the principle axes of the
		// constraint, you will still get strange popping.
		// This ragdoll will have popping around the elbows and knees.
		//
		if ( m_ragdolls[1] )
		{
			enableRagdoll( m_ragdolls[1] );
			const hkArray<hkpConstraintInstance*>& constraints = m_ragdolls[1]->getConstraintArray();
			for( int i = 0; i < constraints.getSize(); i++ )
			{
				hkpConstraintDataUtils::loosenConstraintLimits( constraints[i]->getDataRw(), constraints[i]->getRigidBodyA()->getTransform(), constraints[i]->getRigidBodyB()->getTransform());
			}
		}

		//
		// This ragdoll has had its limited hinge constraints
		// replaced by ragdoll constraints.
		// We will also loosen its constraint limits.
		// This ragdoll will not have any constraints breached.
		// With gravity off, it will just look fixed in space.
		// With gravity on, it will fall but can have strange
		// behavior when interacted with as its constraints limits
		// will be unnaturally loosened.  This is most noticeable
		// around the leg which was stretched backward.
		//
		if ( m_ragdolls[2] )
		{
			enableRagdoll( m_ragdolls[2] );
			const hkArray<hkpConstraintInstance*>& constraints = m_ragdolls[2]->getConstraintArray();
			for( int i = 0; i < constraints.getSize(); i++ )
			{
				hkpConstraintDataUtils::loosenConstraintLimits( constraints[i]->getDataRw(), constraints[i]->getRigidBodyA()->getTransform(), constraints[i]->getRigidBodyB()->getTransform());
			}
		}

		//
		// This ragdoll has had its limited hinge constraints
		// replaced by ragdoll constraints.
		// We will also loosen its constraint limits and then restore them over time.
		// This ragdoll will not have any constraints breached initially.
		// As the limits return to normal, it will have some forces applied
		// to satisfy the constraints.  It will not suffer from unnatural
		// reactions once its limits have been fully restored.
		//
		if ( m_ragdolls[3] )
		{
			enableRagdoll( m_ragdolls[3] );
			const hkArray<hkpEntity*>& entities = reinterpret_cast<const hkArray<hkpEntity*>&>( m_ragdolls[3]->getRigidBodyArray() );
			hkpEaseConstraintsAction* easeConstraintsAction = new hkpEaseConstraintsAction( entities );
			easeConstraintsAction->loosenConstraints();
			easeConstraintsAction->restoreConstraints( EASE_CONSTRAINTS_DEMO_RESTORE_TIME_IN_SECONDS );
			m_world->addAction( easeConstraintsAction );
			easeConstraintsAction->removeReference();
		}

		m_world->unmarkForWrite();

		m_ragdollsEnabled = true;
	}

	if( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_3 ) )
	{
		m_world->markForWrite();
		m_world->setGravity( hkVector4( 0.0f, -9.8f, 0.0f ) );
		m_world->unmarkForWrite();
	}
	else if( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_0 ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_2 ) )
	{
		m_world->markForWrite();
		m_world->setGravity( hkVector4::getZero() );
		m_world->unmarkForWrite();
	}

	static bool slowMo = false;
	if( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_0 ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) )
	{
		slowMo = false;
	}
	else if( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_2 ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_3 ) )
	{
		slowMo = true;
	}

	if( slowMo )
	{
		hkReal frameStepTime = m_timestep;
		m_timestep /= EASE_CONSTRAINTS_DEMO_SLOW_MO_FACTOR;
		hkDefaultPhysics2012Demo::stepDemo();
		m_timestep = frameStepTime;
	}
	else
	{
		hkDefaultPhysics2012Demo::stepDemo();
	}

	return DEMO_OK;
}

hkaRagdollInstance* EaseConstraintsActionDemo::cloneRagdoll( hkaRagdollInstance* sourceRagdoll, bool replaceLimitedHingeConstraintsOnElbow, hkColor::Argb color )
{
	hkaRagdollInstance* newInstance = HK_NULL;

	// Replace elbow's limited hinge constraints with ragdoll constraints.
	// The animation violates the non-limited axes of the elbow constraint
	// which is not supported by hkpEaseConstraintsAction.
	if( replaceLimitedHingeConstraintsOnElbow )
	{
		// *** Parts taken from hkpRagdollInstance::clone
		const int numRigidBodies = sourceRagdoll->getRigidBodyArray().getSize();
		const int numConstraints = sourceRagdoll->getConstraintArray().getSize();

		hkLocalArray<hkpRigidBody*> clonedRigidBodies( numRigidBodies );
		clonedRigidBodies.setSize( numRigidBodies );

		hkLocalArray<hkpConstraintInstance*> clonedConstraints( numConstraints );
		clonedConstraints.setSize( numConstraints );
		{
			for( int rb=0; rb<numRigidBodies; rb++ )
			{
				clonedRigidBodies[rb] = sourceRagdoll->getRigidBodyArray()[rb]->clone();
			}

			for( int c=0; c<numConstraints; c++ )
			{
				hkpRigidBody* clonedChild = clonedRigidBodies[c+1];
				hkpRigidBody* clonedParent = clonedRigidBodies[sourceRagdoll->getSkeleton()->m_parentIndices[c+1]];
				const hkpConstraintInstance* constraint = sourceRagdoll->getConstraintArray()[c];

				if(constraint->getData()->getType() == hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE)
				{
					// gather data from limited hinge constraint
					hkpLimitedHingeConstraintData* limitedHingeData = static_cast<hkpLimitedHingeConstraintData*>(const_cast<hkpConstraintData*>(constraint->getData()));
					hkVector4& childPivot = limitedHingeData->m_atoms.m_transforms.m_transformA.getColumn(3);
					hkVector4& parentPivot = limitedHingeData->m_atoms.m_transforms.m_transformB.getColumn(3);
					hkVector4& childPlane = limitedHingeData->m_atoms.m_transforms.m_transformA.getColumn(0);
					hkVector4& parentPlane = limitedHingeData->m_atoms.m_transforms.m_transformB.getColumn(0);

					// get childTwist axis and compute a parentTwist axis
					// which closely matches the limited hinge's min/max limits.
					hkReal minAng = limitedHingeData->getMinAngularLimit();
					hkReal maxAng = limitedHingeData->getMaxAngularLimit();
					hkReal angExtents = (maxAng - minAng) / 2.0f;
					hkQuaternion minAngRotation; minAngRotation.setAxisAngle( parentPlane, -angExtents + maxAng );
					hkVector4& childTwist = limitedHingeData->m_atoms.m_transforms.m_transformA.getColumn(2);
					hkVector4& parentLimitedAxis = limitedHingeData->m_atoms.m_transforms.m_transformB.getColumn(2);
					hkVector4 parentTwist; parentTwist.setRotatedDir( minAngRotation, parentLimitedAxis );

					hkpRagdollConstraintData* ragdollData = new hkpRagdollConstraintData();
					ragdollData->setInBodySpace( childPivot, parentPivot, childPlane, parentPlane, childTwist, parentTwist );

					// adjust limits to make it like the hinge constraint
					ragdollData->setConeAngularLimit( angExtents );
					ragdollData->setTwistMinAngularLimit( 0.0f );
					ragdollData->setTwistMaxAngularLimit( 0.0f );
					ragdollData->setMaxFrictionTorque( limitedHingeData->getMaxFrictionTorque() );
					ragdollData->setAngularLimitsTauFactor( limitedHingeData->getAngularLimitsTauFactor() );

					clonedConstraints[c] = new hkpConstraintInstance( clonedChild, clonedParent, ragdollData );
					ragdollData->removeReference();
				}
				else
				{
					clonedConstraints[c] = constraint->clone( clonedChild, clonedParent, hkpConstraintInstance::CLONE_DATAS_WITH_MOTORS );
				}
			}
		}

		newInstance = new hkaRagdollInstance( clonedRigidBodies, clonedConstraints, sourceRagdoll->getSkeleton(), sourceRagdoll->m_boneToRigidBodyMap );

		{
			for (int rb=0; rb<numRigidBodies; rb++)
			{
				clonedRigidBodies[rb]->removeReference();
			}

			for (int c=0; c<numConstraints;c++)
			{
				clonedConstraints[c]->removeReference();
			}
		}
	}
	else
	{
		newInstance = sourceRagdoll->clone( hkpConstraintInstance::CLONE_DATAS_WITH_MOTORS );
	}

	setupRagdoll( newInstance, color );

	return newInstance;
}

void EaseConstraintsActionDemo::setupRagdoll( hkaRagdollInstance* ragdollInstance, hkColor::Argb color )
{
	// each ragdoll that is setup will have a different system group
	static int systemGroup = 0;
	systemGroup++;

	// Initialize collision/motion settings for the rigidbody ragdoll parts
	for( int b = 0; b < ragdollInstance->getNumBones(); b++ )
	{
		hkpRigidBody* rb = ragdollInstance->getRigidBodyOfBone(b);

		// Initialize with quality type and collision filter
		rb->setMotionType( hkpMotion::MOTION_DYNAMIC );

		if(b != 0)
		{
			rb->setCollisionFilterInfo( hkpGroupFilter::calcFilterInfo(1, systemGroup, b - 1) );
		}
	}

	// Tweak the mass and inertia tensors of the rigid bodies to improve stability.
	const hkArray<hkpConstraintInstance*>& constraints = ragdollInstance->getConstraintArray();
	hkpInertiaTensorComputer::optimizeInertiasOfConstraintTree( constraints.begin(), constraints.getSize(), ragdollInstance->getRigidBodyOfBone(0) );

	// Disable collisions between bones in the ragdoll
	hkpGroupFilterUtil::disableCollisionsBetweenConstraintBodies( constraints.begin(), constraints.getSize(), systemGroup );

	// add ragdoll to the world and disable it's constraints
	ragdollInstance->addToWorld( m_world, false );
	for(int i = 0; i < constraints.getSize(); i++)
	{
		constraints[i]->setEnabled( false );
	}

	// set color on ragdoll parts (must be done AFTER it's been added to world)
	for( int b =0; b < ragdollInstance->getNumBones(); b++)
	{
		hkpRigidBody* rb = ragdollInstance->getRigidBodyOfBone(b);
		HK_SET_OBJECT_COLOR( (hkUlong)rb->getCollidable(), color );
	}
}

void EaseConstraintsActionDemo::enableRagdoll( hkaRagdollInstance* ragdollInstance )
{
	for( int b =0; b < ragdollInstance->getNumBones(); b++ )
	{
		if( b != 0 )
		{
			hkpConstraintInstance* constraint = ragdollInstance->getConstraintOfBone(b);

			if( !constraint->isEnabled() )
			{
				constraint->setEnabled( true );
			}
		}
	}
}


HK_DECLARE_DEMO(EaseConstraintsActionDemo, HK_DEMO_TYPE_PHYSICS_2012, helpString, detailsString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
