/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Demos/Physics2012/Api/Constraints/DeformableConstraint/DeformableConstraintDemo.h>

#include <Physics2012/Collide/Agent/ConvexAgent/BoxBox/hkpBoxBoxAgent.h>
#include <Physics/Constraint/Data/DeformableFixed/hkpDeformableFixedConstraintData.h>
#include <Physics2012/Dynamics/Constraint/hkpConstraintInstance.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>


DestructionConstraintDemo::DestructionConstraintDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{
	// Setup the camera
	{
		hkVector4 from(0.0f, 0.0f, 10.0f);
		hkVector4 to  (0.0f, 0.0f,  0.0f);
		hkVector4 up  (0.0f, 1.0f,  0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	// Create the world
	{
		hkpWorldCinfo info;
		{
			info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
			info.setBroadPhaseWorldSize( 100.0f );
			info.m_enableDeactivation = false;
		}
		m_world = new hkpWorld(info);

		m_world->lock();

		// Register the single agent required (a hkBoxBoxAgent)
		hkpBoxBoxAgent::registerAgent(m_world->getCollisionDispatcher());

		// Create constraint viewer
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 1.0f;

		setupGraphics();
	}

	// Some vectors to be used for setup
	hkVector4 halfSize(0.5f, 0.5f, 0.5f);
	hkVector4 size;
	size.setMul4(2.0f, halfSize);
	hkVector4 position(size(0), -size(1) -0.1f, 0);

	// Create Box Shape
	hkpBoxShape* boxShape = new hkpBoxShape(halfSize , 0);

	// Create fixed rigid body
	{
		hkpRigidBodyCinfo info;
		{
			info.m_position		. set(0.0f, 0.0f, 0.0f);
			info.m_shape		= boxShape;
			info.m_motionType	= hkpMotion::MOTION_FIXED;
		}

		m_bodyA = new hkpRigidBody(info);
		m_world->addEntity(m_bodyA);
		m_bodyA->removeReference();	
	}

	// Create movable rigid body
	{
		hkpRigidBodyCinfo info;
		{
			hkReal mass = 10.0f;

			// Compute the box inertia tensor
			hkMassProperties massProperties;
			hkpInertiaTensorComputer::computeBoxVolumeMassProperties(halfSize, mass, massProperties);

			info.m_position			= position;
			info.m_shape			= boxShape;
			info.m_mass				= mass;
			info.m_inertiaTensor	= massProperties.m_inertiaTensor;
			info.m_centerOfMass		= massProperties.m_centerOfMass;
			info.m_motionType		= hkpMotion::MOTION_BOX_INERTIA;
		}

		m_bodyB = new hkpRigidBody(info);
		m_world->addEntity(m_bodyB);
		m_bodyB->removeReference();	
	}

	// Remove reference from shape
	boxShape->removeReference();

	// Create Destruction constraint.
	{
		// Set the pivot
		hkVector4 pt;
		pt.setAdd(position, halfSize);
		pt.setComponent<0>(pt.getComponent<0>() - size.getComponent<0>());	// Move pivot to corner of cube
		hkTransform pivot;
		pivot.setIdentity();
		pivot.setTranslation(pt);

		// Create the constraint
		m_constraint = new hkpDeformableFixedConstraintData(); 
		m_constraint->setInWorldSpace(m_bodyA->getTransform(), m_bodyB->getTransform(), pivot, pivot);
		m_constraint->setLinearLimit(5.0f);
		m_constraint->setAngularLimit(HK_REAL_PI * 0.25f);

		// Create instance and add to world
		hkpConstraintInstance* cInstance = new hkpConstraintInstance(m_bodyA, m_bodyB, m_constraint);
		m_world->addConstraint(cInstance); 	
		m_constraint->removeReference();
		cInstance->removeReference();
	}

	m_world->unlock();
}

hkDemo::Result DestructionConstraintDemo::stepDemo()
{
	// Call base class
	return hkDefaultPhysics2012Demo::stepDemo();
}

hkBool DestructionConstraintDemo::objectPicked(const hkgDisplayObject* displayObject, const hkVector4& worldPosition, const hkgViewportPickData* pd)
{
	// Call base class
	const hkBool ret = hkDefaultPhysics2012Demo::objectPicked(displayObject, worldPosition, pd);
	if ( ret && m_mouseSpring )
	{
		// Something was picked
		m_world->lock();
		const hkpRigidBody* pickedRb = m_mouseSpring->getRigidBody();
		m_externalStress = (pickedRb == m_bodyB);

		// Set limited strength on the constraint
		if ( m_externalStress )
		{
			hkSymmetricMatrix3 tLinYield, tLinBreak, tAngYield, tAngBreak;
			hkVector4 v;	
			
			v.set(400.0f, 400.0f, HK_REAL_MAX);			tLinYield.setDiagonal(v);
			v.set(25000.0f, 25000.0f, HK_REAL_MAX);		tLinBreak.setDiagonal(v);
			v.set(HK_REAL_MAX, 8000.0f, HK_REAL_MAX);	tAngYield.setDiagonal(v);
			v.set(HK_REAL_MAX, 50000.0f, HK_REAL_MAX);	tAngBreak.setDiagonal(v);

			m_constraint->setLinearStrength(tLinYield, tLinBreak);
			m_constraint->setAngularStrength(tAngYield, tAngBreak);
		}

		m_world->unlock();
	}

	return ret;
}

void DestructionConstraintDemo::objectReleased()
{
	// Call base class
	hkDefaultPhysics2012Demo::objectReleased();

	if ( m_externalStress )
	{
		m_externalStress = false;

		// Bake current constraint length
		m_world->lock();
		{
			const hkTransform worldFromBodyA = m_bodyA->getTransform();
			const hkTransform worldFromBodyB = m_bodyB->getTransform();

			hkTransform worldFromPivotA;	worldFromPivotA.setMul(worldFromBodyA, m_constraint->m_atoms.m_transforms.m_transformA);
			hkTransform worldFromPivotB;	worldFromPivotB.setMul(worldFromBodyB, m_constraint->m_atoms.m_transforms.m_transformB);

			// Compute distance
			hkVector4 vOffset;
			vOffset.setSub(worldFromPivotB.getTranslation(), worldFromPivotA.getTranslation());
			vOffset._setRotatedInverseDir(worldFromBodyA.getRotation(), vOffset);
			m_constraint->setLinearOffset(vOffset);

			// Compute relative rotation
			hkRotation relRotAB;	relRotAB.setMulInverseMul(worldFromPivotA.getRotation(), worldFromPivotB.getRotation());
			hkQuaternion qOffset;	qOffset.setAndNormalize(relRotAB);
			m_constraint->setAngularOffset(qOffset);

			// Set infinite strength
			hkSymmetricMatrix3 t;	t.m_offDiag.setZero(); t.m_diag = hkVector4::getConstant<HK_QUADREAL_MAX>();
			m_constraint->setLinearStrength(t, t);
			m_constraint->setAngularStrength(t, t);
		}
		m_world->unlock();
	}
}




HK_DECLARE_DEMO( DestructionConstraintDemo, HK_DEMO_TYPE_PHYSICS_2012, "Simple example of a Destruction constraint.", "" );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
