/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Constraints/ConstraintStability/RagdollPoseRecovery/RagdollPoseRecoveryDemo.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>

#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Dynamics/World/Util/hkpWorldCallbackUtil.h>
#include <Physics/Constraint/Data/BallAndSocket/hkpBallAndSocketConstraintData.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintStabilizationUtil.h>
#include <Physics/Constraint/Data/hkpConstraintDataUtils.h>
#include <Physics2012/Internal/Dynamics/Constraints/hkpConstraintProjector.h>
#include <Physics2012/Utilities/Actions/EaseConstraints/hkpEaseConstraintsAction.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Shape/Misc/Transform/hkpTransformShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>


#include <Graphics/Common/Input/Pad/hkgPad.h>

#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>


enum DemoMode
{
	RECOVER_LIMITS = 1,
	RECOVER_POSITIONS = 2,
};

struct RagdollPoseRecoveryVariant
{
	const char*	 m_name;
	DemoMode m_demoMode;
	bool m_recoverLimitsGently;
	bool m_projectConstraints;
	const char* m_details;
};

// suggestions for variants: motor tau + dumping,
// with(out) collision
static const RagdollPoseRecoveryVariant g_variants[] =
{
	{ "Limits breached only. Uncontrolled.", RECOVER_LIMITS, false, false, "" },
	{ "Limits breached only. Ease constraints action.", RECOVER_LIMITS, true, false, "" },

	{ "Positions breached. Uncontrolled.", RECOVER_POSITIONS, false, false, "" },
	{ "Positions breached. Projection.", RECOVER_POSITIONS, false, true, "" },

	{ "Limits+positions breached. Uncontrolled.", DemoMode(RECOVER_LIMITS | RECOVER_POSITIONS), false, false, "" },
	{ "Limits+positions breached. Projection + Ease constrainst action.", DemoMode(RECOVER_LIMITS | RECOVER_POSITIONS), true, true, "" },
};


/// Simple example of constraints projection usage
class ConstraintProjectorListener : public hkpConstraintListener
{
	public:

		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO,ConstraintProjectorListener);

		/// constructor
		ConstraintProjectorListener(hkpWorld* world)
		{
			m_world = world;
			m_world->lock();
			m_world->addConstraintListener(this);
			m_world->unlock();
		}

		/// destructor
		virtual ~ConstraintProjectorListener()
		{
			m_world->lock();
			m_world->removeConstraintListener(this);
			m_world->unlock();
		}

		/// implementation of hkpConstraintListener
		void	constraintViolatedCallback( hkpConstraintInstance* constraint )
		{
			m_criticalSection.enter();
			m_constraintsToProject.pushBack(constraint);
			constraint->addReference();
			m_criticalSection.leave();
		}
			
		/// project and clear violated constraints.
		void	projectAndClear()
		{
			m_criticalSection.enter();
			if(m_constraintsToProject.getSize())
			{
				hkpConstraintProjector	cp(hkpConstraintProjector::HK_REPAIR_VIOLATIONS);
				cp.project(m_constraintsToProject);
			}
			clear();
			m_criticalSection.leave();		
		}

		/// clear violated constraints.
		void	clear()
		{
			m_criticalSection.enter();
			hkReferencedObject::removeReferences( m_constraintsToProject.begin(), m_constraintsToProject.getSize() );
			m_constraintsToProject.clear();
			m_criticalSection.leave();
		}

	public:

		hkpWorld*												m_world;
		hkCriticalSection										m_criticalSection;
		hkArray<hkpConstraintInstance*>							m_constraintsToProject;
};

static hkReal g_originalTau;
static hkReal g_originalDamping;

static void addNosePalmsAndFeetToRagdoll(hkpPhysicsSystem* ragdoll);

RagdollPoseRecoveryDemo::RagdollPoseRecoveryDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
,	m_rndGen(180673)
,	m_doProjection(false)
{
	//
	//	Build a world
	//
	{
		hkpWorldCinfo worldinfo;
		worldinfo.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);
		//worldinfo.m_gravity.set( 0, 0, -9.81f );		
		worldinfo.m_gravity.set( 0, 0, 0 );		
		worldinfo.m_enableDeactivation	=	false;
		worldinfo.setBroadPhaseWorldSize( 200.0f );
		worldinfo.m_maxConstraintViolation	=	0.05f;	// Set maximum constraint violation to 5cm.
		m_world = new hkpWorld( worldinfo );
		m_world->lock();

		//hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Create and attach our listener
	//
	{
		m_violatedConstraintListener	=	new ConstraintProjectorListener(m_world);
	}



	//
	// Visualize constraints
	//
	{
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 0.05f;
	}

	//
	// Misc
	//

	m_shootGunDelay	=	0;
	
	//
	// Create rag dolls
	//
	{
		hkQuaternion		rotation = hkQuaternion::getIdentity();
		hkVector4			position = hkVector4::getZero();
		position(2)	=	5;
		const hkReal height = 2;

		hkpPhysicsSystem*	ragdoll   = GameUtils::createRagdoll( height, position,	rotation, 0+1, GameUtils::RPT_CAPSULE );

		addNosePalmsAndFeetToRagdoll(ragdoll);

		m_ragdolls.pushBack( ragdoll );
		//// invalidate limits
		//if (g_variants[m_env->m_variantId].m_demoMode & RECOVER_LIMITS)
		//{
		//	setupInvalidRagdollPose();
		//}

		//if (g_variants[m_env->m_variantId].m_demoMode & RECOVER_POSITIONS)
		//{
		//	// corrupt positions
		//	hkVector4 pos = ragdoll->getRigidBodies()[0]->getPosition();
		//	for (int bi = 0; bi < ragdoll->getRigidBodies().getSize(); bi++)
		//	{
		//		ragdoll->getRigidBodies()[bi]->setPosition(pos);
		//	}
		//}

		m_world->addPhysicsSystem(ragdoll);

	}

	//
	//	Create a ground box
	//
	{		
		hkVector4 size( 25, 25, 1 );

		hkpRigidBody* rb = GameUtils::createBox( size, 0, hkVector4::getZero() );
		m_world->addEntity( rb );
		rb->removeReference();
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(   0.0f, -5.0f, 5.0f );
		hkVector4 to  (   0.0f,   0.0f, 5.0f );
		hkVector4 up  (   0.0f,   0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up );

		hkVector4 ldir; ldir.set( 0, 0.5f, -1.0f );
		setSoleDirectionLight(m_env, ldir, 0xffffffff );

		setupGraphics();
	}


	// Create the easy-limits action
	//if (g_variants[m_env->m_variantId].m_recoverLimitsGently)
	if (g_variants[m_env->m_variantId].m_demoMode & RECOVER_LIMITS)
	{
		const hkArray<hkpEntity*>& entities = (const hkArray<hkpEntity*>&)m_ragdolls.back()->getRigidBodies();
		m_easeConstraintsAction = new hkpEaseConstraintsAction(entities);
	}
	else
	{
		m_easeConstraintsAction = HK_NULL;
	}

	m_simTime = 0;
	m_constraintRestorationUnderway = false;


	hkpSolverInfo* s = m_world->getSolverInfo();
	g_originalTau = s->m_tau;
	g_originalDamping = s->m_damping;

	m_world->unlock();

	m_settingUp = true;
}

RagdollPoseRecoveryDemo::~RagdollPoseRecoveryDemo()
{
	if (m_easeConstraintsAction)
	{
		m_easeConstraintsAction->removeReference();
	}

	m_world->lock();

	delete m_violatedConstraintListener;
	
	for( int i = 0; i < m_ragdolls.getSize(); i++ )
	{
		m_ragdolls[i]->removeReference();
	}	

	m_world->unlock();
}

hkDemo::Result RagdollPoseRecoveryDemo::stepDemo()
{
	m_doProjection = g_variants[m_env->m_variantId].m_projectConstraints;
	const bool recoverLimitsGently = g_variants[m_env->m_variantId].m_recoverLimitsGently;

	if (m_simTime == 0)
	{
		m_simTime += m_timestep;
		return DEMO_PAUSED;
	}

	m_simTime += m_timestep;

	const hkReal waitTillRecovery = 2.0f;
	const hkReal recoveryTime = recoverLimitsGently ? 2.0f : 0.001f;

	/* Update pose */

	m_world->lock();

	if (m_simTime < waitTillRecovery)
	{
		// invalidate limits
		if (g_variants[m_env->m_variantId].m_demoMode & RECOVER_LIMITS)
		{
			setupInvalidRagdollPose(m_timestep/waitTillRecovery);
		}

		if (g_variants[m_env->m_variantId].m_demoMode & RECOVER_POSITIONS)
		{
			m_world->getSolverInfo()->setTauAndDamping(0.000001f, 0.000001f);
			// corrupt positions
			hkpPhysicsSystem* ragdoll = m_ragdolls.back();
			hkVector4 dstPos = ragdoll->getRigidBodies()[0]->getPosition();
			for (int bi = 0; bi < ragdoll->getRigidBodies().getSize(); bi++)
			{
				hkVector4 currPos = ragdoll->getRigidBodies()[bi]->getPosition();
				hkReal t = m_timestep / (waitTillRecovery-m_simTime);
				hkVector4 newPos;
				newPos.setMul4(1-t, currPos);
				newPos.addMul4(t, dstPos);
				ragdoll->getRigidBodies()[bi]->setPosition(newPos);
			}
		}
	}
	else
	{
		if (m_settingUp == true)
		{
			m_settingUp = false;
			m_world->unlock();

			return DEMO_PAUSED;
		}
		if (g_variants[m_env->m_variantId].m_demoMode & RECOVER_POSITIONS)
		{
			m_world->getSolverInfo()->setTauAndDamping(g_originalTau, g_originalDamping);
		}
	}

	//}

	m_world->unlock();

	/* Limits ease & recovery */

	if (/*recoverLimitsGently || */(g_variants[m_env->m_variantId].m_demoMode & RECOVER_LIMITS))
	{
		if (m_simTime < waitTillRecovery)
		{
			// NOTE: you may call loosenConstraints at any time,
			// even after the action has been added to the world.
			m_easeConstraintsAction->loosenConstraints();
		}
		else if (m_easeConstraintsAction->getWorld() == HK_NULL)
		{
			if (m_constraintRestorationUnderway)
			{
				// action has completed
				//m_constraintRestorationUnderway = false;
			}
			else
			{
				// NOTE: you may call restoreConstraints at any time,
				// even after the action has been added to the world.
				m_easeConstraintsAction->restoreConstraints( recoveryTime );

				m_world->markForWrite();
				m_world->addAction( m_easeConstraintsAction );
				m_world->unmarkForWrite();

				m_constraintRestorationUnderway = true;
			}
		}
	}

	if (recoverLimitsGently)
	{
		int textYPos = (int)getWindowHeight()-24-24-24;

		if (m_constraintRestorationUnderway)
		{
			char buffer[100];
			hkString::sprintf(buffer,  "Restoring constraint limits %3.0f%%.", hkMath::min2(hkReal(100), (m_simTime-waitTillRecovery)/recoveryTime*hkReal(100)) );
			m_env->m_textDisplay->outputText(buffer, 8, textYPos);
		}
		else
		{
			m_env->m_textDisplay->outputText("Limits loosened to current configuration.", 8, textYPos);
		}
	}





	/* Step										*/ 

	hkDemo::Result	result=hkDefaultPhysics2012Demo::stepDemo();
	
	/* Project constraints if required			*/ 
	m_world->lock();
	const int	numViolations	=	m_violatedConstraintListener->m_constraintsToProject.getSize();
	if(m_doProjection && numViolations && m_simTime >= waitTillRecovery)
	{		
		m_violatedConstraintListener->projectAndClear();
		hkpWorldCallbackUtil::firePostSimulationCallback(m_world);
	}
	else
	{
		m_violatedConstraintListener->clear();
	}
	m_world->unlock();	
	
	/* Display info text						*/ 
	hkStringBuf	statusText;
	if(m_doProjection)
		statusText.printf("Projection [ON] , %d constraints projected",numViolations);
	else
		statusText.printf("Projection [OFF]");
	m_env->m_textDisplay->outputText(statusText.cString(),8,(int)getWindowHeight()-24);


	//// Debug: identify bones & constraints
	//for (int ci = 0; ci < m_ragdolls.back()->getConstraints().getSize(); ci++)
	//{
	//	hkpConstraintInstance* c = m_ragdolls.back()->getConstraints()[ci];
	//	hkVector4 pivot = hkpConstraintPivotsUtil::getPivotA(c->getData());
	//	pivot.setTransformedPos(static_cast<hkpRigidBody*>(c->getEntityA())->getTransform(), pivot);
	//	char cIdxStr[10]; hkString::sprintf(cIdxStr, "%d", ci);
	//	m_env->m_textDisplay->outputText3D(cIdxStr, pivot(0), pivot(1), pivot(2), 0xff0000ff);
	//}

	//for (int bi = 0; bi < m_ragdolls.back()->getRigidBodies().getSize(); bi++)
	//{
	//	hkpRigidBody* b = m_ragdolls.back()->getRigidBodies()[bi];
	//	char bIdxStr[10]; hkString::sprintf(bIdxStr, "%d", bi);
	//	hkVector4 pos = b->getPosition();
	//	m_env->m_textDisplay->outputText3D(bIdxStr, pos(0), pos(1), pos(2), 0xff00ff00);
	//}


	return result;
}

static hkVector4 getConstraintPivot(const hkpConstraintInstance* c)
{
	hkVector4 pivot = hkpConstraintDataUtils::getPivotA(c->getData());
	pivot.setTransformedPos(static_cast<hkpRigidBody*>(c->getEntityA())->getTransform(), pivot);
	return pivot;
}

static void rotateBoneAroundPoint(hkpRigidBody* body, hkVector4& pivot, hkQuaternion& rotation)
{
	hkTransform a, b, c;
	a.setIdentity(); b.setIdentity(); c.setIdentity();
	hkVector4 negPivot; negPivot.setNeg3(pivot);
	a.setTranslation(pivot);
	b.setRotation(rotation);
	c.setTranslation(negPivot);

	hkTransform t; t.setMul(b,c); t.setMul(a, t);

	hkTransform bodyT; bodyT.setMul(t, body->getTransform()); 
	body->setTransform(bodyT);
}

void RagdollPoseRecoveryDemo::setupInvalidRagdollPose(hkReal fraction)
{
	const hkArray<hkpConstraintInstance*>& constraints = m_ragdolls.back()->getConstraints();
	const hkArray<hkpRigidBody*>& bodies = m_ragdolls.back()->getRigidBodies();

	// left arm
	{
		hkVector4 pivot = getConstraintPivot(constraints[4]); 
		hkQuaternion quat; quat.setAxisAngle(hkTransform::getIdentity().getColumn(2), 80 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[7], pivot, quat);
		rotateBoneAroundPoint(bodies[8], pivot, quat);
		quat.setAxisAngle(hkTransform::getIdentity().getColumn(1), -180 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[7], pivot, quat); 
		rotateBoneAroundPoint(bodies[8], pivot, quat);
	}

	// right arm
	{
		hkVector4 pivot = getConstraintPivot(constraints[5]); 
		hkQuaternion quat; quat.setAxisAngle(hkTransform::getIdentity().getColumn(2), -80 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[9], pivot, quat);
		rotateBoneAroundPoint(bodies[10], pivot, quat);
		quat.setAxisAngle(hkTransform::getIdentity().getColumn(1), 180 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[9], pivot, quat); 
		rotateBoneAroundPoint(bodies[10], pivot, quat);
	}

	// left leg
	{
		hkVector4 pivot = getConstraintPivot(constraints[2]); 
		hkQuaternion quat; quat.setAxisAngle(hkTransform::getIdentity().getColumn(0), 80 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[3], pivot, quat); 
		rotateBoneAroundPoint(bodies[4], pivot, quat);
		quat.setAxisAngle(hkTransform::getIdentity().getColumn(1), -180 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[3], pivot, quat); 
		rotateBoneAroundPoint(bodies[4], pivot, quat);
	}

	// right leg
	{
		hkVector4 pivot = getConstraintPivot(constraints[3]);
		hkQuaternion quat; quat.setAxisAngle(hkTransform::getIdentity().getColumn(0), 80 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[5], pivot, quat); 
		rotateBoneAroundPoint(bodies[6], pivot, quat);
		quat.setAxisAngle(hkTransform::getIdentity().getColumn(1), 180 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[5], pivot, quat); 
		rotateBoneAroundPoint(bodies[6], pivot, quat);
	}

	// Do head
	{
		hkVector4 pivot = bodies[2]->getPosition();
		hkQuaternion quat; quat.setAxisAngle(hkTransform::getIdentity().getColumn(2), 180 * HK_REAL_DEG_TO_RAD * fraction );
		rotateBoneAroundPoint(bodies[2], pivot, quat); 
	}

}

static void addNosePalmsAndFeetToRagdoll(hkpPhysicsSystem* ragdoll)
{
	// do head
	{
		hkpRigidBody* head = ragdoll->getRigidBodies()[2];
		hkpSphereShape* headShape = (hkpSphereShape*)head->getCollidable()->getShape();
		hkpSphereShape* nose = new hkpSphereShape(headShape->getRadius() / 5.0f);
		hkpTransformShape* noseT = new hkpTransformShape(nose, hkTransform(hkQuaternion::getIdentity(), hkVector4(0.0f, -headShape->getRadius(), 0.0f)));
		hkpShape* array[] = { headShape, noseT };
		hkpListShape* list = new hkpListShape(array, 2);
		head->setShape(list);
		nose->removeReference();
		noseT->removeReference();
		list->removeReference();
	}

	// do limbs
	int bodyIndices[] = {4, 6, 8, 10};
	for (int i = 0; i<4; i++)
	{
		hkpRigidBody* limb = ragdoll->getRigidBodies()[bodyIndices[i]];
		hkpCapsuleShape* limbShape = (hkpCapsuleShape*)limb->getCollidable()->getShape();
		hkpSphereShape* detail = new hkpSphereShape(limbShape->getRadius());
		hkVector4 translate = limbShape->getVertex(i!=2?1:0); translate(1) -= limbShape->getRadius();
		hkpTransformShape* detailT = new hkpTransformShape(detail, hkTransform(hkQuaternion::getIdentity(), translate));
		hkpShape* array[] = { limbShape, detailT };
		hkpListShape* list = new hkpListShape(array, 2);
		limb->setShape(list);
		detail->removeReference();
		detailT->removeReference();
		list->removeReference();
	}
}




static const char helpString[] =	"Using constraint projections to fix large errors in articulated bodies.\r\n"
									"Press 'P' to switch projection on/off\r\n";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( RagdollPoseRecoveryDemo, HK_DEMO_TYPE_PHYSICS_2012 , RagdollPoseRecoveryVariant, g_variants, helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
