/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Constraints/ClonedConstraint/ClonedConstraintDemo.h>


// We will need these collision agents
#include <Physics2012/Collide/Agent/ConvexAgent/BoxBox/hkpBoxBoxAgent.h>
#include <Physics2012/Collide/Agent/ConvexAgent/SphereSphere/hkpSphereSphereAgent.h>

// We need to create a constraint
#include <Physics/Constraint/Data/Hinge/hkpHingeConstraintData.h>
#include <Physics/Constraint/Data/StiffSpring/hkpStiffSpringConstraintData.h>

// We want to turn on constraint viewers for this demo
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Physics2012/Dynamics/Constraint/Util/hkpConstraintPivotsUtil.h>

hkpRigidBody* ClonedConstraintDemo::createSphereMoveableBody( const hkVector4& position, hkpSphereShape* sphereShape )
{
	hkpRigidBody* moveableBody;

	hkpRigidBodyCinfo info;
	info.m_position = position;
	info.m_shape = sphereShape;

	// Compute the inertia tensor
	hkMassProperties massProperties;
	info.m_mass = 20.0f;
	hkpInertiaTensorComputer::computeSphereVolumeMassProperties( 0.5f, info.m_mass, massProperties );
	info.m_inertiaTensor = massProperties.m_inertiaTensor;
	info.m_centerOfMass = massProperties.m_centerOfMass;	
	info.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;

	moveableBody = new hkpRigidBody( info );
	m_world->addEntity( moveableBody );
	moveableBody->removeReference();

	return moveableBody;
}

hkpConstraintInstance* ClonedConstraintDemo::createStiffSpringConstraint( const hkVector4& position, const hkVector4&pivotOffset, hkpRigidBody* bodyA, hkpRigidBody* bodyB )
{
	hkpConstraintInstance* constraint;
	hkpStiffSpringConstraintData* constraintData = new hkpStiffSpringConstraintData();

	if ( !bodyB )
	{
		hkVector4 pivot;
		pivot.setAdd4( position, pivotOffset );

		constraintData->setInWorldSpace( bodyA->getTransform(), hkTransform::getIdentity(), bodyA->getPosition(), pivot );
	}
	else
	{
		constraintData->setInWorldSpace( bodyA->getTransform(), bodyB->getTransform(), bodyA->getPosition(), bodyB->getPosition() );
	}	

	constraint = new hkpConstraintInstance( bodyA, bodyB, constraintData );

	constraintData->removeReference();

	return constraint;
}

void ClonedConstraintDemo::setupConstraints( hkDemoEnvironment* env )
{
	const hkVector4 pivotOffset( -3.0f, 0.0f, 0.0f );
	const hkVector4 cloneOffset( -7.0f, 0.0f, 0.0f );
	hkTransform cloneTransform;
	cloneTransform.setIdentity();
	cloneTransform.setTranslation( cloneOffset );

	const hkReal radius = 0.5f;
	hkpSphereShape* sphereShape;
	{
		sphereShape = new hkpSphereShape( radius );
	}

	//
	// Set body B to NULL and perform deep clone
	//
	hkpRigidBody* moveableBody;
	{
		hkVector4 position( 3.0f, 0.0f, 0.0f );
		moveableBody = createSphereMoveableBody( position, sphereShape );
	}
	sphereShape->removeReference();

	hkpConstraintInstance* constraint = createStiffSpringConstraint( moveableBody->getPosition(), pivotOffset, moveableBody, HK_NULL );
	m_world->addConstraint( constraint); 
	constraint->removeReference();

	hkpRigidBody* moveableBody2;
	{
		hkVector4 newPosition;
		newPosition.setAdd4( moveableBody->getPosition(), cloneOffset );
		moveableBody2 = createSphereMoveableBody( newPosition, sphereShape );
	}

	hkpConstraintInstance* constraint2 = constraint->clone( moveableBody2, HK_NULL, hkpConstraintInstance::CLONE_SHALLOW_IF_NOT_CONSTRAINED_TO_WORLD );
	{
		m_world->addConstraint( constraint2 );
		constraint2->removeReference();

		// This constraint must be explicitly transformed as it is constrained to the world
		constraint2->transform( cloneTransform );
	}

	{
		const unsigned sphereColors = hkColor::rgbFromChars(0, 255, 0, 255 );

		hkUlong originalShapeId = (hkUlong)moveableBody->getCollidable();
		hkUlong clonedShapeId = (hkUlong)moveableBody2->getCollidable();
		HK_SET_OBJECT_COLOR( originalShapeId, sphereColors );
		HK_SET_OBJECT_COLOR( clonedShapeId, sphereColors );
	}

	//
	// Set body B to NULL and perform shallow clone
	//
	hkpRigidBody* moveableBody3;
	{
		hkVector4 position(3.0f, 0.0f, -4.0f);
		moveableBody3 = createSphereMoveableBody( position, sphereShape );
	}

	hkpConstraintInstance* constraint3 = createStiffSpringConstraint( moveableBody3->getPosition(), pivotOffset, moveableBody3, HK_NULL );
	m_world->addConstraint( constraint3 ); 
	constraint3->removeReference();

	hkpRigidBody* moveableBody4 = moveableBody3->clone();
	{
		hkVector4 newPosition;
		newPosition.setAdd4( moveableBody4->getPosition(), cloneOffset );
		moveableBody4->setPosition( newPosition );
		m_world->addEntity( moveableBody4 );
		moveableBody4->removeReference();
	}

	hkpConstraintInstance* constraint4 = constraint3->clone( moveableBody4, HK_NULL, hkpConstraintInstance::CLONE_FORCE_SHALLOW );
	{
		m_world->addConstraint( constraint4 );
		constraint4->removeReference();

		// In this case transforming the cloned constraint will also move the the original constraint
		constraint4->transform( cloneTransform );
	}

	{
		const unsigned sphereColors = hkColor::rgbFromChars(0, 0, 255, 255 );

		hkUlong originalShapeId = (hkUlong)moveableBody3->getCollidable();
		hkUlong clonedShapeId = (hkUlong)moveableBody4->getCollidable();
		HK_SET_OBJECT_COLOR( originalShapeId, sphereColors );
		HK_SET_OBJECT_COLOR( clonedShapeId, sphereColors );
	}

	//
	// Set body B to rigid body with shape
	//
	hkpRigidBody* moveableBody5;
	{
		hkVector4 position( 3.0f, 0.0f, -8.0f );
		moveableBody5 = createSphereMoveableBody( position, sphereShape );
	}

	hkpRigidBody* fixedBody;
	{
		hkpRigidBodyCinfo info;
		info.m_position.set(0.0f, 0.0f, -8.0f);
		info.m_shape = sphereShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		fixedBody = new hkpRigidBody( info );
		moveableBody5->getWorld()->addEntity( fixedBody );
		fixedBody->removeReference();
	}

	hkpConstraintInstance* constraint5 = createStiffSpringConstraint( moveableBody5->getPosition(), pivotOffset, moveableBody5, fixedBody );
	m_world->addConstraint( constraint5 ); 
	constraint5->removeReference();

	hkpRigidBody* moveableBody6;
	{
		hkVector4 newPosition;
		newPosition.setAdd4( moveableBody5->getPosition(), cloneOffset );
		moveableBody6 = createSphereMoveableBody( newPosition, sphereShape );
	}

	hkpRigidBody* fixedBody2;
	{
		hkpRigidBodyCinfo info;
		hkVector4 newPosition;
		info.m_position.setAdd4( fixedBody->getPosition(), cloneOffset );
		info.m_shape = sphereShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		fixedBody2 = new hkpRigidBody( info );
		moveableBody6->getWorld()->addEntity( fixedBody2 );
		fixedBody2->removeReference();
	}

	hkpConstraintInstance* constraint6 = constraint5->clone( moveableBody6, fixedBody2, hkpConstraintInstance::CLONE_SHALLOW_IF_NOT_CONSTRAINED_TO_WORLD );
	{
		m_world->addConstraint( constraint6 );
		constraint6->removeReference();
	}

	{
		const unsigned sphereColors = hkColor::rgbFromChars(255, 255, 0, 255 );

		hkUlong originalShapeAId = (hkUlong)moveableBody5->getCollidable();
		hkUlong originalShapeBId = (hkUlong)fixedBody->getCollidable();
		hkUlong clonedShapeAId = (hkUlong)moveableBody6->getCollidable();
		hkUlong clonedShapeBId = (hkUlong)fixedBody2->getCollidable();
		HK_SET_OBJECT_COLOR( originalShapeAId, sphereColors );
		HK_SET_OBJECT_COLOR( originalShapeBId, sphereColors );
		HK_SET_OBJECT_COLOR( clonedShapeAId, sphereColors );
		HK_SET_OBJECT_COLOR( clonedShapeBId, sphereColors );
	}

	//
	// Set body B to rigid body with no shape
	//
	hkpRigidBody* moveableBody7;
	{
		hkVector4 position( 3.0f, 0.0f, -12.0f );
		moveableBody7 = createSphereMoveableBody( position, sphereShape );
	}

	hkpRigidBody* dummyBody;
	{
		hkpRigidBodyCinfo info;
		info.m_position.set(0.0f, 0.0f, -12.0f);
		info.m_shape = HK_NULL;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		dummyBody = new hkpRigidBody( info );
		moveableBody7->getWorld()->addEntity( dummyBody );
		dummyBody->removeReference();
	}

	hkpConstraintInstance* constraint7 = createStiffSpringConstraint( moveableBody7->getPosition(), pivotOffset, moveableBody7, dummyBody );
	m_world->addConstraint( constraint7 ); 
	constraint7->removeReference();

	hkpRigidBody* moveableBody8;
	{
		hkVector4 newPosition;
		newPosition.setAdd4( moveableBody7->getPosition(), cloneOffset );
		moveableBody8 = createSphereMoveableBody( newPosition, sphereShape );
	}

	hkpRigidBody* dummyBody2;
	{
		hkpRigidBodyCinfo info;
		hkVector4 newPosition;
		info.m_position.setAdd4( dummyBody->getPosition(), cloneOffset );
		info.m_shape = HK_NULL;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		dummyBody2 = new hkpRigidBody( info );
		moveableBody8->getWorld()->addEntity( dummyBody2 );
		dummyBody2->removeReference();
	}

	hkpConstraintInstance* constraint8 = constraint7->clone( moveableBody8, dummyBody2, hkpConstraintInstance::CLONE_SHALLOW_IF_NOT_CONSTRAINED_TO_WORLD );
	{
		m_world->addConstraint( constraint8 );
		constraint8->removeReference();
	}

	{
		const unsigned sphereColors = hkColor::rgbFromChars(255, 0, 255, 255 );

		hkUlong originalShapeId = (hkUlong)moveableBody7->getCollidable();
		hkUlong clonedShapeId = (hkUlong)moveableBody8->getCollidable();
		HK_SET_OBJECT_COLOR( originalShapeId, sphereColors );
		HK_SET_OBJECT_COLOR( clonedShapeId, sphereColors );
	}

	m_world->unlock();
}

ClonedConstraintDemo::ClonedConstraintDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{
	// Setup the camera
	{
		const hkVector4 from(0.0f, 10.0f, 15.0f);
		const hkVector4 to(0.0f, 0.0f, 0.0f);
		const hkVector4 up(0.0f, 1.0f, -0.3f);
		setupDefaultCameras( env, from, to, up );
	}

	// Create the world
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 100.0f );
		m_world = new hkpWorld( info );
		m_world->lock();

		// Register the single agent required (a hkpSphereSphereAgent)
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

		// Create constraint viewer
		m_debugViewerNames.pushBack( hkpConstraintViewer::getName() );
		hkpConstraintViewer::m_scale = 1.0f;

		setupGraphics();
	}

	setupConstraints( env );
}

ClonedConstraintDemo::~ClonedConstraintDemo()
{

}

hkDemo::Result ClonedConstraintDemo::stepDemo()
{
	return hkDefaultPhysics2012Demo::stepDemo();
}



static const char helpString[] =	"This demo shows how to clone and transform constraints.\n"
									"Each pair of stiff spring constraints consists of an original constraint and a cloned constraint.\n"
									"Each cloned constraint is positioned to the left.\n"
									"The green spheres are constrained to the world and cloned using a deep copy.\n"
									"The blue spheres are constrained to the world and cloned using a shallow copy.\n"
									"The yellow spheres are constrained by a constraint between 2 rigid bodies.\n"
									"The purple spheres are constrained to a dummy fixed rigid body.";

HK_DECLARE_DEMO(ClonedConstraintDemo, HK_DEMO_TYPE_PHYSICS_2012, "Cloning and transforming constraints", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
