/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/Voxel/VoxelShapeDemo.h>
#include <Demos/Physics2012/Api/Collide/Shapes/Voxel/UserVoxelShapeCollection.h>
#include <Demos/Physics2012/Api/Collide/Shapes/Voxel/UserVoxelBvTreeShape.h>
#include <Demos/DemoCommon/Utilities/MarchingCubes/MarchingCubes.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Common/Visualize/hkDebugDisplay.h>

#include <Physics2012/Dynamics/World/Util/hkpWorldCallbackUtil.h>
#include <Physics2012/Collide/BroadPhase/hkpBroadPhase.h>
#include <Physics2012/Collide/BroadPhase/hkpBroadPhaseHandlePair.h>

//
// This example implements a dynamic voxelized world and demonstrates its use in
// collision detection as a custom shape in the Havok Physics world.  The voxel
// world can be modified at any time without a significant Havok Physics performance
// impact.
//
// This is accomplished in 3 parts:
//
// 1. The VoxelData manages the voxel data and voxel triangulation.  While
//    this sample uses Marching Cubes, it is not specific to any Havok Physics
//    data and is meant to be physics engine agnostic.
//
// 2. The UserVoxelShapeCollection is a hkpShapeCollection that can be queried
//    for its child shapes.  Each child shape is a voxel, represented by an
//    hkpExtendedMeshShape containing triangles.  It does not store the triangles
//    or hkpExtendedMeshShapes itself; rather the child shapes are generated on the fly
//    using the triangle information stored by the VoxelData.
//
// 3. The UserVoxelBvTreeShape implements the hkpBvTreeShape (mid-phase) interface.
//    Since the voxel world is a single rigid body, the mid-phase is required to speed
//    up query performance.  Otherwise, each colliding rigid body would naively check
//    all triangles in the voxel shape.  Narrow phase collision detection is only
//    performed between each dynamic rigid body and the triangles contained in the
//    voxel(s) it overlaps.
//
// Notes: Performance spikes in this demo are likely due to re-adding the debug draw
//        geometry to the Shape Display Viewer (timer name "setShapeCb").
//
//		  Welding is not supported in this example.
//

static const int kRandomSeed = 12345;

VoxelShapeDemo::VoxelShapeDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
,	m_voxelData(HK_NULL)
,	m_voxelBody(HK_NULL)
,   m_rnd(kRandomSeed)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(90.0f, 90.0f, 70.0f);
		hkVector4 to  (0.0f, 0.0f,  0.0f);
		hkVector4 up  (0.0f, 1.0f,  0.0f);
		setupDefaultCameras( env, from, to, up );
	}


	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM); 
		info.setBroadPhaseWorldSize( 500.0f );

		m_world = new hkpWorld( info );
		m_world->lock();

		setupGraphics();
	}

	//
	// Register the agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Initialize voxel data
	//
	m_voxelData = new MarchingCubes( 50.f );
	{
		const int numParticles = 50;

		// Marching cubes requires this to be uniform
		hkVector4 voxelSize;
		voxelSize.set( 1.25f, 1.25f, 1.25f );

		// Size of the terrain in meters
		hkVector4 extents;
		extents.set( 50.f, 50.f, 50.f );

		m_voxelData->setExtentsAndVoxelSize( extents, voxelSize );

		// Add some influences
		hkArray<Particle> particles( numParticles );
		{
			for (int i = 0; i < numParticles; ++i)
			{
				// Find random position within the voxel shape
				m_rnd.getRandomVectorRange( hkVector4::getZero(), extents, particles[i].m_position );

				// Give it a random value
				particles[i].m_value = m_rnd.getRandRange( 15.f, 35.f );
			}
		}

		// Compute the initial data set
		// (similar to electrostatic potential)
		{
			hkReal* voxelData = m_voxelData->getData().begin();

			int gridX, gridY, gridZ;
			m_voxelData->getGridSize( gridX, gridY, gridZ );

			for (int x = 0; x < gridX; ++x)
			{
				for (int y = 0; y < gridY; ++y)
				{
					for (int z = 0; z < gridZ; ++z)
					{
						hkReal& data = *voxelData;
						data = 0.f;
						for (int i = 0; i < numParticles; ++i)
						{
							Particle& particle = particles[i];

							hkVector4 point;
							m_voxelData->gridPointToPosition( x, y, z, point );

							// Use a 1/r rule
							hkReal distSquared = point.distanceToSquared3( particle.m_position );

							// Cap max value
							distSquared = hkMath::max2( distSquared, 0.01f );

							data += particle.m_value * hkMath::sqrtInverse( distSquared );
						}
						++voxelData;
					}
				}
			}
		}

		// Compute all triangles
		m_voxelData->triangulate();
	}

	{
		// Shape collection; contains all the voxel triangles
		UserVoxelShapeCollection* voxelCollection = new UserVoxelShapeCollection( m_voxelData );

		// Custom voxel bounding volume
		UserVoxelBvTreeShape* voxelShape = new UserVoxelBvTreeShape( voxelCollection );
		voxelCollection->removeReference();

		// Create rigid body
		hkpRigidBodyCinfo cinfo;
		{
			cinfo.m_position.setZero();
			cinfo.m_rotation.setIdentity();

			cinfo.m_shape = voxelShape;
			cinfo.m_motionType = hkpMotion::MOTION_FIXED;
		}
		m_voxelBody = new hkpRigidBody( cinfo );

		m_world->addEntity( m_voxelBody );
		voxelShape->removeReference();

		// Sanity test indexToVoxel produces same result as voxelToIndex
		hkpShapeKey key = voxelCollection->getFirstKey();
		while (key != HK_INVALID_SHAPE_KEY)
		{
			int x, y, z;
			m_voxelData->indexToVoxel( key, x, y, z );

			HK_ON_DEBUG( int index = m_voxelData->voxelToIndex( x, y, z ); )
			HK_ASSERT( 0x0, key == (unsigned long)index );
			key = voxelCollection->getNextKey( key );
		}
	}

	//
	// Add some random bodies to the world
	//
	{
		const int numBodies = 50;
		hkVector4 extents;
		{
			int x, y, z;
			m_voxelData->getGridSize( x, y, z );
			extents.set( x * 1.f, y * 1.f, z * 1.f );
			extents.mul( m_voxelData->getVoxelSize() );
		}

		for (int i = 0; i < numBodies; ++i)
		{
			hkVector4 position;
			m_rnd.getRandomVectorRange( hkVector4::getZero(), extents, position );
			position(1) = extents(1) + 10.f;

			hkpRigidBody* rigidBody = GameUtils::createRandomConvexGeometric( .5f, 1.f, position, 10, &m_rnd );
			m_world->addEntity( rigidBody )->removeReference();		
			HK_SET_OBJECT_COLOR( (hkUlong)rigidBody->getCollidable(), hkColor::rgbFromChars(255, 170, 0) );
		}
	}

	m_world->unlock();

}

static void _calcVelocityToTarget(const hkVector4& position, const hkVector4& target, const hkVector4& gravity, hkReal speedR, hkVector4& velocity)
{
	hkVector4 dist; dist.setSub(target, position); 
	dist(3) = 0.f;
	hkSimdReal distLen = dist.length<3>();
	const hkSimdReal speed = hkSimdReal::fromFloat(speedR);
	if (distLen > hkSimdReal_Eps)
	{
		hkSimdReal time = distLen / speed;
		hkVector4 extraVelocity; extraVelocity.setMul(hkSimdReal_Inv2 * (-time), gravity);
		hkSimdReal extraVelLen = extraVelocity.length<3>();
		if (extraVelLen > speed)
		{
			// clip extra velocity .. to 45degree deflection
			extraVelocity.mul(speed / extraVelLen);
		}
		velocity.setAddMul(extraVelocity, dist, speed / distLen);
	}
	else
	{
		// Ignore:
		velocity = dist;
	}
	velocity(3) = 0.f;
}

hkDemo::Result VoxelShapeDemo::stepDemo()
{
	m_world->lock();

	{
		// See what's under the mouse
		const hkgMouse& mouse = m_env->m_window->getMouse();
		const hkgPad*   pad   = m_env->m_gamePad;

		hkgViewportPickData pickData;
		if(m_env->m_displayWorld && m_env->m_window->getCurrentViewport()->pick(mouse.getPosX(), mouse.getPosY(), m_env->m_displayWorld, pickData))
		{
			hkVector4 hitPos; hitPos.load3( pickData.getClosestObjectWorldPos() );
			hkVector4 normal; normal.load3( pickData.getClosestObjectWorldNormal() );

			// Fire a convex hull at it?
			if (pad->wasButtonPressed(HKG_PAD_BUTTON_1))
			{
				const hkReal speed = 50.f;

				hkVector4 position; position.setZero4();
				hkVector4 velocity; velocity.setZero4();
				{
					hkgCamera* camera = m_env->m_window->getCurrentViewport()->getCamera();
					camera->getFrom( position );

					_calcVelocityToTarget( position, hitPos, m_world->getGravity(), speed, velocity );
				}
				
				// Create convex hull and launch it
				hkpRigidBody* rigidBody = GameUtils::createRandomConvexGeometric( .5f, 1.f, position, 10, &m_rnd );
				rigidBody->setQualityType(HK_COLLIDABLE_QUALITY_BULLET);
				rigidBody->setLinearVelocity( velocity );
				
				m_world->addEntity( rigidBody )->removeReference();		
				HK_SET_OBJECT_COLOR( (hkUlong)rigidBody->getCollidable(), hkColor::rgbFromChars(255, 170, 0) );
			}
			
			// Deform the terrain
			{
				// Rate of growth dependent on timestep.  60 Hz assumed.
				hkReal deformationAmount = m_timestep * 60.f;

				// 1 to grow larger, -1 to grow smaller
				hkReal direction = 0.f;

				if (pad->isButtonPressed(HKG_PAD_BUTTON_2))
				{
					direction = 1.f;
				}
				else if (pad->isButtonPressed(HKG_PAD_BUTTON_3))
				{
					direction = -1.f;
				}
					
				if (direction != 0.f)
				{
					deformationAmount *= direction;

					// Modify the voxel data and retriangulate
					m_voxelData->modifyAtPoint( hitPos, deformationAmount );

					// Force shape display viewer to rebuild shape (expensive!)
					hkpWorldCallbackUtil::fireEntityShapeSet( m_world, m_voxelBody );

					//
					// Recollide nearby bodies
					//
					
					// Perform broadphase query to determine which objects have potentially been affected
					hkArray<hkpEntity*> affectedEntities;
					hkArray<hkpBroadPhaseHandlePair> affectedObjects;
					{
						hkAabb aabb;
						aabb.m_min.setSub( hitPos, m_voxelData->getVoxelSize() );
						aabb.m_max.setAdd( hitPos, m_voxelData->getVoxelSize() );

						// Ask broadphase to collect objects overlapping this voxel
						m_world->getBroadPhase()->querySingleAabb( aabb, affectedObjects );
					}

					for (int i = 0; i < affectedObjects.getSize(); i++)
					{
						hkpBroadPhaseHandlePair *pair = &affectedObjects[i];
						hkpCollidable* collidable = static_cast<hkpCollidable*>(static_cast<hkpTypedBroadPhaseHandle*>(pair->m_b)->getOwner());
						hkpRigidBody* rigidBody = hkpGetRigidBody(collidable);

						// do not process the voxel terrain
						if ( !rigidBody || rigidBody == m_voxelBody )
						{
							continue;
						}

						//	We need to wake up all objects touching this element
						rigidBody->activate();

						affectedEntities.pushBack( rigidBody );
					}

					// Recollide the entities which have been modified.
					// Terrain surface has changed so need to re-evaluate
					// contact points here to avoid being a frame off.
					if (affectedEntities.getSize() > 0)
					{
						m_world->reintegrateAndRecollideEntities( affectedEntities.begin(), affectedEntities.getSize(), hkpWorld::RR_MODE_RECOLLIDE_NARROWPHASE );
					}
				}
			}
		}
	}

	m_world->unlock();
	return hkDefaultPhysics2012Demo::stepDemo();
}

VoxelShapeDemo::~VoxelShapeDemo()
{
	m_voxelData->removeReference();
	m_voxelBody->removeReference();
}


static const char helpString[] = \
"Use \x11 to fire a convex hull at the shape,\n"  \
"    \x12 to add to the terrain shape,\n"  \
"and \x13 to subtract from the terrain shape.";

HK_DECLARE_DEMO(VoxelShapeDemo, HK_DEMO_TYPE_PHYSICS_2012, "Voxelized terrain shape", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
