/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/StaticCompound/Filtering/StaticCompoundShapeFilteringDemo.h>

// Shapes
#include <Physics2012/Internal/Collide/StaticCompound/hkpStaticCompoundShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShapeCinfo.h>

#include <Demos/DemoCommon/Utilities/Geometry/GeometryMaker.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GeometryLoaderUtil.h>

#include <Common/Visualize/Shape/hkDisplayConvex.h>
#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>


static const char g_demoDetails[] =
"Demo showing the use of child shape and per-instance collision filter infos in hkpStaticCompoundShape.\n \
All trees are instances of a single mesh that has different collision layers set for the trunk and the top.\n \
Small tree instances are assigned to a different collision layer using per-instance collision filter infos and masks.\n \
This set up allows us to filter out collisions of big geometries with the relatively small tree instances while still \
being able to filter collisions with selected parts of the bigger instances.";

// Parameters
static const hkReal g_terrainSide = 60;
static const int g_terrainResolution = 16;
static const hkReal g_terrainHeight = 2;
static const int g_terrainNoiseRange = 4;
static const int g_numTrees = 32;
static const int g_numFallingBodies = 16;
static const int g_fallingBodiesFrequency = 25;
static const int g_layerColors[] = { hkColor::BROWN, 0xAA008000 };
static const char* g_treeFiles [] = { "Resources/Physics2012/Objects/BlockTreeTrunk.hkt", "Resources/Physics2012/Objects/BlockTreeTop.hkt" };


StaticCompoundShapeFilteringDemo::StaticCompoundShapeFilteringDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env), m_random(12), m_heightFieldNoise(5, g_terrainNoiseRange), 
	m_numFrames(0), m_fallingBodies(g_numFallingBodies, HK_NULL), m_lastBody(-1)
{	
	// Create physics world	
	{
		hkpWorldCinfo info;	
		info.m_gravity.set(0, -10, 10);
		info.setBroadPhaseWorldSize(256);

		// Required in order to use hkpStaticCompoundShape on SPU
		info.m_useCompoundSpuElf = true;

		m_world = new hkpWorld(info);
	}	

	m_world->lock();

	setupGraphics();

	// Setup a group collision filter disabling collisions of big dynamic bodies against:
	// small tree instances (1 vs 4), the top of big tree instances (1 vs 2) and small dynamic bodies (1 vs 3)
	m_groupFilter = hkRefNew<hkpGroupFilter>(new hkpGroupFilter());			
	m_groupFilter->disableCollisionsBetween(1, 4);	
	m_groupFilter->disableCollisionsBetween(1, 2);
	m_groupFilter->disableCollisionsBetween(1, 3);
	m_world->setCollisionFilter(m_groupFilter);

	// Register collision agents	
	hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());	

	// Set up the camera
	setupDefaultCameras(m_env, hkVector4(-30, 40, 60), hkVector4(0, 0, 0), hkVector4(0, 1, 0), 0.1f, 1000.0f);
		
	// Create a static compound shape
	int terrainInstanceId;
	hkRefPtr<hkpStaticCompoundShape> staticCompoundShape;
	{
		staticCompoundShape = hkRefNew<hkpStaticCompoundShape>(new hkpStaticCompoundShape());

		// Add terrain
		{		
			hkRefPtr<hkpShape> terrainShape = hkRefNew<hkpShape>(createTerrainShape());
			terrainInstanceId = staticCompoundShape->addInstance(terrainShape, hkQsTransform::getIdentity());
		}

		// Add trees		
		{		
			hkRefPtr<hkpShape> treeShape = hkRefNew<hkpShape>(createTreeShape());
			hkArray<hkQsTransform> transforms;			
			calculateTreeTransforms(transforms);

			for (int i = 0; i < transforms.getSize(); ++i)
			{					
				// Add tree instance
				const hkQsTransform& transform = transforms[i];
				const int instanceId = staticCompoundShape->addInstance(treeShape, transform);

				// Override shape collision filter info in small instances to prevent collision with big geometries
				const hkVector4 scale = transform.getScale();
				if (scale(1) < 0.5f)
				{
					const int instanceCollisionLayer = 4;
					staticCompoundShape->setInstanceFilterInfo(instanceId, hkpGroupFilter::calcFilterInfo(instanceCollisionLayer));
					staticCompoundShape->setInstanceFilterInfoMask(instanceId, 0);
				}						
			}
		}		

		// We are finished adding instances, so now we bake the shape into its final form
		staticCompoundShape->bake();
	}

	// Create a fixed body using the static compound shape
	{
		hkpRigidBodyCinfo info;
		info.m_shape = staticCompoundShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		hkpRigidBody* body = new hkpRigidBody(info);
		m_world->addEntity(body);
		replaceDisplayGeometry(body, terrainInstanceId);
		body->removeReference();
	}

	// Create shapes for the falling bodies
	m_bigSphere = hkRefNew<hkpSphereShape>(new hkpSphereShape(5.0f));	
	m_smallSphere = hkRefNew<hkpSphereShape>(new hkpSphereShape(1.5f));	

	m_world->unlock();
}

hkDemo::Result StaticCompoundShapeFilteringDemo::stepDemo()
{
	m_world->lock();
	
	if (m_numFrames % g_fallingBodiesFrequency == 0)
	{
		createFallingBody();
	}	

	m_world->unlock();

	// Display demo description
	{
		const int margin = 15;
		const int windowWidth = m_env->m_window->getWidth();
		const int windowHeight = m_env->m_window->getHeight();
		m_env->m_textDisplay->outputTextWithWrapping(g_demoDetails, margin, margin, windowWidth - 2 * margin, windowHeight - 2 * margin);
	}

	m_numFrames++;
	return hkDefaultPhysics2012Demo::stepDemo();
}

StaticCompoundShapeFilteringDemo::~StaticCompoundShapeFilteringDemo()
{
	m_world->markForWrite();
	
	for (int i = 0; i < m_fallingBodies.getSize(); ++i)
	{
		if (m_fallingBodies[i])	m_fallingBodies[i]->removeReference();
	}

	if (m_world)
	{
		m_world->removeReference();
		m_world = HK_NULL;
	}	
}


//////////////////////////////////////////////////
// Helper functions. No need to read past here.


namespace
{
	struct HeightFunction
	{
	public:
		HeightFunction(const hkPerlinNoise2d& noiseIn) : m_noise(noiseIn) {}

		HK_FORCE_INLINE hkReal operator()(hkReal x, hkReal z) const
		{			
			return g_terrainHeight * m_noise._smooth(x, z);
		}

		const hkPerlinNoise2d& m_noise;
	};
}

void StaticCompoundShapeFilteringDemo::createFallingBody()
{
	// Obtain next position in body array
	int bodyIdx = m_lastBody + 1;	
	if (bodyIdx == g_numFallingBodies)
	{
		bodyIdx = 0;
	}
	hkpRigidBody*& body = m_fallingBodies[bodyIdx];

	// Release current body if already in use
	if (body)
	{
		m_world->removeEntity(body);
		body->removeReference();
	}

	// Create new body
	hkpRigidBodyCinfo info;	
	info.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;			
	info.m_restitution = 0;	
	info.m_position.set(m_random.getRandRange(-0.25f * g_terrainSide, 0.25f * g_terrainSide), 0, -0.5f * g_terrainSide);
	if (bodyIdx % 4)
	{		
		info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(3);
		info.m_shape = m_smallSphere;
		info.m_position(1) = m_smallSphere->getRadius() + 0.1f;
		hkpInertiaTensorComputer::setShapeVolumeMassProperties(m_smallSphere, 1, info);
	}
	else
	{
		info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(1);
		info.m_shape = m_bigSphere;
		info.m_position(1) = m_bigSphere->getRadius() + 0.1f;
		hkpInertiaTensorComputer::setShapeVolumeMassProperties(m_bigSphere, 1, info);
	}
	const hkReal angle = HK_REAL_DEG_TO_RAD * (60 + 60 * m_random.getRandReal01());
	const hkReal maxSpeed = 20;
	info.m_linearVelocity.set(maxSpeed* hkMath::cos(angle), 0, maxSpeed * hkMath::sin(angle));
	body = new hkpRigidBody(info);
	m_world->addEntity(body);	

	// Set display color
	HK_SET_OBJECT_COLOR((hkUlong)body->getCollidable(), hkColor::SLATEGRAY);

	m_lastBody = bodyIdx;
}


void StaticCompoundShapeFilteringDemo::replaceDisplayGeometry(hkpRigidBody* compoundBody, int terrainInstanceId)
{	
	// Remove standard display geometry for the static compound shape
	const hkUlong displayGeometryId = (hkUlong)compoundBody->getCollidable();
	HK_REMOVE_GEOMETRY(displayGeometryId);

	// Iterate over all instances
	const hkpStaticCompoundShape* compoundShape = static_cast<const hkpStaticCompoundShape*>(compoundBody->getCollidable()->getShape());
	const hkArray<hkpStaticCompoundShape::Instance>& instances = compoundShape->getInstances();
	hkGeometry treeGeometries[2];	
	hkBool foundTree = false;
	for (int instanceId = 0; instanceId < instances.getSize(); ++instanceId)
	{
		const hkpBvCompressedMeshShape* mesh = static_cast<const hkpBvCompressedMeshShape*>(instances[instanceId].getShape());

		// Terrain instance
		if (instanceId == terrainInstanceId)
		{
			// Allocate geometry that will be deleted by the hkDisplayConvex object and fill it in
			hkGeometry* geometry = new hkGeometry();
			mesh->convertToGeometry(*geometry);			

			// Create display geometry and add it to the display
			hkDisplayConvex* displayGeometry = new hkDisplayConvex(geometry);
			m_displayGeometries.pushBack(hkRefNew<hkDisplayConvex>(displayGeometry));
			hkDebugDisplay::getInstance().addGeometry(displayGeometry, displayGeometryId + instanceId, 0, 0);			
			HK_SET_OBJECT_COLOR(displayGeometryId + instanceId, hkColor::GRAY);
		}

		// Tree instance
		else
		{
			// Obtain tree geometries
			if (!foundTree)
			{
				// Iterate over all mesh triangles adding them to the geometry corresponding to its collision layer
				hkpShapeBuffer shapeBuffer;
				for (hkpShapeKey key = mesh->getFirstKey(); key != HK_INVALID_SHAPE_KEY; key = mesh->getNextKey(key))
				{
					const hkpTriangleShape* triangle = static_cast<const hkpTriangleShape*>(mesh->getChildShape(key, shapeBuffer));
					hkUint32 collisionFilterInfo = mesh->getCollisionFilterInfo(key);

					// Add triangle to the right geometry
					const hkVector4* vertices = triangle->getVertices();
					hkGeometry& geometry = treeGeometries[collisionFilterInfo - 1];
					const int firstIndex = geometry.m_vertices.getSize();
					geometry.m_vertices.pushBack(vertices[0]);
					geometry.m_vertices.pushBack(vertices[1]);
					geometry.m_vertices.pushBack(vertices[2]);			
					geometry.m_triangles.expandOne().set(firstIndex, firstIndex + 1, firstIndex + 2);			
				}

				// Remove duplicate vertices
				hkGeometryUtils::weldVertices(treeGeometries[0], 0);
				hkGeometryUtils::weldVertices(treeGeometries[1], 0);				

				foundTree = true;
			}

			// Add transformed geometries for tree trunk and top with different colors according to the 
			// collision filter info			
			for (int j = 0; j < 2; ++j)
			{
				// Transform geometry with the instance transform
				hkGeometry* geometry = new hkGeometry(treeGeometries[j]);
				const hkQsTransform& transform = instances[instanceId].getTransform();
				hkMatrix4 transformAsMatrix;
				transformAsMatrix.set(transform);
				hkGeometryUtils::transformGeometry(transformAsMatrix, *geometry);

				// Calculate collision layer
				hkUint32 collisionLayer;
				{					
					collisionLayer = j + 1;
				}

				// Create display geometry and add it to the display
				hkDisplayConvex* displayGeometry = new hkDisplayConvex(geometry);
				m_displayGeometries.pushBack(hkRefNew<hkDisplayConvex>(displayGeometry));
				hkDebugDisplay::getInstance().addGeometry(displayGeometry, displayGeometryId + instanceId, 0, 0);			
				HK_SET_OBJECT_COLOR(displayGeometryId + instanceId, g_layerColors[collisionLayer - 1]);
			}
		}
	}	
}

namespace
{
	struct MyMeshConstructionInfo: public hkpDefaultBvCompressedMeshShapeCinfo
	{
		MyMeshConstructionInfo(const hkGeometry* geometry) : hkpDefaultBvCompressedMeshShapeCinfo(geometry) {}

		hkUint32 getTriangleCollisionFilterInfo(int triangleIndex) const
		{
			return hkpGroupFilter::calcFilterInfo(m_geometry->m_triangles[triangleIndex].m_material);
		}		
	};
}

void StaticCompoundShapeFilteringDemo::calculateTreeTransforms(hkArray<hkQsTransform>& transforms)
{
	const hkReal worldToFunction = (g_terrainNoiseRange - 1) / g_terrainSide;
	const hkReal side_2 = 0.5f * g_terrainSide;
	const hkReal side_4 = 0.25f * g_terrainSide;
	const hkReal side_8 = 0.125f * g_terrainSide;
	HeightFunction heightFunction(m_heightFieldNoise);

	hkVector4 positions[] = 
	{
		// Big trees
		hkVector4(-side_4, 0, -side_4), hkVector4(side_4, 0, -side_4), 
		hkVector4(-side_4, 0, side_4), hkVector4(side_4, 0, side_4),

		// Small trees
		hkVector4(0, 0, -side_4), hkVector4(-side_4, 0, 0), hkVector4(side_4, 0, 0), hkVector4(0, 0, side_4),
		hkVector4(-side_8, 0, -side_8), hkVector4(side_8, 0, -side_8), 
		hkVector4(-side_8, 0, side_8), hkVector4(side_8, 0, side_8)
	};

	transforms.setSize(12);	
	for (int i = 0; i < 12; ++i)
	{
		hkQsTransform& transform = transforms[i];
		transform.setIdentity();

		// Calculate position
		{				
			const hkReal x = positions[i](0);
			const hkReal z = positions[i](2);
			transform.setTranslation(hkVector4(x, heightFunction((x + side_2) * worldToFunction, (z + side_2) * worldToFunction), z));
		}		

		// Calculate rotation
		{							
			hkVector4 axis; 
			axis.set(m_random.getRandReal11() * 0.1f, 1, m_random.getRandReal11() * 0.1f); 
			axis.normalize<3>();				
			transform.setRotation(hkQuaternion(axis, m_random.getRandReal11() * 3));
		}

		// Calculate scale
		if (i < 4)
		{
			// Big tree
			transform.setScale(hkVector4(m_random.getRandRange(1.0f, 1.4f), m_random.getRandRange(0.8f, 1.2f), m_random.getRandRange(1.0f, 1.4f)));
		}
		else
		{
			// Small tree
			transform.setScale(hkVector4(m_random.getRandRange(0.2f, 0.3f), m_random.getRandRange(0.2f, 0.3f), m_random.getRandRange(0.2f, 0.3f)));			
		}
	}	
}

hkpShape* StaticCompoundShapeFilteringDemo::createTreeShape()
{
	// Setup tree shape
	hkpBvCompressedMeshShape* meshShape;
	{		
		// Load tree parts
		hkGeometry treeParts[2];
		for (int i = 0; i < 2; ++i)
		{
			// Open and load resource file
			hkResource* data = HK_NULL;
			hkStringBuf assetFile(HK_GET_DEMOS_ASSET_FILENAME(g_treeFiles[i])); 
			hkAssetManagementUtil::getFilePath(assetFile);
			data = hkSerializeUtil::load(assetFile.cString());
			HK_ASSERT2(0x01234390, data, "Could not load resource");	
			GeometryLoaderUtil::load(data, treeParts[i]);
			data->removeReference();

			// Set material with the collision layer
			hkArray<hkGeometry::Triangle>& triangles = treeParts[i].m_triangles;
			for (int j = 0; j < triangles.getSize(); ++j)
			{
				triangles[j].m_material = i + 1;
			}
		}

		// Join the parts in a single geometry
		hkGeometry treeGeometry(treeParts[0]);
		hkGeometryUtils::appendGeometry(treeParts[1], treeGeometry);

		// Align to demo up direction
		hkQuaternion q(hkVector4(-1,0,0), HK_REAL_PI / 2.0f);
		for (int i = 0; i < treeGeometry.m_vertices.getSize(); ++i) 
		{
			treeGeometry.m_vertices[i]._setRotatedDir(q, treeGeometry.m_vertices[i]);
		}

		// Calculate AABB and move local origin to the base
		hkAabb aabb; aabb.setEmpty();
		for (int i=0; i <treeGeometry.m_vertices.getSize(); ++i) 
		{
			aabb.includePoint(treeGeometry.m_vertices[i]);
		}
		hkVector4 center; aabb.getCenter(center); center(1) = aabb.m_min(1);
		for (int i=0; i<treeGeometry.m_vertices.getSize(); ++i) 
		{
			treeGeometry.m_vertices[i].sub(center);
		}	

		// Create bv compressed mesh shape
		MyMeshConstructionInfo cinfo(&treeGeometry);
		cinfo.m_collisionFilterInfoMode = hkpBvCompressedMeshShape::PER_PRIMITIVE_DATA_8_BIT;
		meshShape = new hkpBvCompressedMeshShape(cinfo);
	}

	return meshShape;
}

hkpShape* StaticCompoundShapeFilteringDemo::createTerrainShape()
{
	// Create geometry using a height function
	hkGeometry terrainGeometry;
	{	
		HeightFunction heightFunction(m_heightFieldNoise);
		hkVector4 up(0, 1, 0);
		hkReal halfSide = 0.5f * g_terrainSide;
		GeometryMaker::Rectangle functionDomain(0.0f, 0.0f, g_terrainNoiseRange - 1, g_terrainNoiseRange - 1);
		GeometryMaker::Rectangle outputDomain(-halfSide, -halfSide, g_terrainSide, g_terrainSide);	
		GeometryMaker::createFromHeightFunction(heightFunction, g_terrainResolution, g_terrainResolution, up, 0, terrainGeometry, functionDomain, outputDomain);
	}

	// Create BvCompressedMesh
	hkpDefaultBvCompressedMeshShapeCinfo	landscapeinfo(&terrainGeometry);
	hkpBvCompressedMeshShape* landscape = new hkpBvCompressedMeshShape(landscapeinfo);

	return landscape;
}



HK_DECLARE_DEMO(StaticCompoundShapeFilteringDemo, HK_DEMO_TYPE_PHYSICS_2012, "Static Compound Shape - Advanced Filtering", "");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
