/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/StaticCompound/DisablingKeys/StaticCompoundShapeKeyDisablingDemo.h>

// Shapes
#include <Physics2012/Internal/Collide/StaticCompound/hkpStaticCompoundShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>

#include <Demos/DemoCommon/Utilities/Geometry/GeometryMaker.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Query/Collector/RayCollector/hkpAllRayHitCollector.h>
#include <Physics2012/Utilities/Deprecated/H1Group/hkpGroupCollisionFilter.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/SceneDataToGeometryConverter/hkxSceneDataToGeometryConverter.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>

#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GeometryLoaderUtil.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/LandscapeRepository.h>
#include <Graphics/Common/Camera/hkgCamera.h>
#include <Common/Visualize/hkDisplayGeometryBuilder.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/hkpShapeDisplayBuilder.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShapeCinfo.h>
#include <Physics2012/Collide/Shape/Convex/hkpConvexShape.h>

// General parameters
static const hkVector4 g_up = hkVector4(0.0f, 1.0f, 0.0f);

// Camera parameters
static const hkVector4 g_cameraFrom = hkVector4(36.189217f, 24.469324f, 37.591488f);
static const hkVector4 g_cameraTo = hkVector4(21.112724f, 11.683920f, 20.208992f);
static const hkReal g_cameraNear = 0.1f;
static const hkReal g_cameraFar = 1000.0f;

// Physics world parameters
static const hkReal g_worldSize = 500;
static const hkReal g_worldGravity = -9.8f;

// Demo parameters
static const hkReal g_terrainSize = 32;
static const int g_numRays = 1;
static const int g_layerColors[] = { 0xFF257273, 0xFF89b23a, 0xFF89b23a, 0xFFbf3f3f };


StaticCompoundShapeKeyDisablingDemo::StaticCompoundShapeKeyDisablingDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env), m_numFrames(0)
{
	// Collision filter
	hkpGroupFilter* filter = new hkpGroupFilter();
	filter->disableCollisionsBetween(1,2);

	// Create physics world
	hkpWorldCinfo info;	
	{
		info.m_gravity.setMul(g_up, g_worldGravity);
		info.setBroadPhaseWorldSize(g_worldSize);
		info.m_enableDeactivation = false;
		info.m_collisionFilter = filter;		

		// Required in order to use hkpStaticCompoundShape on SPU
		info.m_useCompoundSpuElf = true;
	}
	m_world = new hkpWorld(info);
	filter->removeReference();
	m_world->lock();

	setupGraphics();

	// Create block world and the falling spheres
	createLevel();

	// Register collision agents	
	hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

	m_world->unlock();

	// Set up the camera
	setupDefaultCameras( m_env, g_cameraFrom, g_cameraTo, g_up, g_cameraNear, g_cameraFar );
}

void StaticCompoundShapeKeyDisablingDemo::createLevel()
{
	m_staticCompoundShape = new hkpStaticCompoundShape();
	{
		// Create block world
		hkpBvCompressedMeshShape* blockWorld = createBlockWorld();

		// Add instance one
		{
			m_staticCompoundShape->addInstance(blockWorld, hkQsTransform::getIdentity() );
		}

		// Add instance two
		{
			hkQsTransform transform; transform.setIdentity();
			hkVector4 translation; 
			translation.set(0.0f,-15.0f,0.0f);
			transform.setTranslation(translation);
			m_staticCompoundShape->addInstance(blockWorld, transform );
		}

		m_staticCompoundShape->bake();
		blockWorld->removeReference();
	}

	// Create a fixed body for the block world
	{
		hkpRigidBodyCinfo info;
		info.m_shape = m_staticCompoundShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		hkpRigidBody* body = new hkpRigidBody(info);
		m_world->addEntity(body);

		body->removeReference();
		m_staticCompoundShape->removeReference();

		// Delete the display geometry since we will create our own with colors
		HK_REMOVE_GEOMETRY((hkUlong)body->getCollidable());
	}

	// Create the display geometry for the block world 
	createBlockWorldDisplayShapes();

	// Create falling spheres
	if(1)
	{
		hkpSphereShape* sphere = new hkpSphereShape(0.5f);

		for (int i=0; i<g_terrainSize; i++) 
		{
			for (int j=0; j<g_terrainSize; j++)
			{
				// Create a fixed body using the static compound shape
				if ( i%4==0 && j%4==0)
				{
					hkpRigidBodyCinfo info;
					const hkReal centre = g_terrainSize / 2.0f;
					info.m_position.set( (hkReal)i-centre, 10.0f, (hkReal)j-centre);
					info.m_shape = sphere;
					info.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;
					info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(1);
					hkpRigidBody* body = new hkpRigidBody(info);
					m_world->addEntity(body);
					body->removeReference();
				}
			}
		}

		sphere->removeReference();
	}
}

namespace
{
	class MyMeshConstructionInfo : public hkpDefaultBvCompressedMeshShapeCinfo
	{
		public:

			void addConvexShape(const hkpConvexShape* convexShape, 
								const hkQsTransform& transform = hkQsTransform::getIdentity(),
								hkUint32 userData = 0)
			{
				m_convexShapesUserData.pushBack(userData);
				hkpDefaultBvCompressedMeshShapeCinfo::addConvexShape(convexShape, transform);
			}

			virtual hkUint32 getConvexShapeUserData(int convexIndex) const { return m_convexShapesUserData[convexIndex]; }

		protected:

			hkArray<hkUint32> m_convexShapesUserData;
	};
}

hkpBvCompressedMeshShape* StaticCompoundShapeKeyDisablingDemo::createBlockWorld()
{
	MyMeshConstructionInfo cinfo;

	// Noise function to generate block world
	hkPerlinNoise2d noise(8,128);
	{
		noise.m_frequency = 0.065f;
		noise.m_persistence = 0.15f;
		noise.m_amplitude = 0.5;
	}

	// Box shape for block world
	hkVector4 halfs; halfs.set(.5f,.5f,.5f);
	hkpConvexShape* box = new hkpBoxShape(halfs);

	// Map form physical height to layer definition
	int heightToLayer[] = {0,1,2,2,3,3};

	// Build block terrain
	for (int i=0; i<g_terrainSize; ++i)
	{
		for (int j=0; j<g_terrainSize; ++j)
		{
			// Get a terrain height using noise function
			hkReal height = noise.ntn(hkSimdReal::fromInt32(i),hkSimdReal::fromInt32(j)).getReal() * 7.0f + 2.0f;
			height = hkMath::clamp(height,1.0f,6.0f); // clamp terrain height 

			hkQsTransform transform;
			{
				transform.setIdentity();
				const hkReal centre = g_terrainSize / 2.0f;
				hkVector4 translation; translation.set((hkReal)i-centre, hkMath::floor(height), (hkReal)j-centre);
				transform.setTranslation(translation);
			}

			// add box and set the layer as user data
			cinfo.addConvexShape(box, transform, heightToLayer[(int)height-1]);
		}
	}

	// Create and return shape
	hkpBvCompressedMeshShape* blockWorld;
	{
		cinfo.m_userDataMode = hkpBvCompressedMeshShape::PER_PRIMITIVE_DATA_8_BIT;
		blockWorld = new hkpBvCompressedMeshShape(cinfo);
		box->removeReference();
	}

	return blockWorld;
}


void StaticCompoundShapeKeyDisablingDemo::addDisplayBox( const hkpShape* shape, int key ) 
{
	// Setup a shape builder to create individual display shapes for each box
	hkpShapeDisplayBuilder::hkpShapeDisplayBuilderEnvironment env;
	hkpShapeDisplayBuilder builder(env);
	hkArray<hkDisplayGeometry*> displayGeometries; 
	builder.buildDisplayGeometries(shape, displayGeometries );

	HK_ADD_GEOMETRY(displayGeometries, hkTransform::getIdentity(), key);

	// Set color from user info
	HK_SET_OBJECT_COLOR(key , g_layerColors[int(shape->getUserData())] );

	// Remove references to display geometry
	for ( hkArray<hkDisplayGeometry*>::const_iterator iter = displayGeometries.begin(); iter != displayGeometries.end(); iter++ )
	{
		(*iter)->removeReference();
	}
}


void StaticCompoundShapeKeyDisablingDemo::createBlockWorldDisplayShapes()
{
	// Go through each key of the static compound shape and create a display box for it
	hkpShapeBuffer shapeBuffer;
	for (hkpShapeKey key = m_staticCompoundShape->getFirstKey(); key != HK_INVALID_SHAPE_KEY; key = m_staticCompoundShape->getNextKey(key))
	{
		if (m_staticCompoundShape->isShapeKeyEnabled(key) )
		{
			addDisplayBox( m_staticCompoundShape->getChildShape(key, shapeBuffer), key );
		}
	}
}

void StaticCompoundShapeKeyDisablingDemo::calculateRays(hkArray<hkVector4>& from, hkArray<hkVector4>& to)
{
	from.setSize(g_numRays);
	to.setSize(g_numRays);

	const hkReal fps = 1.0f / m_timestep;

	// Calculate aperture angle	
	const hkReal aperturePeriod = 35.0f * fps;
	hkReal apertureAngle = 5.0f + 40.0f * hkMath::abs(hkMath::cos(HK_REAL_PI * m_numFrames / aperturePeriod));
	apertureAngle *= HK_REAL_DEG_TO_RAD;

	const hkReal stepAngle = 2.0f * HK_REAL_PI / hkReal(g_numRays);
	const hkReal rotationPeriod = 7.0f * fps;
	const hkReal rotationStart = 2.0f * HK_REAL_PI * m_numFrames / rotationPeriod;
	for (int i = 0; i < g_numRays; ++i)
	{	
		const hkReal angle = rotationStart + i * stepAngle;

		// Calculate ray direction
		hkVector4 axis; axis.set(-hkMath::sin(angle), 0.0f, hkMath::cos(angle));
		hkQuaternion rotation; rotation.setAxisAngle(axis, apertureAngle);
		hkVector4 direction; direction.setRotatedDir(rotation, hkVector4(0.0f, -1.0f, 0.0f));

		// Calculate ray start and end
		from[i].set(hkMath::cos(angle), 15.0f, hkMath::sin(angle));
		hkSimdReal rayLength; rayLength.setFromFloat(22.0f);
		to[i].setAddMul(from[i], direction, rayLength);
	}
}

hkDemo::Result StaticCompoundShapeKeyDisablingDemo::stepDemo()
{
	m_world->lock();

	m_numFrames++;

	// Get ray start and end points
	hkArray<hkVector4> from;
	hkArray<hkVector4> to;
	calculateRays(from,to);

	for (int i = 0; i<g_numRays; ++i)
	{
		// Ray-cast against the world
		hkpWorldRayCastInput input; 
		{
			input.m_from = from[i];
			input.m_to = to[i];
			input.m_filterInfo = hkpGroupFilter::calcFilterInfo(2);
			input.m_enableShapeCollectionFilter = true;
		}

		hkpWorldRayCastOutput output;
		m_world->castRay(input, output);

		// Draw hits
		if ( output.hasHit() )
		{
			// Display ray and hit point normal
			hkVector4 hitPoint; hitPoint.setSub(input.m_to, input.m_from);
			hitPoint.setAddMul(input.m_from, hitPoint, output.m_hitFraction);

			// Draw ray hit
			HK_DISPLAY_ARROW(hitPoint, output.m_normal, hkColor::BLUE);
			HK_DISPLAY_LINE(input.m_from, hitPoint, hkColor::GREEN );

			// Only disable stuff on the static compound shape
			const hkpShape* shape = output.m_rootCollidable->getShape();
			if ( shape != m_staticCompoundShape )
			{
				continue;
			}

			// Disable key and set block color transparent 
			m_staticCompoundShape->setShapeKeyEnabled(  output.m_shapeKeys[0], false);			
			HK_SET_OBJECT_COLOR( output.m_shapeKeys[0] , 0x00000000 );
		} 
		else
		{
			// Draw ray miss
			HK_DISPLAY_LINE(input.m_from, input.m_to, hkColor::RED );
		}			
	}

	m_world->unlock();
	return hkDefaultPhysics2012Demo::stepDemo();
}

StaticCompoundShapeKeyDisablingDemo::~StaticCompoundShapeKeyDisablingDemo()
{
	// Clean up graphics
	for (hkpShapeKey key = m_staticCompoundShape->getFirstKey(); key != HK_INVALID_SHAPE_KEY; key = m_staticCompoundShape->getNextKey(key))
	{
		HK_REMOVE_GEOMETRY(key);
	}

	// Clean up physics
	if (m_world)
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}
}


HK_DECLARE_DEMO( StaticCompoundShapeKeyDisablingDemo, HK_DEMO_TYPE_PHYSICS_2012, "Static Compound Shape - Disabling Keys", "Shows how to disable and enable shape keys of a hkpStaticCompoundShape. The two blocky terrains are instances of the same (hkpBvCompressedMeshShape) shape. Shape keys are disabled by a ray cast against one instance of the terrain, while the other instance remains unchanged." );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
