/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/StaticCompound/Construction/StaticCompoundShapeConstructionDemo.h>

// Shapes
#include <Physics2012/Internal/Collide/StaticCompound/hkpStaticCompoundShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShapeCinfo.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>

#include <Demos/DemoCommon/Utilities/Geometry/GeometryMaker.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Query/Collector/RayCollector/hkpAllRayHitCollector.h>
#include <Physics2012/Utilities/Deprecated/H1Group/hkpGroupCollisionFilter.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/SceneDataToGeometryConverter/hkxSceneDataToGeometryConverter.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>

#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GeometryLoaderUtil.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/LandscapeRepository.h>


// General parameters
static const hkVector4 g_up = hkVector4(0, 1, 0);

// Camera parameters
static const hkVector4 g_cameraFrom = hkVector4(113.18197f, -41.226677f, -46.726387f);
static const hkVector4 g_cameraTo = hkVector4(100.22651f, -49.847992f, -24.548067f);
static const hkReal g_cameraNear = 0.1f;
static const hkReal g_cameraFar = 1000.0f;

// Physics world parameters
static const hkReal g_worldSize = 500;
static const hkReal g_worldGravity = -9.81f;

// Terrain parameters
static const int g_dimension = 12;
static const int g_spread = 10;
static const int g_displace = 4;
static const int g_heightFactor = 20;

#ifdef HK_REAL_IS_DOUBLE
#	define FILE_TYPE ".hkt"
#else
#	define FILE_TYPE ".hkx"
#endif

static const char* g_treeFile = HK_ASSET_NAME("Resources/Physics2012/Objects/Tree" FILE_TYPE);


StaticCompoundShapeConstructionDemo::StaticCompoundShapeConstructionDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env),
	m_data(),
	m_debrisPackfileData(HK_NULL)
{
	// Disable this warning, as we know the geometry has too much detail in places
	hkError::getInstance().setEnabled( 0xad345a23, false );	// "hkpBvTreeShape::queryAabb() returned more than 256 hkpShapeKey hits"

	// Create physics world
	hkpWorldCinfo info;	
	{
		info.m_gravity.setMul(g_up, g_worldGravity);
		info.setBroadPhaseWorldSize(g_worldSize);

		// Required in order to use hkpStaticCompoundShape on SPU
		info.m_useCompoundSpuElf = true;
	}
	m_world = new hkpWorld(info);

	m_world->lock();

	setupGraphics();

	// Register collision agents	
	hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

	// Create the hkpStaticCompoundShape landscape
	createLevel();

	// Add some falling debris rigid bodies
	addDebris();

	m_world->unlock();

	// Set up the camera
	setupDefaultCameras( m_env, g_cameraFrom, g_cameraTo, g_up, g_cameraNear, g_cameraFar );
}

void StaticCompoundShapeConstructionDemo::createLevel()
{
	// Create the source shapes that will be instanced in the static compound shape
	hkpShape* terrainShape = createTerrainShape();
	hkpShape* treeShape = createTreeShape();
	hkpShape* boulderShape = createBoulderShape();

	// Create the static compound shape
	{
		m_staticCompoundShape = new hkpStaticCompoundShape();

		// Add terrain
		m_staticCompoundShape->addInstance( terrainShape, hkQsTransform::getIdentity() );

		// Add instances of trees and boulders, each with random hkQsTransform (including scale)
		const int step = 1;
		for ( int i=1; i<g_dimension; i += step )
		{
			for ( int j=1; j<g_dimension; j += step )		
			{
				// Trees
				{
					hkQsTransform transform; 
					getNextTransform( i, j, m_data.m_treeScaling, transform );
					if (transform.getScale().length3().getReal() > 1.15f )
					{
						m_staticCompoundShape->addInstance( treeShape, transform );
					}
				}

				// Boulders
				{
					hkQsTransform transform;
					getNextTransform( i, j, m_data.m_boulderScaling, transform );
					if (transform.getScale().length3().getReal() > 0.75f )
					{
						m_staticCompoundShape->addInstance( boulderShape, transform );
					}
				}
			}
		}

		// We are finished adding instances, so now we bake the shape into its final form
		m_staticCompoundShape->bake();
	}

	// Create a fixed body using the static compound shape
	{
		hkpRigidBodyCinfo info;
		info.m_shape = m_staticCompoundShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;
		hkpRigidBody* body = new hkpRigidBody(info);
		m_world->addEntity(body);
		body->removeReference();
	}

	// Remove references to all shapes
	m_staticCompoundShape->removeReference();
	terrainShape->removeReference();
	treeShape->removeReference();
	boulderShape->removeReference();
}

hkDemo::Result StaticCompoundShapeConstructionDemo::stepDemo()
{
	m_world->lock();

	// Cast some rays through the world
	{
		hkQsTransform transform;
		{
			transform.setIdentity();
			hkVector4 axis; axis.set(0,1,0);
			hkReal seconds = m_env->m_frameTimer.getElapsedSeconds();
			hkQuaternion rotation( axis, seconds*0.03f );
			transform.setRotation(rotation);
			hkVector4 translation; translation.set(47.007f, -0.8834f, 57.446f);
			transform.setTranslation(translation);
		}

		hkArray<hkVector4>::const_iterator iter = m_data.m_rays.begin();
		for (int i=0; i<m_data.m_rays.getSize(); ++i) 
		{
			// Raycast against the world
			hkpWorldRayCastInput input; 

			hkVector4 direction = (*iter++);

			input.m_from = direction;
			input.m_to = direction;

			input.m_from.mul(64);
			input.m_to.mul(155);

			input.m_from.setTransformedPos(transform,input.m_from);
			input.m_to.setTransformedPos(transform,input.m_to);

			hkpWorldRayCastOutput output;
			m_world->castRay(input, output);

			// Draw hits
			if ( output.hasHit() )
			{
				hkVector4 hitPoint; hitPoint.setSub(input.m_to, input.m_from);
				hitPoint.setAddMul(input.m_from, hitPoint, output.m_hitFraction);
				HK_DISPLAY_ARROW(hitPoint, output.m_normal, hkColor::BLUE);
				HK_DISPLAY_LINE(input.m_from, hitPoint, (hkColor::GREENYELLOW&0x00ffffff)|0x7f000000 );
			} 
			else
			{
				HK_DISPLAY_LINE(input.m_from, input.m_to, (hkColor::RED&0x00ffffff)|0x7f000000 );
			}			
		}
	}

	m_world->unlock();

	return hkDefaultPhysics2012Demo::stepDemo();
}

StaticCompoundShapeConstructionDemo::~StaticCompoundShapeConstructionDemo()
{
	m_world->markForWrite();

	if (m_world)
	{
		m_world->removeReference();
		m_world = HK_NULL;
	}

	if( m_debrisPackfileData )
	{
		m_debrisPackfileData->removeReference();
	}
}

//
// Below are helper functions just for setup. No need to read past here.
//

StaticCompoundShapeConstructionDemo::DemoData::DemoData()
:	m_heightFieldNoise( 1, g_dimension ),
	m_treeScaling( 7, g_dimension ),
	m_boulderScaling( 2, g_dimension ),
	m_random( 5 )
{
	// tweaks
	m_heightFieldNoise.m_frequency = 0.1175f;
	m_heightFieldNoise.m_persistence = 0.2f;
	m_boulderScaling.m_frequency = 0.3f; 

	createRays(m_rays);
}

void StaticCompoundShapeConstructionDemo::DemoData::createRays( hkArray<hkVector4>& rays)
{
	hkArray<hkVector4> verts;
	{
		const hkReal phi = sqrt(5.0f)/2.0f + 0.5f;
		{ hkVector4 v; v.set(0.0f,  1.0f,  phi, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(0.0f, -1.0f,  phi, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(0.0f,  1.0f, -phi, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(0.0f, -1.0f, -phi, 0.0f); v.normalize3(); verts.pushBack(v); }

		{ hkVector4 v; v.set(1.0f,  phi, 0.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(-1.0f,  phi, 0.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(1.0f, -phi, 0.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(-1.0f, -phi, 0.0f, 0.0f); v.normalize3(); verts.pushBack(v); }

		{ hkVector4 v; v.set(phi, 0.0f,  1.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(-phi, 0.0f,  1.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(phi, 0.0f, -1.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
		{ hkVector4 v; v.set(-phi, 0.0f, -1.0f, 0.0f); v.normalize3(); verts.pushBack(v); }
	}

	for (int j=0; j<3; j++)
	{
		hkStridedVertices strided(verts);
		const hkpConvexVerticesShape shape1(strided);
		const hkArray<hkVector4>& planes = shape1.getPlaneEquations();
		// add new normals that point 
		verts.append(planes.begin(), planes.getSize());
	}

	// filter out rays that are not pointing downwards
	const hkVector4 *p = verts.begin();
	for (int i=0; i<verts.getSize(); i++)
	{
		if ((*p)(1) < -0.7)
		{
			rays.pushBack(*p);
		}
		p++;
	}
}

void StaticCompoundShapeConstructionDemo::getNextTransform( int i, int j, hkPerlinNoise2d& scaleMap, hkQsTransform& transform ) 
{
	hkQuaternion rotation;
	{
		hkVector4 axis; axis.set(m_data.m_random.getRandReal11()*0.1f,1,m_data.m_random.getRandReal11()*0.1f); axis.normalize3(); 
		rotation.setAxisAngle(axis,m_data.m_random.getRandReal11()*3);
	}

	hkVector4 translation;
	{
		translation.set((hkReal)i*g_spread + m_data.m_random.getRandReal11()*g_displace,m_data.m_heightFieldNoise.ntn(hkSimdReal::fromInt32(i),hkSimdReal::fromInt32(j)).getReal()*g_heightFactor-80,(hkReal)j*g_spread + m_data.m_random.getRandReal11()*g_displace);
	}

	hkVector4 scale; 
	{
		hkReal scaleFactor = scaleMap.ntn((hkReal)i,(hkReal)j);
		scale.set(m_data.m_random.getRandReal01(),m_data.m_random.getRandReal01(),m_data.m_random.getRandReal01());
		scale.mul(0.6f); 
		hkVector4 tmp; tmp.set(0.4f,0.4f,0.4f); 
		scale.add(tmp);
		scale.mul( hkMath::abs(scaleFactor*2.5f));
	}

	transform.setIdentity();
	transform.setScale(scale);
	transform.setTranslation(translation);
	transform.setRotation(rotation);
}

hkpShape* StaticCompoundShapeConstructionDemo::createBoulderShape()
{
	// Random vertices for the base boulder shape
	hkpConvexVerticesShape* boulderShape;
	{
		hkPseudoRandomGenerator boulderVertsRandom(2);
		hkArray<hkVector4> boulderVerts;
		for (int i=0; i<128; i++) // many will be removed in the final hull
		{
			hkVector4 point; boulderVertsRandom.getRandomVector11(point); point.mul(1.8f);

			boulderVerts.pushBack( point );
		}

		hkStridedVertices boulderStrided(boulderVerts);
		boulderShape = new hkpConvexVerticesShape(boulderStrided);
	}

	return boulderShape;
}

hkpShape* StaticCompoundShapeConstructionDemo::createTreeShape()
{
	// Setup tree shape
	hkpBvCompressedMeshShape* meshShape;
	{
		hkGeometry treeGeometryOut;
		{
			// Open and load resource file	
			hkResource* data = HK_NULL;
			hkStringBuf assetFile(g_treeFile); 
			hkAssetManagementUtil::getFilePath(assetFile);
			data = hkSerializeUtil::load(assetFile.cString());
			HK_ASSERT2(0x01234390, data, "Could not load resource");	
			GeometryLoaderUtil::load(data, treeGeometryOut);
			data->removeReference();

			hkQuaternion q(hkVector4(-1,0,0), HK_REAL_PI / 2.0f);
			for (int i = 0; i < treeGeometryOut.m_vertices.getSize(); ++i) 
			{
				treeGeometryOut.m_vertices[i]._setRotatedDir(q, treeGeometryOut.m_vertices[i]);
			}

			// Calculate AABB and move local origin to the base
			hkAabb aabb; aabb.setEmpty();
			for (int i=0; i <treeGeometryOut.m_vertices.getSize(); ++i) 
			{
				aabb.includePoint(treeGeometryOut.m_vertices[i]);
			}
			hkVector4 center; aabb.getCenter(center); center(1) = aabb.m_min(1);
			for (int i=0; i<treeGeometryOut.m_vertices.getSize(); ++i) 
			{
				treeGeometryOut.m_vertices[i].sub(center);
			}	
		}

		hkpDefaultBvCompressedMeshShapeCinfo	cinfo(&treeGeometryOut);
		meshShape = new hkpBvCompressedMeshShape(cinfo);
	}

	return meshShape;
}

namespace
{
	struct HeightFunction
	{
	public:
		HeightFunction(const hkPerlinNoise2d& noiseIn):m_noise(noiseIn) {}
		const hkPerlinNoise2d& m_noise;
		HK_FORCE_INLINE hkReal operator()(hkReal x, hkReal z) const
		{
			return m_noise.ntn(hkSimdReal::fromFloat(x),hkSimdReal::fromFloat(z)).getReal()*20-80;
		}
	};
}

hkpShape* StaticCompoundShapeConstructionDemo::createTerrainShape()
{
	// Create terrain surface mesh
	hkGeometry terrainGeometry;
	GeometryMaker::Rectangle functionDomain(0.0f, 0.0f, (hkReal)g_dimension, (hkReal)g_dimension);
	GeometryMaker::Rectangle outputDomain(0.0f, 0.0f, (hkReal)(g_spread*g_dimension), (hkReal)(g_spread*g_dimension));
	hkVector4 up; up.set(0.0f,1.0f,0.0f);
	
	HeightFunction heightFunction(m_data.m_heightFieldNoise);

	GeometryMaker::createFromHeightFunction( heightFunction, g_dimension*2, g_dimension*2, up, 0, terrainGeometry, functionDomain, outputDomain);
	hkpDefaultBvCompressedMeshShapeCinfo	landscapeinfo(&terrainGeometry);
	hkpBvCompressedMeshShape* landscape = new hkpBvCompressedMeshShape(landscapeinfo);

	return landscape;
}

void StaticCompoundShapeConstructionDemo::addDebris()
{
	// Load falling shapes
	hkStringBuf fileName(HK_ASSET_NAME("Resources/Physics2012/Compoundbodies/compoundbodies" FILE_TYPE));
	hkAssetManagementUtil::getFilePath(fileName);
	m_debrisPackfileData = hkSerializeUtil::load(fileName);

	hkArray<hkAabb> aabbs;  
	{
		hkAabb aabb;
		m_staticCompoundShape->getAabb(hkTransform::getIdentity(), 0.0f, aabb);
		hkVector4 p1; p1.setInterpolate(aabb.m_min,aabb.m_max, 0.10f);
		hkVector4 p2; p2.setInterpolate(aabb.m_min,aabb.m_max, 0.90f);
		aabb.setFromLine(p1,p2);
		aabbs.pushBack(aabb);
	}

	AabbSpawnUtil spawnUtil(aabbs);
	HK_ASSERT( 0x215d080c, m_debrisPackfileData );

	hkRootLevelContainer* container = m_debrisPackfileData->getContents<hkRootLevelContainer>();
	HK_ASSERT2(0xa6451543, container != HK_NULL, "Could not load root level object" );

	hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
	HK_ASSERT2(0xa6451544, physicsData != HK_NULL, "Could not find physics data in root level object" );

	const hkArray<hkpRigidBody*>& rigidBodies = physicsData->getPhysicsSystems()[0]->getRigidBodies();

	spawnUtil.m_pseudoRandomGenerator.setSeed(180673);

	for ( int i = 0; i < 50; ++i )
	{
		const hkpRigidBody* srcBody = rigidBodies[i % rigidBodies.getSize()];

		// Create a new body, set the compound shape on it
		hkpRigidBody* newBody = srcBody->clone();
		hkQuaternion rotation; spawnUtil.m_pseudoRandomGenerator.getRandomRotation( rotation );
		newBody->setRotation( rotation );

		hkAabb aabb;  newBody->getCollidable()->getShape()->getAabb(newBody->getTransform(), .05f, aabb );
		hkVector4 objectSize; objectSize.setSub4( aabb.m_max, aabb.m_min );
		hkVector4 position; spawnUtil.getNewSpawnPosition( objectSize, position );
		position(1)+=40.0f;

		newBody->setPosition( position );

		m_world->addEntity( newBody );
		newBody->removeReference();
	}
}




static const char helpString[] = \
"Shows how to construct a hkpStaticCompoundShape, using instances of other shapes at varying transforms.\n \
All trees share the same underlying mesh shape, and all boulders share the same underlying convex vertices shape.";

HK_DECLARE_DEMO(StaticCompoundShapeConstructionDemo, HK_DEMO_TYPE_PHYSICS_2012, "Static Compound Shape - Creation", helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
