/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/Misc/LandscapeComparison/LandscapeDemoUtil.h>

// Shapes
#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Collide/Shape/Deprecated/CompressedMesh/hkpCompressedMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/ExtendedMeshShape/hkpExtendedMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics2012/Collide/Shape/Convex/Sphere/hkpSphereShape.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/HeightField/CompressedSampledHeightField/hkpCompressedSampledHeightFieldShape.h>
#include <Physics2012/Collide/Shape/HeightField/TriSampledHeightField/hkpTriSampledHeightFieldCollection.h>
#include <Physics2012/Collide/Shape/HeightField/TriSampledHeightField/hkpTriSampledHeightFieldBvTreeShape.h>

// Shape building.
#include <Common/Base/Types/Geometry/hkStridedVertices.h>
#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
#include <Physics2012/Collide/Shape/Deprecated/CompressedMesh/hkpCompressedMeshShapeBuilder.h>
#include <Physics2012/Collide/Util/Welding/hkpMeshWeldingUtility.h>

// Mopp creation.
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppCompilerInput.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>

// Deserializing & asset stuff
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>

// Misc utils
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Common/SceneData/SceneDataToGeometryConverter/hkxSceneDataToGeometryConverter.h>

#define CMS_LANDSCAPE_CASTLE "Resources/Common/Landscapes/Sandcastle_9_cms.hkt"
#define EMS_LANDSCAPE_CASTLE "Resources/Common/Landscapes/Sandcastle_9_ems.hkt"

#include <Physics2012/Collide/Shape/Compound/Collection/Mesh/hkpMeshMaterial.h>



LandscapeDemoUtil::LandscapeDemoUtil()
: m_error( DEFAULT_ERROR )
, m_simpleLandscapeNumSides( DEFAULT_SIDES_OF_SIMPLE_LANDSCAPE )
, m_simpleLandscapeScale( DEFAULT_SCALE_SIMPLE_LANDSCAPE )
, m_numVerticesConvexPiece( DEFAULT_NUM_VERTICES_CONVEX_PIECE )
, m_convexRadius( DEFAULT_CONVEX_RADIUS )
, m_random( DEFAULT_RANDOM_SEED )
{
}

LandscapeDemoUtil::~LandscapeDemoUtil()
{
	hkReferencedObject::removeReferences( m_externalData.begin(), m_externalData.getSize() );
}

void LandscapeDemoUtil::setLandscapeBodyCinfo( LandscapeShape landscapeShape, LandscapeSource landscapeSource, hkpRigidBodyCinfo& infoOut )
{
	infoOut.m_shape = createLandscapeShape( landscapeShape, landscapeSource );
	infoOut.m_motionType = hkpMotion::MOTION_FIXED;
	
	// z is up.
	if ( landscapeShape == LANDSCAPE_IS_HEIGHT_FIELD )
	{
		hkQuaternion q;
		{
			hkVector4 axis;
			axis.set( 1.0f, 0.0f, 0.0f );
			hkReal angle = HK_REAL_PI / 2;
			q.setAxisAngle( axis, angle );
		}
		hkVector4 t;
		{
			t.set( 0.0f, m_simpleLandscapeScale * m_simpleLandscapeNumSides * 0.5f, 0.0f );
		}

		infoOut.m_rotation = q;
		infoOut.m_position = t;
	}
}

const hkpShape* LandscapeDemoUtil::createLandscapeShape( LandscapeShape landscapeShape, LandscapeSource landscapeSource )
{
	switch( landscapeSource )
	{
		case LANDSCAPE_SOURCE_IS_SIMPLE:
			return createSimpleLandscapeShape( landscapeShape );
		default:
		case LANDSCAPE_SOURCE_IS_CASTLE:
			return createCastleLandscapeShape( landscapeShape );
	}
}

const hkpShape* LandscapeDemoUtil::createSimpleLandscapeShape( LandscapeShape landscapeShape )
{
	switch( landscapeShape )
	{
		case LANDSCAPE_IS_CMS:
			return createMoppShape( createSimpleCompressedMeshShape(), true );
		case LANDSCAPE_IS_EMS:
			return createMoppShape( createSimpleExtendedMeshShape(), true );
		case LANDSCAPE_IS_HEIGHT_FIELD:
			return createSimpleHeightFieldShape();
		default:
			HK_ASSERT( 0x241f7f2, false );
			return HK_NULL;
	}
}

const hkpShape* LandscapeDemoUtil::createCastleLandscapeShape( LandscapeShape landscapeShape )
{
	switch( landscapeShape )
	{
		case LANDSCAPE_IS_CMS:
			return createMoppShape( createCastleCompressedMeshShape(), true );
		case LANDSCAPE_IS_EMS:
			return createCastleExtendedMeshShape();
		default:
			HK_ASSERT( 0x241f7f2, false );
			return HK_NULL;
	}
}

hkpCompressedMeshShape* LandscapeDemoUtil::createSimpleCompressedMeshShape()
{
	hkpCompressedMeshShapeBuilder builder;

	builder.m_overlapRatio = 0.2f;
	builder.m_stripperPasses = 5000;
	builder.m_weldVertices = true;
	builder.m_TjunctionTolerance = 0;
	builder.m_preserveTjunctions = false;

	hkpCompressedMeshShape* mesh = builder.createMeshShape( m_error, hkpCompressedMeshShape::MATERIAL_ONE_BYTE_PER_TRIANGLE, 18 );

	mesh->setRadius( m_convexRadius );

	// set-up the material table
	mesh->m_meshMaterials = (hkpMeshMaterial*)m_collisionFilterInfos.begin();
	mesh->m_materialStriding = sizeof(hkpMeshMaterial);
	mesh->m_numMaterials = (hkUint16)m_collisionFilterInfos.getSize();

	hkGeometry wavyGeom;
	createWavySurface( m_simpleLandscapeNumSides, wavyGeom );
	builder.beginSubpart( mesh );
	builder.addGeometry( wavyGeom, hkMatrix4::getIdentity(), mesh );
	builder.endSubpart( mesh );
	builder.addInstance( 0, hkMatrix4::getIdentity(), mesh );

	if ( m_numVerticesConvexPiece )
	{
		const int numVerts = m_numVerticesConvexPiece;
		// create an array of random vertices
		hkArray<hkVector4> vertices;
		{
			vertices.reserve( numVerts );
			const hkVector4	ctr = wavyGeom.m_vertices[wavyGeom.m_vertices.getSize()/2];
			for(int i = 0; i < numVerts; i++)
			{
				vertices.pushBackUnchecked( GameUtils::createRandomVertex( hkSphere( hkVector4(0,0,0), 5 ), &m_random ));
				vertices[vertices.getSize()-1](2) += ctr(2);
			}			
		}

		builder.beginSubpart( mesh );
		builder.addConvexPiece( vertices, mesh );
		builder.endSubpart( mesh );
		builder.addInstance( 1, hkMatrix4::getIdentity(), mesh );		
	}

	return mesh;
}

hkpExtendedMeshShape* LandscapeDemoUtil::createSimpleExtendedMeshShape()
{
	hkpStorageExtendedMeshShape* mesh = new hkpStorageExtendedMeshShape( m_convexRadius );

	hkGeometry wavyGeom;
	createWavySurface( m_simpleLandscapeNumSides, wavyGeom );

	hkArray<hkUint8>::Temp collisionFilterInfoIndices;
	for ( int i = 0; i < wavyGeom.m_triangles.getSize(); ++i )
	{
		collisionFilterInfoIndices.pushBack( (hkUint8) wavyGeom.m_triangles[i].m_material );
	}

	hkpExtendedMeshShape::TrianglesSubpart triSubPart;
	{
		triSubPart.m_vertexBase = &wavyGeom.m_vertices.begin()[0](0);
		triSubPart.m_vertexStriding = sizeof( hkVector4 );
		triSubPart.m_numVertices = wavyGeom.m_vertices.getSize();
		triSubPart.m_indexBase = wavyGeom.m_triangles.begin();
		triSubPart.m_indexStriding = sizeof( int ) * 4;
		triSubPart.m_stridingType = hkpExtendedMeshShape::INDICES_INT32;
		triSubPart.m_numTriangleShapes = wavyGeom.m_triangles.getSize();

		if ( m_collisionFilterInfos.getSize() )
		{
			triSubPart.m_materialIndexBase = collisionFilterInfoIndices.begin();
			triSubPart.m_materialIndexStriding = 1;
			triSubPart.setMaterialIndexStridingType(hkpExtendedMeshShape::MATERIAL_INDICES_INT8);
			triSubPart.m_materialBase = (hkpMeshMaterial*)m_collisionFilterInfos.begin();
			triSubPart.m_materialStriding = hkSizeOf(hkUint32);
			triSubPart.setNumMaterials((hkUint16) m_collisionFilterInfos.getSize());
		}
	}
	mesh->addTrianglesSubpart(triSubPart);

	if ( m_numVerticesConvexPiece )
	{
		const int numVerts = m_numVerticesConvexPiece;
		// create an array of random vertices
		hkArray<hkVector4> vertices;
		{
			vertices.reserve( numVerts );
			const hkVector4	ctr = wavyGeom.m_vertices[wavyGeom.m_vertices.getSize()/2];
			for(int i = 0; i < numVerts; i++)
			{
				vertices.pushBackUnchecked( GameUtils::createRandomVertex( hkSphere( hkVector4(0,0,0), 5 ), &m_random ));
				vertices[vertices.getSize()-1](2) += ctr(2);
			}
		}
		hkpConvexVerticesShape::BuildConfig buildConfig;
		{
			buildConfig.m_convexRadius = m_convexRadius;
		}
		hkStridedVertices stridedVertices;
		{
			stridedVertices.m_numVertices = numVerts;
			stridedVertices.m_vertices = (hkReal*) vertices.begin();
			stridedVertices.m_striding = hkSizeOf( hkVector4 );
		}
		const hkpConvexShape *const convexVerticesShape = new hkpConvexVerticesShape( stridedVertices, buildConfig );

		hkpExtendedMeshShape::ShapesSubpart shapeSubPart( &convexVerticesShape, 1 );

		convexVerticesShape->removeReference();
		
		mesh->addShapesSubpart( shapeSubPart );
	}
	
	return mesh;
}

// This is used to initialize the actual heightfield shape.
class SourceHeightFieldShape : public hkpSampledHeightFieldShape
{
	public:		

		SourceHeightFieldShape( const hkpSampledHeightFieldBaseCinfo& ci )
			:	hkpSampledHeightFieldShape(ci)
		{
		}

		HK_FORCE_INLINE hkReal getHeightAtImpl( int x, int z ) const
		{
			return LandscapeDemoUtil::wavyFunction( x, z );
		}

		HK_FORCE_INLINE hkBool getTriangleFlipImpl() const
		{	
			return false;
		}

		virtual void collideSpheres( const CollideSpheresInput& input, SphereCollisionOutput* outputArray) const
		{
			hkSampledHeightFieldShape_collideSpheres(*this, input, outputArray);
		}
};


hkpTriSampledHeightFieldBvTreeShape* LandscapeDemoUtil::createSimpleHeightFieldShape()
{
	hkpTriSampledHeightFieldBvTreeShape* bvTree;
	{
		const int xRes = m_simpleLandscapeNumSides;
		const int zRes = m_simpleLandscapeNumSides;

		hkpSampledHeightFieldBaseCinfo info;
		{
			info.m_scale.set( m_simpleLandscapeScale, m_simpleLandscapeScale, m_simpleLandscapeScale );
			info.m_xRes = xRes;
			info.m_zRes = zRes;
		}

		SourceHeightFieldShape source( info );

		hkpCompressedSampledHeightFieldShape* hfShape = new hkpCompressedSampledHeightFieldShape( &source );

		// Wrap the heightfield in a hkpTriSampledHeightFieldCollection:
		hkpTriSampledHeightFieldCollection* collection = new hkpTriSampledHeightFieldCollection( hfShape );
		hfShape->removeReference();

		// Now wrap the hkpTriSampledHeightFieldCollection in a hkpTriSampledHeightFieldBvTreeShape
		bvTree = new hkpTriSampledHeightFieldBvTreeShape( collection );
		collection->removeReference();
	}

	return bvTree;
}


hkpCompressedMeshShape* LandscapeDemoUtil::createCastleCompressedMeshShape()
{
	hkResource* data = hkSerializeUtil::load("Resources/Physics2012/Landscapes/castle_cms.hkt");
	m_externalData.pushBack(data);

	hkpCompressedMeshShape* compressedMeshShape = data->getContents<hkpCompressedMeshShape>();

	if ( m_collisionFilterInfos.getSize() )
	{
		if ( compressedMeshShape->m_meshMaterials && m_collisionFilterInfos.getSize() )
		{
			for ( int i = 0; i < compressedMeshShape->m_numMaterials; ++i )
			{
				( const_cast<hkpMeshMaterial*>(hkAddByteOffsetConst( compressedMeshShape->m_meshMaterials, i * compressedMeshShape->m_materialStriding )))->m_filterInfo = m_collisionFilterInfos[ i % m_collisionFilterInfos.getSize() ];
			}
		}
	}

	return compressedMeshShape;
}

const hkpShape* LandscapeDemoUtil::createCastleExtendedMeshShape()
{
	const hkpShape* shape = loadFirstShapeInAsset( EMS_LANDSCAPE_CASTLE );

	const hkpShape* ems = shape;
	if ( ems->getType() == hkcdShapeType::MOPP )
	{
		ems = static_cast<const hkpMoppBvTreeShape*>( ems )->getChild();
	}

	HK_ASSERT( 0x4f78a915, ems->getType() == hkcdShapeType::EXTENDED_MESH );

	// If there is a material table in the landscape, we overwrite it with the collision
	// filter infos in this utility so it works with the demo.
	if ( m_collisionFilterInfos.getSize() )
	{
		const hkpExtendedMeshShape* extendedMeshShape = static_cast<const hkpExtendedMeshShape*>( ems );

		for ( int i = 0; i < extendedMeshShape->getNumTrianglesSubparts(); ++i )
		{
			const hkpExtendedMeshShape::Subpart& subPart = extendedMeshShape->getTrianglesSubpartAt( i );
			if ( subPart.m_materialBase && subPart.m_materialStriding )
			{
				for ( int j = 0; j < subPart.getNumMaterials(); ++j )
				{
					( const_cast<hkpMeshMaterial*>(hkAddByteOffsetConst( subPart.m_materialBase, j * subPart.m_materialStriding )))->m_filterInfo = m_collisionFilterInfos[ ( i + j ) % m_collisionFilterInfos.getSize() ];
				}
			}
		}
		for ( int i = 0; i < extendedMeshShape->getNumShapesSubparts(); ++i )
		{
			const hkpExtendedMeshShape::Subpart& subPart = extendedMeshShape->getShapesSubpartAt( i );
			if ( subPart.m_materialBase && subPart.m_materialStriding )
			{
				for ( int j = 0; j < subPart.getNumMaterials(); ++j )
				{
					( const_cast<hkpMeshMaterial*>(hkAddByteOffsetConst( subPart.m_materialBase, j * subPart.m_materialStriding )))->m_filterInfo = m_collisionFilterInfos[ i + j % m_collisionFilterInfos.getSize() ];
				}
			}
		}
	}

	return shape;
}

hkpMoppBvTreeShape* LandscapeDemoUtil::createMoppShape( hkpShapeCollection* meshShape, bool useChunks )
{
	hkpMoppCompilerInput moppInput;
	moppInput.m_enableChunkSubdivision = useChunks;

	// Build the code at runtime
	hkpMoppCode* code = hkpMoppUtility::buildCode( meshShape, moppInput );

	hkpMoppBvTreeShape* moppShape = HK_NULL;
	moppShape = new hkpMoppBvTreeShape( meshShape, code );

	//hkpMeshWeldingUtility::computeWeldingInfo( meshShape, moppShape, hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE );

	meshShape->removeReference();
	code->removeReference();

	return moppShape;
}

float HK_CALL hkgTextureNoise(float x, float y, float z);

hkUint32 LandscapeDemoUtil::getCollisionFilterInfoIndex( int index ) const
{
	if ( m_collisionFilterInfos.getSize() )
	{
		return ( index % m_collisionFilterInfos.getSize() );
	}
	else
	{
		return 0;
	}
}

hkReal LandscapeDemoUtil::wavyFunction( int i, int j )
{
	float x = i * 0.1f;
	float y = j * 0.1f;
	float n = (hkgTextureNoise(x,y,0)+1.0f)/2.0f;
	return n*n*n*48;
	//return 1.6f * hkMath::cos((hkReal)j + i) + 0.3f * hkMath::sin( 2.0f * i);
}

void LandscapeDemoUtil::createWavySurface( int side, hkGeometry& geometry ) const
{
	{
		geometry.m_vertices.setSize( side * side );
		for(int i = 0; i < side; i++)
		{
			for (int j = 0; j < side; j++ )
			{
				hkVector4 vertex ( 
					m_simpleLandscapeScale * (i * 1.0f - (side-1) * 0.5f),
					m_simpleLandscapeScale * (j * 1.0f - (side-1) * 0.5f),
					wavyFunction( i, j ) );
				geometry.m_vertices[i*side + j] = vertex ;
			}
		}
	}

	{
		geometry.m_triangles.setSize( (side-1) * (side-1) * 2);
		int corner = 0;
		int curTri = 0;
		for(int i = 0; i < side - 1; i++)
		{
			for (int j = 0; j < side - 1; j++ )
			{
				geometry.m_triangles[curTri].m_c = corner+1;
				geometry.m_triangles[curTri].m_b = corner+side;
				geometry.m_triangles[curTri].m_a = corner;
				geometry.m_triangles[curTri].m_material = getCollisionFilterInfoIndex( ( i * ( side - 1 ) ) + j );
				curTri++;

				geometry.m_triangles[curTri].m_c = corner+side+1;
				geometry.m_triangles[curTri].m_b = corner+side;
				geometry.m_triangles[curTri].m_a = corner+1;
				geometry.m_triangles[curTri].m_material = getCollisionFilterInfoIndex( ( i * ( side - 1 ) ) + j );
				curTri++;
				corner++; 
			}
			corner++; 
		}
	}
}

const hkpShape* LandscapeDemoUtil::loadFirstShapeInAsset( const char* filename )
{
	// Load the file
	hkSerializeUtil::ErrorDetails loadError;
	hkResource* loadedData;
	loadedData = hkSerializeUtil::load( filename, &loadError );

	HK_ASSERT3(0xa6451543, loadedData != HK_NULL, "Could not load file. The error is:\n" << loadError.defaultMessage.cString() );

	// Get the top level object in the file, which we know is a hkRootLevelContainer
	hkRootLevelContainer* container = loadedData->getContents<hkRootLevelContainer>();
	HK_ASSERT2(0xa6451543, container != HK_NULL, "Could not load root level obejct" );

	// Get the physics data
	hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
	HK_ASSERT2(0xa6451544, physicsData != HK_NULL, "Could not find physics data in root level object" );

	HK_ASSERT2( 0x231a7ac2, physicsData->getPhysicsSystems().getSize() > 0, "There are no physics systems in the asset." );
	hkpPhysicsSystem* system0 = physicsData->getPhysicsSystems()[0];

	HK_ASSERT2( 0xb377381b, system0->getRigidBodies().getSize() > 0, "There are no rigid bodies in the first physics system." );
	hkpRigidBody* system0body0 = system0->getRigidBodies()[0];

	const hkpShape* shape = system0body0->getCollidableRw()->getShape();
	HK_ASSERT2( 0xb377381c, shape, "There first rigid body in the first physics system has no shape." );

	m_externalData.pushBack( loadedData );

	return shape;
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
