/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Demos/Physics2012/Api/Collide/Shapes/HeightField/CoarseGridHeightField/CoarseGridHeightFieldDemo.h>

//
// Need some shapes
//
#include <Physics2012/Collide/Shape/Convex/Box/hkpBoxShape.h>
#include <Physics2012/Collide/Shape/HeightField/SampledHeightField/hkpSampledHeightFieldShape.h>
#include <Physics2012/Collide/Shape/HeightField/SampledHeightField/hkpSampledHeightFieldBaseCinfo.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabb.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics2012/Collide/Query/Collector/RayCollector/hkpClosestRayHitCollector.h>

// We need to display the results of the raycast to confirm correctness.
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Algorithm/Noise/hkPerlinNoise2d.h>



/*
Coarse Height Field Demo

Press 'N' to change the coarseness of the min-max quad tree
Press 'M' to toggle between sweep and line-of-sight mode
*/

class CoarseSampledHeightFieldShape: public hkpSampledHeightFieldShape
{
	public:
		

		CoarseSampledHeightFieldShape( const hkpSampledHeightFieldBaseCinfo& ci, hkUint16* data )
			:	hkpSampledHeightFieldShape(ci),
				m_data(data)
		{
		}

		// Generate a rough terrain
		HK_FORCE_INLINE hkReal getHeightAtImpl( int x, int z ) const
		{
			// Lookup data and return a float
			// We scale the data artificially by 20.0f to make it look interesting
			return 20.0f * hkReal(m_data[x * m_zRes + z]) / hkReal( hkUint16(-1) );
		}

		// Assuming each heightfield quad is defined as four points { 00, 01, 11, 10 },
		// this should return true if the two triangles share the edge p00-p11.
		// Otherwise it should return false if the triangles share the edge p01-p10
		HK_FORCE_INLINE hkBool getTriangleFlipImpl() const
		{	
			return false;
		}

		virtual void collideSpheres( const CollideSpheresInput& input, SphereCollisionOutput* outputArray) const
		{
			hkSampledHeightFieldShape_collideSpheres(*this, input, outputArray);
		}

	private:

		hkUint16* m_data;
};


// We can see that the getHeightAt function is very straight forward and 
// could easily do a displacement map lookup from your render data etc.
// The Triangle flip specifies if given a quad of 4 points [00,01,11,10], does
// the triangle diagonal span 00-11 (true), or 01,10 (false). This is so
// that the collision detection and the display are using the same convention.
// Now that we have a class to use, lets see how easy it is to make one.



CoarseGridHeightFieldDemo::CoarseGridHeightFieldDemo(hkDemoEnvironment* env)
	:	hkDefaultPhysics2012Demo(env, DEMO_FLAGS_NO_SERIALIZE)
{
	m_time = 0.0f;

	// Then we setup a camera in the right place to see our demo.
	{
		hkVector4 from( -5.5f, 17.f, -7.f );
		hkVector4 to  ( 20.5f, 10.f, 20.f );
		hkVector4 up  (  0.0f, 1.0f, 0.0f);
		setupDefaultCameras(env, from, to, up);
	}

	// We create a world the same as always, and let the demo initialize the graphics.
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 200.0f );
		info.m_collisionTolerance = 0.03f;
		m_world = new hkpWorld(info);
		m_world->lock();

		setupGraphics();
	}

	// As we want collision detection, we must register the 
	// agents for the shapes we are going to use. Here we will
	// just use the hkpAgentRegisterUtil to register the common ones
	// and that includes the heightfields.
	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

	// Resolution of the heightfield
	const int xRes = 256;
	const int zRes = 256;


	// We will use perlin noise to create the heightfield
	hkPerlinNoise2d noise(123, xRes);
	noise.m_frequency = .016f*(256.0f/xRes);   
	noise.m_octaves = 8;
	noise.m_persistence = 0.3f;
	noise.m_amplitude = 0.23f;

	// Here we create an array of shorts that our heightfield will lookup
	// This array could easily have been loaded from a texture.
	m_heightData = hkAllocate<hkUint16>(xRes * zRes, HK_MEMORY_CLASS_DEMO);
	{
		for (int x = 0; x < xRes; x++)
		{
			for (int z = 0; z < zRes; z++)
			{
				hkReal f = hkMath::clamp(noise.ntn(hkSimdReal::fromInt32(x),hkSimdReal::fromInt32(z)).getReal(), hkReal(0), hkReal(1));
				m_heightData[x*zRes + z] = static_cast<hkUint16> ( hkUint16(-1)*f );
			}
		}
	}


	// Finally, we want our new heightfield. It will be a fixed body with 
	// the shape that we have defined at the start. 
	{
		hkpSampledHeightFieldBaseCinfo ci;
		ci.m_xRes = xRes;
		ci.m_zRes = zRes;
		ci.m_scale.set(64.0f/xRes,1.0f,64.0f/zRes);

		m_heightFieldShape = new CoarseSampledHeightFieldShape( ci , m_heightData);

		// Now that we have a shape all we need is the fixed body to represent it in the 
		// the simulation. Standard rigid body setup.
		{
			hkpRigidBodyCinfo rci;
			rci.m_motionType = hkpMotion::MOTION_FIXED;
			rci.m_shape = m_heightFieldShape;
			rci.m_friction = 0.2f;

			hkpRigidBody* body = new hkpRigidBody( rci );

			m_world->addEntity(body);
			body->removeReference();
		}
	}

	// Build the coarse tree
	m_coarseness = 3;
	m_heightFieldShape->buildCoarseMinMaxTree(m_coarseness);

	m_world->unlock();
	m_demomode = 0;
}


CoarseGridHeightFieldDemo::~CoarseGridHeightFieldDemo()
{
	m_heightFieldShape->removeReference();
	hkDeallocate<hkUint16>( m_heightData );
}

void CoarseGridHeightFieldDemo::lineOfSight(hkStopwatch& stopwatch)
{
	// Lets have some NPCs walking in circles on the height field
	hkPseudoRandomGenerator random(123);
	const int numNPCs = 20;
	hkVector4 positions[numNPCs];
	for (int i=0; i<numNPCs; i++) 
	{
		// To access the heightfields getHeightAndNormalAt method you need to work in local heightfield coordinate space
		hkReal x = (m_heightFieldShape->m_xRes-1)*hkMath::clamp(random.getRandReal01() + 0.1f*hkMath::sin(m_time * (0.25f+0.02f*i) + 0.3f*i), hkReal(0), hkReal(1));
		hkReal z = (m_heightFieldShape->m_zRes-1)*hkMath::clamp(random.getRandReal01() + 0.1f*hkMath::cos(m_time * (0.25f+0.02f*i) + 0.3f*i), hkReal(0), hkReal(1));
		hkVector4 normal; int triindex;
		hkReal height; 
		m_heightFieldShape->getHeightAndNormalAt((int)x, (int)z, x-(int)x, z-(int)z, normal, height, triindex);
		positions[i].set(x,height+1.3f,z);

		// Convert from heightfield coordinate space to global
		positions[i].mul4( m_heightFieldShape->m_intToFloatScale ); 

		hkVector4 ground; ground.set(x,height,z);
		ground.mul4( m_heightFieldShape->m_intToFloatScale ); 
		// Draw a red line representing the NPC
		HK_DISPLAY_LINE(ground, positions[i], hkColor::RED);
	}

	// For each pair do a raycast to determine line-of-sight.
	for (int i=0; i<numNPCs; i++) 
	{
		for (int j=i+1; j<numNPCs; j++) 
		{
			hkpWorldRayCastInput input;
			input.m_from = positions[i];
			input.m_to = positions[j];
			hkpClosestRayHitCollector output;
			stopwatch.start();
			m_world->castRay(input, output );
			stopwatch.stop();
			if (!output.hasHit()) 
			{
				// Draw a blue line between them if they have line-of-sight
				HK_DISPLAY_LINE( input.m_from, input.m_to, hkColor::LIGHTBLUE);
			}
		}
	}
}

void CoarseGridHeightFieldDemo::sweep(hkStopwatch& stopwatch)
{
	int numrays = 128;
	for (int i=0; i<numrays; i++) 
	{
		hkpWorldRayCastInput input;
		float delta = (m_heightFieldShape->m_xRes-1.0f)/numrays;
		input.m_from.set(delta*(i+0.5f), 20.0f, (m_heightFieldShape->m_zRes-1)*(0.5f+0.2f*hkMath::sin(m_time * 0.3f)));
		input.m_to.set(delta*(i+0.5f), -10.0f, (m_heightFieldShape->m_zRes-1)*(0.5f+0.65f*hkMath::sin(m_time * 0.3f)));
		input.m_from.mul4( m_heightFieldShape->m_intToFloatScale ); 
		input.m_to.mul4( m_heightFieldShape->m_intToFloatScale ); 

		hkpClosestRayHitCollector output;
		stopwatch.start();
		m_world->castRay(input, output );
		stopwatch.stop();

		// To visualise the raycast we make use of a macro defined in "hkDebugDisplay.h" called HK_DISPLAY_LINE.
		// The macro takes three parameters: a start point, an end point and the line color.
		// If a hit is found we display a RED line from the raycast start point to the point of intersection and mark that
		// point with a small RED cross. The intersection point is calculated using: startWorld + (result.m_mindist * endWorld).
		//
		// If no hit is found we simply display a GREY line between the raycast start and end points.

		if( output.hasHit() )
		{
			hkVector4 intersectionPointWorld;
			intersectionPointWorld.setInterpolate4( input.m_from, input.m_to, output.getHit().m_hitFraction );
			HK_DISPLAY_LINE( input.m_from, intersectionPointWorld, hkColor::RED);
			HK_DISPLAY_ARROW( intersectionPointWorld, output.getHit().m_normal, hkColor::CYAN);
		}
		else
		{
			// Otherwise draw as GREY
			HK_DISPLAY_LINE(input.m_from, input.m_to, hkColor::rgbFromChars(200, 200, 200));
		}
	}
}



hkDemo::Result CoarseGridHeightFieldDemo::stepDemo()
{

	if (m_env->m_window->getKeyboard().wasKeyPressed('N' ))
	{
		m_coarseness += 1;
		if ((1<<(m_coarseness+2))>=m_heightFieldShape->m_xRes)  // Only support reasonable coarseness levels
		{
			m_coarseness = 0;
		}
		m_heightFieldShape->buildCoarseMinMaxTree(m_coarseness);
	}

	if (m_env->m_window->getKeyboard().wasKeyPressed('M'))
	{
		m_demomode = (m_demomode+1)%2;
	}

	m_world->lock();
	m_time += float(m_timestep);

	hkStopwatch			stopwatch;
	if (m_demomode==0) 
	{
		sweep(stopwatch);
	}
	else 
	{
		lineOfSight(stopwatch);
	}
	m_world->unlock();

	// Info
	{
		char coarsebuf[1023];
		if (m_coarseness) 
		{
			hkString::sprintf( coarsebuf, "Coarseness (N) : %i (%i x %i)", m_coarseness, 1<<m_coarseness, 1<<m_coarseness); 
		}
		else
		{
			hkString::sprintf( coarsebuf, "Coarseness (N) : off"); 
		}
		char buf[1023];
		hkString::sprintf( buf, "%s \nRaycast time (ms): %.3f  \nMode (M): %s	", coarsebuf, 1000.0f*stopwatch.getElapsedSeconds(), m_demomode ? "Line-of-sight" : "Sweep");
		const int h = getWindowHeight();
		m_env->m_textDisplay->outputText( buf, 20, h-120, 0xffffffff, 1);
	}



	return hkDefaultPhysics2012Demo::stepDemo();
}



////////////////////////////////////////////////////////////////////

static const char helpString[] = \
"This demo shows how to use the optional coarse min-max quadtree on sampled heightfields to optimize raycasts.\n "
"Press n to toggle coarseness level. Press m to switch between sweep and line-of-sight demo.";

HK_DECLARE_DEMO(CoarseGridHeightFieldDemo, HK_DEMO_TYPE_PHYSICS_2012, "Using a coarse min-max quadtree on a heightfield", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
