/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Demos/Physics2012/Api/Collide/Shapes/Convex/ConvexShapesDemo.h>

#include <Physics2012/Collide/Shape/Misc/Bv/hkpBvShape.h>
#include <Physics2012/Collide/Shape/Convex/Triangle/hkpTriangleShape.h>
#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Common/Internal/ConvexHull/hkGeometryUtility.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpConvexRadiusViewer.h>


struct ShapeVariant
{
	const char*	m_name;
	hkpShapeType m_shapeType;
	const char* m_details;
};

static const ShapeVariant g_variants[] =
{
	{ "Box", hkcdShapeType::BOX, "A single Box shape. The green geometry represents the convex 'shell'." },
	{ "Sphere", hkcdShapeType::SPHERE, "A single Sphere shape. Note that the tessellation is only in the display mesh." },
	{ "Triangle", hkcdShapeType::TRIANGLE, "A single Triangle shape. The green geometry represents the convex 'shell'." },
	{ "Capsule", hkcdShapeType::CAPSULE, "A single Capsule shape. Note that the tessellation is only in the display mesh." },
	{ "Cylinder", hkcdShapeType::CYLINDER, "A single Cylinder shape. The green geometry represents the convex 'shell'. Note that the tessellation is only in the display mesh." },
	{ "Convex Vertices", hkcdShapeType::CONVEX_VERTICES, "A single Convex Vertices shape. The green geometry represents the convex 'shell'." }
};


ConvexShapesDemo::ConvexShapesDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{
	const ShapeVariant& variant =  g_variants[m_variantId];

	m_debugViewerNames.pushBack( hkpConvexRadiusViewer::getName() );

	// Setup the camera.
	{
		hkVector4 from(0.0f, 5.0f, 10.0f);
		hkVector4 to  (0.0f, 0.0f,  0.0f);
		hkVector4 up  (0.0f, 1.0f,  0.0f);
		setupDefaultCameras(env, from, to, up);
	}

	// Create the world, setting gravity to zero so body floats.
	hkpWorldCinfo info;
	info.m_gravity.set(0.0f, 0.0f, 0.0f);	
	info.setBroadPhaseWorldSize( 100.0f );
	m_world = new hkpWorld(info);
	m_world->lock();

	setupGraphics();

	// Create the shape
	hkpShape* shape = HK_NULL;
	switch ( variant.m_shapeType )
	{

		// Box
	case hkcdShapeType::BOX:
		{
			hkVector4 halfExtents( 1.0f, 1.0f, 1.0f );
			hkReal convexRadius = 0.05f;

			shape = new hkpBoxShape( halfExtents, convexRadius );

			break;
		}


		// Sphere
	case hkcdShapeType::SPHERE:
		{
			hkReal radius = 1.75f;

			shape = new hkpSphereShape( radius );

			break;
		}


		// Triangle
	case hkcdShapeType::TRIANGLE:
		{
			// Disable backface culling in the HKG renderer
			disableBackFaceCulling();

			float vertices[] = {
				-0.5f, -0.5f,  0.0f, 0.0f, // v0
				0.5f, -0.5f,  0.0f, 0.0f, // v1
				0.0f,  0.5f,  0.0f, 0.0f, // v2
			};

			hkReal convexRadius = 0.05f;
			shape = new hkpTriangleShape( convexRadius );

			int index = 0;
			for (int i = 0; i < 3; i++)
			{
				static_cast<hkpTriangleShape*>(shape)->setVertex(i, hkVector4(vertices[index], vertices[index + 1], vertices[index + 2]));
				index = index + 4;
			}

			break;
		}


		// Capsule
	case hkcdShapeType::CAPSULE:
		{
			hkVector4 top(0.0f, 1.5f, 0.0f);
			hkVector4 bottom(0.0f, -1.0f, 0.0f);
			hkReal radius = 1.5f;

			shape = new hkpCapsuleShape( top, bottom, radius );

			break;
		}


		// Cylinder
	case hkcdShapeType::CYLINDER:
		{
			hkVector4 top(0.0f, 1.5f, 0.0f);
			hkVector4 bottom(0.0f, -1.0f, 0.0f);
			hkReal radius = 1.5f;
			hkReal convexRadius = 0.1f;

			shape = new hkpCylinderShape( top, bottom, radius, convexRadius );

			break;
		}


		// Convex vertices.
		// Note: there are several hkpConvexVerticesShape constructors.
		// See the comments in its header file for details.
	case hkcdShapeType::CONVEX_VERTICES:
		{
			// Data specific to this shape.
			int numVertices = 4;

			// 16 = 4 (size of "each float group", 3 for x,y,z, 1 for padding) * 4 (size of float)
			int stride = sizeof(hkReal) * 4;

			hkReal vertices[] = { // 4 vertices plus padding
				-2.0f, 2.0f, 1.0f, 0.0f, // v0
				1.0f, 3.0f, 0.0f, 0.0f, // v1
				0.0f, 1.0f, 3.0f, 0.0f, // v2
				1.0f, 0.0f, 0.0f, 0.0f  // v3
			};

			hkStridedVertices stridedVerts;
			{
				stridedVerts.m_numVertices = numVertices;
				stridedVerts.m_striding = stride;
				stridedVerts.m_vertices = vertices;
			}

			// Configure a large convex radius, for illustration.
			// The vertices will be shrunk by this radius, but the overall "shell" will be maintained.
			hkpConvexVerticesShape::BuildConfig config;
			config.m_maxShrinkingVerticesDisplacement = 1.0f;
			config.m_maxRelativeShrink = 1.0f;
			config.m_convexRadius = 0.2f;

			shape = new hkpConvexVerticesShape( stridedVerts, config );

			break;
		}

	default:
		break;
	}

	// Make sure that a shape was created
	HK_ASSERT(0, shape);

	// To illustrate using the shape, create a rigid body for it
	{
		// First define a rigid body template
		hkpRigidBodyCinfo rigidBodyInfo;
		{
			rigidBodyInfo.m_position.set(0.0f, 0.0f, 0.0f);
			rigidBodyInfo.m_angularDamping = 0.0f;
			rigidBodyInfo.m_linearDamping = 0.0f;

			rigidBodyInfo.m_shape = shape;

			// Compute the rigid body inertia.
			rigidBodyInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
			hkpInertiaTensorComputer::setShapeVolumeMassProperties( rigidBodyInfo.m_shape, 100.0f, rigidBodyInfo );
		}

		// Create a rigid body (using the template above).
		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);

		// Remove reference since the body now "owns" the Shape.
		shape->removeReference();

		// Finally add body so we can see it, and remove reference since the world now "owns" it.
		m_world->addEntity(rigidBody);
		rigidBody->removeReference();
	}

	m_world->unlock();
}



static const char helpString[] = "Examples of convex primitive shapes";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( ConvexShapesDemo, HK_DEMO_TYPE_PHYSICS_2012, ShapeVariant, g_variants, helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
