/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/CompressedAndExtendedMeshes/CompressedMeshDemo.h>

#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GeometryLoaderUtil.h>

#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Scene/hkxSceneUtils.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Physics2012/Collide/Util/hkpTriangleUtil.h>
#include <Physics2012/Collide/Shape/Compound/Collection/StorageExtendedMesh/hkpStorageExtendedMeshShape.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppUtility.h>
#include <Physics2012/Collide/Shape/Compound/Collection/Mesh/hkpNamedMeshMaterial.h>
#include <Physics2012/Collide/Shape/Query/hkpShapeRayCastInput.h>
#include <Physics2012/Collide/Shape/Query/hkpShapeRayCastOutput.h>
#include <Physics2012/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobQueueUtils.h>
#include <Physics2012/Collide/Query/Multithreaded/RayCastQuery/hkpRayCastQueryJobs.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastOutput.h>
#include <Physics2012/Collide/Util/Welding/hkpMeshWeldingUtility.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/hkpShapeDisplayBuilder.h>
#include <Physics2012/Collide/Query/Collector/PointCollector/hkpRootCdPoint.h>
#include <Physics2012/Collide/Query/Multithreaded/CollisionQuery/hkpCollisionQueryJobQueueUtils.h>
#include <Physics2012/Collide/Query/Collector/PointCollector/hkpFixedBufferCdPointCollector.h>

#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>


enum CompressedMeshDemoVariantType
{
	CASTLE_CMS,
	CASTLE_EMS,
	SIMPLE_TEST,
	LOAD_CMS,
	LOAD_EMS
};

struct CompressedMeshDemoVariant
{
	const char*	 m_name;
	CompressedMeshDemoVariantType  m_type;
	const char* m_details;
};


//static const char helpString[] = "Compressed mesh shape divided into chunks and with quantized vertices. "
static const CompressedMeshDemoVariant g_variants[] =
{
	{ "Create (CMS)", CASTLE_CMS, "A landscape scene imported and converted to compressed mesh format. "
									"You can see the geometry of the chunks in different colors to the left. "
									"The left over (big) triangles are colored in red. "
									"C : show chunks\n"
									"L : wireframe mode\n"
									"T : show T-junctions\n"
									"R : cast rays"},
	{ "Create (EMS)", CASTLE_EMS, "A landscape scene imported and converted to extended mesh format."},
	{ "Load (CMS)", LOAD_CMS, "A compressed mesh shape loaded from file."},
	{ "Load (EMS)", LOAD_EMS, "An extended mesh shape loaded from file."},
	{ "Simple test", SIMPLE_TEST, "A simple compressed mesh shape made up of a ground mesh and a convex piece. "
								"Every other triangle has a different material and it's being used as a collision filter so the box can fall through. "
								"Press R to cast rays that show the material info"}
};

static const char* g_fileNames[] =
{
	"Resources/Common/Landscapes/Sandcastle_9.hkt",
};

static int g_landscapeIndex = 0;

#define CLIENT_LANDSCAPE g_fileNames[g_landscapeIndex]
#define CMS_LANDSCAPE "Resources/Common/Landscapes/Sandcastle_9_cms.hkt"
#define EMS_LANDSCAPE "Resources/Common/Landscapes/Sandcastle_9_ems.hkt"
//#define HIDE_BOXES

//	# of simulated SPUs
#	define NUM_SPUS 1

#define GROUND_LAYER 1
#define BOX_LAYER 2

static hkpMoppBvTreeShape* createMoppShape(hkpShapeCollection* meshShape, bool useChunks)
{
	hkpMoppCompilerInput moppInput;
	moppInput.m_enableChunkSubdivision = useChunks;

	// Build the code at runtime
	hkpMoppCode* code = hkpMoppUtility::buildCode( meshShape, moppInput );

	hkpMoppBvTreeShape* moppShape = HK_NULL;
	moppShape = new hkpMoppBvTreeShape( meshShape, code );

	//hkpMeshWeldingUtility::computeWeldingInfo( meshShape, moppShape, hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE );

	meshShape->removeReference();
	code->removeReference();

	return moppShape;
}

CompressedMeshDemo::CompressedMeshDemo(hkDemoEnvironment* env) : hkDefaultPhysics2012Demo(env),
	m_out( "compressed_mesh.log" ),
	m_semaphore( 0, 1000 ),
	m_compMesh( HK_NULL ),
	m_loadedData(HK_NULL),
	wireframe( false ),
	showWelded( false ),
	showTJ( false ),
	m_showChunks( false ),
	m_showAabb( false ),
	m_showOriginalGeometry( false ),
	m_showBreakdown( true ),
	m_random( 15 )
{
	const CompressedMeshDemoVariant& variant =  g_variants[m_variantId];

	//g_landscapeIndex++;
	const int numLandscapes = sizeof(g_fileNames) / sizeof(char*);
	if ( g_landscapeIndex == numLandscapes )
	{
		g_landscapeIndex = 0;
	}

	//
	// Setup the camera
	//
	{
		hkVector4 from(55.0f, 50.0f, 55.0f);
		hkVector4 to  ( 0.0f,  0.0f,   0.0f);
		hkVector4 up  ( 0.0f,  1.0f,   0.0f);
		setupDefaultCameras(env, from, to, up, 0.1f, 20000.0f);
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo worldInfo;
		{
			worldInfo.m_gravity.set(0.0f, 0.0f, -9.81f);
			worldInfo.setBroadPhaseWorldSize(10000.0f);
			worldInfo.setupSolverInfo( hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM );
			worldInfo.m_enableDeactivation = false;
		}
		m_world = new hkpWorld(worldInfo);
		m_world->lock();


		// Register ALL agents (though some may not be necessary)
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		setupGraphics();
	}

	if ( variant.m_type == LOAD_CMS || variant.m_type == LOAD_EMS )
	{
		// Load the file
		hkSerializeUtil::ErrorDetails loadError;
		m_loadedData = hkSerializeUtil::load( variant.m_type == LOAD_CMS ? CMS_LANDSCAPE : EMS_LANDSCAPE, &loadError );
		HK_ASSERT3(0xa6451543, m_loadedData != HK_NULL, "Could not load file. The error is:\n" << loadError.defaultMessage.cString() );

		// Get the top level object in the file, which we know is a hkRootLevelContainer
		hkRootLevelContainer* container = m_loadedData->getContents<hkRootLevelContainer>();
		HK_ASSERT2(0xa6451543, container != HK_NULL, "Could not load root level obejct" );

		// Get the physics data
		hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
		HK_ASSERT2(0xa6451544, physicsData != HK_NULL, "Could not find physics data in root level object" );
		HK_ASSERT2(0xa6451535, physicsData->getWorldCinfo() != HK_NULL, "No physics cinfo in loaded file - cannot create a hkpWorld" );

		// Add all the physics systems to the world
		for ( int i = 0; i < physicsData->getPhysicsSystems().getSize(); ++i )
		{
			m_world->addPhysicsSystem( physicsData->getPhysicsSystems()[i] );
		}

		m_terrainBody = physicsData->getPhysicsSystems()[0]->getRigidBodies()[0];
		m_terrainShape = m_terrainBody->getCollidable()->getShape();
		m_terrainShape->getAabb( m_terrainBody->getTransform(), 0.1f, m_bounds );
		if ( m_terrainShape->getType() == hkcdShapeType::MOPP )
		{
			m_terrainShape = ((hkpMoppBvTreeShape*)m_terrainShape)->getChild();
		}
	}
	else
	{
		m_terrainShape = createMeshShape( env );
		m_shape = createMoppShape( (hkpShapeCollection*)m_terrainShape, true );
		m_shape->getAabb( hkTransform::getIdentity(), 0.1f, m_bounds );

		hkAabb aabb = m_originalAabb;
		aabb.expandBy( 0.5f );
		if ( !aabb.contains( m_bounds ) )
		{
			m_out << "AABB not OK!" << '\n';
			m_out.flush();
		}

#	ifndef HIDE_GROUND
		//
		//  create the ground mopp
		//
		{

			hkpRigidBodyCinfo terrainInfo;
			{
				hkVector4 size(100.0f, 1.0f, 100.0f);

				terrainInfo.m_shape = m_shape;
				terrainInfo.m_motionType = hkpMotion::MOTION_FIXED;
				terrainInfo.m_friction = 0.5f;

				m_terrainBody = new hkpRigidBody( terrainInfo );
				m_world->addEntity( m_terrainBody );
				m_terrainBody->removeReference();
			}
			terrainInfo.m_shape->removeReference();
		}
#	endif // HIDE_GROUND
	}

	hkVector4 halfExtents;
	hkVector4 centre;
	{
		m_bounds.getHalfExtents( halfExtents );
		m_bounds.getCenter( centre );

		m_minIndex = (halfExtents(0) < halfExtents(1)) ?
			((halfExtents(0) < halfExtents(2)) ? 0 : 2) :
			((halfExtents(1) < halfExtents(2)) ? 1 : 2) ;
	}

	// Setup the camera
	{
		hkVector4 up; up.setZero4();
		up(m_minIndex) = 1.0f;

		hkVector4 from; from.setAddMul4( m_bounds.m_max, up, 5.0f);
		setupDefaultCameras(env, from, centre, up, 0.1f, 10000.0f);
	}

#	ifndef HIDE_BOXES
	//
	// Create objects at random coordinates
	//
	{
		hkVector4 size; size.setAll( 0.5f );
		hkpBoxShape* boxShape =  new hkpBoxShape(size);

		hkpRigidBodyCinfo boxInfo;
		{
			boxInfo.m_mass = 1.0f;
			boxInfo.m_shape = boxShape;
			boxInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;
		}

		//
		// create the boxes
		//
		{
			hkVector4 range; range.setSub4( m_bounds.m_max, m_bounds.m_min );
			range.mul4( 1.0f );
			int numObjects = 200;
			for (int i = 0; i < numObjects; i++)
			{
				hkVector4 pos;
				m_random.getRandomVector11(pos);
				pos.mul4(0.3f);
				pos.mul4(range);
				pos( m_minIndex ) += m_bounds.m_max( m_minIndex ) + i * (range(m_minIndex)) / numObjects + 10;
				pos.add4( centre );

				hkpRigidBody* body;
				{
					boxInfo.m_position = pos;
					body = new hkpRigidBody( boxInfo );
				}

				body->setMaxLinearVelocity( 30 );
				m_world->addEntity(body);
				body->removeReference();
			}

		}

		boxShape->removeReference();
	}
#	endif // HIDE_BOXES

	// setup the linear	cast

	/*hkVector4 boxSize1(1,1,2);
	hkpBoxShape* listShape = new hkpBoxShape(boxSize1);
	//
	// Create a the body that we'll linear cast
	//
	{
		hkVector4 position(0.0f, 0.0f, 80.0f);
		hkpBoxShape* listShape = new hkpBoxShape(boxSize1);
		m_castBody = GameUtils::createRigidBody(listShape, 0.0f, position);
		m_world->addEntity(m_castBody);
		m_castBody->removeReference();
		listShape->removeReference();
	}

	//
	// Some magic to create a second graphical representation of the cast body.
	//
	{
		hkArray<hkDisplayGeometry*> geom;
		hkpShapeDisplayBuilder::hkpShapeDisplayBuilderEnvironment sdbe;
		hkpShapeDisplayBuilder builder(sdbe);
		builder.buildShapeDisplay( m_castBody->getCollidable()->getShape(), hkTransform::getIdentity(), geom );
		m_env->m_displayHandler->addGeometry(geom, m_castBody->getCollidable()->getTransform(), 1 , 0, 0 );
		hkReferencedObject::removeReferences(geom.begin(), geom.getSize());

	}*/

	m_world->unlock();

	m_time = 0.0f;

	//
	// Setup multithreading.
	//
	hkpRayCastQueryJobQueueUtils::registerWithJobQueue( m_jobQueue );

	// Register the collision query functions
	hkpCollisionQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);

	// register the default addCdPoint() function; you are free to register your own implementation here though
	hkpFixedBufferCdPointCollector::registerDefaultAddCdPointFunction();

	// we do not allow the # of active SPUs to drop to zero as this can cause a deadlock.
	m_allowZeroActiveSpus = false;
}

CompressedMeshDemo::~CompressedMeshDemo()
{
	if ( m_world )
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}

	if ( m_loadedData )
	{
		m_loadedData->removeReference();
		m_loadedData = HK_NULL;
	}
}

void CompressedMeshDemo::testLinearCast()
{
	//
	// Setup the output array where the resulting collision points will be returned.
	//
	hkpRootCdPoint* collisionPoints = hkAllocateChunk<hkpRootCdPoint>(1, HK_MEMORY_CLASS_DEMO);

	//
	// Setup the hkpPairLinearCastCommand command.
	//
	hkpPairLinearCastCommand* command;
	{
		command = hkAllocateChunk<hkpPairLinearCastCommand>(1, HK_MEMORY_CLASS_DEMO);

		// Init input data.
		{
			// hkpWorldObject::getCollidable() needs a read-lock on the object
			m_castBody->markForRead();
			m_terrainBody->markForRead();

			command->m_collidableA = m_castBody->getCollidable();
			command->m_collidableB = m_terrainBody->getCollidable();

			m_terrainBody->unmarkForRead();
			m_castBody->unmarkForRead();

			// We perform a linear cast of the first object moving towards the second.
			command->m_from = m_castBody->getPosition();
			command->m_to = m_terrainBody->getPosition();

			//
			// Move around target position.
			//
			hkReal xShift;
			hkReal yShift;
			{
				m_time += m_timestep;
				xShift = 5.5f * hkMath::sin(m_time * 1.0f);
				yShift = 5.5f * hkMath::cos(m_time * 2.0f);
			}
			command->m_to(0) += xShift;
			command->m_to(1) += yShift;
			command->m_to(2) -= 5.0f;
		}

		// Init output data.
		{
			command->m_results			= collisionPoints;
			command->m_resultsCapacity	= 1;
			command->m_numResultsOut	= 0;

			command->m_startPointResults = HK_NULL;
			command->m_startPointResultsCapacity = 0;
			command->m_startPointNumResultsOut = 0;
		}
	}

	//
	// create the job header
	//
	hkpCollisionQueryJobHeader* jobHeader;
	{
		jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	}

	//
	// Setup hkpPairLinearCastJob.
	//
	hkReal queryTolerance = 1.0f;
	m_world->markForRead();
	hkpPairLinearCastJob pairLinearCastJob(m_world->getCollisionInput(), jobHeader, command, 1, m_world->m_collisionInput->m_filter, queryTolerance, &m_semaphore);
	m_world->unmarkForRead();

	//
	// Put the job on the queue, kick-off the PPU/SPU threads and wait for everything to finish.
	//
	{
		//
		// Put the pairGetClosestPointsJob on the job queue.
		//
		pairLinearCastJob.setRunsOnSpuOrPpu();
		m_jobQueue->addJob( pairLinearCastJob, hkJobQueue::JOB_LOW_PRIORITY );

		m_threadPool->processJobQueue( m_jobQueue );
		m_threadPool->waitForCompletion();

		//
		// Wait for the one single job we started to finish.
		//
		m_semaphore.acquire();

	}

	//
	// Display results.
	//
	{
		hkVector4 from	= command->m_from;
		hkVector4 to	= command->m_to;
		hkVector4 path;
		path.setSub4(to, from);

		if ( command->m_numResultsOut > 0 )
		{
			for ( int r = 0; r < command->m_numResultsOut; r++)
			{
				const hkContactPoint& contactPoint = command->m_results[r].m_contact;

				hkReal fraction = contactPoint.getDistance();

				// Calculate the position on the linear cast's path where the first collidable hit the second collidable.
				hkVector4 pointOnPath;
				{
					pointOnPath.setAddMul4( from, path, fraction );
				}

				// Draw the linear cast line from startpoint to hitpoint.
				HK_DISPLAY_LINE( from, pointOnPath, hkColor::RED );

				// Draw the collision normal.
				HK_DISPLAY_ARROW( pointOnPath, contactPoint.getNormal(), hkColor::BLUE );

				// Draw the linear cast line from hitpoint to endpoint.
				HK_DISPLAY_LINE( pointOnPath, to, hkColor::BLACK );

				//
				// Display sphere's position at 'time of collision'.
				//
				{
					hkTransform castTransform = command->m_collidableA->getTransform();
					castTransform.setTranslation(pointOnPath);
					m_env->m_displayHandler->updateGeometry(castTransform, 1, 0);
				}
			}
		}
		else
		{
			// Display the whole linear cast line.
			HK_DISPLAY_LINE( from, to, hkColor::BLACK );

			// Display the casted collidable at cast's endpoint.
			{
				hkTransform castTransform = command->m_collidableA->getTransform();
				m_env->m_displayHandler->updateGeometry(castTransform, 1, 0);
			}
		}
	}

	//
	// Free temporarily allocated memory.
	//
	hkDeallocateChunk(jobHeader,       1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(command,         1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(collisionPoints, 1, HK_MEMORY_CLASS_DEMO);
}

void CompressedMeshDemo::randomRayCastScene()
{
	//	const MoppMultithreadedDemoVariant& variant = g_MoppMultithreadedDemoVariants[m_variantId];

	// Some bounding information
	hkVector4 halfExtents;
	hkVector4 center;
	int minIndex;
	{
		{
			halfExtents.setSub4( m_bounds.m_max, m_bounds.m_min );
			halfExtents.mul4( 0.5f );
			center.setAdd4( m_bounds.m_max, m_bounds.m_min );
			center.mul4( 0.5f );
			minIndex = (halfExtents(0) < halfExtents(1)) ?
				((halfExtents(0) < halfExtents(2)) ? 0 : 2) :
				((halfExtents(1) < halfExtents(2)) ? 1 : 2) ;
		}
	}

	static hkReal time = 0.0f;
	time += m_timestep;

	const int numCommands = 200;

	hkArray<hkpShapeRayCastCommand>	commands(numCommands);
	hkArray<hkpWorldRayCastOutput>	outputs (numCommands);

	// hkpWorldObject::getCollidable() needs a read-lock on the object
	m_terrainBody->markForRead();

	// Not a true array, just a fake one. :)
	const hkpCollidable** collidableArray = hkAlignedAllocate<const hkpCollidable*>(16, 1, HK_MEMORY_CLASS_DEMO);
	collidableArray[0] = m_terrainBody->getCollidable();

	m_terrainBody->unmarkForRead();

	//
	// Setup the raycast commands. Each command represents one ray cast against our MOPP.
	//
	hkPseudoRandomGenerator rand(42);
	for (int i=0; i < numCommands; i++)
	{
		hkpShapeRayCastCommand& command = commands[i];

		// Setup the input
		{
			// init shape data
			{
				command.m_collidables		= collidableArray;
				command.m_numCollidables	= 1;
			}

			// init ray data
			{
				// from & to
				{
					hkVector4 offset;
					{
						offset.set( hkMath::sin(time), hkMath::cos(time), hkMath::sin(time));
						offset.mul4( 20.0f );
					}

					rand.getRandomVector11( command.m_rayInput.m_from );
					rand.getRandomVector11( command.m_rayInput.m_to );
					command.m_rayInput.m_from.mul4( halfExtents );
					command.m_rayInput.m_to.mul4( halfExtents );
					command.m_rayInput.m_from.add4( center );
					command.m_rayInput.m_to.add4( center );
					command.m_rayInput.m_to.add4(offset);
					command.m_rayInput.m_to( minIndex ) = m_bounds.m_min(minIndex) - halfExtents(minIndex) * 2;
					command.m_rayInput.m_from( minIndex ) = m_bounds.m_max(minIndex) + halfExtents(minIndex) * 2;
				}
				command.m_rayInput.m_rayShapeCollectionFilter	= HK_NULL;
				command.m_rayInput.m_filterInfo					= 0;
				command.m_filterType = hkpCollisionFilter::HK_FILTER_UNKNOWN;
				command.m_filterSize = 0;

			}

			// init output struct
			{
				command.m_results			= &outputs[i];
				command.m_resultsCapacity	= 1;
				command.m_numResultsOut	= 0;
			}
		}

		outputs[i].reset();
	}

	//
	// create the job header
	//
	hkpCollisionQueryJobHeader* jobHeader;
	{
		jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	}

	m_world->markForRead();
	//
	// setup hkpShapeRayCastJob
	//
	hkpShapeRayCastJob shapeRayCastJob(m_world->getCollisionInput(), jobHeader, commands.begin(), numCommands, &m_semaphore);
	m_world->unmarkForRead();

	//
	// Put the job on the queue, kick-off the PPU/SPU threads and wait for everything to finish.
	//
	{
		//
		// Put the raycast on the job queue.
		//

		shapeRayCastJob.setRunsOnSpuOrPpu();
		m_jobQueue->addJob( shapeRayCastJob, hkJobQueue::JOB_LOW_PRIORITY );

		m_threadPool->processJobQueue( m_jobQueue );

		m_threadPool->waitForCompletion();

		//
		// Wait for the one single job we started to finish.
		//
		m_semaphore.acquire();

	}

	//
	// Display results.
	//
	{
		for (int r=0; r < numCommands; r++)
		{
			hkpShapeRayCastCommand& command = commands[r];

			// Check the hit.
			if ( command.m_numResultsOut > 0 )
			{
				hkVector4 intersectionPointWorld;
				intersectionPointWorld.setInterpolate4( command.m_rayInput.m_from, command.m_rayInput.m_to, outputs[r].m_hitFraction );

				//	Display the raycast, hit and normal
				HK_DISPLAY_LINE( command.m_rayInput.m_from, intersectionPointWorld, hkColor::LEMONCHIFFON );
				HK_DISPLAY_ARROW( intersectionPointWorld, outputs[r].m_normal, hkColor::GAINSBORO );

				hkStringBuf text;
				hkpShapeKey key = outputs[r].m_shapeKeys[0];
				const hkpMeshMaterial* mat = HK_NULL;
				bool isNamedMaterial = false;
				if ( m_terrainShape->getType() == hkcdShapeType::COMPRESSED_MESH )
				{
					hkpCompressedMeshShape* cms = (hkpCompressedMeshShape*)m_terrainShape;
					mat = cms->getMaterial( key );
					isNamedMaterial = cms->m_namedMaterials.getSize() != 0;
				}
				if ( m_terrainShape->getType() == hkcdShapeType::EXTENDED_MESH )
				{
					hkpExtendedMeshShape* ems = (hkpExtendedMeshShape*)m_terrainShape;
					mat = ems->getMeshMaterial( key );
					isNamedMaterial = true;
				}
				if ( mat == HK_NULL )
				{
					text.appendPrintf( "%x", key );
				}
				else if ( isNamedMaterial )
				{
					hkpNamedMeshMaterial* nameMat = (hkpNamedMeshMaterial*)mat;
					text = nameMat->m_name;
				}
				else
				{
					text.appendPrintf( "%d", mat->m_filterInfo );
				}
				HK_DISPLAY_3D_TEXT( text.cString(), intersectionPointWorld, hkColor::SALMON );
			}
			else
			{
				HK_DISPLAY_LINE( command.m_rayInput.m_from, command.m_rayInput.m_to, hkColor::DARKORCHID );
			}
		}
	}

	hkAlignedDeallocate(collidableArray);
	hkDeallocateChunk(jobHeader, 1, HK_MEMORY_CLASS_DEMO);
}

static bool rayCast = false;
static hkVector4 rayTarget;


void CompressedMeshDemo::testRayCast()
{
	//
	// Get the ground's collidable.
	//
	const hkpCollidable* groundCollidable;
	{
		// hkpWorldObject::getCollidable() needs a read-lock on the object
		m_terrainBody->markForRead();
		groundCollidable = m_terrainBody->getCollidable();
		m_terrainBody->unmarkForRead();
	}

	//
	// Setup the array of collidables which we want to cast our ray against.
	// In this case we only have one collidable.
	//
	const hkpCollidable** collidableArray;
	{
		collidableArray	   = hkAlignedAllocate<const hkpCollidable*>(16, 1, HK_MEMORY_CLASS_DEMO);
		collidableArray[0] = groundCollidable;
	}

	//
	// Setup the output array where the ray's results (in this case only one) will be returned.
	//
	hkpWorldRayCastOutput* rayOutput = hkAllocateChunk<hkpWorldRayCastOutput>(1, HK_MEMORY_CLASS_DEMO);

	//
	// Setup the raycast command.
	//
	const int numCmds = 1;
	hkpShapeRayCastCommand* command;
	{
		command = hkAllocateChunk<hkpShapeRayCastCommand>(numCmds, HK_MEMORY_CLASS_DEMO);

		hkpShapeRayCastCommand* oldCmd = command;
		for ( int i = 0; i < numCmds; ++i )
		{

			// Init shape data.
			{
				command->m_collidables	  = collidableArray;
				command->m_numCollidables = 1;
			}

			hkVector4 target;
			m_random.getRandomVectorRange( m_bounds.m_min, m_bounds.m_max, target );
			target(m_minIndex) = m_bounds.m_min(m_minIndex);

			// Init ray data.
			{
				command->m_rayInput.m_from					   . setZero4();
				command->m_rayInput.m_from(m_minIndex) = m_bounds.m_max(m_minIndex) + 10;
				//command->m_rayInput.m_to					   . set(0.0f, 0.0f, 0.0f);
				command->m_rayInput.m_to = target;
				command->m_rayInput.m_rayShapeCollectionFilter = HK_NULL;
				command->m_rayInput.m_filterInfo			   = 0;
				command->m_filterType = hkpCollisionFilter::HK_FILTER_UNKNOWN;
				command->m_filterSize = 0;
			}

			// Init output struct.
			{
				command->m_results		   = rayOutput;
				command->m_resultsCapacity = 1;
				command->m_numResultsOut   = 0;
			}

			command++;
		}

		command = oldCmd;

	}

	//
	// Create the job header.
	//
	hkpCollisionQueryJobHeader* jobHeader;
	{
		jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	}

	//
	// Setup the raycast job.
	//
	m_world->markForRead();
	hkpShapeRayCastJob shapeRayCastJob(m_world->getCollisionInput(), jobHeader, command, numCmds, &m_semaphore);
	m_world->unmarkForRead();

	//
	// Put the job on the queue, kick-off the PPU/SPU threads and wait for everything to finish.
	//
	{
		//
		// Put the raycast on the job queue.
		//
		shapeRayCastJob.setRunsOnSpuOrPpu();
		m_jobQueue->addJob( shapeRayCastJob, hkJobQueue::JOB_LOW_PRIORITY );

		m_threadPool->processJobQueue( m_jobQueue );
		m_jobQueue->processAllJobs();

		m_threadPool->waitForCompletion();

		//
		// Wait for the one single job we started to finish.
		//
		m_semaphore.acquire();

	}

	//
	// Output results.
	//
	{
		// Display results (just one in this case).
		if ( command->m_numResultsOut > 0 )
		{
			hkpWorldRayCastOutput* output = &command->m_results[0];

			hkVector4 intersectionPointWorld;
			intersectionPointWorld.setInterpolate4(command->m_rayInput.m_from, command->m_rayInput.m_to, output->m_hitFraction );

			// Display hitting ray in RED.
			HK_DISPLAY_LINE(command->m_rayInput.m_from, intersectionPointWorld, hkColor::RED);

			// Display a small RED cross at the point of intersection.
			{
				hkVector4 p = intersectionPointWorld;
				hkVector4 q = intersectionPointWorld;
				p(0) -= 0.2f;
				q(0) += 0.2f;
				HK_DISPLAY_LINE(p, q, hkColor::RED);
				p = intersectionPointWorld;
				q = intersectionPointWorld;
				p(1) -= 0.2f;
				q(1) += 0.2f;
				HK_DISPLAY_LINE(p, q, hkColor::RED);
			}

			// Display hit normal.
			HK_DISPLAY_ARROW( intersectionPointWorld, output->m_normal, hkColor::CYAN );
		}
		else
		{
			// Draw infinite ray as GREY.
			HK_DISPLAY_LINE(command->m_rayInput.m_from, command->m_rayInput.m_to, hkColor::rgbFromChars(200, 200, 200));
		}
	}

	//
	// Free temporarily allocated memory.
	//
	hkDeallocateChunk(jobHeader, 1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(command,   1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(rayOutput, 1, HK_MEMORY_CLASS_DEMO);
	hkAlignedDeallocate(collidableArray);
}

hkDemo::Result CompressedMeshDemo::stepDemo()
{
	if ( m_showChunks && m_compMesh )
	{
		hkAabb bounds;
		for ( int i = 0; i < m_compMesh->m_chunks.getSize(); ++i )
		{
			m_compMesh->getChunkAabb( m_compMesh->m_chunks[i], bounds );
			bounds.m_max.add4( m_geometryTranslation );
			bounds.m_min.add4( m_geometryTranslation );
			if ( m_compMesh->m_chunks[i].m_reference != HK_CMS_NULL_REF )
			{
				HK_DISPLAY_BOUNDING_BOX( bounds, hkColor::YELLOW );
			}
			else
			{
				HK_DISPLAY_BOUNDING_BOX( bounds, hkColor::WHITE );
			}
		}
		for ( int i = 0; i < m_compMesh->m_convexPieces.getSize(); ++i )
		{
			m_compMesh->getConvexPieceAabb( m_compMesh->m_convexPieces[i], bounds );
			bounds.m_max.add4( m_geometryTranslation );
			bounds.m_min.add4( m_geometryTranslation );
			if ( m_compMesh->m_convexPieces[i].m_reference != HK_CMS_NULL_REF )
			{
				HK_DISPLAY_BOUNDING_BOX( bounds, hkColor::YELLOW );
			}
			else
			{
				HK_DISPLAY_BOUNDING_BOX( bounds, hkColor::WHITE );
			}
		}
	}

	if ( m_showAabb )
	{
		HK_DISPLAY_BOUNDING_BOX( m_originalAabb, hkColor::GREEN );
		HK_DISPLAY_BOUNDING_BOX( m_bounds, hkColor::ORANGE );
	}

	if ( rayCast )
	{
		//testRayCast();
		//testLinearCast();
		randomRayCastScene();
	}

	if ( showTJ )
	{
		for ( int i = 0; i < m_builder.m_Tjunctions.getSize(); ++i )
		{
			const hkTjunctionDetector::ProximityInfo& info = m_builder.m_Tjunctions[i];

			const hkVector4& Tvertex = info.m_vertex;

			const hkVector4& v0 = info.m_v0;
			const hkVector4& v1 = info.m_v1;
			const hkVector4& v2 = info.m_v2;

			if ( info.m_type == hkTjunctionDetector::NEAR_FACE )
			{
				HK_DISPLAY_TRIANGLE( v0, v1, v2, hkColor::GREEN );
				HK_DISPLAY_LINE( v0, v1, hkColor::GREEN );
				HK_DISPLAY_LINE( v1, v2, hkColor::GREEN );
				HK_DISPLAY_LINE( v2, v0, hkColor::GREEN );
				HK_DISPLAY_STAR( Tvertex, 0.3f, hkColor::YELLOW );
			}
			else
			{
				HK_DISPLAY_STAR( Tvertex, 0.3f, hkColor::RED );
			}

			if ( info.m_type == hkTjunctionDetector::NEAR_EDGE0 )
			{
				HK_DISPLAY_LINE( v0, v1, hkColor::BLUE );
			}

			if ( info.m_type == hkTjunctionDetector::NEAR_EDGE1 )
			{
				HK_DISPLAY_LINE( v1, v2, hkColor::BLUE );
			}

			if ( info.m_type == hkTjunctionDetector::NEAR_EDGE2 )
			{
				HK_DISPLAY_LINE( v2, v0, hkColor::BLUE );
			}
		}
	}

	if ( showWelded )
	{
		for ( int i = 0; i < m_builder.m_weldedVertices.getSize(); ++i )
		{
			const hkVector4& v = m_builder.m_weldedVertices[i];
			HK_DISPLAY_STAR( v, 0.2f, hkColor::PINK );
		}
	}

	/*if ( showMaterials )
	{
		hkpShapeBuffer shapeBuffer;
		for ( int i = 0; i < m_keys.getSize(); ++i )
		{
			if ( m_keys[i] == HK_INVALID_SHAPE_KEY )
			{
				continue;
			}

			const hkpTriangleShape* triangle = reinterpret_cast<const hkpTriangleShape*>( m_compMesh->getChildShape( m_keys[i], shapeBuffer ) );
			const hkVector4& v0 = triangle->getVertex( 0 );
			const hkVector4& v1 = triangle->getVertex( 1 );
			const hkVector4& v2 = triangle->getVertex( 2 );

			const hkpMeshMaterial* material = m_compMesh->getMaterial( m_keys[i] );
			if ( material != HK_NULL )
			{
				HK_DISPLAY_TRIANGLE( v0, v1, v2, hkColor::MAGENTA );
			}
			else
			{
				HK_DISPLAY_TRIANGLE( v0, v1, v2, hkColor::LIME );
			}
		}
	}*/

	const hkgKeyboard& keys = m_env->m_window->getKeyboard();
	if ( keys.wasKeyPressed( 'L' ) )
	{
		wireframe = !wireframe;
		setGraphicsState(HKG_ENABLED_WIREFRAME, wireframe);
	}
	if ( keys.wasKeyPressed( 'V' ) )
	{
		showWelded = !showWelded;
	}
	if ( keys.wasKeyPressed( 'C' ) )
	{
		m_showChunks = !m_showChunks;
		m_showAabb = !m_showAabb;
	}
	if ( keys.wasKeyPressed( 'T' ) )
	{
		showTJ = !showTJ;
	}
	if ( keys.wasKeyPressed( 'R' ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) )
	{
		rayCast = !rayCast;
		m_random.getRandomVectorRange( m_bounds.m_min, m_bounds.m_max, rayTarget );
		rayTarget(m_minIndex) = m_bounds.m_min(m_minIndex);
	}

	return hkDefaultPhysics2012Demo::stepDemo();
}

int CompressedMeshDemo::testTriangleMapping( hkpShapeCollection* mesh, int& numValidKeys )
{
	numValidKeys = 0;
	int numMismatch = 0;
	hkReal threshold = 0.01f;
	hkpShapeBuffer buffer;
	int numTriangles = 0; // the total number of triangles in the landscape
	for ( int k = 0; k < m_geometryInstances.m_instances.getSize(); ++k )
	{
		const int geomIndex = m_geometryInstances.m_instances[k].m_geometryIdx;
		const hkGeometry& geometry = m_geometryInstances.m_geometries[geomIndex];
		const hkMatrix4& transform = m_geometryInstances.m_instances[k].m_worldFromLocal;
		numTriangles += geometry.m_triangles.getSize();

		// go through all the shape keys associated with this geometry
		for ( int i = 0; i < geometry.m_triangles.getSize(); ++i )
		{
			const hkpShapeKey key = m_keys[k][i];
			if ( key != HK_INVALID_SHAPE_KEY )
			{
				numValidKeys++;

				{
					hkVector4 v1, v2, v3;
					const hkGeometry::Triangle& triangle = geometry.m_triangles[i];

					transform.transformPosition( geometry.m_vertices[triangle.m_a], v1 );
					transform.transformPosition( geometry.m_vertices[triangle.m_b], v2 );
					transform.transformPosition( geometry.m_vertices[triangle.m_c], v3 );

					hkVector4 c1; hkpTriangleUtil::calcCentroid( c1, v1, v2, v3 );

					hkVector4 w1, w2, w3;
					hkpTriangleShape* triangleShape;
					triangleShape = (hkpTriangleShape*)mesh->getChildShape( key, buffer );

					w1 = triangleShape->getVertex(0);
					w2 = triangleShape->getVertex(1);
					w3 = triangleShape->getVertex(2);

					hkVector4 c2; hkpTriangleUtil::calcCentroid( c2, w1, w2, w3 );

					hkVector4 c; c.setSub4( c1, c2 );
					hkReal d = c.length3();
					if ( d > threshold )
					{
						m_out << "MISHSMATCH " << ( numTriangles + i ) << ": " << d << '\n';
						++numMismatch;
					}
				}
			}
		}
	}

	m_out << "num mismatched keys = " << numMismatch << '\n';

	return numTriangles;
}

void CompressedMeshDemo::addDisplayGeometry( hkGeometry geometry, const hkVector4& color, const hkVector4& translate )
{
	for ( int i = 0; i < geometry.m_vertices.getSize(); ++i )
	{
		geometry.m_vertices[i].add4( translate );
	}
	hkgDisplayObject* dispObj = GeometryLoaderUtil::createDisplay( geometry, m_env->m_displayWorld, m_env->m_window->getContext(), color );
	dispObj->removeReference();
}

static void createWavySurface( const int side, hkGeometry& geometry )
{
	hkReal scaleHoriz = 5.0f;
	hkReal scaleVert = 1.0f;
	{
		geometry.m_vertices.setSize( side * side );
		for(int i = 0; i < side; i++)
		{
			for (int j = 0; j < side; j++ )
			{
				hkVector4 vertex (
					scaleHoriz * (i * 1.0f - (side-1) * 0.5f),
					scaleHoriz * (j * 1.0f - (side-1) * 0.5f),
					scaleVert * (0.6f * hkMath::cos((hkReal)j + i) + 0.3f * hkMath::sin( 2.0f * i) ));
				geometry.m_vertices[i*side + j] = vertex ;
			}
		}
	}

	{
		geometry.m_triangles.setSize( (side-1) * (side-1) * 2);
		int corner = 0;
		int curTri = 0;
		for(int i = 0; i < side - 1; i++)
		{
			for (int j = 0; j < side - 1; j++ )
			{
				geometry.m_triangles[curTri].m_c = corner+1;
				geometry.m_triangles[curTri].m_b = corner+side;
				geometry.m_triangles[curTri].m_a = corner;
				geometry.m_triangles[curTri].m_material = 0;
				curTri++;

				geometry.m_triangles[curTri].m_c = corner+side+1;
				geometry.m_triangles[curTri].m_b = corner+side;
				geometry.m_triangles[curTri].m_a = corner+1;
				geometry.m_triangles[curTri].m_material = 0;
				curTri++;
				corner++;
			}
			corner++;
		}
	}
}

static hkpExtendedMeshShape* HK_CALL createExtendedMeshFromGeometry( const hkGeometry& geometry )
{
	hkpExtendedMeshShape* extendedMesh = new hkpExtendedMeshShape();

	hkpExtendedMeshShape::TrianglesSubpart part;
	// set vertices
	part.m_numVertices = geometry.m_vertices.getSize();
	part.m_vertexBase  = &geometry.m_vertices.begin()[0](0);
	part.m_vertexStriding = sizeof(hkVector4);

	// set triangles
	part.m_indexBase = geometry.m_triangles.begin();
	part.m_numTriangleShapes = geometry.m_triangles.getSize();
	part.m_indexStriding = sizeof(hkGeometry::Triangle);
	part.m_stridingType = hkpExtendedMeshShape::INDICES_INT32;

	extendedMesh->addTrianglesSubpart( part );

	return extendedMesh;
}

hkpShape* CompressedMeshDemo::createExtendedMeshShape()
{
	hkpExtendedMeshShape* extendedMesh = new hkpExtendedMeshShape();
	extendedMesh->m_numBitsForSubpartIndex = 13;


	m_keys.setSize( m_geometryInstances.m_instances.getSize() );
	for ( int i = 0; i < m_geometryInstances.m_instances.getSize(); i++ )
	{
		const hkGeometry& geometry = m_geometryInstances.m_geometries[ m_geometryInstances.m_instances[i].m_geometryIdx ];

		// build mapping
		m_keys[i].setSize( geometry.m_triangles.getSize() );
		for ( int j = 0; j < m_keys[i].getSize(); j++ )
		{
			m_keys[i][j] = ( i << ( 32 - extendedMesh->m_numBitsForSubpartIndex ) ) | j;
		}

		hkpExtendedMeshShape::TrianglesSubpart part;
		// set vertices
		part.m_numVertices = geometry.m_vertices.getSize();
		part.m_vertexBase = &geometry.m_vertices.begin()[0](0);
		part.m_vertexStriding = sizeof(hkVector4);

		// set triangles
		part.m_indexBase = geometry.m_triangles.begin();
		part.m_numTriangleShapes = geometry.m_triangles.getSize();
		part.m_indexStriding = sizeof(hkGeometry::Triangle);
		part.m_stridingType = hkpExtendedMeshShape::INDICES_INT32;
		part.m_transform.set( m_geometryInstances.m_instances[i].m_worldFromLocal );

		extendedMesh->addTrianglesSubpart( part );


	}

	int numValidKeys;
	const int numTriangles = testTriangleMapping( extendedMesh, numValidKeys );
	m_out << "num triangles = " << numTriangles << '\n';
	m_out.flush();


	return extendedMesh;
}


hkpShape* CompressedMeshDemo::createCompressedMeshShape()
{
	m_out << "num instances = " << m_geometryInstances.m_instances.getSize() << '\n';
	m_out << "num geometries = " << m_geometryInstances.m_geometries.getSize() << '\n';

	// initialize the builder
	m_error = 0.001f;
	m_builder.m_overlapRatio = 0.2f;
	m_builder.m_stripperPasses = 5000;
	m_builder.m_weldVertices = true;
	m_builder.m_TjunctionTolerance = 1e-7f;
	m_builder.m_createMapping = true;
	m_compMesh = m_builder.createMeshShape( m_error, hkpCompressedMeshShape::MATERIAL_SINGLE_VALUE_PER_CHUNK );

	// build info about each geometry
	m_geomInfo.setSize( m_geometryInstances.m_geometries.getSize() );
	for ( int i = 0; i < m_geometryInstances.m_geometries.getSize(); ++i )
	{
		m_geomInfo[i].m_numInstances = 0;
		m_geomInfo[i].m_subpartID = -1;
	}
	// count the instances
	for ( int i = 0; i < m_geometryInstances.m_instances.getSize(); ++i )
	{
		const int index = m_geometryInstances.m_instances[i].m_geometryIdx;
		if ( m_geomInfo[index].m_numInstances == 0 )
		{
			m_geomInfo[index].m_firstTransform = m_geometryInstances.m_instances[i].m_worldFromLocal;
		}
		m_geomInfo[index].m_numInstances++;
	}

	// add the subparts or transformed geometries
	for ( int i = 0; i < m_geometryInstances.m_geometries.getSize(); ++i )
	{
		m_geomInfo[i].m_subpartID = m_builder.beginSubpart( m_compMesh );
		if ( m_geometryInstances.m_geometries[i].m_vertices.getSize() <= hkpCompressedMeshShape::MAX_CONVEX_VERTICES )
		{
			if ( !m_builder.addConvexPiece( m_geometryInstances.m_geometries[i], m_compMesh ) )
			{
				HK_REPORT( "Convex piece not created, adding it as triangles." );
				m_builder.addGeometry( m_geometryInstances.m_geometries[i], hkMatrix4::getIdentity(), m_compMesh );
			}
		}
		else
		{
			m_builder.addGeometry( m_geometryInstances.m_geometries[i], hkMatrix4::getIdentity(), m_compMesh );
		}
		m_builder.endSubpart( m_compMesh );
	}
	// add the instances
	m_keys.setSize( m_geometryInstances.m_instances.getSize() );
	for ( int i = 0; i < m_geometryInstances.m_instances.getSize(); ++i )
	{
		const int index = m_geometryInstances.m_instances[i].m_geometryIdx;
		m_builder.addInstance( m_geomInfo[index].m_subpartID, m_geometryInstances.m_instances[i].m_worldFromLocal, m_compMesh, &m_keys[i] );
	}

	// compute the AABB in order to know where to place the created display geometry
	hkAabb aabb;
	m_compMesh->getAabb( hkTransform::getIdentity(), 0, aabb );
	hkVector4 halfExtents;
	aabb.getHalfExtents( halfExtents );
	int maxAxis = halfExtents.getMajorAxis3();
	m_geometryTranslation.setZero4();
	m_geometryTranslation( maxAxis ) = 2.2f * halfExtents( maxAxis );

	// create the display geometry
	if ( m_showBreakdown )
	{
		// chunks
		for ( int i = 0; i < m_compMesh->m_chunks.getSize(); ++i )
		{
			hkGeometry chunkGeom;
			hkpCompressedMeshShapeBuilder::chunkToGeometry( m_compMesh, i, chunkGeom);
			addDisplayGeometry( chunkGeom, hkVector4(m_random.getRandReal01() / 2, 0.3f, m_random.getRandReal01(), 1), m_geometryTranslation );
		}

		// convex pieces
		for ( int i = 0; i < m_compMesh->m_convexPieces.getSize(); ++i )
		{
			hkGeometry convexPieceGeom;
			hkpCompressedMeshShapeBuilder::convexPieceToGeometry( m_compMesh, i, convexPieceGeom );
			if ( convexPieceGeom.m_vertices.getSize() != 0 )
			{
				addDisplayGeometry( convexPieceGeom, hkVector4(0.0f, m_random.getRandReal01(), 0.0f, 1), m_geometryTranslation );
			}
		}

		// big triangles
		hkGeometry bigGeometry;
		hkpCompressedMeshShapeBuilder::bigTrianglesToGeometry( m_compMesh, bigGeometry );
		if ( bigGeometry.m_triangles.getSize() != 0 )
		{
			addDisplayGeometry( bigGeometry, hkVector4( 0.9f, 0.2f, 0.2f, 1.0f ), m_geometryTranslation );
		}
	}

	// unit test
	int numValidKeys;
	testTriangleMapping( m_compMesh, numValidKeys );

	// statistics
	m_builder.gatherStatistics( m_compMesh );
	const hkpCompressedMeshShapeBuilder::Statistics& stats = m_builder.m_statistics;

	m_out << "num verts = " << stats.m_numVertices << '\n';
	m_out << "num tris = " << stats.m_numTriangles << '\n';
	m_out << "max extent = " << stats.m_maxExtent << '\n';
	m_out << "max index = " << stats.m_maxIndex << '\n';
	m_out << "error = " << stats.m_error << '\n';
	m_out << "left over triangles = " << stats.m_numBigTriangles << '\n';
	m_out << "left over vertices = " << stats.m_numBigVertices << '\n';
	//m_out << "duplicated vertices = " << stats.m_numExcessVertices << '\n';
	m_out << "num chunks = " << m_compMesh->m_chunks.getSize() << '\n';
	m_out << "num convex pieces = " << m_compMesh->m_convexPieces.getSize() << '\n';
	m_out << "num child shapes = " << m_compMesh->getNumChildShapes() << '\n';
	m_out << "num valid keys = " << numValidKeys << '\n';
	m_out << "size = " << ( stats.m_size >> 10 ) << "k" << '\n';
	m_out << "compress ratio = " << stats.m_compressRatio << '\n';
	m_out << "num strips = " << stats.m_numStrips << '\n';
	m_out << "max overlap = " << stats.m_maxOverlap << ", ratio = " << ( stats.m_maxOverlap / stats.m_maxExtent ) << '\n';
	m_out << "num degenerate triangles = " << stats.m_numDegenerate << '\n';
	m_out << "num T-junctions = " << m_builder.m_Tjunctions.getSize() << '\n';
	m_out << "num welded vertices = " << m_builder.m_weldedVertices.getSize() << '\n';

	int totalTriangles = 0;
	for ( int i = 0; i < m_compMesh->m_chunks.getSize(); ++i )
	{
		const hkpCompressedMeshShape::Chunk& chunk = m_compMesh->m_chunks[i];
		if ( chunk.m_reference == HK_CMS_NULL_REF )
		{
			const int chunkTriangles = chunk.getNumTriangles();
			m_out << "chunk " << i << ": " << ( chunk.m_vertices.getSize() / 3 ) << " vertices, " << chunkTriangles << " triangles" << '\n';
			totalTriangles += chunkTriangles;
		}
		else
		{
			totalTriangles += m_compMesh->m_chunks[chunk.m_reference].getNumTriangles();
			m_out << "chunk " << i << ": reference to " << chunk.m_reference << '\n';
		}
	}
	m_out << "total chunk triangles = " << totalTriangles << '\n';
	//m_out << "final check : " << ( totalTriangles + stats.m_numBigTriangles + stats.m_numDegenerate ) << " == " << numTriangles << '\n';

	m_out.flush();

	return m_compMesh;
}

hkpShape* CompressedMeshDemo::createMeshShape( hkDemoEnvironment* env )
{
	const CompressedMeshDemoVariant& variant = g_variants[m_variantId];

	if ( variant.m_type != SIMPLE_TEST )
	{
		hkLoader loader;
		hkRootLevelContainer* container = loader.load( CLIENT_LANDSCAPE );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scenePtr = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));

		if (scenePtr == HK_NULL || (scenePtr->m_rootNode == HK_NULL) )
		{
			HK_ASSERT(0xa54323c0, "No scene data (or scene data root hkxNode) found. Can't continue.");
			return HK_NULL;
		}

		// get the geometry instances and build the material table
		hkArray<hkxMaterial*> materials;

		hkGeometry geometry;
		hkxSceneDataToGeometryConverter::convertToSingleGeometry( scenePtr, scenePtr->m_rootNode, geometry, materials );
		if ( m_showOriginalGeometry )
		{
			addDisplayGeometry( geometry, hkVector4( 1, 0, 0, 0.8f ), hkVector4::getZero() );
		}
		hkAabbUtil::calcAabb( geometry.m_vertices.begin(), geometry.m_vertices.getSize(), m_originalAabb );

		hkxSceneDataToGeometryConverter::convertToGeometryInstances( scenePtr, scenePtr->m_rootNode, m_geometryInstances, materials );
		m_out << "num material pointers = " << materials.getSize() << '\n';

		if ( variant.m_type == CASTLE_EMS )
		{
			return createExtendedMeshShape();
		}

		return createCompressedMeshShape();
	}
	else
	{
		m_error = 0.001f;

		m_builder.m_overlapRatio = 0.2f;
		m_builder.m_stripperPasses = 5000;
		m_builder.m_weldVertices = true;
		m_builder.m_TjunctionTolerance = 0;
		m_builder.m_preserveTjunctions = false;

		m_compMesh = m_builder.createMeshShape( m_error, hkpCompressedMeshShape::MATERIAL_SINGLE_VALUE_PER_CHUNK, 18 );

		hkGeometry wavyGeom;
		createWavySurface( 40, wavyGeom );
		m_builder.beginSubpart( m_compMesh );
		m_builder.addGeometry( wavyGeom, hkMatrix4::getIdentity(), m_compMesh );
		m_builder.endSubpart( m_compMesh );
		m_builder.addInstance( 0, hkMatrix4::getIdentity(), m_compMesh );

		const int numVerts = 32;
		// create an array of random vertices
		hkArray<hkVector4> vertices;
		{
			vertices.reserve( numVerts );
			for(int i = 0; i < numVerts; i++)
			{
				vertices.pushBackUnchecked( GameUtils::createRandomVertex( hkSphere( hkVector4(0,0,0), 30 ), &m_random ));
			}
		}

		m_builder.beginSubpart( m_compMesh );
		m_builder.addConvexPiece( vertices, m_compMesh );
		m_builder.endSubpart( m_compMesh );
		m_builder.addInstance( 1, hkMatrix4::getIdentity(), m_compMesh );

		// compute the AABB in order to know where to place the created display geometry

		// create the display geometry
			// chunks

			// convex pieces

			// big triangles

		return m_compMesh;
	}
}



HK_DECLARE_DEMO_VARIANT_USING_STRUCT(CompressedMeshDemo, HK_DEMO_TYPE_OTHER, CompressedMeshDemoVariant, g_variants, "A suite of demos showing the features of the (legacy) compressed and extended mesh shapes" );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
