/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/Shapes/BvCompressedMesh/BvCompressedMeshDemo.h>

#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShape.h>
#include <Physics2012/Internal/Collide/BvCompressedMesh/hkpBvCompressedMeshShapeCinfo.h>
#include <Physics2012/Collide/Shape/Convex/Capsule/hkpCapsuleShape.h>
#include <Physics2012/Collide/Shape/Convex/Cylinder/hkpCylinderShape.h>
#include <Physics2012/Collide/Query/CastUtil/hkpWorldRayCastInput.h>
#include <Physics2012/Collide/Query/Collector/RayCollector/hkpAllRayHitCollector.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Common/Base/Algorithm/Noise/hkPerlinNoise2d.h>
#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>
#include <Demos/DemoCommon/Utilities/Geometry/GeometryMaker.h>
#include <Common/Visualize/Shape/hkDisplayConvex.h>

#include <Common/Internal/GeometryProcessing/ConvexHull/hkgpConvexHull.h>


// General parameters
static const hkVector4 g_up = hkVector4(0, 1, 0);

// Camera parameters
static const hkVector4 g_cameraFrom = hkVector4(0, 10, 35);
static const hkVector4 g_cameraTo = hkVector4(0, 0, 0);
static const hkReal g_cameraNear = 0.1f;
static const hkReal g_cameraFar = 1000.0f;

// Physics world parameters
static const hkReal g_worldSize = 500;
static const hkReal g_worldGravity = -9.8f;

// Demo parameters
static const char g_demoDetails[] = "Demo showing the construction and use of a hkpBvCompressedMeshShape. \
\nThe triangles and convex primitives, the rays and dynamic bodies are all colored to indicate their (per-primitive) collision filter layer value. \
\nThe collision filter is configured such that objects in the same layer do not collide with each other. \
\nPer-primitive user data is also stored by the mesh shape, and is displayed alongside each primitives that is hit.";

// Other parameters
static const hkReal g_radius = hkConvexShapeDefaultRadius;
static const int g_meshResolution = 33;
static const hkReal g_meshSide = 32;
static const int g_layerColors[] = { 0xFF0094FF, hkColor::GREEN, hkColor::SADDLEBROWN, hkColor::rgbFromFloats(0.7f, 0.7f, 0.7f) };
static const int g_numRays = 4;
static const int g_noiseSeed = 1;

// Variants.

struct BvCompressedMeshDemoVariant
{
	const char*							m_name;
	bool								m_geom;
	bool								m_shapes;
	const char*							m_details;
};

static const BvCompressedMeshDemoVariant g_variants[] =
{
	{ "Triangles only", true, false, "" },
	{ "Triangles and convex shapes", true, true, "" },
};

BvCompressedMeshDemo::BvCompressedMeshDemo(hkDemoEnvironment* env) : hkDefaultPhysics2012Demo(env), m_numFrames(0)
{			
	// Create physics world
	{	
		hkpWorldCinfo info;					
		info.m_gravity.setMul(g_up, g_worldGravity);
		info.setBroadPhaseWorldSize(g_worldSize);
		info.m_useCompoundSpuElf = true; // this flag must be set to be able to use BvCompressedMeshShape in SPU
		m_world = new hkpWorld(info);
	}

	m_world->lock();
	
	// Setup a group collision filter with collisions between the same layers (1 to 4) disabled
	m_groupFilter = hkRefNew<hkpGroupFilter>(new hkpGroupFilter());			
	m_groupFilter->disableCollisionsBetween(1, 1);
	m_groupFilter->disableCollisionsBetween(2, 2);
	m_groupFilter->disableCollisionsBetween(3, 3);
	m_groupFilter->disableCollisionsBetween(4, 4);
	m_world->setCollisionFilter(m_groupFilter);			

	// Register collision agents
	hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());	

	// Setup graphics	
	setupDefaultCameras(m_env, g_cameraFrom, g_cameraTo, g_up, g_cameraNear, g_cameraFar);
	setupGraphics();
		
	// Create mesh shape
	m_meshShape = hkRefNew<hkpBvCompressedMeshShape>(createBvCompressedMeshShape());

	// Create body with the mesh shape
	{	
		hkpRigidBodyCinfo info;
		info.m_shape = m_meshShape;
		info.m_motionType = hkpMotion::MOTION_FIXED;		
		m_meshBody = hkRefNew<hkpRigidBody>(new hkpRigidBody(info));
		m_world->addEntity(m_meshBody);
	}

	// Substitute default display geometry by our own to show collision layers with colors
	createMeshDisplayGeometry();

	 // Create falling shapes with different collision layers to demonstrate collision filtering
	createFallingShapes();

	// Create a box under the landscape to catch shapes and rays that miss the landscape mesh
	createGroundBox();
		
	m_world->unlock();	
}


namespace
{
	// BV Compressed Mesh construction info structure used to provide custom per triangle filter info and user data.
	// See createBvCompressMeshShape() method below for usage.
	struct MyMeshConstructionInfo : public hkpDefaultBvCompressedMeshShapeCinfo
	{
		MyMeshConstructionInfo( const hkGeometry* geometry ) : hkpDefaultBvCompressedMeshShapeCinfo( geometry ) {}

		~MyMeshConstructionInfo() {}

		hkUint32 getTriangleCollisionFilterInfo( int triangleIndex ) const
		{
			// Calculate triangle center
			hkVector4 center;
			{
				hkVector4 vertices[3]; getVertices( triangleIndex, vertices );
				center.setAdd( vertices[0], vertices[1] ); 
				center.add( vertices[2] );
				center.mul( hkSimdReal::getConstant<HK_QUADREAL_INV_3>() );
			}

			// Assign the collision layer depending on the center height		
			const int collisionLayer = hkMath::clamp(1 + int(4 * center(1) / 3.0f), 1, 4);
			return hkpGroupFilter::calcFilterInfo(collisionLayer);
		}

		hkUint32 getTriangleUserData( int triangleIndex ) const
		{
			// Assign data values sequentially in the range [0,255] 
			return triangleIndex % 256;
		}

		hkUint32 getConvexShapeCollisionFilterInfo( int convexShapeIndex ) const
		{
			// Assign the collision layer depending on the center height		
			const int collisionLayer = hkMath::clamp( 1 + int( 4 * m_hullCenters[convexShapeIndex](1) / 3.0f ), 1, 4 );
			return hkpGroupFilter::calcFilterInfo( collisionLayer );
		}

		hkUint32 getConvexShapeUserData( int convexShapeIndex ) const
		{
			// Assign data values sequentially in the range [0,255] 
			return convexShapeIndex % 256;
		}

		hkArray<hkVector4>	m_hullCenters; // Convex hull centers.
	};
}

hkpBvCompressedMeshShape* BvCompressedMeshDemo::createBvCompressedMeshShape()
{
	// Create a triangulated geometry
	hkGeometry geometry;
	createMeshGeometry( geometry );

	// We use a custom construction info structure where we have overridden the getTriangleCollisionFilterInfo and
	// getTriangleUserData virtual methods to provide our custom collision filter info and user data per triangle  
	// in the shape construction process.
	// In addition, this custom construction info supports convex hulls primitives.
	MyMeshConstructionInfo cInfo( g_variants[m_variantId].m_geom ? &geometry : HK_NULL );
	{
		cInfo.m_convexRadius = g_radius;
		cInfo.m_weldingType = hkpWeldingUtility::WELDING_TYPE_ANTICLOCKWISE;

		// We will use 8 bits per primitive for collision filter infos as we have only 4 collision layers (1 to 4)
		cInfo.m_collisionFilterInfoMode = hkpBvCompressedMeshShape::PER_PRIMITIVE_DATA_8_BIT;

		// We will use 8 bits per primitive for the user data because all our user data values are in the range [0,255]
		cInfo.m_userDataMode = hkpBvCompressedMeshShape::PER_PRIMITIVE_DATA_8_BIT;
	}

	// Add some random convex shapes to the construction info
	if( g_variants[m_variantId].m_shapes )
	{
		hkPseudoRandomGenerator	rndGen(180673);
		hkSimdReal scale; scale.setFromFloat(0.4f);		
		cInfo.m_hullCenters.setSize( geometry.m_vertices.getSize(), hkVector4::getZero() );
		hkUint32 convexHullIndex = 0;
		for( int i=0; i<geometry.m_vertices.getSize(); ++i )
		{
			// Only add shape sometimes
			if ( rndGen.getRandReal01() > 0.9f ) 
			{
				// Set center point and increment index
				cInfo.m_hullCenters[convexHullIndex++] = geometry.m_vertices[i];

				const int shapeType = i % 5;				
				switch( shapeType )
				{
					// Convex hull.
				case 0:
					{
						const int	numHullVertices = 32;
						hkVector4	vertices[numHullVertices];
						for(int j=0; j<numHullVertices; ++j)
						{
							rndGen.getRandomVector11(vertices[j]);
							vertices[j].mul(scale);
							vertices[j].add(geometry.m_vertices[i]);
						}

						hkgpConvexHull		hull; hull.build(vertices,numHullVertices);
						hkArray<hkVector4>	vtx; hull.fetchPositions(hkgpConvexHull::SOURCE_VERTICES, vtx);

						hkpConvexVerticesShape::BuildConfig	cfg;
						cfg.m_convexRadius = cInfo.m_convexRadius;
						cfg.m_shrinkByConvexRadius = false;

						const hkpConvexShape* shape = new hkpConvexVerticesShape( vtx, cfg );
						cInfo.addConvexShape( shape );
						shape->removeReference();
					}
					break;

					// Box
				case 1:
					{
						hkVector4		he; rndGen.getRandomVector01(he); he.mul(scale);
						hkQuaternion	rot; rndGen.getRandomRotation(rot);
						hkQsTransform	trs; trs.setIdentity();
						trs.setTranslation(geometry.m_vertices[i]);
						trs.setRotation(rot);

						const hkpConvexShape* shape = new hkpBoxShape( he, cInfo.m_convexRadius );
						cInfo.addConvexShape( shape, trs );
						shape->removeReference();
					}
					break;

					// Capsule
				case 2:
					{
						hkQuaternion	rot; rndGen.getRandomRotation(rot);
						hkQsTransform	trs; trs.setIdentity(); trs.setTranslation(geometry.m_vertices[i]); trs.setRotation(rot);
						hkVector4		a,b; a.setZero(); b.setZero();
						a(1) += 0.5f;
						b(1) -= 0.5f;

						const hkpConvexShape* shape = new hkpCapsuleShape( a, b, rndGen.getRandRange(0.01f,0.2f) );
						cInfo.addConvexShape( shape, trs );
						shape->removeReference();
					}
					break;

					// Sphere
				case 3:
					{
						hkQsTransform	trs; trs.setIdentity();
						trs.setTranslation( geometry.m_vertices[i] );

						const hkpConvexShape* shape = new hkpSphereShape( rndGen.getRandRange(0.01f,0.3f) );
						cInfo.addConvexShape( shape, trs );
						shape->removeReference();
					}
					break;

					// Cylinder
				case 4:
					{
						hkQuaternion	rot; rndGen.getRandomRotation(rot);
						hkQsTransform	trs; trs.setIdentity(); trs.setTranslation(geometry.m_vertices[i]); trs.setRotation(rot);
						hkVector4		a,b; a.setZero(); b.setZero();
						a(1) += 0.5f;
						b(1) -= 0.5f;

						const hkpConvexShape* shape = new hkpCylinderShape( a, b, rndGen.getRandRange(0.01f,0.2f), 0.0f );
						cInfo.addConvexShape( shape, trs );
						shape->removeReference();
					}
					break;
				}	
			}
		}
	}
	
	// Construct the mesh shape using our construction info
	return new hkpBvCompressedMeshShape( cInfo );
}


hkDemo::Result BvCompressedMeshDemo::stepDemo()
{
	m_world->lock();

	// World ray cast
	{
		hkArray<hkVector4> raysFrom;
		hkArray<hkVector4> raysTo;
		calculateRays(raysFrom, raysTo);

		// Cast rays with alternating collision layers 
		for (int i = 0; i < g_numRays; ++i)
		{			
			hkpWorldRayCastInput input;
			input.m_from = raysFrom[i];
			input.m_to = raysTo[i];

			// Set ray filtering info
			const int collisionLayer = (i % 4) + 1;
			input.m_enableShapeCollectionFilter = true;
			input.m_filterInfo = hkpGroupFilter::calcFilterInfo(collisionLayer);
			
			// Cast ray
			hkpWorldRayCastOutput output;
			m_world->castRay(input, output);

			// Draw result
			drawRayCastOuput(raysFrom[i], raysTo[i], collisionLayer, output);
		}
	}

	m_world->unlock();

	// Output "nicely" formatted text
	{
		const int margin = 15;
		const int windowWidth = m_env->m_window->getWidth();
		const int windowHeight = m_env->m_window->getHeight();
		m_env->m_textDisplay->outputTextWithWrapping(g_demoDetails, margin, margin, windowWidth - 2 * margin, windowHeight - 2 * margin);
	}

	m_numFrames++;
		
	return hkDefaultPhysics2012Demo::stepDemo();
}


BvCompressedMeshDemo::~BvCompressedMeshDemo()
{
	m_world->markForWrite();

	if (m_world)
	{
		m_world->removeReference();
		m_world = HK_NULL;
	}
}


///////////////////////////////
// Helper methods and classes

namespace
{
	struct HeightFunction
	{
		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO, HeightFunction);

		HeightFunction(int seed, int side)
		{
			m_perlinNoise = hkRefNew<hkPerlinNoise2d>(new hkPerlinNoise2d(seed, side));
		}

		HK_FORCE_INLINE hkReal operator()(hkReal x, hkReal z) const 
		{ 
			return 1.5f * (1 + m_perlinNoise->_smooth(x, z));
		}

		hkRefPtr<hkPerlinNoise2d> m_perlinNoise;
	};
}


void BvCompressedMeshDemo::createMeshGeometry(hkGeometry& geometry)
{
	HeightFunction heightFunction(g_noiseSeed, 32);
	GeometryMaker::Rectangle functionDomain(0, 0, 8, 8);
	GeometryMaker::Rectangle outputDomain(-0.5f * g_meshSide, -0.5f * g_meshSide, g_meshSide, g_meshSide);
	GeometryMaker::createFromHeightFunction(heightFunction, g_meshResolution, g_meshResolution, g_up, 0, geometry, functionDomain, outputDomain);

}


void BvCompressedMeshDemo::createMeshDisplayGeometry()
{
	// Remove standard display geometry for the mesh
	const hkUlong displayGeometryId = (hkUlong)m_meshBody->getCollidable();
	HK_REMOVE_GEOMETRY(displayGeometryId);

	// Create our own display geometries with different colors according to the collision layer
	{
		// Allocate geometries for the 4 collision layers. They will be deleted by the hkDisplayConvex objects
		hkGeometry* geometries[4];
		for (int i = 0; i < 4; ++i)
		{
			geometries[i] = new hkGeometry();
		}

		// Use the fact that convertToGeometry stores raw data in material,
		// and that 8 bits of that represent the filter info, to assign colors.
		hkGeometry	rawGeometry; m_meshShape->convertToGeometry(rawGeometry);
		for(int i=0; i<rawGeometry.m_triangles.getSize(); ++i)
		{
			const hkGeometry::Triangle& t = rawGeometry.m_triangles[i];

			hkUint32 collisionFilterInfo = t.m_material & 0xFF;
			if( !m_meshShape->accessCollisionFilterInfoPalette().isEmpty() )
			{
				collisionFilterInfo = m_meshShape->accessCollisionFilterInfoPalette()[collisionFilterInfo];
			}

			const hkVector4 v[] = {rawGeometry.m_vertices[t.m_a],rawGeometry.m_vertices[t.m_b],rawGeometry.m_vertices[t.m_c]};
			hkGeometry& output = *geometries[collisionFilterInfo - 1];
			const int firstIndex = output.m_vertices.getSize();
			output.m_triangles.expandOne().set(firstIndex, firstIndex+1, firstIndex+2, t.m_material);
			output.m_vertices.append(v,3);
		}

		// Create display geometries.
		m_displayGeometries.setSize(4);
		for (int i = 0; i < 4; ++i)
		{		
			// Remove duplicate vertices
			hkGeometryUtils::weldVertices(*geometries[i], 0);

			m_displayGeometries[i] = hkRefNew<hkDisplayConvex>(new hkDisplayConvex(geometries[i]));			
			hkDebugDisplay::getInstance().addGeometry(m_displayGeometries[i], displayGeometryId + i, 0, 0);			
			HK_SET_OBJECT_COLOR(displayGeometryId + i, g_layerColors[i]);
		}		
	}
}

void BvCompressedMeshDemo::createFallingShapes()
{
	// Create falling shape and compute inertia
	hkpRigidBodyCinfo info;		
	info.m_shape = new hkpSphereShape(0.25f);
	info.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;	
	hkpInertiaTensorComputer::setShapeVolumeMassProperties(info.m_shape, 1, info);

	// Obtain mesh aabb and shrink it a bit so all grid positions are well inside the mesh
	hkAabb meshAabb;
	m_meshShape->getAabb(m_meshBody->getTransform(), 0, meshAabb);
	meshAabb.expandBy(hkSimdReal::fromFloat(-1));

	// Create bodies in an horizontal grid
	const int bodiesPerSide = 8;
	hkVector4 gridExtents; gridExtents.setSub(meshAabb.m_max, meshAabb.m_min);
	hkVector4 gridOrigin = meshAabb.m_min;
	gridOrigin.setComponent<1>(meshAabb.m_max.getComponent<1>());
	const hkReal stepX =  gridExtents.getComponent<0>().getReal() / (hkReal) (bodiesPerSide - 1);
	const hkReal stepZ =  gridExtents.getComponent<2>().getReal() / (hkReal) (bodiesPerSide - 1);		
	for (int x = 0; x < bodiesPerSide; ++x)
	{
		for (int z = 0; z < bodiesPerSide; ++z)
		{
			// Alternate collision layers
			const int collisionLayer = ((x + z) % 4) + 1;
			info.m_collisionFilterInfo = hkpGroupFilter::calcFilterInfo(collisionLayer);

			// Create body
			info.m_position.set(x * stepX, 1, z * stepZ);
			info.m_position.add(gridOrigin);
			info.m_restitution = 0;
			info.m_linearDamping = 0.6f;
			hkpRigidBody* body = new hkpRigidBody(info);				
			m_world->addEntity(body);

			// Set color according to the collision layer
			const hkUlong displayGeometryId = hkUlong(body->getCollidable());
			HK_SET_OBJECT_COLOR(displayGeometryId, g_layerColors[collisionLayer - 1]);

			body->removeReference();
		}
	}
	info.m_shape->removeReference();
}

void BvCompressedMeshDemo::createGroundBox()
{
	hkpShape* shape = new hkpBoxShape(hkVector4(0.55f * g_meshSide, 0.5f, 0.55f * g_meshSide));
	hkpRigidBodyCinfo info;
	info.m_shape = shape;
	info.m_motionType = hkpMotion::MOTION_FIXED;
	info.m_position.set(0, -5, 0);
	hkpRigidBody* body = new hkpRigidBody(info);
	shape->removeReference();
	m_world->addEntity(body);
	body->removeReference();
}


void BvCompressedMeshDemo::calculateRays(hkArray<hkVector4>& from, hkArray<hkVector4>& to)
{
	from.setSize(g_numRays);
	to.setSize(g_numRays);
	
	const hkReal fps = 1 / m_timestep;

	// Calculate aperture angle	
	const hkReal aperturePeriod = 35 * fps;
	hkReal apertureAngle = 5 + 40 * hkMath::abs(hkMath::cos(HK_REAL_PI * m_numFrames / aperturePeriod));
	apertureAngle *= HK_REAL_DEG_TO_RAD;

	const hkReal stepAngle = 2 * HK_REAL_PI / hkReal(g_numRays);
	const hkReal rotationPeriod = 30 * fps;
	const hkReal rotationStart = 2 * HK_REAL_PI * m_numFrames / rotationPeriod;
	for (int i = 0; i < g_numRays; ++i)
	{	
		const hkReal angle = rotationStart + i * stepAngle;

		// Calculate ray direction
		hkVector4 axis; axis.set(-hkMath::sin(angle), 0, hkMath::cos(angle));
		hkQuaternion rotation; rotation.setAxisAngle(axis, apertureAngle);
		hkVector4 direction; direction.setRotatedDir(rotation, hkVector4(0, -1, 0));

		// Calculate ray start and end
		from[i].set(hkMath::cos(angle), 15, hkMath::sin(angle));
		hkSimdReal rayLength; rayLength.setFromFloat(30.0f);
		to[i].setAddMul(from[i], direction, rayLength);
	}
}

void BvCompressedMeshDemo::drawRayCastOuput(hkVector4Parameter from, hkVector4Parameter to, 
											int layer, const hkpWorldRayCastOutput& output)
{		
	if (output.hasHit())
	{
		// Draw ray until the hit point
		hkVector4 hitPoint; hitPoint.setSub(to, from);
		hitPoint.setAddMul(from, hitPoint, output.m_hitFraction);					
		HK_DISPLAY_LINE(from, hitPoint, g_layerColors[layer - 1]);

		// Draw hit normal
		{					
			HK_DISPLAY_ARROW(hitPoint, output.m_normal, hkColor::WHITE);
		}

		// Draw the key geometry.
		const hkpShape* shape = output.m_rootCollidable->getShape();
		if (shape->getType() == hkcdShapeType::BV_COMPRESSED_MESH)
		{
			// Retrive child shape.
			hkpShapeBuffer	shapeBuffer;
			const hkpShape*	childShape = m_meshShape->getChildShape(output.m_shapeKeys[0], shapeBuffer);
			
			// Extract child shape geometry from its key.
			hkArray<hkpShapeKey>	keys; keys.pushBack(output.m_shapeKeys[0]);
			hkGeometry				geom; m_meshShape->convertToGeometry(geom, &keys);
			const hkVector4*		vtx = geom.m_vertices.begin();
			
			// Draw wireframe geometry.
			for(int i=0; i<geom.m_triangles.getSize(); ++i)
			{
				const hkGeometry::Triangle& t = geom.m_triangles[i];
				HK_DISPLAY_LINE(vtx[t.m_a], vtx[t.m_b], hkColor::WHITE);
				HK_DISPLAY_LINE(vtx[t.m_b], vtx[t.m_c], hkColor::WHITE);
				HK_DISPLAY_LINE(vtx[t.m_c], vtx[t.m_a], hkColor::WHITE);
			}

			// Print out child shape user data
			hkVector4				center; center.setZero();
			hkSimdReal				maxUp = hitPoint.getComponent<1>();
			for(int i=0; i<geom.m_vertices.getSize(); ++i)
			{
				center.add(vtx[i]);
				maxUp.setMax(maxUp,vtx[i].getComponent<1>());
			}

			hkSimdReal	invNumV; invNumV.setFromFloat(1.0f / (hkReal) geom.m_vertices.getSize());
			center.mul(invNumV);
			center.setComponent<1>(maxUp);
			center(1) += 0.2f;

			hkStringBuf text;					
			text.printf("%d", childShape->getUserData());
			HK_DISPLAY_3D_TEXT(text, center, hkColor::WHITE);
		}
	}
	else
	{
		HK_DISPLAY_LINE(from, to, g_layerColors[layer - 1]);
	}
}



HK_DECLARE_DEMO_VARIANT_USING_STRUCT(BvCompressedMeshDemo, HK_DEMO_TYPE_PHYSICS_2012, BvCompressedMeshDemoVariant, g_variants, "Demo showing the construction and use of hkpBvCompressedMeshShape" );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
