/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */


#include <Demos/demos.h>
#include <Graphics/Common/Window/hkgWindow.h>

#include <Physics2012/Collide/Query/Collector/PointCollector/hkpFixedBufferCdPointCollector.h>
#include <Physics2012/Collide/Query/Multithreaded/CollisionQuery/hkpCollisionQueryJobs.h>


#include <Common/Visualize/hkDebugDisplay.h>

#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>

#include <Demos/Physics2012/Api/Collide/Queries/ClosestPointsMultithreading/ClosestPointsMultithreadingApiDemo.h>

#include <Physics2012/Collide/Query/Multithreaded/CollisionQuery/hkpCollisionQueryJobQueueUtils.h>

#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>

#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>


//	# of simulated SPUs
#	define NUM_SPUS 1


ClosestPointsMultithreadingApiDemo::ClosestPointsMultithreadingApiDemo(hkDemoEnvironment* env)
	: hkDefaultPhysics2012Demo(env), m_useWorldQuery(false), m_numResults(2)
{
	//
	// Setup the camera.
	//
	{
		hkVector4 from( 0.0f, 4.0f, 12.0f);
		hkVector4 to  ( 0.0f,-2.0f,  0.0f);
		hkVector4 up  ( 0.0f, 1.0f,  0.0f);
		setupDefaultCameras(env, from, to, up);
	}

	//
	// Create the world.
	//
	{
		hkpWorldCinfo info;
		info.setupSolverInfo(hkpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM);

		// Set gravity to zero so body floats.
		info.m_gravity.set(0.0f, 0.0f, 0.0f);
		info.setBroadPhaseWorldSize( 100.0f );
		m_world = new hkpWorld(info);
		m_world->lock();

		// Disable backface culling, since we have MOPPs etc.
		disableBackFaceCulling();

		setupGraphics();
	}

	// register all agents(however, we put all objects into the some group,
	// so no collision will be detected
	hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

	{
		hkpGroupFilter* filter = new hkpGroupFilter();
		m_world->setCollisionFilter(filter);
		filter->removeReference();
	}

	//
	// Create some bodies (reuse the ShapeRayCastApi demo)
	//
	createBodies();

	m_world->unlock();


	/// Special case for this demo: we do not allow the # of active SPUs to drop to zero as this can cause a deadlock.
	m_allowZeroActiveSpus = false;

	// Register the collision query functions
	hkpCollisionQueryJobQueueUtils::registerWithJobQueue(m_jobQueue);

	// register the default addCdPoint() function; you are free to register your own implementation here though
	hkpFixedBufferCdPointCollector::registerDefaultAddCdPointFunction();

}


ClosestPointsMultithreadingApiDemo::~ClosestPointsMultithreadingApiDemo()
{
}


void ClosestPointsMultithreadingApiDemo::createBodies()
{
	//
	// Create a simple list shape
	//
	hkpListShape* listShape;
	{
		hkVector4 boxSize1(1,2,1);
		hkpBoxShape* box1 = new hkpBoxShape(boxSize1);

		hkVector4 boxSize2(1,1,3);
		hkpBoxShape* box2 = new hkpBoxShape(boxSize2);

		hkpShape* list[2] = {box1, box2};
		listShape = new hkpListShape(list, 2);

		box1->removeReference();
		box2->removeReference();
	}

	hkVector4 positions[3];
	positions[0].set(0 , 0, 0);
	positions[1].set(-3,-3, 0);
	positions[2].set( 3,-3, 0);

	//
	// Create some bodies
	//

	for (int i=0; i<3; i++)
	{
		m_bodies[i] = GameUtils::createRigidBody(listShape, 1.0f, positions[i]);
		m_world->addEntity(m_bodies[i]);
		m_bodies[i]->removeReference();
	}

	listShape->removeReference();

}


hkDemo::Result ClosestPointsMultithreadingApiDemo::stepDemo()
{

	// Handle user input
	{
		// Toggle between the two modes.
		if ( m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1) )
		{
			m_useWorldQuery = !m_useWorldQuery;
		}

		// Change the number of closest points to find.
		if ( m_useWorldQuery && m_env->m_gamePad->wasButtonPressed(HKG_PAD_DPAD_DOWN) && m_numResults > 1 )
		{
			--m_numResults;
		}
		// 8 is the maximum number of results (2 shapes in the origin list shape * 2 shapes in the destination list shape * 2 bodies)
		else if ( m_useWorldQuery && m_env->m_gamePad->wasButtonPressed(HKG_PAD_DPAD_UP) && m_numResults < 8 )
		{
			++m_numResults;
		}
	}

	if ( m_useWorldQuery )
	{
		return stepDemoWorldGetClosestPoints();
	}
	else
	{
		return stepDemoPairGetClosestPoints();
	}
}

void ClosestPointsMultithreadingApiDemo::makeFakeInput()
{
	if ( (m_physicsStepCounter % 60) == 0)
	{
		m_env->m_gamePad->forceButtonPressed(HKG_PAD_BUTTON_1);
	}
}

hkDemo::Result ClosestPointsMultithreadingApiDemo::stepDemoPairGetClosestPoints()
{

	hkStringBuf buf;
	buf.printf("Using pair closest point query");
	m_env->m_textDisplay->outputText(buf,10,280);
	//
	// Setup the output array where the resulting closest point/shortest distance (in this case only one) will be returned.
	//
	hkpRootCdPoint* closestPoint = hkAllocateChunk<hkpRootCdPoint>(1, HK_MEMORY_CLASS_DEMO);

	//
	// Setup the pairGetClosestPointraycast command.
	//
	hkpPairGetClosestPointsCommand* command;
	{
		command = hkAllocateChunk<hkpPairGetClosestPointsCommand>(1, HK_MEMORY_CLASS_DEMO);

		// Init input data.
		{
			// hkpWorldObject::getCollidable() needs a read-lock on the object
			m_bodies[0]->markForRead();
			m_bodies[2]->markForRead();

			command->m_collidableA = m_bodies[0]->getCollidable();
			command->m_collidableB = m_bodies[2]->getCollidable();

			m_bodies[2]->unmarkForRead();
			m_bodies[0]->unmarkForRead();
		}

		// Init output struct.
		{
			command->m_results			= closestPoint;
			command->m_resultsCapacity	= 1;
			command->m_numResultsOut	= 0;
		}
	}

	//
	// create the job header
	//
	hkpCollisionQueryJobHeader* jobHeader;
	{
		jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	}

	//
	// setup getClosestPointsJob
	//
	hkReal queryTolerance = 10.0f;
	m_world->markForRead();
	hkpPairGetClosestPointsJob pairGetClosestPointsJob(m_world->getCollisionInput(), jobHeader, command, 1, queryTolerance, m_semaphore, 5);
	m_world->unmarkForRead();

	{
		pairGetClosestPointsJob.setRunsOnSpuOrPpu();
		m_jobQueue->addJob( *reinterpret_cast<hkJobQueue::JobQueueEntry*>(&pairGetClosestPointsJob), hkJobQueue::JOB_HIGH_PRIORITY );
		m_threadPool->processJobQueue( m_jobQueue );
		m_threadPool->waitForCompletion();


		//
		// Wait for the one single job we started to finish.
		//
		m_semaphore->acquire();
	}

	//
	// process return values
	//
	{
		//
		// Draw the shortest distance returned by our own command.
		//
		if ( command->m_numResultsOut > 0 )
		{
			const hkContactPoint& pt = command->m_results[0].m_contact;
			drawContactPoint(pt);
		}
	}

	//
	// Free temporarily allocated memory.
	//
	hkDeallocateChunk(jobHeader,    1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(command,      1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(closestPoint, 1, HK_MEMORY_CLASS_DEMO);

	return hkDefaultPhysics2012Demo::stepDemo();
}


hkDemo::Result ClosestPointsMultithreadingApiDemo::stepDemoWorldGetClosestPoints()
{

	hkStringBuf buf;
	buf.printf("Using global closest point query with up to %i results", m_numResults);
	m_env->m_textDisplay->outputText(buf,10,280);

	//
	// Setup the output array where the resulting closest points/shortest distances (in this case two) will be returned.
	//
	hkpRootCdPoint* closestPoint = hkAllocateChunk<hkpRootCdPoint>(m_numResults, HK_MEMORY_CLASS_DEMO);

	//
	// Setup the hkWorldGetClosestPoints command.
	//
	hkpWorldGetClosestPointsCommand* command;
	{
		command = hkAllocateChunk<hkpWorldGetClosestPointsCommand>(1, HK_MEMORY_CLASS_DEMO);

		// Init input data.
		{
			// hkpWorldObject::getCollidable() needs a read-lock on the object
			m_bodies[0]->markForRead();
			command->m_collidable = m_bodies[0]->getCollidable();
			m_bodies[0]->unmarkForRead();
		}

		// Init output struct.
		{
			command->m_results			= closestPoint;
			command->m_resultsCapacity	= m_numResults;
			command->m_numResultsOut	= 0;
		}
	}

	//
	// create the job header
	//
	hkpCollisionQueryJobHeader* jobHeader;
	{
		jobHeader = hkAllocateChunk<hkpCollisionQueryJobHeader>(1, HK_MEMORY_CLASS_DEMO);
	}

	//
	// setup hkpWorldGetClosestPointsJob
	// Put the job on the queue, kick-off the PPU/SPU threads and wait for everything to finish.
	//
	{
		m_world->lockReadOnly();

		{
			hkReal queryTolerance = 10.0f;
			hkpWorldGetClosestPointsJob worldGetClosestPointsJob(m_world->getCollisionInput(), jobHeader, command, 1, m_world->m_broadPhase, queryTolerance, m_semaphore);
			worldGetClosestPointsJob.setRunsOnSpuOrPpu();
			m_jobQueue->addJob( *reinterpret_cast<hkJobQueue::JobQueueEntry*>(&worldGetClosestPointsJob), hkJobQueue::JOB_HIGH_PRIORITY );
		}

		m_threadPool->processJobQueue( m_jobQueue );
		m_threadPool->waitForCompletion();


		//
		// Wait for the one single job we started to finish.
		//
		m_semaphore->acquire();

		m_world->unlockReadOnly();
	}

	//
	// Draw closest distances as returned by the job.
	//
	{
		for (hkUint32 r = 0; r < command->m_numResultsOut; r++)
		{
			const hkContactPoint& pt = command->m_results[r].m_contact;
			drawContactPoint(pt);
		}
	}

	//
	// Free temporarily allocated memory.
	//
	hkDeallocateChunk(jobHeader,    1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(command,      1, HK_MEMORY_CLASS_DEMO);
	hkDeallocateChunk(closestPoint, m_numResults, HK_MEMORY_CLASS_DEMO);

	return hkDefaultPhysics2012Demo::stepDemo();
}


void ClosestPointsMultithreadingApiDemo::drawContactPoint(const hkContactPoint& contactPoint)
{
	// getPosition() returns a point on B by convention
	const hkVector4 pointOnBInWorld   = contactPoint.getPosition();

	// normal goes from B to A by convention
	const hkVector4 normalBtoAInWorld = contactPoint.getNormal();

	// pointOnA = pointOnB + dist * normalBToA
	hkVector4 pointOnAInWorld;
	pointOnAInWorld.setAddMul4(pointOnBInWorld, normalBtoAInWorld, contactPoint.getDistance());

	// Draw the line from A to B and the normal on B
	HK_DISPLAY_LINE(  pointOnAInWorld, pointOnBInWorld, hkColor::GREEN );
}



static const char helpString[] = \
"This demonstrates how to find the closest point(s) between a given pair of bodies using multithreading either on PPU or on SPU (if available).\n" \
"Press \x11 to switch between pair mode and global mode. The pair mode finds the closest points between two rigid bodies. The global mode\n"\
"finds a specified number of closest points between one body and all surrounding shapes, ordered by distance. Press up/down to change the\n"\
"number of closest points to find.";
HK_DECLARE_DEMO( ClosestPointsMultithreadingApiDemo, HK_DEMO_TYPE_OTHER, "Multithreading collision queries.", helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
