/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/ContactPointCallbacks/UserCollisionResponse/UserCollisionResponseDemo.h>
#include <Physics2012/Dynamics/Collide/ContactListener/hkpContactListener.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>

// This class adds contact points to an array which is then used to produce a force
// away from the contact point.
class MyCollisionResolution: public hkReferencedObject, public hkpContactListener
{
	public:
		virtual void contactPointCallback( const hkpContactPointEvent& event ) 
		{
			HK_ASSERT2( 0xf455ea07, event.m_source != hkpCollisionEvent::SOURCE_WORLD, "Do not add this listener to the world." );
			// Disable normal collision response.
			event.m_contactPointProperties->m_flags |= hkContactPointMaterial::CONTACT_IS_DISABLED;
			
			hkContactPoint cp = *event.m_contactPoint;
			
			// The impulse we apply is a function of the contact normal.
			hkVector4 impulse;
			{
				// If the listener was attached to the B body in this collision, then the contact normal
				// is the wrong way around.
				hkReal factor = ( event.m_source == hkpCollisionEvent::SOURCE_A ) ? 1.5f : -1.5f;
				impulse.setMul4( factor, event.m_contactPoint->getNormal() );
			}

			// The sphere is the body to which we added the listener (i.e. the source).
			hkpRigidBody* sphere = event.getBody( event.m_source );

			// We must call accessVelocities prior to applying the impulse, and call updateVelocities afterwards.
			event.accessVelocities( event.m_source );
			sphere->applyLinearImpulse( impulse );
			event.updateVelocities( event.m_source );
		}
};


UserCollisionResponseDemo::UserCollisionResponseDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env)
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(0.0f, 20.0f, 30.0f);
		hkVector4 to(0.0f, 0.0f, 0.0f);
		hkVector4 up(0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Setup the world
	//
	{
		hkpWorldCinfo info;
		info.m_gravity.set(0.0f, -9.8f, 0.0f);
		info.setBroadPhaseWorldSize(100.0f);
		m_world = new hkpWorld( info );
		m_world->lock();

		// Register ALL agents (though some may not be necessary)
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());

		setupGraphics();
	}

	// 
	// Create the table
	//
	{
		const float xdim = 10.0f;
		const float zdim = 10.0f;

		createStaticBox( m_world, 0.0f,     -1.0f,0.0f,		xdim,1.0f,zdim );
		createStaticBox( m_world, -xdim,  1.0f,0.0f,		   1.0f,1,zdim );
		createStaticBox( m_world,  xdim,  1.0f,0.0f,		   1.0f,1,zdim );
		createStaticBox( m_world,     0.0f,  1.0f, zdim,	xdim,1.0f,   1.0f );
		createStaticBox( m_world,     0.0f,  1.0f,-zdim,	xdim,1.0f,   1.0f );
	}

	// 
	// Create a body with a different collision response
	//
	{
		hkpConvexShape* shape = new hkpSphereShape( 0.8f );

		hkpRigidBodyCinfo sphereInfo;
		sphereInfo.m_shape = shape;
		sphereInfo.m_motionType = hkpMotion::MOTION_SPHERE_INERTIA;
		sphereInfo.m_position.set( 0.0f, 1.0f, 0.0f );
		hkMatrix3Util::_setDiagonal( 1.0f,1.0f,1.0f, sphereInfo.m_inertiaTensor );
		sphereInfo.m_mass = 1.0f;
		sphereInfo.m_contactPointCallbackDelay = 0;

		
		m_sphereRigidBody = new hkpRigidBody(sphereInfo);
		m_world->addEntity( m_sphereRigidBody );

		shape->removeReference();
		//
		//	Create our special Collision Resolution Object
		//
		m_collisionResolution = new MyCollisionResolution;
		m_sphereRigidBody->addContactListener( m_collisionResolution );
		
		// Let's start it off with a little horizontal velocity
		hkVector4 velocity(3.0f, 0.0f , -2.0f);
		m_sphereRigidBody->setLinearVelocity(velocity);

	}

	m_world->unlock();
}

UserCollisionResponseDemo::~UserCollisionResponseDemo()
{
	m_world->markForWrite();
	m_sphereRigidBody->removeContactListener( m_collisionResolution );
	m_collisionResolution->removeReference();
	m_sphereRigidBody->removeReference();
	m_world->unmarkForWrite();
}

void UserCollisionResponseDemo::createStaticBox( hkpWorld* world, float centerX, float centerY, float centerZ, float radiusX, float radiusY, float radiusZ )
{
	hkVector4 radii ( radiusX, radiusY, radiusZ );

	hkpShape* boxShape = new hkpBoxShape( radii , 0 );

	hkpRigidBodyCinfo boxInfo;

	boxInfo.m_motionType = hkpMotion::MOTION_FIXED;
	boxInfo.m_shape = boxShape;
	boxInfo.m_position.set( centerX, centerY, centerZ );

	hkpRigidBody* boxRigidBody = new hkpRigidBody(boxInfo);
	world->addEntity( boxRigidBody );
	boxRigidBody->removeReference();
	boxShape->removeReference();
}




static const char helpString[] = \
"We attach a contactListener to a sphere, forcing it away from any bodies with which it is in contact. "
"This is not a very accurate or robust form of collision resolution " 
"but illustrates how the user can implement their own in special cases if desired. "
"For example, controllers for characters, cars and other semi-physical objects. ";

HK_DECLARE_DEMO(UserCollisionResponseDemo, HK_DEMO_TYPE_PHYSICS_2012, "Using a contactListener to implement your own collision response", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
