/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */


#include <Demos/demos.h>
#include <Demos/Physics2012/Api/Collide/ContactPointCallbacks/ShapeKeysInCallback/ShapeKeysInCallbackDemo.h>
#include <Demos/DemoCommon/Utilities/ContactPainter/ContactPainter.h>

#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>

// Builds the landscape
#include <Demos/DemoCommon/Utilities/GameUtils/Landscape/FlatLand.h>

// We need to process and access some collision based info
#include <Physics2012/Dynamics/Collide/ContactListener/hkpContactListener.h>
#include <Physics2012/Collide/Agent/hkpProcessCollisionOutput.h>
#include <Physics2012/Utilities/Collide/ShapeUtils/ShapeKeyPath/hkpShapeKeyPath.h>

// We need to display some collision based info
#include <Common/Visualize/hkDebugDisplay.h>


// Returns true for four of the multiSphereShape's balls in a nice pattern.
static bool isVirtualSubShape( hkpShapeKey key )
{
	return 1 == ( ( key ^ ( key >> 1 ) ^ ( key >> 2 ) ) & 1 );
}

// Draws lines around the appropriate subshape of the provided 
static void highlightTouchedTriangle( const hkpTriangleShape* triangle )
{
	// Elevate the triangle slightly, so it not obscured by the landscape.
	hkVector4 vec0( 0.0f, 0.1f, 0.0f );	vec0.add4( triangle->getVertex(0) );
	hkVector4 vec1( 0.0f, 0.1f, 0.0f );	vec1.add4( triangle->getVertex(1) );
	hkVector4 vec2( 0.0f, 0.1f, 0.0f );	vec2.add4( triangle->getVertex(2) );

	//	Draw the border of the triangle
	HK_DISPLAY_LINE(vec0, vec1, hkColor::WHITE);
	HK_DISPLAY_LINE(vec1, vec2, hkColor::WHITE);
	HK_DISPLAY_LINE(vec2, vec0, hkColor::WHITE);
}

void MyContactListener::contactPointCallback( const hkpContactPointEvent& event )
{
	// Disable contacts with some subshapes. Since this listener is attached to a specific
	// body whose shape hierarchy we know, we simply use the leaf shape key.
	const hkpShapeKey *const shapeKeysOfSource = event.getShapeKeys( event.m_source );
	HK_ASSERT2( 0x1443ab78, shapeKeysOfSource, "This listener was not attached to the expected body." );
	if ( isVirtualSubShape( *shapeKeysOfSource ) )
	{
		event.m_contactPointProperties->m_flags |= hkContactPointMaterial::CONTACT_IS_DISABLED;
	}
	else
	{
		// For those contacts we are not disabling, we get the leaf subshape of the collision
		// partner and, if it is a triangle shape, highlight it. We don't know the structure
		// of the collision partner's shape hierarchy, so we use the shape key path iterator to
		// traverse it.

		const hkpShape* leafShape = HK_NULL;
		hkpShapeKeyPath path( event, 1 - event.m_source );
		hkpShapeKeyPath::Iterator it = path.getIterator();
		while (it.isValid())
		{
			leafShape = it.getShape();
			it.next();	
		}

		if ( leafShape->getType() == hkcdShapeType::TRIANGLE )
		{
			highlightTouchedTriangle( static_cast<const hkpTriangleShape*>( leafShape ) );
		}
	}
}

ShapeKeysInCallbackDemo::ShapeKeysInCallbackDemo( hkDemoEnvironment* env) 
: hkDefaultPhysics2012Demo(env) 
{
	//
	// Setup the camera
	//
	{
		hkVector4 from(6.0f, 6.0f, 12.0f);
		hkVector4 to(0.0f, 1.0f, 0.0f);
		hkVector4 up(0.0f, 1.0f, 0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	//
	// Create the world
	//
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_CONTINUOUS;

		// Turn off deactivation so we can see continuous contact point processing
		info.m_enableDeactivation = false;
		m_world = new hkpWorld( info );

		setupGraphics();
	}

	m_world->lock();

	//
	// Register the agents
	//
	{
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Create a landscape with a tallish shape hierarchy.
	//
	{
		// A hkpListShape containing a hkpMoppBvTreeShape containing an hkpExtendedMeshShape
		// containing hkpTriangleShapes.
		{
			hkpShape* shapes[1];
			{
				m_flatLand = new FlatLand(20);
				hkVector4 scaling( 1.0f, 0.15f, 1.0f );
				m_flatLand->setScaling( scaling );
				shapes[0] = m_flatLand->createMoppShape();
			}
			hkpListShape* listShape = new hkpListShape( shapes, 1 );
			shapes[0]->removeReference();

			hkpRigidBodyCinfo groundInfo;
			{
				groundInfo.m_shape = listShape;
				groundInfo.m_motionType = hkpMotion::MOTION_FIXED;
				// Ensure shapeKeys are stored for the full hierarchy.
				groundInfo.m_numShapeKeysInContactPointProperties = -1;
				// Add some bounce.
				groundInfo.m_restitution = 0.8f;
				groundInfo.m_friction = 1.0f;
			}
			hkpRigidBody* landscape = new hkpRigidBody( groundInfo );
			listShape->removeReference();

			m_world->addEntity( landscape );

			// Color the landscape in transparent dark green
			HK_SET_OBJECT_COLOR( (hkUlong)(landscape->getCollidable()), 0xa0008000 );

			landscape->removeReference();
		}
	}

	//
	// Create a moving object consisting of 8 spheres.
	//
	{
		hkpRigidBodyCinfo multiSpheresInfo;
		{
			hkpSphereShape* sphere = new hkpSphereShape(0.2f);
			
			hkReal size = 0.5f;
			hkpShape* shapes[8];
			for (int i = 0; i < 8; i++)
			{
				hkReal px = i&0x1 ? -size : size;
				hkReal py = i&0x2 ? -size : size;
				hkReal pz = i&0x4 ? -size : size;

				hkVector4 translation( px, py, pz );

				shapes[i] = new hkpConvexTranslateShape( sphere, translation );

			}
			sphere->removeReference();

			hkpListShape* list = new hkpListShape(shapes, 8);
			for(int i = 0; i < 8; i++)
			{
				shapes[i]->removeReference();
			}

			multiSpheresInfo.m_shape = list;
			hkpInertiaTensorComputer::setShapeVolumeMassProperties( multiSpheresInfo.m_shape, 1.0f, multiSpheresInfo );

			multiSpheresInfo.m_qualityType = HK_COLLIDABLE_QUALITY_CRITICAL;
			multiSpheresInfo.m_motionType = hkpMotion::MOTION_BOX_INERTIA;

			// Place the body so it bounces interestingly
			multiSpheresInfo.m_position.set(0.0f, 5.0f, 0.0f);
			hkVector4 axis(1.0f, 2.0f, 3.0f);
			axis.normalize3();
			multiSpheresInfo.m_rotation.setAxisAngle(axis, -0.7f);

			// Add some bounce.
			multiSpheresInfo.m_restitution = 0.5f;
			multiSpheresInfo.m_friction = 0.1f;

			// Ensure enough shape keys are stored.
			multiSpheresInfo.m_numShapeKeysInContactPointProperties = -1;
		}
		m_body = new hkpRigidBody( multiSpheresInfo );
		multiSpheresInfo.m_shape->removeReference();

		m_world->addEntity( m_body );

		// Add the collision event listener to the rigid body
		m_listener = new MyContactListener();
		m_body->addContactListener( m_listener );
	}

	m_world->unlock();
}

hkDemo::Result ShapeKeysInCallbackDemo::stepDemo()
{
	return hkDefaultPhysics2012Demo::stepDemo();
}

ShapeKeysInCallbackDemo::~ShapeKeysInCallbackDemo()
{
	m_world->markForWrite();
	m_body->removeContactListener( m_listener );
	m_listener->removeReference();
	m_world->removeEntity( m_body );
	m_body->removeReference();
	m_flatLand->removeReference();
	m_world->unmarkForWrite();
}



static const char helpString[] = \
"This demo demonstrates the use shape keys in contact point callbacks.\n\n"
"A contactPointListener is attached to the rigid body consisting of "
"eight balls. The listener disables contact points between four of the "
"balls and highlights triangles the body is colliding with.";

HK_DECLARE_DEMO(ShapeKeysInCallbackDemo, HK_DEMO_TYPE_PHYSICS_2012, "Using shape keys in contact point callbacks", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
