/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Demos/Physics2012/Api/Collide/Broadphase/Culling/BroadphaseCullingDemo.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Physics2012/Internal/BroadPhase/TreeBroadPhase/hkpTreeBroadPhase.h>
#include <Physics2012/Dynamics/World/Simulation/hkpSimulation.h>

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Types/Geometry/Aabb/hkAabbUtil.h>
#include <Common/Visualize/hkDebugDisplay.h>

// Variants
static const BroadphaseCullingDemo::Variant	g_variants[]=
{
	{ "Rigid bodies", "", 1024, 0 },
	{ "User objects", "", 0, 1024 },
	{ "Rigid bodies and user objects", "", 1024, 1024 }
};

// Compute a plane equation from triangle vertices, store offset to the origin in W.
static HK_FORCE_INLINE hkVector4	computePlaneFromTriangle(const hkVector4& a, const hkVector4& b, const hkVector4& c)
{
	hkVector4	ab; ab.setSub4(b,a);
	hkVector4	ac; ac.setSub4(c,a);
	hkVector4	plane;
	plane.setCross(ab, ac);
	plane.normalize3();
	plane(3) = -plane.dot3(a);
	return plane;
}

// Generate color from the visible spectrum.
static hkColor::Argb	spectrumColor(hkReal s)
{
	const hkVector4	palette[]={	hkVector4(0,0,0,1), hkVector4(0,0,1,1),
								hkVector4(0,1,1,1), hkVector4(0,1,0,1),
								hkVector4(1,1,0,1), hkVector4(1,0,0,1)};
	const int		numColors(sizeof(palette)/sizeof(palette[0]));
	hkVector4		color(0,0,0,0);
	if(s<=0)		color=palette[0],s=0;
	else if(s>=1)	color=palette[numColors-1],s=1;
	else
	{
		hkReal	ms=s*(numColors-1);
		int		c=(int)ms;
		hkReal	u=ms-c;
		color.setInterpolate4(palette[c],palette[c+1],u);
	}
	return hkColor::rgbFromFloats(color(0),color(1),color(2),1);
}

// Display a solid quad.
static void displaySolidQuad(const hkVector4& a, const hkVector4& b, const hkVector4& c, const hkVector4& d, hkColor::Argb color)
{
	HK_DISPLAY_TRIANGLE(a, b, c, color);
	HK_DISPLAY_TRIANGLE(c, b, d, color);
}

// Display a solid AABB.
static void displaySolidAABB(const hkAabb& aabb, hkColor::Argb color)
{
	const hkVector4	minMax[2]={aabb.m_min, aabb.m_max};
	hkVector4		vertices[8];
	
	for(int i=0; i<8; ++i)
	{
		for(int j=0; j<3; ++j)
		{
			vertices[i](j) = minMax[(i>>j)&1](j);
		}		
	}

	const int quads[][4]=	{ {0,2,1,3}, {4,5,6,7}, {0,1,4,5}, {1,3,5,7}, {3,2,7,6}, {2,0,6,4} };

	for(int i=0; i < (int)(sizeof(quads)/sizeof(quads[0])); ++i)
	{
		displaySolidQuad(vertices[quads[i][0]], vertices[quads[i][1]], vertices[quads[i][2]], vertices[quads[i][3]], color);
	}
}

//
BroadphaseCullingDemo::BroadphaseCullingDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env), 
	m_rand(180673)
{
	const Variant& variant	=	g_variants[m_variantId];
	// Setup the camera and graphics
	{
		// setup the camera
		hkVector4 from(196.0f, 0, 48);
		hkVector4 to  (0.0f, 0.0f,  0.0f);
		hkVector4 up  (0.0f, 0.0f,  1.0f);
		setupDefaultCameras( env, from, to, up, 1.0f, 5000.0f );
	}


	hkAabb	worldAabb; worldAabb.setEmpty();
	{
		hkpWorldCinfo cinfo;
		cinfo.m_gravity.setAll(0);
		cinfo.m_broadPhaseWorldAabb.m_max.set( 100,  100,  8);
		cinfo.m_broadPhaseWorldAabb.m_min.setNeg4(	cinfo.m_broadPhaseWorldAabb.m_max );

		// We need an hkpTreeBroadPhase to perform culling.
		cinfo.m_broadPhaseType = hkpWorldCinfo::BROADPHASE_TYPE_TREE;
		
		worldAabb = cinfo.m_broadPhaseWorldAabb;
		m_world = new hkpWorld(cinfo);
		m_world->lock();

		setupGraphics();
		setAutoDisplayCachingEnabled(true);
	}

	{
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	//
	// Create a random batch of boxes in the world
	//
	if(variant.m_numRigidBodies)
	{
		
		hkpMotion::MotionType motionType = hkpMotion::MOTION_SPHERE_INERTIA;
		hkPseudoRandomGenerator rand(100);

		hkArray<const hkpCollidable*>	collidables;

		GameUtils::createRandomBodies(m_world, worldAabb, variant.m_numRigidBodies, motionType, &rand, collidables);		
	}

	//
	// Create random user object as phantoms.
	//
	if(variant.m_numUserObjects)
	{
		hkpSphereShape*					shape = new hkpSphereShape(1.0f);
		hkArray<hkpBroadPhaseHandle*>	handles;
		hkArray<hkAabb>					aabbs;
		
		for(int i=0; i<variant.m_numUserObjects; ++i)
		{
			// Generate a random transform
			hkTransform transform = hkTransform::getIdentity();
			m_rand.getRandomVector11(transform.getTranslation());			
			transform.getTranslation().mul4(50.0f);
			
			// Create the phantom
			hkpSimpleShapePhantom* phantom = new hkpSimpleShapePhantom(shape, transform);


				// Keep the phantom around.
				m_userObjects.pushBack(phantom);

				// Add the broadphase handle to an array.
				handles.pushBack( (hkpBroadPhaseHandle*) &phantom->getCollidable()->m_broadPhaseHandle);

				// Add the AABB to an array.
				phantom->calcAabb(aabbs.expandOne());
		}
		shape->removeReference();

		// Add these user objects to the hybrid broad-phase.
		hkpTreeBroadPhase*	treeBp = (hkpTreeBroadPhase*) m_world->getBroadPhase();
		treeBp->addUserObjects(handles.getSize(), handles.begin(), aabbs.begin());

	}

	m_rand.getRandomRotation(m_orientations[0]);
	m_rand.getRandomRotation(m_orientations[1]);
	m_animTime					=	0;
	m_cullFarPlane				=	false;
	m_useVelocityForUserObjects	=	true;
	m_sort						=	true;

	// Optimize broad-phase.
	((hkpTreeBroadPhase*) m_world->getBroadPhase())->fullOptimize();

	m_world->unlock();	
}

//
BroadphaseCullingDemo::~BroadphaseCullingDemo()
{
	hkReferencedObject::removeReferences(m_userObjects.begin(), m_userObjects.getSize());
}

//
hkDemo::Result BroadphaseCullingDemo::stepDemo()
{
	hkDemo::Result result = hkDefaultPhysics2012Demo::stepDemo();
	
	// Lock the world and get the tree broad-phase.
	m_world->lock();
	hkpTreeBroadPhase*	broadphase = static_cast<hkpTreeBroadPhase*>(m_world->getBroadPhase());

	// Toggle far plane culling.
	if( m_env->m_window->getKeyboard().wasKeyPressed('F') || m_env->m_window->getGamePad(0).wasButtonPressed(HKG_PAD_BUTTON_L2) )
	{
		m_cullFarPlane = !m_cullFarPlane;
	}

	// Toggle front to back sorting.
	if( m_env->m_window->getKeyboard().wasKeyPressed('B') || m_env->m_window->getGamePad(0).wasButtonPressed(HKG_PAD_BUTTON_1))
	{
		m_sort = !m_sort;
	}

	// Toggle velocity for user objects
	if( m_env->m_window->getKeyboard().wasKeyPressed('V') || m_env->m_window->getGamePad(0).wasButtonPressed(HKG_PAD_BUTTON_2))
	{
		m_useVelocityForUserObjects = !m_useVelocityForUserObjects;
	}

	// Optimize broadphase.
	if( m_env->m_window->getKeyboard().wasKeyPressed('O'))
	{
		broadphase->fullOptimize();
	}

	// Remove one user object at random.
	if( m_env->m_window->getKeyboard().wasKeyPressed('R') && m_userObjects.getSize())
	{
		const int				index = m_rand.getRand32()%m_userObjects.getSize();
		hkpBroadPhaseHandle*	handle = (hkpBroadPhaseHandle*) &m_userObjects[index]->getCollidable()->m_broadPhaseHandle;
		
		broadphase->removeUserObjects(1, &handle);
		
		m_userObjects[index]->removeReference();
		m_userObjects.removeAt(index);
	}

	// Animate user objects
	{
		hkArray<hkpBroadPhaseHandle*>	handles; handles.reserve(m_userObjects.getSize());
		hkArray<hkAabb>					aabbs; aabbs.reserve(m_userObjects.getSize());
		hkArray<hkVector4>				velocities; velocities.reserve(m_userObjects.getSize());
		const hkSimdReal				invDt = 1.0f / m_world->m_simulation->getPhysicsDeltaTime();
		for(int i=m_userObjects.getSize()-1,j=0; j<m_userObjects.getSize(); i=j++)
		{
			// Compute new position.
			hkTransform			src = m_userObjects[i]->getTransform();
			const hkTransform&	dst = m_userObjects[j]->getTransform();
			const hkVector4		oldPosition = src.getTranslation();
			const hkSimdReal	speed = 0.05f;
			const hkSimdReal	radius = 50.0f;
			hkVector4			newPosition;
			newPosition.setInterpolate4(src.getTranslation(), dst.getTranslation(), speed);
			newPosition.mul4(radius / newPosition.length3());

			// Compute velocity.
			hkVector4			velocity;
			velocity.setSub4(newPosition, oldPosition);
			velocity.mul4(invDt);

			// Set new position.
			src.setTranslation(newPosition);
			m_userObjects[i]->setTransform(src);

			// Add velocity to an array. (optional)
			velocities.pushBack(velocity);

			// Add the broadphase handle to an array.
			handles.pushBack( (hkpBroadPhaseHandle*) &m_userObjects[i]->getCollidable()->m_broadPhaseHandle);

			// Add the AABB to an array.
			m_userObjects[i]->calcAabb(aabbs.expandOne());
		}
		
		// Update user objects to the tree broad-phase.
		HK_TIME_CODE_BLOCK("Update user objects",HK_NULL);
		broadphase->updateUserObjects(m_userObjects.getSize(), handles.begin(), aabbs.begin(), m_useVelocityForUserObjects? velocities.begin() : HK_NULL);

	}
	
	// Draw user objects
	for(int i=0; i<m_userObjects.getSize(); ++i)
	{
		hkAabb		aabb; m_userObjects[i]->calcAabb(aabb);
		hkVector4	center; aabb.getCenter(center);
		HK_DISPLAY_BOUNDING_BOX(aabb, hkColor::GREY50);
		HK_DISPLAY_STAR(center, 0.25f, hkColor::WHITE);
	}
	
	// Animate frustum
	const hkReal	spanDuration = 3.0f;
	m_animTime	+=	1.0f / 60.0f / spanDuration;
	while(m_animTime > 1.0f)
	{
		m_animTime			-=	1.0f;
		m_orientations[0]	=	m_orientations[1];
		m_rand.getRandomRotation(m_orientations[1]);
	}
	hkReal			splinePoly = (3 - 2*m_animTime) * m_animTime*m_animTime;
	hkQuaternion	currentOrientation;
	currentOrientation.setSlerp(m_orientations[0], m_orientations[1], hkSimdReal(splinePoly));

	// Create frustum vertices
	hkVector4	nearPlaneQuad[4];
	nearPlaneQuad[0].set(-10,-10,0);
	nearPlaneQuad[1].set(+10,-10,0);
	nearPlaneQuad[2].set(+10,+10,0);
	nearPlaneQuad[3].set(-10,+10,0);

	hkVector4	farPlaneQuad[4];
	hkSimdReal	fovFactor = 4.0f;
	hkReal		farOffset = 75.0f;
	for(int i=0; i<4; ++i)
	{
		farPlaneQuad[i].setMul4(fovFactor, nearPlaneQuad[i]);
		farPlaneQuad[i](2) += farOffset;
	}
	
	// Transform frustum
	hkVector4	direction;
	direction.set(0,0,1,0);
	direction.setRotatedDir(currentOrientation, direction);
	
	for(int i=0; i<4; ++i)
	{
		nearPlaneQuad[i].setRotatedDir(currentOrientation, nearPlaneQuad[i]);
		farPlaneQuad[i].setRotatedDir(currentOrientation, farPlaneQuad[i]);
	}

	// Draw frustum.
	HK_DISPLAY_LINE(nearPlaneQuad[0], nearPlaneQuad[1], hkColor::YELLOW);
	HK_DISPLAY_LINE(nearPlaneQuad[1], nearPlaneQuad[2], hkColor::YELLOW);
	HK_DISPLAY_LINE(nearPlaneQuad[2], nearPlaneQuad[3], hkColor::YELLOW);
	HK_DISPLAY_LINE(nearPlaneQuad[3], nearPlaneQuad[0], hkColor::YELLOW);

	if(m_cullFarPlane)
	{
		HK_DISPLAY_LINE(farPlaneQuad[0], farPlaneQuad[1], hkColor::YELLOW);
		HK_DISPLAY_LINE(farPlaneQuad[1], farPlaneQuad[2], hkColor::YELLOW);
		HK_DISPLAY_LINE(farPlaneQuad[2], farPlaneQuad[3], hkColor::YELLOW);
		HK_DISPLAY_LINE(farPlaneQuad[3], farPlaneQuad[0], hkColor::YELLOW);
	}

	HK_DISPLAY_LINE(farPlaneQuad[0], nearPlaneQuad[0], hkColor::YELLOW);
	HK_DISPLAY_LINE(farPlaneQuad[1], nearPlaneQuad[1], hkColor::YELLOW);
	HK_DISPLAY_LINE(farPlaneQuad[2], nearPlaneQuad[2], hkColor::YELLOW);
	HK_DISPLAY_LINE(farPlaneQuad[3], nearPlaneQuad[3], hkColor::YELLOW);

	// Compute frustum planes.
	hkVector4	planes[6];		

		// Side planes
	planes[0] = computePlaneFromTriangle(nearPlaneQuad[0], nearPlaneQuad[1], farPlaneQuad[0]);
	planes[1] = computePlaneFromTriangle(nearPlaneQuad[1], nearPlaneQuad[2], farPlaneQuad[1]);
	planes[2] = computePlaneFromTriangle(nearPlaneQuad[2], nearPlaneQuad[3], farPlaneQuad[2]);
	planes[3] = computePlaneFromTriangle(nearPlaneQuad[3], nearPlaneQuad[0], farPlaneQuad[3]);

		// Near plane
	planes[4] = computePlaneFromTriangle(nearPlaneQuad[0], nearPlaneQuad[2], nearPlaneQuad[1]);		

		// Far plane
	planes[5] = computePlaneFromTriangle(farPlaneQuad[0], farPlaneQuad[1], farPlaneQuad[2]);

	// Cull using frustum.
	hkArray<const hkpBroadPhaseHandle*>	handles;
	{
		HK_TIME_CODE_BLOCK("Culling",this);
		handles.reserve(4096 / sizeof(const hkpBroadPhaseHandle*));
		if(m_sort)
			broadphase->queryConvexSorted(direction, planes, m_cullFarPlane? 6 : 5, handles);
		else
			broadphase->queryConvex(planes, m_cullFarPlane? 6 : 5, handles);
	}

	// Draw objects AABB visible from the frustum.	
	for(int i=0; i<handles.getSize(); ++i)
	{
		hkAabb	aabb; broadphase->getAabb(handles[i], aabb);
		hkColor::Argb	color = hkColor::YELLOW;
		if(m_sort)
		{
			color = spectrumColor((i+1)/(hkReal)handles.getSize());
		}
		color = hkColor::rgbFromChars(hkColor::getRedAsChar(color), hkColor::getGreenAsChar(color), hkColor::getBlueAsChar(color), 127);
		displaySolidAABB(aabb, color);
	}

	// Display current settings.
	hkStringBuf			str;
	static const char*	sortName[]={"None","Front to back"};
	str.printf("Far plane culling(F): %s\nSort(B): %s\nVelocities(V): %s", m_cullFarPlane?"ON":"OFF", sortName[m_sort?1:0], m_useVelocityForUserObjects?"ON":"OFF");
	m_env->m_textDisplay->outputText(str.cString(), 16, (int)getWindowHeight() - 64);

	m_world->unlock();

	return result;
}



static const char helpString[] = \
"The tree broadphase can be used for culling queries (i.e. frustum).";

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( BroadphaseCullingDemo, HK_DEMO_TYPE_PHYSICS_2012 | HK_DEMO_TYPE_DONT_BOOTSTRAP, BroadphaseCullingDemo::Variant, g_variants, helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
