/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#define SHADOW_EPSILON 0.0001

cbuffer perObject {
	float4x4 g_mWorld			: World;
	float4x4 g_mPrevWorld			: PrevWorld;
	float4x4 g_mWorldInv;
	float4x4 g_mWorldView		: WorldView;
	float4x4 g_mWorldViewProj	: WorldViewProjection;
	float4x4 g_mWorldInvTranspose			: WorldInverseTranspose;
};

cbuffer perCamera {
	float4x4 g_mProj			: Proj;
	float4x4 g_mViewProj;
	float4x4 g_mViewInv			: ViewInverse;
	float4x4 g_mPrevViewProj;

	float4x4 g_mViewToLightProj; // for shadow maps
	float4x4 g_mViewToLightProj1;
	float4x4 g_mViewToLightProj2;
	float4x4 g_mViewToLightProj3;
};

//
// Textures, set assignment so that the behaviour is the same no mater what shader is using it 
//
// first 4 samplers will be shadowe maps if VSM used
sampler g_sSamplerShared  : register(s0);
sampler g_sSamplerClamped : register(s1); 
sampler g_sSamplerShadow  : register(s2);


// Overlapping cube/2d semantics for texture slots not supported in DX10 at the moment
// so have to use #defines to limit pixel shaders visible on compile
// and then select the correct def here (expand to 2darray etc)
#ifdef HKG_SKYMAP
	TextureCube<float4> TexCube0 : register(t0);
	#define Tex2D0 Tex2D1
#else
	Texture2D<float4> Tex2D0 : register(t0);
#endif

Texture2D<float4> Tex2D1 : register(t1);
Texture2D<float4> Tex2D2 : register(t2);
Texture2D<float4> Tex2D3 : register(t3);

Texture2D<float4> Tex2D4 : register(t4);
Texture2D<float4> Tex2D5 : register(t5);
Texture2D<float4> Tex2D6 : register(t6);
Texture2D<float>  DepthPeelSecondDepth : register(t7);

	
//
// Lights 
//
cbuffer perLight {
	float3 g_vLightShadowStartPosWorld; 
	float3 g_vLightShadowStartPosView;
	float4 g_vLightDir : Direction  = {0.0f, 1.0f, 0.0f, 1.0f}; // world space
	float4 g_vLightDirView : Direction;  // view space, for PSVSM
	float4 g_cLightColor : Diffuse = {1.0f, 1.0f, 1.0f, 1.0f};
};
 
// 
// Material
//
cbuffer perMaterial {
	float4 g_cAmbientColor	: MaterialAmbient = {0.0f, 0.0f, 0.0f, 1.0f};
	float4 g_cDiffuseColor	: MaterialDiffuse = {1.0f, 1.0f, 1.0f, 1.0f};
	float4 g_cSpecularColor	: MaterialSpecular = {1.0f, 1.0f, 1.0f, 1.0f};
	float  g_cSpecularPower	: Power = 30.0; // 1..128
};

cbuffer generalSettings {
	float  g_iShadowMapSize		= 1024;
	float  g_fShadowMapDistance = 100; // Distance from light / light proj to end of shadow map volume
	float  g_fShadowVsmEpsilon  = 0.0001; 
	float  g_fShadowVsmBias     = 0.0001; 

	float4 m_fShadowSplitDistances; // view space z distances to the far end of each map

	float4 g_cFogColor;
	float4 g_iFogParams;
	float4 g_iDepthParams; // x == LinearDepth output enabled. y == AlphaBlend enabled. z == Doing a reverse depth peel pass (secondary texture in sampler #3)
	float4 g_fColorParams; // Color range correction enabled?, 1/Gamma, pow(2,Exposure)
};

//
// Structures
//
struct vertexInputT2 {
    float3 position			: POSITION;
    float3 normal			: NORMAL;
 #ifdef TEXTURE_ONE
    float2 texCoord0		: TEXCOORD0;
 #endif
 #ifdef TEXTURE_TWO
    float2 texCoord1		: TEXCOORD1;
 #endif
    float4 diffAmbColor		: COLOR0;
};

struct vertexInputNLT2 {
    float3 position			: POSITION;
 #ifdef TEXTURE_ONE
    float2 texCoord0		: TEXCOORD0;
 #endif
 #ifdef TEXTURE_TWO
    float2 texCoord1		: TEXCOORD1;
 #endif
    float4 diffAmbColor		: COLOR0;
};

struct vertexInputPosOnly {
    float3 position			: POSITION;
};

struct vertexInputNoColorT2 {
    float3 position			: POSITION;
    float3 normal			: NORMAL;
 #ifdef TEXTURE_ONE
    float2 texCoord0		: TEXCOORD0;
 #endif
 #ifdef TEXTURE_TWO
     float2 texCoord1		: TEXCOORD1;
 #endif
};

struct vertexInputNoColorNLT2 {
    float3 position			: POSITION;
 #ifdef HKG_SKYMAP
	float3 normal			: NORMAL;
 #endif
 #ifdef TEXTURE_ONE
    float2 texCoord0		: TEXCOORD0;
 #endif
 #ifdef TEXTURE_TWO
     float2 texCoord1		: TEXCOORD1;
 #endif
};


struct vertexInputT1B {
	float4 position  : POSITION; 	
	float3 normal    : NORMAL; 	
#ifdef TEXTURE_ONE
 	float2 texCoord0 : TEXCOORD0; 	
#endif
	float3 tangent   : TANGENT; // well known slot
	float3 binormal  : BINORMAL;	
};



struct vertexOutputT1B {
	float4 position    : SV_Position;  // clip space
 	float2 texCoord0   : TEXCOORD0;
	float3 posView 	   : TEXCOORD1; 
    float3 L           : TEXCOORD2; // tangent space
	float3 H           : TEXCOORD3;
};

struct vertexOutputT2 {
    float4 position			: SV_Position;
    float3 texCoord0		: TEXCOORD0;
    float2 texCoord1		: TEXCOORD1;
    float3 posView 		    : TEXCOORD2; 
    float4 diffAmbColor		: COLOR0;
    float4 specCol			: COLOR1;
};

struct vertexOutputShadowDepth {
    float4 position			: SV_Position;
    float4 depthZZZW		: TEXCOORD0;
};

struct vertexOutputShadowDepthVSM {
    float4 position			: SV_Position;
    float3 posWorld		    : TEXCOORD0;
};

struct vertexOutputShadowT2 {
    float4 position			: SV_Position;
    float2 texCoord0		: TEXCOORD0;
    float2 texCoord1		: TEXCOORD1;
    float4 posLight			: TEXCOORD2; // ShadowMap 0 lookup coords
    float3 posView 		    : TEXCOORD3; 
    #ifdef HKG_SHADOWS_VSM
  		float4 posLight1		: TEXCOORD4; // ShadowMap 1 lookup coords
		float4 posLight2		: TEXCOORD5; // ShadowMap 2 lookup coords
        float4 posLight3		: TEXCOORD6; // ShadowMap 3 lookup coords
    #endif
    float4 diffAmbColor		: COLOR0;
    float4 specCol			: COLOR1; 
};

struct vertexOutputShadowT1B {
    float4 position			: SV_Position;
    float2 texCoord0		: TEXCOORD0;
    float4 posLight			: TEXCOORD1; // ShadowMap 0 lookup coords
    float3 posView 		    : TEXCOORD2; 
    #ifdef HKG_SHADOWS_VSM
  		float4 posLight1		: TEXCOORD3; // ShadowMap 1 lookup coords
		float4 posLight2		: TEXCOORD4; // ShadowMap 2 lookup coords
        float4 posLight3		: TEXCOORD5; // ShadowMap 3 lookup coords
        float3 L				: TEXCOORD6;
		float3 H				: TEXCOORD7;
	#else
	    float3 L				: TEXCOORD3;
		float3 H				: TEXCOORD4;
	#endif
};

struct pixelOutput
{
    float4 color			: SV_Target0;  
	float4 pzDepth			: SV_Target1; 
};

float4 computeFog( in float viewZ, in float4 c )
{
	float scale = 0;
	float depth = viewZ;
	if (g_iFogParams.x > 2 ) // EXP2
	{	
		float ddensity = depth*g_iFogParams.w;
		scale = 1.0 / exp( ddensity*ddensity ); // 1/(e^((d*density)^2))
	}
	else if (g_iFogParams.x > 1 ) // EXP
	{
		float ddensity = depth*g_iFogParams.w;
		scale = 1.0 / exp( ddensity ); // 1/(e^(d*density))
	}
	else if (g_iFogParams.x > 0 ) // LINEAR
	{
		scale = (g_iFogParams.z - depth) / (g_iFogParams.z - g_iFogParams.y);
	}
	
	scale = clamp(scale, 0, 1);
	return ( (1 - scale) * float4(g_cFogColor.xyz,1) ) + ( scale * c); 
}

float4 colorRangeCorrect( float4 rgba )
{
	float3 c0 = (( 1 - g_fColorParams[0] ) * rgba.rgb );
	float3 c1 =  g_fColorParams[0] * pow( (g_fColorParams[2] * rgba.rgb ), g_fColorParams[1] ); // Exposure and Gamma
	return float4(c0 + c1, rgba.a);
}

//////////////////////////////////////////////////////////////
// 
// Vertex Shaders
//

// No lighting, just transform and pass on texture (and material as diffuse)
vertexOutputT2 VertNoLight(vertexInputNoColorNLT2 In) 
{
    vertexOutputT2 Output;
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Output.position = mul( viewPos, g_mProj);
    Output.posView = viewPos;
	Output.diffAmbColor = g_cDiffuseColor;
    #ifdef TEXTURE_ONE
	#ifdef HKG_SKYMAP
		Output.texCoord0 = normalize(In.position.xyz);
	#else
		Output.texCoord0 = In.texCoord0.xyy;
	#endif
	#else
		Output.texCoord0 = 0;
	#endif	
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif	
		
	Output.specCol = float4(0,0,0,0);
    return Output;
}

// No lighting, just transform and pass on texture (and material as diffuse)
vertexOutputShadowT2 VertNoLightShadowProj(vertexInputNoColorNLT2 In) 
{
    vertexOutputShadowT2 Output;
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Output.position = mul( viewPos, g_mProj);
    Output.diffAmbColor = g_cDiffuseColor;
    #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0;
    #else
    	Output.texCoord0 = 0;
    #endif
    #ifdef TEXTURE_TWO
	    Output.texCoord1 = In.texCoord1;
    #else
		Output.texCoord1 = 0;
    #endif
		
	Output.specCol = float4(0,0,0,0);
    
    	// project pos into light space
    Output.posLight = mul( viewPos, g_mViewToLightProj );

    #ifdef HKG_SHADOWS_VSM
		Output.posLight1 = mul( viewPos, g_mViewToLightProj1 );
		Output.posLight2 = mul( viewPos, g_mViewToLightProj2 );
		Output.posLight3 = mul( viewPos, g_mViewToLightProj3 );
   	#endif

    Output.posView = viewPos;

    return Output;
}

// No lighting, just transform and pass on texture (and vertex color as diffuse)
vertexOutputT2 VertNoLightVC(vertexInputNLT2 In) 
{
    vertexOutputT2 Output;
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Output.position = mul( viewPos, g_mProj);
    Output.posView = viewPos;
    Output.diffAmbColor = In.diffAmbColor;
    #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0.xyy;
	#else
		Output.texCoord0 = 0;
	#endif
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif
				
	Output.specCol = float4(0,0,0,0);
    return Output;
}

// No lighting, just transform and pass on texture (and vertex color as diffuse)
// but with transformed view pos wrt shadow map too (for prelit shadow recvs in the scene)
vertexOutputShadowT2 VertNoLightVCShadowProj(vertexInputNLT2 In) 
{
    vertexOutputShadowT2 Output;
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    
    Output.position = mul( viewPos, g_mProj);
    #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0.xyy;
	#else
		Output.texCoord0 = 0;
	#endif
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif
	
	// project pos into light space
    Output.posLight = mul( viewPos, g_mViewToLightProj );
    
    #ifdef HKG_SHADOWS_VSM
        Output.posLight1 = mul( viewPos, g_mViewToLightProj1 );
		Output.posLight2 = mul( viewPos, g_mViewToLightProj2 );
		Output.posLight3 = mul( viewPos, g_mViewToLightProj3 );
	#endif
    Output.posView = viewPos;
    Output.diffAmbColor = In.diffAmbColor;
    Output.specCol = float4(0,0,0,0);
    return Output;
}

float4 LightVertex( float3 vert, float3 objectNormal, float3 eye )
{
	float3 N = normalize(mul( objectNormal, (float3x3)g_mWorldInvTranspose )); //normal vector
    float3 E = normalize( eye - vert ); //eye vector
    float3 L = g_vLightDir.xyz; //light vector
    float3 H = normalize( E + L ); //half angle vector

	//calculate the diffuse and specular contributions
    float  diff = max(0, dot(N,L));
    float  spec = pow( max(0 , dot(N,H) ), g_cSpecularPower);
    if (diff <= 0)
    	spec = 0;
    
	return float4( diff, diff, diff, spec );
}

void MaterialContribution( float4 vertexColor, float4 lightDiffuseAndSpec, out float4 diff, out float4 spec )
{
	float3 ambColor = vertexColor.rgb * g_cDiffuseColor.rgb * g_cAmbientColor.rgb;
    float3 diffColor = vertexColor.rgb * g_cDiffuseColor.rgb * lightDiffuseAndSpec.rgb * g_cLightColor.rgb;
    
    diff.rgb = diffColor.rgb + ambColor.rgb;
    diff.a = vertexColor.a * g_cDiffuseColor.a; 
    
	spec.rgb = g_cSpecularColor.rgb * g_cLightColor.rgb * lightDiffuseAndSpec.w;
    spec.a = 0;
}

// Vertex based lighting (with specular)
vertexOutputT2 VertOneLight(vertexInputNoColorT2 In) 
{
    vertexOutputT2 Output;
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Output.position = mul( viewPos, g_mProj);
    Output.posView = viewPos;
    #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0.xyy;
	#else
		Output.texCoord0 = 0;
	#endif
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif
	
	//calculate our vectors N, E, L, and H
	float3 worldEyePos = g_mViewInv[3].xyz;
    float3 worldVertPos = mul( In.position, g_mWorld ).xyz;
	float4 light = LightVertex(worldVertPos, In.normal, worldEyePos);
	MaterialContribution(float4(1,1,1,1), light, Output.diffAmbColor, Output.specCol);
	
    return Output;
}

// Vertex based lighting (with specular), also projects the position into the shadow map space
vertexOutputShadowT2 VertOneLightShadowProj(vertexInputNoColorT2 In) 
{
    vertexOutputShadowT2 Output;
    
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    
    Output.position = mul( viewPos, g_mProj);
   #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0;
	#else
		Output.texCoord0 = 0;
	#endif
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif
	
	// project pos into light space
    Output.posLight = mul( viewPos, g_mViewToLightProj );
    
     #ifdef HKG_SHADOWS_VSM
        Output.posLight1 = mul( viewPos, g_mViewToLightProj1 );
		Output.posLight2 = mul( viewPos, g_mViewToLightProj2 );
		Output.posLight3 = mul( viewPos, g_mViewToLightProj3 );
	#endif
  
  	Output.posView =  viewPos;
	
    // Lighting:
    
	//calculate our vectors N, E, L, and H
	float3 worldEyePos = g_mViewInv[3].xyz;
    float3 worldVertPos = mul(In.position, g_mWorld).xyz;
	float4 light = LightVertex(worldVertPos, In.normal, worldEyePos);
	MaterialContribution(float4(1,1,1,1), light, Output.diffAmbColor, Output.specCol);
	
	return Output;
}


// Vertex based lighting (with specular)
vertexOutputT2 VertOneLightVC(vertexInputT2 In) 
{
    vertexOutputT2 Output;
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Output.position = mul( viewPos, g_mProj);
    Output.posView = viewPos;
   #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0.xyy;
	#else
		Output.texCoord0 = 0;
	#endif
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif
	
	//calculate our vectors N, E, L, and H
	float3 worldEyePos = g_mViewInv[3].xyz;
    float3 worldVertPos = mul( In.position, g_mWorld ).xyz;
	float4 light = LightVertex(worldVertPos, In.normal, worldEyePos);
	MaterialContribution(In.diffAmbColor, light, Output.diffAmbColor, Output.specCol);
	
    return Output;
}

// Vertex based lighting (with specular), also projects the position into the shadow map space
vertexOutputShadowT2 VertOneLightVCShadowProj(vertexInputT2 In) 
{
    vertexOutputShadowT2 Output;
    
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    
    Output.position = mul( viewPos, g_mProj);
  #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0;
	#else
		Output.texCoord0 = 0;
	#endif
	#ifdef TEXTURE_TWO
		Output.texCoord1 = In.texCoord1;
	#else
		Output.texCoord1 = 0;
	#endif
	  
	// project pos into light space
    Output.posLight = mul( viewPos, g_mViewToLightProj );
    
    #ifdef HKG_SHADOWS_VSM
        Output.posLight1 = mul( viewPos, g_mViewToLightProj1 );
		Output.posLight2 = mul( viewPos, g_mViewToLightProj2 );
		Output.posLight3 = mul( viewPos, g_mViewToLightProj3 );
	#endif
  	Output.posView = viewPos;
	
    // Lighting:
    
	//calculate our vectors N, E, L, and H
	float3 worldEyePos = g_mViewInv[3].xyz;
    float3 worldVertPos = mul(In.position, g_mWorld).xyz;
	float4 light = LightVertex(worldVertPos, In.normal, worldEyePos);
	MaterialContribution(In.diffAmbColor, light, Output.diffAmbColor, Output.specCol);
	
	return Output;
}


vertexOutputT1B VertOneLightBump( vertexInputT1B In )
{
	vertexOutputT1B Out;

	// copy texture coordinates
	#ifdef TEXTURE_ONE
		Out.texCoord0 = In.texCoord0;
	#else
		Out.texCoord0 = float2(0,0);		
	#endif
	
	// transform position to clip space
	float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Out.position = mul( viewPos, g_mProj);
    Out.posView = viewPos;
   

	// compute the 3x3 tranform from object space to tangent space
	float3x3 objToTangentSpace;
	
	float bumpHeight = 1.0f;
		
	objToTangentSpace[0] = In.tangent * bumpHeight;
	objToTangentSpace[1] = In.binormal * bumpHeight;
	objToTangentSpace[2] = In.normal;
	
    float4 vertexPos = mul(In.position, g_mWorld); // world space position

	// light vector
	float3 lightVec = mul( g_vLightDir, (float3x3)g_mWorldInv );  // transform back to object space
	Out.L = mul( objToTangentSpace, lightVec ); // transform from object to tangent space
	
	// eye vector
	float3 eyeVec = g_mViewInv[3].xyz - vertexPos.xyz; // world space eye vector
	eyeVec = mul(eyeVec, (float3x3) g_mWorldInv );  // transform back to object space
	eyeVec = normalize(eyeVec);
	
	// half-angle vector
	float3 H = normalize(lightVec + eyeVec);
	Out.H = mul(objToTangentSpace, H);	// transform to tangent space

	return Out;
}


// Vertex based lighting (with specular), also projects the position into the shadow map space
vertexOutputShadowT1B VertOneLightBumpShadowProj( vertexInputT1B In) 
{
    vertexOutputShadowT1B Output;
    
    float4 viewPos =  mul( float4(In.position.xyz, 1.0), g_mWorldView);
    Output.position = mul( viewPos, g_mProj);
   
    #ifdef TEXTURE_ONE
		Output.texCoord0 = In.texCoord0;
	#else
		Output.texCoord0 = 0;		
	#endif
	 
		// project pos into light space
    Output.posLight = mul( viewPos, g_mViewToLightProj );
    
    #ifdef HKG_SHADOWS_VSM
        Output.posLight1 = mul( viewPos, g_mViewToLightProj1 );
		Output.posLight2 = mul( viewPos, g_mViewToLightProj2 );
		Output.posLight3 = mul( viewPos, g_mViewToLightProj3 );
    #endif

    Output.posView = viewPos;
  
	// compute the 3x3 tranform from object space to tangent space
	float3x3 objToTangentSpace;
	
	float bumpHeight = 1.0f;
		
	objToTangentSpace[0] = In.tangent * bumpHeight;
	objToTangentSpace[1] = In.binormal * bumpHeight;
	objToTangentSpace[2] = In.normal;
	
    float4 vertexPos = mul(In.position, g_mWorld); // world space position

	// light vector
	float3 lightVec = mul( g_vLightDir, (float3x3)g_mWorldInv );  // transform back to object space
	Output.L = mul( objToTangentSpace, lightVec ); // transform from object to tangent space
	
	// eye vector
	float3 eyeVec = g_mViewInv[3].xyz - vertexPos.xyz; // world space eye vector
	eyeVec = mul(eyeVec, (float3x3) g_mWorldInv );  // transform back to object space
	eyeVec = normalize(eyeVec);
	
	// half-angle vector
	float3 H = normalize(lightVec + eyeVec);
	Output.H = mul(objToTangentSpace, H);	// transform to tangent space

	
	return Output;
}

#ifdef HKG_SHADOWS_VSM

	vertexOutputShadowDepthVSM VertShadowDepth(vertexInputPosOnly In)
	{
		vertexOutputShadowDepthVSM Output;
		
		float3 worldVertPos = mul( float4(In.position,1), g_mWorld);
		Output.position = mul(float4(worldVertPos,1), g_mViewProj); // Light VP in this case
		Output.posWorld = worldVertPos;
		return Output;
	}

#else

	// Vertex transform only, with tcoord0 the enough to work out the transformed depth (no bias)
	vertexOutputShadowDepth VertShadowDepth(vertexOutputShadowDepth In)
	{
		vertexOutputShadowDepth Output;
		Output.position = mul(In.position, g_mWorldViewProj); // Light WVP in this case
		Output.depthZZZW = float4(Output.position.zzz, Output.position.w);
		return Output;
	}

#endif

	
//////////////////////////////////////////////////////////////////
//
// Pixel Shaders
//

#ifdef HKG_DEPTH_PEEL
	#define HKG_TEST_DEPTH_PEEL \
		{ \
			float secondDepth = DepthPeelSecondDepth.Load( int3(In.position.xy, 0) ); \
			if ( secondDepth <= In.position.z ) \
			{ \
				discard;\
			} \
		}
#else
	#define HKG_TEST_DEPTH_PEEL  { }
#endif
	
// No per pixel lighting, just add the input color with the specular, no texture loookup
pixelOutput PixT0( vertexOutputT2 In ) 
{ 
	HKG_TEST_DEPTH_PEEL

	pixelOutput Output;
    Output.color = In.diffAmbColor + float4(In.specCol.rgb,0);
   
	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( Output.color.a - 0.005 );
	
	Output.color = colorRangeCorrect( Output.color );
	
#ifdef HKG_ENABLE_VIEWDEPTH
	if ( g_iFogParams.x > 0)
	{
		Output.color = computeFog( In.posView.z, Output.color );
	}
   	Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
#else
	Output.pzDepth.rgba = 0;
#endif

	return Output;
}

// No per pixel lighting, just modulate the input color with the texture and add specular (afterwards)
pixelOutput PixT1( vertexOutputT2 In ) 
{ 
	HKG_TEST_DEPTH_PEEL
	
	pixelOutput Output;
	
#ifdef HKG_SKYMAP
    float3 sky = TexCube0.Sample( g_sSamplerClamped, In.texCoord0).rgb;
	Output.color = float4(sky,1) * In.diffAmbColor; 	
#else
    Output.color = Tex2D0.Sample(g_sSamplerShared, In.texCoord0.xy) * In.diffAmbColor + float4(In.specCol.rgb,0);
#endif

	//texkill completely transparent regions so that ssao not affected by invisible stuff
	//clip( Output.color.a - 0.005 );
	
	Output.color = colorRangeCorrect( Output.color );
	
#if defined(HKG_ENABLE_VIEWDEPTH) && !defined(HKG_SKYMAP)
	if ( g_iFogParams.x > 0)
    {
		Output.color = computeFog( In.posView.z, Output.color );
    }
    
    Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
#else
	#if defined(HKG_SKYMAP)
		Output.pzDepth.rgba = 1.#INF;
	#else
		Output.pzDepth.rgba = 0;
	#endif
#endif

    return Output;
}

pixelOutput PixT2( vertexOutputT2 In ) 
{ 
    HKG_TEST_DEPTH_PEEL
    
    pixelOutput Output;
    Output.color = Tex2D0.Sample(g_sSamplerShared, In.texCoord0.xy) * Tex2D1.Sample(g_sSamplerShared, In.texCoord1) * In.diffAmbColor + float4(In.specCol.rgb,0);

	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( Output.color.a - 0.005 );

	Output.color = colorRangeCorrect( Output.color );
	
#ifdef HKG_ENABLE_VIEWDEPTH
	if ( g_iFogParams.x > 0)
    {
		Output.color = computeFog( In.posView.z, Output.color );
    }
    
    Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
#else
	Output.pzDepth.rgba = 0;
#endif

    return Output;
}

#ifdef HKG_SHADOWS_VSM

	float4 PixShadowMap( vertexOutputShadowDepthVSM In ): SV_Target0
	{
		float4 depth;
		float3 lightDir = g_vLightShadowStartPosWorld - In.posWorld;
		depth.x = dot(lightDir,g_vLightDir) / g_fShadowMapDistance; 
		depth.y = depth.x * depth.x;
		
		#ifdef HKG_SHADOWS_DISTRIBUTE_16F_OUTPUTS 
			depth.zw = frac(depth.xy * SHADOW_16F_DISTRIBUTE_FACTOR);
			depth.xy = depth.xy - (depth.zw / SHADOW_16F_DISTRIBUTE_FACTOR);
		#else
			depth.zw = depth.xy;
		#endif
		
		return depth;
	}

#else // standard shadow maps
   
	float4 PixShadowMap( vertexOutputShadowDepth In ) : SV_Target0
	{
		float depth = (In.depthZZZW.b / In.depthZZZW.a);
		return float4( depth.xxx, 1.0f );
	}

#endif


// Blinn/Phong lighting model
float4 Phong(float2 dots) 
{
	float NdotL = dots.x;
	float NdotH = dots.y;
	float diffuse = max(NdotL, 0.0);
	float specular = pow(NdotH, g_cSpecularPower);
	if (NdotL <= 0) specular = 0;
	return float4(diffuse, diffuse, diffuse, specular);
}

// Pixel shaders
pixelOutput PixT1Bump( vertexOutputT1B In ) : SV_Target0
{
	HKG_TEST_DEPTH_PEEL
	
	pixelOutput Output;

	float4 colorMap = Tex2D0.Sample(g_sSamplerShared, In.texCoord0);
	float3 N = Tex2D1.Sample(g_sSamplerShared, In.texCoord0)*2.0 - 1.0;

	N = normalize(N);

	float NdotL = dot(N, In.L);
	float NdotH = dot(N, In.H);
	float4 light = Phong(float2(NdotL, NdotH));
	
	float4 c = float4(g_cAmbientColor.rgb*colorMap.rgb +light.rgb*g_cDiffuseColor.rgb*colorMap.rgb + light.w*g_cSpecularColor.rgb, g_cDiffuseColor.a);
	
	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( c.a - 0.005 );

	Output.color = colorRangeCorrect( Output.color );
	
	if ( g_iFogParams.x > 0)
    {
		c = computeFog( In.posView.z, c );
    }
    
    Output.color = c;
    Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
    
	return Output;
}

 
float getLightAmountVSM( float4 posLight, float3 posView, int tindex )
{
	float2 shadowTexCoord = posLight.xy / posLight.w;
	
	// Lookup assumes Bil. filtering supported on the float textures 
	#ifdef HKG_NO_FLOATBLENDING 
	
		float2 texelpos = g_iShadowMapSize * shadowTexCoord;
		float2 lerps = frac( texelpos );
		float2 sourcevals[4];
		float oneTexel = 1.0/g_iShadowMapSize;
	    
		if ( tindex == 0)
		{
			sourcevals[0] = Tex2D0.Sample( g_sSamplerShadow, shadowTexCoord ).rg;  
			sourcevals[1] = Tex2D0.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, 0) ).rg;  
			sourcevals[2] = Tex2D0.Sample( g_sSamplerShadow, shadowTexCoord + float2(0, oneTexel) ).rg;  
			sourcevals[3] = Tex2D0.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, oneTexel) ).rg;  
	    }
		else if ( tindex == 1)
		{
			sourcevals[0] = Tex2D1.Sample( g_sSamplerShadow, shadowTexCoord ).rg;  
			sourcevals[1] = Tex2D1.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, 0) ).rg;  
			sourcevals[2] = Tex2D1.Sample( g_sSamplerShadow, shadowTexCoord + float2(0, oneTexel) ).rg;  
			sourcevals[3] = Tex2D1.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, oneTexel) ).rg;  
	    }
		else if ( tindex == 2)
		{
			sourcevals[0] = Tex2D2.Sample( g_sSamplerShadow, shadowTexCoord ).rg;  
			sourcevals[1] = Tex2D2.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, 0) ).rg;  
			sourcevals[2] = Tex2D2.Sample( g_sSamplerShadow, shadowTexCoord + float2(0, oneTexel) ).rg;  
			sourcevals[3] = Tex2D2.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, oneTexel) ).rg;  
	    }
		else
		{
			sourcevals[0] = Tex2D3.Sample( g_sSamplerShadow, shadowTexCoord ).rg;  
			sourcevals[1] = Tex2D3.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, 0) ).rg;  
			sourcevals[2] = Tex2D3.Sample( g_sSamplerShadow, shadowTexCoord + float2(0, oneTexel) ).rg;  
			sourcevals[3] = Tex2D3.Sample( g_sSamplerShadow, shadowTexCoord + float2(oneTexel, oneTexel) ).rg;  
	    }
			
		// lerp between the shadow values to calculate our light amount
		float2 moments = lerp( lerp( sourcevals[0], sourcevals[1], lerps.x ),
									lerp( sourcevals[2], sourcevals[3], lerps.x ),
									lerps.y );

	#else
	
		float2 moments;
		if ( tindex == 0)
			moments = Tex2D0.Sample( g_sSamplerShadow, shadowTexCoord).rg;
		else if ( tindex == 1)
			moments = Tex2D1.Sample( g_sSamplerShadow, shadowTexCoord).rg;
		else if ( tindex == 2)
			moments = Tex2D2.Sample( g_sSamplerShadow, shadowTexCoord).rg;
		else
			moments = Tex2D3.Sample( g_sSamplerShadow, shadowTexCoord).rg;
	
	#endif
	
	// Rescale light distance and check if we're in shadow
	float3 lightDir = g_vLightShadowStartPosView - posView ;
	float distFromLight = dot(lightDir,g_vLightDirView); 
	float rescaledDistToLight = distFromLight / g_fShadowMapDistance; 
	rescaledDistToLight -= g_fShadowVsmBias; 
	rescaledDistToLight = min( rescaledDistToLight, 0.999f);
	float litFactor = (moments.x > 0.99999f) || (moments.x >= rescaledDistToLight); // || (moments.x > 0.95); // if moment.x (depth) > rescaled, then not in shadow. May add check here for detecting 'off the edge of map'(always lit in our demos)
	
	// Variance shadow mapping
	float E_x2 = moments.y;
	float Ex_2 = moments.x * moments.x;
	float variance = min(max(E_x2 - Ex_2, 0.0) + g_fShadowVsmEpsilon, 1.0);
	float m_d = (moments.x - rescaledDistToLight);
	float p = variance / (variance + (m_d * m_d));

	return max(litFactor, p);
}

float getLightAmount(vertexOutputShadowT2 In)
{
	int t;
	// Refactored for AMD drivers, so that less code on different texture samplers etc in each conditional block..
	// Caused the single pixel horizontal corruption with > 1 shadow map
	float4 posLight;
	if ( In.posView.z > m_fShadowSplitDistances.z)
	{
		posLight = In.posLight3;
		t = 3;
	}
	else if ( In.posView.z > m_fShadowSplitDistances.y)
	{
		posLight = In.posLight2;
		t = 2;
	}
	else if ( In.posView.z > m_fShadowSplitDistances.x)
	{
		posLight = In.posLight1;
		t= 1;
	}
	else
	{
		posLight = In.posLight;
		t = 0;
	}
	
	return getLightAmountVSM( posLight, In.posView, t );
}

float getLightAmount(vertexOutputShadowT1B In)
{
	int t;
	// Refactored for AMD drivers, so that less code on different texture samplers etc in each conditional block..
	// Caused the single pixel horizontal corruption with > 1 shadow map
	float4 posLight;
	float4 posView;
	if ( In.posView.z > m_fShadowSplitDistances.z)
	{
		posLight = In.posLight3;
		t = 3;
	}
	else if ( In.posView.z > m_fShadowSplitDistances.y)
	{
		posLight = In.posLight2;
		t = 2;
	}
	else if ( In.posView.z > m_fShadowSplitDistances.x)
	{
		posLight = In.posLight1;
		t= 1;
	}
	else
	{
		posLight = In.posLight;
		t = 0;
	}
	
	return getLightAmountVSM( posLight, In.posView, t );
}

		
// Pixel shader to use a float based shadow map, will modulate with vertex lit color 
// (does not do per pixel lighting, uses vertex color)	
pixelOutput PixShadowSceneT0( vertexOutputShadowT2 In )
{
	HKG_TEST_DEPTH_PEEL
    
    pixelOutput Output;

	float lightAmount = getLightAmount( In );
	
	float3 materialAmbientOnly = g_cDiffuseColor.rgb * g_cAmbientColor.rgb;
	
	Output.color.rgb = (1-lightAmount)*materialAmbientOnly.rgb + lightAmount*( In.diffAmbColor.rgb + In.specCol.rgb ); // modulate rgb wrt shadow.
    Output.color.a = In.diffAmbColor.a; // modulate alpha as is, shadow doesn't affect it.
	
	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( Output.color.a - 0.005 );

	Output.color = colorRangeCorrect( Output.color );
	
	// Fix for some demos just fir now: no fog or ssao on pre lit objects
	if ( g_iFogParams.x > 0)
	{
		Output.color = computeFog( In.posView.z, Output.color );
	}
	Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
    return Output;
}

// Pixel shader to use a float based shadow map, will modulate with vertex lit color 
// (does not do per pixel lighting, uses vertex color)
pixelOutput PixShadowSceneT1( vertexOutputShadowT2 In )
{
    HKG_TEST_DEPTH_PEEL
    
    pixelOutput Output;

	float lightAmount = getLightAmount( In );
	
	float4 texel = Tex2D4.Sample( g_sSamplerShared, In.texCoord0 );
	float3 materialAmbientOnly = g_cDiffuseColor.rgb * g_cAmbientColor.rgb;
    Output.color.rgb = (1-lightAmount)*materialAmbientOnly.rgb*texel.rgb + lightAmount*(In.diffAmbColor.rgb*texel.rgb + In.specCol.rgb); // modulate rgb wrt shadow.
    Output.color.a = texel.a * In.diffAmbColor.a; // modulate alpha as is, shadow doesn't affect it.

	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( Output.color.a - 0.005 );

	Output.color = colorRangeCorrect( Output.color );
		
	if ( g_iFogParams.x > 0)
    {
		Output.color = computeFog( In.posView.z, Output.color );
    }
    Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
    return Output;
}

// Pixel shader to use a float based shadow map, will modulate with vertex lit color 
// (does not do per pixel lighting, uses vertex color)
pixelOutput PixShadowSceneT2( vertexOutputShadowT2 In )
{
    HKG_TEST_DEPTH_PEEL
    
    pixelOutput Output;

	float lightAmount = getLightAmount( In );
	
	float4 diffuse = float4(((1 - lightAmount)*g_cAmbientColor.rgb),1) + (lightAmount * In.diffAmbColor);
	float4 texel0 = Tex2D4.Sample( g_sSamplerShared, In.texCoord0 );
    float4 texel1 = Tex2D5.Sample( g_sSamplerShared, In.texCoord1 );
    float3 texelTotal = texel0.rgb * texel1.rgb;    
	float3 materialAmbientOnly = g_cDiffuseColor.rgb * g_cAmbientColor.rgb;
    
    Output.color.rgb = (1-lightAmount)*texelTotal*materialAmbientOnly.rgb + lightAmount*(In.diffAmbColor.rgb*texelTotal.rgb + In.specCol.rgb); // modulate rgb wrt shadow.
    Output.color.a = texel0.a * In.diffAmbColor.a; // modulate alpha as is, shadow doesn't affect it.

	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( Output.color.a - 0.005 );

	Output.color = colorRangeCorrect( Output.color );
	
	if ( g_iFogParams.x > 0)
    {
		Output.color = computeFog( In.posView.z, Output.color );
    }
    Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
    return Output;
}

pixelOutput PixShadowSceneT1Bump( vertexOutputShadowT1B In )
{
    HKG_TEST_DEPTH_PEEL
    
    pixelOutput Output;
    
	float lightAmount = getLightAmount( In );
	
    float4 colorMap = Tex2D4.Sample(g_sSamplerShared, In.texCoord0);
	float3 N = Tex2D5.Sample(g_sSamplerShared, In.texCoord0)*2.0 - 1.0;

	N = normalize(N);

	In.L = normalize(In.L);
	In.H = normalize(In.H);

	float NdotL = dot(N, In.L);
	float NdotH = dot(N, In.H);
	float4 light = lightAmount * Phong(float2(NdotL, NdotH));
	
	Output.color.rgb = g_cAmbientColor*colorMap*g_cDiffuseColor + light*g_cDiffuseColor*colorMap + light.w*g_cSpecularColor ;
	Output.color.a = colorMap.a * g_cDiffuseColor.a; // modulate alpha as is, shadow doesn't affect it.

	//texkill completely transparent regions so that ssao not affected by invisible stuff
	clip( Output.color.a - 0.005 );

	Output.color = colorRangeCorrect( Output.color );
	
	if ( g_iFogParams.x > 0)
    {
		Output.color = computeFog( In.posView.z, Output.color );
    }
    Output.pzDepth.rgba = In.posView.z * g_iDepthParams.x;
    return Output;
}


// Velocity map support for default shaders
struct vertexInputVel {
	float3 position				: POSITION;
};

struct vertexOutputVel {
	float4 position         : SV_Position;
	float4 wpCur			: TEXCOORD0;
	float4 wpPrev			: TEXCOORD1;
};
	
vertexOutputVel VertVel( vertexInputVel In)
{
	vertexOutputVel Output;
	
	float3 curWorldVertPos = mul(float4(In.position,1), g_mWorld).xyz;
	float3 prevWorldVertPos = mul(float4(In.position,1), g_mPrevWorld).xyz;
	
	Output.position = mul(float4(curWorldVertPos,1), g_mViewProj); 
	Output.wpCur = Output.position; //SM3.0..
	Output.wpPrev = mul(float4(prevWorldVertPos,1), g_mPrevViewProj);

	return Output;
}

float4 PixVel( vertexOutputVel In ) : SV_Target0
{
	float2 curP  = In.wpCur.xy / In.wpCur.w; 
	float2 prevP = In.wpPrev.xy / In.wpPrev.w; 
	float2 velocityXY;
	velocityXY = curP - prevP;
	velocityXY /= 2;
	
	return float4(velocityXY,0,1);
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
