/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/ObjectComparison/ComparisonUtil.h>

#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/Reflection/Util/hkVariantDataUtil.h>
#include <Common/Base/Reflection/Attributes/hkAttributes.h>
#include <Common/Base/Reflection/Registry/hkVtableClassRegistry.h>

namespace compareHelper //TODO tidy up string creation
{
	hkStringBuf toString(hkReal realIn)
	{
		if (realIn == HK_REAL_MAX)
		{
			return hkStringBuf("Max Real");
		}
		else if (realIn == HK_REAL_MAX)
		{
			return hkStringBuf("-Max Real");
		}
		else if (0 < realIn && realIn <= HK_REAL_EPSILON)
		{
			return hkStringBuf("0.EP");
		}
		else if (HK_REAL_EPSILON <= realIn && realIn < 0)
		{
			return hkStringBuf("-0.EP");
		}

		hkStringBuf text;
		if (floor(realIn) == realIn)
		{
			text.printf("%d",(int)realIn);
		}
		else
		{
			text.printf("%g",realIn);
		}
		return text;
	}

	hkStringBuf toString(const hkVector4& vecIn)
	{
		hkStringBuf text;
		text.printf("%s, %s, %s", compareHelper::toString(vecIn(0)).cString(), compareHelper::toString(vecIn(1)).cString(), compareHelper::toString(vecIn(2)).cString());
		if (vecIn(3) != 0)
		{
			text.appendPrintf(", %s", compareHelper::toString(vecIn(3)).cString());
		}
		return text;
	}

	hkStringBuf toString(const hkQuaternion& quatIn)
	{
		return compareHelper::toString(quatIn.m_vec);
	}
	
	hkStringBuf toString(const hkMatrix3& matIn)
	{
		hkStringBuf text;
		text.printf("\n%s\n%s\n%s",
					compareHelper::toString(matIn.getColumn(0)).cString(),
					compareHelper::toString(matIn.getColumn(1)).cString(),
					compareHelper::toString(matIn.getColumn(2)).cString());
		return text;
	}

	hkStringBuf toString(const hkMatrix4& matIn)
	{
		hkStringBuf text;
		text.printf("\n%s\n%s\n%s\n%s",
			compareHelper::toString(matIn.getColumn(0)).cString(),
			compareHelper::toString(matIn.getColumn(1)).cString(),
			compareHelper::toString(matIn.getColumn(2)).cString(),
			compareHelper::toString(matIn.getColumn(3)).cString());
		return text;
	}	

	hkStringBuf toString(const hkTransform& transIn)
	{
		hkStringBuf text;
		text.appendPrintf("T:%s\n", compareHelper::toString(transIn.getTranslation()).cString());
		text.appendPrintf("R:%s", compareHelper::toString(transIn.getRotation()).cString());
		return text;
	}

	hkStringBuf toString(const hkQsTransform& qsTransIn)
	{
		hkStringBuf text;
		text.appendPrintf("T:%s\n", compareHelper::toString(qsTransIn.m_translation).cString());
		text.appendPrintf("S:%s\n", compareHelper::toString(qsTransIn.m_scale).cString());
		text.appendPrintf("R:%s", compareHelper::toString(qsTransIn.m_rotation.m_vec).cString());
		return text;
	}

	hkBool equals(const hkMatrix3& matA, const hkMatrix3& matB)
	{
		return matA.getColumn(0).equals4(matB.getColumn(0))
			&& matA.getColumn(1).equals4(matB.getColumn(1))
			&& matA.getColumn(2).equals4(matB.getColumn(2));
	}
	hkBool equals(const hkMatrix4& matA, const hkMatrix4& matB)
	{
		return matA.getColumn(0).equals4(matB.getColumn(0))
			&& matA.getColumn(1).equals4(matB.getColumn(1))
			&& matA.getColumn(2).equals4(matB.getColumn(2))
			&& matA.getColumn(3).equals4(matB.getColumn(3));
	}
}

void ComparisonUtil::compareObjects(void* objAStart, void* objBStart)
{
	const hkClass* klassA = hkVtableClassRegistry::getInstance().getClassFromVirtualInstance(objAStart);
	const hkClass* klassB = hkVtableClassRegistry::getInstance().getClassFromVirtualInstance(objBStart);
	HK_ASSERT2(0x285693bb, klassA && klassB, "Object does not have Vtable. Use compareObjects(void*, void*, const hkClass&) instead.");

	hkStringBuf text;	text.printf("Comparing ObjectA Type: %s and ObjectB Type; %s", klassA->getName(), klassB->getName());
	hkcout << text;
	HK_ASSERT2(0x285693bb, klassA->equals(klassB), "Trying to compare objects with different hkClasses");

	compareObjects(objAStart, objBStart, *klassA);
}

void ComparisonUtil::compareObjects(void* objAStart, void* objBStart, const hkClass& objClass)
{
	compareAllMembers(objAStart, objBStart, objClass, hkStringBuf(""), m_diffTree.iterGetFirst());
}

void ComparisonUtil::compareAllMembers(void* objAStart, void* objBStart, const hkClass& objClass, hkStringBuf parentStr, TreeIter parentItr)
{
	int numOfMembers = objClass.getNumMembers();
	for (int mem = 0; mem < numOfMembers; mem++)
	{
		hkClassMember member = objClass.getMember(mem);
		compareMember(objAStart, objBStart, member, parentStr, parentItr);
	}
}

void ComparisonUtil::compareMember(void* objAStart, void* objBStart, const hkClassMember& member, hkStringBuf parentStr, TreeIter parentItr)
{
	if (loopInHierachy(parentItr, member.getName()))
	{
		return;
	}

	int offset = member.getOffset();
	void* objAMember = static_cast<void*>(static_cast<char*>(objAStart) + offset);
	void* objBMember = static_cast<void*>(static_cast<char*>(objBStart) + offset);

	hkClassMember::Type type = member.getType();

	hkStringBuf valueAstr;
	hkStringBuf valueBstr;
	hkBool membersEqual = true; 

	switch(type)
	{
	case hkClassMember::TYPE_POINTER:
		{
			membersEqual = comparePointerMember(objAMember, objBMember, member, valueAstr, valueBstr, parentStr, parentItr);
			break;
		}
	case hkClassMember::TYPE_ARRAY:
		{
			StringValues branch; branch.m_memPath = member.getName();
			TreeIter newBranch = m_diffTree.appendChild(parentItr, branch);

			compareArrayMember(objAMember, objBMember, member, parentStr, newBranch);
			break;
		}
	case hkClassMember::TYPE_STRUCT: //recurse
		{
			StringValues branch; branch.m_memPath = member.getName();
			TreeIter newBranch = m_diffTree.appendChild(parentItr, branch);
			parentStr.appendPrintf("%s/", member.getName());

			compareAllMembers(objAMember, objBMember, member.getStructClass(), parentStr, newBranch);
			break;
		}
	case hkClassMember::TYPE_ENUM:
	case hkClassMember::TYPE_FLAGS:
		{
			membersEqual = compareSimpleType(objAMember, objBMember, member.getSubType(), valueAstr, valueBstr);
			break;
		}
	default:
		{
			membersEqual = compareSimpleType(objAMember, objBMember, type, valueAstr, valueBstr);
			break;
		}
	}

	if (!membersEqual)
	{	
		storeDifference(member, valueAstr, valueBstr, parentStr, parentItr);
	}	
}

void ComparisonUtil::compareArrayMember(void* objAMember, void* objBMember, const hkClassMember& member, hkStringBuf parentStr, TreeIter parentItr)
{
	const hkArray<void*>* memAVal = static_cast<const hkArray<void*>*>( objAMember );
	const hkArray<void*>* memBVal = static_cast<const hkArray<void*>*>( objBMember );

	int arrayASize = memAVal->getSize();
	int arrayBSize = memBVal->getSize();
	
	if (arrayASize==0 && arrayBSize==0) // are they both empty?
	{
		return;
	}

	if (arrayASize != arrayBSize)	//are they even the same size?
	{
		StringValues values;
		values.m_memPath.printf("%s%s", parentStr.cString(), member.getName());
		values.m_valueA.printf("Array Size %d", arrayASize);
		values.m_valueB.printf("Array Size %d", arrayBSize);
		m_differences.pushBack(values);
		return;
	}

	//ok now compare each element
	int arrayAElemSize = member.getArrayMemberSize();
	int arrayBElemSize = member.getArrayMemberSize();
	const void* startA = memAVal->begin();
	const void* startB = memBVal->begin();
	hkClassMember::Type arrayType = member.getArrayType();

	for(int elem = 0; elem < arrayASize; elem++)
	{		
		void* elemA = static_cast<void*>(static_cast<char*>(const_cast<void*>(startA)) + (elem*arrayAElemSize));
		void* elemB = static_cast<void*>(static_cast<char*>(const_cast<void*>(startB)) + (elem*arrayBElemSize));

		hkStringBuf elemAStr;
		hkStringBuf elemBStr;
		hkBool elementsEqual = true;

		switch(arrayType)
		{
		case hkClassMember::TYPE_STRUCT:
			{
				StringValues branch; branch.m_memPath.printf("[%d]", elem);
				TreeIter newBranch = m_diffTree.appendChild(parentItr, branch);
				parentStr.appendPrintf("%s[%d]/", member.getName(), elem);	//TODO: not sure about this string

				compareAllMembers(elemA, elemB, member.getStructClass(), parentStr, newBranch);
				break;
			}
		case hkClassMember::TYPE_POINTER:	
			{
				elementsEqual = comparePointerMember(elemA, elemB, member, elemAStr, elemBStr, parentStr, parentItr, elem);
				break;
			}
		default:
			{
				elementsEqual = compareSimpleType(elemA, elemB, arrayType, elemAStr, elemBStr);
				break;
			}
		}

		if (!elementsEqual)
		{
			storeDifference(member, elemAStr, elemBStr, parentStr, parentItr, elem);
		}
	}
}

hkBool ComparisonUtil::comparePointerMember(void* objAMember, void* objBMember, const hkClassMember& member, hkStringBuf& valueAstr, hkStringBuf& valueBstr, hkStringBuf parentStr, TreeIter parentItr, int elem)
{
	void* derefA = reinterpret_cast<void*>(*reinterpret_cast<int*>( objAMember )); //NB portability issues
	void* derefB = reinterpret_cast<void*>(*reinterpret_cast<int*>( objBMember ));

	//early outs
	{
		if (derefA == derefB)
		{
			return true;
		}
		if (derefA == NULL)
		{
			valueAstr = "Ptr to NULL";
			valueBstr = "Ptr to Non-NULL";
			return false;
		}
		if (derefB == NULL)
		{
			valueAstr = "Ptr to Non-NULL";
			valueBstr = "Ptr to NULL";
			return false;
		}

	}

	if (!member.hasClass())
	{
		return compareSimpleType(derefA, derefB, member.getSubType(), valueAstr, valueBstr);
	}

	else
	{
		StringValues branch; 
		if (elem == -1)
		{
			branch.m_memPath = member.getName();
			parentStr.appendPrintf("%s/", member.getName());
		}
		else
		{
			branch.m_memPath.printf("[%d]", elem);
			parentStr.appendPrintf("%s[%d]/", member.getName(), elem);
		}
		TreeIter newBranch = m_diffTree.appendChild(parentItr, branch);

		//assumption: arrays of pointers will have a struct class and those without are SERIALIZE_IGNORED
		compareAllMembers(derefA, derefB, *(member.getClass()), parentStr, newBranch);
		return true;
	}
}

hkBool ComparisonUtil::compareSimpleType(void* objAMember, void* objBMember, const hkClassMember::Type type, hkStringBuf& valueAstr, hkStringBuf& valueBstr)
{
	hkBool membersEqual = true;
	switch (type)
	{
	case hkClassMember::TYPE_BOOL:
		{
			hkBool memAVal = *reinterpret_cast<hkBool*>( objAMember );
			hkBool memBVal = *reinterpret_cast<hkBool*>( objBMember );
			valueAstr.printf("%s", memAVal ? "True" : "False" );
			valueBstr.printf("%s", memBVal ? "True" : "False" );
			membersEqual = (memAVal == memBVal);
			break;
		}
	case hkClassMember::TYPE_CHAR:
	case hkClassMember::TYPE_INT8:
	case hkClassMember::TYPE_UINT8:
	case hkClassMember::TYPE_INT16:
	case hkClassMember::TYPE_UINT16:
	case hkClassMember::TYPE_INT32:	
	case hkClassMember::TYPE_UINT32:
	case hkClassMember::TYPE_INT64:
	case hkClassMember::TYPE_UINT64:
	case hkClassMember::TYPE_ULONG:
		{	
			hkInt64 memAVal = hkVariantDataUtil::getInt( type, objAMember );
			hkInt64 memBVal = hkVariantDataUtil::getInt( type, objBMember );
			valueAstr.printf("%d", memAVal);
			valueBstr.printf("%d", memBVal);
			membersEqual = (memAVal == memBVal);
			break;
		}	
	case hkClassMember::TYPE_REAL:
	case hkClassMember::TYPE_HALF:
		{
			hkReal memAVal = hkVariantDataUtil::getReal( type, objAMember );
			hkReal memBVal = hkVariantDataUtil::getReal( type, objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = hkMath::equal(memAVal, memBVal);
			break;
		}
	case hkClassMember::TYPE_CSTRING:
	case hkClassMember::TYPE_STRINGPTR:
		{

			hkStringBuf memAVal ( hkVariantDataUtil::getString( type, objAMember ) );
			hkStringBuf memBVal ( hkVariantDataUtil::getString( type, objBMember ) );
			valueAstr.printf("%s", memAVal.cString());
			valueBstr.printf("%s", memBVal.cString());
			membersEqual = (memAVal.compareToIgnoreCase(memBVal) == 0);
			break;
		}
	case hkClassMember::TYPE_VECTOR4:
		{
			hkVector4 memAVal = *reinterpret_cast<hkVector4*>( objAMember );
			hkVector4 memBVal = *reinterpret_cast<hkVector4*>( objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = memAVal.equals4(memBVal);
			break;
		}
	case hkClassMember::TYPE_QUATERNION:
		{
			hkQuaternion memAVal = *reinterpret_cast<hkQuaternion*>( objAMember );
			hkQuaternion memBVal = *reinterpret_cast<hkQuaternion*>( objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = memAVal.m_vec.equals4(memBVal.m_vec);
			break;
		}
	case hkClassMember::TYPE_MATRIX3:
	case hkClassMember::TYPE_ROTATION:
		{
			hkMatrix3 memAVal = *reinterpret_cast<hkMatrix3*>( objAMember );
			hkMatrix3 memBVal = *reinterpret_cast<hkMatrix3*>( objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = compareHelper::equals(memAVal, memBVal);
			break;
		}
	case hkClassMember::TYPE_MATRIX4:
		{
			hkMatrix4 memAVal = *reinterpret_cast<hkMatrix4*>( objAMember );
			hkMatrix4 memBVal = *reinterpret_cast<hkMatrix4*>( objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = compareHelper::equals(memAVal, memBVal);
			break;
		}
	case hkClassMember::TYPE_TRANSFORM:
		{
			hkTransform memAVal = *reinterpret_cast<hkTransform*>( objAMember );
			hkTransform memBVal = *reinterpret_cast<hkTransform*>( objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = memAVal.getTranslation().equals4(memBVal.getTranslation())
						&& compareHelper::equals(memAVal.getRotation(), memBVal.getRotation());
			break;
		}
	case hkClassMember::TYPE_QSTRANSFORM:
		{
			hkQsTransform memAVal = *reinterpret_cast<hkQsTransform*>( objAMember );
			hkQsTransform memBVal = *reinterpret_cast<hkQsTransform*>( objBMember );
			valueAstr = compareHelper::toString(memAVal);
			valueBstr = compareHelper::toString(memBVal);
			membersEqual = memAVal.m_translation.equals4(memBVal.m_translation)
						&& memAVal.m_rotation.m_vec.equals4(memBVal.m_rotation.m_vec)
						&& memAVal.m_scale.equals4(memBVal.m_scale);
			break;
		}
	case hkClassMember::TYPE_VOID:
	case hkClassMember::TYPE_ZERO:
	case hkClassMember::TYPE_VARIANT:
	case hkClassMember::TYPE_FUNCTIONPOINTER:
	case hkClassMember::TYPE_INPLACEARRAY:
	case hkClassMember::TYPE_SIMPLEARRAY:
	case hkClassMember::TYPE_HOMOGENEOUSARRAY:
	case hkClassMember::TYPE_RELARRAY:
	case hkClassMember::TYPE_MAX:
		{
			hkStringBuf str; str.printf("ToDo: write comparison for type %d\n", type);
			HK_WARN(0, str.cString());
			break;
		}
	default:
		{
			HK_ASSERT2(0, false, "Comparision for type hasn't been implemented");
			break;
		}
	}
	return membersEqual;
}

void ComparisonUtil::applyAttributes(const hkClassMember& member, hkStringBuf& valueAstr, hkStringBuf& valueBstr)
{
	const hkVariant* variant = member.getAttribute("hk.Semantics");
	if (variant)
	{
		hkSemanticsAttribute* semanticAtt = reinterpret_cast<hkSemanticsAttribute*>(variant->m_object);
		switch(semanticAtt->m_type)
		{
		case hkSemanticsAttribute::ANGLE:
			{
				valueAstr.append(" (Angle)");
				valueBstr.append(" (Angle)");
				break;
			}
		case hkSemanticsAttribute::COSINE_ANGLE:
			{
				valueAstr.append(" (Cosine Angle)");
				valueBstr.append(" (Cosine Angle)");
				break;
			}
		case hkSemanticsAttribute::UNKNOWN:
		case hkSemanticsAttribute::DISTANCE:
		case hkSemanticsAttribute::NORMAL:
		case hkSemanticsAttribute::POSITION:
		default:
			{
				break;
			}
		}
	}
}

void ComparisonUtil::storeDifference(const hkClassMember& member, hkStringBuf& valueAstr, hkStringBuf& valueBstr, const hkStringBuf parentStr, TreeIter parentItr, const int elem)
{
	applyAttributes(member, valueAstr, valueBstr);

	StringValues values;
	StringValues treeValues;
	if (elem == -1)
	{
		values.m_memPath.printf("%s%s", parentStr.cString(), member.getName());
		treeValues.m_memPath = member.getName();
	}
	else
	{
		values.m_memPath.printf("%s%s[%d]", parentStr.cString(), member.getName(), elem);
		treeValues.m_memPath.printf("[%d]", elem);
	}

	values.m_valueA = valueAstr;
	values.m_valueB = valueBstr;
	treeValues.m_valueA = valueAstr;
	treeValues.m_valueB = valueBstr;

	m_differences.pushBack(values);
	m_diffTree.appendChild(parentItr, treeValues);
}

hkBool ComparisonUtil::loopInHierachy(const TreeIter currentParent, const char* currentMember)
{
	TreeIter ancestor = currentParent;
	while (ancestor)
	{
		if (ancestor->m_value.m_memPath.compareTo(currentMember) == 0)
		{
			return true;
		}
		ancestor = ancestor->m_parent;
	}
	return false;
}

void ComparisonUtil::printDifferences(hkBool printToConsole, hkBool printToFile, const char* filename)
{
	if (!printToConsole && !printToFile)
	{
		return;
	}

	hkOstream outfile = hkOstream(filename);

	hkStringBuf string;
	for(int diff = 0; diff< m_differences.getSize(); diff++)
	{
		string.clear();
		string.printf("Difference between %s\n", m_differences[diff].m_memPath.cString());
		string.appendPrintf("ObjectA: %s\n", m_differences[diff].m_valueA.cString());
		string.appendPrintf("ObjectB: %s\n", m_differences[diff].m_valueB.cString());
		if (printToConsole)
		{
			hkcout << string.cString();
		}
		if (printToFile)
		{
			outfile << string;
		}
	}
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
