/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Menu/MenuDemo.h>
#include <Demos/DemoCommon/Utilities/Tweaker/Tweaker.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/Container/String/hkString.h>
#include <Common/Base/Container/StringMap/hkStorageStringMap.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Serialize/Version/hkVersionUtil.h>
#include <Common/Visualize/hkDebugDisplay.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Texture/hkgTexture.h>

#include <Demos/DemoCommon/DemoFramework/hkPerformanceCounterUtility.h>

#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>

#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>

#include <Graphics/Bridge/StatGraph/hkgStatGraph.h>
#include <Graphics/Bridge/MemoryTrackerGraph/hkgMemoryTrackerGraph.h>
#include <Graphics/Bridge/SeriesGraph/hkgTimerGraph.h>

#include <Graphics/Bridge/System/hkgSystem.h>
#include <Graphics/Common/Font/hkgFont.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>

#ifdef HK_MEMORY_TRACKER_ENABLE
	#include <Common/Base/Memory/Tracker/Default/hkDefaultMemoryTracker.h>

	#include <Common/Base/Memory/Tracker/ScanCalculator/hkTrackerScanCalculator.h>
	#include <Common/Base/Memory/Tracker/ScanCalculator/hkTrackerSnapshotUtil.h>

	#include <Common/Base/Reflection/TypeTree/hkTrackerTypeTreeParser.h>

	#include <Common/Base/Memory/Tracker/Report/hkCategoryReportUtil.h>
	#include <Common/Base/Memory/Tracker/Report/hkProductReportUtil.h>
	#include <Common/Base/Memory/Tracker/Report/hkVdbStreamReportUtil.h>


#endif

#include <Common/Serialize/Packfile/hkPackfileData.h>

#include <Common/Serialize/Packfile/hkPackfileWriter.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>


#include <Common/Base/DebugUtil/DeterminismUtil/hkNetworkedDeterminismUtil.h>
#include <Demos/DemoCommon/DemoFramework/hkFloatingPointExceptionCheck.h>
#include <Demos/DemoCommon/Utilities/GpuStats/GpuStatsUtil.h>

#if !defined(HK_DYNAMIC_DLL)
#	define HK_UNIT_TESTS
#endif

#if defined(HK_INTERNAL_BUILD) && !defined(HK_DYNAMIC_DLL)
#	define HK_PERFORMANCE_TESTS
#endif

#if defined(HK_UNIT_TESTS)
void HK_CALL registerUnitTests( extArray<extStringPtr>& extraNames, extArray<hkDemoEntryRegister*>& allocedDemos);
#endif

#if defined(HK_PERFORMANCE_TESTS)
void HK_CALL registerPerformanceTests( extArray<extStringPtr>& extraNames, extArray<hkDemoEntryRegister*>& allocedDemos);
#endif

extern const unsigned int MenuImageDataSize;
extern const unsigned char MenuImageData[];

extern const unsigned int MenuImageLargeDataSize;
extern const unsigned char MenuImageLargeData[];


// colours for the menu text, normal and current active item.
const hkColor::Argb TEXT_COLOR_NORMAL = 0xffefefef;
const hkColor::Argb TEXT_COLOR_DIRECTORY = 0x7fffffff;
const hkColor::Argb TEXT_COLOR_CURRENT = 0xffffdf00;

//const float MENU_BACKGROUND_COLOR[] = {  30.0f/255.0f,  30.0f/255.0f, 100.0f/255.0f };
const float MENU_BACKGROUND_COLOR[] = { 35.0f/255.0f, 31.0f/255.0f, 32.0f/255.0f };
const float MENU_BACKGROUND_CHANGE[] = {  0.0005f,  -0.0005f, 0.0005f };
const float DEMO_BACKGROUND_COLOR[] = { 35.0f/255.0f, 31.0f/255.0f, 32.0f/255.0f };

// last demo
const char* LASTDEMO_FILENAME = "lastdemo.txt";
const char* LASTDEMO_FILENAME_BOOTSTRAP = "lastdemoBootstrap.txt";
const char* LASTTEST_FILENAME = "lasttest.txt";

MenuDemo* MenuDemo::m_menuDemo = HK_NULL;

MenuDemo* MenuDemo::getMenuDemo()
{
	return m_menuDemo;
}

bool MenuDemo::isHelpOn()
{
	return ( m_helpTimeLeft > 0.0f );
}

void MenuDemo::turnOffHelp()
{
	m_helpTimeLeft = 0.0f;
}

bool MenuDemo::areMonitorsOn()
{
	return m_wantMonitors;
}

void hkSetLastDemo(const char* namein)
{
	hkOfstream out(LASTDEMO_FILENAME);
	if(out.isOk())
	{
		out << namein;
	}
// 	hkStringBuf name = namein;
// 	int lastSlash = name.lastIndexOf('/');
// 	if(lastSlash >= 0)
// 	{
// 		// Never save bootstrap demos as last game
// 		if ( hkString::strStr(name.asLowerCase().cString(), "bootstrap") == HK_NULL )
// 		{
// 			hkString::strNcpy(s_lastDemoName, name.cString(), sizeof(s_lastDemoName));
// 			hkOfstream out(LASTDEMO_FILENAME);
// 			if(out.isOk())
// 			{
// 				out << s_lastDemoName;
// 			}
// 		}
// 	}
// 	return s_lastDemoName;
}


#ifndef HK_DISABLE_BOOTSTRAPPER
extern void hkSetLastDemoBootstrap(const char* namein);
#endif

// Insert spaces between consecutive lower and upper case letters
static const char* insertSpaces(extStringBuf& str)
{
	for(int j=1; j< str.getLength(); j++)
	{
		char c1 = str[j-1];
		char c2 = str[j];
		if(	( (c1>='a'&&c1<='z') && ( (c2>='A'&&c2<='Z') || (c2>='0'&&c2<='9') ) ) ||
			// put a space if we change from numeric to alpha or vice versa
			( (c1>='0'&&c1<='9') && ( (c2>='A'&&c2<='Z') || (c2>='a'&&c2<='z') ) ) ||
			( (c2>='0'&&c2<='9') && ( (c1>='A'&&c1<='Z') || (c1>='a'&&c1<='z') ) ) )
		{
			str.insert(j, " ", 1);
			j++;
		}
	}
	return str.cString();
}

static void canonicalise( extStringBuf& s )
{
	extArray<char> buf;
	buf.reserve( s.getLength()+1 );
	for( const char* p = s.cString(); *p != 0; ++p )
	{
		char c = *p;
		if( (c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || c == '/' || c == ' ' )
		{
			buf.pushBackUnchecked( c );
		}
		else if( c >= 'A' && c <= 'Z' )
		{
			buf.pushBackUnchecked( c + ('a' - 'A') );
		}
	}
	buf.pushBackUnchecked(0);
	s = buf.begin();
}



void HK_CALL hkDrawMenuImage( hkgTexture* t, hkDemoEnvironment* env, bool useCurrentViewport )
{
	hkgWindow* w = env->m_window;
	hkgDisplayContext* c = w->getContext();
	c->lock();

	hkgViewport* cv = w->getContext()->getCurrentViewport();
	hkgViewport* v = w->getWindowOrthoView();
	if (!useCurrentViewport )
		v->setAsCurrent(c);

	float white[4] = { 1.0f, 1.0f, 1.0f, 0.70f}; // change alpha to fade logo

	if (useCurrentViewport)
	{
		// assume for second view
		white[3] = 0.5f;
	}

	c->setCurrentSoleTexture( t, HKG_TEXTURE_MODULATE );

	c->setTexture2DState( true ); // turn on textures for this
	c->setBlendMode(HKG_BLEND_MODULATE); // should be the default anyway
	c->setBlendState( true ); // modulate with background color
	c->setCurrentColor4( white );

	float p[3];
	float uv[2];
	p[2] = -0.01f; //depth

	float tl[3];
	float lr[3];


	unsigned int wh = cv->getHeight() / 2;
	unsigned int ww = cv->getWidth() / 2;

	unsigned int logoH = wh * 2;
	unsigned int logoW = ww * 2;

	// keep square
	if (logoH < logoW)
		logoW = logoH;
	if (logoW < logoH)
		logoH = logoW;

	// adjust if widescreen and not in a widescreen res
	if (w->isWidescreen() && !w->hasWidescreenAspect())
	{
		logoW = (int)( ((float)logoW) * 3.0f/4.0f ); // squash in the logo by its adjusted aspect
	}

	tl[0] = (float)(ww - logoW/2);
	tl[1] = (float)(wh + logoH/2 );

	lr[0] = (float)(ww + logoW/2);
	lr[1] = (float)(wh - logoH/2);

	c->beginGroup( HKG_IMM_TRIANGLE_LIST );

	p[0] = tl[0]; p[1] = tl[1];
	uv[0] = 0.0f; uv[1] = 1.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 0.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1];
	uv[0] = 1.0f; uv[1] = 1.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = tl[0]; p[1] = lr[1];
	uv[0] = 0.0f; uv[1] = 0.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = lr[1];
	uv[0] = 1.0f; uv[1] = 0.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	p[0] = lr[0]; p[1] = tl[1];
	uv[0] = 1.0f; uv[1] = 1.0f;
	c->setCurrentTextureCoord( uv );
	c->setCurrentPosition( p );

	c->endGroup();

	// restore
	if (!useCurrentViewport )
		cv->setAsCurrent(c);

	c->unlock();
}



//
// Menu Constructor
//



MenuDemo::MenuDemo(hkDemoEnvironment* environment)
	:	hkDefaultDemo(environment, true),
		m_defaultDemo(environment->m_options->m_defaultDemo),
		m_currentPath(""),
		m_currentDemo(HK_NULL),
		m_helpTimeMax(15.0f),
		m_helpTimeLeft(15.0f),
		m_paused(false),
		m_wantMonitors(false),
		m_wantTweak(false),
		m_wantCameraData(false),
		m_activeNode(HK_NULL),
		m_statsTree(HK_NULL),
		m_mouseInGraphZoomMode(false),
		m_mousePadTrackedButton(-1),
		m_viewportMouseEventBackup(true),
		m_mouseStatZoomOriginX(0),
		m_mouseStatZoomOriginY(0),
		m_searchIndex(0),
		m_tweaker(HK_NULL),
		m_gpuStats(false)
{
	m_stopwatch.start();
	hkgVec3Copy(m_menuImageBackground, MENU_BACKGROUND_COLOR );
	hkgVec3Copy(m_menuImageBackgroundDirection, MENU_BACKGROUND_CHANGE );
	environment->m_window->setClearColor( m_menuImageBackground );
	environment->m_window->setWantDrawHavokLogo(false);

	m_newTimersGathered = false;

	m_gpuStatsUtil = new GpuStatsUtil;

	m_graphManager = new GraphManager(environment);

	m_mainMemoryGraph = new hkgSeriesGraph;
	m_graphManager->addMemoryGraph(m_mainMemoryGraph);
	int size = 0;

	// set the "singleton" menu demo to me
	m_menuDemo = this;

#if DEMOS_ENABLE_TIMER_CAPTURE
	if (!m_env->m_options->m_notimers)
	{
#if defined(HK_SHOWCASE_BUILD)
		size = 20*1024*1024; // 3MB stats (these are per thread, so this is only for the main thread), default demo makes 2MB for each worker CPU thread, 100KB for each SPU thread.
	#elif defined(HK_PLATFORM_X64)
		size = 8*1024*1024; // 8MB stats (these are per thread, so this is only for the main thread), default demo makes 2MB for each worker CPU thread, 100KB for each SPU thread.
	#else
		size = 3*1024*1024; // 3MB stats (these are per thread, so this is only for the main thread), default demo makes 2MB for each worker CPU thread, 100KB for each SPU thread.
	#endif
	}
#endif

	if (size > 0)
	{
		m_savedStreamBegin = hkMemDebugBlockAlloc<char>( size );
		m_savedStreamCurrent = m_savedStreamBegin;
		m_savedStreamEnd = m_savedStreamBegin + size;
		HK_MEMORY_TRACKER_NEW_RAW("TimerCaptureBuffer", m_savedStreamBegin, size);
		m_simTimersEnd = hkMonitorStream::getInstance().getEnd();
	}
	else
	{
		m_savedStreamBegin = HK_NULL;
		m_savedStreamCurrent =  HK_NULL;
		m_savedStreamEnd = HK_NULL;
		m_simTimersEnd = HK_NULL;
	}

	// bring up the logo
	if (hkgSystem::g_RendererType > hkgSystem::HKG_RENDERER_NULL)
	{
			const char* bestImage = (const char*)MenuImageLargeData;
			int bestImageSize = MenuImageLargeDataSize;

		hkIstream s( bestImage, bestImageSize );
		m_menuImage = hkgTexture::create(environment->m_window->getContext());
		m_menuImage->loadFromPNG( s );
		m_menuImage->realize();
	}
	else
	{
		m_menuImage = HK_NULL;
	}

	hkMonitorStream& stream = hkMonitorStream::getInstance();
	stream.resize( size );
	stream.reset();

	m_performanceCounterUtility = new hkPerformanceCounterUtility( );
	if (environment->m_options->m_enablePerformanceCounter)
	{
		m_performanceCounterUtility->enable((hkPerformanceCounterUtility::CounterSelect)environment->m_options->m_perfomanceCounterOption);
		// Scaling factor to be used for the output timing. Uncomment the line below to measure elapsed seconds.
		// m_performanceCounterUtility->getFrameInfo().m_timerFactor0 = 1.0f / float(hkStopwatch::getTicksPerSecond());
	}

#if defined(HK_UNIT_TESTS)
	registerUnitTests(m_extraNames, m_extraDemos);
#endif

#if defined(HK_PERFORMANCE_TESTS)
	registerPerformanceTests(m_extraNames, m_extraDemos);
#endif


	//	Rebuild the demo database
	hkDemoDatabase::getInstance().rebuildDatabase();

	hkNetworkedDeterminismUtil::create();

#if defined (HK_ENABLE_DETERMINISM_CHECKS)
	hkCheckDeterminismUtil::createInstance();
	m_env->m_options->m_lockFps = -1;
	m_determinismCheckRunIndex = 0;
#endif

	// Copy the demo framework options, just in case another demo tries to mess with them.
	m_originalOptions = *m_env->m_options;

	if( m_defaultDemo != HK_NULL )
	{
		// Attempt to create default demo
		startCurrentDemo();
	}
	else
	{
		// Attempt to retrieve last demo path (not supported on Wii currently).
		hkIstream is(LASTDEMO_FILENAME);
		if( is.isOk() )
		{
			char buffer[512];
			if( is.getline( buffer, 510 ) > 0 )
			{
				m_currentPath = buffer;
			}
		}
	}

}

MenuDemo::~MenuDemo()
{
	hkNetworkedDeterminismUtil::destroy();

	if (m_currentDemo)
	{
		stopCurrentDemo(); // so that stats can be written if required for it
	}

	delete m_tweaker;
	m_mainMemoryGraph->removeReference();
	delete m_graphManager;

	for (int i=0; i< m_extraDemos.getSize(); i++)
	{
		delete m_extraDemos[i];
	}

	if (m_menuImage)
	{
		m_env->m_window->getContext()->lock();
		m_menuImage->release();
		m_env->m_window->getContext()->unlock();
	}

	delete m_performanceCounterUtility;

	if (m_savedStreamBegin)
	{
		hkMemDebugBlockFree<char>( m_savedStreamBegin, int(m_savedStreamEnd-m_savedStreamBegin) );
		HK_MEMORY_TRACKER_DELETE_RAW(m_savedStreamBegin);
	}

	// clear the "singleton" menu demo if it is me
	if ( m_menuDemo == this )
	{
		m_menuDemo = HK_NULL;
	}

#if defined (HK_ENABLE_DETERMINISM_CHECKS)
	hkCheckDeterminismUtil::destroyInstance();
#endif

	HKG_SAFE_DELETE( m_gpuStatsUtil );
}

hkDemo::Result MenuDemo::stepVisualDebugger()
{
	if(!m_paused && m_currentDemo != HK_NULL) // run our current demo if present.
	{
		return m_currentDemo->stepVisualDebugger();
	}

	return hkDemo::DEMO_OK;
}

namespace {

	struct hkNetworkedDeterminismUtil_ControlInfo
	{
		hkgPad m_gamePadA;
		hkgPad m_gamePadB;
		hkgKeyboard m_keyboard;
		hkgMouse m_mouse;

		hkgPad m_gamePadA_nonWindow;
		hkgPad m_gamePadB_nonWindow;

	};

} // namespace

void MenuDemo::advanceFrame()
{
	if (m_currentDemo)
	{
		m_currentDemo->advanceFrame();
	}
	else
	{
		hkDemo::advanceFrame();
	}
}

//
// stepDemo
//
hkDemo::Result MenuDemo::stepDemo()
{
	{
		hkNetworkedDeterminismUtil::ControlCommand controlCommand;
		{
			hkNetworkedDeterminismUtil_ControlInfo info;
			// This actually is false.
			//HK_ASSERT2(0xad903112, m_env->m_gamePad  == &m_env->m_window->getGamePad(0), "GamePad pointers don't match");
			//HK_ASSERT2(0xad903112, m_env->m_gamePadB == &m_env->m_window->getGamePad(1), "GamePad pointers don't match");

			info.m_gamePadA = m_env->m_window->getGamePad(0);
			info.m_gamePadB = m_env->m_window->getGamePad(1);
			info.m_keyboard = m_env->m_window->getKeyboard();
			info.m_mouse    = m_env->m_window->getMouse();

			info.m_gamePadA_nonWindow = *m_env->m_gamePad;
			info.m_gamePadB_nonWindow = *m_env->m_gamePadB;

			controlCommand.m_data.setSize(sizeof(info));
			hkString::memCpy(controlCommand.m_data.begin(), &info, sizeof(info));
		}

		hkNetworkedDeterminismUtil::startStepDemo(controlCommand);
		if (hkNetworkedDeterminismUtil::isClient())
		{
			const hkNetworkedDeterminismUtil_ControlInfo& info = *reinterpret_cast<const hkNetworkedDeterminismUtil_ControlInfo*>(controlCommand.m_data.begin());
			m_env->m_window->setGamePad(0, info.m_gamePadA);
			m_env->m_window->setGamePad(1, info.m_gamePadB);
			m_env->m_window->setKeyboard(info.m_keyboard);
			m_env->m_window->setMouse(info.m_mouse);

			*m_env->m_gamePad = info.m_gamePadA_nonWindow;
			*m_env->m_gamePadB = info.m_gamePadB_nonWindow;
		}
	}

	Result result;

	// run our current demo if present
	if ( m_currentDemo != HK_NULL )
	{
		result = stepCurrentDemo();
	}

	// if we had a default demo then exit
	else if ( m_defaultDemo != HK_NULL )
	{
		result = DEMO_STOP;
	}

	// otherwise just run the menu
	else
	{
		result = stepMenuDemo();
	}

	hkNetworkedDeterminismUtil::endStepDemo();

	return result;
}

void MenuDemo::preRenderDisplayWorld(hkgViewport* v)
{
	if(m_currentDemo != HK_NULL) // run our current game if present.
	{
		m_currentDemo->preRenderDisplayWorld(v);
	}
}

void MenuDemo::postRenderDisplayWorld(hkgViewport* v)
{
	if(m_currentDemo != HK_NULL) // run our current game if present.
	{
		m_currentDemo->postRenderDisplayWorld(v);
	}
	else if (m_menuImage)
	{
		hkDrawMenuImage(m_menuImage, m_env, false);
	}
}

void MenuDemo::renderSecondaryViewport(class hkgViewport* v)
{
	// Menu image
	hkDrawMenuImage(m_menuImage, m_env, true);

	// Some stats etc if enabled
	renderStatGraphs(true);

	// Could render help text etc now too if u like

	// Or even debug display 3D stuff etc
}

void MenuDemo::renderStatGraphs(bool useCurrentViewport)
{
	m_graphManager->renderGraphs(m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, useCurrentViewport, m_mouseInGraphZoomMode);
}

void MenuDemo::postRenderWindow(hkgWindow* w)
{
	if(m_currentDemo != HK_NULL) // run our current game if present.
	{
		m_currentDemo->postRenderWindow(w);
	}

	if (!w->getSecondaryViewport())
	{
		renderStatGraphs(false);
	}
}

void MenuDemo::windowResize(int w, int h)
{

	m_graphManager->onWindowResize(w,h);

}

void MenuDemo::windowDropFile(const char* filename, int x, int y)
{
	if (m_currentDemo)
	{
		m_currentDemo->windowDropFile(filename, x, y);
	}
}

bool MenuDemo::touchDown(int id, int x, int y)
{
	//XX Menu select etc

	if (m_currentDemo)
	{
		return m_currentDemo->touchDown(id, x, y);
	}
	return false;
}

void MenuDemo::touchMove(int id, int x, int y, int dx, int dy)
{
	//XX Menu move etc

	if (m_currentDemo)
	{
		m_currentDemo->touchMove(id, x, y, dx, dy);
	}
}

void MenuDemo::touchUp(int id, int x, int y)
{
	if (m_currentDemo)
	{
		m_currentDemo->touchUp(id, x, y);
	}
}

void MenuDemo::accelerometerChange(const float highPass[3], const float lowPass[3])
{
	if (m_currentDemo)
	{
		m_currentDemo->accelerometerChange(highPass, lowPass);
	}
}

hkDemo::Result MenuDemo::showPausedMenu( bool handleKeysOnly )
{
	if (!handleKeysOnly)
	{
		int dH = m_env->m_window->getTVDeadZoneH();
		int dV = m_env->m_window->getTVDeadZoneV();

		extStringBuf oss;
		oss.printf("[ Paused ]\n\n%c Resume Demo\n%c Restart Demo\n\n%c Toggle Text Statistics\n%c Cycle Graph Statistics\n%c Toggle Memory Graph\n%c Settings\n%c Toggle Help\n%c Single Step\n\n%c Quit",
			// resume,             restart            stats              graph				 memory graph		    settings            help                step               quit
			HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_3, HKG_FONT_BUTTON_1, HKG_FONT_BUTTON_L1, HKG_FONT_BUTTON_SELECT,HKG_FONT_BUTTON_R1, HKG_FONT_BUTTON_L2, HKG_FONT_BUTTON_0, HKG_FONT_BUTTON_2);

		m_env->m_textDisplay->outputText(oss, 20 + dH, 20 + dV, 0xffffffff);
	}

	hkgPad* gamePad = m_env->m_gamePad;
	const hkgMouse& mouse = m_env->m_window->getMouse();

	int px = mouse.getPosX();
	int py = mouse.getPosY();

	bool inGraphBounds = m_graphManager->inStatGraphBounds(px, py);

	if( m_env->wasButtonPressed( HKG_PAD_BUTTON_L2) ) // help
	{
		// Don't pass button press on to demo
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_L2 );
		// If no keyboard, there's no F1 to get the keyboard shortcuts
		// If that's the case, set the helpTime to HK_REAL_MAX
		m_helpTimeLeft = (m_helpTimeLeft <= 0.0f) ? (m_env->m_window->getKeyboard().isConnected() ? m_helpTimeMax : HK_REAL_MAX) : 0.0f;
		m_paused = false;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_R1) ) // settings
	{
		// Don't pass button press on to demo
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_R1 );
		m_wantTweak = true;
		m_paused = false;
	}
// 	else if( m_env->wasButtonPressed( HKG_PAD_DPAD_DOWN) ) // display camera position & target
// 	{
// 		// Don't pass button press on to demo
// 		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_DPAD_DOWN );
// 		m_paused = false;
// 		if ( m_wantCameraData )
// 		{
// 			m_wantCameraData = false;
// 		}
// 		else
// 		{
// 			m_wantCameraData = true;
// 		}
// 	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_3) ) // restart
	{
		// Don't pass button press on to demo
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_1 );
		stopCurrentDemo();
		startCurrentDemo();
		m_paused = false;
		return DEMO_OK;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_1) ) // toggle timer text
	{
		m_wantMonitors = !m_wantMonitors;
		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_L1) ) // cycle timer graphs
	{
		cycleTimerGraph();

		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_SELECT) ) // toggle memory graphs
	{
		toggleMemoryGraphs();

		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_R2) ) // toggle mem graph
	{
		toggleTrackerGraph();

		m_helpTimeLeft = 0.0f;
		m_paused = true;
		return DEMO_PAUSED;
	}
	else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_2) ) // quit
	{
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_2 );
		stopCurrentDemo();
		return DEMO_OK;
	}
	else if ( ( m_env->wasButtonPressed( HKG_PAD_BUTTON_0) ) || // single step
			(inGraphBounds && mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON)) )
	{
		if (inGraphBounds)
		{
			hkBool hasAction = m_graphManager->manageUserAction(px, py);

			if (!hasAction)
			{
				m_mouseStatZoomOriginX = px;
				m_mouseStatZoomOriginY = py;
				if (mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON))
				{
					m_viewportMouseEventBackup = m_env->m_window->getCurrentViewport()->getAcceptsMouseEvents();
					m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(false);
				}
			}
			else
			{
				m_mouseInGraphZoomMode = false;
			}

			return DEMO_PAUSED;
		}

		// Don't pass button press on to demo
		// but fall out and do a single step if not doing a select using space etc in the stat bar
		gamePad->setButtonState( gamePad->getButtonState() ^ HKG_PAD_BUTTON_0 );
	}
	else if ((gamePad->wasButtonReleased(HKG_PAD_BUTTON_0)) ||
			(m_mouseInGraphZoomMode && mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON)) )
	{
		if (m_mouseInGraphZoomMode && mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON))
		{
			m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(m_viewportMouseEventBackup);
		}

		if (inGraphBounds && m_mouseInGraphZoomMode)
		{
			m_graphManager->mouseZoom(m_mouseStatZoomOriginX, m_mouseStatZoomOriginY, px, py);
		}
		m_mouseInGraphZoomMode = false;

		return DEMO_PAUSED;
	}
	else
	{
		return DEMO_PAUSED;
	}
	return DEMO_ERROR;
}

void MenuDemo::toggleTrackerGraph()
{
	// Disable any timer graphs
	m_graphManager->toggleGraph(GraphManager::TIMER, false);
	m_graphManager->toggleGraph(GraphManager::STATS, false);

	// Turn on or off the tracker
	bool newState = !m_graphManager->isEnabled(GraphManager::TRACKER);
	m_graphManager->toggleGraph(GraphManager::TRACKER, newState);
}

void MenuDemo::toggleMemoryGraphs()
{
	m_graphManager->toggleGraph(GraphManager::MEMORY, !m_graphManager->isEnabled(GraphManager::MEMORY));
	m_mainMemoryGraph->setNumChannels(1);
	m_mainMemoryGraph->setFormat(hkgTimerGraph::SI_PREFIX, "B");
}

void MenuDemo::cycleTimerGraph()
{
	// Disable any tracker graph
	m_graphManager->toggleGraph(GraphManager::TRACKER, false);


	if (m_graphManager->isEnabled(GraphManager::TIMER))
	{
		m_graphManager->toggleGraph(GraphManager::TIMER, false);

		m_graphManager->toggleGraph(GraphManager::STATS, true);
	}
	else if (m_graphManager->isEnabled(GraphManager::STATS))
	{
		m_graphManager->toggleGraph(GraphManager::STATS, false);
	}
	else
	{
		m_graphManager->toggleGraph(GraphManager::TIMER, true);
		m_graphManager->m_timerGraph->setFormat(hkgSeriesGraph::FLOAT_VALUES, "ms");

		// Add series from the timers
		extArray< extStringPtr > timerNames;
		m_currentDemo->getGraphedTimers( timerNames );
		for( int i=0; i<timerNames.getSize(); ++i )
		{
			if ((hkString::strCmp(timerNames[i],"Physics") == 0) || (hkString::strCmp(timerNames[i],"Physics 2012") == 0))
			{
				// Force white
				m_graphManager->m_timerGraph->addSeries( timerNames[i], hkColor::WHITE );
			}
			else if ((hkString::strCmp(timerNames[i],"Destruction") == 0) || (hkString::strCmp(timerNames[i],"Destruction 2012") == 0))
			{
				// Force yellow
				m_graphManager->m_timerGraph->addSeries( timerNames[i], hkColor::YELLOW );
			}
			else
			{
				m_graphManager->m_timerGraph->addSeries( timerNames[i] );
			}
		}
	}
}

hkDemo::Result MenuDemo::showTweakMenu()
{
	int dH = m_env->m_window->getTVDeadZoneH();
	int dV = m_env->m_window->getTVDeadZoneV();

	hkVariant* options = m_currentDemo->getOptions();
	if( options )
	{
		const hkgWindow* window = m_env->m_window;
		{
			extStringBuf oss;
			oss.printf("[ Tweaking ]\n\n%c Tweak up\n%c Tweak down\n%c Restart with new parameters\n\n%c Stop tweaking\n%c %s",
				HKG_FONT_BUTTON_3, HKG_FONT_BUTTON_2, HKG_FONT_BUTTON_1, HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_0, m_paused ? "Resume" : "Pause" );
			hkReal lineHeight = m_env->m_textDisplay->getFont()->getCharHeight();
			m_env->m_textDisplay->outputText(oss, 20 + dH, int(window->getHeight()) - int(lineHeight*9) - dV, TEXT_COLOR_NORMAL);
		}
		HK_ASSERT(0x12cdf4de, options->m_object);
		HK_ASSERT(0x12cdf4df, options->m_class);
		if( m_tweaker && m_tweaker->getData() != options->m_object )
		{
			delete m_tweaker;
			m_tweaker = HK_NULL;
		}
		if( m_tweaker==HK_NULL )
		{
			m_tweaker = new Tweaker( options->m_object, *options->m_class );
			extern const hkClass MenuDemoEmptyClass;
			hkVersionUtil::copyDefaults( options->m_object, MenuDemoEmptyClass, *options->m_class );
			m_tweaker->m_activateButton = 0;
			m_tweaker->m_activateKey = 0;
			m_tweaker->m_clearButton = 0;
			m_tweaker->m_tweakUpButton = HKG_PAD_BUTTON_2; // ^
			m_tweaker->m_tweakDownButton = HKG_PAD_BUTTON_3; // O
			m_tweaker->m_tweakDownKey = '2';
			m_tweaker->m_tweakDownKey2 = 0;
			m_tweaker->m_tweakUpKey = '3';
			m_tweaker->m_clearButton = 0;
			m_tweaker->setMode( Tweaker::MODE_ACTIVE );
			m_tweaker->toNextSibling();
		}

		if( m_env->wasButtonPressed( HKG_PAD_BUTTON_0) ) // X
		{
			m_paused = !m_paused;
		}
		else if( m_env->wasButtonPressed( HKG_PAD_BUTTON_1) ) // []
		{
			stopCurrentDemo();
			startCurrentDemo();
			return DEMO_OK;
		}

		m_tweaker->update(m_env);
	}
	else
	{
		extStringBuf oss;
		oss.printf("[ Tweaking ]\n\n%c Resume Demo\n%c Previous Menu\n\nNo tweakable parameters", HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_0);
		m_env->m_textDisplay->outputText(oss, 20 + dH, 20 + dV, TEXT_COLOR_NORMAL);

		if( m_env->wasButtonPressed( HKG_PAD_BUTTON_0) )
		{
			m_wantTweak = false;
		}
	}

	if(m_paused)
	{
		return DEMO_PAUSED;
	}
	return DEMO_ERROR;
}

void MenuDemo::stepMainMemoryGraph(const hkMemorySystem::MemoryStatistics& stats)
{
	if( m_graphManager->isEnabled(GraphManager::MEMORY) )
	{
		for( int i = 0; i < stats.m_entries.getSize(); ++i )
		{
			const hkMemorySystem::MemoryStatistics::Entry& entry = stats.m_entries[i];

			// Simple hash function to have the same hue for the two series
			hkUint32 hash = (entry.m_allocatorName[0] + entry.m_allocatorName[1]*256);
			hash = (hash * (hash + 3) *239) % 255;
			hkReal hue = hash / 255.0f;
			int color = hkColor::rgbFromHSV( hue, 0.6f, 1.0f );


			// display heap in use
			if( hkString::strCmp(entry.m_allocatorName.cString(), "Heap") == 0 )
			{
				m_mainMemoryGraph->addSeries( "Heap in use", color, false );
				m_mainMemoryGraph->addData( "Heap in use", 0, (hkReal)entry.m_allocatorStats.m_inUse );
			}
		}
		m_mainMemoryGraph->postProcess();
	}
}

void MenuDemo::stepTimerGraph( extArray<hkTimerData>& threadStreamInfos, extArray<hkTimerData>& spuStreamInfos )
{
	const int numThreads = threadStreamInfos.getSize();
	const int numSpus = spuStreamInfos.getSize();
	if( !m_graphManager->isEnabled(GraphManager::TIMER) || (numThreads <= 0 && numSpus <= 0) )
	{
		return;
	}

	hkMonitorStreamFrameInfo frameInfo;
	{
		frameInfo.m_indexOfTimer0 = 0;
		frameInfo.m_indexOfTimer1 = -1;
		frameInfo.m_timerFactor0 = 1e3f / float(hkStopwatch::getTicksPerSecond());	// to ms
		frameInfo.m_timerFactor1 = 1;
		frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;
	}

	extArray<hkTimerData> streamInfos( numThreads + numSpus );
	{
		for( int i=0; i<numThreads; ++i )
		{
			streamInfos[i] = threadStreamInfos[i];
		}
		for( int i=0; i<numSpus; ++i )
		{
			streamInfos[numThreads+i] = spuStreamInfos[i];
		}
	}

	m_graphManager->m_timerGraph->update( frameInfo, streamInfos.begin(), numThreads, numSpus );
}

void MenuDemo::stepStatGraph( extArray<hkTimerData>& threadStreamInfos, extArray<hkTimerData>& spuStreamInfos )
{
	const int numThreads = threadStreamInfos.getSize();
	const int numSpus = spuStreamInfos.getSize();
	if( !m_graphManager->isEnabled(GraphManager::STATS) || (numThreads <= 0 && numSpus <= 0) )
	{
		return;
	}

	hkMonitorStreamFrameInfo frameInfo;
	{
		frameInfo.m_indexOfTimer0 = 0;
		frameInfo.m_indexOfTimer1 = -1;
		//the ms marker lines assume this (us, not ms):
		frameInfo.m_timerFactor0 = 1e6f / float(hkStopwatch::getTicksPerSecond());
		frameInfo.m_timerFactor1 = 1;
		frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;
	}

	// Create a node tree per thread
	extArray<hkMonitorStreamAnalyzer::Node*> threadTrees;
	{
		for (int destId = 0; destId < threadStreamInfos.getSize(); ++destId)
		{
			hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(threadStreamInfos[destId].m_streamBegin, threadStreamInfos[destId].m_streamEnd, frameInfo, "/", false );

			hkMonitorStreamAnalyzer::Node* perThreadPerFrame = new hkMonitorStreamAnalyzer::Node(HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY);
			perThreadPerFrame->m_children.pushBack( currentTree );
			threadTrees.pushBack( perThreadPerFrame );
		}

		for (int destId = 0; destId < spuStreamInfos.getSize(); ++destId)
		{
			hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(spuStreamInfos[destId].m_streamBegin, spuStreamInfos[destId].m_streamEnd, frameInfo, "/", false);

			hkMonitorStreamAnalyzer::Node* perThreadPerFrame = new hkMonitorStreamAnalyzer::Node(HK_NULL, "/", hkMonitorStreamAnalyzer::Node::NODE_TYPE_DIRECTORY);
			perThreadPerFrame->m_children.pushBack( currentTree );
			threadTrees.pushBack( perThreadPerFrame );
		}
	}

	// update may alter the node tree given to disable timers
	m_graphManager->m_statGraph->update( threadTrees, frameInfo, numThreads, numSpus );

	if (!m_mouseInGraphZoomMode)
	{
		int mouseX = m_env->m_window->getMouse().getPosX();
		int mouseY = m_env->m_window->getMouse().getPosY();

		hkMonitorStreamAnalyzer::Node* timerNodeUnderMouse = m_graphManager->m_statGraph->findTimerNodeAtSample(threadTrees, frameInfo, numThreads, numSpus, mouseX, mouseY );
		if (timerNodeUnderMouse)
		{
			extStringBuf nodeInfo;
			nodeInfo.printf("%s/%s:%.3fms", timerNodeUnderMouse->m_parent? timerNodeUnderMouse->m_parent->m_name : "", timerNodeUnderMouse->m_name, timerNodeUnderMouse->m_value[frameInfo.m_indexOfTimer0] * 0.001f );
			m_env->m_textDisplay->outputText( nodeInfo, mouseX, (int)( getWindowHeight() - mouseY - 16));
		}
	}

	for (int tt=0; tt < threadTrees.getSize(); ++tt)
	{
		delete threadTrees[tt];
	}
}


void MenuDemo::showMemoryTracker()
{
	if (!m_paused)
	{
		m_graphManager->m_trackerGraph->update(m_timestep);
	}

	if (!m_mouseInGraphZoomMode)
	{
		int mouseX = m_env->m_window->getMouse().getPosX();
		int mouseY = m_env->m_window->getMouse().getPosY();

		int index =  m_graphManager->m_trackerGraph->pickOrderedBlockIndex(mouseX, mouseY);

		if (index >= 0)
		{
			const hkgMemoryTrackerGraph::OrderedBlock& orderedBlock = m_graphManager->m_trackerGraph->getOrderedBlock(index);
			const hkTrackerScanSnapshot::Block* block = orderedBlock.m_block;

			hkArray<char> text;
			hkOstream stream(text);

			stream << "Category: " << hkCategoryReportUtil::getCategoryName(hkCategoryReportUtil::Category(orderedBlock.m_category)) << "\n";
			stream << "Type: ";

			hkScanReportUtil::appendBlockType(block, stream);
			stream << "\n";
			stream << "Size: " << hkScanReportUtil::MemorySize(orderedBlock.m_size);

			text.pushBack(0);

			m_env->m_textDisplay->outputText( text.begin(), mouseX, (int)( getWindowHeight() - mouseY - 16));
		}
	}
}

void MenuDemo::showMonitorAnalysis(int xOffset, extArray<hkTimerData>& threadStreamInfos, extArray<hkTimerData>& spuStreamInfos )
{
	hkMonitorStreamFrameInfo frameInfo;
	{
		frameInfo.m_indexOfTimer0 = 0;
		frameInfo.m_indexOfTimer1 = -1;
		frameInfo.m_timerFactor0 = 1e3f / float(hkStopwatch::getTicksPerSecond());
		frameInfo.m_timerFactor1 = 1;
		frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;
	}

	if (m_statsTree == HK_NULL)
	{
		m_statsTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(threadStreamInfos[0].m_streamBegin, threadStreamInfos[0].m_streamEnd, frameInfo, "/", true );
		m_activeNode = HK_NULL;
	}
	else
	{
		//
		// Take the monitor streams from the last frame and blend them into the results for smooth display. If paused, just show the actual
		// timings from last frame.
		//

		hkReal blendFromLastFrame = m_paused ? 0 : .8f;

		for (int destId = 0; destId < threadStreamInfos.getSize(); ++destId)
		{
			hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(threadStreamInfos[destId].m_streamBegin, threadStreamInfos[destId].m_streamEnd, frameInfo, "/", true );
			hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary(m_statsTree, currentTree, destId, 0, blendFromLastFrame );
			delete currentTree;
		}
		for (int destId = 0; destId < spuStreamInfos.getSize(); ++destId)
		{
			hkMonitorStreamAnalyzer::Node* currentTree = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(spuStreamInfos[destId].m_streamBegin, spuStreamInfos[destId].m_streamEnd, frameInfo, "/", true );
			hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary(m_statsTree, currentTree, destId + threadStreamInfos.getSize(), 0, blendFromLastFrame );
			delete currentTree;
		}
	}

	{

		// do we have enough stats to init the active node.
		// As the sims can start off with no active node (no children in stats, esp multithread demos)
		if (!m_activeNode && m_statsTree && (m_statsTree->m_children.getSize() > 0))
		{
			m_statsTree->m_userFlags |= 1;
			m_activeNode = m_statsTree->m_children[0];
		}

		// Navigation
		if (m_paused || (m_env->m_gamePad->getButtonState() & HKG_PAD_BUTTON_0) || (m_env->m_gamePadB->getButtonState() & HKG_PAD_BUTTON_0))
		{
			hkMonitorStreamAnalyzer::CursorKeys keys;
			keys.m_upPressed    = m_env->wasButtonPressed(HKG_PAD_DPAD_UP);
			keys.m_downPressed  = m_env->wasButtonPressed(HKG_PAD_DPAD_DOWN);
			keys.m_leftPressed  = m_env->wasButtonPressed(HKG_PAD_DPAD_LEFT);
			keys.m_rightPressed = m_env->wasButtonPressed(HKG_PAD_DPAD_RIGHT);
			m_activeNode = hkMonitorStreamAnalyzer::navigateMonitors( keys, m_activeNode );
		}

		hkArray<char> osBuf;
		hkOstream os(osBuf);

		os  << "Hold " << HKG_FONT_BUTTON_0 << " and use "
			<< HKG_FONT_BUTTON_UP << HKG_FONT_BUTTON_DOWN << HKG_FONT_BUTTON_LEFT << HKG_FONT_BUTTON_RIGHT
			<< " to navigate\n\n";

		{
			hkMonitorStreamAnalyzer::CombinedThreadSummaryOptions options;
			options.m_activeNode = m_activeNode;
			options.m_displayPartialTree = true;
			options.m_downArrowChar = HKG_FONT_BUTTON_DOWN;
			options.m_rightArrowChar = HKG_FONT_BUTTON_RIGHT;
			options.m_indentationToFirstTimerValue = -1;
			options.m_timerColumnWidth = 16;
			options.m_tabSpacingForTimerNames = 1;
			options.m_tabSpacingForTimerValues = 1;
			options.m_useTabsNotSpacesForColumns = false;

			hkMonitorStreamAnalyzer::showCombinedThreadSummaryForSingleFrame(m_statsTree, threadStreamInfos.getSize(), spuStreamInfos.getSize(), os, options );
		}

		int dH = m_env->m_window->getTVDeadZoneH();
		int dV = m_env->m_window->getTVDeadZoneV();

		m_env->m_textDisplay->outputText( osBuf.begin(), 10 + dH, 10 + dV, 0xffffffff, -1, -1, true /*monospace*/ );
	}
}

static int keyToButtonFontMapping[][2] =
{
	{ HKG_VKEY_SPACE,	HKG_FONT_BUTTON_0		},
	{ '1',				HKG_FONT_BUTTON_1		},
	{ '2',				HKG_FONT_BUTTON_2		},
	{ '3',				HKG_FONT_BUTTON_3		},
	{ HKG_VKEY_UP,		HKG_FONT_BUTTON_UP		},
	{ HKG_VKEY_DOWN,	HKG_FONT_BUTTON_DOWN	},
	{ HKG_VKEY_LEFT,	HKG_FONT_BUTTON_LEFT	},
	{ HKG_VKEY_RIGHT,	HKG_FONT_BUTTON_RIGHT	},
	{ HKG_VKEY_SHIFT,	HKG_FONT_BUTTON_SELECT	},
	{ HKG_VKEY_RETURN,	HKG_FONT_BUTTON_START	},
	{ HKG_VKEY_DELETE,	HKG_FONT_BUTTON_L1		},// also map to left mouse button
	{ HKG_VKEY_END,		HKG_FONT_BUTTON_R1		},// also map to right mouse button
	{ HKG_VKEY_INSERT,	HKG_FONT_BUTTON_L2		},
	{ HKG_VKEY_HOME,	HKG_FONT_BUTTON_R2		}
};

char keyToButtonFont( int k )
{
	int numMappings = (int) HK_COUNT_OF(keyToButtonFontMapping);
	for (int i=0; i<numMappings; i++)
	{
		int from = keyToButtonFontMapping[i][0];
		int to   = keyToButtonFontMapping[i][1];

		if (from == (int)k)
		{
			return char(to);
		}
	}

	return 0;
}

hkDemo::Result MenuDemo::stepCurrentDemo()
{
	hkgPad* gamePad = m_env->m_gamePad;

	m_currentDemo->waitForStepCompletion();

	m_currentDemo->onStepCompleted();

	switch (m_env->m_options->m_replayType)
	{
		case hkDemoFrameworkOptions::REPLAY_RECORD:
			m_replay.recordFrame( m_env, m_timestep );
			// Uncomment to save the replay info every frame (to repro crashes, etc.)
			//m_replay.save( m_env->m_options->m_inputFilename );
			break;

		case hkDemoFrameworkOptions::REPLAY_PLAYBACK:
			m_replay.playbackFrame( m_env, m_timestep );

			break;
	}


	if( m_env->wasButtonPressed( HKG_PAD_START) )  // pause pressed?
	{
		if( m_wantTweak )
		{
			m_wantTweak = false;
			m_paused = false;
		}
		else
		{
			m_paused = !m_paused;

			if (m_paused)
			{
				tickFrame(*m_env, false); // an extra tick to restore the proper dynamic vb
			}
		}
	}
	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_F1) )
	{
		m_helpTimeLeft = ( m_helpTimeLeft == HK_REAL_MAX ) ? 0 : HK_REAL_MAX;
	}

	const bool newTimersGathered = m_newTimersGathered;
	if ( m_paused && m_newTimersGathered && m_savedStreamBegin && (hkMonitorStream::getInstance().getCapacity() != HK_NULL) )
	{
		// Switch current monitor stream for this thread to continue to capture graphics timers
		hkAlgorithm::swap<hkPadSpu<char*> >(m_savedStreamBegin, hkMonitorStream::getInstance().m_start);
		hkAlgorithm::swap(m_savedStreamCurrent, hkMonitorStream::getInstance().m_end);
		hkAlgorithm::swap(m_savedStreamEnd, hkMonitorStream::getInstance().m_capacity);
		hkMonitorStream::getInstance().m_capacityMinus16 = hkMonitorStream::getInstance().m_capacity - 32;
		hkMonitorStream::getInstance().reset();
		m_pausedSimTimers = m_simTimersEnd;
		m_newTimersGathered = false;
	}

	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_TAB) )
	{
		m_wantTweak = !m_wantTweak;
	}

	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_F9) )
	{
		m_gpuStats = !m_gpuStats;
		m_helpTimeLeft = 0.0f;
	}

	bool inSingleStep = false;
	if( m_wantTweak ) // show paused context menu
	{
		hkDemo::Result res = showTweakMenu();
		if (res != DEMO_ERROR )
		{
			return res;
		}
	}
	else if( m_paused ) // show paused context menu
	{
		if( (m_wantMonitors || m_graphManager->isEnabled(GraphManager::TIMER) || m_graphManager->isEnabled(GraphManager::STATS))
			&& m_currentDemo && m_savedStreamBegin)
		{
			extArray<hkTimerData> threadStreams;
			extArray<hkTimerData> spuStreams;

			m_currentDemo->getTimerStreamInfo( threadStreams, spuStreams, 6 );

			// Append saved simulation timers to this threads timers (now have graphics in them)
			hkUlong simTimerSize = m_pausedSimTimers - m_savedStreamBegin;
			if (simTimerSize > 0)
			{
				hkUlong availableSpace = hkMonitorStream::getInstance().m_capacity - hkMonitorStream::getInstance().m_end;
				if ( availableSpace > simTimerSize )
				{
					hkString::memCpy( hkMonitorStream::getInstance().getEnd(), m_savedStreamBegin, (int)simTimerSize );
					threadStreams[0].m_streamEnd = hkMonitorStream::getInstance().m_end + simTimerSize;
				}
			}

			if (m_wantMonitors)
			{
				showMonitorAnalysis(0, threadStreams, spuStreams );
			}

			if ( (m_graphManager->isEnabled(GraphManager::TIMER) || m_graphManager->isEnabled(GraphManager::STATS))
			 && m_savedStreamBegin && m_pausedSimTimers)
			{
				// For this thread, just use the saved physics timers, otherwise the time range will keep getting bigger
				extArray<hkTimerData> threadStreamsNoGraphics;
				threadStreamsNoGraphics = threadStreams;
				threadStreamsNoGraphics[0].m_streamBegin = m_savedStreamBegin;
				threadStreamsNoGraphics[0].m_streamEnd = m_pausedSimTimers;
				stepStatGraph(threadStreamsNoGraphics, spuStreams);
				if( newTimersGathered )
				{
					stepTimerGraph(threadStreamsNoGraphics, spuStreams);
				}
			}

			// Reset just this thread (used for graphics)
			hkMonitorStream::getInstance().reset();
		}
		if ( m_env->wasButtonPressed(HKG_PAD_BUTTON_0) && m_graphManager->isEnabled(GraphManager::MEMORY) && m_currentDemo)
		{
				hkMemorySystem::MemoryStatistics stats;
				hkMemorySystem::getInstance().getMemoryStatistics(stats);
				stepMainMemoryGraph(stats);
		}

		if ( m_graphManager->isEnabled(GraphManager::TRACKER))
		{
			// Display any special tracker info
			showMemoryTracker();
		}

		if( m_gpuStats )
		{
			m_gpuStatsUtil->handleKeys( *m_env );
			showGpuStats();
		}

		hkDemo::Result res = showPausedMenu( m_wantMonitors || m_gpuStats );

		if (res != DEMO_ERROR )
		{
			return res;
		}

		inSingleStep = m_paused;
		if (inSingleStep) // then we won't have ticked the frame fully as where paused
		{
			tickFrame(*m_env, false);
		}
	}

	if( m_gpuStats )
	{
		showGpuStats();
	}

	if( m_currentDemo->getError() != HK_NULL )
	{
		m_env->m_textDisplay->outputText(m_currentDemo->getError(), 20, 20, TEXT_COLOR_CURRENT);
		return DEMO_OK;
	}
	else if(0)
	{
		extStringBuf oss;
		oss.printf("Menu : %c", HKG_FONT_BUTTON_START);
		m_env->m_textDisplay->outputText(oss, 20, 20, 0x00777777);
	}

	//
	// Display camera position & target
	//
	if ( m_wantCameraData )
	{
		int windowHeight = getWindowHeight();
		int dH           = m_env->m_window->getTVDeadZoneH();
		int dV           = m_env->m_window->getTVDeadZoneV();

		float* cameraFrom = m_env->m_window->getViewport(0)->getCamera()->getFromPtr();
		float* cameraTo   = m_env->m_window->getViewport(0)->getCamera()->getToPtr();
		extStringBuf oss;
		oss.printf("Camera position : %.2f %.2f %.2f\nCamera target   : %.2f %.2f %.2f", cameraFrom[0], cameraFrom[1], cameraFrom[2], cameraTo[0], cameraTo[1], cameraTo[2]);
		m_env->m_textDisplay->outputText(oss, 20 + dH, windowHeight - dV - 50, 0xffffffff);
	}

	if(m_helpTimeLeft > 0.0f && !m_wantTweak )
	{
		if( m_helpTimeLeft != HK_REAL_MAX )
		{
			m_helpTimeLeft -= m_timestep;
		}
		int overallindex = hkDemoDatabase::getInstance().findDemo( m_currentPath.cString());
		if( overallindex >= 0 )
		{
			const hkDemoEntry& demo = hkDemoDatabase::getInstance().getDemos()[overallindex];
			extStringBuf help;
			help = (demo.m_details ? demo.m_details : "<No detailed help for this demo>");
			if( m_currentDemo->getOptions() )
			{
				help.appendPrintf("\nPress <Tab> or %c %c to tweak this demo", HKG_FONT_BUTTON_START, HKG_FONT_BUTTON_R1);
			}

			// Descriptions for key-pressed callbacks
			hkInplaceBitField<HKG_KEYBOARD_NUM_VKEYS> keyUsed;

			const hkArrayBase<KeyPressCallbackInfo>* bindingArrays[3] =
			{
				&m_currentDemo->getKeyPressCallbackInfo(),
				&m_currentDemo->getKeyHeldCallbackInfo(),
				&m_currentDemo->getKeyReleasedCallbackInfo()
			};

			const char* bindingHelpStrings[3] = { "press", "held", "released" };

			for (int bindingType = 0; bindingType<3; bindingType++)
			{
				const hkArrayBase<KeyPressCallbackInfo>& keyInfos = *bindingArrays[bindingType];
				hkBool addedShortcutDescription = false;

				if( keyInfos.getSize() && m_helpTimeLeft == HK_REAL_MAX )
				{
					for( int i = keyInfos.getSize()-1; i >= 0; i -= 1 )
					{
						const KeyPressCallbackInfo& k = keyInfos[i];
						char buttonFont = keyToButtonFont(k.vkey);

						if (!buttonFont && !m_env->m_window->getKeyboard().isConnected() )
						{
							// No keyboard attached, and no button. Don't bother displaying.
							continue;
						}

						if( k.description != HK_NULL && keyUsed.get(k.vkey) == false )
						{
							// Only append e.g. "Key held shortcuts" if there's a valid description
							// There might not be, if we have multiple things bound to the same key
							if (!addedShortcutDescription)
							{
								help.appendPrintf("\n\nKey %s shortcuts:\n", bindingHelpStrings[bindingType] );
								addedShortcutDescription = true;
							}

							keyUsed.set(k.vkey);
							if (buttonFont && !m_env->m_window->getKeyboard().isConnected() )
							{
								help.appendPrintf("\n\t%c   %s", buttonFont, k.description );
							}
							else
							{
								help.appendPrintf("\n\t%8s %s", hkgKeyboard::nameOfVkey(HKG_KEYBOARD_VKEY(k.vkey)), k.description );
							}
						}
					}
				}
			}


			int dH = m_env->m_window->getTVDeadZoneH();
			int dV = m_env->m_window->getTVDeadZoneV();
			const int winWidth = m_env->m_window->getSecondaryViewport()? m_env->m_window->getSecondaryViewport()->getWidth() : m_env->m_window->getWidth();
			m_env->m_textDisplay->outputTextWithWrapping(help, 20, 20 + dV, winWidth - 40 - (2 * dH), -1, TEXT_COLOR_NORMAL);
			//m_env->m_textDisplay->outputText(helpBuf.begin(), 20 + dH, 20 + dV, TEXT_COLOR_NORMAL);
		}
	}

	if (!m_childDemoJustCreated)
	{
		m_env->m_displayHandler->clear(); // clear debug display before a potential world step
	}
	m_childDemoJustCreated = false;


	// Only gather stats if you have to (so if showing graphs, gathering stats, or if VDB is enabled)
	//bool vdbEnbabledAndRunning = (stepResult != hkDemo::DEMO_PAUSED) && m_currentDemo->visualDebuggerEnabled();
	if ( m_currentDemo->visualDebuggerEnabled() || m_wantMonitors ||  m_graphManager->isEnabled(GraphManager::TIMER)
	|| m_graphManager->isEnabled(GraphManager::MEMORY) || m_graphManager->isEnabled(GraphManager::STATS) || m_performanceCounterUtility->isEnabled() )
	{
		extArray<hkTimerData> threadStreams;
		extArray<hkTimerData> spuStreams;

		m_currentDemo->getTimerStreamInfo( threadStreams, spuStreams, 6 );

		if ( m_currentDemo->visualDebuggerEnabled() )
		{
			m_currentDemo->addTimersToVdb( threadStreams, spuStreams );

			stepVisualDebugger();
		}

		if ( !m_paused )
		{
			// On screen monitors
			if( m_wantMonitors )
			{
				showMonitorAnalysis( 0, threadStreams, spuStreams );
			}

			if ( m_graphManager->isEnabled(GraphManager::MEMORY))
			{
				hkMemorySystem::MemoryStatistics stats;
				hkMemorySystem::getInstance().getMemoryStatistics(stats);
				stepMainMemoryGraph(stats);
			}

			if ( m_graphManager-> isEnabled(GraphManager::TIMER))
			{
				stepTimerGraph( threadStreams, spuStreams );
			}

			if ( m_graphManager->isEnabled(GraphManager::STATS))
			{
				stepStatGraph( threadStreams, spuStreams );
			}
		}

		if( m_performanceCounterUtility->isEnabled() )
		{
			for ( int i = 0; i < threadStreams.getSize(); ++i )
			{
				m_performanceCounterUtility->captureExternalFrame( threadStreams[i].m_streamBegin, threadStreams[i].m_streamEnd, i );
			}
			for ( int i = 0; i < spuStreams.getSize(); ++i )
			{
				m_performanceCounterUtility->captureExternalFrame( spuStreams[i].m_streamBegin, spuStreams[i].m_streamEnd, i + threadStreams.getSize() );
			}
		}
	}

	if (m_graphManager->isEnabled(GraphManager::TRACKER))
	{
		// Show tracker graph
		showMemoryTracker();
	}

	if (m_env->m_mousePickingEnabled)
	{
		// assumes graph is shown from 0,0
		const hkgMouse& mouse = m_env->m_window->getMouse();
		int px = mouse.getPosX();
		int py = mouse.getPosY();

		bool inStatBounds = m_graphManager->inStatGraphBounds(px, py);

		if(m_paused)
		{
			if( m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_SHIFT) )
			{
				//if paused, any picking functions move under the shift+pad0 combination
				//let the demo itself decide how to handle consecutive picking instructions
				//instead of basing it on m_mouseactive.
				if( m_mouseActive )
				{
					m_currentDemo->mouseUp();
				}
				m_currentDemo->mouseDown();
				m_mouseActive = true;
			}
		}
		else if( !m_mouseActive &&
			(   m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_SPACE) ||
				( inStatBounds && mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON)) ||
				m_env->wasButtonPressed( HKG_PAD_BUTTON_0) // == Space in pseudo pad on PC, so on PC button 0 is used up anyway..
			  ) )
		{
			// do any other 2D Menu picking here
			if (inStatBounds)
			{
				bool hasAction =m_graphManager->manageUserAction(px, py);
				if (!hasAction)
				{
					m_mouseStatZoomOriginX = px;
					m_mouseStatZoomOriginY = py;
					m_mouseInGraphZoomMode = true;

					// disable the hkg camera mode while we zoom
					if (mouse.wasButtonPressed(HKG_MOUSE_LEFT_BUTTON))
					{
						m_viewportMouseEventBackup = m_env->m_window->getCurrentViewport()->getAcceptsMouseEvents();
						m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(false);
					}
				}
				else
				{
					m_mouseInGraphZoomMode = false;
				}
			}
			else
			{
				m_currentDemo->mouseDown(); // pick object under cursor
				m_mouseActive = true;
			}

			m_mousePadTrackedButton = m_env->wasButtonPressed( HKG_PAD_BUTTON_0)? HKG_PAD_BUTTON_0: -1;
		}
		else if (m_mouseInGraphZoomMode || m_mouseActive)
		{
			if( m_env->m_window->getKeyboard().wasKeyReleased(HKG_VKEY_SPACE) ||
				(m_mouseInGraphZoomMode && mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON)) ||
				( (m_mousePadTrackedButton != -1) && (gamePad->wasButtonReleased((HKG_PAD_BUTTON)m_mousePadTrackedButton)) ) )
			{
				if (m_mouseInGraphZoomMode)
				{
					if (mouse.wasButtonReleased(HKG_MOUSE_LEFT_BUTTON))
					{
						m_env->m_window->getCurrentViewport()->setAcceptsMouseEvents(m_viewportMouseEventBackup);
					}

					m_mouseInGraphZoomMode = false;
					if (inStatBounds)
					{

						m_graphManager->mouseZoom(m_mouseStatZoomOriginX,m_mouseStatZoomOriginY,  px, py);
					}
				}
				else // m_mouseActive
				{
					m_currentDemo->mouseUp();
				}

				m_mouseActive = false;
				m_mousePadTrackedButton = -1;
			}
			else if (m_mouseActive)
			{
				m_currentDemo->mouseDrag();
			}
		}
	}

	// Reset timer streams prior to stepping
	m_currentDemo->resetTimerStreams();

	// Start the counter for the next step
	m_performanceCounterUtility->startCounter();

	m_newTimersGathered = true;

	//
	//	Step the demo
	//
	Result stepResult;
	{
		if (m_env->m_options->m_fpuExceptionFlags)
		{
			hkPushFPUState( m_env->m_options->m_fpuExceptionFlags );
		}

		m_currentDemo->m_env->checkInputDeterminism();
		stepResult = m_currentDemo->stepDemo();

		if ( m_env->m_options->m_fpuExceptionFlags )
		{
			hkPopFPUState();
		}
		m_frameCounter ++;
	}

#if defined (HK_ENABLE_DETERMINISM_CHECKS)
	if ( m_frameCounter >= m_currentDemo->m_bootstrapIterations )
	{
		if ( m_determinismCheckRunIndex == 0)
		{
			stopCurrentDemo();
			m_determinismCheckRunIndex = 1;
			startCurrentDemo();
		}
		else if ( m_determinismCheckRunIndex == 1)
		{
			m_determinismCheckRunIndex = 0;
			stepResult = DEMO_STOP;
		}
	}
#endif

	m_simTimersEnd = hkMonitorStream::getInstance().getEnd();

	//
	// Capture the frame
	//
	m_performanceCounterUtility->stopCounter();

	PostStepActions actions = m_currentDemo->getPostStepActions();

	if (actions & DEMO_PSA_CLEAR_TIMERS)
	{
		// Clear timers
		m_activeNode = HK_NULL;
		if (m_statsTree)
		{
			delete m_statsTree;
		}
		m_statsTree = HK_NULL;

		m_graphManager->toggleGraph(GraphManager::TIMER , false);
		m_graphManager->toggleGraph(GraphManager::STATS , false);
	}

	switch( stepResult )
	{
		case DEMO_OK:
		{
			if (inSingleStep) return DEMO_PAUSED;
			break;
		}


		case DEMO_PAUSED:
		{
			m_paused = true;
			return DEMO_PAUSED;
		}
		case DEMO_STOP:
		{
			stopCurrentDemo();

			if (m_env->m_repetitionIndex == (m_env->m_options->m_numRepetitions - 1))
			{
				m_env->m_repetitionIndex = 0;

				// if we were running a single demo, given by -g demo
				// then we should stop now
				if( m_env->m_options->m_defaultDemo )
				{
					return DEMO_STOP;
				}
			}
			else
			{
				m_env->m_repetitionIndex++;
				startCurrentDemo();
			}

			break;
		}
		case DEMO_RESTART:
		{
			stopCurrentDemo();
			startCurrentDemo();
			break;
		}
		case DEMO_ERROR:
		{
			break;
		}
		default:
		{
			HK_ASSERT2(0x194d9b84, 0, "Should be unreachable");
		}
	}



	return DEMO_OK;
}

void MenuDemo::writeSimpleMemoryStatistics()
{
#if  defined(HK_MEMORY_TRACKER_ENABLE)

#	define HK_REPORT_PREFIX "mem_"

	if( hkMemoryTracker* tracker = hkMemoryTracker::getInstancePtr() )
	{
		hkTrackerScanSnapshot* scanSnapshot = hkTrackerSnapshotUtil::createSnapshot();

		{
			hkOstream stream(HK_REPORT_PREFIX "hkVdbStreamReportUtil_generateReport.txt");
			hkVdbStreamReportUtil::generateReport( scanSnapshot, stream );
		}

		{
			hkOstream stream(HK_REPORT_PREFIX "hkProductReportUtil_reportSummary.txt");
			hkProductReportUtil::reportSummary(scanSnapshot, HK_NULL, stream);
		}

		{
			hkOstream stream(HK_REPORT_PREFIX "hkCategoryReportUtil_report.txt");
			hkCategoryReportUtil::report(scanSnapshot, HK_NULL, stream);
		}


		scanSnapshot->removeReference();
	}
#endif
}

void MenuDemo::stopCurrentDemo()
{
	switch (m_env->m_options->m_replayType)
	{
		case hkDemoFrameworkOptions::REPLAY_RECORD:
			m_replay.endRecordingAndSave(m_env->m_options->m_inputFilename);
			break;

		case hkDemoFrameworkOptions::REPLAY_PLAYBACK:
			m_replay.endPlayback(m_env);
			break;
	}

	m_performanceCounterUtility->saveFileAndReset();

	// memory dump
	if ( m_env->m_options->m_saveMemoryStatistics)
	{
		hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );
		writeSimpleMemoryStatistics();
		hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );
	}

	m_env->m_displayHandler->clear();

	hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_AUTO );

	if (m_env->m_options->m_fpuExceptionFlags)
	{
		hkPushFPUState( m_env->m_options->m_fpuExceptionFlags );
	}

	m_currentDemo->preDeleteDemo();
	delete m_currentDemo;
	m_currentDemo = HK_NULL;

	// Restore the demo options
	*m_env->m_options = m_originalOptions;

	if ( m_env->m_options->m_fpuExceptionFlags )
	{
		hkPopFPUState();
	}
	HK_ON_DETERMINISM_CHECKS_ENABLED(
		if ( m_determinismCheckRunIndex >= 0 )
		{
			hkCheckDeterminismUtil::getInstance().finish();
		}
	);

	hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );

	// reset all monitors
	{
		m_activeNode = HK_NULL;
		if (m_statsTree)
		{
			delete m_statsTree;
		}
		m_statsTree = HK_NULL;

		m_graphManager->toggleGraph(GraphManager::TIMER, false);
		m_graphManager->toggleGraph(GraphManager::STATS, false);
		m_graphManager->toggleGraph(GraphManager::TRACKER, false);
		m_graphManager->toggleGraph(GraphManager::MEMORY, false);
	}

	hkgVec3Copy(m_menuImageBackground, MENU_BACKGROUND_COLOR );
	m_env->m_window->setClearColor( m_menuImageBackground );
	m_env->m_window->setWantDrawHavokLogo(false);
}

void MenuDemo::startCurrentDemo()
{
	{
		hkTextDisplay& t = *m_env->m_textDisplay;
		int x = int( 0.5f * (m_env->m_window->getWidth() - 14/*width of msg in chars*/ * t.getFont()->getCharWidth() ) );
		int y = int( 0.5f * (getWindowHeight() - 3/*lines in msg*/ * t.getFont()->getCharHeight() ) );
		t.wipeText();
		t.outputText( " \n  Loading...  \n " , x, y);
		debugRenderNow().wait(false);
	}
	switch (m_env->m_options->m_replayType)
	{
		case hkDemoFrameworkOptions::REPLAY_RECORD:
			m_replay.startRecording();
			break;

		case hkDemoFrameworkOptions::REPLAY_PLAYBACK:
			m_replay.startPlayback(m_env->m_options->m_inputFilename, m_env);
			break;
	}

	m_env->m_window->getContext()->lock();

	if (m_env->m_options->m_printList)
	{
		hkDemoDatabase::getInstance().dbgPrintAllDemos();
	}	


	m_childDemoJustCreated = true;
	// Find the index of the default or current demo
	const char* demoName = ( m_defaultDemo != HK_NULL )? m_defaultDemo: m_currentPath.cString();
	int index = hkDemoDatabase::getInstance().findDemo( demoName );

	char buf[256];
	hkString::snprintf(buf, 256, "Starting demo index %i (use -g %i to run this demo)\n", index, index);
	HK_REPORT( buf );

	m_env->m_window->setClearColor( DEMO_BACKGROUND_COLOR );
	m_env->m_window->setWantDrawHavokLogo(true);

	if ( index >= 0)
	{
		hkReferencedObject::lockAll();

		// Create the demo
		const hkDemoEntry& demo = hkDemoDatabase::getInstance().getDemos()[index];
		m_env->m_demoPath = demo.m_demoPath;
		m_env->m_menuPath = demo.m_menuPath;
		m_env->m_resourcePath = demo.m_resourcePath;
		m_env->m_variantId = demo.m_variantId;
		m_env->m_demoTypeFlags = demo.m_demoTypeFlags;

		m_currentPath = demo.m_menuPath;

#if defined (HK_ENABLE_DETERMINISM_CHECKS)
		if ( hkString::strStr( m_env->m_demoPath, "/Bootstrap/"))
		{
			m_determinismCheckRunIndex = -1;
			hkCheckDeterminismUtil::destroyInstance();
		}
		else if ( m_determinismCheckRunIndex == -1)
		{
			m_determinismCheckRunIndex = 0;
			hkCheckDeterminismUtil::createInstance();
		}


		if ( m_determinismCheckRunIndex == 0)
		{
			hkCheckDeterminismUtil::getInstance().startWriteMode();
		}
		else if ( m_determinismCheckRunIndex == 1)
		{
			hkCheckDeterminismUtil::getInstance().startCheckMode();
		}
#endif

		if (m_env->m_options->m_fpuExceptionFlags)
		{
			hkPushFPUState( m_env->m_options->m_fpuExceptionFlags );
		}

		//
		//	Build the demo
		//
		m_currentDemo = (*demo.m_func)(m_env);
		m_currentDemo->postConstruct();

		if ( m_env->m_options->m_fpuExceptionFlags )
		{
			hkPopFPUState();
		}

		m_currentDemo->setDemoName(m_currentPath);
		m_helpTimeLeft = m_helpTimeMax;

		m_performanceCounterUtility->setDemoName( m_env->m_menuPath.cString() );

		hkReferencedObject::unlockAll();

		extArray<hkTimerData> threadStreams;
		extArray<hkTimerData> spuStreams;

		m_currentDemo->getTimerStreamInfo( threadStreams, spuStreams, 6 );

		m_performanceCounterUtility->resetNumThreads( threadStreams.getSize(), spuStreams.getSize() );

		if( demo.m_demoTypeFlags & HK_DEMO_TYPE_SHOW_PERFORMANCE_GRAPHS )
		{
			cycleTimerGraph();
		}

		if (demo.m_demoTypeFlags & HK_DEMO_TYPE_SHOW_MEMORY_GRAPHS)
		{
			toggleMemoryGraphs();
		}
		// Create / load the demo scene
		m_currentDemo->createScene();

		m_frameCounter = 0;
		m_paused = false;
	}
	else
	{
		// Otherwise show the error demo
		m_currentDemo = new ErrorDemo(m_env, demoName);
	}

	m_env->m_window->getContext()->unlock();

}


hkDemo::Result MenuDemo::stepMenuDemo()
{
	const extArray<hkDemoEntry>& alldemos = hkDemoDatabase::getInstance().getDemos();
	int startx = 20 + m_env->m_window->getTVDeadZoneH();
	int starty = m_env->m_window->getTVDeadZoneV();
	int stepx = 13;

	int stepy = 20;
	hkTextDisplay* textDisplay = m_env->m_textDisplay;
	if( m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_OEM_2_FORWARDSLASH) )
	{
		m_searchString += "/";
		m_searchIndex = 0;
	}
	if( m_searchString.getLength() )
	{
		// keyboard accelerators
		const hkgKeyboard& kb = m_env->m_window->getKeyboard();
		const int ranges[][2] = { {'A','Z'}, {'0','9'}, {' ',' '} };
		for( int rangeIndex = 0; rangeIndex < int(HK_COUNT_OF(ranges)); ++rangeIndex )
		{
			for( int i = ranges[rangeIndex][0]; i <= ranges[rangeIndex][1]; ++i )
			{
				if( kb.wasKeyPressed( HKG_KEYBOARD_VKEY(i) ) )
				{
					char s[2]; s[0] = char(i); s[1] = 0;
					if( ! kb.getKeyState(HKG_VKEY_SHIFT) )
					{
						s[0] = hkString::toLower(s[0]);
					}
					m_searchString += s;
				}
			}
		}
		m_searchString.replace("  "," ");
		if( m_searchString.getLength() == 2 && m_searchString[1] == ' ' )
		{
			m_searchString.chompEnd(1);
		}

		starty += stepy;
		// show the current search string
		{
			extStringBuf search("(", m_searchString.cString()+1 );
			search.replace(" ", ") (");
			search.append( ((int(m_stopwatch.getElapsedSeconds()*2))&1) ? "|)" : " )");
			extStringBuf displayString; displayString.printf("Searching for: %s\n<Tab> to cancel, <Space> for multiple search terms", search.cString());
			textDisplay->outputText( displayString, startx, starty);
		}
		starty += stepy * 2;

		extArray<const hkDemoEntry*> entries;
		entries.reserve( (getWindowHeight() - 2*starty) / stepy );
		for (int demoIndex = 0; demoIndex < alldemos.getSize() && entries.getSize() < entries.getCapacity(); demoIndex++)
		{
			extStringBuf hayStack(alldemos[demoIndex].m_menuPath); canonicalise(hayStack);
			extStringBuf needle(m_searchString.cString()+1); // +1 skip leading '/'
			canonicalise(needle);
			extArray<const char*>::Temp bits;
			needle.split(' ', bits);
			hkBool32 match = true; // all bits have to match
			for( int i = 0; i < bits.getSize(); ++i )
			{
				if( hayStack.indexOf(bits[i]) == -1 )
				{
					match = false;
					break;
				}
			}
			if( match )
			{
				entries.pushBack( alldemos.begin() + demoIndex );
			}
		}
		m_searchIndex = hkMath::min2( entries.getSize(), m_searchIndex );
		if( entries.getSize() == 0 )
		{
			starty += stepy;
			textDisplay->outputText( "No demos found", startx, starty, TEXT_COLOR_NORMAL);
		}
		for (int i = 0; i < entries.getSize(); i++)
		{
			starty += stepy;
			hkColor::Argb color = (i == m_searchIndex) ? TEXT_COLOR_CURRENT : TEXT_COLOR_NORMAL;
			textDisplay->outputText( entries[i]->m_menuPath, startx, starty, color);
		}
		if( entries.getSize() == entries.getCapacity() )
		{
			starty += stepy;
			textDisplay->outputText( "...", startx, starty);
		}

		int numEntries = entries.getSize();

		if (m_env->wasButtonPressed( HKG_PAD_DPAD_UP) && numEntries)
		{
			m_searchIndex = (m_searchIndex + numEntries - 1) % numEntries;
		}
		else if (m_env->wasButtonPressed( HKG_PAD_DPAD_DOWN) && numEntries )
		{
			m_searchIndex = (m_searchIndex + 1) % numEntries;
		}
		else if (m_env->wasButtonPressed( HKG_PAD_START))
		{
			if( m_searchIndex < entries.getSize() )
			{
				m_currentPath = entries[m_searchIndex]->m_menuPath;
				hkSetLastDemo(m_currentPath.cString());
#ifndef HK_DISABLE_BOOTSTRAPPER
				if (! hkString::strStr( m_currentPath.cString(), "ootstrap") )
					hkSetLastDemoBootstrap(m_currentPath.cString());
#endif
				startCurrentDemo();
			}
		}
		else if (kb.wasKeyPressed(HKG_VKEY_TAB)) //cancel
		{
			m_searchString = "";
		}
		if( kb.wasKeyPressed( HKG_VKEY_BACK ) || m_env->wasButtonPressed( HKG_PAD_DPAD_LEFT) )
		{
			m_searchString.slice(0, m_searchString.getLength()-1);
		}
		if( kb.getKeyState( HKG_VKEY_CONTROL ) && kb.wasKeyPressed( HKG_KEYBOARD_VKEY('W') ) )
		{
			int sep = m_searchString.lastIndexOf(' ');
			if( sep > -1 )
			{
				m_searchString.chompEnd( m_searchString.getLength() - sep );
			}
			else
			{
				m_searchString = "";
			}
		}
		if( kb.getKeyState( HKG_VKEY_CONTROL ) && kb.wasKeyPressed( HKG_KEYBOARD_VKEY('C') ) )
		{
			m_searchString = "";
		}


		return DEMO_OK;
	}

	// get the entries for this level
	extArray<Entry> entries;
	int selectedIndex = 0;
	{
		for (int tr=0; tr<2; tr++)
		{
			hkStorageStringMap<int, extContainerAllocator> alreadyHadIt;

			// initialize the path
			extStringBuf path; path = m_currentPath;
			path.pathDirname();
			if(path.getLength())
			{
				path += "/";
			}
			int skip = path.getLength();

			for (int i = 0; i < alldemos.getSize(); i++)
			{
				extStringBuf n(alldemos[i].m_menuPath);
				if( n.startsWith(path) && !n.endsWith("Menu") )
				{
					int slash = n.indexOf('/', skip);
					int len;
					const char* help = HK_NULL;

					if(slash >= 0)
					{
						len = slash - skip;
					}
					else
					{
						len = n.getLength() - skip;
						help = alldemos[i].m_help;
					}
					extStringBuf name( n.cString()+skip, len);

					if( alreadyHadIt.hasKey(name) == false)
					{
						extStringBuf s(path,name);
						Entry e(name, s, help, i);
						entries.pushBack(e);
						alreadyHadIt.insert(name, 1);
					}
				}
			}
			if ( entries.isEmpty() )
			{
				m_currentPath = "";
				continue;
			}

			for(int k=0; k<entries.getSize(); ++k)
			{
				if ( m_currentPath == entries[k].m_fullPath.cString())
				{
					selectedIndex = k;
					break;
				}
			}
			HK_ASSERT(0x59ae7dff, entries.getSize() );
			break;
		}
	}

	// Show Build Number
	{
		const int h = (m_env->m_window->getSecondaryViewport() ? m_env->m_window->getSecondaryViewport()->getHeight() : getWindowHeight()) - m_env->m_window->getTVDeadZoneV();
		const int dx =  m_env->m_window->getTVDeadZoneH();

		extStringBuf buf;
#ifdef HK_DEBUG
		buf.printf("Havok %s - DEBUG", HAVOK_SDK_VERSION_NUM_STRING);
#else
		buf.printf("Havok %s", HAVOK_SDK_VERSION_NUM_STRING);
#endif

		if (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_CONSOLE)
		{
			buf.appendJoin(" - Renderer: ", hkgSystem::getRendererString());
		}

		textDisplay->outputText( buf, 20+dx, h-40, TEXT_COLOR_NORMAL, 1);

	}

	// show status and location
	{
		starty += stepy;
		if( getStatus() )
		{
			textDisplay->outputText(getStatus(), startx, starty);
		}
		starty += stepy;

		extStringBuf loc = m_currentPath;
		int slash;
		while( (slash = loc.indexOf('/')) != -1)
		{
			extStringBuf label;
			label.set(loc, slash+1);

			loc.chompStart(slash+1);
			starty += stepy;
			textDisplay->outputText(insertSpaces(label), startx, starty, TEXT_COLOR_NORMAL);
			startx += stepx;
		}
	}

	// show current entries
	{
		for(int i = 0; i < entries.getSize(); ++i)
		{
			starty += stepy;
			extStringBuf name(entries[i].m_name);

			insertSpaces(name);

			bool isDir = entries[i].isDir();
			hkColor::Argb color = (i == selectedIndex) ? TEXT_COLOR_CURRENT : TEXT_COLOR_NORMAL;
			extStringBuf str( isDir? "[" : " ", name, isDir? "]" : "");
			textDisplay->outputText(str, startx, starty, color);
		}
	}


	// process user events
	{
		const int numEntries = entries.getSize();

		// next item
		if (	m_env->wasButtonPressed( HKG_PAD_DPAD_UP)
			||	m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_UP))
		{
			selectedIndex = (selectedIndex + numEntries - 1) % numEntries;
		}
		// prev item
		else if (	m_env->wasButtonPressed( HKG_PAD_DPAD_DOWN)
				 ||	m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_DOWN))
		{
			selectedIndex = (selectedIndex + 1) % numEntries;
		}
		// up a level
		else if(	m_env->wasButtonPressed( HKG_PAD_DPAD_LEFT)
				||	m_env->wasButtonPressed( HKG_PAD_BUTTON_1)
				||	m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_BACK) )
		{
			selectedIndex = -1;
			addStatus("");
		}
		// down a level
		else if( m_env->wasButtonPressed( HKG_PAD_DPAD_RIGHT) )
		{
			if( entries[selectedIndex].isDir() )
			{
				m_currentPath += "/.";
				selectedIndex = 0;
				addStatus("");
			}
		}
		// start a demo, or go down a level
		else if (	m_env->wasButtonPressed( HKG_PAD_BUTTON_0)
				 || m_env->wasButtonPressed( HKG_PAD_START)
				 ||	m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_RETURN) )
		{
			if( entries[selectedIndex].isDir() == false)
			{
				hkSetLastDemo(m_currentPath.cString());
#ifndef HK_DISABLE_BOOTSTRAPPER
				if (! hkString::strStr( m_currentPath.cString(), "ootstrap") )
					hkSetLastDemoBootstrap(m_currentPath.cString());
#endif
				startCurrentDemo();
			}
			else // directory
			{
				m_currentPath += "/.";
				selectedIndex = 0;
				addStatus("");
			}
		}
		// cancel the game
		else if (m_env->wasButtonPressed( HKG_PAD_BUTTON_1)) //cancel
		{
			// can't cancel the menu demo
		}
		// show help for current demo
		else
		{
			if(entries[selectedIndex].isDir() == false)
			{
				extStringBuf help = entries[selectedIndex].m_help;
				int end = help.indexOf('\n');
				if(end > 0)
				{
					help.slice(0,end);
				}
				addStatus(help);
			}
			else
			{
				addStatus("");
			}

			// keyboard accelerators
			const hkgKeyboard& kb = m_env->m_window->getKeyboard();
			for( int i = 0; i < entries.getSize(); ++i )
			{
				int j = (i + selectedIndex + 1) % entries.getSize();
				if(entries[j].m_name.getLength() > 0)
				{
					int c = hkString::toUpper( entries[j].m_name[0] );
					if( kb.wasKeyPressed( HKG_KEYBOARD_VKEY(c) ) )
					{
						selectedIndex = j;
						break;
					}
				}
			}
		}
	}

	// update our current demo
	{
		int lastSlash = m_currentPath.lastIndexOf('/');
		if( lastSlash >= 0)
		{
			m_currentPath.slice(0, lastSlash); // empty string if / not found
		}
		else
		{
			m_currentPath = "";
		}

		if ( selectedIndex >= 0)
		{
			if ( m_currentPath.getLength() )
			{
				m_currentPath += "/";
			}
			m_currentPath += entries[selectedIndex].m_name;
		}
	}

	// Can change current menu background color here if you like



	return DEMO_OK;
}


void MenuDemo::addStatus(const char* s)
{
	if(s && s[0])
	{
		m_statusLine = s;
	}
	else
	{
		m_statusLine.printf("Use %c%c%c%c to navigate and %c or %c to start.",
		HKG_FONT_BUTTON_UP, HKG_FONT_BUTTON_DOWN, HKG_FONT_BUTTON_LEFT,	HKG_FONT_BUTTON_RIGHT,
		HKG_FONT_BUTTON_0, HKG_FONT_BUTTON_START);
	}
}

const char* MenuDemo::getStatus()
{
	return m_statusLine;
}

void MenuDemo::showGpuStats()
{
	const hkgDisplayContext *context = m_env->m_displayHandler->getContext();
	if ( !context )
	{
		return;
	}

	hkStringBuf str;
	m_gpuStatsUtil->display( *context, str );

	int x = m_env->m_window->getTVDeadZoneH() + 16;
	int y = m_env->m_window->getTVDeadZoneV() + 16;

	m_env->m_textDisplay->outputText(str, x, y, 0xffffffff);
	m_env->m_textDisplay->outputText(str, x-1, y+1, 0xff000000);
}


#include <Demos/DemoCommon/DemoFramework/hkTestDemo.h>

#if defined(HK_PERFORMANCE_TESTS) && !defined(MENU_DEMO_TESTS_REGISTERED) // todo: destruction preview includes this file - break apart more cleanly

#include <Demos/DemoCommon/DemoFramework/hkPerformanceTestDemo.h> 
static hkDemo* HK_CALL hkCreatePerformanceTestDemo( hkDemoEnvironment* env )
{
	return new PerformanceTestDemo(env);
}

void HK_CALL registerPerformanceTests( extArray<extStringPtr>& extraNames, extArray<hkDemoEntryRegister*>& allocedDemos)
{
	// extraDemos keeps track of all extra items which are registered at the end.
	extArray<hkDemoEntryRegister*> extraDemos;

	//
	// Go through the test database and add new demos for each test
	// These ones are dynamically allocated
	//
	{
		for( int ti = 0; hkPerformanceTestDatabase[ti] != HK_NULL; ti += 1 )
		{
			const hkPerformanceTestEntry* testEntry = hkPerformanceTestDatabase[ti];

			extStringBuf path(testEntry->m_path, "/Magic.cpp");
			path.replace('\\','/');

			extraNames.pushBack(path.cString());

			hkDemoEntryRegister* registeredDemo = new hkDemoEntryRegister(
				hkCreatePerformanceTestDemo, HK_DEMO_TYPE_TEST, extraNames.back(), -1, HK_NULL, "", "");
			extraDemos.pushBack(registeredDemo);
			allocedDemos.pushBack(registeredDemo);
		}
	}
	extStringBuf prefix = "UnitTest/";
	{
		// Special case 'All tests'
		extStringPtr& l_path = extraNames.expandOne();
		l_path.printf("%sAll PerformanceTests/Magic.cpp", prefix.cString() );
		hkDemoEntryRegister* registeredDemo = new hkDemoEntryRegister(
			hkCreatePerformanceTestDemo, HK_DEMO_TYPE_TEST, l_path, -1, HK_NULL, "Run All Performance Tests", "");
		extraDemos.pushBack(registeredDemo);
		allocedDemos.pushBack( registeredDemo );
	}
}
#endif // HK_PERFORMANCE_TESTS && !MENU_DEMO_TESTS_REGISTERED

#if defined(HK_UNIT_TESTS) && !defined(MENU_DEMO_TESTS_REGISTERED)
static hkDemo* HK_CALL hkCreateUnitTestDemo( hkDemoEnvironment* env )
{
	return new TestDemo(env);
}
static hkDemo* HK_CALL hkCreateUnitTestDemoWithAutoRecovery( hkDemoEnvironment* env )
{
	return new TestDemo(env, true);
}

void HK_CALL registerUnitTests( extArray<extStringPtr>& extraNames, extArray<hkDemoEntryRegister*>& allocedDemos)
{
	// extraDemos keeps track of all extra items which are registered at the end.
	extArray<hkDemoEntryRegister*> extraDemos;

	//
	// Go through the test database and add new demos for each test
	// These ones are dynamically allocated
	//
	{
		for( int ti = 0; hkUnitTestDatabase[ti] != HK_NULL; ti += 1 )
		{
			const hkTestEntry* testEntry = hkUnitTestDatabase[ti];

			extStringBuf path(testEntry->m_path, "/Magic.cpp");
			path.pathNormalize();

			extraNames.pushBack(path.cString());

			hkDemoEntryRegister* registeredDemo = new hkDemoEntryRegister(
				hkCreateUnitTestDemo, HK_DEMO_TYPE_TEST, extraNames.back(), -1, HK_NULL, "", "", true);
			extraDemos.pushBack(registeredDemo);
			allocedDemos.pushBack(registeredDemo);
		}
	}
	//
	//	Extract the categories from each test game and make them a new game as well
	//
	{
		if( extraNames.getSize() != 0 )
		{
			// Push back each category as a separate game too
			extArray<const char*> categories;
			for( int ti = 0; hkUnitTestDatabase[ti] != HK_NULL; ti += 1 )
			{
				const hkTestEntry* testEntry = hkUnitTestDatabase[ti];
				if (categories.indexOf(testEntry->m_category) == -1)
				{
					categories.pushBack(testEntry->m_category);
				}
			}


			extStringBuf prefix = "UnitTest/";

			for (int i=0; i < categories.getSize(); i++)
			{
				extStringPtr& l_path = extraNames.expandOne();
				l_path.printf("%s%s/Magic.cpp", prefix.cString(), categories[i] );

				hkDemoEntryRegister* registeredDemo = new hkDemoEntryRegister(
					hkCreateUnitTestDemoWithAutoRecovery, HK_DEMO_TYPE_TEST, l_path, -1, HK_NULL, "Run Test", "");
				extraDemos.pushBack(registeredDemo);
				allocedDemos.pushBack( registeredDemo );
			}

			{
				// Special case 'All tests'
				extStringPtr& l_path = extraNames.expandOne();
				l_path.printf("%sAll Tests/Magic.cpp", prefix.cString() );
				hkDemoEntryRegister* registeredDemo = new hkDemoEntryRegister(
					hkCreateUnitTestDemoWithAutoRecovery, HK_DEMO_TYPE_TEST, l_path, -1, HK_NULL, "Run All Unit Tests", "");
				extraDemos.pushBack(registeredDemo);
				allocedDemos.pushBack( registeredDemo );
			}
		}
	}
}
#endif // HK_UNIT_TEST && !MENU_DEMO_TESTS_REGISTERED

//
// Graph Manager
//

GraphManager::GraphManager( hkDemoEnvironment* env)
 :  m_statGraph(0),
 m_trackerGraph(0),
 m_timerGraph(0),
 m_showMemoryGraphs(false),
 m_env(env)
 {
 }

void GraphManager::renderGraphs( int origZoomX, int origZoomY, bool useCurrentViewport, bool mouseOverlay )
{
	int mouseX = m_env->m_window->getMouse().getPosX();
	int mouseY = m_env->m_window->getMouse().getPosY();

	if (isEnabled(TIMER))
	{
		m_timerGraph->render( m_env->m_options->m_statBarX, m_env->m_options->m_statBarY, useCurrentViewport);
	}

	if (isEnabled(STATS))
	{
		m_statGraph->display( m_env->m_options->m_statBarX, m_env->m_options->m_statBarY, useCurrentViewport);

		if (mouseOverlay)
		{
			if ( (mouseX >= m_env->m_options->m_statBarX) && (mouseX < (m_env->m_options->m_statBarX + m_statGraph->getDisplayWidth())) &&
				(mouseY >= m_env->m_options->m_statBarY) && (mouseY < (m_env->m_options->m_statBarY + m_statGraph->getDisplayHeight())) )
			{
				m_statGraph->displayZoomExtents( origZoomX, origZoomY , mouseX, mouseY, useCurrentViewport);
			}
		}
	}

	if (isEnabled(MEMORY))
	{
		int numGraphs = m_memoryGraphs.getSize();
		int width = m_env->m_window->getWidth()/numGraphs;
		int x = 0;
		int y = m_env->m_window->getHeight()- ( m_env->m_window->getHeight()/m_env->m_options->m_statBarScreenFraction);
		for (int i = 0 ; i < numGraphs; ++i)
		{
			m_memoryGraphs[i]->render(x, y, useCurrentViewport);
			x +=width;
		}
	}

	if (isEnabled(TRACKER))
	{
		m_trackerGraph->display( m_env->m_options->m_statBarX, m_env->m_options->m_statBarY, useCurrentViewport);

		if (mouseOverlay)
		{
			if ( (mouseX >= m_env->m_options->m_statBarX) && (mouseX < (m_env->m_options->m_statBarX + m_trackerGraph->getDisplayWidth())) &&
				(mouseY >= m_env->m_options->m_statBarY) && (mouseY < (m_env->m_options->m_statBarY + m_trackerGraph->getDisplayHeight())) )
			{
				m_trackerGraph->displayZoomExtents( origZoomX, origZoomY, mouseX, mouseY, useCurrentViewport);
			}
		}
	}
}

void GraphManager::onWindowResize( int w, int h )
{
	if ( (w > 0) && (h > 5) )
	{
		if (isEnabled(TIMER))
		{
			m_timerGraph->setDisplayContext( m_env->m_window->getContext(), w, h/m_env->m_options->m_statBarScreenFraction );
		}
		if (isEnabled(STATS))
		{
			m_statGraph->init( m_statGraph->getType(), m_env->m_window->getContext(), w, h/m_env->m_options->m_statBarScreenFraction, 0x0 );
		}

		if (isEnabled(MEMORY))
		{
			int numGraphs = m_memoryGraphs.getSize();
			for (int i = 0 ; i < numGraphs; ++i)
			{
				m_memoryGraphs[i]->setDisplayContext( m_env->m_window->getContext(), w/numGraphs, h/m_env->m_options->m_statBarScreenFraction );
			}
		}

		if (isEnabled(TRACKER))
		{
			m_trackerGraph->resize( m_env->m_window->getContext(), w, h/m_env->m_options->m_statBarScreenFraction );
		}
	}
}

hkBool GraphManager::inStatGraphBounds(int px, int py) const
{
	const int x = m_env->m_options->m_statBarX;
	const int y = m_env->m_options->m_statBarY;
	int width = 0;
	int height = 0;

	if (isEnabled(STATS))
	{
		width = m_statGraph->getDisplayWidth();
		height = m_statGraph->getDisplayHeight();
	}

	if (isEnabled(TRACKER))
	{
		width = m_trackerGraph->getDisplayWidth();
		height = m_trackerGraph->getDisplayHeight();
	}

	return px > x && px < x + width && py > y && py < y + height;
}



void GraphManager::toggleGraph( GraphManager::Graph graph, bool enable)
{
	if (isEnabled(graph) == enable)
	{
		return;
	}
	switch (graph)
	{
		case TRACKER:
		{
			if( enable )
			{
				
				hkTrackerSnapshot snapshot;
				snapshot.init();
				
				HK_ASSERT(0x2423432, snapshot.checkConsistent() == HK_SUCCESS);

				m_trackerGraph = hkRefNew<hkgMemoryTrackerGraph>(new hkgMemoryTrackerGraph());
				hkgViewport* secondView = m_env->m_window->getSecondaryViewport();
				if (secondView)
					m_trackerGraph->init( m_env->m_window->getContext(), snapshot, secondView->getWidth(), 2*secondView->getHeight()/m_env->m_options->m_statBarScreenFraction);
				else
					m_trackerGraph->init( m_env->m_window->getContext(), snapshot, m_env->m_window->getWidth(), m_env->m_window->getHeight()/m_env->m_options->m_statBarScreenFraction);
			}
			else
			{
				m_trackerGraph = HK_NULL;
			}
			break;
		}
		case TIMER:
		{
			if (enable)
			{
				m_timerGraph = hkRefNew<hkgTimerGraph>(new hkgTimerGraph());

				m_timerGraph->setDisplayContext( m_env->m_window->getContext(), m_env->m_window->getWidth(), m_env->m_window->getHeight()/m_env->m_options->m_statBarScreenFraction );
			}
			else
			{
				m_timerGraph = HK_NULL;
			}
			break;
		}
		case STATS:
		{
			if (enable)
			{
				m_statGraph = hkRefNew<hkgStatGraph> (new hkgStatGraph());
				hkgViewport* secondView = m_env->m_window->getSecondaryViewport();
				if (secondView)
				{
					m_statGraph->init( HKG_STAT_GRAPH_BAR, m_env->m_window->getContext(), secondView->getWidth(), 2*secondView->getHeight()/m_env->m_options->m_statBarScreenFraction, 0x0);
				}
				else
				{
					m_statGraph->init( HKG_STAT_GRAPH_BAR, m_env->m_window->getContext(), m_env->m_window->getWidth(), m_env->m_window->getHeight()/m_env->m_options->m_statBarScreenFraction, 0x0);
				}
				hkMonitorStreamAnalyzer::g_lastFrameTime = m_env->m_options->m_statBarStartTime * 1000.0f;
			}
			else
			{
				m_statGraph = HK_NULL;
			}
			break;
		}

		case MEMORY:
		{
			if (enable)
			{
				int numGraphs = m_memoryGraphs.getSize();
				int w = m_env->m_window->getWidth();
				int h = m_env->m_window->getHeight();
				for (int i = 0 ; i < numGraphs; ++i)
				{
					m_memoryGraphs[i]->setDisplayContext( m_env->m_window->getContext(), w/numGraphs, h/m_env->m_options->m_statBarScreenFraction );
				}
			}
			m_showMemoryGraphs = enable;
			break;
		}
		default:
		// Should not get here.
		HK_ASSERT(0x0, false);
		break;
	}
	// Prevent graphs from overlapping.
	HK_ASSERT( 0x0, !(m_timerGraph && m_statGraph) );
	// Draw logo if the tracker is not activated
	m_env->m_window->setWantDrawHavokLogo( m_statGraph == HK_NULL && m_timerGraph == HK_NULL );
}

hkBool GraphManager::manageUserAction( int px, int py )
{
		hkBool hasAction = false;
		if (isEnabled(GraphManager::STATS))
		{
			hasAction = m_statGraph->userAction( px, py) != hkgStatGraph::HKG_STAT_GRAPH_ACTION_NONE;
		}
		if (isEnabled(GraphManager::TRACKER))
		{
			hasAction = m_trackerGraph->userAction(px, py) != hkgMemoryTrackerGraph::ACTION_NONE;
		}

		return hasAction;
}

void GraphManager::mouseZoom( int origX, int origY, int px, int py )
{
	if (isEnabled(GraphManager::STATS))
	{
		m_statGraph->zoom( origX, origY, px, py );
	}
	if (isEnabled(GraphManager::TRACKER))
	{
		m_trackerGraph->zoom( origX, origY, px, py );
	}
}

bool GraphManager::isEnabled( Graph graph ) const
{
	switch (graph)
	{
		case TIMER:
			return m_timerGraph != HK_NULL;
		case STATS:
			return m_statGraph != HK_NULL;
		case TRACKER:
			return m_trackerGraph != HK_NULL;
		case MEMORY:
			return m_showMemoryGraphs;
		default:
			return false;
	}
}

void GraphManager::addMemoryGraph( hkgSeriesGraph* memoryGraph )
{
	m_memoryGraphs.pushBack( memoryGraph);
	// Reinit the graphs with new width.
	if (isEnabled(MEMORY))
	{
		int numGraphs = m_memoryGraphs.getSize();
		int w = m_env->m_window->getWidth();
		int h = m_env->m_window->getHeight();
		for (int i = 0 ; i < numGraphs; ++i)
		{
			m_memoryGraphs[i]->setDisplayContext( m_env->m_window->getContext(), w/numGraphs, h/m_env->m_options->m_statBarScreenFraction );
		}
	}
}

void GraphManager::removeMemoryGraph( hkgSeriesGraph* memoryGraph)
{
	int i  = m_memoryGraphs.indexOf(memoryGraph);

	if (i > 0 )
	{
		m_memoryGraphs.removeAtAndCopy(i);
	}
}



HK_DECLARE_DEMO(MenuDemo, HK_DEMO_TYPE_MENU, "Display a menu of available demos", "I'm the menu demo");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
