/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_MARCHING_CUBES_H
#define HK_MARCHING_CUBES_H

#include <Common/Base/Math/Vector/hkIntVector.h>

// Contains the data for the voxelized world and creates & stores triangles for each voxel
// Uses the Marching Cubes algorithm (http://en.wikipedia.org/wiki/Marching_cubes) to triangulate the voxels.
//
// Note: This is a naive implementation and not optimized for performance or memory.

class MarchingCubes : public hkReferencedObject
{
public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);

	MarchingCubes( hkReal surfaceValue = 0.f );

	virtual ~MarchingCubes( );

	struct VoxelTriangles
	{
		public:
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO, VoxelTriangles );

			// Array of vertices
			hkArray<hkVector4> m_vertices;

			// Triangle indices.  We know we won't ever have more than 255 per voxel so use hkUint8
			hkArray<hkUint8> m_indices;
	};

	struct Voxel
	{
		public:
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO, Voxel );
	
			// The eight corners of the voxel
			hkVector4 p[8];
			
			// Value at each corner
			hkReal val[8];
	};

	void setExtentsAndVoxelSize( const hkVector4& extents, const hkVector4& voxelSize );

	void positionToGridPoint( const hkVector4& position, int& gridX, int& gridY, int& gridZ ) const;

	void positionToGridPoint( const hkVector4& position, hkIntVector& gridOut ) const;

	void gridPointToPosition( int x, int y, int z, hkVector4& positionOut ) const;

	void getGridSize( int& x, int& y, int& z ) const;

	hkReal getValueAt( const int& x, const int& y, const int& z ) const;

	const VoxelTriangles& getTrianglesAt( const int& x, const int& y, const int& z ) const;

	const VoxelTriangles& getTrianglesAt( const hkUint32& voxelIndex ) const;

	hkBool isVoxelEmpty( const hkUint32& voxelIndex ) const;

	hkUint32 getNumVoxels() const;

	// Convert between grid point (x,y,z) and grid indices
	int gridPointToIndex( int x, int y, int z ) const;
	void indexToGridPoint( int index, int& xOut, int& yOut, int& zOut ) const;

	// Convert between voxel (x,y,z) and voxel indices
	int voxelToIndex( int x, int y, int z ) const;
	void indexToVoxel( int index, int& xOut, int& yOut, int& zOut ) const;

	//
	// Computes the triangles for a particular voxel
	//
	void triangulateVoxel( const hkUint32& voxelX, const hkUint32& voxelY, const hkUint32& voxelZ );

	//
	// Triangulates all voxels
	//
	void triangulate( );

	/// Given a point, apply weighted modification of 8 voxel corners
	void modifyAtPoint( const hkVector4& point, hkReal amount );

	hkArray<hkReal>& getData() { return m_data; }

	const hkVector4& getVoxelSize() { return m_voxelSize; }

	void setSurfaceValue( hkReal surfaceValue );

private:

	// Triangulate a single voxel
	void triangulateVoxel( const Voxel& voxel, VoxelTriangles& trianglesOut );

	// Interpolate between two vertices
	void vertexInterp( const hkVector4& p1, const hkVector4& p2, const hkReal& valp1, const hkReal& valp2, hkVector4& vertexOut );

	// Sets the value for one corner of a voxel
	void setVoxelCorner( const int& cornerId, const int& gridX, const int& gridY, const int& gridZ, Voxel& voxel );

	// Surface value to represent
	hkReal m_surfaceValue;

	// Array of voxels; triangles contained within each voxel
	hkArray<VoxelTriangles> m_triangleData;

	// Raw data; one hkReal per grid point
	hkArray<hkReal> m_data;

	// Extents
	hkUint32 m_gridX;
	
	hkUint32 m_gridY;

	hkUint32 m_gridZ;

	// Size of one voxel
	hkVector4 m_voxelSize;

	hkVector4 m_invVoxelSize;

		/// length of m_voxelSize
	hkReal m_voxelLength;

	/// Edge table - precomputed edges required for Marching Cubes
	static const int s_edgeTable[256];

	/// Triangle table - precomputed; required for Marching Cubes
	static const int s_triangleTable[256][16];
};

#endif //HK_MARCHING_CUBES_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
