/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Geometry/GeometryMaker.h>


void HK_CALL GeometryMaker::createQuad(hkReal sideX, hkReal sideZ, int defaultMaterial, hkGeometry& geometryOut, const hkTransform& transform)
{
	// Vertices layout from +Y to origin:
	//	0 - 3
	//	| / |
	//	1 - 2
	const hkReal halfX = 0.5f * sideX;
	const hkReal halfZ = 0.5F * sideZ;
	hkVector4 vertices[] =
	{
		hkVector4(-halfX, 0, -halfZ),
		hkVector4(-halfX, 0, halfZ),
		hkVector4(halfX, 0, halfZ),
		hkVector4(halfX, 0, -halfZ)
	};
	geometryOut.m_vertices.append(vertices, 4);

	hkGeometry::Triangle triangles[2];
	triangles[0].set(0 ,1, 3, defaultMaterial);
	triangles[1].set(1, 2, 3, defaultMaterial);
	geometryOut.m_triangles.append(triangles, 2);

	applyTransform(transform, geometryOut);
}


void GeometryMaker::createPyramid(hkReal sideX, hkReal sideZ, hkReal heightY, int defaultMaterial, hkGeometry& geometryOut, const hkTransform& transform)
{
	//Vertices layout seen from +Y to origin:
	//  1---4
	//	|\ /|
	//	| 0 |
	//	|/ \|
	//	2---3
	const hkReal halfX = 0.5f * sideX;
	const hkReal halfZ = 0.5F * sideZ;
	hkArray<hkVector4>& vertices = geometryOut.m_vertices;
	vertices.pushBack(hkVector4(0, heightY, 0));
	vertices.pushBack(hkVector4(-halfX, 0, -halfZ));
	vertices.pushBack(hkVector4(-halfX, 0, halfZ));
	vertices.pushBack(hkVector4(halfX, 0, halfZ));
	vertices.pushBack(hkVector4(halfX, 0, -halfZ));

	hkGeometry::Triangle triangles[4];
	triangles[0].set(0, 1, 2, defaultMaterial);
	triangles[1].set(0, 2, 3, defaultMaterial);
	triangles[2].set(0, 3, 4, defaultMaterial);
	triangles[3].set(0, 4, 1, defaultMaterial);
	const int numTriangles = 4;
	geometryOut.m_triangles.append(triangles, numTriangles);

	applyTransform(transform, geometryOut);
}


void HK_CALL GeometryMaker::createBox(hkVector4Parameter halfExtents, int defaultMaterial, hkGeometry& geometryOut, const hkTransform& transform)
{
	hkVector4* vertices = geometryOut.m_vertices.expandBy(8);

	//	 4 -- 5
	//  /|   /|    Y
	// 0____1 |	   |__X
	// | 7--|-6	  /
	// |/   |/	 Z
	// 3____2
	//
	vertices[0].set(-halfExtents(0), halfExtents(1), halfExtents(2));
	vertices[1].set(halfExtents(0), halfExtents(1), halfExtents(2));
	vertices[2].set(halfExtents(0), -halfExtents(1), halfExtents(2));
	vertices[3].set(-halfExtents(0), -halfExtents(1), halfExtents(2));
	vertices[4].set(-halfExtents(0), halfExtents(1), -halfExtents(2));
	vertices[5].set(halfExtents(0), halfExtents(1), -halfExtents(2));
	vertices[6].set(halfExtents(0), -halfExtents(1), -halfExtents(2));
	vertices[7].set(-halfExtents(0), -halfExtents(1), -halfExtents(2));

	hkArray<hkGeometry::Triangle>& triangles = geometryOut.m_triangles;
	triangles.expandBy(1)->set(3, 2, 1, defaultMaterial);
	triangles.expandBy(1)->set(3, 1, 0, defaultMaterial);
	triangles.expandBy(1)->set(6, 7, 4, defaultMaterial);
	triangles.expandBy(1)->set(6, 4, 5, defaultMaterial);
	triangles.expandBy(1)->set(4, 7, 3, defaultMaterial);
	triangles.expandBy(1)->set(4, 3, 0, defaultMaterial);
	triangles.expandBy(1)->set(2, 6, 5, defaultMaterial);
	triangles.expandBy(1)->set(2, 5, 1, defaultMaterial);
	triangles.expandBy(1)->set(7, 6, 2, defaultMaterial);
	triangles.expandBy(1)->set(7, 2, 3, defaultMaterial);
	triangles.expandBy(1)->set(1, 5, 4, defaultMaterial);
	triangles.expandBy(1)->set(1, 4, 0, defaultMaterial);

	applyTransform(transform, geometryOut);
}


void HK_CALL GeometryMaker::createTjunctions(int defaultMaterial, hkGeometry& geometry)
{
	hkVector4 vertices[] =
	{
		hkVector4(-1, 0, -1),
		hkVector4(1, 0, -1),
		hkVector4(1, 0, 1),
		hkVector4(-1, 0, 1),
		hkVector4(-1.0001f, 0, 0),
		hkVector4(-2, 0, -1),
		hkVector4(-2, 0, 1),
		hkVector4(0, 0, -1.001f),
		hkVector4(1, 0, -2),
		hkVector4(-1, 0, -2),
		hkVector4(1.01f, 0, 0),
		hkVector4(2, 0, 1),
		hkVector4(2, 0, -1),
		hkVector4(0, 0, 1.1f),
		hkVector4(-1, 0, 2),
		hkVector4(1, 0, 2),
		hkVector4(-0.5f, 0.001f, 0),
		hkVector4(-0.4f, 0.1f, 0),
		hkVector4(-0.6f, 0.1f, 0),
		hkVector4(-1.0001f, 0, -1.0001f),
		hkVector4(-1.0001f, 0, -1.1f),
		hkVector4(-1.1f, 0, -1.0001f)
	};
	const int numVertices = 22;

	hkGeometry::Triangle triangles[8];
	triangles[0].set(0, 3, 1, defaultMaterial);
	triangles[1].set(1, 3, 2, defaultMaterial);
	triangles[2].set(4, 5, 6, defaultMaterial);
	triangles[3].set(7, 8, 9, defaultMaterial);
	triangles[4].set(10, 11, 12, defaultMaterial);
	triangles[5].set(13, 14, 15, defaultMaterial);
	triangles[6].set(16, 17, 18, defaultMaterial);
	triangles[7].set(19, 20, 21, defaultMaterial);
	int numTriangles = 8;

	geometry.m_vertices.append(vertices, numVertices);
	geometry.m_triangles.append(triangles, numTriangles);
}


void HK_CALL GeometryMaker::applyTransform(const hkTransform& transform, hkGeometry& geometry)
{
	hkArray<hkVector4>& vertices = geometry.m_vertices;
	for (int i = 0; i < vertices.getSize(); ++i)
	{
		vertices[i].setTransformedPos(transform, vertices[i]);
	}
}


void HK_CALL GeometryMaker::createPointGrid(int xRes, int zRes, hkReal step, hkArray<hkVector4>& grid)
{
	hkReal xOrigin = 0.5f * (1 - xRes) * step;
	hkReal zOrigin = 0.5f * (1 - zRes) * step;
	grid.setSize(xRes * zRes);
	hkVector4* point = grid.begin();
	for (int z = 0; z < zRes; ++z)
	{
		for (int x = 0; x < xRes; ++x)
		{
			point->set(xOrigin + x * step, 0, zOrigin + z * step);
			++point;
		}
	}
}

void HK_CALL GeometryMaker::createPointDisk(hkReal radius, int outerRes, int numRings, hkArray<hkVector4>& disk)
{
	disk.expandOne().setZero();
	for (int i = numRings; i > 0; --i)
	{
		hkReal fraction = i / hkReal(numRings);
		const hkReal ringRadius = fraction * radius;
		const int numPointsInRing = (int)(outerRes * fraction);
		const hkReal angleStep = 2 * HK_REAL_PI / hkReal(numPointsInRing);
		hkReal angle = 0;
		for (int j = 0 ; j < numPointsInRing; ++j)
		{
			disk.expandOne().set(ringRadius * hkMath::cos(angle), 0, ringRadius * hkMath::sin(angle));
			angle += angleStep;
		}
	}
}

void HK_CALL GeometryMaker::createIcosahedron(hkReal radius, int defaultMaterial, hkGeometry& geometryInOut)
{
	int firstVertex = geometryInOut.m_vertices.getSize();
	hkVector4* vertices = geometryInOut.m_vertices.expandBy(12);
	createIcosahedronVertices(radius, vertices);

	// Beware! This is completely dependent on the order the vertices are created in createIcosahedronVertices
	hkGeometry::Triangle* triangles = geometryInOut.m_triangles.expandBy(20);
	triangles[0].set(firstVertex + 1, firstVertex + 4, firstVertex + 0, defaultMaterial);
	triangles[1].set(firstVertex + 4, firstVertex + 9, firstVertex + 0, defaultMaterial);
	triangles[2].set(firstVertex + 4, firstVertex + 5, firstVertex + 9, defaultMaterial);
	triangles[3].set(firstVertex + 8, firstVertex + 5, firstVertex + 4, defaultMaterial);
	triangles[4].set(firstVertex + 1, firstVertex + 8, firstVertex + 4, defaultMaterial);
	triangles[5].set(firstVertex + 1, firstVertex + 10, firstVertex + 8, defaultMaterial);
	triangles[6].set(firstVertex + 10, firstVertex + 3, firstVertex + 8, defaultMaterial);
	triangles[7].set(firstVertex + 8, firstVertex + 3, firstVertex + 5, defaultMaterial);
	triangles[8].set(firstVertex + 3, firstVertex + 2, firstVertex + 5, defaultMaterial);
	triangles[9].set(firstVertex + 3, firstVertex + 7, firstVertex + 2, defaultMaterial);
	triangles[10].set(firstVertex + 3, firstVertex + 10, firstVertex + 7, defaultMaterial);
	triangles[11].set(firstVertex + 10, firstVertex + 6, firstVertex + 7, defaultMaterial);
	triangles[12].set(firstVertex + 6, firstVertex + 11, firstVertex + 7, defaultMaterial);
	triangles[13].set(firstVertex + 6, firstVertex + 0, firstVertex + 11, defaultMaterial);
	triangles[14].set(firstVertex + 6, firstVertex + 1, firstVertex + 0, defaultMaterial);
	triangles[15].set(firstVertex + 10, firstVertex + 1, firstVertex + 6, defaultMaterial);
	triangles[16].set(firstVertex + 11, firstVertex + 0, firstVertex + 9, defaultMaterial);
	triangles[17].set(firstVertex + 2, firstVertex + 11, firstVertex + 9, defaultMaterial);
	triangles[18].set(firstVertex + 5, firstVertex + 2, firstVertex + 9, defaultMaterial);
	triangles[19].set(firstVertex + 11, firstVertex + 2, firstVertex + 7, defaultMaterial);
}

void HK_CALL GeometryMaker::createIcosahedronVertices(hkReal radius, hkVector4 verticesOut[12])
{
	// The magic numbers are used to obtain an icosahedron of radius 1. See http://www.opengl.org.ru/docs/pg/0208.html
	const hkSimdReal cx = hkSimdReal::fromFloat(0.525731112119133606f);
	const hkSimdReal x = radius * cx;
	const hkSimdReal cz = hkSimdReal::fromFloat(0.850650808352039932f);
	const hkSimdReal z = radius * cz;
	const hkSimdReal zero = hkSimdReal::getConstant<HK_QUADREAL_0>();

	verticesOut[0].set(-x, zero, z, zero);
	verticesOut[1].set(x, zero, z, zero);
	verticesOut[2].set(-x, zero, -z , zero);
	verticesOut[3].set(x, zero, -z, zero);
	verticesOut[4].set(zero, z, x, zero);
	verticesOut[5].set(zero, z, -x, zero);
	verticesOut[6].set(zero, -z, x, zero);
	verticesOut[7].set(zero, -z, -x, zero);
	verticesOut[8].set(z, x, zero , zero);
	verticesOut[9].set(-z, x, zero, zero);
	verticesOut[10].set(z, -x, zero, zero);
	verticesOut[11].set(-z, -x, zero, zero);
}

void HK_CALL GeometryMaker::createTetrahedronVertices(hkReal radius, hkVector4 verticesOut[4])
{
	// sqrt(2) / 3
	const hkReal a = 0.47140452079103168293389624140323f;

	// sqrt(2) / sqrt(3)
	const hkReal b = 0.81649658092772603273242802490196f;

	const hkReal c = -radius * a;
	const hkReal d = -radius / 3.0f;
	const hkReal e = radius * b;

	verticesOut[0].set(0, radius, 0);
	verticesOut[1].set(-2.0f * c, d, 0);
	verticesOut[2].set(c, d, e);
	verticesOut[3].set(c, d, -e);
}

void HK_CALL GeometryMaker::createTetragonalBipyramidVertices(hkReal radius, hkVector4 verticesOut[6])
{
	verticesOut[0].set(-1, 0, 0);
	verticesOut[1].set(1, 0, 0);
	verticesOut[2].set(0, radius, 0);
	verticesOut[3].set(0, -radius, 0);
	verticesOut[4].set(0, 0, radius);
	verticesOut[5].set(0, 0, -radius);
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
