/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/GameUtils/TweakerUtils.h>

#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/Reflection/hkClassMember.h>
#include <Common/Base/Reflection/Attributes/hkAttributes.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Font/hkgFont.h>

#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>
#include <Common/Base/Reflection/Registry/hkVtableClassRegistry.h>
#include <Common/Base/Container/RelArray/hkRelArray.h>

#define NOT(A) (!(A))

template <typename T>
T& lookupMember(void* start)
{
	return *reinterpret_cast<T*>( start );
}

template <typename T>
const T& lookupMember(const void* start)
{
	return *reinterpret_cast<const T*>( start );
}

const hkClass* TweakerUtils::getInstanceClass( const hkClass* baseClass, const void* instance )
{
	if ( ( instance != 0 ) && baseClass->hasVtable() )
	{
		const hkClass* c = hkBuiltinTypeRegistry::getInstance().getVtableClassRegistry()->getClassFromVirtualInstance( instance );

		if ( c )
		{
			return c;
		}
	}

	return baseClass;
}

template<typename T>
struct TweakerTraits;

template<>
struct TweakerTraits<hkReal>
{
	static void format(extStringBuf& out, hkReal x) { out.printf("%f", x); }
};

template<>
struct TweakerTraits<hkInt32>
{
	static void format(extStringBuf& out, hkInt32 x) { out.printf("%d", x); }
};

template<>
struct TweakerTraits<hkUint32>
{
	static void format(extStringBuf& out, hkInt32 x) { out.printf("%u", x); }
};

template<>
struct TweakerTraits<hkInt16>
{
	static void format(extStringBuf& out, hkInt16 x) { out.printf("%d", x); }
};

template<>
struct TweakerTraits<hkUint16>
{
	static void format(extStringBuf& out, hkUint16 x) { out.printf("%u", x); }
};

template<>
struct TweakerTraits<hkBool>
{
	static void format(extStringBuf& out, hkBool x) { out.printf("%s", (x) ? "True" : "False"); }
};

static void processText(class hkTextDisplay& disp, hkBool displayText, extStringBuf& str, float x, float y, hkColor::Argb color, extStringBuf& fullName, extArray<TweakerUtils::NameAndDisplayLocation>* nameAndDisplayLocationOut) 
{
	if( displayText ) 
	{
		disp.outputText(str, x, y, color);
	}

	if( nameAndDisplayLocationOut )
	{
		TweakerUtils::NameAndDisplayLocation nl;
		nl.m_name = fullName;
		nl.m_xLocation = x;
		nl.m_yLocation = y;
		nameAndDisplayLocationOut->pushBack( nl );
	}
}

template <typename T>
void displayArrayMemberPOD(const void* memberData, const hkClassMember::Type arrayType, const hkClassMember::Type subType, const char *name, int arrayIndex,
							   class hkTextDisplay& disp, float x, float& y, 
							   const char* currentFullNameIn, 
							   TweakerUtils::DisplayOptions& options)
{
	const void* ptrAddr = static_cast<const void*>(memberData);
	const T* arrayPtr;
	int arraySz;
	if( arrayType == hkClassMember::TYPE_ARRAY)
	{
		arrayPtr = static_cast<const hkArray<T>*>( ptrAddr )->begin();
		arraySz = static_cast<const hkArray<T>*>( ptrAddr )->getSize();
	}
	else
	{
		arrayPtr = static_cast<const hkRelArray<T>*>( ptrAddr )->begin();
		arraySz = static_cast<const hkRelArray<T>*>( ptrAddr )->getSize();
	}

	for( int j = 0; j < arraySz; j++ )
	{
		extStringBuf s;

		extStringBuf valueString;
		TweakerTraits<T>::format( valueString, arrayPtr[j] );
		s.printf("%s[%d]: %s", name, j, valueString.cString());

		extStringBuf arrayElementName;
		arrayElementName.printf( "%s[%d]", currentFullNameIn, j);
		hkColor::Argb color = ( j == arrayIndex ? options.m_selectedColor : hkColor::WHITE );
		processText(disp, options.m_allowTextDisplay, s, x+20, y, color, arrayElementName, options.m_nameAndDisplayLocationOut);
		y += disp.getFont()->getCharHeight() * 1.2f;
	}
}


hkStringPtr TweakerUtils::getData( const hkClassMember& member, const void* memberData )
{
	extStringBuf str;
	extStringBuf valueString;
	
	switch ( member.getType() )
	{
		case hkClassMember::TYPE_REAL :
		{
			TweakerTraits<hkReal>::format( valueString, lookupMember<hkReal>(memberData) );
			str.printf("%s", valueString.cString() );
			break;
		}
		case hkClassMember::TYPE_INT32 :
		{
			TweakerTraits<hkInt32>::format( valueString, lookupMember<hkInt32>(memberData) );
			str.printf("%s", valueString.cString());
			break;
		}
		case hkClassMember::TYPE_UINT32 :
		{
			TweakerTraits<hkUint32>::format( valueString, lookupMember<hkUint32>(memberData) );
			str.printf("%s", valueString.cString());
			break;
		}
		case hkClassMember::TYPE_INT16 :	
		{
			TweakerTraits<hkInt16>::format( valueString, lookupMember<hkInt16>(memberData) );
			str.printf("%s", valueString.cString());
			break;
		}
		case hkClassMember::TYPE_UINT16 :
		{
			TweakerTraits<hkUint16>::format( valueString, lookupMember<hkUint16>(memberData) );
			str.printf("%s", valueString.cString());
		}
		case hkClassMember::TYPE_BOOL :
		{
			TweakerTraits<hkBool>::format( valueString, lookupMember<hkBool>(memberData) );
			str.printf("%s", valueString.cString());
			break;
		}

		case hkClassMember::TYPE_VECTOR4:
		case hkClassMember::TYPE_QUATERNION:
		{
			const hkReal* r = reinterpret_cast<const hkReal*>(memberData);
			str.printf("(%f %f %f %f)", r[0], r[1], r[2], r[3] );
			break;
		}
		case hkClassMember::TYPE_ENUM :
		{
			const hkClassEnum& e = member.getEnumType();
			int value = member.getEnumValue(memberData);
			const char* valueName = HK_NULL;
			if( e.getNameOfValue( value, &valueName) == HK_SUCCESS )
			{
				str.printf("%s\n", valueName);
			}
			break;
		}
		case hkClassMember::TYPE_STRUCT :
		{
			str.printf(">\n");
			break;
		}
		case hkClassMember::TYPE_POINTER:
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			void* ptrTo = *static_cast<void*const*>(ptrAddr);

			if ( ptrTo == HK_NULL )
			{
				str.printf("null\n");
			}
			else
			{
				switch( member.getSubType() )
				{
				case hkClassMember::TYPE_CHAR:
					{
						str.printf("\"%s\"\n", ptrTo );

						break;
					}
				case hkClassMember::TYPE_STRUCT:
					{
						const hkClass& structClass = member.getStructClass();
						const char* className = getInstanceClass( &structClass, ptrTo )->getName();

						str.printf("<%s>", className );

						break;
					}
				default:
					{
						str.printf("???\n" );

						break;
					}
				}
			}
			break;
		}
		case hkClassMember::TYPE_ARRAY :
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			const hkArray<void*>* arrayPtr = static_cast<const hkArray<void*>*>( ptrAddr );
			int arraySz = arrayPtr->getSize();
			str.printf("[0..%d]\n", arraySz-1 );
			break;
		}
		case hkClassMember::TYPE_RELARRAY :
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			const hkRelArray<void*>* arrayPtr = static_cast<const hkRelArray<void*>*>( ptrAddr );
			int arraySz = arrayPtr->getSize();
			str.printf("[0..%d]\n", arraySz-1 );
			break;
		}
		case hkClassMember::TYPE_CSTRING:
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			void* ptrTo = *static_cast<void*const*>(ptrAddr);

			if ( ptrTo == HK_NULL )
			{
				str.printf("null\n");
			}
			else
			{
				str.printf("\"%s\"\n", ptrTo );
			}
			break;
		}
		
		case hkClassMember::TYPE_FLAGS:
		{
			str.printf("[");
			const hkClassEnum& cenum = member.getEnumType();
			hkArray<const char*> bits;
			int leftOvers;
			cenum.decomposeFlags( member.getEnumValue(memberData), bits, leftOvers );
			for( int i = bits.getSize() - 1; i >= 0; --i )
			{
				str += bits[i];
				if( i != 0 ) str += " | ";
			}
			if( leftOvers )
			{
				extStringBuf tmp;
				tmp.printf("%s%s %x", str.cString(), str.getLength() ? " | " : "", leftOvers );
				str = tmp;
			}
			str += "]";
			break;
		}
		case hkClassMember::TYPE_STRINGPTR:
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			const hkStringPtr* strPtr = static_cast<const hkStringPtr*>(ptrAddr);

			if ( strPtr == HK_NULL )
			{
				str.printf("null\n");
			}
			else
			{
				str.printf("\"%s\"\n", strPtr->cString() );
			}
			break;
		}
		default :
		{
			str.printf("! ERROR !\n");
			break;
		}
	}
	return str.cString();
}

void TweakerUtils::displayMemberData(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										class hkTextDisplay& disp, float x, float& y,
										const char* currentFullNameIn, 
										DisplayOptions& options)
{
	extStringBuf path = memberPath;

	if (path.getLength() == 0)
		return;

	// Strip leading '/'
	while (path[0] == '/')
		path.chompStart(1);

	// Find next '/'
	int idx = path.indexOf('/');
	if (idx < 0) idx = path.getLength();

	int leftBracketIndex = path.indexOf('[');
	if ( leftBracketIndex < 0 ) leftBracketIndex = path.getLength();

	hkBool isArray = leftBracketIndex < idx;
	extStringBuf memberName;
	int arrayIndex = -1;

	if ( isArray )
	{
		int rightBracketIndex = path.indexOf(']');
		extStringBuf indexString( path+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.set( path, leftBracketIndex );
		path.chompStart(rightBracketIndex+1);
	}
	else if(idx > 0)
	{
		memberName.set( path, idx );
		path.chompStart(idx); // now holds remaining path
	}

	//Show the class
	int indents = 0;
	for (int memberIndex = 0; memberIndex < rootKlass.getNumMembers(); ++memberIndex )
	{
		const hkClassMember& member = rootKlass.getMember(memberIndex);
		if ( options.m_hideMember && options.m_hideMember(member) )
		{
			continue;
		}

		const char* displayName = member.getName();
		bool newLine = false;
		bool endent = false;
		{
			const hkVariant* ui = member.getAttribute("hk.Ui");
			if( ui != HK_NULL && ui->m_object != HK_NULL )
			{
				const hkUiAttribute* uiAttr = (const hkUiAttribute*)ui->m_object;
				newLine = uiAttr->m_endGroup2;
				if( hkString::strLen(uiAttr->m_group) > 0 )
				{
					++indents;
				}
				endent = uiAttr->m_endGroup;
				if( hkString::strLen(uiAttr->m_label) > 0 )
				{
					displayName = uiAttr->m_label;
				}
			}
		}

		extStringBuf currentFullName(currentFullNameIn, "/", member.getName() );

		bool isSelected = hkString::strCmp(member.getName(), memberName.cString()) == 0;
		hkColor::Argb col = (isSelected) ? options.m_selectedColor : hkColor::WHITE;

		const void* memberData = static_cast<const void*>(static_cast<const char*>(rootData) + member.getOffset());

		extStringBuf str;
		extStringBuf valueString;
		switch (member.getType())
		{
		case hkClassMember::TYPE_REAL :
			{
				TweakerTraits<hkReal>::format( valueString, lookupMember<hkReal>(memberData) );
				str.printf("%s: %s", displayName, valueString.cString() );
				break;
			}
		case hkClassMember::TYPE_INT32 :
			{
				TweakerTraits<hkInt32>::format( valueString, lookupMember<hkInt32>(memberData) );
				str.printf("%s: %s", displayName, valueString.cString());
				break;
			}
		case hkClassMember::TYPE_UINT32 :	
			{
				TweakerTraits<hkUint32>::format( valueString, lookupMember<hkUint32>(memberData) );
				str.printf("%s: %s", displayName, valueString.cString());
				break;
			}
		case hkClassMember::TYPE_INT16 :	
			{
				TweakerTraits<hkInt16>::format( valueString, lookupMember<hkInt16>(memberData) );
				str.printf("%s: %s", displayName, valueString.cString());
				break;
			}
		case hkClassMember::TYPE_UINT16 :
			{
				TweakerTraits<hkUint16>::format( valueString, lookupMember<hkUint16>(memberData) );
				str.printf("%s: %s", displayName, valueString.cString());
			}
		case hkClassMember::TYPE_BOOL :
			{
				TweakerTraits<hkBool>::format( valueString, lookupMember<hkBool>(memberData) );
				str.printf("%s: %s", displayName, valueString.cString());
				break;
			}
		
		case hkClassMember::TYPE_VECTOR4:
		case hkClassMember::TYPE_QUATERNION:
			{
				const hkReal* r = reinterpret_cast<const hkReal*>(memberData);
				str.printf("%s: (%f %f %f %f)", displayName, r[0], r[1], r[2], r[3] );
				break;
			}
			break;
		case hkClassMember::TYPE_ENUM :
			{
				const hkClassEnum& e = member.getEnumType();
				int value = member.getEnumValue(memberData);
				const char* valueName = HK_NULL;
				if( e.getNameOfValue( value, &valueName) == HK_SUCCESS )
				{
					str.printf("%s: %s\n", displayName, valueName);
				}
			}
			break;
		case hkClassMember::TYPE_STRUCT :
			{
				str.printf("%s >\n", displayName);
			}
			break;
		case hkClassMember::TYPE_POINTER:
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				void* ptrTo = *static_cast<void*const*>(ptrAddr);

				if ( ptrTo == HK_NULL )
				{
					str.printf("%s -> null\n", displayName );
				}
				else
				{
					switch( member.getSubType() )
					{
						case hkClassMember::TYPE_CHAR:
						{
							str.printf("%s -> \"%s\"\n", displayName, ptrTo );

							break;
						}
						case hkClassMember::TYPE_STRUCT:
						{
							const hkClass& structClass = member.getStructClass();
							const char* className = getInstanceClass( &structClass, ptrTo )->getName();

							str.printf("%s -> <%s>", displayName, className );

							break;
						}
						default:
						{
							str.printf("%s -> ???\n", displayName );

							break;
						}
					}
				}
			}
			break;
		case hkClassMember::TYPE_ARRAY :
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				const hkArray<void*>* arrayPtr = static_cast<const hkArray<void*>*>( ptrAddr );
				int arraySz = arrayPtr->getSize();
				str.printf("%s[0..%d]\n", displayName, arraySz-1 );
			}
			break;
		case hkClassMember::TYPE_RELARRAY :
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				const hkRelArray<void*>* arrayPtr = static_cast<const hkRelArray<void*>*>( ptrAddr );
				int arraySz = arrayPtr->getSize();
				str.printf("%s[0..%d]\n", displayName, arraySz-1 );
			}
			break;
		case hkClassMember::TYPE_CSTRING:
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				void* ptrTo = *static_cast<void*const*>(ptrAddr);

				if ( ptrTo == HK_NULL )
				{
					str.printf("%s -> null\n", displayName );
				}
				else
				{
					str.printf("%s -> \"%s\"\n", displayName, ptrTo );
				}
			}
			break;
		case hkClassMember::TYPE_FLAGS:
			{
				str.printf("%s -> [", displayName );
				const hkClassEnum& cenum = member.getEnumType();
				hkArray<const char*> bits;
				int leftOvers;
				cenum.decomposeFlags( member.getEnumValue(memberData), bits, leftOvers );
				for( int i = bits.getSize() - 1; i >= 0; --i )
				{
					str += bits[i];
					if( i != 0 ) str += " | ";
				}
				if( leftOvers )
				{
					extStringBuf tmp;
					tmp.printf("%s%s %x", str.cString(), str.getLength() ? " | " : "", leftOvers );
					str = tmp;
				}
				str += "]";
			}
			break;
			case hkClassMember::TYPE_STRINGPTR:
			{
				const void* ptrAddr = static_cast<const void*>(memberData);
				const hkStringPtr* strPtr = static_cast<const hkStringPtr*>(ptrAddr);

				if ( strPtr == HK_NULL )
				{
					str.printf("%s -> null\n", displayName );
				}
				else
				{
					str.printf("%s -> \"%s\"\n", displayName, strPtr->cString() );
				}
			}
			break;
		default :
			{
				str.printf("%s\n", displayName);
			}
			break;
		}
		
		if(newLine)
		{
			indents = 0;
			y += disp.getFont()->getCharHeight() * 0.4f;
		}

		processText(disp, options.m_allowTextDisplay, str, x+indents*16, y, col, currentFullName, options.m_nameAndDisplayLocationOut );

		y += disp.getFont()->getCharHeight() * 1.2f;

		if( endent && indents > 0 )
		{
			--indents;
		}

		// Chase structs
		if ((isSelected) && (member.getType() == hkClassMember::TYPE_STRUCT))
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			displayMemberData(path, ptrAddr, member.getStructClass(), disp, x+20, y, 
				currentFullName, options);
		}

		// Chase pointers
		if (isSelected && (member.getType() == hkClassMember::TYPE_POINTER) && (member.getSubType() == hkClassMember::TYPE_STRUCT))
		{
			const void* ptrAddr = static_cast<const void*>(memberData);
			void* ptrTo = *static_cast<void*const*>(ptrAddr);

			if ( ptrTo )
			{
				const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

				displayMemberData(path, ptrTo, *instanceClass, disp, x+20, y,
					currentFullName, options);
			}
		}

		// Chase arrays
		if ( isArray && isSelected && ((member.getType() == hkClassMember::TYPE_ARRAY) || (member.getType() == hkClassMember::TYPE_RELARRAY)))
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :		
				{
					displayArrayMemberPOD<hkReal>( static_cast<const void*>(memberData), member.getType(), member.getSubType(), memberName.cString(), arrayIndex,  
						disp, x, y, currentFullName.cString(), options );
					break;
				}
			case hkClassMember::TYPE_INT32 :
				{
					displayArrayMemberPOD<hkInt32>( static_cast<const void*>(memberData), member.getType(), member.getSubType(), memberName.cString(), arrayIndex,  
						disp, x, y, currentFullName.cString(), options );
					break;
				}
			case hkClassMember::TYPE_UINT32 :
				{
					displayArrayMemberPOD<hkUint32>( static_cast<const void*>(memberData), member.getType(), member.getSubType(), memberName.cString(), arrayIndex,  
						disp, x, y, currentFullName.cString(), options );
					break;
				}
			case hkClassMember::TYPE_INT16 :
				{
					displayArrayMemberPOD<hkInt16>( static_cast<const void*>(memberData), member.getType(), member.getSubType(), memberName.cString(), arrayIndex,  
						disp, x, y, currentFullName.cString(), options );
					break;
				}
			case hkClassMember::TYPE_UINT16 :
				{
					displayArrayMemberPOD<hkUint16>( static_cast<const void*>(memberData), member.getType(), member.getSubType(), memberName.cString(), arrayIndex,  
						disp, x, y, currentFullName.cString(), options );
					break;
				}
			case hkClassMember::TYPE_BOOL :
				{
					displayArrayMemberPOD<hkBool>( static_cast<const void*>(memberData), member.getType(), member.getSubType(), memberName.cString(), arrayIndex,  
						disp, x, y, currentFullName.cString(), options  );
					break;
 				}
			case hkClassMember::TYPE_POINTER:
				{
					const void* ptrAddr = static_cast<const void*>(memberData);
					const void* const * arrayPtr;
					int arraySz;
					if(member.getType() == hkClassMember::TYPE_ARRAY)
					{
						arrayPtr = static_cast<const hkArray<void*>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkArray<void*>*>( ptrAddr )->getSize();
					}
					else
					{
						arrayPtr = static_cast<const hkRelArray<void*>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkRelArray<void*>*>( ptrAddr )->getSize();
					}

					for( int j = 0; j < arraySz; j++ )
					{
						const void* ptrTo = arrayPtr[j];
						const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

						const char* instanceClassName = ( ptrTo ? instanceClass->getName() : "null" );

						extStringBuf s;
						s.printf( "%s[%d] -> <%s>", memberName.cString(), j, instanceClassName );

						extStringBuf arrayElementName;
						arrayElementName.printf( "%s[%d]", currentFullName.cString(), j);

						if ( j == arrayIndex )
						{
							processText(disp, options.m_allowTextDisplay, s, x+20, y, options.m_selectedColor, arrayElementName, options.m_nameAndDisplayLocationOut);
							y += disp.getFont()->getCharHeight() * 1.2f;

							if ( ptrTo )
							{
								displayMemberData(path, ptrTo, *instanceClass, disp, x+40, y, 
									arrayElementName, options);
							}
						}
						else
						{
							processText(disp, options.m_allowTextDisplay, s, x+20, y, 0xffffffff, arrayElementName, options.m_nameAndDisplayLocationOut);
							y += disp.getFont()->getCharHeight() * 1.2f;
						}
					}

					break;
				}
			case hkClassMember::TYPE_STRUCT:
				{
					const void* ptrAddr = static_cast<const void*>(memberData);
					const char* arrayPtr;
					int arraySz;
					if(member.getType() == hkClassMember::TYPE_ARRAY)
					{
						arrayPtr = static_cast<const hkArray<char>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkArray<char>*>( ptrAddr )->getSize();
					}
					else
					{
						arrayPtr = static_cast<const hkRelArray<char>*>( ptrAddr )->begin();
						arraySz = static_cast<const hkRelArray<char>*>( ptrAddr )->getSize();
					}
					int structSz = member.getArrayMemberSize();

					for( int j = 0; j < arraySz; j++ )
					{
						extStringBuf s;
						s.printf( "%s[%d]", memberName.cString(), j );

						extStringBuf arrayElementName;
						arrayElementName.printf( "%s[%d]", currentFullName.cString(), j);

						if ( j == arrayIndex )
						{
							processText(disp, options.m_allowTextDisplay, s, x+20, y, options.m_selectedColor, arrayElementName, options.m_nameAndDisplayLocationOut);
							y += disp.getFont()->getCharHeight() * 1.2f;

							displayMemberData(path, arrayPtr + j*structSz, member.getStructClass(), disp, x+40, y, 
								arrayElementName, options );
						}
						else
						{
							processText(disp, options.m_allowTextDisplay, s, x+20, y, 0xffffffff, arrayElementName, options.m_nameAndDisplayLocationOut);
							y += disp.getFont()->getCharHeight() * 1.2f;
						}
					}

					break;
				}
			default:
				{
					break;
				}
			}
		}
	}
}




int getMemberIdx( const char* memberName, const hkClass& klass)
{
	int memberIdx = -1;
	for (int i=0; i < klass.getNumMembers(); i++)
	{
		const hkClassMember& member = klass.getMember(i);
		const hkStringBuf name(member.getName());
		if ( name == memberName )
		{
			memberIdx = i;
		}
	}
	return memberIdx;
}

hkStringPtr TweakerUtils::getClass(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
									void*& data, const hkClass*& klass,
									HideMemberFunc hideMember )
{
	hkStringBuf path = memberPath;

	// Strip leading '/'
	while (path[0] == '/')
		path.chompStart(1);

	// Empty string
	if (path.getLength() == 0)
	{
		data = const_cast<void*>(rootData);
		klass = &rootKlass;
		return "/";
	}

	// Find next '/'
	int idx = path.indexOf('/');
	if (idx < 0) idx = path.getLength();

	// find next '['

	int leftBracketIndex = path.indexOf( '[' );
	if (leftBracketIndex < 0) leftBracketIndex = path.getLength();

	hkBool isArray = ( leftBracketIndex < idx );
	hkStringBuf memberName;
	int arrayIndex = -1;
	hkStringBuf memberNameAndIndex;

	if ( isArray )
	{
		int rightBracketIndex = path.indexOf(']');
		hkStringBuf indexString; indexString.set( path.cString()+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.set(path.cString(), leftBracketIndex );
		memberNameAndIndex.set(path.cString(), rightBracketIndex+1);
		path.chompStart(rightBracketIndex+1);
	}
	else
	{
		memberName.set(path.cString(), idx);
		path.chompStart(idx); // now holds remaining path
	}

	// Check the klass has this member
	int memberIdx = getMemberIdx( memberName, rootKlass);

	// Member not found so return
	if (memberIdx == -1)
	{
		data = const_cast<void*>(rootData);
		klass = &rootKlass;
		return "";
	}

	const hkClassMember& member = rootKlass.getMember( memberIdx );

	// Chase pointers to structs / classes

	if (path.getLength() > 0)
	{
		if (member.getType() == hkClassMember::TYPE_STRUCT)
		{
			const void* ptrAddr = static_cast<const char*>(rootData) + member.getOffset();
			hkStringBuf sb("/", memberName,
				TweakerUtils::getClass(path, ptrAddr, member.getStructClass(), data, klass) );
			return sb.cString();
		}
		else if (( member.getType() ==  hkClassMember::TYPE_POINTER) && (member.getSubType() == hkClassMember::TYPE_STRUCT))
		{
			const void* ptrAddr = static_cast<const char*>(rootData) + member.getOffset();
			const void* ptrTo = *static_cast<const void*const*>(ptrAddr);
			const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

			hkStringBuf sb("/", memberName, 
				TweakerUtils::getClass(path, ptrTo, *instanceClass, data, klass));
			return sb.cString();
		}
		else if (((member.getType() == hkClassMember::TYPE_ARRAY ) || ( member.getType() == hkClassMember::TYPE_RELARRAY )) && isArray )
		{
			if ( member.getSubType() == hkClassMember::TYPE_STRUCT )
			{
				const void* ptrAddr = static_cast<const char*>(rootData) + member.getOffset();

				const char* arrayPtr;
				if(member.getType() == hkClassMember::TYPE_ARRAY)
				{
					arrayPtr = static_cast<const hkArray<char>*>( ptrAddr )->begin();
				}
				else
				{
					arrayPtr = static_cast<const hkRelArray<char>*>( ptrAddr )->begin();
				}

				int sz = member.getArrayMemberSize();
				const void* memberData = arrayPtr + sz*arrayIndex;

				hkStringBuf sb("/", memberNameAndIndex, 
					TweakerUtils::getClass(path, memberData, member.getStructClass(), data, klass));
				return sb.cString();
			}
			else if ( member.getSubType() == hkClassMember::TYPE_POINTER )
			{
				const void* arrayData = static_cast<const void*>(static_cast<const char*>(rootData) + member.getOffset());
				void* const* arrayPtr;
				if(member.getType() == hkClassMember::TYPE_ARRAY)
				{
					arrayPtr = static_cast<const hkArray<void*>*>( arrayData )->begin();
				}
				else
				{
					arrayPtr = static_cast<const hkRelArray<void*>*>( arrayData )->begin();
				}				
				const void* ptrTo = *(arrayPtr + arrayIndex);
				const hkClass* instanceClass = getInstanceClass( &member.getStructClass(), ptrTo );

				hkStringBuf sb("/", memberNameAndIndex, 
						TweakerUtils::getClass(path, ptrTo, *instanceClass, data, klass));
				return sb.cString();
			}
			else
			{
				return memberPath;
			}
		}
	}

	// Return this class
	data = const_cast<void*>(rootData);
	klass = &rootKlass;
	return "/";
}

hkStringPtr TweakerUtils::setReal( const char* memberPath, void* rootData, const hkClass& rootKlass, float newValue )
{
	if( memberPath==HK_NULL || memberPath[0]==0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringPtr classPath = TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass);

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	// deal with arrays

	int rightBracketIndex = memberName.indexOf( ']' );
	int arrayIndex = -1;

	if ( rightBracketIndex != -1 )
	{
		int leftBracketIndex = memberName.indexOf( '[' );
		hkStringBuf indexString; indexString.set( memberName.cString() + leftBracketIndex + 1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);
	if (memberIdx < 0)
		return classPath;

	const hkClassMember& member = klass->getMember(memberIdx);
	void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());

	hkStringBuf str;
	switch (member.getType())
	{
	case hkClassMember::TYPE_REAL :
		{
			hkReal& value = lookupMember<hkReal>(memberData);

			value = newValue;
		}
		break;
	case hkClassMember::TYPE_INT32 :
		{
			hkInt32& value = lookupMember<hkInt32>(memberData);
			value = hkInt32(newValue);
		}
		break;
	case hkClassMember::TYPE_UINT32 :
		{
			hkUint32& value = lookupMember<hkUint32>(memberData);
			value = hkUint32(newValue);
		}
		break;
	case hkClassMember::TYPE_INT16 :
		{
			hkInt16& value = lookupMember<hkInt16>(memberData);
			value = hkInt16(newValue);
		}
		break;
	case hkClassMember::TYPE_UINT16 :
		{
			hkUint16& value = lookupMember<hkUint16>(memberData);
			value = hkUint16(newValue);
		}
		break;
	case hkClassMember::TYPE_BOOL :
		{
			bool& value = lookupMember<bool>(memberData);
			value = (newValue!=0.0f);
		}
		break;
	case hkClassMember::TYPE_ENUM :
		{
			// for backward compatibility we clear the data on zero, otherwise do nothing to enums
			if ( newValue == 0.0f )
			{
				const hkClassEnum& e = member.getEnumType();
				member.setEnumValue(memberData, e.getItem(0).getValue());
			}
		}
		break;
	case hkClassMember::TYPE_ARRAY:
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkArray<hkReal>* arrayPtr = static_cast<hkArray<hkReal>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = newValue;
				}
				break;
			case hkClassMember::TYPE_INT32 :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkArray<hkInt32>* arrayPtr = static_cast<hkArray<hkInt32>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = static_cast<hkInt32>(newValue);
				}
				break;
			default:
				break;
			}
		}
		break;
	case hkClassMember::TYPE_RELARRAY:
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkRelArray<hkReal>* arrayPtr = static_cast<hkRelArray<hkReal>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = newValue;
				}
				break;
			case hkClassMember::TYPE_INT32 :
				{
					void* ptrAddr = static_cast<void*>(memberData);
					hkRelArray<hkInt32>* arrayPtr = static_cast<hkRelArray<hkInt32>*>( ptrAddr );
					(*arrayPtr)[arrayIndex] = static_cast<hkInt32>(newValue);
				}
				break;
			default:
				break;
			}
		}
		break;
	default:
		break;
	}

	return memberPath;
}

static hkInt32 convertTweakerFloatToInt(float f)
{
	// If between -1 and 1, set to 1 or -1
	if ( hkMath::fabs(f) < 1.0f)
	{
		return (f > 0) ? 1 : -1;
	}

	// Else round towards zero
	return hkInt32(f);
}


void HK_CALL tweakMemberPOD(void* memberData, const hkClassMember::Type type, float offset, const hkReal threshold, TweakerUtils::FloatTweakType floatTweakType)
{
	switch (type)
	{ 
		case hkClassMember::TYPE_REAL :
		{

			switch(floatTweakType)
			{
			case TweakerUtils::MULTIPLICATIVE:
				{
					hkReal& value = lookupMember<hkReal>(memberData);
					value *= (value > 0) ? 1.0f + offset : 1.0f - offset;


					// On 0 snap to +ve or -ve
					if ( value == 0.0f )
					{
						value = (offset > 0) ? threshold : -threshold;
					}

					// Snap to 0 exactly
					if (hkMath::fabs(value) < threshold)
					{
						value = 0.0f;
					}
					break;
				}
			case TweakerUtils::ADDITIVE:
				{
					hkReal& value = lookupMember<hkReal>(memberData);
					value += offset;	
					break;
				}
			default:
				{
					break;
				}
			}
		}
		break;
	case hkClassMember::TYPE_INT32 :
		{
			hkInt32& value = lookupMember<hkInt32>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value += offsetAsInt;
		}
		break;
	case hkClassMember::TYPE_UINT32 :
		{
			hkUint32& value = lookupMember<hkUint32>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value += offsetAsInt;
		}
		break;
	case hkClassMember::TYPE_INT16 :
		{
			hkInt16& value = lookupMember<hkInt16>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value = hkInt16( value + offsetAsInt);
		}
		break;
	case hkClassMember::TYPE_UINT16 :
		{
			hkUint16& value = lookupMember<hkUint16>(memberData);
			hkInt32 offsetAsInt = convertTweakerFloatToInt(offset);
			value = hkUint16( value + offsetAsInt);
		}
		break;
	case hkClassMember::TYPE_BOOL :
		{
			bool& value = lookupMember<bool>(memberData);
			value = !value;
		}
		break;
	default:
		{
			break;
		}
	}

}


template <typename T>
void tweakArrayMemberPOD(void* memberData, int arrayIndex, const hkClassMember::Type type, float offset, const hkReal threshold,  TweakerUtils::FloatTweakType floatTweakType)
{
	void* ptrAddr = static_cast<void*>(memberData);
	hkArray<T>* arrayPtr = static_cast<hkArray<T>*>( ptrAddr );
	tweakMemberPOD( (void*)&((*arrayPtr)[arrayIndex]), type, offset, threshold, floatTweakType);
}

template <typename T>
void tweakRelArrayMemberPOD(void* memberData, int arrayIndex, const hkClassMember::Type type, float offset, const hkReal threshold,  TweakerUtils::FloatTweakType floatTweakType)
{
	void* ptrAddr = static_cast<void*>(memberData);
	hkRelArray<T>* arrayPtr = static_cast<hkRelArray<T>*>( ptrAddr );
	tweakMemberPOD( (void*)&((*arrayPtr)[arrayIndex]), type, offset, threshold, floatTweakType);
}

void applyValueRange(const hkClassMember& member, void* memberData)
{
	const hkRangeRealAttribute* rangeReal = HK_NULL;
	{
		const hkVariant* variant = member.getAttribute("hk.RangeReal");
		if(variant != HK_NULL)
		{
			rangeReal = (const hkRangeRealAttribute*)variant->m_object;
		}
	}
	const hkRangeInt32Attribute* rangeInt32 = HK_NULL;
	{
		const hkVariant* variant = member.getAttribute("hk.RangeInt32");
		if(variant != HK_NULL)
		{
			rangeInt32 = (const hkRangeInt32Attribute*)variant->m_object;
		}
	}

	switch (member.getType())
	{
		case hkClassMember::TYPE_REAL :
		{
			if(rangeReal != HK_NULL)
			{
				hkReal& value = *(hkReal*)(memberData);
				value = hkMath::clamp(value, rangeReal->m_absmin, rangeReal->m_absmax);
			}
			break;
		}
		case hkClassMember::TYPE_INT32 :
		{
			if(rangeInt32 != HK_NULL)
			{
				hkInt32& value = *(hkInt32*)(memberData);
				value = hkMath::clamp(value, rangeInt32->m_absmin, rangeInt32->m_absmax);
			}
			break;
		}
		case hkClassMember::TYPE_UINT32 :
		{
			if(rangeInt32 != HK_NULL)
			{
				hkUint32& value = *(hkUint32*)(memberData);
				value = hkMath::clamp(value, hkUint32(hkMath::max2(0, rangeInt32->m_absmin)), hkUint32(rangeInt32->m_absmax));
			}
			break;
		}
		case hkClassMember::TYPE_INT16 :
		{
			if(rangeInt32 != HK_NULL)
			{
				hkInt16& value = *(hkInt16*)(memberData);
				value = hkMath::clamp(value, hkInt16(rangeInt32->m_absmin), hkInt16(rangeInt32->m_absmax));
			}
			break;
		}
		case hkClassMember::TYPE_UINT16 :
		{
			if(rangeInt32 != HK_NULL)
			{
				hkUint16& value = *(hkUint16*)(memberData);
				value = hkMath::clamp(value, hkUint16(hkMath::max2(0, rangeInt32->m_absmin)), hkUint16(rangeInt32->m_absmax));
			}
			break;
		}
		default:
			break;
	}
}

hkStringPtr TweakerUtils::tweakData(const char* memberPath, void* rootData, const hkClass& rootKlass, float offset, const hkReal threshold, FloatTweakType floatTweakType)
{
	if (memberPath==HK_NULL || memberPath[0] == 0)
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	// deal with arrays

	int rightBracketIndex = memberName.indexOf( ']' );
	int arrayIndex = -1;

	if ( rightBracketIndex != -1 )
	{
		int leftBracketIndex = memberName.indexOf( '[' );
		hkStringBuf indexString; indexString.set( memberName.cString()+leftBracketIndex + 1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);
	if (memberIdx < 0)
		return classPath.cString();

	const hkClassMember& member = klass->getMember(memberIdx);
	void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());

	hkStringBuf str;
	switch (member.getType())
	{
	case hkClassMember::TYPE_REAL :
	case hkClassMember::TYPE_INT32 :
	case hkClassMember::TYPE_UINT32 :
	case hkClassMember::TYPE_INT16 :
	case hkClassMember::TYPE_UINT16 :
	case hkClassMember::TYPE_BOOL :
		{
			tweakMemberPOD( memberData, member.getType(), offset, threshold, floatTweakType );
			applyValueRange( member, memberData );
			break;
		}
		break;
	case hkClassMember::TYPE_ENUM :
		{
			const hkClassEnum& e = member.getEnumType();
			int value = member.getEnumValue(memberData);

			// Find item with current value
			int itemIdx = 0;
			while ( e.getItem(itemIdx).getValue() != value)
				itemIdx++;

			itemIdx += (offset > 0) ? 1 : e.getNumItems() -1;
			itemIdx = itemIdx % e.getNumItems();

			member.setEnumValue(memberData, e.getItem(itemIdx).getValue());
		}
		break;
	case hkClassMember::TYPE_ARRAY :
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :		tweakArrayMemberPOD<hkReal>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_INT32 :	tweakArrayMemberPOD<hkInt32>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_UINT32 :	tweakArrayMemberPOD<hkUint32>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_INT16 :	tweakArrayMemberPOD<hkInt16>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_UINT16 :	tweakArrayMemberPOD<hkUint32>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_BOOL :		tweakArrayMemberPOD<hkBool>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			// case hkClassMember::TYPE_ENUM :
				default:
				{
					break;
				}
			}
		}
		break;
	case hkClassMember::TYPE_RELARRAY :
		{
			switch( member.getSubType() )
			{
			case hkClassMember::TYPE_REAL :		tweakRelArrayMemberPOD<hkReal>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_INT32 :	tweakRelArrayMemberPOD<hkInt32>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_UINT32 :	tweakRelArrayMemberPOD<hkUint32>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_INT16 :	tweakRelArrayMemberPOD<hkInt16>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_UINT16 :	tweakRelArrayMemberPOD<hkUint32>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			case hkClassMember::TYPE_BOOL :		tweakRelArrayMemberPOD<hkBool>(memberData, arrayIndex, member.getSubType(), offset, threshold, floatTweakType); break;
			// case hkClassMember::TYPE_ENUM :
			default:
				{
					break;
				}
			}
		}
		break;
	default:
		break;
	}

	return memberPath;
}


hkStringPtr TweakerUtils::clearData(const char* memberPath, void* rootData, const hkClass& rootKlass)
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	int memberIdx = getMemberIdx(memberName, *klass);
	if (memberIdx < 0)
		return classPath.cString();

	const hkClassMember& member = klass->getMember(memberIdx);
	void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());

	hkStringBuf str;
	switch (member.getType())
	{
	case hkClassMember::TYPE_REAL :
		{
			hkReal& value = lookupMember<hkReal>(memberData);
			value = 0.0f;
		}
		break;
	case hkClassMember::TYPE_INT32 :
		{
			hkInt32& value = lookupMember<hkInt32>(memberData);
			value = 0;
		}
		break;
	case hkClassMember::TYPE_UINT32 :
		{
			hkUint32& value = lookupMember<hkUint32>(memberData);
			value = 0;
		}
		break;
	case hkClassMember::TYPE_BOOL :
		{
			bool& value = lookupMember<bool>(memberData);
			value = static_cast<bool>(0);
		}
		break;
	case hkClassMember::TYPE_ENUM :
		{
			const hkClassEnum& e = member.getEnumType();
			member.setEnumValue(memberData, e.getItem(0).getValue());
		}
		break;
	default:
		break;
	}

	return memberPath;
}

void HK_CALL TweakerUtils::displayData(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
								class hkTextDisplay& disp, float x, float y,
								TweakerUtils::DisplayOptions* options )
{
	float yCopy = y;

	hkStringBuf currentFullName;
	TweakerUtils::DisplayOptions defaultOptions;
	displayMemberData(memberPath, rootData, rootKlass, disp, x, yCopy, 
		currentFullName, options ? *options : defaultOptions);
}


hkStringPtr TweakerUtils::getNextSiblingPath(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										HideMemberFunc hideMember )
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	int leftBracketIndex = memberName.indexOf('[');
	int arrayIndex = -1;

	if ( leftBracketIndex != -1 )
	{
		int rightBracketIndex = memberName.indexOf(']');
		hkStringBuf indexString; indexString.set( memberName.cString()+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);

	if ( (memberIdx != -1) && (arrayIndex != -1) )
	{
		const hkClassMember& member = klass->getMember(memberIdx);

		if ( member.getType() == hkClassMember::TYPE_ARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkArray<char>* arrayPtr = static_cast<hkArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}
		else if ( member.getType() == hkClassMember::TYPE_RELARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkRelArray<char>* arrayPtr = static_cast<hkRelArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}
	}

	do
	{
		memberIdx = ( memberIdx + 1 ) % klass->getNumMembers();
	} while ( hideMember && hideMember(klass->getMember(memberIdx)) );

	classPath += klass->getMember(memberIdx).getName();
	return classPath.cString();
}

hkStringPtr TweakerUtils::getPrevSiblingPath(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										HideMemberFunc hideMember )
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	hkStringBuf memberName = memberPath;
	if (classPath.getLength() > 0)
		memberName.replace(classPath, "");

	int leftBracketIndex = memberName.indexOf('[');
	int arrayIndex = -1;

	if ( leftBracketIndex != -1 )
	{
		int rightBracketIndex = memberName.indexOf(']');
		hkStringBuf indexString; indexString.set( memberName.cString()+leftBracketIndex+1, rightBracketIndex - leftBracketIndex - 1 );
		arrayIndex = hkString::atoi( indexString.cString() );
		memberName.slice( 0, leftBracketIndex );
	}

	int memberIdx = getMemberIdx(memberName, *klass);

	if ( (memberIdx != -1) && (arrayIndex != -1) )
	{
		const hkClassMember& member = klass->getMember(memberIdx);

		if ( member.getType() == hkClassMember::TYPE_ARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkArray<char>* arrayPtr = static_cast<hkArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + sz - 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}
		else if ( member.getType() == hkClassMember::TYPE_RELARRAY )
		{
			void* memberData = static_cast<void*>(static_cast<char*>(data) + member.getOffset());
			hkRelArray<char>* arrayPtr = static_cast<hkRelArray<char>*>(memberData);
			int sz = arrayPtr->getSize();

			int newIndex = (arrayIndex + sz - 1) % sz;

			hkStringBuf s;
			s.printf( "%s%s[%d]", classPath.cString(), memberName.cString(), newIndex );
			return s.cString();
		}	
	}

	if ( memberIdx == -1 )
	{
		memberIdx = 0;
	}

	do
	{
		memberIdx = ( memberIdx - 1 + klass->getNumMembers()) % klass->getNumMembers();
	} while ( hideMember && hideMember(klass->getMember(memberIdx)) );

	classPath += klass->getMember(memberIdx).getName();
	return classPath.cString();
}

hkStringPtr TweakerUtils::getChildPath(	const char* memberPath, const void* rootData, const class hkClass& rootKlass,
										HideMemberFunc hideMember )
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	if( hkString::endsWith(memberPath,"]") )
	{
		// getClass only returns the class of the array element type if a slash is at the end
		hkStringBuf path(memberPath, "/");

		const hkClass* klass = 0;
		void* data;
		hkStringBuf classPath (TweakerUtils::getClass(path, rootData, rootKlass, data, klass));

		if ( klass && ( klass->getNumMembers() > 0 ) )
		{
			for( int i = 0; i < klass->getNumMembers(); i++ )
			{
				const hkClassMember& member = klass->getMember(i);
				if( NOT( hideMember && hideMember(member) ) )
				{
					classPath += member.getName();
					return classPath.cString();
				}
			}

			// all of the members filtered out so we just return the parent path that was passed in
			return memberPath;
		}

		return memberPath;
	}

	hkStringBuf path;
	path = memberPath;

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(path, rootData, rootKlass, data, klass));

	int memberIdx;
	{
		hkStringBuf memberName = path;
		if (classPath.getLength() > 0)
			memberName.replace(classPath, "");

		memberIdx = getMemberIdx(memberName, *klass);
	}

	// Couldn't find the member
	if (memberIdx == -1)
	{
		if (rootKlass.getNumMembers() > 0)
		{
			for( int i = 0; i < rootKlass.getNumMembers(); i++ )
			{
				const hkClassMember& member = rootKlass.getMember(i);
				if( NOT( hideMember && hideMember(member) ) )
				{
					classPath += member.getName();
					return classPath.cString();
				}
			}
		}
	}

	const hkClassMember& member = klass->getMember(memberIdx);

	if ( member.getType() ==  hkClassMember::TYPE_STRUCT)
	{
		const hkClass& childClass = member.getStructClass();
		if (childClass.getNumMembers() > 0)
		{
			for( int i = 0; i < childClass.getNumMembers(); i++ )
			{
				const hkClassMember& mem = childClass.getMember(i);
				if( NOT( hideMember && hideMember(mem) ) )
				{
					hkStringBuf sb(memberPath, "/", mem.getName());
					return sb.cString();
				}
			}

			return memberPath;
		}
	}
	else if (( member.getType() ==  hkClassMember::TYPE_POINTER) && (member.getSubType() == hkClassMember::TYPE_STRUCT))
	{
		const void* ptrAddr = const_cast<const char*> (static_cast<char*>(data) + member.getOffset() );
		const void* ptrTo = *static_cast<const void*const*>(ptrAddr);

		const hkClass& childClass = *getInstanceClass( &member.getStructClass(), ptrTo );

		if ( ptrTo && ( childClass.getNumMembers() > 0 ) )
		{
			for( int i = 0; i < childClass.getNumMembers(); i++ )
			{
				const hkClassMember& mem = childClass.getMember(i);
				if( NOT( hideMember && hideMember(mem) ) )
				{
					hkStringBuf sb(memberPath, "/", mem.getName());
					return sb.cString();
				}
			}

			return memberPath;
		}
	}
	else if ( member.getType() == hkClassMember::TYPE_ARRAY )
	{
		// check to see if it is an empty array or not

		const void* ptrAddr = const_cast<const char*>( static_cast<char*>(data) + member.getOffset() );
		const hkArray<char>* arrayPtr = static_cast<const hkArray<char>*>( ptrAddr );

		if ( arrayPtr->getSize() )
		{
			hkStringBuf sb(memberPath, "[0]"); // first element
			return sb.cString();
		}
	}
	else if ( member.getType() == hkClassMember::TYPE_RELARRAY )
	{
		// check to see if it is an empty array or not

		const void* ptrAddr = const_cast<const char*>( static_cast<char*>(data) + member.getOffset() );
		const hkRelArray<char>* arrayPtr = static_cast<const hkRelArray<char>*>( ptrAddr );

		if ( arrayPtr->getSize() )
		{
			hkStringBuf sb(memberPath, "[0]"); // first element
			return sb.cString();
		}
	}

	return memberPath;
}

hkStringPtr TweakerUtils::getParentPath(const char* memberPath, const void* rootData, const class hkClass& rootKlass)
{
	if (memberPath == HK_NULL || memberPath[0] == 0 )
		return "/";

	hkStringBuf path = memberPath;
	if ( path.endsWith("]") )
	{
		int leftBracketIndex = path.lastIndexOf('[');
		path.slice(0, leftBracketIndex);
		return path.cString();
	}

	const hkClass* klass;
	void* data;
	hkStringBuf classPath (TweakerUtils::getClass(memberPath, rootData, rootKlass, data, klass));

	// Remove trailing /
	while (classPath.endsWith("/"))
		classPath.chompEnd(1);

	return classPath.cString();
}


// Helper utils
void DemoMouseTweaker::TweakAlgorithmSpec::setAdditive(hkReal delta)
{
	m_positiveOffset = delta;
	m_negativeOffset = -delta;
	m_threshold = 0.0f;
	m_floatTweakType = TweakerUtils::ADDITIVE;
}

void DemoMouseTweaker::TweakAlgorithmSpec::setMultiplicative(int numMultsPerFactorOfTen, hkReal threshold)
{
	hkReal delta = hkMath::pow( hkReal(10.0f), 1.0f / (hkReal) numMultsPerFactorOfTen) - 1.0f;
	m_positiveOffset = delta;
	m_negativeOffset = ( -delta / ( 1.0f + delta ) );
	m_threshold =  threshold;
	m_floatTweakType = TweakerUtils::MULTIPLICATIVE;
}

void DemoMouseTweaker::getCurrentTweakRectangleHeight(const MouseTweakerSettings& settings, float& yOut)
{
	hkStringBuf currentFullName;
	extArray<TweakerUtils::NameAndDisplayLocation> nameAndDisplayLocationOut;

	TweakerUtils::DisplayOptions options(&nameAndDisplayLocationOut, settings.m_hideMembers, 0, false );	// Don't *display* anything, just parse and fill out the locations
	TweakerUtils::displayData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, 0.0f, 0.0f, &options);

	// Get the last height
	yOut = 0.0f;
	if( nameAndDisplayLocationOut.getSize() != 0 )
	{
		yOut =  nameAndDisplayLocationOut.back().m_yLocation;
	}

}


hkBool DemoMouseTweaker::tweak(MouseTweakerSettings& settings)
{ 
	if( settings.m_demoEnvironment == HK_NULL )
	{
		HK_WARN_ALWAYS(0x6ee18950, "Tweaking disabled since m_demoEnvironment not set. Probably you want to set it to hkDemo::m_env");
		return false;
	}
	
	if( settings.m_tweakee == HK_NULL )
	{
		const int yOffset = 20; //getWindowHeight() - 400;
		const int xOffset = settings.m_demoEnvironment->m_window->getWidth() - 400;

		settings.m_demoEnvironment->m_textDisplay->outputText("Tweaking disabled since m_tweakee not set.\n Probably you want to set it to your\n demo 'settings' structure ", xOffset, yOffset );
		return false;
	}

	if( settings.m_klass == HK_NULL )
	{
		const int yOffset = 20; //getWindowHeight() - 400;
		const int xOffset = settings.m_demoEnvironment->m_window->getWidth() - 400;

		settings.m_demoEnvironment->m_textDisplay->outputText("Tweaking disabled since m_klass not set.\n Probably you want to set it to the\n 'class' instance of your demos\n 'settings' structure.", xOffset, yOffset );
		return false;
	}


	const int yOffset = settings.m_yOffset;
	const int xOffset = settings.m_demoEnvironment->m_window->getWidth() - settings.m_xOffset;


	const int fontWidth = (int)(settings.m_demoEnvironment->m_textDisplay->getFont()->getCharHeight() * 1.2f);

	const hkBool mouseTweaking = (HK_PLATFORM_IS_CONSOLE == 0);

	hkBool inTweakRectangle = true;
	hkBool left = false, right = false;

	if( mouseTweaking )
	{
		// Check left/right extents of mouse to see if in the tweak box
		{
			if( settings.m_demoEnvironment->m_window->getMouse().getPosX() < xOffset - 10)
			{
				// To the left of (allowable rectangle of) tweak display
				inTweakRectangle = false;
			}

			if( settings.m_demoEnvironment->m_window->getMouse().getPosX() > xOffset + settings.m_rectangleWidth )
			{
				// To the right of (allowable rectangle of) tweak display
				inTweakRectangle = false;
			}
		}

		if( inTweakRectangle )
		{
			hkStringBuf currentFullName;
			extArray<TweakerUtils::NameAndDisplayLocation> nameAndDisplayLocationOut;

			TweakerUtils::DisplayOptions options(&nameAndDisplayLocationOut, settings.m_hideMembers, 0xFFFF8040, false );	// Don't *display* anything, just parse and fill out the locations
			TweakerUtils::displayData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, (float) xOffset, (float) yOffset, &options);
			

			int yPos = settings.m_demoEnvironment->m_window->getHeight()- settings.m_demoEnvironment->m_window->getMouse().getPosY();

			// Brute force look at all the text strings and see which one we match. Do not assume they are in any order!
			int closestDistToLocation = 1000000;
			int index = -1;
			for(int i =0; i < nameAndDisplayLocationOut.getSize(); i++)
			{
				//hkcout << nameAndDisplayLocationOut[i].m_name << "\t\t\t" << nameAndDisplayLocationOut[i].yLocation << "\n";
				int distToLocation = yPos - (int)nameAndDisplayLocationOut[i].m_yLocation;
				if( distToLocation < closestDistToLocation
					&& distToLocation >= 0)
				{
					closestDistToLocation = distToLocation;
					index = i;
				}
			}

			// If we found a string and we're within fontwidth of it (check this in case we're now below the lowest displayed tweakable ie out side the bottom of the rectangle), we're good
			if( index != -1 && 
				(yPos - nameAndDisplayLocationOut[index].m_yLocation < fontWidth) )
			{
				//hkcout << nameAndDisplayLocationOut[index].m_name << "\t\t\t" << nameAndDisplayLocationOut[index].yLocation << "\n";
				settings.m_tweakName = nameAndDisplayLocationOut[index].m_name;


				left = settings.m_demoEnvironment->m_window->getMouse().wasButtonPressed( HKG_MOUSE_LEFT_BUTTON );
				right = settings.m_demoEnvironment->m_window->getMouse().wasButtonPressed( HKG_MOUSE_RIGHT_BUTTON );

				if ( right || left )
				{
					hkStringBuf oldPath (settings.m_tweakName.cString());
					settings.m_tweakName = TweakerUtils::getChildPath(settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass);

					// If we changed path, we're done
					if( !(settings.m_tweakName == oldPath) )
					{
						left = right = false;
						const_cast<hkgMouse&> (settings.m_demoEnvironment->m_window->getMouse()).setButtonState( 0 );
						settings.m_demoEnvironment->m_window->setMousePosition( settings.m_demoEnvironment->m_window->getMouse().getPosX(), settings.m_demoEnvironment->m_window->getMouse().getPosY() - fontWidth );
					}
				}
			}
		}
	}
	else	// keyboard or gamepad
	{
		int down, up; 

		down = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN );
		up = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP );
		left = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT );
		right = settings.m_demoEnvironment->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT );

		TweakerUtils::DisplayOptions options(HK_NULL, settings.m_hideMembers, 0xFFFF8040, true );

		TweakerUtils::displayData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, (float) xOffset, (float) yOffset, &options);

		if ( down )
		{
			settings.m_tweakName = TweakerUtils::getNextSiblingPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass );
			return false;
		}

		if( up )
		{
			settings.m_tweakName = TweakerUtils::getPrevSiblingPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass );
			return false;
		}

		if ( left )
		{
			settings.m_tweakName = TweakerUtils::getParentPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass);
			return false;
		}

		if ( right )
		{
			settings.m_tweakName = TweakerUtils::getChildPath( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass);
			return false;
		}

		
		if (settings.m_demoEnvironment->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1))
		{
			left = true;
		}

		if (settings.m_demoEnvironment->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_2))
		{
			right = true;
		}
		

	}

	hkBool tweaked = false;

	if ( left || right )
	{
		tweaked = true;

		hkBool neg = left;

		TweakAlgorithmSpec tweakAlgorithmSpec;

		// Put special tweaking here
		if( settings.m_getTweakAlgorithmSpecFunc )//settings.m_tweakName.endsWith("something"))
		{
			(*settings.m_getTweakAlgorithmSpecFunc)( settings.m_tweakName, tweakAlgorithmSpec);
		}

		if( tweakAlgorithmSpec.m_floatTweakType == TweakerUtils::NO_TWEAK )
		{
			return tweaked;
		}

		TweakerUtils::tweakData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, 
			!neg ? float(tweakAlgorithmSpec.m_positiveOffset): float(tweakAlgorithmSpec.m_negativeOffset), tweakAlgorithmSpec.m_threshold, tweakAlgorithmSpec.m_floatTweakType );

	}

	if( mouseTweaking )
	{
		TweakerUtils::DisplayOptions options(HK_NULL, settings.m_hideMembers, 0xffffff00, true );
		TweakerUtils::displayData( settings.m_tweakName.cString(), settings.m_tweakee, *settings.m_klass, *settings.m_demoEnvironment->m_textDisplay, 
			(float) xOffset, (float) yOffset, 
			&options );
	}

	return tweaked;
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
