/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GeometryLoaderUtil.h>

#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/System/Io/IArchive/hkIArchive.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/Math/Matrix/hkMatrix3Util.h>

#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Resource/hkResource.h>

#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/SceneData/SceneDataToGeometryConverter/hkxSceneDataToGeometryConverter.h>
#include <Common/GeometryUtilities/Misc/hkGeometryUtils.h>

#include <Common/Internal/ConvexHull/hkGeometryUtility.h>

#include <Graphics/Common/Geometry/hkgGeometry.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Common/Geometry/FaceSet/hkgFaceSet.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Material/hkgMaterial.h>

#include <Common/Base/KeyCode.h>
#ifdef HK_FEATURE_PRODUCT_PHYSICS_2012
#include <Physics2012/Collide/Shape/Compound/Collection/SimpleMesh/hkpSimpleMeshShape.h>
#include <Physics2012/Collide/Shape/Misc/Transform/hkpTransformShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTranslate/hkpConvexTranslateShape.h>
#include <Physics2012/Collide/Shape/Compound/Collection/List/hkpListShape.h>

#include <Physics2012/Dynamics/Entity/hkpRigidBody.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>

#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Collide/Shape/Convex/ConvexTransform/hkpConvexTransformShape.h>
#include <Physics2012/Collide/Shape/Compound/Tree/Mopp/hkpMoppBvTreeShape.h>
#include <Physics2012/Collide/Shape/Convex/ConvexVertices/hkpConvexVerticesShape.h>

namespace GeometryLoaderUtil
{
	// 
	typedef hkPointerMap< const hkpShape*, int > DoneShapesMap;

	// Convert from a shape to geometry (legacy)
	void HK_CALL load( const hkpShape* shape, const hkTransform& transform, hkGeometry& geometry );

	// Convert from a simple mesh shape to a geometry
	void	HK_CALL load( const hkpSimpleMeshShape* shape, const hkTransform& transform, hkGeometry& geometry);

	// Load from a shape into a collection of instanced geometries
	void	HK_CALL loadShapeRecursive( const hkpShape* shape, const hkTransform& transform, GeometryInstances& geometriesOut, DoneShapesMap& doneShapes, hkGeometry* currentRawGeometry = HK_NULL );

	int s_numConvexShapes;

}


#endif

//
void GeometryLoaderUtil::merge( const hkGeometry& input, hkGeometry& output )
{
	const int ibase = output.m_vertices.getSize();
	hkVector4* vertices = output.m_vertices.expandBy( input.m_vertices.getSize() );
	hkGeometry::Triangle* triangles = output.m_triangles.expandBy( input.m_triangles.getSize() );
	for( int i=0; i<input.m_vertices.getSize(); ++i )
	{
		vertices[i] = input.m_vertices[i];
	}
	for( int i=0;i<input.m_triangles.getSize();++i )
	{
		triangles[i].set(	input.m_triangles[i].m_a+ibase,
							input.m_triangles[i].m_b+ibase,
							input.m_triangles[i].m_c+ibase,
							input.m_triangles[i].m_material );
	}
}

//
void GeometryLoaderUtil::merge( const hkGeometry& input, const hkMatrix4& transform, hkGeometry& output )
{
	const int ibase = output.m_vertices.getSize();
	hkVector4* vertices = output.m_vertices.expandBy( input.m_vertices.getSize() );
	hkGeometry::Triangle* triangles = output.m_triangles.expandBy( input.m_triangles.getSize() );
	for( int i=0; i<input.m_vertices.getSize(); ++i )
	{
		transform.transformPosition( input.m_vertices[i], vertices[i] );
	}
	for( int i=0;i<input.m_triangles.getSize();++i )
	{
		triangles[i].set(	input.m_triangles[i].m_a+ibase,
							input.m_triangles[i].m_b+ibase,
							input.m_triangles[i].m_c+ibase,
							input.m_triangles[i].m_material );
	}
}

//
void GeometryLoaderUtil::scale(hkGeometry& geom,const hkVector4& s)
{
	for(int i=0;i<geom.m_vertices.getSize();++i)
	{
		geom.m_vertices[i].mul4(s);
	}
}

//
hkReal GeometryLoaderUtil::normalize(hkGeometry& geom, hkReal size , hkTransform* transformOut)
{
	if(geom.m_vertices.getSize()==0) 
	{
		return 0;
	}
	hkAabb	box;
	box.m_min=geom.m_vertices[0];
	box.m_max=geom.m_vertices[0];
	for(int i=1;i<geom.m_vertices.getSize();++i)
	{
		box.m_max.setMax4(box.m_max,geom.m_vertices[i]);
		box.m_min.setMin4(box.m_min,geom.m_vertices[i]);
	}
	hkVector4	ext;
	ext.setSub4(box.m_max,box.m_min);
	hkVector4	ctr;
	ctr.setAdd4(box.m_max,box.m_min);
	ctr.mul4(0.5f);
	hkReal		maxLength=ext(ext.getMajorAxis3());
	hkReal		scl=1/maxLength*size;
	for(int i=0;i<geom.m_vertices.getSize();++i)
	{
		geom.m_vertices[i].sub4(ctr);
		geom.m_vertices[i].mul4(scl);
	}
	if(transformOut)
	{
		*transformOut	=	hkTransform::getIdentity();
		hkMatrix3Util::_setDiagonal(scl,scl,scl, transformOut->getRotation());
		transformOut->getTranslation().setMul4(scl,ctr);
	}
	return scl;
}

//
struct hkBspLoader
{
	struct Directory
	{
		hkInt32	m_offset;
		hkInt32	m_length;
	};

	struct Texture
	{
		char	m_name[64];
		hkInt32	m_flags;
		hkInt32	m_contents;
	};

	struct Vertex
	{
		float	m_position[3];
		float	m_textcoord[2][2];
		float	m_normal[3];
		hkUint8	m_color[4];
	};

	struct Face
	{
		hkInt32	m_texture;
		hkInt32	m_effect;
		hkInt32	m_type;
		hkInt32	m_firstVertex;
		hkInt32	m_numVertices;
		hkInt32	m_firstMeshVertex;
		hkInt32	m_numMeshVertex;
		hkInt32	m_lightMap;
		hkInt32	m_lmStart[2];
		hkInt32	m_lmEnd[2];
		float	m_lmOrg[3];
		float	m_lmTng[2][3];
		float	m_normal[3];
		hkInt32	m_patchRes[2];
	};

	template <typename T>
	static void	readDirectoryContent(hkIArchive& stream, const Directory& directory, int elementSize, hkArray<T>& elements)
	{
		HK_ASSERT2(0, directory.m_length % elementSize == 0, "Invalid element size");
		HK_ASSERT2(0, hkSizeOf(T) >= elementSize, "Invalid element size");
		elements.setSize(directory.m_length / elementSize);
		if( hkSeekableStreamReader* sr = stream.getStreamReader()->isSeekTellSupported() )
		{
			sr->seek(directory.m_offset, hkSeekableStreamReader::STREAM_SET);
			if(sizeof(T) == elementSize)
			{
				stream.readRaw(elements.begin(), elementSize * elements.getSize());
			}
			else
			{
				for(int i=0; i<elements.getSize(); ++i) stream.readRaw(&elements[i], elementSize);
			}		
		}
		else
		{
			HK_ASSERT(0,0);
		}
	}

	static HK_FORCE_INLINE void evaluateQuadraticCurve(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkSimdRealParameter u, hkVector4& output)
	{
		hkVector4	u0; u0.setInterpolate(a, b, u);
		hkVector4	u1; u1.setInterpolate(b, c, u);
		output.setInterpolate(u0, u1, u);
	}

	static HK_FORCE_INLINE void	evaluateQuadraticPatch(const hkVector4 cps[3][3], hkSimdRealParameter u, hkSimdRealParameter v, hkVector4& output)
	{
		hkVector4	u0; evaluateQuadraticCurve(cps[0][0], cps[1][0], cps[2][0], u, u0);
		hkVector4	u1; evaluateQuadraticCurve(cps[0][1], cps[1][1], cps[2][1], u, u1);
		hkVector4	u2; evaluateQuadraticCurve(cps[0][2], cps[1][2], cps[2][2], u, u2);
		evaluateQuadraticCurve(u0, u1, u2, v, output);
	}

	static void	tesselateQuadraticPatch(const hkVector4 cps[3][3], int resX, int resY, int mat, hkGeometry& geometry)
	{
		const int	vi = geometry.m_vertices.getSize();
		hkVector4*	vtx = geometry.m_vertices.expandBy(resX*resY);
		
		for(int j=0; j<resY; ++j)
		{
			hkSimdReal	v; v.setFromFloat(j / (hkReal)(resY-1));
			for(int i=0; i<resX; ++i)
			{
				hkSimdReal	u; u.setFromFloat(i / (hkReal)(resX-1));
				evaluateQuadraticPatch(cps, u, v, vtx[j*resX+i]);
			}
		}

		for(int j=0; j<(resY-1); ++j)
		{
			for(int i=0; i<(resX-1); ++i)
			{
				const int i00 = vi + (j * resX + i);
				const int i10 = vi + (j * resX + i + 1);
				const int i11 = vi + ((j + 1) * resX + i + 1);
				const int i01 = vi + ((j + 1) * resX + i);
				geometry.m_triangles.expandOne().set(i00, i10, i11, 0);
				geometry.m_triangles.expandOne().set(i00, i11, i01, 0);
			}
		}
	}
	
	static bool	load(hkIArchive& stream, hkGeometry& geometry)
	{
		// Load header and directories.
		const int	numDirs = 17;
		Directory	directories[numDirs];
		{
			char	magic[4]; stream.readArrayGeneric(magic,1,4);
			int		version = stream.read32();
			if(version < 0x2e) return false;
			for(int i=0; i<numDirs; ++i)
			{
				directories[i].m_offset = stream.read32();
				directories[i].m_length = stream.read32();
			}
		}

		// Load content.
		hkArray<Texture>	textures;	readDirectoryContent(stream, directories[1], 72, textures);
		hkArray<Vertex>		vertices;	readDirectoryContent(stream, directories[10], 44, vertices);
		hkArray<hkInt32>	triangles;	readDirectoryContent(stream, directories[11], 4, triangles);
		hkArray<Face>		faces;		readDirectoryContent(stream, directories[13], 104, faces);		

		// Generate geometry.
		geometry.clear();

		// Generate vertices.
		hkSimdReal	scale; scale.setFromFloat(0.03f);
		geometry.m_vertices.setSize(vertices.getSize());
		for(int i=0; i<vertices.getSize(); ++i)
		{
			geometry.m_vertices[i].set((hkReal)vertices[i].m_position[0], (hkReal)vertices[i].m_position[1], (hkReal)vertices[i].m_position[2], 0.0f);
			geometry.m_vertices[i].mul(scale);
		}

		// Generate triangulation.
		for(int i=0; i<faces.getSize(); ++i)
		{
			const Face&		face = faces[i];
			const int		baseIndex = face.m_firstVertex;
			
			if(face.m_texture >= 0 && face.m_texture < textures.getSize())
			{
				const Texture&	texture = textures[face.m_texture];
				if(texture.m_flags & 4)			continue;
				if(texture.m_flags == 0x4c20)	continue;
			} else continue;

			switch(face.m_type)
			{
			case 1:
			case 3:
				{
					if(face.m_numMeshVertex)
					{
						const hkInt32*	indices = &triangles[face.m_firstMeshVertex];
						for(int fi=0; fi<face.m_numMeshVertex; fi+=3)
						{
							geometry.m_triangles.expandOne().set(baseIndex+indices[fi], baseIndex+indices[fi+2], baseIndex+indices[fi+1], face.m_texture);
						}
					}
					else
					{		
						HK_WARN(0, "Untriangulated faces not supported");
					}
				}
				break;
			case 2:
				{
					const hkVector4*	cps = &geometry.m_vertices[face.m_firstVertex];
					for(int y=0; y < (face.m_patchRes[1]-1); y += 2)
					{
						for(int x=0; x < (face.m_patchRes[0]-1); x += 2)
						{
							hkVector4	patch[3][3];
							#define GET_CP(_i_,_j_) cps[(y + (_j_)) * face.m_patchRes[0] + x + (_i_)]
							patch[0][0] = GET_CP(0,0); patch[1][0] = GET_CP(1,0); patch[2][0] = GET_CP(2,0);
							patch[0][1] = GET_CP(0,1); patch[1][1] = GET_CP(1,1); patch[2][1] = GET_CP(2,1);
							patch[0][2] = GET_CP(0,2); patch[1][2] = GET_CP(1,2); patch[2][2] = GET_CP(2,2);
							#undef GET_CP
							tesselateQuadraticPatch(patch, 8, 8, face.m_texture, geometry);
						}
					}
				}
				break;
			default:
				HK_WARN(0, "Face type "<<face.m_type<<" not implemented.");
			}
		}
		
		return true;
	}

};

//
bool GeometryLoaderUtil::load(const char* filename, hkGeometry& geometry, hkAabb* aabbOut)
{
	hkStringBuf assetFile(filename); hkAssetManagementUtil::getFilePath(assetFile);	
	geometry.clear();
	if(assetFile.endsWith(".bsp"))
	{
		// Bsp file.
		hkIArchive	stream(filename);
		if(stream.isOk())
		{
			if(hkBspLoader::load(stream, geometry))
			{
				hkGeometryUtils::weldVertices(geometry, 0.0f);
				
				hkAabb	aabb; aabb.setEmpty();
				for(int i=0; i<geometry.m_vertices.getSize(); ++i) aabb.includePoint(geometry.m_vertices[i]);
				
				hkVector4	center; aabb.getCenter(center);
				for(int i=0; i<geometry.m_vertices.getSize(); ++i) geometry.m_vertices[i].sub(center);
			}
		}
	}
	else
	{
		// Assume Havok asset.
		hkResource* data = hkSerializeUtil::load(assetFile.cString(), HK_NULL, hkSerializeUtil::LOAD_FORCED);
		if(data)
		{
			GeometryLoaderUtil::load(data, geometry);
			data->removeReference();
		}
	}

	hkAabb		aabb; aabb.setEmpty();
	for(int i=0; i<geometry.m_vertices.getSize(); ++i) aabb.includePoint(geometry.m_vertices[i]);
	hkVector4	center; aabb.getCenter(center);
	for(int i=0; i<geometry.m_vertices.getSize(); ++i) geometry.m_vertices[i].sub(center);
	if(aabbOut)
	{
		aabbOut->m_min.setSub(aabb.m_min, center);
		aabbOut->m_max.setSub(aabb.m_max, center);
	}

	return geometry.m_triangles.getSize() > 0;
}

//
#ifdef HK_FEATURE_PRODUCT_PHYSICS_2012
void GeometryLoaderUtil::load(const hkpSimpleMeshShape* shape, const hkTransform& transform, hkGeometry& geometry)
{
	hkGeometry	geom;
	geom.m_vertices.setSize(shape->m_vertices.getSize());
	geom.m_triangles.setSize(shape->m_triangles.getSize());
	for(int i=0;i<geom.m_vertices.getSize();++i)
	{
		geom.m_vertices[i]._setTransformedPos(transform, shape->m_vertices[i]);
	}
	for(int i=0;i<geom.m_triangles.getSize();++i)
	{
		geom.m_triangles[i].m_a	=	shape->m_triangles[i].m_a;
		geom.m_triangles[i].m_b	=	shape->m_triangles[i].m_b;
		geom.m_triangles[i].m_c	=	shape->m_triangles[i].m_c;
	}
	merge(geom,geometry);
}

void GeometryLoaderUtil::load( const hkpShape* shape, const hkTransform& transform, hkGeometry& geometry)
{

}

void GeometryLoaderUtil::loadShapeRecursive( const hkpShape* shape, const hkTransform& transform, GeometryInstances& geometryInstances, GeometryLoaderUtil::DoneShapesMap& doneShapes, hkGeometry* currentRawGeometry )
{
	// Recurse into wrappers
	switch( shape->getType() )
	{
	case hkcdShapeType::TRANSFORM:
		{
			const hkpTransformShape* trs = static_cast<const hkpTransformShape* >(shape);
			hkTransform	finalTransform; finalTransform.setMul(transform, trs->getTransform());
			loadShapeRecursive( trs->getChildShape(), finalTransform, geometryInstances, doneShapes, currentRawGeometry );
		}
		return;

	case hkcdShapeType::CONVEX_TRANSFORM:
		{
			const hkpConvexTransformShape* trs = static_cast<const hkpConvexTransformShape* >(shape);
			hkTransform unscaledTransform; trs->getTransform( &unscaledTransform );
			hkTransform	finalTransform; finalTransform.setMul(transform, unscaledTransform);
			loadShapeRecursive( trs->getChildShape(), finalTransform, geometryInstances, doneShapes, currentRawGeometry );
		}
		return;

	case hkcdShapeType::CONVEX_TRANSLATE:
		{
			const hkpConvexTranslateShape* trs = static_cast<const hkpConvexTranslateShape* >(shape);
			hkTransform baseTransform = hkTransform::getIdentity();
			baseTransform.setTranslation(trs->getTranslation());
			hkTransform	finalTransform; finalTransform.setMul(transform, baseTransform);
			loadShapeRecursive( trs->getChildShape(), finalTransform, geometryInstances, doneShapes, currentRawGeometry );
		}
		return;

	case hkcdShapeType::MOPP:
		{
			const hkpMoppBvTreeShape* mopp  = static_cast< const hkpMoppBvTreeShape* >( shape );
			const hkpShapeCollection* collection = mopp->getShapeCollection();
			loadShapeRecursive( collection, transform, geometryInstances, doneShapes, currentRawGeometry );
		}
		return;

	default:
		break;
	}

	// Containers
	const hkpShapeContainer* container = shape->getContainer();
	if( container )
	{
		// Create a new instance
		GeometryInstances::Instance& instance = geometryInstances.m_instances.expandOne();
		instance.m_worldFromLocal.set(transform);
		instance.m_geometryIdx = doneShapes.getWithDefault( shape, -1 );
		if( instance.m_geometryIdx == -1 )
		{
			// Create a new geometry
			instance.m_geometryIdx = geometryInstances.m_geometries.getSize();
			currentRawGeometry = &geometryInstances.m_geometries.expandOne();
			doneShapes.insert( shape, instance.m_geometryIdx );

			hkpShapeBuffer buffer;
			for( hkpShapeKey key = container->getFirstKey(); key!=HK_INVALID_SHAPE_KEY; key=container->getNextKey(key) )
			{
				const hkpShape* childShape = container->getChildShape( key, buffer );
				if( (void*)childShape == &buffer )
				{
					// Shape created on-the-fly, need to add it to the raw geom
					loadShapeRecursive( childShape, hkTransform::getIdentity(), geometryInstances, doneShapes, currentRawGeometry );
				}
				else
				{
					// Shape stored elsewhere, check if its done already etc
					loadShapeRecursive( childShape, hkTransform::getIdentity(), geometryInstances, doneShapes, HK_NULL );
				}
			}
		}
		return;
	}

	// Leaf shapes
	switch( shape->getType() )
	{
		case hkcdShapeType::TRIANGLE:
			{
				// Append to the raw geometry
				HK_ASSERT( 0x0, currentRawGeometry );
				const hkpTriangleShape* tri = static_cast<const hkpTriangleShape* >(shape);
				int baseIndex = currentRawGeometry->m_vertices.getSize();
				hkVector4* vertices = currentRawGeometry->m_vertices.expandBy(3);
				vertices[0]._setTransformedPos(transform, tri->getVertex(0));
				vertices[1]._setTransformedPos(transform, tri->getVertex(1));
				vertices[2]._setTransformedPos(transform, tri->getVertex(2));
				hkGeometry::Triangle& t = currentRawGeometry->m_triangles.expandOne();
				t.set(baseIndex, baseIndex+1, baseIndex+2);
			}
			break;

		case hkcdShapeType::CONVEX_VERTICES:
			{
				const hkpConvexVerticesShape* cvx = static_cast<const hkpConvexVerticesShape* >(shape);
				if( currentRawGeometry )
				{
					// Append to the current raw geometry
					hkArray<hkVector4> vertices; cvx->getOriginalVertices(vertices);
					hkArray<hkVector4> trsVertices(vertices.getSize());
					for(int i=0; i<vertices.getSize(); ++i)
					{
						trsVertices[i]._setTransformedPos( transform, vertices[i] );
					}
					
					hkGeometry geom;
					hkGeometryUtility::createConvexGeometry( trsVertices, geom );
					merge( geom, hkMatrix4::getIdentity(), *currentRawGeometry );
				}
				else
				{
					// Create new instance
					GeometryInstances::Instance& instance = geometryInstances.m_instances.expandOne();
					instance.m_worldFromLocal.set(transform);
					instance.m_geometryIdx = doneShapes.getWithDefault( shape, -1 );
					if( instance.m_geometryIdx == -1 )
					{
						// Create new geometry
						instance.m_geometryIdx = geometryInstances.m_geometries.getSize();
						hkGeometry& geom = geometryInstances.m_geometries.expandOne();
						doneShapes.insert( shape, instance.m_geometryIdx );

						hkArray<hkVector4> vertices;
						cvx->getOriginalVertices(vertices);
						hkGeometryUtility::createConvexGeometry( vertices, geom );
						
						s_numConvexShapes++;
					}
				}
			}
			break;

		default:
			HK_WARN_ALWAYS(0x64900302, "Shape type " << shape->getType() << " conversion to geometry not implemented");
			break;
	}
}
#endif

//
bool GeometryLoaderUtil::load( const hkRootLevelContainer* level, GeometryInstances& geometriesOut )
{
	#ifdef HK_FEATURE_PRODUCT_PHYSICS_2012
	DoneShapesMap doneShapes;
	s_numConvexShapes = 0;
	#endif
	for( int varIdx=0; varIdx<level->m_namedVariants.getSize(); ++varIdx )
	{
		const hkRootLevelContainer::NamedVariant& variant = level->m_namedVariants[varIdx];
		if( hkString::strCasecmp( variant.getName(), "Scene Data" ) == 0 )
		{
			// Use the scene data converter
			hkxScene* scene = static_cast<hkxScene*>( variant.getObject() );
			hkLocalArray<hkxMaterial*> materials(8);
			hkxSceneDataToGeometryConverter::convertToGeometryInstances( scene, scene->m_rootNode, geometriesOut, materials );
		}
		else if( hkString::strCasecmp( variant.getName(), "hkpPhysicsData" ) == 0 )
		{
#ifdef HK_FEATURE_PRODUCT_PHYSICS_2012
			// Examine each top level shape in turn
			hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( variant.getObject() );
			const hkArray<hkpPhysicsSystem*>& systems = physicsData->getPhysicsSystems();
			for( int i=0; i<systems.getSize(); ++i )
			{
				const hkArray<hkpRigidBody*>& rbs = systems[i]->getRigidBodies();
				for( int j=0; j<rbs.getSize(); ++j )
				{
					const hkpCollidable* collidable = rbs[j]->getCollidable();
					if( collidable )
					{
						const hkpShape*	shape = collidable->getShape();
						if( shape )
						{
							loadShapeRecursive( shape, rbs[j]->getTransform(), geometriesOut, doneShapes );
						}
					}
				}
			}
#else
			HK_WARN_ALWAYS(0x506D2D9B, "Physics data found but not loaded.");
#endif
		}
	}
	return(true);
}

//
void GeometryLoaderUtil::load( const hkResource* resource, hkGeometry& geometry )
{
	GeometryInstances geometryInstances;
	load( resource, geometryInstances );
	HK_ASSERT( 0x0, geometryInstances.m_geometries.getSize() > 0 );
	for( int i=0; i<geometryInstances.m_instances.getSize(); ++i )
	{
		merge(	geometryInstances.m_geometries[ geometryInstances.m_instances[i].m_geometryIdx ],
				geometryInstances.m_instances[i].m_worldFromLocal,
				geometry );
	}
}

//
void GeometryLoaderUtil::load( const hkResource* resource, GeometryLoaderUtil::GeometryInstances& geometriesOut )
{
	hkStringBuf	type( resource->getContentsTypeName() );
	if(type == "hkRootLevelContainer")
	{
		load( resource->getContents<hkRootLevelContainer>(), geometriesOut );
	}
	else if(type == "hkGeometry")
	{
		hkGeometry& geometry = geometriesOut.m_geometries.expandOne();
		GeometryInstances::Instance& instance = geometriesOut.m_instances.expandOne();
		instance.m_geometryIdx = geometriesOut.m_geometries.getSize() - 1;
		instance.m_worldFromLocal = hkMatrix4::getIdentity();

		geometry.appendGeometry(*resource->getContents<hkGeometry>());
	}
	else if(type == "hkpSimpleMeshShape")
	{
#ifdef HK_FEATURE_PRODUCT_PHYSICS_2012
		hkGeometry& geometry = geometriesOut.m_geometries.expandOne();
		GeometryInstances::Instance& instance = geometriesOut.m_instances.expandOne();
		instance.m_geometryIdx = 0;
		instance.m_worldFromLocal = hkMatrix4::getIdentity();
		load( resource->getContents<hkpSimpleMeshShape>(), hkTransform::getIdentity(), geometry );
#else
		HK_WARN_ALWAYS(0x506D2D9B, "Physics data found but not loaded.");
#endif
	}	
}


hkgGeometry* GeometryLoaderUtil::createGeometry(const hkGeometry& mesh, hkgDisplayContext* context, const hkVector4& color)
{
	hkgGeometry* displayGeom = 0;
 
	// Create geometry for a triangle mesh
	if (mesh.m_triangles.getSize() > 0)
	{
		context->lock();
		displayGeom=hkgGeometry::create();
		const int				nTri		=	mesh.m_triangles.getSize();
		hkgMaterialFaceSet*		fs			=	hkgMaterialFaceSet::create();
		hkgFaceSet*				faceSet		=	hkgFaceSet::create(context);
		hkgVertexSet*			vs			=	hkgVertexSet::create(context);
		faceSet->setVertexSet(vs);
		vs->release();

		fs->addFaceSet(faceSet);faceSet->release();

		displayGeom->addMaterialFaceSet(fs);fs->release();

		vs->setNumVerts(nTri*3,HKG_VERTEX_FORMAT_POS|HKG_VERTEX_FORMAT_NORMAL);
		vs->lock(HKG_LOCK_WRITEDISCARD);

		int vertsAdded=0;

		for(int i=0;i<nTri;i++)
		{
			const hkGeometry::Triangle&	tri=mesh.m_triangles[i];
			const hkVector4 v[]={	mesh.m_vertices[tri.m_a],
				mesh.m_vertices[tri.m_b],
				mesh.m_vertices[tri.m_c]};
			hkVector4 	ab,ac,n;
			ab.setSub4(v[1],v[0]);
			ac.setSub4(v[2],v[0]);
			n.setCross(ab,ac);
			n.normalize3();
			for(int j=0;j<3;j++)
			{
				vs->setVertexComponentData(HKG_VERTEX_COMPONENT_POS,vertsAdded,&v[j]);
				vs->setVertexComponentData(HKG_VERTEX_COMPONENT_NORMAL,vertsAdded,&n);
				vertsAdded++;
			}	
		}
		vs->unlock();
		hkgFaceSetPrimitive* hkgprim = new hkgFaceSetPrimitive(HKG_TRI_LIST);	
		hkgprim->setIndices16(HK_NULL,nTri*3);
		hkgprim->setVertexBufferStartIndex(0);
		faceSet->addPrimitive(hkgprim);
		faceSet->initialize();
		context->unlock();

		hkgMaterial* mat=hkgMaterial::create();
		mat->setDiffuseColor(float(color(0)),float(color(1)),float(color(2)),float(color(3)));
		displayGeom->setMaterialOnAll(mat);
		mat->release();
	}

	// Create geometry for convex vertices
	else
	{
		hkGeometry convexGeometry;
		hkGeometryUtility::createConvexGeometry(mesh.m_vertices, convexGeometry);		
		displayGeom = createGeometry(convexGeometry, context, color);
	}

	return(displayGeom);
}


hkgDisplayObject* GeometryLoaderUtil::createDisplay(const hkArray<const hkGeometry*>& meshes,hkgDisplayWorld* display,hkgDisplayContext* context, const hkVector4* color)
{
	hkArray<hkgGeometry*>	displayGeoms;
	hkPseudoRandomGenerator	rnd(1806);
	for(int i=0;i<meshes.getSize();++i)
	{
		hkVector4 meshColor;
		if (color)
		{
			meshColor = *color;
		}
		else
		{		
			hkVector4		rndColor;
			hkVector4		white(1,1,1,1);		
			rnd.getRandomVector01(rndColor);
			meshColor.setInterpolate4(rndColor,white,0.7f);
			meshColor.mul4(0.8f);
			meshColor(3)=1;
		}
		hkgGeometry*	displayGeom=createGeometry(*meshes[i],context,meshColor);
		if(displayGeom)
		{
			displayGeoms.pushBack(displayGeom);
		}
	}
	hkgDisplayObject*	displayObject=0;
	if(displayGeoms.getSize())
	{
		displayObject=hkgDisplayObject::create();
		for(int i=0;i<displayGeoms.getSize();++i)
		{
			displayObject->addGeometry(displayGeoms[i]);
			displayGeoms[i]->removeReference();
		}			
		displayObject->setStatusFlags(HKG_DISPLAY_OBJECT_STATIC|HKG_DISPLAY_OBJECT_NOT_CULLED|HKG_DISPLAY_OBJECT_SHADOW);
		if (display)
		{
			display->addDisplayObject(displayObject);
		}		
	}
	return(displayObject);
}

//
hkgDisplayObject* GeometryLoaderUtil::createDisplay(const hkGeometry& mesh,hkgDisplayWorld* display,hkgDisplayContext* context,const hkVector4& color)
{
	hkgGeometry*		displayGeom=createGeometry(mesh,context,color);
	hkgDisplayObject*	displayObject=0;
	if(displayGeom)
	{
		displayObject=hkgDisplayObject::create();
		displayObject->addGeometry(displayGeom);displayGeom->removeReference();
		displayObject->setStatusFlags(HKG_DISPLAY_OBJECT_STATIC|HKG_DISPLAY_OBJECT_NOT_CULLED|HKG_DISPLAY_OBJECT_SHADOW);
		if (display)
		{
			display->addDisplayObject(displayObject);
		}		
	}
	return(displayObject);
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
