/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#ifndef HK_DEMOCOMMON_DRAWUTILS_H
#define HK_DEMOCOMMON_DRAWUTILS_H

#include <Common/Visualize/hkDebugDisplay.h>

struct DrawUtils
{
	//+hk.MemoryTracker(ignore=True)
	static void HK_CALL drawCircle(const hkVector4& o,const hkVector4& d,hkReal r,hkColor::Argb color);
	static void HK_CALL drawSphere(const hkVector4& o,hkReal r,hkColor::Argb color,int depth=3);
	static void HK_CALL drawArc(const hkVector4& o,hkReal r,const hkVector4& a,const hkVector4& b,const hkVector4& c,int depth,hkColor::Argb color);
	static void HK_CALL drawArrowWithScaledHead(const hkVector4& startPos, const hkVector4& endPos, hkSimdRealParameter headScale, const hkColor::Argb color);
};

struct hkDrawBase
{
	//+hk.MemoryTracker(ignore=True)
	static void HK_CALL drawPoint(hkVector4Parameter a, hkColor::Argb color);
	static void HK_CALL drawLine(hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color);
	static void HK_CALL wireTriangle(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color);
	static void HK_CALL fillTriangle(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color);
	static void HK_CALL fillRectangle(hkVector4Parameter center, hkVector4Parameter x, hkVector4Parameter y, hkColor::Argb color);
	static void HK_CALL fillQuad(hkVector4Parameter bl, hkVector4Parameter br, hkVector4Parameter tr, hkVector4Parameter tl, hkColor::Argb color);
	static void HK_CALL wireRectangle(hkVector4Parameter center, hkVector4Parameter x, hkVector4Parameter y, hkColor::Argb color);
	static void HK_CALL pushOffset( hkVector4Parameter off );
	static void HK_CALL popOffset();
	static hkColor::Argb darken( hkColor::Argb c );

	static hkVector4 s_x2d;
	static hkVector4 s_y2d;
	static hkVector4 s_n2d;

	struct Vector : public hkVector4
	{
		Vector( hkVector4Parameter v4 ) : hkVector4(v4) {}
		Vector( hkVector2 v2 )
		{
			setMul4( hkSimdReal(v2.x), s_x2d );
			addMul4( hkSimdReal(v2.y), s_y2d );
		}
	};
	typedef const Vector& VectorParameter;
};

struct hkDrawRectangle : public hkDrawBase
{
	//+hk.MemoryTracker(ignore=True)
	hkDrawRectangle()
	{
		m_center.setZero4();
		m_x = s_x2d;
		m_y = s_y2d;
	}

	hkDrawRectangle& fillColor( hkColor::Argb c )
	{
		if( !m_x.equals3(m_y, 1e-5f) )
		{
			fillRectangle(m_center, m_x, m_y, c);
		}
		return *this;
	}
	hkDrawRectangle& wireFrame( hkColor::Argb c )
	{
		wireRectangle(m_center, m_x, m_y, c);
		return *this;
	}

	hkDrawRectangle& center( hkVector4Parameter p )
	{
		m_center = p;
		return *this;
	}
	hkDrawRectangle& axes( hkVector4Parameter x, hkVector4Parameter y )
	{
		m_y = x;
		m_x = y;
		return *this;
	}
	hkDrawRectangle& expand( hkReal e )
	{
		hkReal lx = m_x.fastNormalizeWithLength3();
		m_x.mul4( lx + e );
		hkReal ly = m_y.fastNormalizeWithLength3();
		m_y.mul4( ly + e );
		return *this;
	}
	hkDrawRectangle& sideLengths( hkReal s0, hkReal s1 )
	{
		m_x.fastNormalize3(); m_x.mul4( s0 );
		m_y.fastNormalize3(); m_y.mul4( s1 );
		return *this;
	}
	hkDrawRectangle& sideSquare( hkReal s )
	{
		return sideLengths(s,s);
	}
	hkDrawRectangle& thickLine3d( hkVector4Parameter v0, hkVector4Parameter v1, hkVector4Parameter n, hkReal w);

	hkDrawRectangle& thickLine2d( VectorParameter v0, VectorParameter v1, hkReal w)
	{
		hkVector4 n; n = s_n2d;
		return thickLine3d(v0, v1, n, w);
	}

	hkVector4 m_center;
	hkVector4 m_y;
	hkVector4 m_x;
};

struct hkDrawLine : public hkDrawBase
{
	//+hk.MemoryTracker(ignore=True)
	hkDrawLine& ray( hkVector4Parameter o, hkVector4Parameter d, hkColor::Argb c )
	{
		hkVector4 p; p.setAdd4( o, d );
		drawLine( o, p, c);
		return *this;
	}
	hkDrawLine& ray( hkVector4Parameter o, hkVector4Parameter d, hkReal t, hkColor::Argb c )
	{
		hkVector4 p; p.setAddMul4( o, d, t );
		drawLine( o, p, c);
		return *this;
	}
	hkDrawLine& points( hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb c )
	{
		drawLine( a, b, c);
		return *this;
	}
	hkDrawLine& points( hkVector4Parameter a, hkVector4Parameter b, hkReal t, hkColor::Argb c )
	{
		hkVector4 p; p.setInterpolate4(a,b,t);
		drawLine( a, p, c);
		return *this;
	}
	hkDrawLine& arrow( hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb c )
	{
		hkVector4 n2d; n2d = s_n2d;
		drawLine( a, b, c);
		hkVector4 toB; toB.setSub4(b,a);
		hkVector4 perp; perp.setCross(toB,n2d);
		for( int i = 0; i < 2 ; ++i )
		{
			hkVector4 arm; arm.setAddMul4(b,perp,0.1f); arm.setAddMul4(arm, toB, -0.2f);
			drawLine(b, arm, c);
			perp.setNeg3(perp);
		}
		return *this;
	}

	hkDrawLine& thickLine2d( hkVector4Parameter a, hkVector4Parameter b, hkReal w, hkColor::Argb c )
	{
		hkDrawRectangle().thickLine2d(a,b,w).fillColor(c);
		return *this;
	}

	hkDrawLine& thickLine2d( const hkVector2& a, const hkVector2& b, hkReal w, hkColor::Argb c )
	{
		hkVector4 p1; p1.setMul4( hkSimdReal(a.x), s_x2d ); p1.addMul4( a.y, s_y2d );
		hkVector4 p2; p2.setMul4( hkSimdReal(b.x), s_x2d ); p2.addMul4( b.y, s_y2d );

		hkDrawRectangle().thickLine2d(p1,p2,w).fillColor(c);
		return *this;
	}

	hkDrawLine& thickArrow2d( const hkVector2& a, const hkVector2& b, hkReal w, hkColor::Argb c );
	hkDrawLine& thickArrow3d( const hkVector4& a, const hkVector4& b, const hkVector4& normal, hkReal w, hkColor::Argb c );

	hkDrawLine& linestrip( const hkVector2* p, int np, hkColor::Argb c )
	{
		for (int i = 0; i < np - 1; ++i)
		{
			hkVector4 p1; p1.setMul4( hkSimdReal(p[i].x),   s_x2d );  p1.addMul4( p[i].y,  s_y2d );
			hkVector4 p2; p2.setMul4( hkSimdReal(p[i+1].x), s_x2d ); p2.addMul4( p[i+1].y, s_y2d );
			drawLine( p1, p2, c );
		}
		return *this;
	}
	hkDrawLine& lineloop( const hkVector2* p, int np, hkColor::Argb c )
	{
		linestrip(p,np,c);
		hkVector4 p1; p1.setMul4( hkSimdReal(p[np-1].x), s_x2d ); p1.addMul4( p[np-1].y,  s_y2d );
		hkVector4 p2; p2.setMul4( hkSimdReal(p[0].x), s_x2d ); p2.addMul4( p[0].y, s_y2d );
		drawLine( p1, p2, c );
		return *this;
	}
	hkDrawLine& thickLines3d( const hkVector4* verts, int numVerts, hkBool32 loop, hkVector4Parameter n, hkReal w, hkColor::Argb color );
	hkDrawLine& thickLines3d( const hkVector4* verts, const hkVector4* normals, int numVerts, hkReal w, hkColor::Argb color );
	hkDrawLine& thickLine3d( const hkVector4& start, const hkVector4& end, const hkVector4& normal, hkReal w, hkColor::Argb color );
};

struct hkDrawTriangle : public hkDrawBase
{
	//+hk.MemoryTracker(ignore=True)
	hkDrawTriangle& points( const hkVector4* p )
	{
		m_a = p[0];
		m_b = p[1];
		m_c = p[2];
		return *this;
	}

	hkDrawTriangle& points( hkVector4Parameter p0, hkVector4Parameter p1, hkVector4Parameter p2 )
	{
		m_a = p0;
		m_b = p1;
		m_c = p2;
		return *this;
	}
	hkDrawTriangle& fillColor( hkColor::Argb c )
	{
		fillTriangle(m_a, m_b, m_c, c);
		return *this;
	}
	hkDrawTriangle& wireFrame( hkColor::Argb c )
	{
		wireTriangle(m_a, m_b, m_c, c);
		return *this;

	}
	hkDrawTriangle& expand( hkReal r );

	hkVector4 m_a;
	hkVector4 m_b;
	hkVector4 m_c;
};

struct hkDrawPoint : public hkDrawBase
{
	//+hk.MemoryTracker(ignore=True)
	hkDrawPoint& point( hkVector4Parameter p, hkColor::Argb c )
	{
		drawPoint(p,c);
		return *this;
	}
	hkDrawPoint& wireCross( hkVector4Parameter p, hkReal rad, hkColor::Argb c );
	hkDrawPoint& wirePip( hkVector4Parameter p, hkReal rad, hkColor::Argb c );
	hkDrawPoint& fillPip( VectorParameter p, hkReal rad, hkColor::Argb c );
	hkDrawPoint& circle3d( hkVector4Parameter p, hkReal rad, hkVector4Parameter norm, hkColor::Argb c );
	hkDrawPoint& circle3d( hkVector4Parameter p, hkReal rad, hkVector4Parameter norm, hkVector4Parameter startPerp, hkReal start, hkReal end, int numSegments, hkColor::Argb c );
	hkDrawPoint& wireCircle3d( hkVector4Parameter p, hkReal rad, hkVector4Parameter norm, hkColor::Argb c );
	hkDrawPoint& circle2d( hkVector4Parameter p, hkReal rad, hkColor::Argb c )
	{
		return circle3d(p,rad,hkVector4(s_n2d),c);
	}
	hkDrawPoint& arc2d( hkVector4Parameter p1, hkVector4Parameter p2, hkVector4Parameter center, hkReal rad, hkColor::Argb c );
};

namespace hkDrawAnyConvert
{
	inline void toVector4( const hkVector2& p, hkVector4& out, int id )
	{
		hkVector4 t;
		t.setMul4( hkSimdReal(p.x), hkDrawBase::s_x2d );
		t.addMul4( hkSimdReal(p.y), hkDrawBase::s_y2d );
		t.setInt24W( id );
		out = t;
	}
	inline void toVector4( hkVector4Parameter p, hkVector4& out, int id )
	{
		out = p;
		out.setInt24W(id);
	}
}

struct hkDrawAny : public hkDrawBase
{
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkDrawAny);
	hkDrawAny();
	~hkDrawAny();

	template<typename VEC> hkDrawAny& points( const VEC& p0 );
	template<typename VEC> hkDrawAny& points( const VEC& p0, const VEC& p1 );
	template<typename VEC> hkDrawAny& points( const VEC& p0, const VEC& p1, const VEC& p2 );
	template<typename VEC> hkDrawAny& points( const VEC& p0, const VEC& p1, const VEC& p2, const VEC& p3 );
	template<typename VEC> hkDrawAny& points( const VEC* src, int n, int stride=sizeof(VEC) );

	template<typename VEC> hkDrawAny& indexed( const VEC* v, int i0 );
	template<typename VEC> hkDrawAny& indexed( const VEC* v, int i0, int i1 );
	template<typename VEC> hkDrawAny& indexed( const VEC* v, int i0, int i1, int i2 );
	template<typename VEC> hkDrawAny& indexed( const VEC* v, int i0, int i1, int i2, int i3 );
	template<typename VEC> hkDrawAny& indexed( const VEC* src, const int* indices, int n );

	template<typename VEC> hkDrawAny& offsets( const VEC& base, const VEC& off0 );

		/// Append the first vertex.
	hkDrawAny& closeLoop();
		/// Replace all points with octahedrons.
	hkDrawAny& octahedron( hkReal topRad, hkReal sideRad=-1);
	hkDrawAny& square( hkReal rad );
	hkDrawAny& diamond( hkReal rad );

	hkDrawAny& pushOffset( hkReal d, int axis );
	hkDrawAny& popOffset();

	hkDrawAny& scale( hkReal factor );

	hkDrawAny& fillDart( hkReal rad, hkColor::Argb color );
	hkDrawAny& label( hkColor::Argb c, const char* txt = HK_NULL );
	hkDrawAny& stroke( hkColor::Argb c, const hkReal rad = 0 );
	hkDrawAny& fill( hkColor::Argb c );
	hkDrawAny& fillSquare( hkReal rad, hkColor::Argb c );
	hkDrawAny& thickLine( hkReal w, hkColor::Argb c );
	hkDrawAny& thickArrow( hkReal w, hkColor::Argb c );

	hkArray<hkVector4> m_pts;
	hkArray<int> m_polys;
	int m_offsetPops;
};

#include <Demos/DemoCommon/Utilities/Drawing/DrawUtils.inl>

#endif //HK_DEMOCOMMON_DRAWUTILS_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
