/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Drawing/DrawUtils.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Math/Vector/hkVector4Util.h>
#include <Common/Base/Algorithm/Collide/LineSegment/hkLineSegmentUtil.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h> //remove
#include <Common/Base/Math/Quaternion/hkQuaternionUtil.h>

// drawing offsets - like glPushMatrix, glPopMatrix
static hkVector4 s_drawingOffset;
static hkVector4 s_drawingOffsetStack[8];
static int s_drawingOffsetIndex;

/*static*/ void HK_CALL DrawUtils::drawCircle(const hkVector4& o,const hkVector4& d,hkReal r,hkColor::Argb color)
{
	hkQuaternion	orientation;
	hkVector4		normal=d;
	const int		steps=64;
	normal.normalize3();
	hkQuaternionUtil::_computeShortestRotation(hkVector4(0,0,1,0),normal, orientation);
	hkVector4		p;
	p.setRotatedDir(orientation,hkVector4(r,0,0,0));
	p.add4(o);
	for(int i=1;i<=steps;++i)
	{
		const hkReal	angle=i/(hkReal)steps*HK_REAL_PI*2;
		hkVector4		v(0,0,0,0);
		v(0)	=	r*hkMath::cos(angle);
		v(1)	=	r*hkMath::sin(angle);
		hkVector4		c;
		c.setRotatedDir(orientation,v);
		c.add4(o);
		HK_DISPLAY_LINE(p,c,color);
		p=c;
	}
}

/*static*/ void HK_CALL DrawUtils::drawArc(const hkVector4& o,hkReal r,const hkVector4& a,const hkVector4& b,const hkVector4& c,int depth,hkColor::Argb color)
{
	hkVector4	ab,bc,ca;
	ab.setAdd4(a,b);ab.mul4(0.5f);ab.normalize3();
	bc.setAdd4(b,c);bc.mul4(0.5f);bc.normalize3();
	ca.setAdd4(c,a);ca.mul4(0.5f);ca.normalize3();
	if(depth>0)
	{		
		drawArc(o,r,a,ab,ca,depth-1,color);
		drawArc(o,r,b,bc,ab,depth-1,color);
		drawArc(o,r,c,ca,bc,depth-1,color);
		drawArc(o,r,ab,bc,ca,depth-1,color);
	}
	else
	{
		hkVector4	ct;
		ct.setAdd4(a,b);
		ct.add4(c);
		ct.mul4(1/3.0f);
		ct.normalize3();
		ct.mul4(r);ct.add4(o);
		ab.mul4(r);ab.add4(o);
		bc.mul4(r);bc.add4(o);
		ca.mul4(r);ca.add4(o);
		HK_DISPLAY_LINE(ct,ab,color);
		HK_DISPLAY_LINE(ct,bc,color);
		HK_DISPLAY_LINE(ct,ca,color);
	}
}

/*static*/ void HK_CALL DrawUtils::drawArrowWithScaledHead(const hkVector4& startPos, const hkVector4& endPos, hkSimdRealParameter headScale, const hkColor::Argb color)
{
	HK_DISPLAY_LINE( startPos, endPos, color );
	{
		// Draw a scaled down arrow so that its arms are smaller.
		hkVector4 arrowScaledNormal;
		{
			arrowScaledNormal.setSub( endPos, startPos); 
			arrowScaledNormal.mul( headScale );
		}
		hkVector4 arrowStart; arrowStart.setAddMul( endPos, arrowScaledNormal, hkSimdReal_Minus1 );
		HK_DISPLAY_ARROW( arrowStart, arrowScaledNormal, color );
	}
}

/*static*/ void HK_CALL DrawUtils::drawSphere(const hkVector4& o,hkReal r,hkColor::Argb color,int depth)
{
	drawArc(o,r,hkVector4(+1,0,0,0),hkVector4(0,+1,0,0),hkVector4(0,0,+1,0),depth,color);
	drawArc(o,r,hkVector4(-1,0,0,0),hkVector4(0,0,+1,0),hkVector4(0,+1,0,0),depth,color);
	drawArc(o,r,hkVector4(+1,0,0,0),hkVector4(0,0,+1,0),hkVector4(0,-1,0,0),depth,color);
	drawArc(o,r,hkVector4(-1,0,0,0),hkVector4(0,-1,0,0),hkVector4(0,0,+1,0),depth,color);
	drawArc(o,r,hkVector4(+1,0,0,0),hkVector4(0,0,-1,0),hkVector4(0,+1,0,0),depth,color);
	drawArc(o,r,hkVector4(-1,0,0,0),hkVector4(0,+1,0,0),hkVector4(0,0,-1,0),depth,color);
	drawArc(o,r,hkVector4(+1,0,0,0),hkVector4(0,-1,0,0),hkVector4(0,0,-1,0),depth,color);
	drawArc(o,r,hkVector4(-1,0,0,0),hkVector4(0,0,-1,0),hkVector4(0,-1,0,0),depth,color);
}


typedef hkDrawBase::VectorParameter VectorParameter;

namespace
{
	struct Rect
	{
		hkVector4 bot[2];
		hkVector4 top[2];
	};
}

void hkDrawBase::pushOffset(hkVector4Parameter a)
{
	HK_ASSERT(0x3655d7eb, s_drawingOffsetIndex+1 < (int)HK_COUNT_OF(s_drawingOffsetStack));
	s_drawingOffsetStack[s_drawingOffsetIndex++] = s_drawingOffset;
	s_drawingOffset.add4(a);
}

void hkDrawBase::popOffset()
{
	HK_ASSERT(0x3655d7ea, s_drawingOffsetIndex > 0 );
	s_drawingOffset = s_drawingOffsetStack[--s_drawingOffsetIndex];
}

hkColor::Argb hkDrawBase::darken( hkColor::Argb c )
{
	return (unsigned(c) >> 1) & ~0x80808080;
}

void hkDrawBase::drawPoint(hkVector4Parameter a,  hkColor::Argb color)
{
	hkVector4 ao; ao.setAdd4(a,s_drawingOffset);
	HK_DISPLAY_STAR( ao, 1.0f, color);
}

void hkDrawBase::drawLine(hkVector4Parameter a, hkVector4Parameter b, hkColor::Argb color)
{
	hkVector4 ao; ao.setAdd4(a,s_drawingOffset);
	hkVector4 bo; bo.setAdd4(b,s_drawingOffset);
	HK_DISPLAY_LINE( ao,bo, color);
}

void hkDrawBase::fillTriangle(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color)
{
	hkVector4 ao; ao.setAdd4(a,s_drawingOffset);
	hkVector4 bo; bo.setAdd4(b,s_drawingOffset);
	hkVector4 co; co.setAdd4(c,s_drawingOffset);
	HK_DISPLAY_TRIANGLE( ao,bo,co, color);
}

void hkDrawBase::wireTriangle(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkColor::Argb color)
{
	drawLine( a,b, color);
	drawLine( b,c, color);
	drawLine( c,a, color);
}

void hkDrawBase::fillRectangle(hkVector4Parameter center, hkVector4Parameter x, hkVector4Parameter y, hkColor::Argb color)
{
	hkVector4 a = center; a.add4(y); a.add4(x);
	hkVector4 b = center; b.add4(y); b.sub4(x);
	hkVector4 c = center; c.sub4(y); c.sub4(x);
	hkVector4 d = center; d.sub4(y); d.add4(x);

	fillTriangle( a,b,c, color);
	fillTriangle( a,c,d, color);
}

void hkDrawBase::fillQuad(hkVector4Parameter a, hkVector4Parameter b, hkVector4Parameter c, hkVector4Parameter d, hkColor::Argb color)
{
	fillTriangle( a,b,c, color);
	fillTriangle( a,c,d, color);
}

void hkDrawBase::wireRectangle(hkVector4Parameter center, hkVector4Parameter x, hkVector4Parameter y, hkColor::Argb color)
{
	hkVector4 a = center; a.add4(y); a.add4(x);
	hkVector4 b = center; b.add4(y); b.sub4(x);
	hkVector4 c = center; c.sub4(y); c.sub4(x);
	hkVector4 d = center; d.sub4(y); d.add4(x);

	drawLine( a,b, color);
	drawLine( b,c, color);
	drawLine( c,d, color);
	drawLine( d,a, color);
}

hkDrawRectangle& hkDrawRectangle::thickLine3d( hkVector4Parameter v0, hkVector4Parameter v1, hkVector4Parameter n, hkReal w)
{
	hkVector4 x; x.setSub4(v1,v0); x.mul4(0.5f);
	hkVector4 c; c.setAdd4(v0,x);
	hkVector4 y; y.setCross( n, x );
	if( y.normalize3IfNotZero() == HK_SUCCESS )
	{
		y.mul4(w);
		m_center = c;
		m_x = x;
		m_y = y;
	}
	else
	{
		m_x = x;
		m_y = x;
	}
	return *this;
}

// Convert a line segment into 4 corner points of a rectangle
static inline void fattenLine( hkVector4Parameter v0, hkVector4Parameter v1, hkVector4Parameter norm, hkReal width, hkVector4& edge, hkVector4* bot, hkVector4* top )
{
	edge.setSub4(v1, v0);
	hkVector4 perp; perp.setCross( norm, edge );
	perp.normalize3(); perp.mul4(width);
	bot[0].setSub4( v0, perp );
	bot[1].setAdd4( v0, perp );
	top[0].setSub4( v1, perp );
	top[1].setAdd4( v1, perp );
}

// given two adjacent rectangles, move a single point on the top of one and
// the corresponding point on the bottom of the other so they no longer overlap.
static int bevelRectanglePair(
	const hkVector4* cbot,       hkVector4* ctop, hkVector4Parameter cedge,
	      hkVector4* nbot, const hkVector4* ntop, hkVector4Parameter nedge,
	hkVector4Parameter norm)
{
	hkVector4 cross; cross.setCross( cedge, nedge );
	hkReal dot = cross.dot3(norm);
	if( hkMath::fabs( dot) < 1e-3f ) return 0;
	int side = dot > 0 ? 1 : 0; // which side gets shortened

	hkLineSegmentUtil::ClosestLineSegLineSegResult result;
	hkLineSegmentUtil::closestLineSegLineSeg( cbot[side], cedge, nbot[side], nedge, result );

	ctop[side] = result.m_closestPointA;
	nbot[side] = result.m_closestPointA;
	return side;
}

hkDrawLine& hkDrawLine::thickLines3d( const hkVector4* verts, int numVerts, hkBool32 loop, hkVector4Parameter norm, hkReal width, hkColor::Argb color )
{
	if( numVerts >= 3 )
	{
		
		hkVector4 cbot[2]; // Current edge & bottom, top of rectangle
		hkVector4 ctop[2];
		hkVector4 cedge;
		fattenLine(verts[0], verts[1], norm, width, cedge, cbot, ctop);

		hkVector4 lbot[2]; // Last, only used for loop
		hkVector4 ltop[2];
		hkVector4 ledge;
		if( loop )
		{
			fattenLine(verts[numVerts-1], verts[0], norm, width, ledge, lbot, ltop);
			int bevelSide = bevelRectanglePair( lbot, ltop, ledge, cbot, ctop, cedge, norm );
			fillTriangle( ltop[0], ltop[1], cbot[!bevelSide], color );
			//		cbot[bevelSide] = ltop[bevelSide];
		}
		for( int i = 1; i < numVerts-1; ++i )
		{
			hkVector4 nbot[2]; // Next bottom, top & edge
			hkVector4 ntop[2];
			hkVector4 nedge;

			hkVector4 edge; edge.setSub4(verts[i], verts[i+1]);
			// Check for zero-length edges, so we don't try to draw thin triangles
			if (edge.lengthSquared3().getReal() < HK_REAL_EPSILON)
			{
				continue;
			}

			fattenLine(verts[i], verts[i+1], norm, width, nedge, nbot, ntop);
			int bevelSide = bevelRectanglePair( cbot, ctop, cedge, nbot, ntop, nedge, norm );

			fillQuad( cbot[0], cbot[1], ctop[1], ctop[0], color );
			fillTriangle( ctop[0], ctop[1], nbot[!bevelSide], color );

			//		nbot[bevelSide] = ctop[bevelSide];

			cbot[0] = nbot[0];
			cbot[1] = nbot[1];
			ctop[0] = ntop[0];
			ctop[1] = ntop[1];
			cedge = nedge;
		}
		if( loop )
		{
			int bevelSide = bevelRectanglePair( cbot, ctop, cedge, lbot, ltop, ledge, norm );
			fillQuad( cbot[0], cbot[1], ctop[1], ctop[0], color );
			fillTriangle( ctop[0], ctop[1], lbot[!bevelSide], color );
			//		lbot[bevelSide] = ctop[bevelSide];
			fillQuad( lbot[0], lbot[1], ltop[1], ltop[0], color );
		}
		else
		{
			fillQuad( cbot[0], cbot[1], ctop[1], ctop[0], color );
		}
	}
	else if( numVerts == 2 )
	{
		hkDrawRectangle().thickLine3d(verts[0], verts[1], norm, width ).fillColor(color);
	}
	else if( numVerts == 1 )
	{
		hkDrawPoint().fillPip( verts[0], width, color );
	}
	return *this;
}

hkDrawLine& hkDrawLine::thickLines3d( const hkVector4* verts, const hkVector4* normals, int numVerts, hkReal width, hkColor::Argb color )
{
	//const hkVector4 norm = normals[0];
	if( numVerts >= 3 )
	{
		
		hkVector4 cbot[2]; // Current edge & bottom, top of rectangle
		hkVector4 ctop[2];
		hkVector4 cedge;
		fattenLine(verts[0], verts[1], normals[0], width, cedge, cbot, ctop);

		hkVector4 lbot[2]; // Last, only used for loop
		hkVector4 ltop[2];
		hkVector4 ledge;
// 		if( loop )
// 		{
// 			fattenLine(verts[numVerts-1], verts[0], norm, width, ledge, lbot, ltop);
// 			int bevelSide = bevelRectanglePair( lbot, ltop, ledge, cbot, ctop, cedge, norm );
// 			fillTriangle( ltop[0], ltop[1], cbot[!bevelSide], color );
// 			//		cbot[bevelSide] = ltop[bevelSide];
// 		}
		for( int i = 1; i < numVerts-1; ++i )
		{
			hkVector4 nbot[2]; // Next bottom, top & edge
			hkVector4 ntop[2];
			hkVector4 nedge;

			hkVector4 edge; edge.setSub4(verts[i], verts[i+1]);
			// Check for zero-length edges, so we don't try to draw thin triangles
			if (edge.lengthSquared3().getReal() < HK_REAL_EPSILON)
			{
				continue;
			}

			fattenLine(verts[i], verts[i+1], normals[i], width, nedge, nbot, ntop);
			int bevelSide = bevelRectanglePair( cbot, ctop, cedge, nbot, ntop, nedge, normals[i-1] );

			//fillQuad( cbot[0], cbot[1], ctop[1], ctop[0], color );
			fillQuad( cbot[0], ctop[0], ctop[1], cbot[1], color );
			
			
// 			{
// 				hkVector4 midpoint; midpoint.setInterpolate4(verts[i], verts[i+1], .5f);
// 
// 				HK_DISPLAY_ARROW(midpoint, normals[i], 0xFF0000FF);
// 			}
			
			fillTriangle( ctop[1], ctop[0], nbot[!bevelSide], color );

			//		nbot[bevelSide] = ctop[bevelSide];

			cbot[0] = nbot[0];
			cbot[1] = nbot[1];
			ctop[0] = ntop[0];
			ctop[1] = ntop[1];
			cedge = nedge;
		}
// 		if( loop )
// 		{
// 			int bevelSide = bevelRectanglePair( cbot, ctop, cedge, lbot, ltop, ledge, norm );
// 			fillQuad( cbot[0], cbot[1], ctop[1], ctop[0], color );
// 			fillTriangle( ctop[0], ctop[1], lbot[!bevelSide], color );
// 			//		lbot[bevelSide] = ctop[bevelSide];
// 			fillQuad( lbot[0], lbot[1], ltop[1], ltop[0], color );
// 		}
// 		else
		{
			//fillQuad( cbot[0], cbot[1], ctop[1], ctop[0], color );
			fillQuad( cbot[0], ctop[0], ctop[1], cbot[1], color );
		}
	}
	else if( numVerts == 2 )
	{
		hkDrawRectangle().thickLine3d(verts[0], verts[1], normals[0], width ).fillColor(color);
	}
	else if( numVerts == 1 )
	{
		hkDrawPoint().fillPip( verts[0], width, color );
	}
	return *this;
}

hkDrawLine& hkDrawLine::thickLine3d( const hkVector4& start, const hkVector4& end, const hkVector4& normal, hkReal w, hkColor::Argb color )
{
	hkLocalArray<hkVector4> verts(2);
	verts.pushBack(start);
	verts.pushBack(end);

	thickLines3d(verts.begin(), verts.getSize(), false, normal, w, color);

	return *this;
}

hkDrawLine& hkDrawLine::thickArrow2d( const hkVector2& bot, const hkVector2& top, hkReal w, hkColor::Argb color )
{
	// premultiply w into all quantities
	hkVector2 line; line.setSub( top, bot );
	hkReal lineLen = line.normalizeWithLength();
	hkVector2 axis; axis.setMul( line, w ); // vector of length w along axis
	hkVector2 perp; perp.setPerp( axis ); //ditto perpendicular

	const hkReal WING_H = 6; // height of wing triangle (units of w)
	const hkReal WING_W = 4; // width of wing triangle
	const hkReal WING_C = 3; // wing cutout height at join (must be less than H)
	const hkReal dartH = (WING_H) + WING_H/WING_W;
	if( lineLen >= (dartH-WING_C) * w ) // normal length
	{
		// pointy arrow hat
		hkVector2 mtop; mtop.setMul( axis, WING_H/WING_W ); mtop.setSub( top, mtop );
		hkVector2 rtop; rtop.setAdd( mtop, perp );
		hkVector2 ltop; ltop.setSub( mtop, perp );
		fillTriangle( Vector(rtop), Vector(ltop), Vector(top), color );

		// main body of line
		hkVector2 rbot; rbot.setAdd( bot, perp );
		hkVector2 lbot; lbot.setSub( bot, perp );
		fillQuad( Vector(rbot), Vector(lbot), Vector(ltop), Vector(rtop), color );

		// wings
		hkVector2 mid; mid.setAdd( mtop, bot ); mid.setMul( mid, 0.5f );
		for(int i = 0; i < 2; ++i )
		{
			hkVector2 rbase,a,b,c;
			rbase.setAdd( mid, perp ); // midpoint of wing attachment
			const hkReal half = 0.5f * (WING_H - WING_C);
			a.setAddMul( rbase, axis, -half ); // lower wing attachment
			b.setAddMul( rbase, axis, +half ); // high wing attachment
			c.setAddMul( b, axis, -WING_H );
			c.setAddMul( c, perp, WING_W ); // wing tip
			if( i == 0 )
			{
				perp.set( -perp.x, -perp.y ); // flip for next iter
			}
			else
			{
				hkVector2 t = a; a = b; b = t; // fix winding
			}
			fillTriangle( Vector(a), Vector(b), Vector(c), color );
		}
	}
	else // very short arrows are drawn as darts
	{
		hkVector2 tbot; tbot.setAddMul( top, axis, -dartH );
		hkVector2 rtip; rtip.setAddMul( tbot, perp, WING_W+1 );
		hkVector2 ltip; ltip.setAddMul( tbot, perp, -WING_W-1 );
		fillTriangle( Vector(ltip), Vector(top), Vector(bot), color );
		fillTriangle( Vector(bot), Vector(top), Vector(rtip), color );
		return *this;
	}
	
	return *this;
}

hkDrawLine& hkDrawLine::thickArrow3d( const hkVector4& a, const hkVector4& b, const hkVector4& n, hkReal w, hkColor::Argb c )
{	
	const hkReal arrowHeadWidth = w * 2.5f;
	const hkReal arrowHeadHeight = w * 5.0f;

	hkVector4 _arrow;
	_arrow.setSub4(b, a);
	const hkReal length = (hkReal)_arrow.normalizeWithLength3();						

	hkVector4 lineStart;
	lineStart = a;

	hkVector4 lineEnd;						
	lineEnd.setInterpolate4(a, b, (length - arrowHeadHeight) / length);

	hkDrawLine().thickLine3d(lineStart, lineEnd, n, w, c);

	hkVector4 perp;
	perp.setCross(_arrow, n);						

	hkVector4 headLeft;
	headLeft.setAddMul4(lineEnd, perp, arrowHeadWidth);

	hkVector4 headRight;
	headRight.setAddMul4(lineEnd, perp, -arrowHeadWidth);

	hkDrawTriangle().points(headLeft, b, headRight).fillColor(c);

	return *this;
}

hkDrawTriangle& hkDrawTriangle::expand( hkReal r )
{
	hkVector4 c; c.setAdd4( m_a, m_b ); c.add4(m_c); c.mul4(1.0f/3.0f);
	hkVector4 da; da.setSub4( m_a, c ); da.fastNormalize3(); m_a.addMul4( r, da );
	hkVector4 db; db.setSub4( m_b, c ); db.fastNormalize3(); m_b.addMul4( r, db );
	hkVector4 dc; dc.setSub4( m_c, c ); dc.fastNormalize3(); m_c.addMul4( r, dc );
	return *this;
}

hkDrawPoint& hkDrawPoint::wireCross( hkVector4Parameter p, hkReal rad, hkColor::Argb c )
{
	hkVector4 off[3];
	off[0] = hkVector4::getConstant(HK_QUADREAL_1000); off[0].mul4(rad);
	off[1] = hkVector4::getConstant(HK_QUADREAL_0100); off[1].mul4(rad);
	off[2] = hkVector4::getConstant(HK_QUADREAL_0010); off[2].mul4(rad);

	for( int i = 0; i < 3; ++i )
	{
		hkVector4 p0; p0.setSub4( p, off[i] );
		hkVector4 p1; p1.setAdd4( p, off[i] );
		drawLine( p0, p1, c );
	}
	return *this;
}

hkDrawPoint& hkDrawPoint::wirePip( hkVector4Parameter p, hkReal rad, hkColor::Argb c )
{
	hkVector4 top = hkVector4::getConstant(HK_QUADREAL_1000); top.setAddMul4( p, top,  rad );
	hkVector4 bot = hkVector4::getConstant(HK_QUADREAL_1000); bot.setAddMul4( p, bot, -rad );
	hkVector4 pts[5];
	pts[0] = hkVector4::getConstant(HK_QUADREAL_0100); pts[0].setAddMul4( p, pts[0],  rad );
	pts[1] = hkVector4::getConstant(HK_QUADREAL_0010); pts[1].setAddMul4( p, pts[1],  rad );
	pts[2] = hkVector4::getConstant(HK_QUADREAL_0100); pts[2].setAddMul4( p, pts[2], -rad );
	pts[3] = hkVector4::getConstant(HK_QUADREAL_0010); pts[3].setAddMul4( p, pts[3], -rad );
	pts[4] = pts[0];
	for( int i = 0; i < 4; ++i )
	{
		drawLine( top, pts[i], c );
		drawLine( bot, pts[i], c );
		drawLine( pts[i], pts[i+1], c );
	}
	return *this;
}

hkDrawPoint& hkDrawPoint::fillPip( VectorParameter p, hkReal rad, hkColor::Argb c )
{
	hkVector4 top = hkVector4::getConstant(HK_QUADREAL_1000); top.setAddMul4( p, top,  rad );
	hkVector4 bot = hkVector4::getConstant(HK_QUADREAL_1000); bot.setAddMul4( p, bot, -rad );
	hkVector4 pts[5];
	pts[0] = hkVector4::getConstant(HK_QUADREAL_0100); pts[0].setAddMul4( p, pts[0],  rad );
	pts[1] = hkVector4::getConstant(HK_QUADREAL_0010); pts[1].setAddMul4( p, pts[1],  rad );
	pts[2] = hkVector4::getConstant(HK_QUADREAL_0100); pts[2].setAddMul4( p, pts[2], -rad );
	pts[3] = hkVector4::getConstant(HK_QUADREAL_0010); pts[3].setAddMul4( p, pts[3], -rad );
	pts[4] = pts[0];
	for( int i = 0; i < 4; ++i )
	{
		fillTriangle( top, pts[i], pts[i+1], c );
		fillTriangle( pts[i], bot, pts[i+1], c );
	}
	return *this;
}

hkDrawPoint& hkDrawPoint::circle3d( hkVector4Parameter p, hkReal rad, hkVector4Parameter norm, hkColor::Argb color )
{
	hkVector4 x,y;
	{
		hkMatrix3 basis;
		hkVector4Util::buildOrthonormal(norm, basis);
		x.setMul4( rad, basis.getColumn(1) );
		y.setMul4( rad, basis.getColumn(2) );
	}
	// to generate the sin lookup table
	// N=12; ", ".join("%ff"%math.sin(i*math.pi/(2*(1+N))) for i in range(1, N+1))
	const hkReal lut[12] = { 0.120537f, 0.239316f, 0.354605f, 0.464723f, 0.568065f, 0.663123f, 0.748511f, 0.822984f, 0.885456f, 0.935016f, 0.970942f, 0.992709f };
	int step = 4; // num of points in a quadrant
	for( int i = 0; i < 4; ++i ) // for each quadrant
	{
		hkVector4 cur; cur.setAdd4( p, x );
		for( int yi = 0, xi = HK_COUNT_OF(lut)-1; yi < (int)(HK_COUNT_OF(lut)); yi += step, xi -= step )
		{
			hkVector4 next;
			next.setAddMul4( p, y, lut[yi] );
			next.addMul4( lut[xi], x );
			fillTriangle( p, cur, next, color );
			cur = next;
		}
		hkVector4 next; next.setAdd4( p, y );
		fillTriangle( p, cur, next, color );

		// rotate 90 and go again
		hkVector4 t = x;
		x = y;
		y.setNeg4(t);
	}
	return *this;
}

hkDrawPoint& hkDrawPoint::circle3d( hkVector4Parameter p, hkReal rad, hkVector4Parameter norm, hkVector4Parameter startPerp, hkReal start, hkReal end, int numSegments, hkColor::Argb c )
{
	const hkReal increment = (end - start) / numSegments;

	hkLocalArray<hkVector4> pointsOnArc(numSegments + 1);
	
	for( hkReal theta = start; theta < (end + increment); theta += increment )
	{	
		hkReal clampedTheta = hkMath::clamp(theta, start, end);
		
		hkQuaternion rotation;
		rotation.setAxisAngle(norm, clampedTheta);

		hkVector4 direction;
		direction.setRotatedDir(rotation, startPerp);
		
		pointsOnArc.expandOne().setAddMul4(p, direction, rad);
	}	

	for( int segementIndex = 0; segementIndex < pointsOnArc.getSize() - 1; ++segementIndex )
	{
		hkDrawTriangle().fillTriangle(p, pointsOnArc[segementIndex], pointsOnArc[segementIndex+1], c);
	}	

	return *this;
}

hkDrawPoint& hkDrawPoint::wireCircle3d( hkVector4Parameter p, hkReal rad, hkVector4Parameter norm, hkColor::Argb color )
{
	hkVector4 x,y;
	{
		hkMatrix3 basis;
		hkVector4Util::buildOrthonormal(norm, basis);
		x.setMul4( rad, basis.getColumn(1) );
		y.setMul4( rad, basis.getColumn(2) );
	}
	// to generate the sin lookup table
	// N=12; ", ".join("%ff"%math.sin(i*math.pi/(2*(1+N))) for i in range(1, N+1))
	const hkReal lut[12] = { 0.120537f, 0.239316f, 0.354605f, 0.464723f, 0.568065f, 0.663123f, 0.748511f, 0.822984f, 0.885456f, 0.935016f, 0.970942f, 0.992709f };
	int step = 4; // num of points in a quadrant
	for( int i = 0; i < 4; ++i ) // for each quadrant
	{
		hkVector4 cur; cur.setAdd4( p, x );
		for( int yi = 0, xi = HK_COUNT_OF(lut)-1; yi < (int)(HK_COUNT_OF(lut)); yi += step, xi -= step )
		{
			hkVector4 next;
			next.setAddMul4( p, y, lut[yi] );
			next.addMul4( lut[xi], x );
			drawLine( cur, next, color);
			cur = next;
		}
		hkVector4 next; next.setAdd4( p, y );
		drawLine( cur, next, color );

		// rotate 90 and go again
		hkVector4 t = x;
		x = y;
		y.setNeg4(t);
	}
	return *this;
}

hkDrawPoint& hkDrawPoint::arc2d( hkVector4Parameter p1, hkVector4Parameter p2, hkVector4Parameter center, hkReal rad, hkColor::Argb c )
{
	hkSimdReal radSimd = hkSimdReal(rad);
	const int numSegments = 16;
	hkSimdReal dt; dt.setFromFloat( 1.0f / hkReal(numSegments) );

	hkVector4 dir1; dir1.setSub4(p1, center);
	hkVector4 dir2; dir2.setSub4(p2, center);
	dir1.normalize3();
	dir2.normalize3();

	hkQuaternion dRotation, currentRotation;
	{
		hkQuaternion totalRot; hkQuaternionUtil::_computeShortestRotation(dir1, dir2, totalRot);
		dRotation.setSlerp(hkQuaternion::getIdentity(), totalRot, dt);
		currentRotation = dRotation;
	}

	hkVector4 prev = dir1;

 	for (int i=0; i<numSegments; i++)
 	{
 		hkVector4 next;
		next.setRotatedDir(currentRotation, dir1);

		hkVector4 temp1, temp2;
		temp1.setAddMul4(center, next, radSimd);
		temp2.setAddMul4(center, prev, radSimd);

		fillTriangle(center, temp1, temp2, c);

		prev = next;
		currentRotation.setMul(currentRotation, dRotation);
 	}

	return *this;
}

hkVector4 hkDrawBase::s_x2d = hkVector4( 1,0,0,1);
hkVector4 hkDrawBase::s_y2d = hkVector4( 0,1,0,1);
hkVector4 hkDrawBase::s_n2d = hkVector4( 0,0,1,0);

hkDrawAny::hkDrawAny() : m_offsetPops(0)
{
}

hkDrawAny::~hkDrawAny()
{
	for( int i = 0; i < m_offsetPops; ++i )
	{
		hkDrawBase::popOffset();
	}
}

hkDrawAny& hkDrawAny::closeLoop()
{
	if( m_pts.getSize() )
	{
		hkVector4 s = m_pts[0];
		m_pts.pushBack( s );
	}	
	return *this;
}

hkDrawAny& hkDrawAny::octahedron( hkReal topRad, hkReal sideRad)
{
	sideRad = sideRad < 0 ? topRad : sideRad;
	hkArray<hkVector4> newPts; newPts.reserve( m_pts.getSize() * 6 );

	for( int srcIndex = 0; srcIndex < m_pts.getSize(); ++srcIndex )
	{
		const hkVector4& p = m_pts[srcIndex];
		int idxStart = newPts.getSize();
		{
			hkVector4* n = newPts.expandByUnchecked(6);
			hkVector4 v;
			v = hkVector4::getConstant(HK_QUADREAL_0100); v.setAddMul4( p, v,  sideRad ); n[0] = v;
			v = hkVector4::getConstant(HK_QUADREAL_0010); v.setAddMul4( p, v,  sideRad ); n[1] = v;
			v = hkVector4::getConstant(HK_QUADREAL_0100); v.setAddMul4( p, v, -sideRad ); n[2] = v;
			v = hkVector4::getConstant(HK_QUADREAL_0010); v.setAddMul4( p, v, -sideRad ); n[3] = v;
			v = hkVector4::getConstant(HK_QUADREAL_1000); v.setAddMul4( p, v,  topRad ); n[4] = v; // TOP
			v = hkVector4::getConstant(HK_QUADREAL_1000); v.setAddMul4( p, v, -topRad ); n[5] = v; // BOT
		}

		int previ = 3;
		for( int i = 0; i < 4; ++i )
		{
			int* triTop = m_polys.expandBy(8);
			triTop[0] = -3;
			triTop[1] = idxStart + 4;
			triTop[2] = idxStart + previ;
			triTop[3] = idxStart + i;
			int* triBot = triTop + 4;
			triBot[0] = -3;
			triBot[1] = idxStart + previ;
			triBot[2] = idxStart + 5;
			triBot[3] = idxStart + i;
			previ = i;
		}
	}
	m_pts.swap( newPts );
	return *this;
}

static void _polygon( hkArray<hkVector4>& pts, hkArray<int>& polys, const hkVector4* off, int noff )
{
	hkArray<hkVector4> newPts;
	newPts.reserve( pts.getSize() * noff );
	for( int i = 0; i < pts.getSize(); ++i )
	{
		hkVector4* p = newPts.expandByUnchecked(noff);
		for( int j = 0; j < noff; ++j )
		{
			p[j].setAdd4( pts[i], off[j] );
		}
	}

	polys.clear();
	polys.reserve( (noff+1) * pts.getSize() );
	for( int i = 0; i < pts.getSize(); ++i )
	{
		int* q = polys.expandByUnchecked(noff+1);
		q[0] = -noff;
		for( int j = 0; j < noff; ++j )
		{
			q[j+1] = noff*i + j;
		}
	}
	pts.swap( newPts );
}

static void _triangles( const hkArrayBase<hkVector4>& pts, const hkVector4* tris, int numTriPts, hkColor::Argb color )
{
	for( int i = 0; i < pts.getSize(); ++i )
	{
		for( int j = 0; j < numTriPts; j += 3 )
		{
			hkVector4 a; a.setAdd4( pts[i], tris[j  ] );
			hkVector4 b; b.setAdd4( pts[i], tris[j+1] );
			hkVector4 c; c.setAdd4( pts[i], tris[j+2] );
			hkDrawBase::fillTriangle( a, b, c, color );
		}
	}
}

hkDrawAny& hkDrawAny::square( hkReal rad )
{
	hkVector4 off[4];
	off[0].setAdd4( s_y2d, s_x2d ); off[0].mul4(rad);
	off[1].setSub4( s_y2d, s_x2d ); off[1].mul4(rad);
	off[2].setNeg4(off[0]);
	off[3].setNeg4(off[1]);
	_polygon( m_pts, m_polys, off, HK_COUNT_OF(off) );
	return *this;
}

hkDrawAny& hkDrawAny::diamond( hkReal rad )
{
	hkVector4 off[4];
	off[0].setMul4( rad, s_x2d );
	off[1].setMul4( rad, s_y2d );
	off[2].setNeg4(off[0]);
	off[3].setNeg4(off[1]);
	_polygon( m_pts, m_polys, off, HK_COUNT_OF(off) );	
	return *this;
}

hkDrawAny& hkDrawAny::pushOffset( hkReal d, int axis )
{
	m_offsetPops += 1;
	hkVector4 v; v.setZero4(); v(axis) = d;
	hkDrawBase::pushOffset( v );
	return *this;
}

hkDrawAny& hkDrawAny::popOffset()
{
	HK_ASSERT(0x7007216f, m_offsetPops >= 1);
	m_offsetPops -= 1;
	hkDrawBase::popOffset();
	return *this;
}

hkDrawAny& hkDrawAny::scale( hkReal amt )
{
	hkVector4 normal = hkVector4::getConstant(HK_QUADREAL_0010);
	hkArray<hkVector4> segs;
	hkArray<hkVector4> pts;
	{
		HK_ASSERT(0x5ff40abc, m_pts[0].equals3(m_pts.back(),0) );
		hkVector4 prev = m_pts[0];
		for( int i = 1; i < m_pts.getSize(); ++i )
		{
			hkVector4 cur = m_pts[i];
			hkVector4 d; d.setSub4( cur, prev );
			hkVector4 p; p.setCross( normal, d ); p.normalize3();
			hkVector4 a; a.setAddMul4( prev, p, 1.0f - amt );
			hkVector4 b; b.setAddMul4( cur, p, 1.0f - amt );
			segs.pushBack( a );
			segs.pushBack( d );
			prev = cur;
		}
	}
	{
		hkVector4 pp = segs[ segs.getSize()-2 ];
		hkVector4 pd = segs[ segs.getSize()-1 ];
 		for( int i = 0; i < segs.getSize(); i += 2 )
		{
			hkVector4 cp = segs[i  ];
			hkVector4 cd = segs[i+1];
			hkLineSegmentUtil::ClosestLineSegLineSegResult result;
			int res = hkLineSegmentUtil::closestLineSegLineSeg( pp, pd, cp, cd, result );
			if( res != 0 )
			{
				pts.pushBack( result.m_closestPointA );
				pts.expandOne().setSub4( result.m_closestPointA, result.m_closestAminusClosestB );
			}
			else
			{
				pts.pushBack( result.m_closestPointA );
			}
			pp = cp;
			pd = cd;
		}
	}

	hkVector4 prev = pts.back();
	for( int i = 0; i < pts.getSize(); ++i )
	{
		drawLine( prev, pts[i], 0xffffffff );
		prev = pts[i];
	}

	return *this;
}

hkDrawAny& hkDrawAny::label( hkColor::Argb c, const char* txt )
{
	hkStringBuf s;
	txt = txt==HK_NULL ? "" : txt;
	for( int i = 0; i < m_pts.getSize(); ++i )
	{
		hkVector4 v; v.setAdd4( s_drawingOffset, m_pts[i]);
		s.printf("%i %s", v.getInt24W(), txt );
		HK_DISPLAY_3D_TEXT( s.cString(), v, c );
	}
	return *this;
}

hkDrawAny& hkDrawAny::stroke( hkColor::Argb color, hkReal radius )
{
	if( radius > 0 )
	{
		// todo: bevel
		hkVector4 pts;
		if( m_polys.getSize() )
		{
			for( int curPolyIndex = 0; curPolyIndex < m_polys.getSize(); /**/ )
			{
				int npart = -m_polys[curPolyIndex];
				curPolyIndex += 1;
				HK_ASSERT(0x7fa2b8f2, npart > 0);
				int previ = m_polys[curPolyIndex+npart-1];
				for( int i = 0; i < npart; ++i )
				{
					int curi = m_polys[curPolyIndex + i];
					hkDrawLine().thickLine2d( m_pts[curi], m_pts[previ], radius, color );
					previ = curi;
				}
				curPolyIndex += npart;
			}
		}
		else
		{
			for( int i = 0; i < m_pts.getSize() - 1; ++i )
			{
				hkDrawLine().thickLine2d( m_pts[i], m_pts[i+1], radius, color );
			}
		}
	}
	else
	{
		hkVector4 pts;
		if( m_polys.getSize() )
		{
			for( int curPolyIndex = 0; curPolyIndex < m_polys.getSize(); /**/ )
			{
				int npart = -m_polys[curPolyIndex];
				curPolyIndex += 1;
				HK_ASSERT(0x7fa2b8f2, npart > 0);
				int previ = m_polys[curPolyIndex+npart-1];
				for( int i = 0; i < npart; ++i )
				{
					int curi = m_polys[curPolyIndex + i];
					drawLine( m_pts[curi], m_pts[previ], color );
					previ = curi;
				}
				curPolyIndex += npart;
			}
		}
		else
		{
			for( int i = 0; i < m_pts.getSize() - 1; ++i )
			{
				drawLine( m_pts[i], m_pts[i+1], color);
			}
		}
	}
	return *this;
}

hkDrawAny& hkDrawAny::fill( hkColor::Argb color )
{
	if( m_polys.getSize() )
	{
		for( int curPolyIndex = 0; curPolyIndex < m_polys.getSize(); /**/ )
		{
			int npart = -m_polys[curPolyIndex];
			curPolyIndex += 1;
			//HK_ASSERT(0x7fa2b8f3, npart > 0);
			//HK_ASSERT(0x7fa2b8f3, npart == 3);
			int i0 = m_polys[curPolyIndex  ];
			int i1 = m_polys[curPolyIndex+1];
			int i2 = m_polys[curPolyIndex+2];
			fillTriangle( m_pts[i0], m_pts[i1], m_pts[i2], color );
			curPolyIndex += npart;
		}
	}
	else
	{
		hkArray<hkVector4> pts; pts = m_pts;
		while( pts.getSize() >= 3 )
		{
			int np = pts.getSize();
			for( int i0 = 0; i0 < np; ++i0 )
			{
				int i1 = (i0+1) % np;
				int i2 = (i0+2) % np;
				hkVector2 v0; v0.load( &pts[i0](0) );
				hkVector2 v1; v1.load( &pts[i1](0) );
				hkVector2 v2; v2.load( &pts[i2](0) );

				if( v2.leftOfLine(v0,v1))
				{
					hkBool32 ok = true;
					for( int j = 0; j < np - 3; ++j )
					{
						int ij = (i0+3+j) % np;
						hkVector2 v; v.load( &pts[ij](0) );
						if( v.leftOfLine(v0,v1) && v.leftOfLine(v1,v2) && v.leftOfLine(v2,v0) )
						{
							ok = false;
							break;
						}
					}
					if( ok )
					{
						fillTriangle( pts[i0], pts[i1], pts[i2], color );
						pts.removeAtAndCopy(i1);
						break;
					}
				}				
			}
			HK_ASSERT(0x73e3e251, pts.getSize() < np );
		}
	}
	return *this;
}

hkDrawAny& hkDrawAny::fillSquare( hkReal rad, hkColor::Argb c )
{
	hkVector4 off[6];
	off[0].setAdd4( s_y2d, s_x2d ); off[0].mul4(rad);
	off[1].setSub4( s_y2d, s_x2d ); off[1].mul4(rad);
	off[2].setNeg4(off[0]);
	off[3] = off[2];
	off[4].setNeg3(off[1]);
	off[5] = off[0];
	_triangles( m_pts, off, HK_COUNT_OF(off), c );
	return *this;
}

hkDrawAny& hkDrawAny::thickLine( hkReal w, hkColor::Argb c )
{
	HK_ASSERT(0x2a3986d9, m_pts.getSize() == 2 ); // unfinished!
	hkVector4 edge, bot[2], top[2];
	fattenLine( m_pts[0], m_pts[1], s_n2d, w, edge, bot, top);
	fillTriangle(bot[1], bot[0], top[0], c);
	fillTriangle(top[1], bot[1], top[0], c);
	return *this;
}

hkDrawAny& hkDrawAny::thickArrow( hkReal w, hkColor::Argb c )
{
	hkVector2 src; src.setProject( m_pts[0], s_x2d, s_y2d );
	hkVector2 dst; dst.setProject( m_pts[1], s_x2d, s_y2d );
	hkDrawLine().thickArrow2d( src, dst, w, c);
	return *this;
}

hkDrawAny& hkDrawAny::fillDart( hkReal rad, hkColor::Argb color )
{
	hkVector4 edge; edge.setSub4( m_pts[1], m_pts[0] );
	hkVector4 perp; perp.setCross( edge, s_n2d );
	perp.fastNormalize3();
	hkVector4 bot0; bot0.setAddMul4( m_pts[0], perp, -rad );
	hkVector4 bot1; bot1.setAddMul4( m_pts[0], perp,  rad );
	fillTriangle( bot0, bot1, m_pts[1], color );
	return *this;
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
