/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Havok Bootstrapper

//
// Includes
//
#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Bootstrap/BootstrapDemo.h>

#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Common/Font/hkgFont.h>
#include <Common/Serialize/Resource/hkResource.h>

#include <Common/Base/KeyCode.h>
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
#include <Demos/DemoCommon/DemoFramework/hkDefaultPhysics2012Demo.h>
#include <Physics2012/Utilities/Serialize/hkpHavokSnapshot.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>
#include <Physics2012/Utilities/VisualDebugger/hkpPhysicsContext.h>
#endif

#include <Common/Base/Reflection/Registry/hkTypeInfoRegistry.h>
#include <Common/Base/Reflection/Registry/hkVtableClassRegistry.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/Memory/hkMemoryStreamReader.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkCheckDeterminismUtil.h>
#include <Common/Base/DebugUtil/DeterminismUtil/hkNetworkedDeterminismUtil.h>
#include <Common/Base/DebugUtil/MemoryExceptionTestingUtil/hkMemoryExceptionTestingUtil.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>

#include <Demos/DemoCommon/DemoFramework/hkPerformanceCounterUtility.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Graphics/Bridge/System/hkgSystem.h>

#include <Common/Internal/Misc/hkSystemDate.h>

#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/Memory/Allocator/Malloc/hkMallocAllocator.h>
#include <Common/Visualize/hkVisualDebugger.h>
#include <Common/Base/Memory/Tracker/ScanCalculator/hkTrackerSnapshotUtil.h>
#include <Common/Base/Memory/Tracker/Report/hkVdbStreamReportUtil.h>
#include <Common/Base/System/Io/Util/hkLoadUtil.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>

#include <Common/Base/System/Hardware/hkHardwareInfo.h>

// For processor info
#include <Common/Base/Fwd/hkwindows.h>




// Demos for stats bootstrap
struct StatsBootstrapDemoInstance
{
	char* m_name;
};

// Support for the last successful bootstrap demo
extern const char* LASTDEMO_FILENAME_BOOTSTRAP;

	// set the next define to force the bootstrap to execute a single demo only
//#define SPECIAL_DEMO "Physics2012/Api/Dynamics/Constraints/BallAndSocketRope/Length 200"
//#define SPECIAL_DEMO "Examples/Physics/Continuous/BrickWall/8x8x3 Continuous"
//#define SPECIAL_DEMO "Physics2012/Test/Stress/Feature/Collide/BroadPhaseRayCastStress"
//#define SPECIAL_DEMO "Behavior/Test/Feature/TransitionTest"

void hkSetLastDemoBootstrap(const char* namein)
{
	hkOfstream out(LASTDEMO_FILENAME_BOOTSTRAP);
	if(out.isOk())
	{
		out << namein;
	}
}

extern const char* hkSetLastDemo(const char* namein);

// Demo type masks

// Individual products
static const int DEMO_MASK_AI = HK_DEMO_TYPE_AI;
static const int DEMO_MASK_ANIMATION = HK_DEMO_TYPE_ANIMATION;
static const int DEMO_MASK_BEHAVIOR = HK_DEMO_TYPE_BEHAVIOR;
static const int DEMO_MASK_CLOTH = HK_DEMO_TYPE_CLOTH;
static const int DEMO_MASK_DESTRUCTION = HK_DEMO_TYPE_DESTRUCTION;
static const int DEMO_MASK_DESTRUCTION_2012 = HK_DEMO_TYPE_DESTRUCTION_2012 |
											  HK_DEMO_TYPE_DESTRUCTION_CRITICAL |
											  HK_DEMO_TYPE_DESTRUCTION_CRITICAL_FILEBASED;
static const int DEMO_MASK_PHYSICS = HK_DEMO_TYPE_PHYSICS;
static const int DEMO_MASK_PHYSICS_2012 = HK_DEMO_TYPE_PHYSICS_2012;
static const int DEMO_MASK_SCRIPT = HK_DEMO_TYPE_SCRIPT;

// Compound/special masks
static const int DEMO_MASK_ALL = DEMO_MASK_PHYSICS |
								 DEMO_MASK_PHYSICS_2012 |
								 DEMO_MASK_ANIMATION |
								 DEMO_MASK_CLOTH |
								 DEMO_MASK_CLOTH |
								 DEMO_MASK_DESTRUCTION_2012 |
								 DEMO_MASK_DESTRUCTION |
								 DEMO_MASK_AI |
								 DEMO_MASK_SCRIPT;

static const int DEMO_MASK_ALL_CRITICAL = HK_DEMO_TYPE_CRITICAL | DEMO_MASK_ALL;

static const int DEMO_MASK_ALL_STATS = HK_DEMO_TYPE_STATS;

static const int DEMO_MASK_AI_STATS = HK_DEMO_TYPE_STATS | DEMO_MASK_AI;

static const int DEMO_MASK_ANIMATION_STATS = HK_DEMO_TYPE_STATS | DEMO_MASK_ANIMATION;

static const int DEMO_MASK_STATS_CRITICAL = HK_DEMO_TYPE_STATS |
											HK_DEMO_TYPE_CRITICAL;

static const int DEMO_MASK_DETERMINISM = DEMO_MASK_PHYSICS |
										 DEMO_MASK_PHYSICS_2012 |
										 DEMO_MASK_AI |
										 DEMO_MASK_BEHAVIOR;

static const int DEMO_MASK_DETERMINISM_CRITICAL = HK_DEMO_TYPE_CRITICAL | DEMO_MASK_DETERMINISM;

static const int DEMO_MASK_SERIALIZE = HK_DEMO_TYPE_SERIALIZE;

// Warning and report IDs to disable
static hkUint32 s_ignoredIds[] =
{
	0x6d7b6e97, // Report section in hkaiNavMeshGenerationUtils
	0x3e0c88c6, // Report section around hkgAssetConverter::convertMesh
	0xAF55ADDE, // hkFreeListAllocator warnings
	0x7247e2de, // Report section in hkaiNavMeshSimplificationUtils::simplifyNavMesh
	0x517e0a1d, // Report section in hkaiNavMeshDebugUtils::reportMemoryUsed

	// animation
	0x36118e94, // spline info
	0x1e663ab1, // quantized info
	0x9832bf32, // predictive info
	0x3214badc, // predictive info
	0x982BC8F8, // performance warning
	0x54E4323E, // skeleton mapping
	0x4fa31823, // loading TK files
	0x0DCDD09D, // quantized sampleTracks performance warning
	0xABBAFC8F, // retargeting warning
	0x651F7AA5, // old stuff in old assets

	// Cloth reports
	0xC1074000, // Matching collidable <...> with transform (bone) <...> [Method: ...].
	0xC1074875, // Report: Processing Cloth Data "Roy Cape" succeeded.

	// Destruction reports
	0x3005e1de, // hkdDestructionTrackerViewer
	0x47cc533b, // hkDefaultDestruction2012Demo::printSceneStats
	0xABBA4566, // hkdAssetProcessingUtil

	// Graphics/loading
	0x704DCE71, // Trying to load a PNG texture which is too large
	0xABBA55FE, // Could not load a texture [...] from any search path, ignoring it.
	0x00023461, // hkgMeshAssetUtil.cpp: No shaders. Ignoring ...
};


// Definitions of variants
struct BootstrapVariant
{
	const char*	m_name;
	BootstrapDemo::TestType m_testType;
	int m_demoTypeMask;
	const char* m_details;
};

#define FOR_ALL_PRODUCTS( MACRO )  MACRO(AI, Ai),\
	MACRO(ANIMATION, Animation),\
	MACRO(BEHAVIOR, Behavior),\
	MACRO(CLOTH, Cloth),\
	MACRO(DESTRUCTION, Destruction),\
	MACRO(DESTRUCTION_2012, Destruction2012),\
	MACRO(PHYSICS, Physics),\
	MACRO(PHYSICS_2012, Physics2012),\
	MACRO(SCRIPT, Script),

#define BOOTSTRAP_PRODUCT_ALL(UPPER, lower) { "Bootstrap" #lower, BootstrapDemo::TEST_NORMAL, DEMO_MASK_ ## UPPER,	"Running all  " #lower " demos." }
#define BOOTSTRAP_PRODUCT_CRITICAL(UPPER, lower) { "Critical/BootstrapCritical" #lower,	BootstrapDemo::TEST_NORMAL, HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_ ## UPPER,	"Running all critical " #lower " demos." }


static const BootstrapVariant g_variants[] =
{
{ "BootstrapAll",			BootstrapDemo::TEST_NORMAL, DEMO_MASK_ALL,			"Running all demos", },
{ "BootstrapSerialize",		BootstrapDemo::TEST_NORMAL, DEMO_MASK_SERIALIZE,	"Running demos using serialization" },
{ "BootstrapCritical",		BootstrapDemo::TEST_NORMAL, DEMO_MASK_ALL_CRITICAL,	"Running all critical demos" },

// Product-specific BootstrapAll variants
FOR_ALL_PRODUCTS( BOOTSTRAP_PRODUCT_ALL )

// Product-specific BootstrapCritical variants
FOR_ALL_PRODUCTS( BOOTSTRAP_PRODUCT_CRITICAL )

{ "BootstrapStats",					BootstrapDemo::TEST_STATISTICS, 				DEMO_MASK_ALL_STATS,		"Running demos to gather statistics" },
{ "BootstrapStatsAI",				BootstrapDemo::TEST_STATISTICS, 				DEMO_MASK_AI_STATS,			"Running AI demos to gather statistics" },
{ "BootstrapStatsAnimation",		BootstrapDemo::TEST_STATISTICS, 				DEMO_MASK_ANIMATION_STATS,			"Running Animation demos to gather statistics" },
{ "BootstrapStatsCritical",			BootstrapDemo::TEST_STATISTICS_CRITICAL,		DEMO_MASK_STATS_CRITICAL,	"Running critical demos to gather statistics" },
{ "BootstrapStatsDetailedTimings",	BootstrapDemo::TEST_STATISTICS_DETAILED, 		DEMO_MASK_ALL_STATS,		"Running demos to gather detailed statistics" },
{ "BootstrapStatsSingleThreaded",	BootstrapDemo::TEST_STATISTICS_SINGLE_THREADED, DEMO_MASK_ALL_STATS,		"Running demos to gather single threaded statistics" },

{ "SerializeAll (Binary)", BootstrapDemo::TEST_SERIALIZE_BINARY, DEMO_MASK_ALL, "Binary serializing all demos" },
{ "SerializeAll (XML)",    BootstrapDemo::TEST_SERIALIZE_XML,    DEMO_MASK_ALL, "XML serializing all demos" },

{ "MemSnapShotAll",    BootstrapDemo::TEST_MEMORY_SNAPSHOT,    DEMO_MASK_ALL, "Memory snapshotting all demos" },

#if defined(HK_ENABLE_DETERMINISM_CHECKS)
	{ "DeterminismAll",    BootstrapDemo::TEST_DETERMINISM, DEMO_MASK_DETERMINISM, "Determinism testing for all supported demos" },
	{ "Mt DeterminismAll",    BootstrapDemo::TEST_MULTITHREADING_DETERMINISM, DEMO_MASK_DETERMINISM, "Multi threading determinism testing for all supported demos" },

	{ "DeterminismCritical",    BootstrapDemo::TEST_DETERMINISM, DEMO_MASK_DETERMINISM_CRITICAL, "Determinism testing for all supported critical demos" },

	#if defined(HK_FEATURE_PRODUCT_PHYSICS)
		{ "DeterminismPhysics",    BootstrapDemo::TEST_DETERMINISM, DEMO_MASK_PHYSICS, "Determinism testing Physics demos" },
	#endif
	#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
		{ "DeterminismPhysics2012",    BootstrapDemo::TEST_DETERMINISM, DEMO_MASK_PHYSICS_2012, "Determinism testing Physics 2012 demos" },
		{ "Mt DeterminismPhysics2012", BootstrapDemo::TEST_MULTITHREADING_DETERMINISM, DEMO_MASK_PHYSICS_2012, "Multi threading determinism testing Physics 2012 demos" },
	#endif
	#if defined(HK_FEATURE_PRODUCT_AI)
		{ "DeterminismAI",    BootstrapDemo::TEST_DETERMINISM, DEMO_MASK_AI, "Determinism testing AI demos" },
		{ "MT DeterminismAI",    BootstrapDemo::TEST_MULTITHREADING_DETERMINISM, DEMO_MASK_AI, "Determinism testing AI demos" },
	#endif
	#if defined(HK_FEATURE_PRODUCT_BEHAVIOR)
		{ "DeterminismBehavior",    BootstrapDemo::TEST_DETERMINISM, DEMO_MASK_BEHAVIOR, "Determinism testing Behavior demos" },
		{ "MT DeterminismBehavior",    BootstrapDemo::TEST_MULTITHREADING_DETERMINISM, DEMO_MASK_BEHAVIOR, "Determinism testing Behavior demos" },
	#endif
#endif


#if defined(HK_ENABLE_MEMORY_EXCEPTION_UTIL)
{ "Memory Exceptions",    BootstrapDemo::TEST_MEMORY_EXCEPTIONS, DEMO_MASK_PHYSICS_2012, "Memory exceptions in Physics 2012 demos" },
#endif
};

static void getDemoLogPathFromBDI( int bdi, extStringBuf& pathOut )
{
	pathOut.printf("%d", bdi);
	pathOut.prepend( "logs/" );
	pathOut.append( ".log" );
}


void BootstrapDemo::mtSafeDeleteDemo(hkDemo* demo)
{
	if ( demo )
	{
		if(m_env->m_options->m_saveOutputToLog)
		{
			hkDemoConsole::getInstance().popStreamWriter();
			// The demo completed safely, so delete the log.
			extStringBuf demoLogPath;
			getDemoLogPathFromBDI(m_demoIndex, demoLogPath);
			hkStringBuf demoErrors(demo->getError());
			addHeaderToTestLogfile(demoLogPath.cString(), m_env->m_menuPath.cString(), (demoErrors.getLength()<=0), m_statsTotalTime);
		}

		hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_AUTO );
		demo->preDeleteDemo();
		delete demo;
		hkReferencedObject::setLockMode( hkReferencedObject::LOCK_MODE_NONE );
	}

	// Restore the demo options
	*m_env->m_options = m_originalOptions;
}

static void hkOutputStatsSummaryToFile( const char* filename, const hkArray<hkDemo::RegressionInfo>& statsRecords, int numThreads, int numSpus );



void BootstrapDemo::preRenderDisplayWorld(hkgViewport* v)
{
	if (m_demo != HK_NULL )
	{
		m_demo->preRenderDisplayWorld(v);
	}
}

void BootstrapDemo::postRenderDisplayWorld(hkgViewport* v)
{
	if (m_demo != HK_NULL )
	{
		m_demo->postRenderDisplayWorld(v);
	}
}

//
// Constructor
//
BootstrapDemo::BootstrapDemo(hkDemoEnvironment* env)
	:	hkDemo(env),
		m_demoTypeMask(0),
		m_testType(TEST_NORMAL),
		m_demo(HK_NULL),
		m_demoIndex(0),
		m_steps(0),
		m_allocatedData(HK_NULL),
		m_originalWorld(HK_NULL),
		m_counter(0),
		m_statsTotalTime(0.0f),
		m_statsNumSamples(0),
		m_runsPerGame(2),
		m_runIndex(0)
{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
	m_forceMultithreadedSimulationBackup = hkpWorld::m_forceMultithreadedSimulation;
#endif

#if !defined (HK_ENABLE_DETERMINISM_CHECKS) || defined (HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
	m_runsPerGame = 1;
#endif

	m_originalTexturePixelsLimit = hkgTexture::s_texturePixelsLimit;
	hkgTexture::s_texturePixelsLimit = 1024*2048;
	m_originalOptions = *m_env->m_options;

	//
	// Process variant settings
	//
	{
		const BootstrapVariant& variant =  g_variants[env->m_variantId];
		m_demoTypeMask = variant.m_demoTypeMask;
		m_testType = variant.m_testType;
	}

#if defined(HK_ENABLE_DETERMINISM_CHECKS) && ! defined(HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
	{
		//m_options.m_stepsPerDemo = 150;
		hkCheckDeterminismUtil::createInstance();
	}
#endif

	// Set m_forcedNumThreads and m_forcedNumSpus, to use when generating stats etc.
	if (m_testType == TEST_STATISTICS_SINGLE_THREADED)
	{
		m_forcedNumThreads = 1;
		m_forcedNumSpus = 1;
	}
	else
	{
		m_forcedNumThreads = hkHardwareInfo::getNumHardwareThreads();
		m_forcedNumSpus = 5;
	}
	m_forcedNumSpus = 0;

#if defined (HK_ENABLE_MEMORY_EXCEPTION_UTIL)
	hkMemoryExceptionTestingUtil::create();
#endif

	//
	// Get suitable demo entries
	//
	{
		const extArray<hkDemoEntry>& db = hkDemoDatabase::getInstance().getDemos();

		// Don't allow duplicate demos (same menu path). This can cause loops in the bootstrapper, when we try to resume
		// past a crashed demo.
		hkStringMap<bool> demoRegistered;
		demoRegistered.reserve( db.getSize() );

		for (int i=0; i<db.getSize(); i++)
		{
			const hkDemoEntry& entry = db[i];

			// Set useDemoTest to the intersection of the demo flags and the bootstrap mask
			// (if any demo flags are in the bootstrap mask useDemoTest will be non-zero).
			int useDemoTest = 0;
			useDemoTest = ( entry.m_demoTypeFlags & m_demoTypeMask );

			// If the bootstrap mask is DEMO_MASK_STATS_CRITICAL clear useDemoTest unless all flags in the
			// bootstrap mask (i.e. both the stats flag and the critical flag) are in the demo flags.
			if ( m_demoTypeMask == DEMO_MASK_STATS_CRITICAL || m_demoTypeMask == DEMO_MASK_AI_STATS || m_demoTypeMask == DEMO_MASK_ANIMATION_STATS )
			{
				useDemoTest =  !(( entry.m_demoTypeFlags & m_demoTypeMask ) ^ m_demoTypeMask );
			}
			// If the bootstrap mask contains HK_DEMO_TYPE_CRITICAL clear useDemoTest for the current demo unless
			// it's both critical itself and at least one of the non-critical flags (e.g. Physics etc.) in the mask
			// is also in the demo.
			else if ( m_demoTypeMask & HK_DEMO_TYPE_CRITICAL )
			{
				if (((entry.m_demoTypeFlags & HK_DEMO_TYPE_CRITICAL) == 0) ||
					((useDemoTest & ~HK_DEMO_TYPE_CRITICAL) == 0))
				{
					useDemoTest = 0;
				}
			}

#if !defined(SPECIAL_DEMO)
			if ( useDemoTest )
#endif
			{
				bool useDemo = true;

					// take all demos if not special demo selected
#ifdef SPECIAL_DEMO
				useDemo = entry.m_menuPath.startsWith(SPECIAL_DEMO);
#endif

				useDemo &= !( entry.m_demoTypeFlags & HK_DEMO_TYPE_DONT_BOOTSTRAP );
				useDemo &= !(( entry.m_demoTypeFlags & HK_DEMO_TYPE_BOOTSTRAP_FIRST_VARIANT_ONLY ) && (entry.m_variantId > 0));

				if ( m_testType == TEST_DETERMINISM || m_testType == TEST_MULTITHREADING_DETERMINISM )
				{
					// these don't work for determinism checks
					useDemo &=  hkString::strStr( entry.m_menuPath.cString(), "ObjectsOnLandscape") == HK_NULL; 


					// These are not physics demos
					useDemo &=  ! entry.m_menuPath.startsWith("Common/Api/Base/DetailedTimers");
					useDemo &=  ! entry.m_menuPath.startsWith("Common/Api/Base/Streams");
					useDemo &=  ! entry.m_menuPath.startsWith("Common/Api/MemoryWalk");
					useDemo &=  ! entry.m_menuPath.startsWith("Common/Base/CustomAttributes");

					useDemo &=  ! entry.m_menuPath.startsWith("CommonApi");

					useDemo &=  ! entry.m_menuPath.startsWith("DemoCommon");
					useDemo &=  ! entry.m_menuPath.startsWith("Animation");

					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/Api/Dynamics/MemoryIssues");
					useDemo &=  ! entry.m_menuPath.startsWith("PhysicsApi/VehiclePhysics/SerializedVehicle");
					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/Test/Feature/Dynamics/AsynchronuousSpecialEffectsThread");	// nondeterministic second thread
					useDemo &=  ! entry.m_menuPath.startsWith("Resources/Physics2012/UseCase/AssetStreaming");

					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/UseCase/CharacterControl");						// non deterministic
					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/UseCase/Fracture");								// slow and not deterministic

					// and those are not needed and/or too slow
					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/Test/Performance/BenchmarkSuite");		// way too slow
					useDemo &=  ! entry.m_menuPath.startsWith("Physics/Test/Performance/BenchmarkSuite/Piles/Big Borg Cube");  // Out of mem	
					useDemo &=  ! entry.m_menuPath.startsWith("Physics/Test/Performance/BenchmarkSuite/Piles/Fractured pillars");  // Out of mem
					useDemo &=  ! ( entry.m_menuPath.startsWith("Geometry/") && (entry.m_menuPath.endsWith(".hkt") || entry.m_menuPath.endsWith(".hkx")) );		// too slow 

					useDemo &=  ! entry.m_menuPath.startsWith("Resources/Common/Api/Serialize/SimpleLoad/Simple"); // unrelevant

					useDemo &=  ! entry.m_menuPath.endsWith("BatchAddRemoveBodies"); // slow
					useDemo &=  ! entry.m_menuPath.startsWith("PhysicsApi/CollisionDetection/Raycasting"); // unrelevant
					useDemo &=  ! entry.m_menuPath.startsWith("PhysicsApi/ContinuousPhysics/DiscreteVsContinuous"); // two worlds in the demo, not supported

					// vehicles are not deterministic because of the phantom used
					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/Api/Vehicle");							// non deterministic
					useDemo &=  ! entry.m_menuPath.startsWith("ShowCase/Gdc2007");								// non deterministic



					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/UseCase/Fountain");				// broadphase border resets position in nondeterminstic order

					useDemo &=  ! entry.m_menuPath.startsWith("Examples/Physics/Ragdoll/SlidingRagdolls"); //demo has nonstandard mt usage, checkDeterminimsUtil breaks
					useDemo &=  ! entry.m_menuPath.endsWith("DestructibleWalls/Destructible Walls");		// uses a second world to simulate the wall
					useDemo &=  ! entry.m_menuPath.endsWith("DestructibleWalls/Single Brick Tmp");			// uses a second world to simulate the wall

					useDemo &=  ! entry.m_menuPath.endsWith("MovingVsFixedCollision");

					// Awfuly slow
					useDemo &=  ! entry.m_menuPath.endsWith("BroadPhaseRayCastStress/16-bit broadphase");

					
					useDemo &=  ! entry.m_menuPath.endsWith("Shift Broadphase Only"); // start index ~175
					useDemo &=  ! entry.m_menuPath.endsWith("Shift Coordinate Space");

					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/Api/Dynamics/World/SlidingWorld");				// breaks in 6.1 due to inconsistent state in the broadphase
					useDemo &=  ! entry.m_menuPath.startsWith("Physics2012/UseCase/Welding");				// no asset


					//
					// AI demos
					//
					useDemo &=  ! entry.m_menuPath.startsWith("Ai/ShowCase/Gdc2010/Mountain, 10001 characters"); // sloooooow
					useDemo &=  ! entry.m_menuPath.startsWith("Ai/UseCase/DynamicEnvironment"); // currently a problem

					//
					// Behavior demos
					//
					useDemo &=  ( hkString::strStr( entry.m_menuPath.cString(), "AsynchronousOnDemandAssetLoading") == HK_NULL ); // breaks, needs investigation

				}



				if( m_testType == TEST_SERIALIZE_BINARY || m_testType == TEST_SERIALIZE_XML )
				{
					if( (entry.m_demoTypeFlags & HK_DEMO_TYPE_PHYSICS_2012) == 0 )
					{
						useDemo = false;
					}
				}

				if( demoRegistered.getWithDefault(entry.m_menuPath.cString(), false) )
				{
					// Already have one version of this demo.
					useDemo = false;
				}

				if ( useDemo )
				{
					DemoEntry newEntry;
					newEntry.m_entry = &entry;
					m_entries.pushBack(newEntry);
					demoRegistered.insert( entry.m_menuPath.cString(), true );
				}
			}
		}
	}

	// Disable warnings
	for (int warnIdx = 0; warnIdx < (int) HK_COUNT_OF(s_ignoredIds); warnIdx++)
	{
		hkError::getInstance().setEnabled( s_ignoredIds[warnIdx], false);
	}


	//
	// Start from last/next demo? (Not supported on Wii currently)
	//
	const BootstrapVariant& variant =  g_variants[env->m_variantId];

	if ((env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_2) || m_env->m_options->m_runLastDemo) ||
		(env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_3) || m_env->m_options->m_runNextDemo) ||
		variant.m_testType == BootstrapDemo::TEST_NORMAL_CONTINUE )
	{
		hkIfstream is(LASTDEMO_FILENAME_BOOTSTRAP);
		if( is.isOk() )
		{
			char name[1024];
			if( is.getline(name, 1024) > 0 )
			{
				// Set demo index
				for (int i = 0; i < m_entries.getSize(); i++ )
				{
					const hkDemoEntry* entry = m_entries[i].m_entry;
					if (entry->m_menuPath.endsWith(name))
					{
						m_demoIndex = i;

						if( env->m_gamePad->isButtonPressed(HKG_PAD_BUTTON_3) || m_env->m_options->m_runNextDemo )
						{
							m_demoIndex++;
						}

						// if this demo is flagged as "don't bootstrap", keep looking for a match.
						if( (entry->m_demoTypeFlags & HK_DEMO_TYPE_DONT_BOOTSTRAP) == 0)
						{
							break;
						}
					}
				}
			}
		}
	}

	if (m_env->m_options->m_bootstrapDemoIndex != -1)
	{
		m_demoIndex = m_env->m_options->m_bootstrapDemoIndex;
		hkcout << "Bootstrap demo index: " << m_demoIndex << "\n";
	}
}

static void outputRegressionString(const hkDemo::RegressionInfo& demoRegression, hkOfstream& regressionOut )
{
	extStringBuf regressionData;
	regressionData.printf("[REGRESSION:%s:%s:%f]\n", demoRegression.m_name.cString(), demoRegression.getUnitsString(), demoRegression.m_value );
	regressionOut << regressionData;
}

BootstrapDemo::~BootstrapDemo()
{
	hkgTexture::s_texturePixelsLimit = m_originalTexturePixelsLimit;

	for (int warnIdx = 0; warnIdx < (int) HK_COUNT_OF(s_ignoredIds); warnIdx++)
	{
		hkError::getInstance().setEnabled( s_ignoredIds[warnIdx], true);
	}

	hkSetLastDemoBootstrap("[DONE]"); // For automated testing.
	hkcout << "[DONE]\n";
	hkcout.flush();
	mtSafeDeleteDemo(m_demo);

#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
	hkpWorld::m_forceMultithreadedSimulation = m_forceMultithreadedSimulationBackup;
#endif

#if defined(HK_ENABLE_DETERMINISM_CHECKS) && ! defined(HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
	{
		hkCheckDeterminismUtil::destroyInstance();
	}
#endif

#if defined (HK_ENABLE_MEMORY_EXCEPTION_UTIL)
	hkMemoryExceptionTestingUtil::destroy();
#endif

	const char* statsDir = (m_env->m_options->m_statsDir == HK_NULL ? "Statistics/" : m_env->m_options->m_statsDir);


	// Create the directory if it doesn't exist
	/*hkFileSystem::Result mkdirRes = */ hkFileSystem::getInstance().mkdir(statsDir);

	hkStringBuf regressionFile(statsDir, getPlatform(true), "_Regression.txt");
	hkOfstream regressionOut(regressionFile.cString());

	if ( ( m_testType == TEST_STATISTICS )
		|| ( m_testType == TEST_STATISTICS_SINGLE_THREADED )
		|| ( m_testType == TEST_STATISTICS_DETAILED ))
	{
		hkSort( m_demoRegressions.begin(), m_demoRegressions.getSize() );

		// Write regression timings out to file.
		if (regressionOut.isOk())
		{
 			for (int i=0; i < m_demoRegressions.getSize(); i++)
 			{
				outputRegressionString( m_demoRegressions[i], regressionOut);
 			}
		}

		// Add the stats to the multiplatform stats file.
		hkStringBuf allStatsFile(statsDir, "Multiplatform_Performance_Statistics");
		hkOutputStatsSummaryToFile( allStatsFile.cString(), m_demoRegressions, m_forcedNumThreads, m_forcedNumSpus );

		// Add the stats to the current platform's stats file.
		hkStringBuf platStatsFile( statsDir, getPlatform(true), "_Performance_Statistics");
		hkOutputStatsSummaryToFile( platStatsFile.cString(), m_demoRegressions, m_forcedNumThreads, m_forcedNumSpus );
		// Create a unique stats file for this (day's) stats run.
		char stringDate[80];
		hkSystemDate::getStringDate(stringDate);
		hkStringBuf uniqueStatsFile;
		uniqueStatsFile.printf("%s%s_%s_%s", statsDir, getPlatform(true), HAVOK_SDK_VERSION_STRING, stringDate ); 
		hkOutputStatsSummaryToFile( uniqueStatsFile.cString(), m_demoRegressions, m_forcedNumThreads, m_forcedNumSpus );
	}
	else // Just report the total time for each product
	{
		// sort?
		for ( hkStorageStringMap<int>::Iterator iter = m_pathToProductIndex.getIterator();
			m_pathToProductIndex.isValid(iter);
			iter = m_pathToProductIndex.getNext(iter)  )
			{
				extStringBuf fullName("Total time for ", m_pathToProductIndex.getKey(iter), " demos");
				RegressionInfo ri;
				ri.m_name = fullName;
				ri.m_value = m_productTimes[ m_pathToProductIndex.getValue(iter) ];
				ri.m_type = RegressionInfo::REGRESSION_SECONDS;

				outputRegressionString( ri, regressionOut);

#if defined(HK_PLATFORM_IS_CONSOLE)
				HK_REGRESSION_REPORT( ri.m_name, ri.getUnitsString(), ri.m_value );
#endif

			}
	}
}

static void fillStatus(extArray<char>& bar, int cur, int max, int maxchars)
{
	if(cur > max)
	{
		cur = max;
	}
	bar.clear();
	if(maxchars > 0)
	{
		bar.reserve(maxchars+1);
		bar.setSize(maxchars, ' ');
		bar[0] = '[';
		for(int i = 1; i < maxchars * cur / max; ++i)
		{
			bar[i] = '#';
		}
		bar.back() = ']';
	}
	bar.pushBack(0);
}

hkStringBuf BootstrapDemo::getStatsFileName( const char* extension )
{
	hkStringBuf statsFileName;

	hkStringBuf fileName (m_entries[m_demoIndex].m_entry->m_menuPath);
	fileName.pathBasename();
	fileName.replace('/', '_');

	const char * statsDir = "Statistics/";
	statsFileName += (m_env->m_options->m_statsDir == HK_NULL ? statsDir : m_env->m_options->m_statsDir);
	if ( m_demoTypeMask == DEMO_MASK_STATS_CRITICAL )
	{
		statsFileName += "Crit_";
	}
	statsFileName += getPlatform();
	statsFileName += "_";
	statsFileName += fileName;
	statsFileName += extension;

	return statsFileName;
}

//
// step demo
//
hkDemo::Result BootstrapDemo::stepDemo()
{
	const int START_INDEX = 0;
	const int END_INDEX = (m_env->m_options->m_bootstrapEndDemoIndex <= 0) ? m_entries.getSize() : m_env->m_options->m_bootstrapEndDemoIndex + 1;

	if( m_demoIndex < START_INDEX )
	{
		m_demoIndex = START_INDEX;
	}

	//
	// No demos left?
	//
	if( m_demoIndex >= END_INDEX )
	{
		return DEMO_STOP;
	}

	const hkDemoEntry* entry = m_entries[m_demoIndex].m_entry;

	//
	// Need to start the next demo?
	//
	if(m_demo == HK_NULL)
	{
		{
			static int coolTheJets;
			if( coolTheJets > 0 )
			{
				coolTheJets -= 1;
				return DEMO_OK;
			}
			coolTheJets = 2;
		}
		{
			// Use fixed thread settings for various bootstrap types.
			if (( m_testType == TEST_MULTITHREADING_DETERMINISM )
				|| ( m_testType == TEST_STATISTICS )
				|| ( m_testType == TEST_STATISTICS_SINGLE_THREADED )
				|| ( m_testType == TEST_STATISTICS_DETAILED )
				|| ( m_testType == TEST_STATISTICS_CRITICAL ) )
			{
				m_env->m_options->m_numThreads = m_forcedNumThreads;
				m_env->m_options->m_numSpus = m_forcedNumSpus;

				if ( m_testType == TEST_STATISTICS_SINGLE_THREADED )
				{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
					hkpWorld::m_forceMultithreadedSimulation = false;
#endif
					m_env->m_options->m_forceMT = false;
				}
				else
				{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
					hkpWorld::m_forceMultithreadedSimulation = true;
#endif
					m_env->m_options->m_forceMT = true;
				}
			}

			hkprintf("%i:%s\n", m_demoIndex, entry->m_menuPath.cString());
			
			if(m_env->m_options->m_saveOutputToLog)
			{
				extStringBuf demoLog;
				getDemoLogPathFromBDI(m_demoIndex, demoLog);
				
				hkRefPtr<hkStreamWriter> logWriter = hkFileSystem::getInstance().openWriter( demoLog.cString() );
				HK_ASSERT(0x0, logWriter->isOk());
				hkDemoConsole::getInstance().addStreamWriter( logWriter );
			}

			hkStringBuf buf; buf.printf(">>> Starting %s'", entry->m_menuPath.cString());
			HK_REPORT( buf.cString() );

			m_env->m_menuPath = entry->m_menuPath;
			m_env->m_demoPath = entry->m_demoPath;
			m_env->m_variantId = entry->m_variantId;
			m_env->m_demoTypeFlags = entry->m_demoTypeFlags;
			m_env->m_resourcePath = entry->m_resourcePath;
			hkSetLastDemoBootstrap(entry->m_menuPath.cString());

			// Clean up as much mem as wqe can from last demo
			{
				m_env->m_displayHandler->clearAndDeallocate();
				m_env->m_textDisplay->clearAndDeallocate();
			}

			{
				hkMemorySystem::getInstance().garbageCollect();
				//hkMemorySystem::getInstance().printStatistics( hkcout );
			}


#if ! defined (HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
			if ( m_testType == TEST_DETERMINISM || m_testType == TEST_MULTITHREADING_DETERMINISM )
			{
				if ( m_runIndex == 0 )
				{
					hkCheckDeterminismUtil::getInstance().startWriteMode(true);
				}
				else
				{
					hkCheckDeterminismUtil::getInstance().startCheckMode();
				}
			}
#endif

			// Create the demo
			hkReferencedObject::lockAll();
			m_env->m_window->getContext()->lock(); // ctors can assume that HKG is locked normally for it

			m_statsTicksPreCreate = hkStopwatch::getTickCounter();
			m_demo = (*entry->m_func)(m_env);
			m_demo->postConstruct();
			m_statsTicksPostCreate = hkStopwatch::getTickCounter();

			m_env->m_window->getContext()->unlock();
			hkReferencedObject::unlockAll();

			// Reset timer info
			{
				hkMonitorStream::getInstance().reset();
				m_statsTotalTime = 0.0f;
				m_statsNumSamples = 0;
			}

			// Create / load the demo scene
			m_env->m_window->getContext()->lock();
			m_demo->createScene();
			m_env->m_window->getContext()->unlock();

			m_steps = 0;
		}

		if (m_testType == TEST_SERIALIZE_BINARY || m_testType == TEST_SERIALIZE_XML)
		{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
			// keep our old one (so that the game dtor can do whatever it is it usually does)
			m_originalWorld = static_cast<hkDefaultPhysics2012Demo*>(m_demo)->m_world;

			// remove the world from the context used by the visualize lib
			static_cast<hkDefaultPhysics2012Demo*>(m_demo)->getPhysicsViewerContext()->removeWorld(m_originalWorld);
#endif
			m_counter = 0;
		}

		if (m_testType == TEST_STATISTICS_DETAILED )
		{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
			hkDefaultPhysics2012Demo* physicsDemo = static_cast<hkDefaultPhysics2012Demo*>(m_demo);
			physicsDemo->m_world->markForWrite();

			if ( physicsDemo->m_vdb == HK_NULL )
			{
				physicsDemo->m_vdb = new hkVisualDebugger( physicsDemo->m_contexts, physicsDemo->m_vdbClassReg );
			}
			physicsDemo->m_vdb->addDefaultProcess("Statistics");
			physicsDemo->m_vdb->capture( getStatsFileName( ".hkm" ).cString() );

			physicsDemo->m_world->unmarkForWrite();
#endif
		}
	}

	const int stepsMax = m_demo->m_bootstrapIterations;

	//
	// Status bar
	//
	{
		int charwidth  = (int)m_env->m_textDisplay->getFont()->getCharWidth();
		int charheight = (int)m_env->m_textDisplay->getFont()->getCharHeight();

		extStringBuf status;
		status.printf("%s\n", entry->m_menuPath.cString());

		{
			extInplaceArray<char, 200> bar;
			int maxchars = (m_env->m_window->getWidth() / charwidth) - 20;
			fillStatus(bar, m_steps, stepsMax, maxchars);
			status.appendPrintf("iter %3i / %3i - %s\n", m_steps, stepsMax, bar.begin());

			int numdemos = m_entries.getSize();
			fillStatus(bar, m_demoIndex, numdemos, maxchars);
			status.appendPrintf("demo %3i / %3i - %s", m_demoIndex, numdemos, bar.begin());
		}

		int starty = m_env->m_window->getHeight() - 4 * charheight;
		m_env->m_textDisplay->outputText(status, charwidth*2, starty, hkUint32(-1), 1, -1, true);
	}

	m_env->m_displayHandler->clear();

	//
	// Divert according to test types
	//
	switch (m_testType)
	{
	case TEST_NORMAL:
	case TEST_NORMAL_CONTINUE:
	case TEST_MEMORY_SNAPSHOT:
		NormalStepDemo();
		break;

	case TEST_STATISTICS:
	case TEST_STATISTICS_SINGLE_THREADED:
	case TEST_STATISTICS_DETAILED:
	case TEST_STATISTICS_CRITICAL:
		{
			if( hkMemorySystem::getInstance().getDebugInterface() )
			{
				HK_ERROR(0x6fe32291, "Trying to generate statistics using debug memory.");
			}
			StatsStepDemo();
			break;
		}

	case TEST_SERIALIZE_BINARY:
	case TEST_SERIALIZE_XML:
		SerializeStepDemo();
		break;

	case TEST_DETERMINISM:
	case TEST_MULTITHREADING_DETERMINISM:
		DeterminismStepDemo();
		break;
	case TEST_MEMORY_EXCEPTIONS:
		MemoryExceptionsStepDemo();
		break;

	default:
		HK_ASSERT2(0x0, false, "Test type unaccounted for");
		break;
	}

	return DEMO_OK;
}


hkDemo::Result BootstrapDemo::stepVisualDebugger()
{
	if (m_demo)
	{
		m_demo->stepVisualDebugger();
	}
	return DEMO_OK;
}

void BootstrapDemo::resetTimerStreams()
{
	if (m_demo)
	{
		m_demo->resetTimerStreams();
	}
	else
	{
		hkDemo::resetTimerStreams();
	}
}

bool BootstrapDemo::visualDebuggerEnabled()
{
	if (m_demo)
	{
		return m_demo->visualDebuggerEnabled();
	}
	else
	{
		return false;
	}
}

const char* BootstrapDemo::getPlatform(hkBool simple)
{
#if defined(HK_PLATFORM_WINRT)
	//xx make system calls
	#ifdef HK_ARCH_X64
		return "WinRT x64";
	#elif defined(HK_ARCH_ARM)
		return "WinRT Arm";
	#else
		return "WinRT x86";
	#endif
#elif defined(HK_PLATFORM_DURANGO)
	return "XboxOne";
#elif defined(HK_PLATFORM_WIN32)
	if (simple)
	{
		return "PC";
	}

	struct _SYSTEM_INFO info;
	GetSystemInfo( &info );

	const char* processor;
	switch (info.wProcessorArchitecture)
	{
	case PROCESSOR_ARCHITECTURE_INTEL :
		// INTEL really means x86 here (see MSDN docs)
		// so check the vendor-specific wProcessorLevel as well.
		switch( info.wProcessorLevel )
		{
		case 6:
			processor = "x86_Intel";
			break;
		case 15:
			processor = "x86_AMD";
			break;
		default:
			hkStringBuf unknownProcessor;
			unknownProcessor.printf("x86_UnknownChipset%d", info.wProcessorLevel);
			processor = unknownProcessor.cString();
			break;
		}
		break;
	case PROCESSOR_ARCHITECTURE_IA64 :
		processor = "IA64";
		break;
	case PROCESSOR_ARCHITECTURE_AMD64 :
		processor = "x64";
		break;
	case PROCESSOR_ARCHITECTURE_UNKNOWN :
	default:
		processor = "UnknownArch";
		break;
	}

	static char buffer[1024];
	hkString::sprintf(buffer, "%s_rev_%x_%d_processors", processor, info.wProcessorRevision, info.dwNumberOfProcessors );
	return buffer;
#elif defined(HK_PLATFORM_LINUX)
	return "Linux";
#elif defined(HK_PLATFORM_WIIU)
	return "WIIU";
#elif defined(HK_PLATFORM_GC)
	return "NGC";
#elif defined(HK_PLATFORM_MAC)
	return "Mac";
#elif defined(HK_PLATFORM_PS3_PPU)
	return "PS3";
#elif defined(HK_PLATFORM_PSP)
	return "PSP";
#elif defined(HK_PLATFORM_PSVITA)
	return "PsVita";
#elif defined(HK_PLATFORM_CTR)
	return "3DS";
#elif defined(HK_PLATFORM_ANDROID)
	return "Android";
#elif defined(HK_PLATFORM_PS4)
	return "PS4";
#elif defined(HK_PLATFORM_TIZEN)
	return "Tizen";
#else
	return "UndefinedPlat";
#endif
}

void BootstrapDemo::StatsStepDemo()
{
	hkStopwatch timer;
	timer.start();
	HK_ON_DEBUG( Result result = ) m_demo->stepDemo();
	HK_ASSERT(0, result == DEMO_OK );
	timer.stop();

	// Skip 1st frame
	if (m_statsNumSamples > 0)
		m_statsTotalTime += timer.getSplitSeconds();
	m_statsNumSamples++;

	hkMonitorStreamFrameInfo frameInfo;
	frameInfo.m_heading = "Timer values are usecs";
	frameInfo.m_indexOfTimer0 = 0;
	frameInfo.m_indexOfTimer1 = -1;
	frameInfo.m_timerFactor0 = 1e6f / float(hkStopwatch::getTicksPerSecond());
	frameInfo.m_timerFactor1 = 1.0f;

	m_steps++;

	//
	// Check for demo ended
	//
	if ( m_steps >= m_demo->m_bootstrapIterations )
	{
		hkStringBuf demoName( m_entries[m_demoIndex].m_entry->m_menuPath );
		demoName.pathBasename();

		hkStringBuf statsFileName = getStatsFileName( ".txt" );

		hkReal average = (m_statsTotalTime*1000000)/m_statsNumSamples;

		// critical statistics
		// will read in previous average times from a file and see if current
		// average time is significantly slower or faster
		if ( m_demoTypeMask == DEMO_MASK_STATS_CRITICAL )
		{
			char readString[80] = "";

			// temporary list to store previous average times
			hkArray< hkReal > previousAverageTimes;
			previousAverageTimes.clear();

			// istream has to be in its own scope so the handle on the file is lost
			// this allows the out stream to write the file later on, can cause problems on the xbox360 without this
			{
				hkIfstream istr(statsFileName.cString());
				if (istr.isOk())
				{
					while(1)
					{
						int nread = istr.getline(readString, 80, '\n');

						// add previous average times to list
						if (nread > 0)
						{
							previousAverageTimes.pushBack(hkString::atof(readString));
						}
						else
						{
							break;
						}
					}
				}
			}

			// if the file is empty
			if (!previousAverageTimes.isEmpty())
			{
				// check for a significant difference faster or slower than 10%
				hkReal oldAverage = previousAverageTimes.back();
				hkBool performanceDifference = (((oldAverage / average) > 1.1f) || ((average / oldAverage) > 1.1f));
				if (performanceDifference)
				{
					// throw error if significant performance difference
					char averageString[80];
					hkString::sprintf (averageString, "%f", average);
					HK_WARN (0x55e10a7e, "Performance of demo: " << demoName << " took "
						<< averageString << " micrsoeconds. (At least 10 percent difference.)" );
				}
			}
			// print out file again
			hkOfstream ofstr(statsFileName.cString());

			for (int i = 0; i < previousAverageTimes.getSize(); i++)
			{
				ofstr.printf("%f\n", previousAverageTimes[i] );
			}
			ofstr.printf("%f\n", average );
		}

		hkArray<hkDemo::RegressionInfo> thisDemoRegressions;
		m_demo->getRegressionInfo(thisDemoRegressions, average);
		m_demoRegressions.append(thisDemoRegressions);

#if defined(HK_PLATFORM_IS_CONSOLE)
		for (int ri=0; ri<thisDemoRegressions.getSize(); ri++)
		{
			const hkDemo::RegressionInfo& regression = thisDemoRegressions[ri];
			HK_REGRESSION_REPORT( regression.m_name, regression.getUnitsString(), regression.m_value );
		}
#endif

		mtSafeDeleteDemo(m_demo);
		m_demo = HK_NULL;
		m_env->m_displayHandler->clear(); // clear debug display

		m_demoIndex++;
	}

}

void BootstrapDemo::NormalStepDemo()
{
	hkStopwatch timer; timer.start();

	m_demo->makeFakeInput();
	Result result = m_demo->stepDemo();

	m_statsTotalTime += timer.getSplitSeconds();
	m_statsNumSamples++;

	m_steps++;

	const int stepsMax = m_demo->m_bootstrapIterations;

	//
	// Check for demo ended
	//
	if( (result != DEMO_OK && result != DEMO_PAUSED)
		|| m_steps >= stepsMax
		|| m_env->wasButtonPressed(HKG_PAD_BUTTON_0) )
	{

		if(result != DEMO_OK)
		{
			HK_WARN(0x2057da93, "Demo " << m_entries[m_demoIndex].m_entry->m_menuPath << " exited early!");
		}

		if( m_testType == TEST_MEMORY_SNAPSHOT )
		{
			hkFileSystem::getInstance().mkdir("mem");
			hkStringBuf sb; sb.printf("mem/memsnap%05i.hkmem", m_demoIndex);
			hkTrackerScanSnapshot* snapshot = hkTrackerSnapshotUtil::createSnapshot();
			hkOstream os(sb.cString());
			hkVdbStreamReportUtil::generateReport(snapshot, os);
			snapshot->removeReference();
		}

		mtSafeDeleteDemo(m_demo);
		m_demo = HK_NULL;
		m_env->m_displayHandler->clear(); // clear debug display

		outputTimings();

		m_demoIndex++;
		if( hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_NULL )
		{
			// skip demos which won't work with the null renderer
			while( m_demoIndex < m_entries.getSize() && (m_entries[m_demoIndex].m_entry->m_demoTypeFlags & HK_DEMO_TYPE_NEEDS_REAL_RENDERER) )
			{
				m_demoIndex += 1;
			}
		}
	}
}

void BootstrapDemo::SerializeStepDemo()
{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
	//HK_ASSERT(0x0, m_demo->getType() == hkDefaultDemo::DEMO_TYPE_PHYSICS);
	hkDefaultPhysics2012Demo* demo = static_cast<hkDefaultPhysics2012Demo*>(m_demo);

	if (m_counter < 1 )
	{
		m_counter++;
		m_env->m_displayHandler->clearDisplay();
		demo->cleanupGraphics();

		if (demo->m_world)
		{
			hkArray<char> memStream;
			//
			// Save and dealloc old
			//
			{
				hkArrayStreamWriter writer( &memStream, hkArrayStreamWriter::ARRAY_BORROW );
				HK_ON_DEBUG( hkBool res = ) hkpHavokSnapshot::save(demo->m_world, &writer, m_testType==TEST_SERIALIZE_BINARY );
				HK_ASSERT( 0x215d080d, res );
			}
			if(0) // debugging dumps
			{
				hkOstream dumpb("dump.bin");
				dumpb.write( memStream.begin(), memStream.getSize() );
				hkOstream dumpt("dump.txt");
				dumpt.write( memStream.begin(), memStream.getSize() );
			}

			if (m_originalWorld != demo->m_world)
			{
				demo->m_world->removeReference();
			}

			if (m_allocatedData)
			{
				m_allocatedData->removeReference();
				m_allocatedData = HK_NULL;
			}


			//
			// Test file load times on PlayStation(R)2 etc:
			if (0)
			{
				{
					hkOstream o("test.hkb");
					o.write(memStream.begin(), memStream.getSize());
				}
				hkIstream i("test.hkb");
				hkpPhysicsData* physicsData = hkpHavokSnapshot::load(i.getStreamReader(), &m_allocatedData);
				demo->m_world = physicsData->createWorld();
			}
			else
			{
				//
				// Load and create new
				//
				hkMemoryStreamReader reader( memStream.begin(), memStream.getSize(), hkMemoryStreamReader::MEMORY_INPLACE );
				hkpPhysicsData* physicsData = hkpHavokSnapshot::load(&reader, &m_allocatedData);
				if( (demo->m_flags & hkDefaultPhysics2012Demo::DEMO_FLAGS_NO_SERIALIZE) == 0 )
				{
					demo->m_world = physicsData->createWorld();
				}
			}

			//
			// some extra debugging:
			//
			/*
			{
				game->m_debugViewerNames.pushBack( hkpBroadphaseViewer::getName()  );
				game->m_debugViewerNames.pushBack( hkpConstraintViewer::getName()  );
				game->m_debugViewerNames.pushBack( hkpContactPointViewer::getName()  );
				game->m_debugViewerNames.pushBack( hkpSimulationIslandViewer::getName()  );
			}
			*/
			demo->setupGraphics();
		}
	}

	if	( demo->m_world &&
		( demo->m_flags & hkDefaultPhysics2012Demo::DEMO_FLAGS_NO_SERIALIZE) == 0 )
	{
		demo->m_world->stepDeltaTime(.016f);
	}

	m_steps++;

	const int stepsMax = m_demo->m_bootstrapIterations;

	if( m_steps >= stepsMax || m_env->wasButtonPressed(HKG_PAD_BUTTON_0) )
	{
		//if (m_demo->getType() == hkDefaultDemo::DEMO_TYPE_PHYSICS )
		{
			demo->m_world = m_originalWorld;
			m_env->m_displayHandler->clearDisplay();
			demo->cleanupGraphics();
			if (m_allocatedData)
			{
				m_allocatedData->removeReference();
				m_allocatedData = HK_NULL;
			}
		}

		mtSafeDeleteDemo(m_demo);
		m_demo = HK_NULL;

		m_demoIndex++;
	}
#endif // HK_FEATURE_PRODUCT_PHYSICS_2012
}

void BootstrapDemo::DeterminismStepDemo()
{
#if ! defined (HK_ENABLE_DETERMINISM_CHECKS)
	HK_ASSERT2( 0xf0212345, false, "Please enable the determinism utility to use this feature");
#endif

	hkStopwatch timer; timer.start();

	m_demo->makeFakeInput();
	m_demo->m_env->checkInputDeterminism();
	m_demo->stepDemo();

	m_statsTotalTime += timer.getSplitSeconds();
	m_statsNumSamples++;

		// increment step
	m_steps++;

	// if demo is not ended yet, continue
	if( m_steps < m_demo->m_bootstrapIterations)
	{
		return;
	}

	mtSafeDeleteDemo(m_demo);
	m_demo = HK_NULL;

	outputTimings();

#if ! defined (HK_ENABLE_NETWORKED_DETERMINISM_UTIL)
	if (m_testType == TEST_DETERMINISM || m_testType == TEST_MULTITHREADING_DETERMINISM)
	{
		hkCheckDeterminismUtil::getInstance().finish();
	}
#endif
	m_env->m_displayHandler->clear(); // clear debug display

	if (++m_runIndex >= m_runsPerGame)
	{
		m_runIndex = 0;

		const int numRepetitions = m_env->m_options->m_numRepetitions;
		if (numRepetitions == 1 || m_env->m_repetitionIndex + 1 == numRepetitions)
		{
			m_env->m_repetitionIndex = 0;
			m_demoIndex++;
		}
		else
		{
			m_env->m_repetitionIndex++;
		}
	}
}

void BootstrapDemo::MemoryExceptionsStepDemo()
{
#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
	hkDefaultPhysics2012Demo* demo = static_cast<hkDefaultPhysics2012Demo*>(m_demo);
	hkpWorld* world = demo->m_world;
	if( !world )
	{
		goto GOTO_NEXT_DEMO;
	}
#endif

	if (m_steps == 0)
	{
		hkMemoryExceptionTestingUtil::startNewDemo();
	}

	m_demo->stepDemo();

	hkMemoryExceptionTestingUtil::endFrame();

	// increment step
	m_steps++;

	// if demo is not ended yet, continue
	if( m_steps < m_demo->m_bootstrapIterations)
	{
		return;
	}

#if defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
GOTO_NEXT_DEMO:
#endif
	mtSafeDeleteDemo(m_demo);
	m_demo = HK_NULL;
	m_env->m_displayHandler->clear(); // clear debug display

	if (++m_runIndex >= m_runsPerGame)
	{
		m_demoIndex++;
		m_runIndex = 0;
	}
}

void BootstrapDemo::advanceFrame()
{
	if (m_demo)
	{
		m_demo->advanceFrame();
	}
	else
	{
		hkDemo::advanceFrame();
	}
}

void BootstrapDemo::outputTimings()
{
	hkUint64 now = hkStopwatch::getTickCounter();
	hkUint64 ticksPerSecond = hkStopwatch::getTicksPerSecond();
	hkUint64 wallTicks = now - m_statsTicksPreCreate;
	hkUint64 ctorTicks = m_statsTicksPostCreate - m_statsTicksPreCreate;
	hkReal wallSecs = hkReal((wallTicks*4096) / ticksPerSecond) / (4096);
	hkReal ctorSecs = hkReal((ctorTicks*4096) / ticksPerSecond) / (4096);
	hkReal simSecs = m_statsTotalTime;
	hkcout.printf("\n%.1f, %.1f, %.1f # {Elapsed, stepDemo, constructor} times\n", wallSecs, simSecs, ctorSecs );

	//
	// Update bootstrap totals
	//

	// Replace e.g. "Ai/Api/blah/..." -> "Ai"
	hkStringBuf menuPath = static_cast<hkStringBuf>(m_entries[m_demoIndex].m_entry->m_menuPath);
	int firstSeparator = menuPath.indexOf('/');
	if(firstSeparator >= 0)
		menuPath.chompEnd( menuPath.getLength() - firstSeparator );

	// Find the index in the map (if it exists)
	int productIndex = m_pathToProductIndex.getWithDefault(menuPath.cString(), -1);
	if (productIndex == -1)
	{
		productIndex = m_productTimes.getSize();
		m_productTimes.expandOne() = 0.0f;
		m_pathToProductIndex.insert(menuPath, productIndex);
	}

	m_productTimes[productIndex] += wallSecs;
}

static hkOfstream* createAndAppend(const char* filename)
{
	hkArray<char>::Temp existing;
	hkLoadUtil(filename).toArray(existing);
	existing.removeAllAndCopy('\r'); // Remove \r characters added by cygwin when the Wii reads text files.

	hkOfstream* outFile = new hkOfstream(filename);
	if(outFile->isOk())
	{
		outFile->write( existing.begin(), existing.getSize() );
	}
	return outFile;
}


static void hkOutputStatsSummaryToFile( const char* filename, const hkArray<hkDemo::RegressionInfo>& regressionInfos, int numThreads, int numSpus )
{
	// CSV

	{
		hkStringBuf csvFilename(filename, ".csv");

		hkStringBuf oldStats;
		hkStringBuf newStats;

		// Read the existing file into a hkString.
		{
			hkStringBuf existing;
			if( hkLoadUtil(csvFilename).toString(existing) )
			{
				// Remove unwanted chars that may have crept in from everyday use.
				existing.replace("\r", "");

				// Excel has a habit of appending commas to the info string, remove them
				int newLine = existing.indexOf('\n');
				hkStringBuf firstLine( existing.cString(), newLine);
				firstLine.replace(",", "");
				oldStats.append(firstLine);
				oldStats.append( existing.cString()+newLine);
			}
		}

		// Remove the help message from the start of the file so that we can easily add new columns, etc.
		const char* csvInfoStr = "NOTE: The numerical values are the average StepDemo() time for each demo in microseconds.\n\n,";
		if( oldStats.indexOf(csvInfoStr) != -1)
		{
			oldStats.chompStart( hkString::strLen(csvInfoStr) );
		}

		// Use the first line from the old stats and add version string, platform and number of threads.
		int idx = (oldStats.indexOf('\n') > 0) ? oldStats.indexOf('\n') : oldStats.getLength();
		newStats.set( oldStats.cString(), idx);
		newStats.appendJoin( ",Havok ", HAVOK_SDK_VERSION_NUM_STRING, " ", BootstrapDemo::getPlatform(true));

		hkStringBuf threadInfo;
		threadInfo.printf( " (%d Threads)\n", numThreads );
		newStats += threadInfo;
		if(oldStats.getLength() - idx > 0)
		{
			oldStats.chompStart(idx+1);
		}
		else
		{
			oldStats = "";
		}

		// Determine the number of columns in the csv file.
		int numCols = 0;
		{
			for( int i = 0; i != -1; )
			{
				i = newStats.indexOf(',', i);
				if( i >= 0 )
				{
					numCols++;
					i++;
				}
			}
		}

		// Add the stats.
		{
			hkStringBuf tmpBuf ;
			for (int i=0; i < regressionInfos.getSize(); i++)
			{
				if(regressionInfos[i].m_type != hkDemo::RegressionInfo::REGRESSION_AVERAGE_STEP_TIME_MICROSECONDS)
					continue;

				//Construct a demo string which matches the start of the csv file
				//this consists of the demo path and demo name (name wrapped in quotes!) separated by a comma
				hkStringBuf curDemo, demoName, demoPath;
				demoPath = demoName = regressionInfos[i].m_name;
				demoName.chompStart(demoName.lastIndexOf("/")+1);
				demoPath.chompEnd(demoPath.getLength() - demoPath.lastIndexOf("/"));
				curDemo.printf("%s,\"%s\"", demoPath.cString(), demoName.cString());
				// Add any lines from the original file that come before the current stats demo.
				if( oldStats.getLength() > 0 )
				{
					//check if old stats begins with the curDemo string, if not, append the next line to the newStats string
					//using strcmp will pop out of this loop when the demo string has lexicographic priority over the csv demo string
					//this keeps the contents of the file in alphabetical order even when new demos are added
					while(oldStats.getLength() > 0 && hkString::strCmp( oldStats.cString(), curDemo.cString() ) < 0 )
					{
						idx = ((oldStats.indexOf('\n') > 0) ? oldStats.indexOf('\n') : oldStats.getLength());
						//add the current line in old stats to new stats
						tmpBuf = oldStats ;
						tmpBuf.chompEnd(tmpBuf.getLength() - idx);
						newStats += tmpBuf;
						newStats += ",\n";
						//move old stats on by the length of a line
						if(oldStats.getLength() - idx > 0)
						{
							oldStats.chompStart(idx+1);
						}
						else
						{
							oldStats = "";
						}
					}
				}

				if( oldStats.indexOf( curDemo ) != -1)
				{
					//we have matched the current demo to the top line in the oldStats string
					//so pull the contents of this line into the curDemo string
					idx = (oldStats.indexOf('\n') > 0) ? oldStats.indexOf('\n') : oldStats.getLength();
					curDemo = oldStats ;
					curDemo.chompEnd(curDemo.getLength() - idx) ;
					if(oldStats.getLength() - idx > 0)
					{
						oldStats.chompStart(idx+1);
					}
					else
					{
						oldStats = "";
					}
				}
				else
				{
					//new demo, add in dummy columns to keep excel happy
					for( int j = 0; j < numCols-1; j++ )
					{
						curDemo += ",";
					}
				}
				//format a line for insertion into the newStats string
				hkStringBuf curLine;
				curLine.printf( "%s,%f\n", curDemo.cString(), regressionInfos[i].m_value );
				newStats += curLine;
			}

			// Add any remaining lines from the original file.
			while( oldStats.getLength() > 0 )
			{
				idx = (oldStats.indexOf('\n') > 0) ? oldStats.indexOf('\n') : oldStats.getLength();
				tmpBuf = oldStats;
				tmpBuf.chompEnd(tmpBuf.getLength() - idx);
				newStats += tmpBuf;
				newStats += "\n";
				if(oldStats.getLength() - idx > 0)
				{
					oldStats.chompStart(idx+1);
				}
				else
				{
					oldStats = "";
				}
			}
		}

		// Add the help string to the start of the file.
		hkStringBuf csvFileContents(csvInfoStr);
		csvFileContents += newStats;

		// Write out the new csv stats file.
		hkOfstream* newCsvFile = new hkOfstream( csvFilename.cString() );
		if(newCsvFile->isOk())
		{
			newCsvFile->write( csvFileContents.cString(), csvFileContents.getLength() );
		}
		delete newCsvFile;
	}

	// HTML
	{
		hkStringBuf htmlFilename(filename, ".html");
		hkOfstream* htmlFile = createAndAppend( htmlFilename.cString() );

		if(htmlFile->isOk())
		{
			hkStringBuf terminator("\n</body></html>");
			htmlFile->getStreamWriter()->seek(0, hkStreamWriter::STREAM_END);

			if (htmlFile->getStreamWriter()->tell() < terminator.getLength())
			{
				// New Empty File
				htmlFile->printf("<html>\n"
					"\t<link href=\"stats.css\" rel=\"stylesheet\" type=\"text/css\">\n"
					"\t<body>\n");
				htmlFile->printf(terminator.cString());
			}

			int eofPos = htmlFile->getStreamWriter()->tell();
			htmlFile->getStreamWriter()->seek( eofPos-terminator.getLength(), hkStreamWriter::STREAM_SET);

			htmlFile->printf("<p>Havok Version: %s</p>\n", HAVOK_SDK_VERSION_NUM_STRING);
			htmlFile->printf("<p>Platform: %s</p>\n", BootstrapDemo::getPlatform());
			htmlFile->printf("<p>Threads: %d</p>\n", numThreads );
			htmlFile->printf("<table>\n<tr><td><p>Demo Path</p></td><td><p>Demo Name</p></td><td><p>Average StepDemo() time - microseconds</p></td></tr>\n");
			for (int i=0; i < regressionInfos.getSize(); i++)
			{
				if(regressionInfos[i].m_type != hkDemo::RegressionInfo::REGRESSION_AVERAGE_STEP_TIME_MICROSECONDS)
					continue;

				int splitIndex = hkString::lastIndexOf(regressionInfos[i].m_name, '/');
				hkStringBuf lhs(regressionInfos[i].m_name.cString(), splitIndex);

				htmlFile->printf("<tr> <td><p>%s</p></td> <td><p>%s</p></td> <td align=right><p>%7.3f</p></td></tr>\n",
					lhs.cString(),
					regressionInfos[i].m_name.cString()+splitIndex+1,
					regressionInfos[i].m_value );
			}
			htmlFile->printf("</table>\n");
			htmlFile->printf("<hr/>\n%s", terminator.cString());
		}

		delete htmlFile;
	}
}


#if !defined( HK_PLATFORM_WII ) || defined(HK_INTERNAL_BUILD)
static const char helpString[] = "Test bootstrap [Hold \x12\\\x13 and press \x1A to continue from last\\next demo]";
#else
static const char helpString[] = "Test bootstrap";
#endif

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( BootstrapDemo, HK_DEMO_TYPE_OTHER, BootstrapVariant, g_variants, helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
