/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */


#include <Demos/demos.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>


// Vertex Deformation
#include <Common/Base/Config/hkConfigSimd.h>
#include <Animation/Animation/Deform/Skinning/Fpu/hkaFPUSkinningDeformer.h>
#include <Animation/Animation/Deform/Skinning/Simd/hkaSimdSkinningDeformer.h>

// Mesh info
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Mesh/hkxMeshSection.h>
#include <Common/SceneData/Mesh/hkxVertexBuffer.h>
#include <Common/SceneData/Mesh/hkxVertexUtil.h>

// Skeletal Animation
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Animation/hkaAnimationBinding.h>
#include <Animation/Animation/Animation/hkaAnnotationTrack.h>
#include <Animation/Animation/Animation/hkaAnimation.h>
#include <Animation/Animation/Animation/Interleaved/hkaInterleavedUncompressedAnimation.h>
#include <Animation/Animation/Motion/hkaAnimatedReferenceFrame.h>
#include <Animation/Animation/Motion/Default/hkaDefaultAnimatedReferenceFrame.h>
#include <Animation/Animation/Rig/hkaBone.h>
#include <Animation/Animation/Rig/hkaSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Animation/Rig/hkaBoneAttachment.h>
#include <Animation/Animation/Rig/hkaSkeletonUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperData.h>

// Deformation
#include <Animation/Animation/Deform/Skinning/hkaMeshBinding.h>

// Display
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

// Graphics Stuff
#include <Graphics/Common/Math/hkgMath.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/Geometry/VertexSet/hkgVertexSet.h>
#include <Graphics/Common/DisplayContext/hkgDisplayContext.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Bridge/System/hkgSystem.h>

// Asset Management
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

// Serialization
#include <Common/Base/Reflection/hkClass.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>


void HK_CALL AnimationUtils::drawWeightedVertices( const hkxVertexBuffer& buffer, int bone )
{
	const hkxVertexDescription& vdesc = buffer.getVertexDesc();
	const hkxVertexDescription::ElementDecl* pdecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0 );
	const hkxVertexDescription::ElementDecl* wdecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0 );
	const hkxVertexDescription::ElementDecl* idecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );

	// If we have position, weights and indices
	if (pdecl && wdecl && idecl)
	{
		const char* pbase = static_cast<const char*>( buffer.getVertexDataPtr(*pdecl) );
		const char* wbase = static_cast<const char*>( buffer.getVertexDataPtr(*wdecl) );
		int pstride = pdecl->m_byteStride;
		int wstride = wdecl->m_byteStride;

		int numVerts = buffer.getNumVertices();

		if(idecl->m_type == hkxVertexDescription::HKX_DT_UINT8)
		{
			const char* ibase = static_cast<const char*>( buffer.getVertexDataPtr(*idecl) );
			int istride = idecl->m_byteStride;

			for( int vertIndex = 0; vertIndex < numVerts; ++vertIndex )
			{
				const hkFloat32* pos = reinterpret_cast<const hkFloat32*>(pbase + (pstride * vertIndex));
				const hkUint8* weights = reinterpret_cast<const hkUint8*>(wbase + (wstride * vertIndex));
				const hkUint8* indices = reinterpret_cast<const hkUint8*>(ibase + (istride * vertIndex));

				hkReal col=0;
				for (int i=0; i < 4; i++)
				{
					if (indices[i] == bone)
					{
						col += hkReal(weights[i])/hkReal(255);
					}
				}

				hkColor::Argb color = hkColor::rgbFromFloats(col,col,col);
				hkVector4 position; position.load<3,HK_IO_NATIVE_ALIGNED>(pos);
				HK_DISPLAY_STAR(position, hkReal(5), color);
			}
		}
		else if(idecl->m_type == hkxVertexDescription::HKX_DT_INT16)
		{
			const hkUint16* ibase = static_cast<const hkUint16*>( buffer.getVertexDataPtr(*idecl) );
			int istride = idecl->m_byteStride/sizeof(hkUint16);

			for( int vertIndex = 0; vertIndex < numVerts; ++vertIndex )
			{
				const hkFloat32* pos = reinterpret_cast<const hkFloat32*>(pbase + (pstride * vertIndex));
				const hkUint8* weights = reinterpret_cast<const hkUint8*>(wbase + (wstride * vertIndex));
				const hkUint16* indices = reinterpret_cast<const hkUint16*>(ibase + (istride * vertIndex));

				hkReal col=0;
				for (int i=0; i < 4; i++)
				{
					if (indices[i] == bone)
					{
						col += hkReal(weights[i])/hkReal(255);
					}
				}

				hkColor::Argb color = hkColor::rgbFromFloats(col,col,col);
				hkVector4 position; position.load<3,HK_IO_NATIVE_ALIGNED>(pos);
				HK_DISPLAY_STAR(position, hkReal(5), color);
			}
		}
	}
}

void HK_CALL AnimationUtils::drawVertices( const hkxVertexBuffer& buffer, hkColor::Argb color )
{
	const hkxVertexDescription& vdesc = buffer.getVertexDesc();
	const hkxVertexDescription::ElementDecl* pdecl = vdesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0 );
	
	if (pdecl)
	{
		const char* pbase = static_cast<const char*>( buffer.getVertexDataPtr(*pdecl) );
		int pstride = pdecl->m_byteStride;
		int numVerts = buffer.getNumVertices();

		for( int vertIndex = 0; vertIndex < numVerts; ++vertIndex )
		{
			const hkFloat32* pos = reinterpret_cast<const hkFloat32*>(pbase + (pstride * vertIndex));
			hkVector4 position; position.load<3,HK_IO_NATIVE_ALIGNED>(pos);
			HK_DISPLAY_STAR(position, hkReal(5), color);
		}
	}
}

void AnimationUtils::drawPose( const hkaPose& pose, const hkQsTransform& worldFromModel, hkColor::Argb color, hkReal size, hkBool showOrigin, hkBool showLabels)
{
#ifndef HK_DISABLE_DEBUG_DISPLAY
	const hkaSkeleton* skeleton = pose.getSkeleton();

	HK_DISPLAY_MODEL_SPACE_POSE( skeleton->m_bones.getSize(), skeleton->m_parentIndices.begin(), pose.getSyncedPoseModelSpace().begin(), worldFromModel, color );


	if (showLabels)
	{
		hkDebugDisplay& d = hkDebugDisplay::getInstance();
		const hkInt16* parents = skeleton->m_parentIndices.begin();
		const hkQsTransform* bones = pose.getSyncedPoseModelSpace().begin();
		for (int i=0; i<skeleton->m_bones.getSize(); i++)
		{
			hkVector4 p1, p2;
			p1 = bones[i].getTranslation();
			if (parents[i] == -1)
			{
				p2 = p1;
			}
			else
			{
				p2 = bones[parents[i]].getTranslation();
			}

			hkVector4 bonemid;
			bonemid.setAdd(p1, p2); bonemid.mul(hkSimdReal_Half);
			bonemid._setTransformedPos(worldFromModel, bonemid);

			hkStringBuf s;
			s.printf("%d", i);
			d.display3dText(s.cString(), bonemid, color, 0, 0);
		}
	}
#endif
}

void AnimationUtils::drawMapping( const hkaSkeletonMapperData& mapping, const hkQsTransform& worldFromModelA, const hkQsTransform& worldFromModelB, hkReal size, hkBool showRootTranslation, hkBool showLabels )
{
#ifndef HK_DISABLE_DEBUG_DISPLAY

	HK_ASSERT2(0x22440047, mapping.m_partitionMap.getSize() == 0, "This function does not work with partitions.");

	// Setup color map storage
	hkColor::Argb black = hkColor::BLACK;
	hkArray<hkColor::Argb> boneIdxToColorMapA( mapping.m_skeletonA->m_bones.getSize(), black );
	hkArray<hkColor::Argb> boneIdxToColorMapB( mapping.m_skeletonB->m_bones.getSize(), black );

	// Current index into color palette
	int curColorIndex = 0;

	// Store colors for simple mappings
	for ( int simpleMappingIt = 0; simpleMappingIt < mapping.m_simpleMappings.getSize(); simpleMappingIt++ )
	{
		const hkColor::Argb color = hkColor::getPaletteColor( curColorIndex++ );

		const hkaSkeletonMapperData::SimpleMapping& simpleMapping = mapping.m_simpleMappings[ simpleMappingIt ];
		boneIdxToColorMapA[ simpleMapping.m_boneA ] = color;
		boneIdxToColorMapB[ simpleMapping.m_boneB ] = color;
	}

	// Store colors for chain mappings
	for ( int chainMappingIt = 0; chainMappingIt < mapping.m_chainMappings.getSize(); chainMappingIt++ )
	{
		const hkColor::Argb color = hkColor::getPaletteColor( curColorIndex++ );

		const hkaSkeletonMapperData::ChainMapping& chainMapping = mapping.m_chainMappings[ chainMappingIt ];
		for ( int boneItA = chainMapping.m_startBoneA; boneItA < chainMapping.m_endBoneA; boneItA++ )
		{
			boneIdxToColorMapA[ boneItA ] = color;
		}
		for ( int boneItB = chainMapping.m_startBoneB; boneItB < chainMapping.m_endBoneB; boneItB++ )
		{
			boneIdxToColorMapB[ boneItB ] = color;
		}
	}

	// Unmapped bones will have black by default

	// Draw ref poses with mapped colors
	for ( int skelIt = 0; skelIt < 2; skelIt++ )
	{
		const hkaSkeleton* skeleton;
		const hkQsTransform* worldFromModel;
		hkArray<hkColor::Argb>* boneIdxToColorMap;

		if ( skelIt == 0 )
		{
			skeleton = mapping.m_skeletonA;
			worldFromModel = &worldFromModelA;
			boneIdxToColorMap = &boneIdxToColorMapA;
		}
		else
		{
			skeleton = mapping.m_skeletonB;
			worldFromModel = &worldFromModelB;
			boneIdxToColorMap = &boneIdxToColorMapB;
		}

		hkaPose pose( skeleton );
		pose.setToReferencePose();
		const hkQsTransform* modelSpacePose = pose.getSyncedPoseModelSpace().begin();
		const hkInt16* parentIndices = skeleton->m_parentIndices.begin();

		// Copied from displayModelSpacePose
		{
			for (hkInt16 boneIdxIt = 0; boneIdxIt < skeleton->m_bones.getSize(); boneIdxIt++)
			{
				const hkColor::Argb color = (*boneIdxToColorMap)[boneIdxIt];

				hkVector4 p1;

				p1._setTransformedPos(*worldFromModel, modelSpacePose[boneIdxIt].getTranslation());

				hkSimdReal boneLen = hkSimdReal_1;

				const hkInt16 parent = parentIndices[boneIdxIt];

				// Display connections between child and parent
				if (parent == -1)
				{
					hkVector4 p2;
					p2._setTransformedPos(*worldFromModel, hkVector4::getZero());
					HK_DISPLAY_LINE(p1, p2, color);			
				}
				else
				{
					hkVector4 p2;
					p2._setTransformedPos(*worldFromModel, modelSpacePose[parent].getTranslation());
					HK_DISPLAY_LINE(p1, p2, color);

					hkVector4 bone; bone.setSub(p1,p2);
					boneLen.setClamped(bone.length<3>(), hkSimdReal::fromFloat(hkReal(0.1f)), hkSimdReal::fromFloat(hkReal(10)));
				}

				const hkVector4 worldPosition = p1;
				hkQuaternion worldRotation; worldRotation.setMul(worldFromModel->getRotation(), modelSpacePose[boneIdxIt].getRotation());

				// Display local axis 
				{ 
					hkVector4 boneLocalFrame;			
					const hkSimdReal boneAxisSize = hkSimdReal::fromFloat(size) * boneLen;

					hkVector4 p2;
					boneLocalFrame.setMul(hkVector4::getConstant<HK_QUADREAL_1000>(), boneAxisSize);
					p2._setRotatedDir(worldRotation, boneLocalFrame);
					p2.add(worldPosition);
					HK_DISPLAY_LINE(worldPosition, p2, 0x7fff0000);

					boneLocalFrame.setMul(hkVector4::getConstant<HK_QUADREAL_0100>(), boneAxisSize);
					p2._setRotatedDir( worldRotation, boneLocalFrame);
					p2.add(worldPosition);
					HK_DISPLAY_LINE(worldPosition, p2, 0x7f00ff00);

					boneLocalFrame.setMul(hkVector4::getConstant<HK_QUADREAL_0010>(), boneAxisSize);
					p2._setRotatedDir(worldRotation, boneLocalFrame);
					p2.add(worldPosition);
					HK_DISPLAY_LINE(worldPosition, p2, 0x7f0000ff);
				}

				// Copied from AnimationUtils::drawPose
				{
					if (showLabels)
					{
						hkVector4 p1model, p2model;
						p1model = modelSpacePose[boneIdxIt].getTranslation();
						if (parentIndices[boneIdxIt] == -1)
						{
							p2model = p1model;
						}
						else
						{
							p2model = modelSpacePose[parentIndices[boneIdxIt]].getTranslation();
						}

						hkVector4 bonemid;
						bonemid.setAdd(p1model, p2model); bonemid.mul(hkSimdReal_Half);
						bonemid._setTransformedPos(*worldFromModel, bonemid);

						hkStringBuf s;
						s.printf("[%d]%s", boneIdxIt, skeleton->m_bones[boneIdxIt].m_name.cString() );
						HK_DISPLAY_3D_TEXT(s.cString(), bonemid, color);
					}
				}
			}
		}
	}

#endif
}

template <class SkinBinding>
void HK_CALL AnimationUtils::setupSkinBinding( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, SkinBinding* binding)
{
	const hkxVertexDescription& iDesc = iBuffer->getVertexDesc();
	const hkxVertexDescription::ElementDecl* iposDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_POSITION, 0);
	const hkxVertexDescription::ElementDecl* inormDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_NORMAL, 0);
	const hkxVertexDescription::ElementDecl* itangDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_TANGENT, 0);
	const hkxVertexDescription::ElementDecl* ibinormDecl = iDesc.getElementDecl(hkxVertexDescription::HKX_DU_BINORMAL, 0);
	const hkxVertexDescription::ElementDecl* boneWeights = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDWEIGHTS, 0 );
	const hkxVertexDescription::ElementDecl* boneIndices = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );

	hkString::memSet(binding, 0, sizeof(SkinBinding));
	
	if (iposDecl && input->m_deformPosition )
	{
		binding->m_iPosBase = reinterpret_cast<const hkFloat32*>( iBuffer->getVertexDataPtr(*iposDecl) );
		binding->m_iPosStride = (hkUint8)( iposDecl->m_byteStride / sizeof(hkFloat32) );
	}

	if (inormDecl && input->m_deformNormal )
	{
		binding->m_iNormBase = reinterpret_cast<const hkFloat32*>( iBuffer->getVertexDataPtr(*inormDecl) );
		binding->m_iNormStride = (hkUint8)( inormDecl->m_byteStride / sizeof(hkFloat32) );
	}

	if (itangDecl && input->m_deformTangent )
	{
		binding->m_iTangentBase = reinterpret_cast<const hkFloat32*>( iBuffer->getVertexDataPtr(*itangDecl) );
		binding->m_iTangentStride = (hkUint8)( itangDecl->m_byteStride  / sizeof(hkFloat32) );
	}

	if (ibinormDecl && input->m_deformBinormal )
	{
		binding->m_iBinormBase = reinterpret_cast<const hkFloat32*>( iBuffer->getVertexDataPtr(*ibinormDecl) );
		binding->m_iBinormStride = (hkUint8)( ibinormDecl->m_byteStride / sizeof(hkFloat32) );
	}

	if (boneWeights)
	{
		binding->m_iWeightBase = reinterpret_cast<const hkUint8*>( iBuffer->getVertexDataPtr(*boneWeights) );
		binding->m_iWeightStride = (hkUint8)( boneWeights->m_byteStride );
	}

	if (boneIndices )
	{
		binding->setBoneIndicesDataPtr(*iBuffer);
	}
	binding->m_bonesPerVertex = 4;

	HKG_VERTEX_FORMAT f = oBuffer->getVertexFormat();
	binding->m_oPosBase = (f & HKG_VERTEX_FORMAT_POS) && input->m_deformPosition ? (hkFloat32*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_POS) : HK_NULL;
	binding->m_oNormBase = (f & HKG_VERTEX_FORMAT_NORMAL) && input->m_deformNormal ? (hkFloat32*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_NORMAL) : HK_NULL;
	binding->m_oBinormBase = (f & HKG_VERTEX_FORMAT_BITANGENT) && input->m_deformBinormal ? (hkFloat32*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_BITANGENT) : HK_NULL;
	binding->m_oTangentBase = (f & HKG_VERTEX_FORMAT_TANGENT) && input->m_deformTangent ? (hkFloat32*)oBuffer->getVertexComponentStartPointer(HKG_VERTEX_COMPONENT_TANGENT) : HK_NULL;
	binding->m_oPosStride = (hkUint8)( binding->m_oPosBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_POS) / sizeof(hkFloat32) : HK_NULL );
	binding->m_oNormStride = (hkUint8)( binding->m_oNormBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_NORMAL) / sizeof(hkFloat32) : HK_NULL );
	binding->m_oBinormStride = (hkUint8)( binding->m_oBinormBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_BITANGENT) / sizeof(hkFloat32) : HK_NULL );
	binding->m_oTangentStride = (hkUint8)( binding->m_oTangentBase ? oBuffer->getVertexComponentStride(HKG_VERTEX_COMPONENT_TANGENT) / sizeof(hkFloat32) : HK_NULL );

	// SIMD aligned outputs or not? Input needs to be aligned only for SIMD skinners and are with our demos and scene data
	binding->m_outputBufferIsAligned = binding->m_oPosBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oPosBase) >= HK_FLOAT_ALIGNMENT) && (binding->m_oNormStride % 4 == 0) : true;
	binding->m_outputBufferIsAligned = binding->m_outputBufferIsAligned && ( binding->m_oNormBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oNormBase) >= HK_FLOAT_ALIGNMENT) && (binding->m_oNormStride % 4 == 0) : true );
	binding->m_outputBufferIsAligned = binding->m_outputBufferIsAligned && ( binding->m_oBinormBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oBinormBase) >= HK_FLOAT_ALIGNMENT) && (binding->m_oBinormStride % 4 == 0) : true );
	binding->m_outputBufferIsAligned = binding->m_outputBufferIsAligned && ( binding->m_oTangentBase? (hkxVertexUtil::getAlignment((hkUlong)binding->m_oTangentBase) >= HK_FLOAT_ALIGNMENT) && (binding->m_oTangentStride % 4 == 0) : true );

	binding->m_numVerts = iBuffer->getNumVertices();
}

template void HK_CALL AnimationUtils::setupSkinBinding<hkaSkinBinding>( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, hkaSkinBinding* binding);
template void HK_CALL AnimationUtils::setupSkinBinding<hkaSkinLargeBinding>( const hkaVertexDeformerInput* input, const hkxVertexBuffer* iBuffer, hkgVertexSet* oBuffer, hkaSkinLargeBinding* binding);

hkBool HK_CALL AnimationUtils::usingInt8BoneIndices(const class hkxVertexBuffer* iBuffer)
{
	const hkxVertexDescription& iDesc = iBuffer->getVertexDesc();
	const hkxVertexDescription::ElementDecl* boneIndices = iDesc.getElementDecl( hkxVertexDescription::HKX_DU_BLENDINDICES, 0 );
	
	if(boneIndices != HK_NULL)
	{
		if( boneIndices->m_type == hkxVertexDescription::HKX_DT_UINT8 )
		{
			return true;
		}
		else if( boneIndices->m_type == hkxVertexDescription::HKX_DT_INT16)
		{
			return false;
		}
	}

	return true;
}

void HK_CALL AnimationUtils::skinMesh(hkQsTransform* poseLocal, const hkArray<hkRefPtr<hkaMeshBinding> >& skinBinding, const hkQsTransform& worldTransform, const hkgSceneDataConverter& graphicsScene)
{
	
	if( skinBinding.getSize() < 0 )
	{
		return;
	}

	// Assumes that all skin bindings are pointing to the same hkaSkeleton
	hkaSkeleton* skeleton = skinBinding[0]->m_skeleton;	
	hkArray<hkQsTransform> poseLocalAsArray(poseLocal, skeleton->m_bones.getSize(), skeleton->m_bones.getSize());	
	hkaPose poseLocalAsPose(hkaPose::LOCAL_SPACE, skeleton, poseLocalAsArray);
	hkTransform worldTransformAsTransform; worldTransform.copyToTransform(worldTransformAsTransform);
	skinMesh(poseLocalAsPose, skinBinding, worldTransformAsTransform, graphicsScene);
		
}

void HK_CALL AnimationUtils::skinMesh(hkQsTransform* poseLocalAsArray, const hkArray<hkRefPtr<hkaMeshBinding> >& skinBinding, const hkArray< hkRefPtr<hkgDisplayObject> >& displayObjects, const hkQsTransform& worldFromModel, const hkgSceneDataConverter& graphicsScene)
{
	if( skinBinding.getSize() < 0 )
	{
		return;
	}

	// Assumes that all skin bindings are pointing to the same hkaSkeleton
	hkaSkeleton* skeleton = skinBinding[0]->m_skeleton;	

	// Construct a pose object for skinning
	hkaPose poseLocalAsPose(hkaPose::LOCAL_SPACE, skeleton, poseLocalAsArray, skeleton->m_bones.getSize());

	for( hkInt32 i = 0; i < skinBinding.getSize(); ++i )
	{
		// Assumes either a straight map (null map) or a single one (1 palette)
		int boneCount = skeleton->m_bones.getSize();
		const hkInt16* usedBones = skinBinding[i]->m_mappings.getSize() > 0 ? skinBinding[i]->m_mappings[0].m_mapping.begin() : HK_NULL;
		int numUsedBones = usedBones? skinBinding[i]->m_mappings[0].m_mapping.getSize() : boneCount;

		// Work with the pose in world
		const hkArray<hkQsTransform>& poseInWorld = poseLocalAsPose.getSyncedPoseModelSpace();

		// Construct the composite world transform
		hkLocalArray<hkTransform> compositeWorldInverse( boneCount );
		compositeWorldInverse.setSize( boneCount );

		// Multiply through by the bind pose inverse world inverse matrices
		for (int p=0; p < numUsedBones; p++)
		{
			int boneIndex = usedBones? usedBones[p] : p;
			compositeWorldInverse[p].setMul(poseInWorld[ boneIndex ], skinBinding[i]->m_boneFromSkinMeshTransforms[ boneIndex ]);
		}

		// Construct the world from model as an hkTransform, not hkQsTransform
		hkTransform worldFromModelAsTransform; worldFromModelAsTransform.set(worldFromModel.m_rotation, worldFromModel.m_translation);

		// Skin the mesh
		AnimationUtils::skinMesh(*(skinBinding[i]->m_mesh), displayObjects[i], worldFromModelAsTransform, compositeWorldInverse.begin(), compositeWorldInverse.getSize(), graphicsScene);
	}	
}

void HK_CALL AnimationUtils::skinMesh(const class hkxMesh& inputMesh, const hkTransform& worldTransform, const hkTransform* worldCompositeMatrices, int numWorldCompositeMatrices, const hkgSceneDataConverter& graphicsScene)
{
	const int displayObjIdx = hkgAssetConverter::findFirstInMapping( graphicsScene.m_meshes, &inputMesh ); // Note : will ignore instancing..
	if ( displayObjIdx >=0 )
	{
		// Ste the graphics transform
		hkgDisplayObject* dObj = (hkgDisplayObject*)graphicsScene.m_meshes[ displayObjIdx ].m_hkgObject;
		
		skinMesh( inputMesh, dObj, worldTransform, worldCompositeMatrices, numWorldCompositeMatrices, graphicsScene );
	}
}

static inline bool _usesVB( const hkgDisplayObject* d, const hkgVertexSet* v )
{
	const int ng = d->getNumGeometry();
	for (int gi=0; gi < ng; ++gi)
	{
		const hkgGeometry* g = d->getGeometry(gi);
		const int nmfs = g->getNumMaterialFaceSets();
		for (int mfi=0; mfi < nmfs; ++mfi)
		{
			const hkgMaterialFaceSet* m = g->getMaterialFaceSet(mfi);
			const int nfs = m->getNumFaceSets();
			for (int mfs=0; mfs < nfs; ++mfs)
			{
				const hkgFaceSet* f = m->getFaceSet(mfs);
				if (f->getVertexSet() == v)
					return true;
			}
		}
	}
	return false;
}
void HK_CALL AnimationUtils::skinMesh(const class hkxMesh& inputMesh, class hkgDisplayObject* dObj, const hkTransform& worldTransform, const hkTransform* worldCompositeMatrices, int numWorldCompositeMatrices, const class hkgSceneDataConverter& graphicsScene)
{
	HK_TIME_CODE_BLOCK("AnimationUtils::skinMesh", HK_NULL);
	// Ste the graphics transform
	hkTransform t = worldTransform;
	t.getTranslation().setW(hkSimdReal_1);	// hkTransform makes no guarantees about this entry, but display objects assume all 16 components are used!
	t.getColumn(0).zeroComponent<3>();
	t.getColumn(1).zeroComponent<3>();
	t.getColumn(2).zeroComponent<3>();
	dObj->setTransformAligned( t );

	hkaVertexDeformerInput vdi;
	vdi.m_deformPosition = true;
	vdi.m_deformNormal = true;

	HKG_VERTEX_FORMAT lockFormat = HKG_VERTEX_FORMAT_POS | 
		(vdi.m_deformNormal? HKG_VERTEX_FORMAT_NORMAL : 0) |
		(vdi.m_deformTangent? HKG_VERTEX_FORMAT_TANGENT: 0) |
		(vdi.m_deformBinormal? HKG_VERTEX_FORMAT_BITANGENT : 0);

	graphicsScene.m_context->lock();

	// For each mesh section
	for (int s=0; s < inputMesh.m_sections.getSize(); ++s)
	{
		const hkxMeshSection& msc = *inputMesh.m_sections[s];

		hkxVertexBuffer* vIn = msc.m_vertexBuffer;

		// Find the vertex buffer for display that corresponds to this loaded vertex buffer
		extArray<int> vbInstances;
		hkgAssetConverter::findAllMappings( graphicsScene.m_vertexBuffers, vIn, vbInstances );
		if (vbInstances.getSize() >=0)
		{
			hkgVertexSet* graphicsBuffer = HK_NULL;
			for (int vi=0; vi < vbInstances.getSize(); ++vi)
			{
				int idx = vbInstances[vi];
				hkgReferencedObject* hkgObj = graphicsScene.m_vertexBuffers[ idx ].m_hkgObject;
				hkgVertexSet* vTest = static_cast<hkgVertexSet*> ( hkgObj );
				if ( _usesVB( dObj, vTest) )
				{
					graphicsBuffer = vTest;
					break;
				}
			}

			if (!graphicsBuffer) 
				continue;

			graphicsBuffer->partialLock(HKG_LOCK_WRITEONLY, lockFormat);

			// Do all the setup up to this point, but if we have graphics disabled, don't do any real work.
			if (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_NULL)
			{
				#if (HK_CONFIG_SIMD == HK_CONFIG_SIMD_DISABLED) && !defined(HK_PLATFORM_WIIU)
					if( AnimationUtils::usingInt8BoneIndices(vIn) )
					{
						hkaSkinBinding binding;
						AnimationUtils::setupSkinBinding(&vdi, vIn, graphicsBuffer, &binding);
						hkaFPUSkinningDeformer::deform(worldCompositeMatrices, numWorldCompositeMatrices, binding);
					}
					else
					{
						hkaSkinLargeBinding binding;
						AnimationUtils::setupSkinBinding(&vdi, vIn, graphicsBuffer, &binding);
						hkaFPUSkinningDeformer::deform(worldCompositeMatrices, numWorldCompositeMatrices, binding);
					}
				#else
					if( AnimationUtils::usingInt8BoneIndices(vIn) )
					{
						hkaSkinBinding binding;
						AnimationUtils::setupSkinBinding(&vdi, vIn, graphicsBuffer, &binding);
						hkaSimdSkinningDeformer::deform(worldCompositeMatrices, numWorldCompositeMatrices, binding);
					}
					else
					{
						hkaSkinLargeBinding binding;
						AnimationUtils::setupSkinBinding(&vdi, vIn, graphicsBuffer, &binding);
						hkaSimdSkinningDeformer::deform(worldCompositeMatrices, numWorldCompositeMatrices, binding);
					}
				#endif
			}
			
			graphicsBuffer->unlock();
		}
	}

	graphicsScene.m_context->unlock();
}

void HK_CALL AnimationUtils::drawAttachments( const hkDemoEnvironment* env, const hkQsTransform* poseInWorld, const hkArray< hkRefPtr< hkaBoneAttachment > >& boneAttachments, const hkArray< class hkgDisplayObject* >& attachmentObjects )
// Move the attachments
{
	HK_ASSERT2( 0x1f4bd39c, boneAttachments.getSize() == attachmentObjects.getSize(), "Size mismatch" );


	for ( int a = 0; a < boneAttachments.getSize(); a++ )
	{
		if ( attachmentObjects[ a ] != HK_NULL )
		{
			const hkaBoneAttachment* ba = boneAttachments[ a ];
			hkMatrix4 worldFromBone; worldFromBone.set( poseInWorld[ ba->m_boneIndex ] );
			hkMatrix4 worldFromAttachment; worldFromAttachment.setMul( worldFromBone, ba->m_boneFromAttachment );
			env->m_sceneConverter->updateAttachment( attachmentObjects[ a ], worldFromAttachment );
		}
	}
}

void HK_CALL AnimationUtils::createGraphicsMeshesForAttachments( const hkDemoEnvironment* env, const hkArray< hkRefPtr< hkaBoneAttachment > >& boneAttachments, hkArray< class hkgDisplayObject* >& attachmentObjectsOut )
{
	attachmentObjectsOut.clear();
	attachmentObjectsOut.setSize( boneAttachments.getSize() );

	for ( int a = 0; a < boneAttachments.getSize(); ++a )
	{
		const hkaBoneAttachment* ba = boneAttachments[ a ];
		hkgDisplayObject* hkgObject = HK_NULL;

		//Check the attachment is a mesh
		if ( hkString::strCmp( ba->m_attachment.getClass()->getName(), hkxMeshClass.getName() ) == 0 )
		{
			hkgObject = env->m_sceneConverter->findFirstDisplayObjectUsingMesh( static_cast< hkxMesh* >( ba->m_attachment.val() ) );

			if ( hkgObject != HK_NULL )
			{
				hkgObject->setStatusFlags( hkgObject->getStatusFlags() | HKG_DISPLAY_OBJECT_DYNAMIC );
			}
		}

		attachmentObjectsOut[ a ] = hkgObject;
	}
}

hkaAnimationBinding* AnimationUtils::loadAnimation( hkLoader& loader, const char* assetName )
{
	hkStringBuf assetFile(assetName); hkAssetManagementUtil::getFilePath(assetFile);
	hkRootLevelContainer* container = loader.load( assetFile.cString() );
	HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
	hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
	HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
	HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
	return ac->m_bindings[0];
}

hkaDefaultAnimationControl* AnimationUtils::loadControl( hkLoader& loader, const char* assetName )
{
	hkaAnimationBinding* binding = loadAnimation( loader, assetName );
	hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl( binding );
	control->easeOut(hkReal(0));
	return control;
}

// Skin
void HK_CALL AnimationUtils::skinMesh( const hkaPose& pose, const hkArray<hkRefPtr<hkaMeshBinding> >& skinBindings, const hkTransform& transform, const hkgSceneDataConverter& sceneConverter )
{
	int boneCount = pose.getSkeleton()->m_bones.getSize();
	// Work with the pose in world
	const hkArray<hkQsTransform>& poseInWorld = pose.getSyncedPoseModelSpace();

	// Construct the composite world transform
	hkLocalArray<hkTransform> compositeWorldInverse( boneCount );
	compositeWorldInverse.setSize( boneCount );

	// Skin the meshes
	for (int i=0; i < skinBindings.getSize(); i++)
	{
		// assumes either a straight map (null map) or a single one (1 palette)
		const hkInt16* usedBones = skinBindings[i]->m_mappings.getSize() > 0 ? skinBindings[i]->m_mappings[0].m_mapping.begin() : HK_NULL;
		int numUsedBones = usedBones? skinBindings[i]->m_mappings[0].m_mapping.getSize() : boneCount;

		// Multiply through by the bind pose inverse world inverse matrices
		for (int p=0; p < numUsedBones; p++)
		{
			int boneIndex = usedBones? usedBones[p] : p;
			compositeWorldInverse[p].setMul( poseInWorld[ boneIndex ], skinBindings[i]->m_boneFromSkinMeshTransforms[ boneIndex ] );
		}

		AnimationUtils::skinMesh( *skinBindings[i]->m_mesh, transform, compositeWorldInverse.begin(), compositeWorldInverse.getSize(), sceneConverter );
	}
}

// Compute the blend params that will produce the desired velocity
void HK_CALL AnimationUtils::computeBlendParams( hkReal desiredVel, hkReal walkVel, hkReal runVel, hkReal walkDur, hkReal runDur, hkReal& blend, hkReal& walkSpeed, hkReal& runSpeed )
{
	// Analytical solution of blending aproximation
	// Solution is second root of quadratic equation 

	const hkReal runWalkRatio = runDur / walkDur;

	const hkReal wVratio = walkVel*runWalkRatio;
	const hkReal rVratio = runVel*runWalkRatio;
	const hkReal rVratio2 = rVratio*runWalkRatio;
	const hkReal dVratio = desiredVel*runWalkRatio;

	blend = (-2.0f*wVratio+walkVel+rVratio2-sqrt(walkVel*walkVel-2.0f*walkVel*rVratio2+runVel*rVratio2*runWalkRatio*runWalkRatio+4.0f*(-walkVel*dVratio+wVratio*dVratio+rVratio2*desiredVel-rVratio2*dVratio)))/(2.0f*(walkVel-wVratio-rVratio+rVratio2));

	blend = hkMath::clamp(blend, hkReal(0.f), hkReal(1.f) );
	runSpeed  = (1.0f-blend) * runWalkRatio + blend;
	walkSpeed = blend * (1.0f / runWalkRatio) + (1.0f - blend);	

}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
