/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_TEXT_DISPLAY_H
#define HK_TEXT_DISPLAY_H

#include <Common/Base/Ext/hkBaseExt.h>
#include <Graphics/Common/Font/hkgFontDefines.h>
#include <Common/Base/Types/Color/hkColor.h>
#ifndef __HAVOK_PARSER__
#include <wchar.h>
#endif

class hkgFont;
class hkgCamera;
class hkgWindow;
class hkgViewport;

/// Utility class for persistent text display.
class hkTextDisplay : public hkReferencedObject
{
	//+hk.MemoryTracker(ignore=True)
	public:

		EXT_DECLARE_REFOBJECT_ALLOCATOR();

			/// Create a text display attached to window 'window'
		hkTextDisplay( hkgWindow* window );

		virtual ~hkTextDisplay();

			/// Render the stored text onto 'window'.
		void displayText(hkgWindow* window, hkgViewport* v);
		void displayJust3DText(hkgWindow* window, hkgViewport* v);
		void displayJust2DText(hkgWindow* window, bool useCurrentViewport = false );

			/// Request 'str' to be rendered at absolute screen position x,y.
		void outputText(const char* str, hkUint32 x, hkUint32 y, hkColor::Argb color = hkColor::WHITE, int frames = 1, int highlightLine = -1, bool monospace = false) { outputText(str, (float)x, (float)y, color, frames, highlightLine, monospace ); }
		void outputText(const char* str, int x, int y, hkColor::Argb color = hkColor::WHITE, int frames = 1, int highlightLine = -1, bool monospace = false) { outputText(str, (float)x, (float)y, color, frames, highlightLine, monospace  ); }
		virtual void outputText(const char* str, float x, float y, hkColor::Argb color = hkColor::WHITE, int frames = 1, int highlightLine = -1, bool monospace = false);

			/// Request 'str' to be rendered at world position x,y,z (which will be the lower left of the text)
		virtual void outputText3D(const char* str, float x, float y, float z, hkColor::Argb color = hkColor::WHITE, int frames = 1, bool monospace = false );

			/// Clear all current text.
		virtual void wipeText();

		void clearAndDeallocate();

		const hkgFont* getFont() const { return m_font; }

			/// 2D lines visible
		int getNumVisibleLines(const hkgWindow* win) const;

		/// Set how slash t gets interpreted 
		void setTabStopSize(bool monospace, int pixels);
		int getTabStopSize(bool monospace) const;

		void setDepthTest3D( bool read, bool write );

			/// Request 'str' to be rendered at absolute screen position x,y.
			/// If the text runs over sizeX it automatically wraps to the next line.
			/// If the text runs over sizeY it is truncated. Negative sizeY disables truncation.
			/// The method also breaks word appropriately on spaces rather than in the middle.
		virtual void outputTextWithWrapping(const char* str, int x, int y, int sizeX, int sizeY, hkColor::Argb color = hkColor::WHITE, bool monospace = false);

		struct TextLayoutCallback
		{
			//+hk.MemoryTracker(ignore=True)
			virtual void text(const char* p, int len) = 0;
			virtual ~TextLayoutCallback() {}
		};
		virtual void layoutTextWithWrapping(hkgFont* f, const char* str, int sizeX, int sizeY, TextLayoutCallback& cb);

		struct OutputString
		{
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO_FRAMEWORK, OutputString);
			extStringPtr str;
			int frames; 
			int highlightLine;
			float x;
			float y;
			float c[4]; // color rgb alpha
			float tabStopSize;
			bool monospace;
		};

		struct OutputString3D
		{
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO_FRAMEWORK, OutputString3D);
			extStringPtr str;
			int frames; 
			float x;
			float y;
			float z;
			float c[4]; // color rgb alpha
			float tabStopSize;
			bool monospace;
		};

	public:
		bool m_holdTextForDebug;

	private:

		extArray<OutputString> m_strings;
		extArray<OutputString3D> m_strings3D;
		mutable class hkCriticalSection* m_stringLock;
		hkgWindow*			m_curWindow;
		hkgFont*			m_font;
		hkgFont*			m_monoSpaceFont; // may be the same ptr as the main font if it is monospace anyway
		hkgCamera*			m_fontCam; // ortho cam for the font

		bool				m_depthRead3D;
		bool				m_depthWrite3D;

};

class hkTextLog : public hkReferencedObject
{
	public:
	HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
		hkTextLog( int numLines, int top, int left );
		~hkTextLog();

		void outputText( const char* str  );

			// Call this every frame
		void displayLog( hkTextDisplay& display );

	protected:

		class hkCriticalSection* m_stringLock;
		extArray<extStringPtr> m_lines;
		int m_top;
		int m_left;
		int m_numLines;
};


#endif // HK_TEXT_DISPLAY_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
