/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#include <Demos/demos.h>

#include <Demos/DemoCommon/DemoFramework/hkDemo.h>
#include <Graphics/Common/hkGraphics.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Camera/hkgCamera.h>
#include <Graphics/Common/Font/hkgFont.h>
#include <Graphics/Bridge/System/hkgSystem.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Monitor/hkMonitorStream.h>
#include <Common/Base/Container/String/hkUtf8.h>


hkTextDisplay::~hkTextDisplay()
{
	m_font->release();
	m_monoSpaceFont->release();
	m_fontCam->release();

	delete m_stringLock;
}

void hkTextDisplay::wipeText()
{
	m_strings.clear();
	m_strings3D.clear();
}

struct hkBox2D
{
	int minX;
	int maxX;
	int minY;
	int maxY;
};


class TextDistanceMore
{
public:
	TextDistanceMore(hkgCamera* cam) : m_camera(cam) { }
	hkBool operator() ( const hkTextDisplay::OutputString3D& a, const hkTextDisplay::OutputString3D& b )
	{
		float v[3];
		hkgVec3Sub(v, &a.x, m_camera->getFromPtr() );
		const float da = hkgVec3LengthSqrd(v);
		hkgVec3Sub(v, &b.x, m_camera->getFromPtr() );
		const float db = hkgVec3LengthSqrd(v);
		return ( da < db ); // reverse order.
	}

	hkgCamera* m_camera;
};

void hkTextDisplay::setDepthTest3D( bool read, bool write )
{
	m_depthRead3D = read;
	m_depthWrite3D = write;
}


void hkTextDisplay::displayJust3DText(hkgWindow* window, hkgViewport* v)
{
	HKG_TIMER_BEGIN("3DText", this);
	m_stringLock->enter();

	// render text first:
	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();
	ctx->setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars

	v->setAsCurrent( ctx );
	hkgFont::setDrawState( ctx );

	
	hkgCamera* cam = v->getCamera();

	// 3D text first
	HK_ALIGN16(float m[16]);
	float charWP = m_font->getCharWidth();
	float charHP = m_font->getCharHeight();
	float charWM = m_monoSpaceFont->getCharWidth();
	float charHM = m_monoSpaceFont->getCharHeight();
	float ratioP = charWP / charHP; // will keep the ratio, alter the height
	float ratioM = charWM / charHM; // will keep the ratio, alter the height
	float tsP = m_font->getTabStopSize();
	float tsM = m_monoSpaceFont->getTabStopSize();

	if (m_strings3D.getSize() > 0)
	{
		// Want depth for 3D text so that it fits in scene properly
		HKG_ENABLED_STATE state = ctx->getEnabledState();
		bool hasWrite = (state & HKG_ENABLED_ZWRITE) > 0;
		if ( hasWrite != m_depthWrite3D)
			ctx->setDepthWriteState(true);

		bool hasRead = (state & HKG_ENABLED_ZREAD) > 0;
		if ( hasRead != m_depthRead3D)
			ctx->setDepthReadState(true);

		// sort the 3d text array wrt to camera
		// as 3d text freqently overlaps, it is better to
		// draw in depth order to improve visual effect
		// ( the use of the double arg template doses not compile in VC6 )
#if !defined(HK_COMPILER_MSVC_VERSION) || (HK_COMPILER_MSVC_VERSION >= 1300)		
		HKG_TIMER_BEGIN("HeapSort 3D Text", this);
		hkAlgorithm::heapSort<OutputString3D, TextDistanceMore>(m_strings3D.begin(), m_strings3D.getSize(), TextDistanceMore(cam));
		HKG_TIMER_END();
#endif	
	}

	for (int j = m_strings3D.getSize() - 1; j >= 0; --j)
	{
		OutputString3D& s = m_strings3D[j];    
		hkgFont* f = s.monospace? m_monoSpaceFont : m_font;

		ctx->pushMatrix();	
		float dist = cam->computeBillboardMatrix( &s.x, m);
		float newCharH = cam->computeIconVerticalDrawSize(dist, (int)( s.monospace? charHM : charHP), (int)(v->getHeight()) );
		ctx->multMatrix(m);
		f->setCharHeight(newCharH);
		f->setCharWidth(newCharH * (s.monospace? ratioM : ratioP) );
		f->setTabStopSize( s.tabStopSize );
		f->render( window->getContext(), s.str.cString(), 0, 0, s.c); // render font at 0,0 wrt to billboard matrix

		if (!m_holdTextForDebug)
			s.frames--;
		if (s.frames <= 0)
		{
			m_strings3D.removeAt(j); // this is why we recurse in reverse order.
		}

		ctx->popMatrix();
	}

	m_font->setCharHeight(charHP);
	m_font->setCharWidth(charWP);
	m_font->setTabStopSize(tsP);

	m_monoSpaceFont->setCharHeight(charHM);
	m_monoSpaceFont->setCharWidth(charWM);
	m_monoSpaceFont->setTabStopSize(tsM);

	m_stringLock->leave();
	ctx->unlock();
	HKG_TIMER_END();
}

void hkTextDisplay::displayJust2DText(hkgWindow* window, bool useCurrentViewport)
{
	HKG_TIMER_BEGIN("2DText", this);
		
	m_stringLock->enter();

	// render text first:
	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();
	ctx->setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars

	float tsP = m_font->getTabStopSize();
	float tsM = m_monoSpaceFont->getTabStopSize();

	// 2D text goes to full screen
	hkgViewport* v = useCurrentViewport? ctx->getCurrentViewport() : window->getWindowOrthoView();
	if (!useCurrentViewport)
	{
		v->setAsCurrent( ctx );
	}
	hkgFont::setDrawState( ctx );



	// Don't want depth for 2D text or its borders.
	HKG_ENABLED_STATE state = ctx->getEnabledState();
	if (state & HKG_ENABLED_ZWRITE)
		ctx->setDepthWriteState(false);
	if (state & HKG_ENABLED_ZREAD)
		ctx->setDepthReadState(false);

	ctx->setLightingState(false);

	//
	// 2D text
	//
	m_fontCam->computeOrtho( 0, float(v->getWidth()), 0, float(v->getHeight()), 0, 1);
	m_fontCam->setAsCurrent( window->getContext() );
	ctx->setCurrentSoleTexture(HK_NULL, HKG_TEXTURE_MODULATE); // for 360, incase we have to modify the font texture for extra chars

	HKG_TIMER_BEGIN("BoundingBoxes", this)

	hkLocalArray<hkBox2D> boxes(m_strings.getSize());
	hkLocalArray<hkBox2D> highlights(10);
	boxes.setSize(m_strings.getSize());

	{
		// Find the extents of the text
		for (int s=0 ; s < m_strings.getSize(); s++)
		{
			float maxWidthP = 0;
			float widthP = 0;	
			int height = 1; // max height
			for(hkUtf8::Iterator iter(m_strings[s].str); iter.advance(); /**/)
			{
				hkUtf8::CodePoint cp = iter.current();
				// Handle \n
				if( cp == '\n' )
				{
					height++;
					widthP = 0;
				}
				else if( cp == '\t' )
				{
					int tab = (int)m_strings[s].tabStopSize;
					int cwidth = (int)( m_strings[s].monospace? m_monoSpaceFont->getCharWidth() : m_font->getCharWidth() );
					int w = hkMath::hkFloatToInt(widthP * cwidth);
					widthP = float( ((w + tab) / tab) * tab) / cwidth; // next whole multiple of tab pixels, in units of font relative widths
				}
				else
				{
					float w = 0;
					if (m_strings[s].monospace)
						m_monoSpaceFont->preResolve( cp, w );
					else
						m_font->preResolve( cp, w );

					widthP += w;
				}
				maxWidthP = maxWidthP < widthP? widthP: maxWidthP;
			}

			// highlights?
			int cwidth = (int)( m_strings[s].monospace? m_monoSpaceFont->getCharWidth() : m_font->getCharWidth() );
			int cheight = (int)( m_strings[s].monospace? m_monoSpaceFont->getCharHeight() : m_font->getCharHeight() );

			if (m_strings[s].highlightLine >= 0 )
			{
				hkBox2D& b = highlights.expandOne();
			
				b.minX = (int)(m_strings[s].x);
				b.maxX = (int)((m_strings[s].x) + (maxWidthP * cwidth) + 5);
				b.minY = (int)(m_strings[s].y) + (m_strings[s].highlightLine * cheight);
				b.maxY = b.minY + cheight ;
			}

			// main background
			boxes[s].minX = (int)(m_strings[s].x);
			boxes[s].maxX = (int)((m_strings[s].x) + maxWidthP * cwidth + 5);
			boxes[s].minY = (int)(m_strings[s].y);
			boxes[s].maxY = (int)((m_strings[s].y) + height * cheight);
		}
	}


	// Merge boxes
	{
		for (int k=0; k < boxes.getSize(); k++)
		{
			for (int j=0; j < boxes.getSize() && k < boxes.getSize();)
			{
				if (j!=k)
				{
					const int tolerance = 12;
					//Check for overlap
					if ((boxes[k].minX - tolerance < boxes[j].maxX) && (boxes[k].maxX + tolerance > boxes[j].minX) &&
						(boxes[k].minY - tolerance < boxes[j].maxY) && (boxes[k].maxY + tolerance > boxes[j].minY))
					{
						//Merge j into k
						boxes[k].minX = hkMath::min2(boxes[k].minX, boxes[j].minX);
						boxes[k].minY = hkMath::min2(boxes[k].minY, boxes[j].minY);
						boxes[k].maxX = hkMath::max2(boxes[k].maxX, boxes[j].maxX);
						boxes[k].maxY = hkMath::max2(boxes[k].maxY, boxes[j].maxY);

						//Erase j
						boxes[j] = boxes[boxes.getSize()-1];
						boxes.popBack();
						k=0;
					}
					else
					{
						j++;
					}
				}
				else
				{
					j++;
				}

			}
		}
	}

	window->getContext()->setTexture2DState( false );
	
	float highlightcolor[4] = { 0.75f, 0.75f, 0.75f, 0.5f};
	float white[4] = { 0.5f, 0.5f, 0.5f, 0.5f};
	float white2[4] = { 0.2f, 0.2f, 0.5f, 0.5f};
	window->getContext()->setCurrentColor4( white );

	// Draw an alpha background overlay 
	for (int i=0; i < boxes.getSize(); i++)
	{
		float p[3];	p[2] = -0.01f; //depth (0..-2)
		float tl[2]; float lr[2];
		const int border = 5;
		tl[0] = (float)boxes[i].minX - border; 
		tl[1] = (float)v->getHeight() - boxes[i].minY + border; 
		lr[0] = (float)boxes[i].maxX + border; 
		lr[1] = (float)v->getHeight() - boxes[i].maxY - border;

		window->getContext()->setCurrentColor4( white );
		window->getContext()->beginGroup( HKG_IMM_TRIANGLE_LIST );
		p[0] = tl[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1];
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentColor4( white2 );
		window->getContext()->setCurrentPosition( p );
		window->getContext()->endGroup();

		float black[4] = { 0.0f, 0.0f, 0.0f, 1.0f};
		window->getContext()->setCurrentColor4( black );
		window->getContext()->beginGroup( HKG_IMM_LINES );
		p[0] = tl[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1];
		window->getContext()->setCurrentPosition( p );
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = tl[1];
		window->getContext()->setCurrentPosition( p );
		window->getContext()->endGroup();
	}
	
	// highlights
	for (int i=0; i < highlights.getSize(); i++)
	{
		float p[3];	p[2] = -0.005f; //depth (0..-2)
		float tl[2]; float lr[2];
		const int border = 2;
		tl[0] = (float)highlights[i].minX - border; 
		tl[1] = (float)v->getHeight() - highlights[i].minY + border; 
		lr[0] = (float)highlights[i].maxX + border; 
		lr[1] = (float)v->getHeight() - highlights[i].maxY - border;

		window->getContext()->setCurrentColor4( highlightcolor );
		window->getContext()->beginGroup( HKG_IMM_TRIANGLE_LIST );
		p[0] = tl[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = tl[1]; 
		window->getContext()->setCurrentPosition( p );
		p[0] = tl[0]; p[1] = lr[1];
		window->getContext()->setCurrentPosition( p );
		p[0] = lr[0]; p[1] = lr[1]; 
		window->getContext()->setCurrentColor4( white2 );
		window->getContext()->setCurrentPosition( p );
		window->getContext()->endGroup();
	}

	HKG_TIMER_END();

	window->getContext()->setTexture2DState( true );

	int offset = int(v->getHeight()) - int(m_font->getCharHeight());

	for (int k = m_strings.getSize() - 1; k >= 0; --k)
	{
		OutputString& s = m_strings[k];
		hkgFont* f = s.monospace? m_monoSpaceFont : m_font;

		f->setTabStopSize( s.tabStopSize ); // per string tab stop settings (so stats can differ from default 4 char etc)
		f->render( window->getContext(), s.str.cString(), s.x, offset - s.y, s.c);

		if (!m_holdTextForDebug)
			s.frames--;
		if (m_strings[k].frames <= 0)
		{
			m_strings.removeAt(k);
		}
	}

	m_font->setTabStopSize(tsP);
	m_monoSpaceFont->setTabStopSize(tsM);

	m_stringLock->leave();
	ctx->unlock();

	HKG_TIMER_END();
}

void hkTextDisplay::displayText(hkgWindow* window, hkgViewport* v0)
{
	HKG_TIMER_BEGIN("Text", this);
	m_stringLock->enter();
		displayJust3DText( window, v0 );
		displayJust2DText( window );
	m_stringLock->leave();
	HKG_TIMER_END();
}

struct AccumulateTextCallback : public hkTextDisplay::TextLayoutCallback
{
	virtual void text( const char* p, int n )
	{
		m_out.append(p,n);
	}
	extArray<char> m_out;
};

void hkTextDisplay::outputTextWithWrapping(const char* str, int x, int y, int sizeX, int sizeY, hkColor::Argb color, bool monospace)
{
	AccumulateTextCallback cb;
	layoutTextWithWrapping(monospace? m_monoSpaceFont : m_font, str, sizeX, sizeY, cb);
	cb.m_out.pushBack(0);
	outputText(cb.m_out.begin(), (float)x, (float)y, color, monospace);
}

void hkTextDisplay::layoutTextWithWrapping(hkgFont* f, const char* str, int sizeX, int sizeY, TextLayoutCallback& cb )
{
	int charWidth = int( f->getCharWidth() );
	int charHeight = int( f->getCharHeight() );
	int linesLeft = sizeY > 0 ? hkMath::max2( 1, sizeY /  charHeight ) : -1;
	int charsPerLine = sizeX / charWidth;
	if( charsPerLine < 2 || str == HK_NULL )
	{
		return; // early out if no room
	}

	const char* cur = str;
	while( *cur && linesLeft )
	{
		float pixelsUsed = 0; // horizontal pixels used on this line
		int lastWhitespace = 0; // byteIndex of last whitespace
		int bytesThisLine = 0;

		// find out how many characters we can consume
		hkUtf8::Iterator iter(cur);
		int bytesThisChar;
		while( true )
		{
			if( iter.advance(&bytesThisChar) == false ) // no more input
			{
				if( bytesThisLine )
				{
					cb.text(cur, bytesThisLine);
				}
				cur += bytesThisLine;
				break;					//EARLY OUT exit loop for newline
			}

			hkUtf8::CodePoint cp = iter.current();

			// We'd like to wrap at whitespace if possible.
			// Remember in case we need to wrap.
			if( (cp == ' ') || (cp == '\t') )
			{
				lastWhitespace = bytesThisLine;
			}

			bytesThisLine += bytesThisChar;

			if( cp == '\n' )
			{
				cb.text(cur, bytesThisLine);
				cur += bytesThisLine;
				break;					//EARLY OUT exit loop for newline
			}

			if( cp == '\t' )
			{
				int pi = hkMath::hkFloatToInt(pixelsUsed);
				int tab = (int)f->getTabStopSize();
				pixelsUsed = float( ((pi + tab) / tab) * tab);
			}
			else
			{
				float adv = 0;
				m_font->preResolve( cp, adv );
				pixelsUsed += adv * charWidth;
			}

			// We ran out of space. We will need to split the line
			if( pixelsUsed > sizeX )
			{ 
				if( lastWhitespace > 0 ) // there was whitespace, wrap there
				{
					bytesThisLine = lastWhitespace;
					cb.text( cur, bytesThisLine);
					cb.text( "\n", 1 );

					// consume any whitespace immediately after the break
					hkUtf8::Iterator ws(cur + bytesThisLine);
					while( ws.advance(&bytesThisChar) )
					{
						if( ws.current()==' ' || ws.current()=='\t')
						{
							bytesThisLine += bytesThisChar;
						}
						else
						{
							break;
						}
					}
					cur += bytesThisLine;
				}
				else // no whitespace, we must hyphenate
				{
					bytesThisLine -= bytesThisChar; // rewind one char
					cb.text( cur, bytesThisLine );
					cb.text( "-\n", 2 );
					cur += bytesThisLine;
				}
				break;
			}
		}

		linesLeft -= 1;
	}

	// If we exited because we ran out of lines, show that text is truncated
	if( *cur != 0 ) 
	{
		cb.text("<...>", 5);
	}
}

void hkTextDisplay::outputText(const char* str, float x, float y, hkColor::Argb color, int frames, int highlightLine, bool monospace )
{
	int slen = hkUtf8::strLen(str);
	if( slen > 0 )
	{
		m_stringLock->enter();

		OutputString& s = m_strings.expandOne();

		s.frames = frames;
		s.highlightLine = highlightLine;
		// Move text in a bit on consoles (TV deadzone, normally we give about 20 pixels)
		s.x = x + m_curWindow->getTVDeadZoneH();
		s.y = y; 
		s.tabStopSize = monospace? m_monoSpaceFont->getTabStopSize() : m_font->getTabStopSize();
		s.monospace = monospace;
		s.str = str;
		
		s.c[0] = (float)hkColor::getRedAsFloat( color );
		s.c[1] = (float)hkColor::getGreenAsFloat( color );
		s.c[2] = (float)hkColor::getBlueAsFloat( color );
		s.c[3] = (float)hkColor::getAlphaAsFloat( color );
		if (s.c[3] < 0.05f)
			s.c[3] = 1.0f; // forgot the alpha in the color def.

		m_stringLock->leave();
	}
}

void hkTextDisplay::outputText3D(const char* str, float x, float y, float z, hkColor::Argb color, int frames, bool monospace )
{
	int slen = hkUtf8::strLen(str);
	if( slen > 0 )
	{
		m_stringLock->enter();

		OutputString3D& s = m_strings3D.expandOne();

		s.frames = frames;
		s.x = x;
		s.y = y; 
		s.z = z; 
		s.tabStopSize = monospace? m_monoSpaceFont->getTabStopSize() : m_font->getTabStopSize();
		s.monospace = monospace;
		s.str = str;

		s.c[0] = (float)hkColor::getRedAsFloat( color );
		s.c[1] = (float)hkColor::getGreenAsFloat( color );
		s.c[2] = (float)hkColor::getBlueAsFloat( color );
		s.c[3] = (float)hkColor::getAlphaAsFloat( color );
		if (s.c[3] < 0.05f)
			s.c[3] = 1.0f; // forgot the alpha in the color def.

		m_stringLock->leave();
	}
}


int hkTextDisplay::getNumVisibleLines(const hkgWindow* window) const
{
	int wh = window->getHeight();
	int ch = (int)(getFont()->getCharHeight());
	return wh / ch;	
}

void hkTextDisplay::setTabStopSize(bool monospace, int pixels)
{
	if (monospace && m_monoSpaceFont)
	{
		m_monoSpaceFont->setTabStopSize((float)pixels);
	}
	else if (!monospace && m_font)
	{
		m_font->setTabStopSize((float)pixels);
	}
}

int hkTextDisplay::getTabStopSize(bool monospace) const
{
	if (monospace && m_monoSpaceFont)
	{
		return (int)m_monoSpaceFont->getTabStopSize();
	}
	else if (!monospace && m_font)
	{
		return (int)m_font->getTabStopSize();
	}
	return 0;
}

hkTextDisplay::hkTextDisplay(hkgWindow* window)
{
	m_holdTextForDebug = false;
	m_stringLock = new hkCriticalSection(1000);

	m_depthRead3D = true;
	m_depthWrite3D = true;

	m_curWindow = window;

	m_font = hkgFont::create();
	
	if (hkgSystem::g_RendererType > hkgSystem::HKG_RENDERER_NULL)
	{
		//XX add winrt 
	#if !defined(HK_PLATFORM_RVL) && !defined(HK_PLATFORM_PSVITA) && !defined(HK_PLATFORM_ANDROID) && !defined(HK_PLATFORM_WIIU)  && !defined(HK_PLATFORM_WINRT) && !defined(HK_PLATFORM_DURANGO)// enable Unicode TrueType fonts for other platforms. Works on NGP, just default fread stream is very slow to load at the moment
		// System Fonts are usually not redistributable.
		// Most free Unicode fonts do not have all languages, so pick appropriate one here:

		bool needMono = true;

		char path[MAX_PATH];
		GetWindowsDirectoryA(path, MAX_PATH);
		hkStringBuf pbuf( path, "\\Fonts\\ARIALUNI.TTF");
		if ( !m_font->loadFontFromFile( window->getContext(), pbuf ))
		{
			needMono = false; // sazanami-gothic is mono space, as is the default
			
			// Japanese + English:
			if (!m_font->loadFontFromFile( window->getContext(), "./Resources/Common/Fonts/Sazanami/sazanami-gothic.ttf" )) 	
			{
				// Simple English only ASCII texture, not from TTF file
				m_font->loadFromBuiltin( window->getContext(), false );
			}
		}

		if (needMono)
		{
			char path[MAX_PATH];
			GetWindowsDirectoryA(path, MAX_PATH);
			hkStringBuf pbuf( path, "\\Fonts\\courbd.ttf"); // courier new, bold
			//hkStringBuf pbuf( path, "\\Fonts\\lucon.ttf"); // lucida console
			m_monoSpaceFont = hkgFont::create();
			if (!m_monoSpaceFont->loadFontFromFile( window->getContext(), pbuf ))
			{
				m_monoSpaceFont->removeReference();
				needMono = false;
			}
		}


	#else // no Unicode etc
		
		bool needMono = false;
		m_font->loadFromBuiltin( window->getContext(), false );

	#endif

		if (!needMono)
		{
			m_monoSpaceFont = m_font;
			m_font->addReference();
		}
	}
	else // No font at all 
	{
		m_monoSpaceFont = m_font;
		m_font->addReference();
	}

	m_fontCam = hkgCamera::create();
	m_fontCam->computeModelView(true);
}

void hkTextDisplay::clearAndDeallocate()
{
	m_strings.clearAndDeallocate();
	m_strings3D.clearAndDeallocate();
}

hkTextLog::hkTextLog( int numLines, int top, int left )
{
	m_stringLock = new hkCriticalSection(1000);
	m_top = top;
	m_left = left;
	m_numLines = numLines;
}
hkTextLog::~hkTextLog()
{
	delete m_stringLock;
}


void hkTextLog::outputText( const char* str  )
{
	m_stringLock->enter();
	if (m_lines.getSize() == m_numLines )
	{
		m_lines.removeAtAndCopy( 0 );
	}
	m_lines.pushBack( str );
	m_stringLock->leave();
}

void hkTextLog::displayLog( hkTextDisplay& display )
{
	m_stringLock->enter();
	for (int i = 0; i < m_lines.getSize(); ++i )
	{
		display.outputText(m_lines[i].cString(), m_left, m_top + 20 * i );
	}
	m_stringLock->leave();
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
