/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/DemoFramework/hkTestDemo.h>
#include <Demos/DemoCommon/DemoFramework/hkDemoConsole.h>
#include <Demos/DemoCommon/DemoFramework/hkDemoFramework.h>
#include <Demos/DemoCommon/DemoFramework/hkFloatingPointExceptionCheck.h>

#include <Graphics/Common/Input/Pad/hkgPad.h>
#include <Graphics/Common/Window/hkgWindow.h>

#include <Common/Base/Memory/System/Debug/hkDebugMemorySystem.h>
#include <Common/Base/System/Io/OStream/hkOStream.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/System/Error/hkDefaultError.h>
#include <Common/Base/UnitTest/hkUnitTest.h>


#include <Common/Base/Fwd/hkstandardheader.h>
#include HK_STANDARD_HEADER(stdlib)
#include HK_STANDARD_HEADER(string)
#include HK_STANDARD_HEADER(setjmp)

extern const char* LASTTEST_FILENAME;

#if defined HK_COMPILER_MSVC
	// C4611 interaction between '_setjmp' and C++ object destruction is non-portable
	// yes we know.
#	pragma warning(disable: 4611)
#endif

using namespace std;

/////////////////////////////

class TestDemo;
static jmp_buf s_jumpBuffer;
static TestDemo* s_testGame;
static extStringBuf* s_errorText = HK_NULL;

const char * SUMMARY_FILE_NAME = "XboxUnitTestSummary.txt";

static const char* nthLine(const char* s, int n)
{
	const char* p = s;
	while(*p && n)
	{
		n -= *p++ == '\n';
	}
	return p;
}


hkBool HK_CALL hkTestReport(hkBool32 cond, const char* desc, const char* file, int line)
{
	if(!cond)
	{

		s_testGame->failCurrentTest(TestDemo::FAIL_NORMAL);

		//.\Utilities\Menu\MenuDemo.cpp(28) : fatal error C1189: #error :  test
		hkStringBuf err;
		err.printf("%s (%i) : error : %s\n", file, line, desc);
		if (s_errorText)
		{
			*s_errorText += err;
		}
		else
		{
			OutputDebugStringA(err);
			OutputDebugStringA("\n");
			hkprintf("%s", err.cString() );
		}
	}
	return bool(cond);
}

void hkSetLastTest(const char* testName)
{
	hkOfstream out(LASTTEST_FILENAME);
	if(out.isOk())
	{
		out << testName;
	}
}

extern HK_EXPORT_COMMON hkTestReportFunctionType hkTestReportFunction;

TestDemo::TestDemo(hkDemoEnvironment* env, hkBool autoRecoverFromErrors, hkTestEntry* overrideTest)
	:	hkDefaultDemo(env),
		m_autoRecoverFromErrors(autoRecoverFromErrors),
		m_firstLineShown(0),
		m_numPasses(0),
		m_numFails(0)
{
	// Init test function
	m_origTestReportFunction = hkTestReportFunction;
	hkTestReportFunction = &hkTestReport;

	m_error.m_owner = this;

	const hkStringBuf envPath(env->m_menuPath);
	const bool runAllTests = envPath.endsWith("All Tests");

#ifndef HK_DYNAMIC_DLL
	for( int ti = 0; hkUnitTestDatabase[ti] != HK_NULL; ti += 1 )
	{
		const hkTestEntry* test = hkUnitTestDatabase[ti];

		// match the path normalization that happens in registerUnitTests().
		hkStringBuf testPath(test->m_path);
		testPath.pathNormalize();

		if (runAllTests || testPath.endsWith(envPath) || envPath.endsWith(test->m_category) )
		{
			m_tests.pushBack(test);
		}
	}
#endif
	m_currentTest = 0;

	if(overrideTest)
	{
		m_tests.pushBack(overrideTest);
	}

	s_errorText = &m_error.m_errorBuffer;

#if (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	// The demo sets these up and we can't have multiple instances of the hkSpuUtil
	// So just expose the demo's copies to the unit tests
	hkUnitTest::s_jobQueue = m_jobQueue;
	hkUnitTest::s_taskQueue = m_taskQueue;
	hkUnitTest::s_threadPool = m_threadPool;
#endif

	m_keepAliveTimer.start();
}

TestDemo::~TestDemo()
{
	// Clear test function
	hkTestReportFunction = m_origTestReportFunction;

#if (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	hkUnitTest::s_jobQueue = HK_NULL;
	hkUnitTest::s_taskQueue = HK_NULL;
	hkUnitTest::s_threadPool = HK_NULL;
#endif

	hkcout.printf("[%i passes, %i fails] ('F'ailure, 'W'arning, 'A'ssert, 'E'rror)\n", m_numPasses, m_numFails);
	s_errorText = HK_NULL;
	m_env->m_exitCode =  m_numFails > 0;
}

hkDemo::Result TestDemo::stepDemo()
{
	// Window dressing
	{
		int windowLines = m_env->m_textDisplay->getNumVisibleLines(m_env->m_window);
		const char* p = m_passString.cString();
		int textLines = 0;
		while(*p)
		{
			textLines += *p++ == '\n';
		}

		{
			m_env->m_textDisplay->outputText( nthLine(m_testString, m_firstLineShown), 70, 40);
			m_env->m_textDisplay->outputText( nthLine(m_failString, m_firstLineShown), 40, 40, 0xff0000);
			m_env->m_textDisplay->outputText( nthLine(m_passString, m_firstLineShown), 40, 40, 0x00ff00);
		}

		if( textLines + 6 > windowLines )
		{
			if(m_firstLineShown > 0)
			{
				m_env->m_textDisplay->outputText("^\n^", 10, 40);
				if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_UP) )
				{

					{
						int amount = 1;
						m_firstLineShown = hkMath::max2(0, m_firstLineShown-amount);
					}
				}
			}

			if( textLines + 6 - m_firstLineShown > windowLines )
			{
				m_env->m_textDisplay->outputText("v\nv", 10, (int)getWindowHeight() - 40);

				if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_DOWN) )
				{
					{
						int amount = 1;
						m_firstLineShown = hkMath::min2(textLines+6-windowLines, m_firstLineShown+amount);
					}
				}
			}
		}
		{
			hkStringBuf s;
			s.printf("[%i passes, %i fails] ('F'ailure, 'W'arning, 'A'ssert, 'E'rror)", m_numPasses, m_numFails);
			m_env->m_textDisplay->outputText(s, 10, 10);
		}
	}

	// Run tests
	hkStopwatch watch; watch.start();
	while (m_currentTest < m_tests.getSize())
	{
		const hkTestEntry* currentTest = m_tests[m_currentTest];

		hkError* savedError = HK_NULL;
		if( m_autoRecoverFromErrors )
		{
			// change error handler to attempt to continue
			savedError = &hkError::getInstance();
			savedError->addReference();
			hkError::replaceInstance( &m_error );
		}

		m_currentTestOk = true;
		s_testGame = this;

		testRun(currentTest);
		//watch.stop();	hkprintf( "%f, %s\n", watch.getElapsedSeconds(), currentTest->m_name );

		// restore default error handler
		if( savedError )
		{
			m_error.addReference();
			hkError::replaceInstance(savedError);
		}

		if( m_currentTestOk )
		{
			m_numPasses++;
			m_passString += "O";
			m_passedTests += (currentTest -> m_name);
			m_passedTests += "\n";
		}
		else
		{
			m_numFails++;
			m_failedTests += (currentTest -> m_name);
			m_failedTests += "\n";
		}
		m_passString += "\n";
		m_failString += "\n";

		// Next test and immediately run the test until we willed 100msecs
		m_currentTest++;
		watch.stop();
		watch.resume();
		if (watch.getElapsedSeconds() > 0.1f )
		{
			break;
		}
	}



	// Finished
	if (m_currentTest == m_tests.getSize())
	{
		// this out put needs to match "[HKTEST FINISH]\n" or the python script will not
		// be able to find the finish tag
		hkprintf( "\n[HKTEST FINISH]\n" );
		hkprintf("[%i passes, %i fails]\n", m_numPasses, m_numFails);
		hkSetLastTest("[DONE]");
		//In order to provide info for continious integration builds
		//Write info to file when unit testing on Xbox 360
		HK_REGRESSION_REPORT( "Unit Tests Failures", "%", 100.0f - 100.0f * hkReal( m_numPasses ) / hkReal ( m_numPasses + m_numFails ) );
		if (m_env->m_options->m_renderer
			&& m_env->m_options->m_renderer[0] == 'n')
		{
			return DEMO_STOP;
		}
		m_currentTest ++;
	}
	else if (m_currentTest < m_tests.getSize() && m_keepAliveTimer.getElapsedSeconds() > 15.0f)
	{
		// Occasionally print some output, to keep console connections alive when testing.
		hkprintf(".");
		m_keepAliveTimer.reset();
		m_keepAliveTimer.start();
	}

	return DEMO_OK;
}

static hkReal unitTestTimeThresholdInSeconds
#if HK_PLATFORM_IS_CONSOLE == 1
	= 5.0f;
#else
	= 1.0f;
#endif

static void getTestLogPath(const hkTestEntry* entry, extStringBuf& pathOut )
{
	pathOut.append(entry->m_name);
	pathOut.prepend( "logs/" );
	pathOut.append( ".log" );
}

void TestDemo::testRun(const hkTestEntry* entry)
{
	hkFPUExceptionDisabler disableFpuExceptions;
	const char* name = entry->m_name;
	m_testString += entry->m_name;
	m_testString += "\n";

	m_error.m_errorBuffer = "";
	extStringBuf pathOut;
	getTestLogPath(entry, pathOut);
	hkStopwatch timer;

	hkSetLastTest(entry->m_name);

	if( setjmp(s_jumpBuffer) == 0 )
	{
		hkprintf("Running %s unit test...", name);
		if (hkDebugMemorySystem * dms = hkMemorySystem::getInstance().getDebugInterface())
		{
			extStringBuf testBookmark("START ", entry->m_path );
			dms->addBookmark(testBookmark.cString());
		}

		timer.start();
		(*entry->m_func)();
		timer.stop();

		if (hkDebugMemorySystem * dms = hkMemorySystem::getInstance().getDebugInterface())
		{
			extStringBuf testBookmark("END ", entry->m_path );
			dms->addBookmark(testBookmark.cString());
		}
		hkprintf(" finished");
		if (timer.getElapsedSeconds() > unitTestTimeThresholdInSeconds)
		{
			hkprintf(" (took %f seconds).\n", timer.getElapsedSeconds() );
		}
		else
		{
			hkprintf(".\n");
		}
		
	}
	else
	{
		//entry-
		hkStringBuf path( entry->m_path );
		path.replace( "Test/UnitTest", ".\\" );
		path.replace( "/", "\\" );
		m_error.m_errorBuffer.appendPrintf("%s (%i) : '%s' was aborted\n", __FILE__, __LINE__, name);
	}

	if (m_error.m_errorBuffer.getLength()!=0)
	{
		hkprintf("\n[HKTEST BEGIN %s]\n", entry->m_name);
		hkprintf("%s", m_error.m_errorBuffer.cString());
		OutputDebugStringA(m_error.m_errorBuffer.cString());
		OutputDebugStringA("\n");
		hkprintf("[HKTEST END]\n");
	}

	if(m_env->m_options->m_saveOutputToLog)
	{
		// write unit test output to logfile using test name
		{
			hkOstream stream(pathOut);
			if (stream.isOk())
			{
				stream << m_error.m_errorBuffer.cString();
			}
		}
		addHeaderToTestLogfile(pathOut.cString(), entry->m_name, m_currentTestOk, timer.getElapsedSeconds());
	}
	hkcout.flush();
}


void TestDemo::failCurrentTest(FailReason f)
{
	if(m_currentTestOk)
	{
		m_currentTestOk = false;
		const char* fails[FAIL_MAX] = { "F", "A", "E" };
		m_failString += fails[f];
	}
}

static void errorReport(const char* msg, void* errorReportObject)
{
	hkprintf( "%s", msg );
}

TestDemoError::TestDemoError() : hkDefaultError( ( hkErrorReportFunction )( errorReport ) ), m_owner(HK_NULL)
{
}

int TestDemoError::message(Message msg, int id, const char* description, const char* file, int line)
{
	if( isEnabled(id) )
	{
		switch (msg)
		{
			case hkDefaultError::MESSAGE_REPORT:
				if (id > 0)
				{
					m_errorBuffer.appendPrintf("%s (%i) : Report 0x%x: %s\n", file, line, id, description);
				}
				break;
			case hkDefaultError::MESSAGE_ASSERT:
			{
				m_owner->failCurrentTest(TestDemo::FAIL_ASSERT);
				hkStringBuf mg( description );
				mg.replace( "\n", "; " );
				m_errorBuffer.appendPrintf("%s (%i) : assert error 0x%x: %s\n", file, line, id, mg.cString());
				longjmp(s_jumpBuffer, 1);
				break;
			}
			case hkDefaultError::MESSAGE_ERROR:
			{
				m_owner->failCurrentTest(TestDemo::FAIL_ERROR);

				hkStringBuf mg( description );
				mg.replace( "\n", "; " );
				m_errorBuffer.appendPrintf("%s (%i) : fatal error 0x%x: %s\n", file, line, id, mg.cString());
				longjmp(s_jumpBuffer, 1);
				break;
			}
			default:
			{
					// suspend warnings fired from:
					// - hkRigidBody::checkPerformance
				if ( id == 0x2ff8c16f || id == 0x34df5494 )
				{
					break;
				}

				m_owner->m_passString += "W";
				hkStringBuf mg( description );
				mg.replace( "\n", "; " );
				m_errorBuffer.appendPrintf("%s (%i) : warning 0x%x: %s\n", file, line, id, mg.cString());

				break;
			}
		}
	}
	return 0;
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
