/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#include <Demos/demos.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Camera/hkgCamera.h>
#include <Graphics/Common/Texture/SkyBox/hkgSkyBox.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>
#include <Graphics/Common/Material/hkgMaterial.h>
#include <Graphics/Common/Movie/hkgMovieRecorder.h>

#include <Graphics/Bridge/System/hkgSystem.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

#include <Demos/DemoCommon/DemoFramework/hkPseudoPad.h>

#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Visualize/hkVirtualFramebufferServer.h>


class NullDisplayHandler : public hkgDisplayHandler
{
public:
	HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO_FRAMEWORK );
	NullDisplayHandler(hkgDisplayWorld* displayWorld, hkgDisplayContext* displayContext, class hkgWindow* window)
	:	hkgDisplayHandler(displayWorld, displayContext, window)
	{}

	//
	// "null" hkDebugDisplayHandler interface
	// unfortunately we can't derive directly from the base class since demo code assumes an hkgDisplayHandler
	//
	virtual hkResult addGeometry(hkDisplayGeometry* geometry, hkUlong id, int tag, hkUlong shapeIdHint) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult addGeometry(const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkUlong id, int tag, hkUlong shapeIdHint, hkGeometry::GeometryType createDyanamicGeometry = hkGeometry::GEOMETRY_STATIC) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult addGeometryInstance(hkUlong originalGeomId, const hkTransform& transform, hkUlong id, int tag, hkUlong shapeIdHint) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult setGeometryPickable( hkBool isPickable, hkUlong id, int tag ) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult setGeometryVisibility(int geometryIndex, bool isEnabled, hkUlong id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult setGeometryColor(hkColor::Argb color, hkUlong id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult setGeometryColor(hkColor::Argb color, hkgDisplayObject* displayObject) HK_OVERRIDE { return hkgDisplayHandler::setGeometryColor(color, displayObject); }
	virtual hkResult setGeometryTransparency(float alpha, hkUlong id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult updateGeometry(const hkTransform& transform, hkUlong id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult updateGeometry( const hkMatrix4& transform, hkUlong id, int tag ) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult skinGeometry(hkUlong* ids, int numIds, const hkMatrix4* poseModel, int numPoseModel, const hkMatrix4& worldFromModel, int tag ) HK_OVERRIDE { return HK_SUCCESS; }
	virtual void addTextureSearchPath(const char* path) HK_OVERRIDE { }
	virtual void clearTextureSearchPaths() HK_OVERRIDE { }
	virtual hkResult removeGeometry(hkUlong id, int tag, hkUlong shapeIdHint) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult updateCamera(const hkVector4& from, const hkVector4& to, const hkVector4& up, hkReal nearPlane, hkReal farPlane, hkReal fov, const char* name) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayPoint(const hkVector4& position, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayLine(const hkVector4& start, const hkVector4& end, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayTriangle(const hkVector4& a, const hkVector4& b, const hkVector4& c, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayText(const char* text, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult display3dText(const char* text, const hkVector4& pos, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayPoint2d(const hkVector4& position, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayLine2d(const hkVector4& start, const hkVector4& end, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayTriangle2d(const hkVector4& a, const hkVector4& b, const hkVector4& c, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayText2d(const char* text, const hkVector4& pos, hkReal sizeScale, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayAnnotation(const char* text, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayBone(const hkVector4& a, const hkVector4& b, const hkQuaternion& orientation, hkColor::Argb color, int tag ) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayGeometry(const hkArrayBase<hkDisplayGeometry*>& geometries, const hkTransform& transform, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult displayGeometry(const hkArrayBase<hkDisplayGeometry*>& geometries, hkColor::Argb color, int id, int tag) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult sendMemStatsDump(const char* data, int length) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkResult holdImmediate() HK_OVERRIDE { return HK_SUCCESS; }
	virtual void lockForUpdate() HK_OVERRIDE {  }
	virtual void unlockForUpdate() HK_OVERRIDE { }
	virtual hkResult addGeometryLazily( const hkReferencedObject* source, hkDisplayGeometryBuilder* builder, const hkTransform& transform, hkUlong id, int tag, hkUlong shapeIdHint) HK_OVERRIDE { return HK_SUCCESS; }
	virtual hkBool doesSupportHashes() const HK_OVERRIDE { return false; }
	virtual hkResult addGeometryHash( const hkReferencedObject* source, hkDisplayGeometryBuilder* builder, const Hash& hash, const hkAabb& aabb, hkColor::Argb color, const hkTransform& transform, hkUlong id, int tag ) HK_OVERRIDE { return HK_FAILURE; }
	virtual hkResult updateGeometryVerts( hkUlong geomID, const hkArray<hkVector4>& newVerts, int tag ) HK_OVERRIDE { return HK_SUCCESS; }
};

static int HK_CALL _getWindowDeviceFlag( int d )
{
	switch (d)
	{
	case -1:
		return 0;
	case 0:
		return HKG_WINDOW_PRIMARY_DEVICE;
	case 1:
		return HKG_WINDOW_SECONDARY_DEVICE;
	case 2:
		return HKG_WINDOW_TERTIARY_DEVICE;
	case 3:
		return HKG_WINDOW_QUATERNARY_DEVICE;
	default:
		break;
	}
	return 0;
}

void setupDebugRenderNow();

bool initRendererAndEnv(hkDemoEnvironment& env)
{
	hkDemoFrameworkOptions& options = *(env.m_options);

	// Now initialize the graphics, using hkgSystem
	hkgSystem::init(options.m_renderer);


	hkgWindow* window = hkgWindow::create();
	window->setShadowMapSize( options.m_shadowMapRes );

	HKG_WINDOW_CREATE_FLAG windowFlags = _getWindowDeviceFlag( options.m_graphicsDevice ) |
		(options.m_windowed ? HKG_WINDOW_WINDOWED : HKG_WINDOW_FULLSCREEN) |
		(options.m_renderStereo ? HKG_WINDOW_HINT_STEREO : 0 ) |
		options.m_windowHint;

	if (options.m_enableMsaa)
	{
		windowFlags = windowFlags | HKG_WINDOW_MSAA |
			(HKG_WINDOW_HINT_MSAASAMPLES & (options.m_msaaSamples << HKG_WINDOW_HINT_MSAASAMPLES_SHIFT)) |
			(HKG_WINDOW_HINT_MSAAQUALITY & (options.m_msaaQuality << HKG_WINDOW_HINT_MSAAQUALITY_SHIFT));
	}

	if (options.m_enableVsync)
	{
		windowFlags = windowFlags | HKG_WINDOW_VSYNC |
			(HKG_WINDOW_HINT_VSYNCINTERVAL & (options.m_vsyncInterval << HKG_WINDOW_HINT_VSYNCINTERVAL_SHIFT));
	}

	void* nativePlatformHandle = HK_NULL;

	bool initOk = window->initialize( windowFlags,
		HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH32, options.m_width, options.m_height,
		options.m_windowTitle, nativePlatformHandle);

	// check that render is actually ok (shader drivern and platform has shaders for instance)
	// PC / variable gfx card renderers that could be shader lib driven (consoles later on)
	bool shaderLibDriven = 
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_OGLES2) || // GLSL 2
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX11) ||   // HLSL 4.0 / 5.0
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX9) ||   // HLSL 2.0 / 3.0
		(hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_OGLS);     // Cg   2.0

#if defined(HK_PLATFORM_WIIU) || defined(HK_PLATFORM_WINRT) || defined(HK_PLATFORM_PS4) || defined(HK_PLATFORM_DURANGO)
	if(hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_NULL)
#else
	if (shaderLibDriven)
#endif
	{
		if ( (!initOk) && (hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX11) )
		{
			hkprintf("HKG: DirectX not installed properly or older OS, so trying older renderer (DirectX9)\n");

			window->release();

			hkgSystem::quit();
			hkgSystem::init( "d3d9" );

			window = hkgWindow::create();
			window->setShadowMapSize( options.m_shadowMapRes );
			initOk = window->initialize( windowFlags,
				HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH32, options.m_width, options.m_height,
				options.m_windowTitle);
		}

		if ( (!initOk) || (HKG_GET_RENDER_CAPS(window).m_vertexShaderMajorVersion < 2))
		{
			hkprintf("HKG: Have to change to a fixed function renderer as current hardware does not support shader model 2 or higher!\n");

			// (D3D9, OGLES2) -> OGLES
			const char* fixedFunc = "ogles";

			window->release();

			hkgSystem::quit();
			hkgSystem::init( fixedFunc );

			window = hkgWindow::create();
			window->setShadowMapSize( options.m_shadowMapRes );
			initOk = window->initialize( windowFlags,
				HKG_WINDOW_BUF_COLOR | HKG_WINDOW_BUF_DEPTH32, options.m_width, options.m_height,
				options.m_windowTitle);
		}
		else 
		{
			// WiiU always suppports shadows, check on PC. 360/PS3 need to be cleaned up to get full shadows again
			if ( (HKG_GET_RENDER_CAPS(window).m_vertexShaderMajorVersion >= 3) && ((hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX9) || (hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_DX11)) )
			{
				// then dealling with a good graphics card, so we can assume shadows etc unlesss told otherwise
				if (!options.m_forceNoShadows)
				{
					options.m_enableShadows = true;
				}
			}
		}
	}


	if (hkgSystem::g_RendererType != hkgSystem::HKG_RENDERER_CONSOLE)
	{
#ifdef HK_DEBUG
		hkprintf("HKG: Using the %s renderer\n", hkgSystem::getRendererString() );
#endif
	}

	// Consoles that are ShaderLib drivern


	window->getContext()->lock();

#if defined HK_PLATFORM_WIN32 && !defined(HK_PLATFORM_WINRT) && !defined(HK_PLATFORM_DURANGO)
	if (options.m_windowed && options.m_repositionConsole && (options.m_xPos != -1) && (options.m_yPos != -1))
	{
		RECT visible;
		::SystemParametersInfo(SPI_GETWORKAREA, 0, &visible, 0);
		
		SetWindowPos( (HWND)(window->getPlatformHandle()), NULL,
			visible.left + options.m_xPos, visible.top + options.m_yPos,
			options.m_width, options.m_height, 0);
	}
	//SendMessage( (HWND)(window->getPlatformHandle()), 0x0112/*WM_SYSCOMMAND*/, 0xF030/*SC_MAXIMIZE*/, 0);
#endif

	// don't allow viewport resizing
	window->setWantViewportBorders(false);
	window->setWantViewportResizeByMouse(false);


	// setup navigation style
	if (options.m_invertFlyMode)
		window->getViewport(0)->setInvertLook(true);

	if (options.m_trackballMode > 0)
	{
		window->getViewport(0)->setNavigationMode(HKG_CAMERA_NAV_TRACKBALL);
		if (options.m_trackballMode == 2) // Max
			window->getViewport(0)->setMouseConvention(HKG_MC_3DSMAX);
		else if (options.m_trackballMode == 3) // Maya
			window->getViewport(0)->setMouseConvention(HKG_MC_MAYA);
	}

	window->setMouseGain( float(options.m_mouseGain) );

	if (options.m_fakeTouchEvents)
	{
		window->setMouseIsTouchEvent(true);
		window->setUseVirtualGamePad(true, true /*start hidden */);
	}

	for(int i = 0; i < 2; ++i)
	{
		window->clearBuffers();
		window->swapBuffers();
	}

	// Demos that use two pads will attempt to use the CTRL key for switching between pads.
	// Here we attempt to "free" the CTRL key for other uses if not needed for pads.
	hkPseudoPad* pad0 = new hkPseudoPad ();
	hkPseudoPad* pad1 = new hkPseudoPad ();

	if (window->hasGamePads())
	{
		hkprintf("HKG: Found a gamepad, so unless you have specified otherwise (-fakegamepad), only that input will be used.\n");
	}

	env.m_window = window;
	env.m_displayWorld = hkgDisplayWorld::create();
	env.m_sceneConverter = new hkgSceneDataConverter( env.m_displayWorld, window->getContext() );
	env.m_reportingLevel = (hkDemoEnvironment::ReportingLevel) options.m_reportingLevel;
	env.m_textDisplay = new hkTextDisplay(window);
	env.m_gamePad = pad0;
	env.m_gamePadB = pad1;
	if ( hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_NULL)
	{
		env.m_displayHandler = new NullDisplayHandler(env.m_displayWorld, env.m_window->getContext(), env.m_window);
	}
	else
	{
		env.m_displayHandler = new hkgDisplayHandler(env.m_displayWorld, env.m_window->getContext(), env.m_window);
	}
	

	if( shaderLibDriven )
	{
		env.m_displayHandler->setShaderLib( env.m_sceneConverter->m_shaderLibrary );
	}	

	hkDefaultDemo::setupLights(&env);

	window->getContext()->unlock();

	setupDebugRenderNow();

	if (options.m_enableVirtualFramebuffer)
	{
		env.m_virtualFrameBufferServer = new hkVirtualFramebufferServer();

		env.m_virtualFrameBufferServer->registerGamepadCallback( &env );
		env.m_virtualFrameBufferServer->registerKeyboardCallback( &env );
		env.m_virtualFrameBufferServer->registerMouseCallback( &env );
		env.m_virtualFrameBufferServer->registerFileDropCallback( &env );

		env.m_virtualFrameBufferServer->serve();
	}

	if (options.m_recordMovie)
	{
		env.m_movieRecorder = hkgMovieRecorder::create( window );
		env.m_movieRecorder->startRecording( options.m_movieFilename, options.m_motionBlurSamples );
	}

	return true;
}

void hkDemoEnvironment::onVirtualMouseUpdate( const hkVirtualMouse& m )
{
	const hkgMouse& cm = m_window->getMouse();
	// XX expand and make generic for all rotated devices
	int sX = int( m.m_screenX * m_window->getWidth() );
	int sY = int( m.m_screenY * m_window->getHeight() );
	if ((cm.getPosX() != sX) || (cm.getPosY() != sY))
	{
		m_window->processMouseMove(sX, sY, false);
	}

	//XX for now, the enums are the same, 
	HKG_MOUSE_BUTTON cmB = cm.getButtonState();
	if (cmB != (hkUint32)m.m_buttons)
	{
		hkVirtualMouse::Button buttons[] = { hkVirtualMouse::MOUSE_RIGHT_BUTTON, hkVirtualMouse::MOUSE_MIDDLE_BUTTON, hkVirtualMouse::MOUSE_LEFT_BUTTON };
		for (int mi=0; mi < 3; ++mi)
		{
			if ((cmB & buttons[mi]) != ( (hkUint32)m.m_buttons & buttons[mi]))
			{
				m_window->processMouseButton(buttons[mi], (m.m_buttons & buttons[mi]) != 0, sX, sY, false);
			}
		}
	}

	if (m.m_wheelDelta != 0)
	{
		m_window->processMouseWheel((int)m.m_wheelDelta, sX, sY, false);
	}
}

void hkDemoEnvironment::onVirtualKeyEventUpdate( const hkVirtualKeyEvent& m )
{
	const hkgKeyboard& kb = m_window->getKeyboard();
	if (kb.getKeyState( (HKG_KEYBOARD_VKEY)m.m_key ) != m.m_state )
	{
		m_window->processKey( m.m_key, m.m_state );
	}
}

void hkDemoEnvironment::onVirtualGamepadUpdate( const hkVirtualGamepad& g )
{
	const hkgPad& p = m_window->getGamePad(g.m_gamePadNum);
	short padNum = (short)g.m_gamePadNum;

	// Sticks
	if ( (p.getStickPosX(0) != g.m_sticks[0].x ) ||
		 (p.getStickPosY(0) != g.m_sticks[0].y ) )
	{
		m_window->processPadStickMove(padNum, 0, g.m_sticks[0].x, g.m_sticks[0].y );
	}

	if ( (p.getStickPosX(1) != g.m_sticks[1].x ) ||
		 (p.getStickPosY(1) != g.m_sticks[1].y ) )
	{
		m_window->processPadStickMove(padNum, 1, g.m_sticks[1].x, g.m_sticks[1].y );
	}

	// Buttons
	//XXX enums the same for now
	HKG_PAD_BUTTON pB = p.getButtonState();
	if (pB != (hkUint32)g.m_buttons)
	{
		hkVirtualGamepad::Button curButton = hkVirtualGamepad::PAD_BUTTON_0;
		while (curButton <= hkVirtualGamepad::PAD_BUTTON_RSTICK)
		{
			if ((pB & curButton) != ( (hkUint32)g.m_buttons & curButton))
			{
				m_window->processPadButton(padNum, curButton, (g.m_buttons & curButton) != 0);
			}
			curButton = (hkVirtualGamepad::Button)(curButton << 1);
		}
	}
	
	// Triggers
	if ( p.getTriggerPos(0) != g.m_triggers[0].z )
		m_window->processPadAnalogTrigger(padNum, HKG_PAD_LEFT_ANALOG_TRIGGER, g.m_triggers[0].z );

	if ( p.getTriggerPos(1) != g.m_triggers[1].z )
		m_window->processPadAnalogTrigger(padNum, HKG_PAD_RIGHT_ANALOG_TRIGGER, g.m_triggers[1].z );

}

void hkDemoEnvironment::onVirtualFileDrop( const hkVirtualFileDrop& d ) 
{
	if ( m_window->hasWindowDropFileFunction() )
	{
		for (int f=0; f < d.m_files.getSize(); ++f)
		{
			m_window->handleFileDrop( (const char*) d.m_files[f], (int)( d.m_screenX * m_window->getWidth() ), (int)( d.m_screenY * m_window->getWidth() ) );
		}
	}
}

bool startRenderFrame( hkDemoEnvironment& env )
{
	HKG_TIMER_BEGIN_LIST("_Render", "UpdatePads");

	hkgWindow* window = env.m_window;

	// Normally, pad1 is emulated by holding CTRL (pad0 is emulated when CTRL is not being held).
	// When only one pad is present, we want pad1 to be emulated with no CTRL so we flip the flag.
	// COM-305: If the user wants to use CTRL, we bypass this.
	hkBool pad0DesiredCtrlState;
	if (env.m_controlKeyReserved)
	{
		pad0DesiredCtrlState = window->getKeyboard().getKeyState(HKG_VKEY_CONTROL);
	}
	else
	{
		pad0DesiredCtrlState = env.getDesiredKeyboardCtrlStateForGamePad();
	}

	
	HKG_TIMER_SPLIT_LIST("ClearBuffers");
	
	window->getContext()->lock();
	bool graphicsWindowOK = window->clearBuffers();

	if (env.m_virtualFrameBufferServer)
	{
		if (env.m_virtualFrameBufferServer->getNumConnectedClients() > 0)
		{
			env.m_window->setRemoteInputControlled(true);
		}

		HKG_TIMER_SPLIT_LIST("hkVirtualFrameBufferServer::step");
		// We step after peek mesages, which is just before clear buffers, as in HKG that is where the current input is toggled too
		// so our remote callbacks here from step should be able to over write them
		env.m_virtualFrameBufferServer->step(); 

		if (env.m_virtualFrameBufferServer->getNumConnectedClients() < 1)
		{
			env.m_window->setRemoteInputControlled(false);
		}
	}

	// update pseudo pad after remote update
	((hkPseudoPad*)env.m_gamePad)->updatePad(0, window, 0.01f, env.m_options->m_forceKeyboardGamepad, pad0DesiredCtrlState);
	((hkPseudoPad*)env.m_gamePadB)->updatePad(1, window, 0.01f, env.m_options->m_forceKeyboardGamepad, !pad0DesiredCtrlState);

	env.m_inRenderLoop = true;

	window->getContext()->unlock();

	HKG_TIMER_END_LIST();

	return graphicsWindowOK;
}

static void _sendVdbCamera( hkgViewport* v, int index )
{
	hkgCamera* c = v->getCamera();

	float floatFrom[3]; c->getFrom(floatFrom);
	float floatTo[3]; c->getTo(floatTo);
	float floatUp[3]; c->getUp(floatUp);

	hkVector4 from; from.set( floatFrom[0], floatFrom[1], floatFrom[2] );
	hkVector4 to;   to.set( floatTo[0], floatTo[1], floatTo[2] );
	hkVector4 up;   up.set( floatUp[0], floatUp[1], floatUp[2] );

	if ( index == 0 )
	{
		HK_UPDATE_CAMERA(from, to, up, c->getNear(), c->getFar(), c->getFOV(), "Demo Framework");
	}
	else
	{	
		hkStringBuf name;
		name.printf("Demo Framework V%d", index);
		HK_UPDATE_CAMERA(from, to, up, c->getNear(), c->getFar(), c->getFOV(), name.cString());
	}
}

void sendCurrentFrameOverNetwork( hkDemoEnvironment& env )
{
	hkgWindow* window = env.m_window;
	if (env.m_virtualFrameBufferServer && (env.m_virtualFrameBufferServer->getNumConnectedClients() > 0) && window->supportsRenderTargetLock() )
	{
	
		HKG_TIMER_SPLIT_LIST("hkVirtualFrameBufferServer::sendFramebuffer");
		
		bool isInSecondaryRender = window->getContext()->getCurrentViewport() == window->getSecondaryViewport();
		int displayBufferId = 0;
		if (isInSecondaryRender)
			displayBufferId  = 2;
		else if (window->getCurrentStereoMode() == HKG_STEREO_RIGHT_EYE)
			displayBufferId  = 1;

		hkgFrameBufferDesc fd;
		hkgFrameBufferRect lockRect;
		hkVirtualFramebufferRelativeRect lockRectRel;
		bool subRect = env.m_virtualFrameBufferServer->getFramebufferRectOfInterest( lockRectRel );
		
		if (subRect)
		{
			if ((lockRectRel.m_framebufferId >= 0) && ( displayBufferId != lockRectRel.m_framebufferId ) )
			{
				// this buffer is not of current interest
				return;
			}
			else // check if 0 area of interest
			{
				if ( (lockRectRel.m_startX == lockRectRel.m_endX) || (lockRectRel.m_startY == lockRectRel.m_endY) )
				{
					return;
				}
			}
		}
			float ww = (float)window->getWidth();
			float wh = (float)window->getHeight();

		if (isInSecondaryRender)
		{
				ww = (float)window->getSecondaryViewport()->getWidth();
				wh = (float)window->getSecondaryViewport()->getHeight();
		}



		if (subRect)
		{
		
			lockRect.m_startPixelX = (hkUint32)( lockRectRel.m_startX * ww);
			if (lockRectRel.m_endX < 0 )
				lockRect.m_endPixelX = -1;
			else
				lockRect.m_endPixelX = (hkInt32)( lockRectRel.m_endX*ww );

			lockRect.m_startPixelY = (hkUint32)( lockRectRel.m_startY * wh);
			if (lockRectRel.m_endY < 0 )
				lockRect.m_endPixelY = -1;
			else 
				lockRect.m_endPixelY = (hkInt32)( lockRectRel.m_endY*wh );
		}

		if ( window->lockCurrentRenderTarget( fd, subRect? &lockRect : HK_NULL ) )
		{
			hkVirtualFramebuffer f;
			bool rgbOrder = (fd.m_format == HKG_TEXTURE_PIXEL_RGBA) || (fd.m_format == HKG_TEXTURE_PIXEL_RGB) || (fd.m_format == HKG_TEXTURE_PIXEL_ARGB);
			f.m_format = rgbOrder? hkVirtualFramebuffer::DATA_RGB : hkVirtualFramebuffer::DATA_BGR;
			f.m_rowOrder = (fd.m_orientation == HKG_TEXTURE_00_UPPER_LEFT)? hkVirtualFramebuffer::DATA_TOP_LEFT : hkVirtualFramebuffer::DATA_LOWER_LEFT;

			f.m_fullBufferWidthInPixels = (hkUint32)( ww );
			f.m_fullBufferHeightInPixels = (hkUint32)( wh );
			
			f.m_startX = fd.m_rect.m_startPixelX;
			f.m_startY = fd.m_rect.m_startPixelY;
			f.m_endX = fd.m_rect.m_endPixelX;
			f.m_endY = fd.m_rect.m_endPixelY;

			int ySize = f.m_endY >= 0 ? f.m_endY - f.m_startY : f.m_fullBufferHeightInPixels - f.m_startY;

			f.m_dataSizeInBytes = fd.m_rowPitchInBytes*ySize;
			f.m_rowPitchInBytes = fd.m_rowPitchInBytes;
			f.m_pixelStrideInBytes = fd.m_pixelStrideInBytes;
			f.m_data = fd.m_data;
			
			f.m_displayBufferId = 0;
			if (isInSecondaryRender)
				f.m_displayBufferId = 2;
			else if (window->getCurrentStereoMode() == HKG_STEREO_RIGHT_EYE)
				f.m_displayBufferId = 1;
				
			//XX iOS (move device rotation i/f to base class)
				f.m_displayRotation = hkVirtualFramebuffer::DISPLAY_0_DEGREES;

			env.m_virtualFrameBufferServer->sendFrameBuffer(&f);

			window->unlockCurrentRenderTarget();
		}
	}
}

void tickFrame(hkDemoEnvironment& env, bool justCamera)
{
	hkgWindow* window = env.m_window;
	if (!window || (window->getWidth() == 0) || (window->getHeight() == 0) )
	{
		return;
	}
	HKG_TIMER_BEGIN("_Render", HK_NULL);
	HKG_TIMER_BEGIN("tick", HK_NULL);

	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();



	int frameNum = ctx->advanceFrame();

	// Primary world:
	if (env.m_displayWorld)
	{
		env.m_displayWorld->advanceToFrame(frameNum, !justCamera, env.m_window);
		env.m_window->getViewport(0)->getCamera()->advanceToFrame(frameNum);
	}

	// Other views:
	for (int v=0; v < env.m_viewportData.getSize(); ++v)
	{
		hkgDisplayWorld* w = env.m_viewportData[v].m_displayWorld;
		if (w)
		{
			// ok if the same world as other views as it will see the frame num is the same.
			w->advanceToFrame(frameNum, !justCamera, env.m_window);
		}
		env.m_window->getViewport( env.m_viewportData[v].m_windowViewportIndex )->getCamera()->advanceToFrame(frameNum);
	}

	ctx->unlock();
	
	HKG_TIMER_END();
	HKG_TIMER_END();
}

void clearFrameData(hkDemoEnvironment& env)
{
	hkgWindow* window = env.m_window;
	for( int viewportIndex = 0; viewportIndex < window->getNumViewports(); ++viewportIndex )
	{
		// only reason there won't be a displayWorld is if the demo is doing the rendering itself.
		hkDemoEnvironment::ViewportData* viewportData = env.getViewportData( viewportIndex );
		hkTextDisplay* td = viewportData && viewportData->m_textDisplay? viewportData->m_textDisplay : env.m_textDisplay;
		if ( td )
		{
			td->clearAndDeallocate();
		}
	}
}

void renderFrame( hkDemoEnvironment& env, hkDemo* demo )
{
	HKG_TIMER_BEGIN("Render", HK_NULL);

	hkgWindow* window = env.m_window;

	if (!window || (window->getWidth() == 0) || (window->getHeight() == 0) || (hkgSystem::g_RendererType == hkgSystem::HKG_RENDERER_NULL))
	{
		HKG_TIMER_BEGIN("SwapBuffers", HK_NULL);
		if (window)
		{
			window->getContext()->lock();
			window->swapBuffers();
			window->getContext()->unlock();
		}
		HKG_TIMER_END();

		clearFrameData(env);
		HKG_TIMER_END(); // render
		return; // nothing to render too..
	}

	hkgDisplayContext* ctx = window->getContext();
	ctx->lock();

	if (window->getCurrentStereoMode() == HKG_STEREO_RIGHT_EYE)
	{
		window->setCurrentStereoMode(HKG_STEREO_LEFT_EYE); // always start with left
	}

	do 
	{

#ifdef HK_GRAPHICS_TIMERS_ENABLED
		if (window->getCurrentStereoMode() == HKG_STEREO_LEFT_EYE)
		{
			HKG_TIMER_BEGIN_LIST("LeftEye", "Get Viewport");
		}
		else if (window->getCurrentStereoMode() == HKG_STEREO_RIGHT_EYE)
		{
			HKG_TIMER_BEGIN_LIST("RightEye", "Get Viewport");
		}
		else
		{
			HKG_TIMER_BEGIN_LIST("MainWindow", "Get Viewport");
		}
#endif

		hkgViewport* masterView = window->getCurrentViewport();
		if (env.m_shareCameraBetweenViewports)
		{
			const hkgCamera* fromC = masterView->getCamera(); 
			for( int viewportIndex = 0; viewportIndex < window->getNumViewports(); ++viewportIndex )
			{
				// only reason there won't be a displayWorld is if the demo is doing the rendering itself.
				hkgViewport* v = window->getViewport(viewportIndex);
				if (v != masterView)
				{
					hkgCamera* toC = v->getCamera();
					// Don't copy to different modes as doesn't make sense
					if ((toC->getProjectionMode() == HKG_CAMERA_PERSPECTIVE) && (fromC->getProjectionMode() == HKG_CAMERA_PERSPECTIVE) )
					{
						float aspect = toC->getAspect();
						toC->copy( *fromC );
						toC->setAspect(aspect);
						toC->computeProjection();
						toC->computeFrustumPlanes();
					}
				}
			}
		}

		for( int viewportIndex = 0; viewportIndex < window->getNumViewports(); ++viewportIndex )
		{
			// only reason there won't be a displayWorld is if the demo is doing the rendering itself.
			hkgViewport* v = window->getViewport(viewportIndex);
			hkDemoEnvironment::ViewportData* viewportData = env.getViewportData( viewportIndex );
		
			HKG_TIMER_SPLIT_LIST("SetViewport");

			v->setAsCurrent( ctx );

			_sendVdbCamera( v, viewportIndex );

			if (v->getSkyBox())
			{
				HKG_TIMER_SPLIT_LIST("SkyBox");
				v->getSkyBox()->render( ctx, v->getCamera() );
			}
		
			hkgDisplayWorld* dw = viewportData && viewportData->m_displayWorld? viewportData->m_displayWorld : env.m_displayWorld;
			if (dw)
			{
				dw->setFrameTime( float(env.m_frameTimer.getLastFrameTime()) );
			}
			HKG_TIMER_SPLIT_LIST("DemoPreRenderWorld");

			if (demo)
			{
				demo->preRenderDisplayWorld( v );
			}

			HKG_TIMER_SPLIT_LIST("DisplayWorld");
			if (dw)
			{
				// can't alter the world in the middle of a render pass, so it will lock itself
				dw->render( ctx, true, env.m_options->m_enableShadows ); // culled with shadows (if any setup)

				if (env.m_options->m_edgedFaces)
				{
					HKG_COLOR_MODE origColorMode = ctx->getColorMode();
					HKG_BLEND_MODE origBlendMode = ctx->getBlendMode();
					HKG_ENABLED_STATE origEnabledState = ctx->getEnabledState();

					ctx->setWireframeState(true);
					ctx->setDepthReadState(true);
					ctx->setDepthWriteState(false);
					ctx->setLightingState(false);
					ctx->setTexture2DState(false);
					ctx->setBlendState(true);

					ctx->setLineWidth(8);

					// Most assets will be loaded using the ShaderLib
					// Currently that lib only supports lit senarios
					// wheras here we want the wireframe to be unlit etc.
					// So we will enforce no per material shader (not required as default shaders will be ok for most things, except particles etc)

					// Render white wireframe with low alpha
					hkgMaterial* globalWireMaterial = hkgMaterial::create();
					globalWireMaterial->setDiffuseColor(1,1,1,0.1f);
					ctx->setCurrentMaterial(globalWireMaterial, HKG_MATERIAL_VERTEX_HINT_NONE); // Global mat
					globalWireMaterial->removeReference();

					ctx->setColorMode( HKG_COLOR_GLOBAL | HKG_COLOR_GLOBAL_SHADER_COLLECTION );

					hkgCamera* curCamera = v->getCamera();
					float origFrom[3];
					curCamera->getFrom(origFrom);
					float* currentTo = curCamera->getToPtr();

					float newFrom[3];
					hkgVec3Sub( newFrom, currentTo, origFrom);
					hkgVec3Scale( newFrom, 1.0e-3f );
					hkgVec3Add( newFrom, origFrom);
					curCamera->setFrom( newFrom );
					curCamera->computeModelView(false);
					curCamera->setAsCurrent(ctx);

					// Render from this shifted POV
					dw->render( ctx, true, false );

					ctx->setLineWidth(-1);

					// Reset context mode
					ctx->setColorMode(origColorMode);
					ctx->matchState(origEnabledState, ctx->getCullfaceMode(), origBlendMode, ctx->getAlphaSampleMode());
					curCamera->setFrom(origFrom);
					curCamera->computeModelView(false);
					curCamera->setAsCurrent(ctx);
				}
			}

			HKG_TIMER_SPLIT_LIST("DemoPostRenderWorld");
			if (demo)
			{
				demo->postRenderDisplayWorld( v );
			}


			HKG_TIMER_SPLIT_LIST("DrawImmediate");
			hkgDisplayHandler* dh = viewportData && viewportData->m_displayHandler? viewportData->m_displayHandler : env.m_displayHandler;
			if (dh)
			{
				dh->drawImmediate();
			}
		}

		HKG_TIMER_SPLIT_LIST("PostEffects");
		
		window->applyPostEffects();
		
		HKG_TIMER_SPLIT_LIST("Final Pass Objects");
		{
			for( int viewportIndex = 0; viewportIndex < window->getNumViewports(); ++viewportIndex )
			{
				// only reason there won't be a displayWorld is if the demo is doing the rendering itself.
				hkDemoEnvironment::ViewportData* viewportData = env.getViewportData( viewportIndex );
				hkgDisplayWorld* dw = viewportData && viewportData->m_displayWorld? viewportData->m_displayWorld : env.m_displayWorld;
				if ( dw )
				{
					window->getViewport(viewportIndex)->setAsCurrent( ctx );
					dw->finalRender(ctx, true /* frustum cull */);
				}
			}
		}

		masterView->setAsCurrent( ctx );

		HKG_TIMER_SPLIT_LIST("DemoPostRenderWindow");
		if (demo)
		{
			demo->postRenderWindow( window );
		}

		// Draw text after all post effects etc
		HKG_TIMER_SPLIT_LIST("Display3DText");
		{
			for( int viewportIndex = 0; viewportIndex < window->getNumViewports(); ++viewportIndex )
			{
				// only reason there won't be a displayWorld is if the demo is doing the rendering itself.
				hkDemoEnvironment::ViewportData* viewportData = env.getViewportData( viewportIndex );
				hkTextDisplay* td = viewportData && viewportData->m_textDisplay? viewportData->m_textDisplay : env.m_textDisplay;
				if ( td )
				{
					hkgViewport* v = window->getViewport(viewportIndex);
					td->displayJust3DText(window, v );
				}
			}
		}

		// Text after post render window (as it displays the stats bar etc)
		if ( !window->getSecondaryViewport() )
		{
			HKG_TIMER_SPLIT_LIST("Display2DText");
		
			env.m_textDisplay->displayJust2DText(window, false);
		}

		// We commonly use 'swap discard' (better for SLi/CrossFire etc) as the swap effect, so 
		// that means you neeed tyo save the RT before Swap or its contents may be erased
		// A side effect of this is that the image will not have the Havok logo on the lower corner

		bool preSwapSendBuffer = window->renderTargetReadyPreSwap();
		if (preSwapSendBuffer && env.m_virtualFrameBufferServer)
		{
			HKG_TIMER_SPLIT_LIST("ConsoleViewer");
			sendCurrentFrameOverNetwork(env);
		}

		if (preSwapSendBuffer && env.m_options->m_saveFrames)
		{
			HKG_TIMER_SPLIT_LIST("SaveBMP");

			char filename[128];
			hkString::sprintf(filename,"frame%05i.bmp",env.m_options->m_numSaveFrames);
			window->saveCurrentRenderTargetToBmp(filename);
			env.m_options->m_numSaveFrames++;
		}

		if ( preSwapSendBuffer && env.m_options->m_recordMovie )
		{
			env.m_movieRecorder->addFrame();
		}

		HKG_TIMER_SPLIT_LIST("SwapBuffers");

		window->swapBuffers();

		if (!preSwapSendBuffer && env.m_virtualFrameBufferServer)
		{
			HKG_TIMER_SPLIT_LIST("ConsoleViewer");
			sendCurrentFrameOverNetwork(env);
		}

		if (!preSwapSendBuffer && env.m_options->m_recordMovie)
		{
			env.m_movieRecorder->addFrame();	
		}

		if (!preSwapSendBuffer && env.m_options->m_saveFrames)
		{
			HKG_TIMER_SPLIT_LIST("SaveBMP");

			char filename[128];
			hkString::sprintf(filename,"frame%05i.bmp",env.m_options->m_numSaveFrames);
			window->saveCurrentRenderTargetToBmp(filename);
			env.m_options->m_numSaveFrames++;
		}

		window->toggleCurrentStereoTargetEye();

		HKG_TIMER_END_LIST();

	}  while (window->getCurrentStereoMode() == HKG_STEREO_RIGHT_EYE);

	hkgViewport* secondaryView = window->getSecondaryViewport();
	if (secondaryView)
	{
		HKG_TIMER_BEGIN("SecondaryViewport", HK_NULL);

		float sc[] = { 0.25f, 0.25f, 0.25f };
		window->clearSecondaryViewport(sc);

		//secondaryView is normally an othroview, for extra menus, stats etc
		secondaryView->setAsCurrent( ctx );

		if (demo)
		{
			// menu demo renders stats etc in second view if available
			demo->renderSecondaryViewport(secondaryView);
		}

		env.m_textDisplay->displayJust2DText(window, true);

		bool preSwapSendBuffer = window->renderTargetReadyPreSwap();
		if (preSwapSendBuffer && env.m_virtualFrameBufferServer)
		{
			sendCurrentFrameOverNetwork(env);
			
		}
		
		if ( preSwapSendBuffer && env.m_options->m_recordMovie )
		{
			env.m_movieRecorder->addFrame();
		}
		
		window->swapSecondaryViewport();
		
		if (!preSwapSendBuffer && env.m_virtualFrameBufferServer)
		{
			sendCurrentFrameOverNetwork(env);
		
		}
		
		if (!preSwapSendBuffer &&  env.m_options->m_recordMovie )
		{
			env.m_movieRecorder->addFrame();
		}

		HKG_TIMER_END();
	}

	HKG_TIMER_BEGIN("waitForCompletion", HK_NULL);
	window->waitForCompletion();
	HKG_TIMER_END(); 

	env.m_inRenderLoop = false;

	ctx->unlock();

	HKG_TIMER_END(); // render
}

void quitRendererAndEnv(hkDemoEnvironment& env)
{
	if (env.m_virtualFrameBufferServer)
	{
		env.m_virtualFrameBufferServer->removeReference();
		env.m_virtualFrameBufferServer = HK_NULL;
	}

	if (env.m_options->m_recordMovie)
	{
		env.m_movieRecorder->stopRecording();
		env.m_movieRecorder->removeReference();
	}

	hkgDisplayContext* ctx = env.m_window->getContext();
	ctx->lock();
	{
		env.m_window->setWindowResizeFunction( HK_NULL, HK_NULL );
		env.m_window->setFullscreen(false, env.m_window->getWidth(), env.m_window->getHeight());

		delete (hkPseudoPad*)env.m_gamePad;
		delete (hkPseudoPad*)env.m_gamePadB;
		
		delete env.m_textDisplay;
		delete env.m_displayHandler;
		env.m_sceneConverter->release();
		env.m_displayWorld->release();
		for (int vi=0; vi < env.m_viewportData.getSize(); ++vi)
		{
			hkDemoEnvironment::ViewportData& vd = env.m_viewportData[vi];
			if (vd.m_textDisplay) delete vd.m_textDisplay;
			if (vd.m_displayHandler) delete vd.m_displayHandler;
			if (vd.m_sceneConverter) vd.m_sceneConverter->release();
			if (vd.m_displayWorld) vd.m_displayWorld->release();
		}
		env.m_viewportData.setSize(0);
	}
	ctx->unlock();
	env.m_window->release();

	// All hkg objects should be gone by now:
	hkgSystem::quit();
}



void hkDemoFrameTimer::init( const hkDemoEnvironment& env )
{
	int lockFps = env.m_options->m_lockFps;

	m_previousFpsLimit = lockFps;
	m_minSecondsFrame = lockFps ? (1.0f / lockFps) : 0;
	m_minTicksPerFrame = lockFps ? (hkStopwatch::getTicksPerSecond() / lockFps) : 0;
}

void hkDemoFrameTimer::updateAndWait( const hkDemoEnvironment& env )
{
	hkDemoFrameworkOptions& options = *(env.m_options);
	if( (options.m_lockFps > 0) || options.m_showFps || options.m_consoleFps)
	{
		HKG_TIMER_BEGIN_LIST("_Render", "SoftwareFrameLock");

		if (options.m_lockFps != m_previousFpsLimit)
		{
			init( env );
		}

		while ( (options.m_lockFps > 0) && (m_minSecondsFrame - m_ticksSoFar.getElapsedSeconds() > 0 ))
		{
			// Busy wait..
		}

		hkUint64 ticks = m_ticksSoFar.getElapsedTicks();
		m_frameTimes[m_curFrame++] = static_cast<int>(ticks);
		m_curFrame = m_curFrame % HK_DEMO_FRAMELOCK_MAX_FRAMES;

		int minTime = m_frameTimes[0];
		int maxTime = m_frameTimes[0];
		for(int i = 1; i < HK_DEMO_FRAMELOCK_MAX_FRAMES; ++i)
		{
			if(m_frameTimes[i] < minTime)
			{
				minTime = m_frameTimes[i];
			}
			else if(m_frameTimes[i] > maxTime)
			{
				maxTime = m_frameTimes[i];
			}
		}

		if (options.m_consoleFps && (m_curFrame == (HK_DEMO_FRAMELOCK_MAX_FRAMES - 1)) )
		{
			static int interval = 0;
			++interval;
			if (interval == 20) // 20x max frames == 200 normally
			{
				if (minTime > 0)
				{
					float fpsMin = float(hkStopwatch::getTicksPerSecond()) / float(minTime);
					float fpsMax = float(hkStopwatch::getTicksPerSecond()) / float(maxTime);
					hkprintf("FPS min:%.2f max: %.2f\n", fpsMin, fpsMax);
					interval = 0;
				}
			}
		}
		if (options.m_showFps)
		{
			int fps = 1 + int(hkStopwatch::getTicksPerSecond() / (minTime+1));
			hkStringBuf str; str.printf("%i", fps);
			int x  = env.m_window->getWidth() - env.m_window->getTVDeadZoneH() - 100;
			int y  = env.m_window->getTVDeadZoneV() + 16;
			env.m_textDisplay->outputText(str, x-1, y+1, 0x7f000000);
			env.m_textDisplay->outputText(str, x, y, 0x7fffffff);
		}

		HKG_TIMER_END_LIST();
	}
	else // still track frame times so that we can use them in demos
	{
		hkUint64 ticks = m_ticksSoFar.getElapsedTicks();
		m_frameTimes[m_curFrame++] = static_cast<int>(ticks);
		m_curFrame = m_curFrame % HK_DEMO_FRAMELOCK_MAX_FRAMES;
	}
	
	m_ticksSoFar.reset();
	m_ticksSoFar.start();
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
