/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */
#include <Demos/demos.h>
#include <Demos/DemoCommon/DemoFramework/hkDemoDatabase.h>
#include <Demos/DemoCommon/DemoFramework/hkDemo.h>
#include <Demos/DemoCommon/DemoFramework/hkDemoFramework.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/System/Io/FileSystem/hkNativeFileSystem.h>
#include <Common/Base/Reflection/hkClassEnum.h>
#include <Common/Base/Ext/hkBaseExt.h>

#include <Common/Base/KeyCode.h>

hkDemoDatabase::hkDemoDatabase()
{
}


hkDemoEntryRegister::hkDemoEntryRegister( DemoCreationFunction func, int type, const char* path,
										 hkDemoEntryRegister* entries, int numVariants, int sizeOfVariantStruct,
										 const char*const* variantNames, const char* help, const char*const* details, bool actuallyReg )
{
	for (int i = 0; i < numVariants;i++)
	{
		const char* variantName = *variantNames;
		if ( !variantName )
		{
			break;
		}
		const char* detail = *details;
		if ( !detail )
		{
			detail = "";
		}
		const char* h = (help) ? help : detail;

		new (&entries[i]) hkDemoEntryRegister( func, type, path, i, variantName, h, detail, actuallyReg );
		variantNames = hkAddByteOffsetConst( variantNames, sizeOfVariantStruct );
		details = hkAddByteOffsetConst( details, sizeOfVariantStruct );
	}
}

// for demos which create other entries
hkDemoEntryRegister::hkDemoEntryRegister(DemoCreationFunction func, int type, const char* path,
					DemoEntriesCreationFunction entryCreateFunc, 
					const char* help, const char* details, bool actuallyReg )
{
	new(this)hkDemoEntryRegister( func, type, path, -1, HK_NULL, help, details, actuallyReg ); 
	m_createEntryFunc = entryCreateFunc;
}


hkDemoEntryRegister::hkDemoEntryRegister( DemoCreationFunction func, int type, const char* path,
										 hkDemoEntryRegister* entries, int numEntries,
										 const hkClassEnum& cenum, 
										 const char* help, const char* details, bool actuallyReg )
{
	int nitems = cenum.getNumItems();
	HK_ASSERT(0x7d4ed233, numEntries >= nitems );
 	for( int i = 0; i < nitems; i++ )
	{
		const hkClassEnum::Item& item = cenum.getItem(i);
		new (&entries[i]) hkDemoEntryRegister( func, type, path, item.getValue(), item.getName(), help, details, actuallyReg );
	}
}

hkDemoEntryRegister::hkDemoEntryRegister(	DemoCreationFunction func, int typeflags, const char* path,
										 int variant, const char* variantName, 
										 const char* help, const char* details, bool actuallyReg )
										 :	m_func(func), m_createEntryFunc(HK_NULL), m_demoTypeFlags(typeflags), m_demoPath(path),
										 m_variantId( variant ), m_variantName( variantName ),
										 m_help(help), m_details(details)
{
	// If we don't auto reg, we assume that it will be in a static array of demos / tests etc
	if (actuallyReg )
	{
		m_next = hkDemoDatabase::s_demoList;
		hkDemoDatabase::s_demoList = this;
	}
	else
	{
		m_next = HK_NULL;
	}
}

void hkDemoEntryRegister::registerDemo()
{
	HK_ASSERT2( 0xf0323454, !m_next, "Demo already registered");
	m_next = hkDemoDatabase::s_demoList;
	hkDemoDatabase::s_demoList = this;
}

// to sort the demos array alphabetically and by variant id using hkSort
class demoEntryPathAndVariantIdLess
{
private:
	hkBool32 areVariants( const hkDemoEntry* a, const hkDemoEntry* b )
	{
		// check if they have a valid variant field
		if( (a->m_variantId >= 0) && (b->m_variantId >= 0) ) 
		{
			// if so, check if the base path is the same
			extStringBuf sa(a->m_menuPath); sa.pathDirname();
			extStringBuf sb(b->m_menuPath); sb.pathDirname();
			return sb==sa;
		}
		else
		{
			return false;
		}
	}
	// true if the paths are the same and a->variantId < b->variantId
	HK_FORCE_INLINE hkBool ifSameLessVariantId( const hkDemoEntry* a, const hkDemoEntry* b )
	{
		// if they are variants of the same demo check the variant id
		return areVariants(a,b) && ( a->m_variantId < b->m_variantId );
	}

public:

	HK_FORCE_INLINE hkBool operator() ( const hkDemoEntry* a, const hkDemoEntry* b )
	{
		// order by variant id if they are variants of the same demo
		if (areVariants(a,b))
			return ( a->m_variantId < b->m_variantId );

		// alphabetically otherwise
		return hkString::strCmp(a->m_menuPath, b->m_menuPath) < 0;
	}
};


void findCommonRootPath( const extArray< hkDemoEntry* >& demos, extStringBuf& prefix )
{
	prefix = "";
	//
	// Find the common root, be careful if there is only one demo
	// special: if we see "*/Demo/" means <toplevel>/Demo
	{
		int i = 0;
		while( i < demos.getSize() && prefix.getLength() == 0 )
		{
			if(demos[i]->m_menuPath[0] != '*') // get a regular path.
			{
				prefix = demos[i]->m_menuPath;
				prefix.pathBasename();
				prefix.lowerCase();
			}
			++i;
		}
	}
	for(int i = 0; i < demos.getSize() && prefix.getLength() != 0; ++i)
	{
		if( demos[i]->m_menuPath[0] != '*')
		{
			extStringBuf gp(demos[i]->m_menuPath);
			gp.lowerCase();
			while( !gp.startsWith(prefix) )
			{
				// commonRoot is foo/bar/ we want the second last /
				int lastSlash = prefix.lastIndexOf('/', 0, prefix.getLength() - 1 );
				if(lastSlash >= 0)
				{
					// want the trailing / in the commonRoot
					prefix.slice(0, lastSlash + 1);
				}
				else
				{
					prefix = "";
				}
			}
		} // if != *
	} // for
}

// Uncomment and modify this to produce build a specific subset of demos e.g. for a client or trade show
//#define USING_DEMO_FILTERS

#ifdef USING_DEMO_FILTERS
const char* demoFilters[] = 
{
	"DemoCommon/Utilities", // You need to include this for MenuDemo, and probably want BootstrapDemo too
	"Ai/Api"
};
#endif

// Note: Behavior/Animation only seek to set-apart PHYSICS_2012 demos, so as such only uses the PHYSICS_2012 prefixes
static const char* remapPrefixes[] =
{
	"HKAI_PHYSICS",
	"HKB_PHYSICS_2012",
	"HKA_PHYSICS_2012"
};

static void s_removeDemoBasePath( hkStringBuf& path )
{
	// strip everything up to the demo root folder
	const char root[] = "/demos/";
	int start = path.indexOfCase(root);
	if( start != -1 )
	{
		// if the path contains one of remapPrefixes, preserve it's location at the start.
		int prefixIndex = -1;
		for (int i=0; i< (int) HK_COUNT_OF(remapPrefixes); i++)
		{
			if( path.startsWith(remapPrefixes[i]) ) 
			{
				prefixIndex = i;
				break;
			}
		}

		path.chompStart(start + hkString::strLen(root) );
		if(prefixIndex != -1)
		{
			path.prepend( remapPrefixes[prefixIndex] );
		}
	}
}

void hkDemoDatabase::rebuildDatabase(bool isContinuousIntegration /*= false*/)
{
	m_demos.clear();

	//
	// Normalize and collect all demo register entries
	// into a temporary array
	//
	extArray<hkDemoEntry*> demos;	
	hkDemoEntryRegister* e = s_demoList;
	{
		hkStringBuf menuPath;
		while(e)
		{
			hkStringBuf resourcePath = e->m_demoPath;
			resourcePath.pathNormalize();
			s_removeDemoBasePath( resourcePath );

			HK_ASSERT3(0x6e7879d4, resourcePath.lastIndexOf('/') >= 0,  "probably a metrowerks build with no 'pragma fullpath_file on': " << resourcePath );

			// Copy as the menu path
			menuPath = resourcePath; // remove the demo filename for the menu
			menuPath.pathDirname();

			// If we have a variant, append the variant name
			if( e->m_variantName )
			{
				menuPath += "/";
				menuPath += e->m_variantName;
			}

	#ifdef USING_DEMO_FILTERS
			{
				extStringBuf name(menuPath);
				hkBool32 valid = false;
				for (int f=0; f < HK_COUNT_OF(demoFilters); f++)
				{
					valid |= name.startsWith(demoFilters[f]);
				}
				if (!valid)
				{
					e = e->m_next;
					continue;
				}
			}
	#endif

			// Store the entry
			hkDemoEntry* entry = new hkDemoEntry();
			entry->m_func = e->m_func;
			entry->m_demoTypeFlags = e->m_demoTypeFlags;
			entry->m_menuPath = menuPath;
			entry->m_demoPath = resourcePath;
			entry->m_variantId = e->m_variantId;
			entry->m_help = e->m_help;
			entry->m_details = e->m_details;

			// Some AI/Behavior/Animation demos have multiple variants depending on physics type (or lack thereof)
			// Possibly move them to a different directory.
			// Below is a table (with Ai as an example) of the intended directory structure.

			//	Products	|            Demo Variant Type
			//	registered	| P					| NP		| no phys
			//	------------+-------------------+-----------+--------
			//	P, NP		| Ai/Physics2012/	| default	| Ai/NoPhysics/
			//	P			| default			| n/a		| Ai/NoPhysics/
			//	NP			| n/a				| default	| Ai/NoPhysics/
			//	none		| n/a				| n/a		| default

			bool isAiDemo = ( entry->m_demoTypeFlags & HK_DEMO_TYPE_AI );
			bool isBehaviorDemo = ( entry->m_demoTypeFlags & HK_DEMO_TYPE_BEHAVIOR );
			bool isAnimationDemo = ( entry->m_demoTypeFlags & HK_DEMO_TYPE_ANIMATION );

			if ( isAiDemo || isBehaviorDemo || isAnimationDemo )
			{
				hkBool32 startsWithPrefix = false;
				for (int i=0; i< (int) HK_COUNT_OF(remapPrefixes); i++)
				{
					startsWithPrefix |= entry->m_menuPath.startsWith(remapPrefixes[i]);
				}

				if ( startsWithPrefix )
				{
					const char* originalRootDir = HK_NULL;
					if ( isAiDemo )
					{
						originalRootDir = "Ai/";
					}
					else if ( isBehaviorDemo )
					{
						originalRootDir = "Behavior/";
					}
					else if ( isAnimationDemo )
					{
						originalRootDir = "Animation/";
					}
					else
					{
						HK_ASSERT(0x22440111, false);
					}

					hkStringPtr newRootDir;
					newRootDir = originalRootDir;
					if(entry->m_demoTypeFlags & HK_DEMO_TYPE_USES_PHYSICS_2012)
					{
#if defined(HK_FEATURE_PRODUCT_PHYSICS)
						newRootDir.printf("%s%s", originalRootDir, "Physics2012/");
#endif
					}
					else if( (entry->m_demoTypeFlags & (HK_DEMO_TYPE_USES_PHYSICS | HK_DEMO_TYPE_USES_PHYSICS_2012) ) == 0)
					{
#if defined(HK_FEATURE_PRODUCT_PHYSICS) || defined(HK_FEATURE_PRODUCT_PHYSICS_2012)
						newRootDir.printf("%s%s", originalRootDir, "NoPhysics/");
#endif
					}

					if( hkString::strCmp( newRootDir, originalRootDir ) != 0 )
					{
						entry->m_demoTypeFlags &= ~HK_DEMO_TYPE_CRITICAL; // clear flag for "non-main" physics versions
						if (isAiDemo && isContinuousIntegration)
						{
							entry->m_demoTypeFlags |= HK_DEMO_TYPE_BOOTSTRAP_FIRST_VARIANT_ONLY;
						}
					}

					const char* subDir = hkString::strStr(entry->m_menuPath, originalRootDir);
					HK_ASSERT(0x62ca738a, subDir);
					subDir += hkString::strLen(originalRootDir);
					entry->m_menuPath = extStringBuf( newRootDir, subDir );
				}
			}

			if ( e->m_createEntryFunc )	// now we have a create other demos function
			{
				e->m_createEntryFunc(*entry, demos);
				delete entry;
			}
			else
			{
				demos.pushBack(entry);
			}
			e = e->m_next;

		}
	}

	HK_ASSERT2(0x4467bbc3, demos.getSize() != 0, "No demos registered!");

	// find a common root for the demos, if any
	{
		extStringBuf prefix;
		findCommonRootPath( demos, prefix );
		m_prefix = prefix;
	}

	// strip the common root plus from the names
	int prefixLen = m_prefix.getLength();
	for(int i = 0; i < demos.getSize(); ++i)
	{
		// prune the prefix from each name 
		
		int chomp = demos[i]->m_menuPath[0] == '*' ? 2 // this comes from unit tests, */means "put me at the demos level"???
			: m_prefix.getLength() > 0 ? prefixLen
			: 0;
		if( chomp )
		{
			demos[i]->m_menuPath = demos[i]->m_menuPath+chomp;
		}
	}

	//
	// Sort the array alphabetically and by ID in the same variant group
	//
	hkSort( demos.begin(), demos.getSize(), demoEntryPathAndVariantIdLess() );

	// Collect the menu order files into an array
	extern const char* DemoOrder[];
	// Build the member demos, sorting according to the menu order
	for (int i = 0; DemoOrder[i] != 0; i++)
	{
		// the demos are already sorted alphabetically
		int j=0;
		// look for the first matching path.. 
		while( j < demos.getSize() && !( demos[j]!=HK_NULL && hkString::beginsWith( demos[j]->m_menuPath, DemoOrder[i]) ) ) 
			++j;
		// ..and add it together with all the following matching ones
		while( j < demos.getSize() && demos[j]!=HK_NULL && hkString::beginsWith(demos[j]->m_menuPath, DemoOrder[i]) ) 
		{
			m_demos.pushBack( *(demos[j]) );
			delete demos[j];
			demos[j] = HK_NULL;
			++j;
		}
	}

	// Store any leftover demos
	for (int j = 0; j < demos.getSize(); j++)
	{
		if(demos[j] != HK_NULL)
		{
			m_demos.pushBack( *(demos[j]) );
		}
		delete demos[j];
	}
	// cleanup
	demos.clear();

	/*
	//
	// Write the demo list to a file
	//
	hkOstream stream(HK_DEMO_LIST_FILENAME);
	if ( stream.isOk() )
	{
	stream << ";Havok '" << productCode << "' demo list:" << '\n' << '\n';
	for (int j = 0; j < m_demos.getSize(); j++)
	{
	stream << m_demos[j].m_menuPath << '\n';
	}
	}
	*/
}

void hkDemoDatabase::dbgPrintAllDemos()
{
	for(int i = 0; i < m_demos.getSize(); ++i)
	{
		HK_REPORT("Demo " << i << ": " << m_demos[i].m_menuPath.cString());
	}
}

// Case insensitive demo index finder
int hkDemoDatabase::findDemo(const char* cname) const
{
	// see if it's a number (demo index)
	{
		int value = 0;
		
		for( int i = 0; /**/; ++i )
		{
			int c = cname[i];
			if( c >= '0' && c <= '9' )
			{
				value = (value*10) + c-'0';
			}
			else if( c == 0) // we must match the whole string to handle demo names with leading numbers
			{
				return (value < m_demos.getSize()) ? value : -1;
			}
			else // not a digit? fall through to string check
			{
				break;
			}
		}
	}
	// try full path
	for(int i = 0; i < m_demos.getSize(); ++i)
	{
		if( hkString::strCasecmp( m_demos[i].m_menuPath.cString(), cname ) == 0 )
		{
			return i;
		}
	}
	// try to match the name only
	for(int i = 0; i < m_demos.getSize(); ++i)
	{
		int lastSlash = hkString::lastIndexOf(m_demos[i].m_menuPath, '/');
		if( lastSlash >= 0 && hkString::strCasecmp( m_demos[i].m_menuPath.cString()+lastSlash+1, cname ) == 0 )
		{
			return i;
		}
	}
	// try the requested name as a demo prefix
	for(int i = 0; i < m_demos.getSize(); ++i)
	{
		int lastSlash = hkString::lastIndexOf(m_demos[i].m_menuPath, '/');
		if( lastSlash >= 0 && hkString::strNcasecmp( m_demos[i].m_menuPath.cString()+lastSlash+1, cname, hkString::strLen(cname) ) == 0 )
		{
			return i;
		}
	}

	return -1;
}

// Create a demo by index
hkDemo* hkDemoDatabase::createDemo(int index, hkDemoEnvironment* env) const
{
	env->m_demoPath		= m_demos[index].m_demoPath;
	env->m_menuPath     = m_demos[index].m_menuPath;
	env->m_variantId	= m_demos[index].m_variantId;
	env->m_resourcePath	= m_demos[index].m_resourcePath;
	return (*m_demos[index].m_func)(env);
}

// Find and create a demo, otherwise return the error demo
hkDemo* hkDemoDatabase::createDemo(const char* name, hkDemoEnvironment* env) const
{
	int i = findDemo(name);
	if( i != -1 )
	{
		return createDemo(i, env);
	}

	return new ErrorDemo(env, name);
}



hkDemoEntryRegister* hkDemoDatabase::s_demoList;
HK_LOCAL_SINGLETON_IMPLEMENTATION(hkDemoDatabase);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
