/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#ifndef HK_DEMOFRAMEWORK_DEMO_H
#define HK_DEMOFRAMEWORK_DEMO_H

#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>
#include <Graphics/Common/Input/Keyboard/hkgKeyboard.h>
#include <Common/Base/Thread/Pool/hkThreadPool.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Demos/DemoCommon/DemoFramework/Callable.h>
#include <Common/Visualize/hkVirtualFramebufferServer.h>
#include <Common/Base/Container/StringMap/hkStorageStringMap.h>

struct hkDemoEntry;
class hkgWindow;
class hkgDisplayHandler;
class hkgSceneDataConverter;
class hkgDisplayWorld;
class hkgPad;
class hkPerformanceCounterUtility;
struct hkDemoFrameworkOptions;
class hkMonitorStreamAnalyzer;
struct hkTimerData;
class hkResource;
class hkgMovieRecorder;

#define DEMO_OPTIONS_DECLARE(OPTIONS) \
		virtual hkVariant* getOptions() { return &m_optionsVariant; } \
		static hkVariant m_optionsVariant; \
		static OPTIONS m_options

#define DEMO_OPTIONS_DEFINE(OWNER,OPTIONS) \
	extern const hkClass OWNER##OPTIONS##Class; \
	OWNER::OPTIONS OWNER::m_options; \
	hkVariant OWNER::m_optionsVariant = { &OWNER::m_options, &OWNER##OPTIONS##Class }

#define HK_DEMO_FRAMELOCK_MAX_FRAMES 10
class hkDemoEnvironment;

class hkDemoFrameTimer
{
public:
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkDemoFrameTimer);

	hkDemoFrameTimer()
		: m_curFrame(0), m_previousFpsLimit(-10000)
	{
		for (int i=0; i < HK_DEMO_FRAMELOCK_MAX_FRAMES; ++i)
		{
			m_frameTimes[i] = 0;
		}

		m_ticksSoFar.start();
		m_totalTicks.start();
	}

	void init( const hkDemoEnvironment& env );
	void updateAndWait( const hkDemoEnvironment& env );

	hkReal getElapsedSeconds() const { return m_totalTicks.getElapsedSeconds(); }
	hkReal getLastFrameTime() const { return (float)( (double)m_frameTimes[ ((m_curFrame-1) > 0) ? (m_curFrame-1) : (HK_DEMO_FRAMELOCK_MAX_FRAMES-1) ] / (double)hkStopwatch::getTicksPerSecond() ); }

protected:

	int m_frameTimes[HK_DEMO_FRAMELOCK_MAX_FRAMES];
	int m_curFrame;

	hkStopwatch m_ticksSoFar;
	hkStopwatch m_totalTicks;

	int m_previousFpsLimit;
	hkReal m_minSecondsFrame;
	hkUint64 m_minTicksPerFrame;

};

// Demo environment
class hkDemoEnvironment :
	public hkVirtualGamepadHandler,
	public hkVirtualKeyEventHandler,
	public hkVirtualMouseHandler,
	public hkVirtualFileDropHandler
{
	public:
		HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkDemoEnvironment);

		hkDemoEnvironment();

		// When mapping keys to gamepad buttons, the Ctrl key is used to determine which pad a key affects.
		// This function returns true if the Ctrl key needs to be down to affect m_gamePad (index 0)
		// or m_gamePadB (index 1).  This function is useful if a demo is handling additional
		// keys explicitly and wants to correlate them with the gamepads.  But it's better to
		// call wasKeyPressed or wasKeyReleased below which will automate the procedure.
		hkBool getDesiredKeyboardCtrlStateForGamePad( int padIndex=0 ) const;

		// You can find out if a key has been pressed or released when the ctrl key is in the state
		// associated with a particular gamepad.
		hkBool wasKeyPressed( int padIndex, HKG_KEYBOARD_VKEY key ) const;
		hkBool wasKeyReleased( int padIndex, HKG_KEYBOARD_VKEY key ) const;
		hkBool getKeyState( int padIndex, HKG_KEYBOARD_VKEY key ) const;

		// check for both game pads
		bool   wasButtonPressed( unsigned int key ) const;
		bool   isButtonPressed( unsigned int key ) const;

		// COM-305 : You can use reserve/release in your demos if you want to use the CTRL key (PC only)
		//	         for something different than switching game pads
		void reserveControlKey();
		void releaseControlKey();

		void checkInputDeterminism() const;

		// Environment
		hkgWindow*	m_window;

		// Default viewport, or for any viewport in which m_viewport has null data
		hkgDisplayHandler* m_displayHandler;
		hkgSceneDataConverter* m_sceneConverter;
		hkgDisplayWorld* m_displayWorld;
		hkTextDisplay* m_textDisplay;

		// You can specify completely separate worlds etc for each viewport, or just specific overrides (leave any null to use above default ones)
		struct ViewportData
		{
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkDemoEnvironment::ViewportData);
			ViewportData() : m_windowViewportIndex(0), m_displayHandler(HK_NULL), m_sceneConverter(HK_NULL), m_displayWorld(HK_NULL), m_textDisplay(HK_NULL) {}

			int m_windowViewportIndex;

			hkgDisplayHandler* m_displayHandler;
			hkgSceneDataConverter* m_sceneConverter;
			hkgDisplayWorld* m_displayWorld;
			hkTextDisplay* m_textDisplay;
		};
		extArray<ViewportData> m_viewportData;
		ViewportData* getViewportData( int windowViewportIndex ); // can return null, in which case use defaults

		hkgPad* m_gamePad;
		hkgPad* m_gamePadB;
		hkBool m_controlKeyReserved;
		hkBool m_mousePickingEnabled;
		hkBool m_shareCameraBetweenViewports;
		hkBool m_showWorldAxis; // Display world axis in the lower-left corner (default: false)
		hkBool m_inRenderLoop;

		hkgMovieRecorder* m_movieRecorder;

		hkDemoFrameworkOptions* m_options;
		hkDemoFrameTimer m_frameTimer;

		// Current demo
		extStringPtr m_menuPath;
		extStringPtr m_demoPath;
		extStringPtr m_resourcePath;

		int m_variantId;

		// Grows from 0 to hkDemoFrameworkOptions::m_numRepetitions - 1 with each consecutive execution of a demo.
		int m_repetitionIndex;

		int m_demoTypeFlags;
		int m_cpuMhz; // similar to x86 of this speed

		class hkVirtualFramebufferServer* m_virtualFrameBufferServer;

		enum ReportingLevel
		{
			REPORT_OFF		= 0,
			REPORT_ERROR	= 1,
			REPORT_ASSERT	= 2,
			REPORT_WARN		= 3,
			REPORT_INFO		= 4,
			REPORT_DEBUG	= 5,
			REPORT_ALL		= REPORT_DEBUG
		};

		ReportingLevel m_reportingLevel;	// Specifies the desired level of console output

		char m_exitCode;	// Exit Code to return from framework

		// Remote access
		virtual void onVirtualMouseUpdate( const hkVirtualMouse& m );
		virtual void onVirtualKeyEventUpdate( const hkVirtualKeyEvent& m );
		virtual void onVirtualGamepadUpdate( const hkVirtualGamepad& g );
		virtual void onVirtualFileDrop( const hkVirtualFileDrop& d );

};

// Base demo class
class hkDemo : public hkReferencedObject
{
	public:

		HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_DEMO);

		hkDemo(hkDemoEnvironment* env);
		~hkDemo();

		// Step result
		enum Result
		{
			DEMO_OK,
			DEMO_PAUSED,
			DEMO_STOP,
			DEMO_RESTART,
			DEMO_ERROR
		};

		// called after demo's constructor has been executed
		virtual void postConstruct() { ; }

		/// Called after the constructor. Responsible for creating the destruction world and loading the scene.
		virtual hkResult createScene()	{	return HK_SUCCESS;	}

		// called right before the demo is deleted
		virtual void preDeleteDemo() { ; }

		// Signal to the base system that the frame has changed.
		virtual void advanceFrame();

		// Run a single step of the demo
		virtual Result stepDemo();

		// Special signal flags that a demo
		enum PostStepActions
		{
			DEMO_PSA_NONE = 0x0,
			DEMO_PSA_CLEAR_TIMERS = 0x1, // Timers should be cleared (bootstrap-style demos)
		};

		virtual PostStepActions getPostStepActions () const;

		virtual void waitForStepCompletion() {}

		virtual void onStepCompleted() {}

		// Perform whatever extra operations independent of simulation being run/paused.
		virtual void preRenderDisplayWorld(class hkgViewport* view) {}
		virtual void postRenderDisplayWorld(class hkgViewport* view) {}
		virtual void postRenderWindow(class hkgWindow* window) {}

		virtual void renderSecondaryViewport(class hkgViewport* v) {}

		// Run single step the VDB, if one
		virtual Result stepVisualDebugger() = 0;
		virtual bool visualDebuggerEnabled() = 0;

		virtual void getNumTimerStreams( int& numThreadStreams, int& numSpuStreams, int maxThreads = 0x7fffffff ) const;
		virtual void getTimerStreamInfo( extArray<hkTimerData>& threadStreams, extArray<hkTimerData>& spuStreams, int maxThreads = 0x7fffffff );

		virtual void resetTimerStreams();
		virtual void addTimersToVdb( const extArray<hkTimerData>& threadStreams, const extArray<hkTimerData>& spuStreams ) {}

		// Which timer names to include in the running timer graphs
		virtual void getGraphedTimers( extArray<extStringPtr>& timerNames ) {}

		// What tweakable parameters does this demo have
		virtual hkVariant* getOptions() { return HK_NULL; }

		// Get any error text
		const char* getError() { return m_error; }

		virtual void mouseDown() {}	// Mouse has been pressed
		virtual void mouseUp() {}	// Mouse has been released
		virtual void mouseDrag() {}	// Mouse has been dragged

		virtual void windowResize(int w, int h) {} // window has resize. Handy to redo some 2d graphs etc
		virtual void windowDropFile(const char* filename, int x, int y) {} // window has a file dropped. The const char is temporary, strDup if you want to keep it.

		// Touch events (can fake with -faketouch at cmdline if you want to test with Left Mouse to touch)
		virtual bool touchDown(int id, int x, int y) { return false; }
		virtual void touchMove(int id, int x, int y, int dx, int dy) {}
		virtual void touchUp(int id, int x, int y) {}

		// Accelerometer changed (high pass and low pass filter of that raw data given)
		virtual void accelerometerChange(const float highPass[3], const float lowPass[3]) {}

		void setDemoName(const char* name);

			/// Recursively scan a directory for hkx files and create a demo for each found entry
		static void HK_CALL scanDirectoryAndCreateDemosForHavokFiles(const char* path, const char* ending, hkDemoEntry& thisEntry, hkBool recurseFolders, extArray<hkDemoEntry*>& entriesOut );

		// Called by the BootstrapDemo
		// Artificial input when running in bootstrap mode
		virtual void makeFakeInput() = 0;

		//
		// Regression information.
		// This is used by the BootstrapDemo to determine what data to output.
		// Most demos will output their average steptime by default. This is controlled by m_addAverageTimeRegression.
		//
		// Extra regression data for "one-off" information, e.g. build time or compression ratio, can be added by
		// calling addRegression().
		//
		// If addRegression() is called subsequent times with the same string parameter, the old value will be replaced.
		// This is useful for qunatities like average-frame timings.
		//
		struct RegressionInfo
		{
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkDemo::RegressionInfo);

			enum RegressionType
			{
					/// The average demo step time, used specifically for the BootstrapDemo
				REGRESSION_AVERAGE_STEP_TIME_MICROSECONDS,
				REGRESSION_MICROSECONDS,
				REGRESSION_SECONDS,
				REGRESSION_BYTES,
				REGRESSION_RATIO,
				REGRESSION_ITERATIONS,
			};

			const char* getUnitsString() const;

			extStringPtr m_name;
			hkEnum<RegressionType, hkUint8> m_type;
			hkReal m_value;

			bool operator < (const RegressionInfo& other) const { return hkString::strCmp( m_name.cString(), other.m_name.cString() ) < 0 ? true : false; }
		};

			/// Add a regression with the specified information.
			/// If a regression by that name already exists for this demo, it will be overwritten.
			/// The full regression name will have the demo's menupath prepended.
		virtual void addRegression( const char* regressionName, RegressionInfo::RegressionType regType, hkReal val );

			/// Add regression information for this demo.
			/// The default implementation adds an "average step time" regression, and appends m_additionalRegressions.
		virtual void getRegressionInfo( hkArray<RegressionInfo>& regressions, hkReal averageStepTime );

			/// Extra regression information.
		extArray<RegressionInfo> m_additionalRegressions;

			/// Whether or not to add a regression for the average step time. This should usually be left true, but can
			/// be disabled for demos that do no work during the stepDemo() method.
		hkBool m_addAverageTimeRegression; //+default(true)

			/// Used by addRegression to determine whether to add or replace, based on the name.
		hkStorageStringMap<int, extContainerAllocator> m_regressionNameToIndexMap;


		// Our current demo environment
		hkDemoEnvironment* m_env;

		// Number of times to the demo will be stepped by the bootstrapper
		int m_bootstrapIterations; // +default(100)

		typedef Callable0<int> KeyPressCallback;
			/// Bind a vkey to a callback.
			/// If the description is null, the key will not be listed by menudemo.
			/// Note that vkey codes are not exactly ascii codes. In particular, lower case
			/// ascii values do not correspond to vkeys.
		int bindKeyPressed(int vkey, const char* descr, const KeyPressCallback& cb );
		int bindKeyHeld(int vkey, const char* descr, const KeyPressCallback& cb );
		int bindKeyReleased(int vkey, const char* descr, const KeyPressCallback& cb );

	protected:

		extStringPtr m_error;
		extStringPtr m_name;

	public:

		struct KeyPressCallbackInfo
		{
			HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE,hkDemo::KeyPressCallbackInfo);
			KeyPressCallbackInfo() : description(HK_NULL), vkey(0)/*, cb(0,0)*/ {}
			const char* description;
			int vkey;
			KeyPressCallback cb;
		};

		const hkArrayBase<KeyPressCallbackInfo>& getKeyPressCallbackInfo() const { return m_keyPressCallbacks; }
		const hkArrayBase<KeyPressCallbackInfo>& getKeyHeldCallbackInfo() const { return m_keyHeldCallbacks; }
		const hkArrayBase<KeyPressCallbackInfo>& getKeyReleasedCallbackInfo() const { return m_keyReleaseCallbacks; }

	protected:

		extArray<KeyPressCallbackInfo> m_keyPressCallbacks;
		extArray<KeyPressCallbackInfo> m_keyHeldCallbacks;
		extArray<KeyPressCallbackInfo> m_keyReleaseCallbacks;

};

//
// A few utility functions for binding to keys
// Usage:
//	 bindKeyPressed(HKG_VKEY_F5, "Toggle testBool", KeyPressCallback::BoundFunction(&hkToggleBooleanValue, &m_testBool) );
//

// b = !b
int hkToggleBooleanValue( hkBool* b );

// i++
int hkIncrementIntegerValue( int* i );

// i--
int hkDecrementIntegerValue( int* i );

// i = max(i-1, 0)
int hkDecrementIntegerValueNonNegative( int* i );

// i = max(i-1, 1)
int hkDecrementPositiveIntegerValue( int* i );

// Error demo
class ErrorDemo : public hkDemo
{
	public:
		HK_DECLARE_CLASS_ALLOCATOR(HK_MEMORY_CLASS_BASE);
		ErrorDemo(hkDemoEnvironment* env, const char* failed)
			: hkDemo(env), m_failed(failed), m_iterations(1000)
		{
		}

		Result stepDemo()
		{
			hkStringBuf os;
			os.printf("Creation of '%s' demo failed. Was it registered?", m_failed);
			m_env->m_textDisplay->outputText(os, 20, 20);
			return --m_iterations >= 0 ? DEMO_OK : DEMO_STOP;
		}

		Result stepVisualDebugger()
		{
			return DEMO_OK;
		}

		void makeFakeInput() {}

		bool visualDebuggerEnabled() { return false; }

		const char* m_failed;
		int m_iterations;
};

#endif // HK_DEMOFRAMEWORK_DEMO_H

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
