/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */


#include <Demos/demos.h>
#include <Demos/DemoCommon/DemoFramework/hkDefaultPhysics2012Demo.h>

#include <Common/Base/KeyCode.h>

#include <Physics2012/Utilities/Actions/MouseSpring/hkpMouseSpringAction.h>

#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/System/hkBaseSystem.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>

#include <Common/Base/DebugUtil/MemoryExceptionTestingUtil/hkMemoryExceptionTestingUtil.h>
#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>

#include <Physics2012/Dynamics/World/Simulation/hkpSimulation.h>

#include <Graphics/Common/hkGraphics.h>
#include <Graphics/Common/Input/Pad/hkgPad.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/Camera/hkgCamera.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Common/DisplayObject/hkgInstancedDisplayObject.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

#include <Common/Serialize/Util/hkBuiltinTypeRegistry.h>
#include <Common/Base/Reflection/Registry/hkVtableClassRegistry.h>

#include <Common/SceneData/Graph/hkxNode.h>
#include <Common/SceneData/Mesh/hkxMesh.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Attributes/hkxAttributeGroup.h>

#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Bridge/System/hkgSystem.h>
#include <Graphics/Bridge/SeriesGraph/hkgSeriesGraph.h>

#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpBroadphaseViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpConvexRadiusViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpActiveContactPointViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpInactiveContactPointViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpToiContactPointViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpToiCountViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpWeldingViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpConstraintViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpSweptTransformDisplayViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpRigidBodyCentreOfMassViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpRigidBodyInertiaViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpSimulationIslandViewer.h>
//#include <Physics2012/Utilities/VisualDebugger/Viewer/Vehicle/hkpVehicleViewer.h>

#include <Physics2012/Utilities/VisualDebugger/Viewer/Collide/hkpShapeDisplayViewer.h>
#include <Physics2012/Utilities/VisualDebugger/Viewer/Dynamics/hkpPhantomDisplayViewer.h>

#include <Physics2012/Utilities/VisualDebugger/hkpPhysicsContext.h>
#include <Physics2012/Utilities/Dynamics/TimeSteppers/hkpAsynchronousTimestepper.h>

#include <Common/Visualize/Process/hkDebugDisplayProcess.h>

#include <Common/Visualize/hkProcess.h>
#include <Common/Visualize/hkProcessRegisterUtil.h>
#include <Common/Visualize/hkVisualDebugger.h>
#include <Common/Visualize/hkProcessFactory.h>

#include <Physics2012/Dynamics/hkpDynamics.h>
#include <Physics2012/Dynamics/World/hkpPhysicsSystem.h>
#include <Physics2012/Dynamics/World/hkpSimulationIsland.h>

#include <Physics2012/Collide/Agent/hkpProcessCollisionInput.h>

#include <Common/Base/Thread/JobQueue/hkJobQueue.h>

#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/Menu/MenuDemo.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>

#include <Common/Base/Thread/Pool/hkCpuThreadPool.h>
#include <Physics2012/Utilities/Serialize/hkpHavokSnapshot.h>


#	include <Common/Base/Fwd/hkwindows.h>

#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamAnalyzer.h>



class hkDemoEnvironment;

DEMO_OPTIONS_DEFINE(hkDefaultPhysics2012Demo, PhysicsOptions);

static const char* solverMemoryGraphName = "Solver memory in use";


hkDefaultPhysics2012Demo::hkDefaultPhysics2012Demo(hkDemoEnvironment* env, DemoFlags flags)
	:	hkDefaultDemo(env),
		m_mouseSpring(HK_NULL),
		m_mouseSpringMaxRelativeForce(1000.0f),
		m_physicsViewersContext(HK_NULL),
		m_displayToiInformation(false),
		m_world(HK_NULL),
		m_flags(flags)
{
	hkReferencedObject::lockAll();
	m_physicsStepCounter = 0;
	m_raytraceViewport = 0;

	m_splash = HK_NULL;
	hkBool displaySplash = (hkString::strStr(env->m_demoPath, "Api" ) == HK_NULL);
	if (displaySplash && !(flags & DEMO_FLAGS_NO_SPLASH_SCREEN))
	{
		m_splash = new SplashScreen(this, SplashScreen::PHYSICS_2012_SPLASH, true);
		m_splash->setMessage("Initializing");
	}


	// Disable warnings that commonly occur in the demos
	hkError::getInstance().setEnabled(0x6e8d163b, false); // hkpMoppUtility.cpp:18
	hkError::getInstance().setEnabled(0x34df5494, false); //hkGeometryUtility.cpp:26
	hkError::getInstance().setEnabled(0xf013323d, false); //hkBroadphaseBorder

	m_oldForceMultithreadedSimulation = hkpWorld::m_forceMultithreadedSimulation;
	hkpWorld::m_forceMultithreadedSimulation = env->m_options->m_forceMT;

	m_physicsViewersContext = new hkpPhysicsContext;
	if ( m_env->m_window )
	{
		hkpPhysicsContext::registerAllPhysicsProcesses(); // all physics only ones
	}

	// Most physics demos look good with shadows

	m_forcedFPSChange = false;
	m_simulationStarted = false;

	if (m_env->m_window)
	{
		setupLights(m_env); // after we decide if we want shadows
	}

	// Which timers to graph by default
	{
		m_graphedTimerNames.pushBack( "Physics 2012" );
		m_graphedTimerNames.pushBack( "BroadPhase" );
		m_graphedTimerNames.pushBack( "NarrowPhase" );
		m_graphedTimerNames.pushBack( "Integrate" );
		//m_graphedTimerNames.pushBack( "TOIs" );
	}

	// Add a memory graph for solver buffer use
	{
		m_solverMemoryGraph = new hkgSeriesGraph();
		m_solverMemoryGraph->setFormat( hkgSeriesGraph::SI_PREFIX, "B" );
		m_solverMemoryGraph->addSeries( solverMemoryGraphName );

		MenuDemo::getMenuDemo()->m_graphManager->addMemoryGraph( m_solverMemoryGraph );
	}

	hkReferencedObject::unlockAll();

#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
	hkpWorld::registerWithJobQueue( m_jobQueue );
#endif
}


void hkDefaultPhysics2012Demo::setAutoInstancingEnabled( bool on )
{
	hkpShapeDisplayViewer* viewer = getLocalViewer<hkpShapeDisplayViewer>();
	if (viewer)
	{
		viewer->setInstancingEnabled( on );
	}
}

void hkDefaultPhysics2012Demo::setAutoDisplayCachingEnabled( bool on )
{
	hkpShapeDisplayViewer* viewer = getLocalViewer<hkpShapeDisplayViewer>();
	if (viewer)
	{
		viewer->setDisplayBodyCachingEnabled( on );
	}
	if (m_env->m_displayHandler)
	{
		m_env->m_displayHandler->setDisplayBodyCachingEnabled( on );
	}
}

void hkDefaultPhysics2012Demo::destroyWorld()
{
	if ( ( m_world != HK_NULL ) && ( m_env->m_options->m_renderParallelWithSimulation && m_simulationStarted ) )
	{
		m_world->finishMtStep( );
	}
	hkReferencedObject::lockAll();
	if(m_world)
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}

	hkReferencedObject::unlockAll();
}

hkDefaultPhysics2012Demo::~hkDefaultPhysics2012Demo()
{
	if( m_mouseSpring != HK_NULL)
	{
		hkpWorld* mouseWorld = m_mouseSpring->getWorld();
		if ( mouseWorld )
		{
			mouseWorld->lock();
			mouseWorld->removeAction( m_mouseSpring );
			mouseWorld->unlock();
		}

		m_mouseSpring->removeReference();
	}

	destroyWorld();

	hkReferencedObject::lockAll();

	shutdownVDB();

	if (m_physicsViewersContext)
	{
		m_physicsViewersContext->removeReference();
		m_physicsViewersContext = HK_NULL;
	}

	hkReferencedObject::unlockAll();

	if (m_forcedFPSChange)
	{
		m_env->m_options->m_lockFps = m_oldFPSLock;
	}

	MenuDemo::getMenuDemo()->m_graphManager->removeMemoryGraph(m_solverMemoryGraph);
	m_solverMemoryGraph->removeReference();

	delete m_splash;

	hkpWorld::m_forceMultithreadedSimulation = m_oldForceMultithreadedSimulation;
}



void hkDefaultPhysics2012Demo::addOrRemoveThreads()
{

	hkDefaultDemo::addOrRemoveThreads();

}



void hkDefaultPhysics2012Demo::setupContexts(hkArray<hkProcessContext*>& contexts)
{
	if ( m_world )
	{
		m_world->markForWrite();
		if ( (m_physicsViewersContext->findWorld(m_world) < 0) )
		{
			m_physicsViewersContext->addWorld(m_world);
		}
		m_world->unmarkForWrite();
	}
	contexts.pushBack( m_physicsViewersContext );

	// Add viewers to the demo display.
	// Uncomment these to use them (and uncomment the #includes above)
	//m_debugViewerNames.pushBack( hkpBroadphaseViewer::getName()  );
	//  m_debugViewerNames.pushBack( hkpConstraintViewer::getName()  );
	// m_debugViewerNames.pushBack( hkpActiveContactPointViewer::getName()  );
	//	m_debugViewerNames.pushBack( hkpInactiveContactPointViewer::getName()  );
	//	m_debugViewerNames.pushBack( hkpRigidBodyCentreOfMassViewer::getName()  );
	//	m_debugViewerNames.pushBack( hkpRigidBodyInertiaViewer::getName()  );
	// m_debugViewerNames.pushBack( hkpSimulationIslandViewer::getName()  );
	//	m_debugViewerNames.pushBack( hkpVehicleViewer::getName()  );
	//	m_debugViewerNames.pushBack( hkStatisticsProcess::getName()  );
	//	m_debugViewerNames.pushBack( hkpSweptTransformDisplayViewer::getName() );
	//	m_debugViewerNames.pushBack( hkpToiContactPointViewer::getName()  );
	//  m_debugViewerNames.pushBack( hkpToiCountViewer::getName()  );

	// A viewer to look at the collision radius of convex objects.
	// m_debugViewerNames.pushBack( hkpConvexRadiusViewer::getName() );

	// These are the three "default" viewers which display shapes,
	// phantoms and debug lines.

	if (1)// BETTER TO USE (but affects historical regression):   kgSystem::g_RendererType > hkgSystem::HKG_RENDERER_NULL )
	{
		m_debugViewerNames.pushBack( hkpShapeDisplayViewer::getName() );
		m_debugViewerNames.pushBack( hkpPhantomDisplayViewer::getName() );
		m_debugViewerNames.pushBack( hkDebugDisplayProcess::getName() );
	}

	// register all our classes we know about with the vdb for tweaking
	if (m_vdbClassReg)
	{
		m_vdbClassReg->registerList(hkBuiltinTypeRegistry::StaticLinkedTypeInfos, hkBuiltinTypeRegistry::StaticLinkedClasses);
	}
}



void hkDefaultPhysics2012Demo::waitForStepCompletion()
{
	if ( m_env->m_options->m_renderParallelWithSimulation )
	{
		if ( !m_simulationStarted )
		{
			return;
		}

		m_simulationStarted = false;
		if(	m_world && m_world->m_simulationType == hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED &&	m_threadPool != HK_NULL )
		{
			m_world->checkUnmarked();
			m_jobQueue->processAllJobs();
			m_threadPool->waitForCompletion();
			m_world->finishMtStep( m_jobQueue, m_threadPool );
		}
	}
}

void hkDefaultPhysics2012Demo::startNewSimulateStep()
{
	m_simulationStarted = true;
	if(	m_world )
	{
		if ( m_world->m_simulationType == hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED && m_threadPool != HK_NULL )
		{
			m_world->checkUnmarked();
			m_world->initMtStep( m_jobQueue, m_timestep );
			m_threadPool->processJobQueue( m_jobQueue );
		}
		else
		{
			m_world->stepDeltaTime(m_timestep);
		}
	}
}

hkDemo::Result hkDefaultPhysics2012Demo::stepDemo()
{
	if(m_world)
	{
		m_world->checkUnmarked();

		// Feed solver memory graph
		if( m_solverMemoryGraph && MenuDemo::getMenuDemo()->m_graphManager->isEnabled(GraphManager::MEMORY) )
		{
			m_world->markForWrite();
			hkWorldMemoryAvailableWatchDog::MemUsageInfo memInfo;
			m_world->calcRequiredSolverBufferSize( memInfo );
			m_world->unmarkForWrite();

			m_solverMemoryGraph->setNumChannels(1);
			m_solverMemoryGraph->addData( solverMemoryGraphName, 0, (hkReal)memInfo.m_maxRuntimeBlockSize );
			m_solverMemoryGraph->postProcess();
		}

		//
		// Step the world
		//

#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
		if ( m_world->m_simulationType == hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED )
		{
			if ( m_env->m_options->m_renderParallelWithSimulation )
			{
				Result res = hkDefaultDemo::stepDemo();
				if (res == DEMO_OK)
				{
					startNewSimulateStep();
				}
				m_physicsStepCounter++;
				addOrRemoveThreads();
				return res;
			}
			else
			{
				hkMemoryExceptionTestingUtil::startFrame();
				hkCheckDeterminismUtil::workerThreadStartFrame(true);

				HK_ON_DEBUG(hkpStepResult stepResult =) m_world->stepMultithreaded( m_jobQueue, m_threadPool, m_timestep );
				HK_ASSERT2( 0xf0236545, stepResult == HK_STEP_RESULT_SUCCESS, "Your physics simulation ran out of memory" );

				hkCheckDeterminismUtil::workerThreadFinishFrame();
				hkMemoryExceptionTestingUtil::endFrame();

			}
		}
		else
#endif
		{
			hkMemoryExceptionTestingUtil::startFrame();
			hkCheckDeterminismUtil::workerThreadStartFrame(true);

			m_world->stepDeltaTime(m_timestep);

			hkCheckDeterminismUtil::workerThreadFinishFrame();
			hkMemoryExceptionTestingUtil::endFrame();
		}

		//	updateDisplay( m_world );
		
		//
		// Save snapshot
		//

		hkVariant* optionsVar = getOptions();
		if( m_world && optionsVar->m_class && hkDefaultPhysics2012DemoPhysicsOptionsClass.isSuperClass(*optionsVar->m_class) )
		{
			PhysicsOptions* options = static_cast<PhysicsOptions*>(optionsVar->m_object);
			if( options->m_saveWorldSnapshot )
			{
				options->m_saveWorldSnapshot = false;
				hkpHavokSnapshot::save(m_world, hkOstream("PhysicsSnapshot.hkt").getStreamWriter() );
			}
		}
	}

	m_physicsStepCounter++;
	return hkDefaultDemo::stepDemo();
}

hkpStepResult hkDefaultPhysics2012Demo::stepAsynchronously(hkpWorld* world, hkReal frameDeltaTime, hkReal physicsDeltaTime)
{

	hkpStepResult result;

	if (  m_world->m_simulationType == hkpWorldCinfo::SIMULATION_TYPE_MULTITHREADED )
	{

		world->setFrameTimeMarker( frameDeltaTime );

		world->advanceTime();
		while ( !world->isSimulationAtMarker() )
		{
			HK_ASSERT( 0x11179564, world->isSimulationAtPsi() );

			{
				// Interact from game to physics
			}

			hkCheckDeterminismUtil::workerThreadStartFrame(true);

			m_world->initMtStep( m_jobQueue, m_timestep );

			m_threadPool->processJobQueue( m_jobQueue );
			m_jobQueue->processAllJobs( );

			m_threadPool->waitForCompletion();

			m_world->finishMtStep( m_jobQueue, m_threadPool );

			hkCheckDeterminismUtil::workerThreadFinishFrame();
		}

		result = HK_STEP_RESULT_SUCCESS;
	}
	else
	{
		hkAsynchronousTimestepper::stepAsynchronously(world, frameDeltaTime, physicsDeltaTime);
		result = HK_STEP_RESULT_SUCCESS;
	}

	return result;
}


void hkDefaultPhysics2012Demo::preDeleteDemo()
{
}


hkBool hkDefaultPhysics2012Demo::objectPicked( const hkgDisplayObject* displayObject, const hkVector4& worldPosition, const hkgViewportPickData* pd )
{
	HK_ASSERT(0x65b2643b, m_env->m_displayHandler);
	hkgDisplayHandler* dhandler = m_env->m_displayHandler;

	if( displayObject && m_world )
	{
		hkUlong id = 0;
		void* userPtr = HK_NULL;
		if (displayObject->getStatusFlags() & HKG_DISPLAY_OBJECT_INSTANCED)
		{
			id = ((const hkgInstancedDisplayObject*)displayObject)->getUserDataFromIndex( pd->getClosestGeometryIndex() );
		}
		else
		{
			id = dhandler->getDisplayObjectId( displayObject );
			userPtr = displayObject->getUserPointer();
		}

		// HACK!  We know the id is actually the address of the Collidable
		if ( (id > 0) || (userPtr != HK_NULL) )
		{
			hkpRigidBody* rb = HK_NULL;
			// Check if doLink->m_id is a valid address.
			if (id > 0)
			{
				if ( (id & 0x03) == 0x03) // 0x1 == swept transform from, 0x2 = swept transform to, 0x3 = convex radius (ok to pick)
				{
					id = hkClearBits(id, 0x03);
				}

				if (id % 4 != 0 || id < 64)
				{
					return false;
				}

				hkpCollidable* col = reinterpret_cast<hkpCollidable*>( id );
				if( col->getOwner() )
				{
					rb = hkpGetRigidBody(col);
				}
			}
			else
			{
				rb = reinterpret_cast<hkpRigidBody*>( userPtr ); // true in our demos that use assets
			}

			if( rb && !rb->isFixed() )
			{
				hkpWorld* mouseWorld = rb->getWorld();
				if (mouseWorld)
				{
					hkVector4 positionAinA;
					positionAinA.setTransformedInversePos( rb->getTransform(), worldPosition );

					const hkReal springDamping = 0.5f;
					const hkReal springElasticity = 0.3f;
					const hkReal objectDamping = 0.95f;

					mouseWorld->lock();
					{
						m_mouseSpring = new hkpMouseSpringAction( positionAinA, worldPosition, springDamping, springElasticity, objectDamping, rb, &m_mouseSpringAppliedCallbacks );
						m_mouseSpring->setMaxRelativeForce(m_mouseSpringMaxRelativeForce);

						mouseWorld->addAction( m_mouseSpring );
					}
					mouseWorld->unlock();
				}

				return true;
			}
		}
	}
	return false;
}

void hkDefaultPhysics2012Demo::objectDragged( const hkVector4& newWorldSpacePoint )
{
	// has mousespring been deleted when entity was deleted?
	if( m_mouseSpring != HK_NULL )
	{
		m_world->lock();
		m_mouseSpring->setMousePosition( newWorldSpacePoint );
		m_world->unlock();
	}
}

void hkDefaultPhysics2012Demo::objectReleased()
{
	if( m_mouseSpring != HK_NULL)
	{
		hkpWorld* mouseWorld = m_mouseSpring->getWorld();
		if ( mouseWorld )
		{
			mouseWorld->lock();
			{
				mouseWorld->removeAction( m_mouseSpring );
				hkpRigidBody* body = static_cast<hkpRigidBody*>(m_mouseSpring->getEntity());
				body->activate();
			}
		}

		m_mouseSpring->removeReference();

		if (mouseWorld)
		{
			mouseWorld->unlock();
		}

		m_mouseSpring = HK_NULL;
	}
}


static void HK_CALL displayTimePoint( hkgDisplayContext* displayContext, hkTime time, hkReal physicsDeltaTime, hkReal scale )
{
	hkReal angle = 2.0f * HK_REAL_PI * scale * time / physicsDeltaTime;

	hkReal sx = hkMath::sin( angle );
	hkReal cx = hkMath::cos( angle );

	hkReal size = 40.0f;

	hkReal rs = size;
	hkReal re = size * 0.5f;

	// ortho view in window is 0..-2
	hkVector4 start( rs * sx, rs * cx, -0.001f);
	hkVector4 end  ( re * sx, re * cx, -0.001f);

	hkVector4 offset(size + 10.0f, size + 10.0f, 0.0f);
	start.add4( offset );
	end  .add4( offset );

	displayContext->setCurrentPosition( &start(0) );
	displayContext->setCurrentPosition( &end(0) );

	//HK_DISPLAY_LINE( start, end, color );
}


static void HK_CALL displayTimeWheel(hkDemoEnvironment* env, hkpWorld* world, hkReal physicsDeltaTime)
{
	hkgViewport* origView = env->m_window->getCurrentViewport();
	hkgViewport* orthoView = env->m_window->getWindowOrthoView();

	// Set as the current view
	hkgDisplayContext* context = env->m_window->getContext();
	context->lock();
	orthoView->setAsCurrent( context );

	context->setDepthReadState(false);
	context->setDepthWriteState(true);
	context->setTexture2DState(false);

	//
	//	Display a nice time bar to see the physics steps
	//

	{
		context->beginGroup( HKG_IMM_LINES );

		context->setCurrentColorPacked((unsigned int)hkColor::BLUE);// reinterpret_cast<float*>(&color) );
		for (int i = 0; i < 8; i++)
		{
			displayTimePoint( context, hkTime(i* physicsDeltaTime), physicsDeltaTime, 0.125f );
		}

		context->setCurrentColorPacked((unsigned int)hkColor::GREEN);// reinterpret_cast<float*>(&color) );

		hkTime time = world->m_simulation->getSimulateUntilTime();
		displayTimePoint( context, time, physicsDeltaTime, 0.125f );

		context->endGroup();
	}

	origView->setAsCurrent( context );
	context->unlock();
}

void hkDefaultPhysics2012Demo::updateDisplay(hkpWorld* w)
{
	// XX Should only do this if we have rbs that need to be
	// update displayObjects from Asset files (no demos currently)
	if (0)
	{
		w->markForRead();

		const hkArray<hkpSimulationIsland*>& activeIslands = w->getActiveSimulationIslands();
		for(int i = 0; i < activeIslands.getSize(); i++)
		{
			const hkArray<hkpEntity*>& activeEntities = activeIslands[i]->getEntities();
			for(int j = 0; j < activeEntities.getSize(); j++)
			{
				hkpRigidBody* rigidBody = static_cast<hkpRigidBody*>(activeEntities[j]);
				hkpPropertyValue p = rigidBody->getProperty(HK_PROPERTY_DISPLAY_PTR);
				hkgDisplayObject* dispObj = reinterpret_cast<hkgDisplayObject*>( p.getPtr() );
				if (dispObj)
				{
					hkMatrix4 m;
					hkTransform transform;
					rigidBody->approxCurrentTransform( transform );
					m.set( transform );

					dispObj->setTransform((float*)&m);
				}
			}
		}

		w->unmarkForRead();
	}
}

void hkDefaultPhysics2012Demo::postRenderDisplayWorld(hkgViewport* )
{
	if (m_displayToiInformation)
	{
		hkReal physicsDeltaTime = m_world->m_simulation->getPhysicsDeltaTime();
		if ( hkMath::equal( physicsDeltaTime, 0.0f) )
		{
			physicsDeltaTime = 1.0f;
		}
		displayTimeWheel(m_env, m_world, physicsDeltaTime);
	}


}

void hkDefaultPhysics2012Demo::enableDisplayingToiInformation(hkBool enable)
{
	m_displayToiInformation = enable;
}

static hkxNode* _findMeshNodeByName(hkxNode* node, const char* name)
{
	if (!node) return HK_NULL;

	if (node->m_name && node->m_object.getClass())
	{
		if (hkString::strCmp( node->m_name, name )==0)
		{
			// have a node called the correct thing, do we have a mesh?
			if ( node->m_object && (hkString::strCmp( node->m_object.getClass()->getName(), hkxMeshClass.getName()) ==0) )
			{
				return node;
			}
		}
	}
	for(int ni=0; ni < node->m_children.getSize(); ++ni)
	{
		hkxNode* m = _findMeshNodeByName( node->m_children[ni], name );
		if (m)
			return m;
	}
	return HK_NULL;
}

static hkgDisplayObject* _findDisplayObject( hkxNode* node, extArray<hkgAssetConverter::NodeMapping>& mappings)
{
	for (int mi=0; mi < mappings.getSize(); ++mi)
	{
		if (mappings[mi].m_origNode == node)
			return (hkgDisplayObject*)( mappings[mi].m_hkgObject );
	}
	return HK_NULL;
}

static bool _isProxy( hkxNode* node )
{
	bool isRb= false;
	bool isShape = false;
	for (int gi=0; gi < node->m_attributeGroups.getSize(); ++gi)
	{
		hkxAttributeGroup* group = &node->m_attributeGroups[gi];
		if ( group->m_attributes.getSize() > 0 )
		{
			if (hkString::strCmp( group->m_name, "hkRigidBody") == 0)
			{
				isRb = true;
			}
			else if (hkString::strCmp( group->m_name, "hkShape") == 0)
			{
				isShape = true;
			}
		}
	}

	return isRb && !isShape;
}


static void _hidePhysicsNodes(hkxNode* node, extArray<hkgAssetConverter::NodeMapping>& nodes, hkgDisplayWorld* world)
{
	// See if it is in itself an rb or not. If it is then return.
	int gi;
	for (gi=0; gi < node->m_attributeGroups.getSize(); ++gi)
	{
		hkxAttributeGroup* group = &node->m_attributeGroups[gi];
		if ( (group->m_attributes.getSize() > 0) && (hkString::strCmp( group->m_name, "hkRigidBody") == 0) )
			return; // it is an rb, not a shape or ramdom mesh
	}

	if ( hkString::strCmp( node->m_object.getClass()->getName(), hkxMeshClass.getName()) == 0 )
	{
		hkgDisplayObject* toBeRemoved = _findDisplayObject( node, nodes );
		if (toBeRemoved)
		{
			if (world->removeDisplayObject(toBeRemoved) != HK_NULL)
			{
				toBeRemoved->removeReference(); // world ref
			}
		}
	}

	// recurse
	for (int ci=0; ci < node->m_children.getSize(); ++ci)
	{
		_hidePhysicsNodes(node->m_children[ci], nodes, world);
	}
}

static void _mergeCompoundNodes(hkgDisplayObject* dispObj, hkxNode* node, extArray<hkgAssetConverter::NodeMapping>& nodes, const hkArray<hkpRigidBody*>& rbs )
{
	// See if it is in itself an rb or not. If it is then return.
	int gi;
	for (gi=0; gi < node->m_attributeGroups.getSize(); ++gi)
	{
		hkxAttributeGroup* group = &node->m_attributeGroups[gi];
		if ( (group->m_attributes.getSize() > 0) && (hkString::strCmp( group->m_name, "hkRigidBody") == 0) )
			return; // it is an rb, not a shape or ramdom mesh
	}

	// could be an old style export scene, so will not have attributes
	// check for rbs of the same name..
	if (node->m_name)
	{
		for (int rbl = 0; rbl < rbs.getSize(); ++rbl)
		{
			if (rbs[rbl]->getName() && (hkString::strCmp( rbs[rbl]->getName(), node->m_name) == 0) )
				return; // rb by the same name as this node so can assume it is this one
		}
	}

	// merge the mesh (if it is a mesh) into the rb, transforming it as required.
	if ( node->m_object && hkString::strCmp( node->m_object.getClass()->getName(), hkxMeshClass.getName()) == 0 )
	{
		hkgDisplayObject* toBeMerged = _findDisplayObject( node, nodes);

		const float* tA = toBeMerged->getTransform();
		const float* tB = dispObj->getTransform();
		float bInv[16];
		float geomT[16];
		hkgMat4Invert(bInv, tB);
		hkgMat4Mult(geomT, bInv, tA);

		int numGeom = toBeMerged->getNumGeometry();
		for (gi=numGeom-1; gi >= 0; --gi)
		{
			hkgGeometry* geom = toBeMerged->removeGeometry(gi);
			geom->transform(geomT); // bake in relative transform
			dispObj->addGeometry(geom);
			geom->removeReference(); // remove ref given back by previous remove
		}
	}

	// recurse
	for (int ci=0; ci < node->m_children.getSize(); ++ci)
	{
		_mergeCompoundNodes(dispObj, node->m_children[ci], nodes, rbs );
	}
}

void hkDefaultPhysics2012Demo::loadAndAddRigidBodies( hkLoader* loader, const char* filename )
{
	//hkStringBuf assetFile("Resources/Physics2012/levels/test_level.hkx"); hkAssetManagementUtil::getFilePath(assetFile);
	hkStringBuf assetFile(filename); hkAssetManagementUtil::getFilePath(assetFile);
	//hkStringBuf assetFile("Resources/Physics2012/levels/onemesh_test_level.hkx"); hkAssetManagementUtil::getFilePath(assetFile);
	hkRootLevelContainer* container = loader->load( assetFile.cString() );
	HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
	hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));

	HK_ASSERT2(0x27343635, scene, "No scene loaded");
	m_env->m_sceneConverter->convert( scene, hkgAssetConverter::CONVERT_ALL );

	hkpPhysicsData* physics = reinterpret_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ));
	HK_ASSERT2(0x27343635, physics, "No physics loaded");

	// Physics
	if (physics)
	{
		const hkArray<hkpPhysicsSystem*>& psys = physics->getPhysicsSystems();

		// Tie the two together
		for (int i=0; i<psys.getSize(); i++)
		{
			hkpPhysicsSystem* system = psys[i];

			// Associate the display and physics (by name)
			if (scene)
			{
				hkDefaultPhysics2012Demo::addPrecreatedDisplayObjectsByName( psys[i]->getRigidBodies(), scene );
			}
			// add the lot to the world
			m_world->addPhysicsSystem(system);
		}
	}
}

int hkDefaultPhysics2012Demo::addPrecreatedDisplayObjectsByName( const hkArray<hkpRigidBody*>& rbs, const hkxScene* scene, hkArray<hkgDisplayObject*>* createdDisplayObjects, bool castShadows )
{
	return addPrecreatedDisplayObjectsByNameInternal(
		m_env,
		m_world,
		rbs,
		scene,
		createdDisplayObjects,
		castShadows );
}

int hkDefaultPhysics2012Demo::addPrecreatedDisplayObjectsByNameInternal(
	hkDemoEnvironment* env,
	const hkpWorld* rbsWorld,
	const hkArray<class hkpRigidBody*>& rbs,
	const class hkxScene* scene,
	hkArray<hkgDisplayObject*>* createdDisplayObjects,
	bool castShadows )
{
	int numConverted = 0;
//	hkxScene* scene = (hkxScene*) ontainer->findObjectByType( hkxSceneClass.getName() );

	env->m_displayWorld->lock();
	if (createdDisplayObjects)
	{
		createdDisplayObjects->setSize(rbs.getSize(), HK_NULL);
	}
	for (int rbi=0; rbi < rbs.getSize(); ++rbi)
	{
		const char* name = rbs[rbi]->getName();
		hkxNode* meshNode = _findMeshNodeByName(scene->m_rootNode, name);
		hkgDisplayObject* dispObj = _findDisplayObject( meshNode, env->m_sceneConverter->m_meshNodes );

		if (dispObj)
		{
			rbsWorld->markForRead();
			hkUlong id = (hkUlong)( rbs[rbi]->getCollidable() );
			rbsWorld->unmarkForRead();

			// set flag before adding to help internal state (esp for shadow casters)
			if ( !rbs[rbi]->isFixed() )
			{
				int idx = env->m_displayWorld->findDisplayObject(dispObj);
				if (idx >= 0)
				{
					env->m_window->getContext()->lock();

					bool isProxy = _isProxy( meshNode );
					if (isProxy) // just hide physics children
					{
						for (int ci=0; ci < meshNode->m_children.getSize(); ++ci)
						{
							_hidePhysicsNodes( meshNode->m_children[ci], env->m_sceneConverter->m_meshNodes, env->m_displayWorld );
						}
					}
					else // may be compund
					{
						for (int ci=0; ci < meshNode->m_children.getSize(); ++ci)
						{
							_mergeCompoundNodes(dispObj, meshNode->m_children[ci], env->m_sceneConverter->m_meshNodes, rbs );
						}
					}

					bool removedObj = false;
					if (castShadows || !isProxy)
					{
						HK_ON_DEBUG(hkgDisplayObject* ret = ) env->m_displayWorld->removeDisplayObject( dispObj );// gives back ref
						HK_ASSERT(0x0, ret);

						removedObj = true;
						if ( castShadows )
						{
							dispObj->setStatusFlags( dispObj->getStatusFlags() | HKG_DISPLAY_OBJECT_DYNAMIC | HKG_DISPLAY_OBJECT_SHADOW );
						}
					}

					dispObj->computeAABB();

					if (removedObj)
					{
						env->m_displayWorld->addDisplayObject(dispObj);
						dispObj->release();
					}

					env->m_window->getContext()->unlock();

				}
			}

			if (createdDisplayObjects)
			{
				(*createdDisplayObjects)[rbi] = dispObj;
			}

			env->m_displayHandler->addPrecreatedDisplayObject( id, dispObj );
			numConverted++;
		}
	}

	// due to merging, we may be left with some empty display objects at this stage, so we can remove them
	// to avoid work
	for (int doi=0; doi < env->m_displayWorld->getNumDisplayObjects();)
	{
		if (env->m_displayWorld->getDisplayObject(doi)->getNumGeometry() < 1)
		{
			env->m_displayWorld->removeDisplayObject(doi)->removeReference();
		}
		else
		{
			doi++;
		}
	}

	env->m_displayWorld->unlock();

	return numConverted;
}

hkgDisplayObject* hkDefaultPhysics2012Demo::findMeshDisplay( const char* meshName, const hkxScene* scene )
{
	if (meshName)
	{
		hkxNode* node = _findMeshNodeByName(scene->m_rootNode, meshName);
		hkgDisplayObject* dispObj = _findDisplayObject( node, m_env->m_sceneConverter->m_meshNodes );
		return dispObj;
	}
	return HK_NULL;
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
