/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/DemoFramework/MonitorStatsRecorder.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#define MAX_MONITOR_BUFFER ((1024 * 1024 * 1024) - 1) // The amount of memory available to store the timers of all to be recorded frames of all threads! (1GB... the maximum amount the hkArray can hold)

// Pass in a repetition index of -1 to generate the aggregate stats file name
static void generateFileName(const hkDefaultDemo* demo, int repetitionIndex, hkStringBuf& fileNameOut)
{
	const hkDemoFrameworkOptions* options = demo->m_env->m_options;
	fileNameOut.set(options->m_outputRecordDir);

	// Terminate '.' and '..' with '/' so they are considered as directory names
	if (fileNameOut.compareTo(".") == 0 || fileNameOut.compareTo("..") == 0)
	{
		fileNameOut.append("/");
	}

	// If the last character of the file name is '/' we will append the full demo name
	if (fileNameOut.endsWith("/"))
	{
		// Substitute non alphanumeric characters in the full demo name with '_'
		hkStringBuf	demoName(demo->m_menuPath.cString());
		for(int i=0;i<demoName.getLength();++i)
		{
			if(	!(	(demoName[i]>='a' && demoName[i]<='z') ||
				(demoName[i]>='A' && demoName[i]<='Z') ||
				(demoName[i]>='0' && demoName[i]<='9')))
			{
				demoName.getArray()[i] = '_';
			}
		}

		fileNameOut.append(demoName);
	}

	// Append repetition number
	if (options->m_numRepetitions > 1)
	{
		if (repetitionIndex < 0)
		{
			fileNameOut.appendPrintf(".1-%d", options->m_numRepetitions);
		}
		else
		{
			fileNameOut.appendPrintf(".%d", repetitionIndex + 1);
		}
	}

	// Append format extension
	if (options->m_outputRecordFormat == MonitorStatsRecorder::XML)
	{
		fileNameOut.append(".xml");
	}
	else
	{
		fileNameOut.append(".csv");
	}
}

MonitorStatsRecorder::MonitorStatsRecorder(hkDefaultDemo* demo)
{
	const hkDemoFrameworkOptions* demoOptions = demo->m_env->m_options;
	m_demo = demo;
	m_firstFrame = demoOptions->m_firstFrameToRecord;
	m_numFrames = demoOptions->m_numFramesToRecord;
	m_numMonitors = demoOptions->m_numMonitors;
	m_monitorNames = demoOptions->m_firstMonitor;
	m_outputFormat = (OutputFormat)demoOptions->m_outputRecordFormat;
	if (m_outputFormat == CSV && m_numMonitors == 0)
	{
		HK_ERROR(0x65c6b36b, "You must specifiy at least one monitor name when using CSV output format");
	}
	generateFileName(demo, demo->m_env->m_repetitionIndex, m_fileName);

	// Calculate number of threads to record
	m_numThreads = demoOptions->m_recordAllThreads ? 1 + m_demo->m_threadPool->getNumThreads() : 1;
	if (m_outputFormat == XML && m_numThreads > 1)
	{
		HK_WARN(0x3c8e24a7, "Reporting of thread pool monitors is not supported for XML output format. Only the main thread monitors will be recorded");
		m_numThreads = 1;
	}

	// Create monitor stream analyzer
	const int bufferSize = MAX_MONITOR_BUFFER;
	m_analyzer = new hkMonitorStreamAnalyzer(bufferSize, m_numThreads);

	hkStringBuf statusText;
	statusText.printf("Recording frames [%d => %d] to '%s'", m_firstFrame, m_numFrames + m_firstFrame, m_fileName.cString());
	HK_REPORT(statusText.cString());
}


MonitorStatsRecorder::~MonitorStatsRecorder()
{
	delete m_analyzer;
}


bool MonitorStatsRecorder::appendFrame()
{
	const int frameIndex = m_demo->m_stepCounter - m_firstFrame;
	if (frameIndex < 0 || frameIndex > m_numFrames)
	{
		return false;
	}

	hkMonitorStreamFrameInfo frameInfo;
	frameInfo.m_heading.set("");
	frameInfo.m_indexOfTimer0 = 0;
	frameInfo.m_indexOfTimer1 = -1;
	frameInfo.m_timerFactor0 = 1e3f / float(hkStopwatch::getTicksPerSecond());
	frameInfo.m_timerFactor1 = 1;
	frameInfo.m_absoluteTimeCounter = hkMonitorStreamFrameInfo::ABSOLUTE_TIME_TIMER_0;

	// Get main thread timer
	hkTimerData info;
	info.m_streamBegin = hkMonitorStream::getInstance().getStart();
	info.m_streamEnd = hkMonitorStream::getInstance().getEnd();
	extArray<hkTimerData> threadStreams;
	threadStreams.pushBack(info);

	// Get thread pool timers
	if (m_numThreads > 1)
	{
		if ( m_demo->m_threadPool != HK_NULL )
		{
			m_demo->m_threadPool->appendTimerData( threadStreams, extAllocator::getInstance() );
		}
	}
	if (threadStreams.getSize() != m_numThreads)
	{
		HK_ERROR(0x7f768324, "The number of monitor streams does not match the number of threads provided on construction of the monitor stats recorder.");
		return true;
	}

	// Copy timers to the analyzer buffer
	for (int threadIndex = 0; threadIndex < m_numThreads; ++threadIndex)
	{
		frameInfo.m_threadId = threadIndex;
		m_analyzer->captureFrameDetails(threadStreams[threadIndex].m_streamBegin, threadStreams[threadIndex].m_streamEnd, frameInfo);
	}

	if ((frameIndex == m_numFrames - 1) && m_fileName.getLength())
	{
		m_analyzer->checkAllThreadsCapturedSameNumFrames();

		hkLocalArray<hkMonitorStreamAnalyzer::Node*> threadsStatsRoot(m_numThreads);
		for (int i = 0; i < m_numThreads; ++i)
		{
			threadsStatsRoot.pushBack(m_analyzer->makeStatisticsTreeForMultipleFrames(i, true));
		}
		if (threadsStatsRoot[0]->m_children.getSize() != m_numFrames)
		{
			HK_ERROR(0x61b99f88, "The number of frames in the statistics is different from the number of recorded frames. This can be caused by frames getting dropped because of insufficient space allocated for the hkMonitorStreamAnalyzer::m_data buffer. Make sure the MAX_MONITOR_BUFFER define is set to a large enough value to hold all timer data for all threads during all frames.");
			return true;
		}

		if (m_outputFormat == XML)
		{
			exportToXml((const hkMonitorStreamAnalyzer::Node**)threadsStatsRoot.begin(), m_fileName.cString());
		}
		else
		{
			exportToCsv((const hkMonitorStreamAnalyzer::Node**)threadsStatsRoot.begin(), m_fileName.cString());
		}

		HK_REPORT("Record finished");

		aggregateStats();

		return true;
	}

	return false;
}


void MonitorStatsRecorder::exportToXml(const hkMonitorStreamAnalyzer::Node** threadsStatsRoot, const char* filename) const
{
	// Create output file
	HK_REPORT("Writing stats to: " << filename);
	hkOfstream	stream(filename);
	if (!stream.isOk())
	{
		HK_ERROR(0x388bf9b0, "Error creating file " << filename);
		return;
	}

	// Header
	stream << "<?xml version='1.0' encoding='utf-8'?>\n";
	stream << "<MonitorRecords>\n";

	// We will process only the stats of the main thread
	const hkArray<hkMonitorStreamAnalyzer::Node*>& frameStats = threadsStatsRoot[0]->m_children;
	hkStringBuf rootName("");
	for (int i = 0; i < frameStats.getSize(); ++i)
	{
		stream << "<Frame value='" << i << "'>\n";
		processStatsNodeXml(frameStats[i], rootName, stream);
		stream << "</Frame>\n";
	}
	stream << "</MonitorRecords>";
}


void MonitorStatsRecorder::processStatsNodeXml(
	const hkMonitorStreamAnalyzer::Node* node, const hkStringBuf& parentName, hkOstream& stream) const
{
	hkStringBuf fullName(parentName);
	hkStringBuf nodeName(node->m_name);

	if (nodeName.getLength())
	{
		if (parentName.getLength())
		{
			fullName.printf("%s/%s",parentName.cString(), node->m_name);
		}
		else
		{
			fullName = nodeName;
		}
		fullName.replace(' ','_');

		bool add = m_numMonitors ? false : true;

		// If we have a list of monitors, check if the current node is one of them
		if (!add)
		{
			for (int i = 0; i < m_numMonitors; ++i)
			{
				if (fullName.compareToIgnoreCase(m_monitorNames[i]) == 0)
				{
					add = true;
					break;
				}
			}
		}

		if (add)
		{
			stream << "<R name='" << fullName.cString() << "' value='" << node->m_value[0] << "'/>\n";
		}
	}

	for (int i = 0; i < node->m_children.getSize(); ++i)
	{
		processStatsNodeXml(node->m_children[i], fullName, stream);
	}
}

void MonitorStatsRecorder::exportValuesToCsv(const hkReal* values, const char* filename) const
{
	// Create output file
	HK_REPORT("Writing stats to: " << filename);
	hkOfstream	stream(filename);
	if (!stream.isOk())
	{
		HK_ERROR(0x388bf9b0, "Error creating file " << filename);
		return;
	}

	// Write headers
	stream << "Frame";
	const char* separator = " , ";
	for (int threadIndex = 0; threadIndex < m_numThreads; ++threadIndex)
	{
		for (int j = 0; j < m_numMonitors; ++j)
		{
			hkStringBuf name(m_monitorNames[j]);
			name.pathBasename();
			stream << separator << name.cString();
			if (m_numThreads > 1)
			{
				stream << "_" << threadIndex;
			}
		}
	}
	stream << "\n";

	// Print out monitor values for all frames
	const hkReal* valuePtr = values;
	const int numValuesPerFrame = m_numThreads * m_numMonitors;
	hkLocalArray<double> average(numValuesPerFrame); average.setSize(numValuesPerFrame, 0);
	hkLocalArray<hkReal> minValue(numValuesPerFrame); minValue.setSize(numValuesPerFrame, HK_REAL_MAX);
	hkLocalArray<hkReal> maxValue(numValuesPerFrame); maxValue.setSize(numValuesPerFrame, 0);
	for (int frameIndex = 0; frameIndex < m_numFrames; ++frameIndex)
	{
		stream << frameIndex;
		for (int i = 0; i < numValuesPerFrame; ++i)
		{
			hkReal value = *(valuePtr++);
			stream << separator << value;
			if (value < minValue[i])
			{
				minValue[i] = value;
			}
			if (value > maxValue[i])
			{
				maxValue[i] = value;
			}
			average[i] += value;
		}
		stream << "\n";
	}

	// Calculate average per value
	for (int i = 0; i < numValuesPerFrame; ++i)
	{
		average[i] /= (double) m_numFrames;
	}

	// Calculate aggregated variance per value
	hkLocalArray<double> stdDev(numValuesPerFrame); stdDev.setSize(numValuesPerFrame, 0);
	valuePtr = values;
	for (int frameIndex = 0; frameIndex < m_numFrames; ++frameIndex)
	{
		for (int i = 0; i < numValuesPerFrame; ++i)
		{
			double diff = (*valuePtr - average[i]);
			stdDev[i] += diff * diff;
			++valuePtr;
		}
	}

	// Print out calculated values
	stream << "\nAverage";
	for (int i = 0; i < numValuesPerFrame; ++i)
	{
		stream << separator << average[i];
	}
	stream << "\nNormStdDev";
	for (int i = 0; i < numValuesPerFrame; ++i)
	{
		double value = hkMath::sqrt(stdDev[i] / (double) m_numFrames) / average[i];
		stream << separator << value;
	}
	stream << "\nMin";
	for (int i = 0; i < numValuesPerFrame; ++i)
	{
		stream << separator << minValue[i];
	}
	stream << "\nMax";
	for (int i = 0; i < numValuesPerFrame; ++i)
	{
		stream << separator << maxValue[i];
	}
}


void MonitorStatsRecorder::exportToCsv(const hkMonitorStreamAnalyzer::Node** threadsStatsRoot, const char* filename) const
{
	// Allocate a big buffer for all thread monitor values
	int bufferSize = m_numFrames * m_numThreads * m_numMonitors;
	hkLocalArray<hkReal> monitorValues(bufferSize);
	monitorValues.setSize(bufferSize, 0);

	// Extract monitor values from stat trees
	hkStringBuf rootName("");
	for (int threadIndex = 0; threadIndex < m_numThreads; ++threadIndex)
	{
		const hkMonitorStreamAnalyzer::Node* threadStats = threadsStatsRoot[threadIndex];
		hkReal* values = &monitorValues[threadIndex * m_numMonitors];
		for (int frameIndex = 0; frameIndex < m_numFrames; ++frameIndex)
		{
			const hkMonitorStreamAnalyzer::Node* frameStats = threadStats->m_children[frameIndex];
			processStatsNodeCsv(frameStats, rootName, values);
			values += m_numThreads * m_numMonitors;
		}
	}

	exportValuesToCsv(monitorValues.begin(), filename);
}


void MonitorStatsRecorder::processStatsNodeCsv(
	const hkMonitorStreamAnalyzer::Node* node, const hkStringBuf& parentName, hkReal* values) const
{
	hkStringBuf fullName(parentName);
	hkStringBuf nodeName(node->m_name);

	if (nodeName.getLength())
	{
		if (parentName.getLength())
		{
			fullName.printf("%s/%s",parentName.cString(), node->m_name);
		}
		else
		{
			fullName = nodeName;
		}
		fullName.replace(' ','_');


		int monitorIndex = 0;
		for (; monitorIndex < m_numMonitors; ++monitorIndex)
		{
			if (fullName.compareToIgnoreCase(m_monitorNames[monitorIndex]) == 0)
			{
				values[monitorIndex] = node->m_value[0];
				break;
			}
		}
	}

	for (int i = 0; i < node->m_children.getSize(); ++i)
	{
		processStatsNodeCsv(node->m_children[i], fullName, values);
	}
}

void MonitorStatsRecorder::aggregateStats()
{
	const int repetitionIndex = m_demo->m_env->m_repetitionIndex;
	const int numRepetitions = m_demo->m_env->m_options->m_numRepetitions;
	if (numRepetitions == 1 || m_outputFormat != CSV || repetitionIndex + 1 < numRepetitions)
	{
		return;
	}

	const int numValuesPerFrame = m_numThreads * m_numMonitors;
	const int numValues = m_numFrames * numValuesPerFrame;
	hkLocalArray<hkReal> monitorValues(numValues);
	monitorValues.setSize(numValues, 0);

	hkStringBuf fileName;
	char buffer[1024];

	// Accumulate the values of the stats file for each repetition
	for (int i = 0; i < numRepetitions; ++i)
	{
		generateFileName(m_demo, i, fileName);
		hkIstream fileIn(fileName.cString());
		if (!fileIn.isOk())
		{
			HK_ERROR(0x3a8187dd, "Error opening file " << fileName);
		}

		// Skip header line
		while (fileIn.getline(buffer, 1024) == -1) {};

		hkReal* currentValue = &monitorValues[0];
		for (int j = 0; j < m_numFrames; ++j)
		{
			// Skip frame number
			int frameNumber;
			fileIn >> frameNumber;

			// Read and accumulate monitor values
			hkReal valueRead;
			char comma;
			for (int k = 0; k < numValuesPerFrame; ++k)
			{
				do
				{
					fileIn >> comma;
				} while (comma != ',');
				fileIn >> valueRead;
				*currentValue += valueRead;
				currentValue++;
			}
		}
	}

	// Average all values across repetitions
	hkReal* currentValue = &monitorValues[0];
	hkReal numRepetitionsReal = (hkReal) numRepetitions;
	for (int i = 0; i < numValues; ++i)
	{
		*currentValue /= numRepetitionsReal;
		currentValue++;
	}

	// Export as .csv
	generateFileName(m_demo, -1, fileName);
	exportValuesToCsv(&monitorValues[0], fileName);
}

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
