/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Demos/Common/Test/Feature/ConvexHull/ConvexHullDemo.h>


// Serialization
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Base/Reflection/Registry/hkTypeInfoRegistry.h>

// Asset mgt
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/Base/Types/Geometry/hkStridedVertices.h>

// Misc.
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>

// Convex hull
#include <Common/Internal/GeometryProcessing/ConvexHull/hkgpConvexHull.h>

static hkPseudoRandomGenerator	g_rndgen(180673);
static int						g_seed(-1);
struct eOperation { enum _ { NONE, INTERSECTION, SUBTRACTION, VORONOI_CELLS, VORONOI_CVT, VORONOI_EDGES, ABS_SCALE, SLICE };};

struct ConvexHullDemoType { enum _ {
	RANDOM_POTATO,
	RANDOM_CYLINDER,
	RANDOM_SPHERE,
	RANDOM_CUBIC_CURVE,
	RANDOM_PLANAR_POTATO,
	RANDOM_PLANAR_CIRCLE,
	RANDOM_PLANES,
	RANDOM_PLANAR_PLANES,
	SIMPLIFIED_RND_POTATO,
	SIMPLIFIED_RND_POTATO_CNT,
	SIMPLIFIED_RND_POTATO_2D,
	SIMPLIFIED_RND_POTATO_CNT_2D,
	BOOL_3D_INT_POTATOES,
	BOOL_3D_SUB_POTATOES,
	BOOL_2D_INT_POTATOES,
	BOOL_2D_SUB_POTATOES,
	VORONOI_CELLS,
	VORONOI_EDGES,
	PLANAR_VORONOI_CELLS,
	PLANAR_VORONOI_EDGES,
	PLANAR_CLOUD_3D,
	PLANAR_CVT,
	PLANAR_SLICE,
	ABS_SCALING,
}; };

static const ConvexHullDemo::Variant	g_variants[]=
	{
		ConvexHullDemo::Variant(-1,												"Loop all")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_POTATO,				"Random potato")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_CYLINDER,			"Random cylinder")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_SPHERE,				"Random sphere")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_CUBIC_CURVE,			"Random cubic curve")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_PLANAR_POTATO,		"Random planar potato")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_PLANAR_CIRCLE,		"Random planar circle")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_PLANES,				"Random planes")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::RANDOM_PLANAR_PLANES,		"Random planar planes")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::SIMPLIFIED_RND_POTATO,		"Simplified random potato")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::SIMPLIFIED_RND_POTATO_CNT,	"Simplified random potato with containment")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::SIMPLIFIED_RND_POTATO_2D,	"Simplified random planar potato")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::SIMPLIFIED_RND_POTATO_CNT_2D,"Simplified random planar potato with containment")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::BOOL_3D_INT_POTATOES,		"Intersection of potatoes")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::BOOL_3D_SUB_POTATOES,		"Subtraction of potatoes")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::BOOL_2D_INT_POTATOES,		"Intersection of planar potatoes")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::BOOL_2D_SUB_POTATOES,		"Subtraction of planar potatoes")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::VORONOI_CELLS,				"Voronoi cells")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::VORONOI_EDGES,				"Voronoi edges")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::PLANAR_VORONOI_CELLS,		"Planar Voronoi cells")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::PLANAR_VORONOI_EDGES,		"Planar Voronoi edges")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::PLANAR_CLOUD_3D,			"Planar cloud as 3D hull")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::PLANAR_CVT,					"Planar centroidal Voronoi tesselation")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::PLANAR_SLICE,				"Planar slice")
	,	ConvexHullDemo::Variant(ConvexHullDemoType::ABS_SCALING,				"Absolute scaling")
	};

//
						ConvexHullDemo::ConvexHullDemo(hkDemoEnvironment* env) : hkDefaultDemo(env)
{	
	hkVector4 from(16.0f, 0.0f, 0.0f);
	hkVector4 to(0.0f, 0.0f, 0.0f);
	hkVector4 up(0.0f, 0.0f, 1.0f);	
	setupDefaultCameras(env,from,to,up,0.01f,1000.0f);
	setupGraphics();

	m_currentTest				=	0x7fffffff;
	m_numSteps					=	0;
	m_numLoops					=	0;
	m_numSamples				=	1024;
	m_numHulls					=	0;
	m_timeAccumulator			=	0;
	m_accuracy					=	1;
	m_loopDemos					=	true;
	m_variant					=	g_variants+m_env->m_variantId;
	m_hullPerSeconds.reserve(256);
	m_stopWatch.start();
	if(m_variant->m_test>=0)
	{
		m_loopDemos		=	false;
		m_currentTest	=	m_variant->m_test;
	}
}

static hkReal hkMath_rand01(unsigned& seed)
{
	const unsigned a = 1103515245;
	const unsigned c = 12345;
	const unsigned m = unsigned(-1) >> 1;
	seed = (a * seed + c ) & m;
	return (hkReal(seed) / m);
}

//
static void				draw(const hkgpConvexHull& ch,const hkArray<int>& vpf,const hkArray<int>& vid,const hkVector4& offset,const hkVector4& basecolor,bool drawEdges=true)
{
	hkVector4	light;
	light.set(1,1,1,0);
	light.normalize3();
	if(ch.getDimensions()>=0)
	{
		if(ch.getDimensions()>2)
		{
			for(hkgpConvexHull::Triangle* t=ch.getFirstTriangle();t;t=ch.getNext(t))
			{
				const hkVector4			p=ch.getPlaneEquation(t);
				const hkReal			l=hkMath::fabs(p.dot3(light))*0.8f+0.2f;
				const int				c=hkColor::rgbFromFloats(basecolor(0)*l,basecolor(1)*l,basecolor(2)*l,basecolor(3));
				hkgpConvexHull::Vertex*	v[]={ch.getVertex(t,0),ch.getVertex(t,1),ch.getVertex(t,2)};
				hkVector4				x[]={ch.getSourcePosition(v[0]),ch.getSourcePosition(v[1]),ch.getSourcePosition(v[2])};
				x[0].add4(offset);
				x[1].add4(offset);
				x[2].add4(offset);
				HK_DISPLAY_TRIANGLE(x[0],x[1],x[2],c);
			}
		}
		if((drawEdges || ch.getDimensions()<3) && (vid.getSize()>0))
		{
			const hkColor::Argb	color=hkColor::rgbFromFloats(basecolor(0),basecolor(1),basecolor(2),1);//drawEdges? (int)hkColor::WHITE : (int)hkColor::rgbFromFloats(basecolor(0),basecolor(1),basecolor(2),1);
			hkArray<hkVector4>	positions;
			const int*			vi=&vid[0];
			ch.fetchPositions(hkgpConvexHull::SOURCE_VERTICES,positions);
			for(int i=0;i<positions.getSize();++i)
			{
				positions[i].add4(offset);
			}
			for(int i=0;i<vpf.getSize();++i)
			{
				const int count=vpf[i];
				for(int j=0;j<count;++j)
				{
					HK_DISPLAY_LINE(positions[vi[j]],positions[vi[(j+1)%count]],color);
				}
				vi+=count;
			}
		}
	}
}

//
static void				draw(const hkgpConvexHull& ch,const hkVector4& basecolor,bool drawEdges, const hkVector4& offset=hkVector4::getZero())
{
	hkArray<int>	vpf,vid;
	ch.generateIndexedFaces(hkgpConvexHull::INTERNAL_VERTICES,vpf,vid, true);
	draw(ch,vpf,vid,offset,basecolor,drawEdges);
}

//
#include <Common/Internal/GeometryProcessing/hkGeometryProcessing.h>

//
static hkReal			weight(const hkVector4& x)
{
	hkReal	bw = x(1)+6;
	return hkMath::pow(1+bw,8);
}

//
static void				recurseComputeCOM(const hkVector4& a, const hkVector4& b, const hkVector4& c, int depth, hkReal& area, hkVector4& com)
{
	if(depth>0)
	{
		hkVector4	ab; ab.setInterpolate4(a,b,0.5f);
		hkVector4	bc; bc.setInterpolate4(b,c,0.5f);
		hkVector4	ca; ca.setInterpolate4(c,a,0.5f);
		recurseComputeCOM(a,ab,ca,depth-1,area,com);
		recurseComputeCOM(b,bc,ab,depth-1,area,com);
		recurseComputeCOM(c,ca,bc,depth-1,area,com);
		recurseComputeCOM(ab,bc,ca,depth-1,area,com);
	}
	else
	{
		const hkReal	bw = hkGeometryProcessing::triangleArea2(a,b,c).getReal()/6.0f;
		hkReal			vw[] = {weight(a),weight(b),weight(c)};
		for(int i=0;i<3;++i) vw[i]=bw*vw[i];
		com.addMul4(vw[0],a);
		com.addMul4(vw[1],b);
		com.addMul4(vw[2],c);
		area	+=	vw[0]+vw[1]+vw[2];
	}
}

//
static hkVector4		computeNonUniformCOM(hkgpConvexHull* hull)
{
	hkVector4			center = hull->getCentroid();
	hkArray<hkVector4>	vertices; hull->fetchPositions(hkgpConvexHull::INTERNAL_VERTICES,vertices);
	hkVector4			com; com.setZero4();
	hkReal				area = 0;
	for(int i=vertices.getSize()-1,j=0;j<vertices.getSize();i=j++)
	{
		recurseComputeCOM(center,vertices[i],vertices[j],2,area,com);
	}
	com.mul4(1.0f/area);
	return com;
}

//
hkDefaultDemo::Result	ConvexHullDemo::stepDemo()
{
	int				maxStepPerTest	=	256;
	eOperation::_	operation		=	eOperation::NONE;
	bool			fromPlanes		=	false;
	bool			forcePlanar		=	false;
	bool			sortInputs		=	true;
	bool			doRotate		=	true;
	bool			doPreScale		=	true;
	bool			doPostScale		=	true;
	bool			doSimplify		=	false;
	bool			doConfine		=	false;
	bool			doDrawPoints	=	false;	
	hkReal			cvtSOR			=	1.0f;
	hkReal			cvtRANGE		=	4.0f;
	hkReal			shrinkLength	=	0.0f;
	hkVector4		projectionPlane	=	hkVector4::getZero();
	if(g_seed!=-1) g_rndgen.setSeed(g_seed);
	switch(m_currentTest)
	{

		// So called 'potato' convex hull
	case	ConvexHullDemoType::RANDOM_POTATO:
		{
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;

		// Random cylinder
	case	ConvexHullDemoType::RANDOM_CYLINDER:
		{			
			sortInputs=false;
			m_points.clear();
			for(int i=0;i<m_numSamples;++i)
			{
				const hkReal	a=(i/(hkReal)m_numSamples)*2*HK_REAL_PI;
				const hkReal	r=1;
				hkVector4		x;
				x(0)=r*hkMath::cos(a);
				x(1)=r*hkMath::sin(a);
				x(2)=-2*r;
				m_points.pushBack(x);
				x(2)=-x(2);
				m_points.pushBack(x);
			}			
		}
		break;

		// Random sphere
	case	ConvexHullDemoType::RANDOM_SPHERE:
		{
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].fastNormalize3NonZero();
			}
		}
		break;

		// Random cubic curve
	case	ConvexHullDemoType::RANDOM_CUBIC_CURVE:
		{
			hkVector4	c[4];
			m_points.setSize(m_numSamples);			
			g_rndgen.getRandomVector11(c[0]);
			g_rndgen.getRandomVector11(c[1]);
			g_rndgen.getRandomVector11(c[2]);
			g_rndgen.getRandomVector11(c[3]);
			for(int i=0;i<4;++i)
			{
				c[i].mul4(3);
			}
			for(int i=0;i<m_points.getSize();++i)
			{
				const hkReal	t=i/(hkReal)(m_points.getSize()-1);
				hkVector4		d[3];
				d[0].setInterpolate4(c[0],c[1],t);
				d[1].setInterpolate4(c[1],c[2],t);
				d[2].setInterpolate4(c[2],c[3],t);
				d[0].setInterpolate4(d[0],d[1],t);
				d[1].setInterpolate4(d[1],d[2],t);
				m_points[i].setInterpolate4(d[0],d[1],t);
			}
		}
		break;

		// Random planar potato
	case	ConvexHullDemoType::RANDOM_PLANAR_POTATO:
		{
			forcePlanar		=	true;
			doPostScale		=	false;
			doDrawPoints	=	true;
			projectionPlane.set(0,0,1,0);
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].mul4(hkVector4(1,1,0,0));
			}
		}
		break;

		// Random circle
	case	ConvexHullDemoType::RANDOM_PLANAR_CIRCLE:
		{
			forcePlanar		=	true;
			doPostScale		=	false;
			doDrawPoints	=	true;
			projectionPlane.set(0,0,1,0);
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].mul4(hkVector4(1,1,0,0));
				m_points[i].normalize3IfNotZero();
			}
		}
		break;

		// Random planes
	case	ConvexHullDemoType::RANDOM_PLANES:
		{
			fromPlanes		=	true;
			doRotate		=	false;
			doPreScale		=	false;
			doPostScale		=	false;
			maxStepPerTest	=	16;
			m_points.setSize(4+m_numSamples/4);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].normalize3IfNotZero();
				m_points[i](3)=-2;
			}
		}
		break;

		// Random planar planes
	case	ConvexHullDemoType::RANDOM_PLANAR_PLANES:
		{
			fromPlanes		=	true;
			doRotate		=	false;
			doPreScale		=	false;
			doPostScale		=	false;
			maxStepPerTest	=	16;
			projectionPlane.set(1,0,0,0);
			m_points.setSize(3+m_numSamples/32);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].subMul4(projectionPlane.dot4xyz1(m_points[i]),projectionPlane);
				m_points[i].normalize3IfNotZero();
				m_points[i](3)=-2;
			}
		}
		break;

		// Simplified random potato
	case	ConvexHullDemoType::SIMPLIFIED_RND_POTATO:
		{
			doSimplify		=	true;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}		
		break;

		// Simplified random potato with containment
	case	ConvexHullDemoType::SIMPLIFIED_RND_POTATO_CNT:
		{
			doSimplify		=	true;
			doConfine		=	true;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}		
		break;		

		// Simplified random planar potato
	case	ConvexHullDemoType::SIMPLIFIED_RND_POTATO_2D:
		{
			forcePlanar		=	true;
			doSimplify		=	true;
			doDrawPoints	=	true;
			doPostScale		=	false;
			doPostScale		=	false;
			doRotate		=	false;
			projectionPlane.set(1,0,0,0);
			m_points.setSize(m_numSamples / 8);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].mul4(hkVector4(0,1,1,0));
			}
		}
		break;

		// Simplified random planar potato with containment
	case	ConvexHullDemoType::SIMPLIFIED_RND_POTATO_CNT_2D:
		{
			forcePlanar		=	true;
			doSimplify		=	true;
			doConfine		=	true;
			doDrawPoints	=	true;
			doPostScale		=	false;
			doPostScale		=	false;
			doRotate		=	false;
			projectionPlane.set(1,0,0,0);
			m_points.setSize(m_numSamples / 8);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].mul4(hkVector4(0,1,1,0));
			}
		}		
		break;		

		// Intersection of potatoes
	case	ConvexHullDemoType::BOOL_3D_INT_POTATOES:
		{
			operation		=	eOperation::INTERSECTION;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;

		// Subtraction of potatoes
	case	ConvexHullDemoType::BOOL_3D_SUB_POTATOES:
		{
			operation		=	eOperation::SUBTRACTION;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;

		// Intersection of planar potatoes
	case	ConvexHullDemoType::BOOL_2D_INT_POTATOES:
		{
			forcePlanar		=	true;
			doRotate		=	false;
			doPreScale		=	false;
			doPostScale		=	false;
			operation		=	eOperation::INTERSECTION;
			m_numSamples	=	24;
			projectionPlane.set(1,0,0,0);
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].subMul4(projectionPlane.dot4xyz1(m_points[i]),projectionPlane);
			}
		}
		break;

		// Subtraction of planar potatoes
	case	ConvexHullDemoType::BOOL_2D_SUB_POTATOES:
		{
			forcePlanar		=	true;
			doRotate		=	false;
			doPreScale		=	false;
			doPostScale		=	false;
			operation		=	eOperation::SUBTRACTION;
			m_numSamples	=	24;
			projectionPlane.set(1,0,0,0);
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].subMul4(projectionPlane.dot4xyz1(m_points[i]),projectionPlane);
			}
		}
		break;

		// Voronoi cells of a random potato
	case	ConvexHullDemoType::VORONOI_CELLS:
		{
			operation		=	eOperation::VORONOI_CELLS;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;

		// Voronoi edges of a random potato
	case	ConvexHullDemoType::VORONOI_EDGES:
		{
			operation		=	eOperation::VORONOI_EDGES;
			maxStepPerTest	=	16;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;
		// Voronoi cells of a planar random potato
	case	ConvexHullDemoType::PLANAR_VORONOI_CELLS:
		{
			forcePlanar		=	true;
			doPostScale		=	false;
			operation		=	eOperation::VORONOI_CELLS;
			projectionPlane.set(0,0,1,0);
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].subMul4(projectionPlane.dot4xyz1(m_points[i]),projectionPlane);
			}
		}
		break;

		// Voronoi edges of a planar random potato
	case	ConvexHullDemoType::PLANAR_VORONOI_EDGES:
		{
			forcePlanar		=	true;
			doPostScale		=	false;
			operation		=	eOperation::VORONOI_EDGES;
			projectionPlane.set(0,0,1,0);
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].subMul4(projectionPlane.dot4xyz1(m_points[i]),projectionPlane);
			}
		}
		break;
		// Planar cloud as 3D hull
	case	ConvexHullDemoType::PLANAR_CLOUD_3D:
		{
			forcePlanar		=	false;
			doPostScale		=	false;
			doPostScale		=	false;
			doRotate		=	false;
			doDrawPoints	=	true;
			m_points.setSize(100*100);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
				m_points[i].mul4(hkVector4(0,1,1,0));
			}
		}
		break;
		// Planar CVT
	case	ConvexHullDemoType::PLANAR_CVT:
		{
			forcePlanar		=	true;
			doPostScale		=	false;
			doPreScale		=	false;
			doRotate		=	false;
			operation		=	eOperation::VORONOI_CVT;
			maxStepPerTest	=	256;
			cvtSOR			=	1.8f;
			cvtRANGE		=	6.0f;
			if(m_numSteps==0)
			{
				projectionPlane.set(1,0,0,0);
				m_points.setSize(256);
				for(int i=0;i<m_points.getSize();++i)
				{
					g_rndgen.getRandomVector11(m_points[i]);
					m_points[i].mul4(cvtRANGE*0.99f);
					m_points[i].subMul4(projectionPlane.dot4xyz1(m_points[i]),projectionPlane);
				}
			}			
		}
		break;
		// Planar slice
	case	ConvexHullDemoType::PLANAR_SLICE:
		{
			operation		=	eOperation::SLICE;
			m_points.setSize(m_numSamples);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;
		// Absolute scaling
	case	ConvexHullDemoType::ABS_SCALING:
		{
			operation		=	eOperation::ABS_SCALE;
			shrinkLength	=	0.25f * (1-hkMath::pow(1-m_numSteps/(hkReal)maxStepPerTest,2));
			forcePlanar		=	false;
			doPostScale		=	false;
			doPostScale		=	false;
			doRotate		=	false;
			m_points.setSize(6);
			if(!m_loopDemos) g_rndgen.setSeed(m_numLoops*757454);
			for(int i=0;i<m_points.getSize();++i)
			{
				g_rndgen.getRandomVector11(m_points[i]);
			}
		}
		break;
	default:
		m_currentTest=m_numSteps=0;
		return(ConvexHullDemo::stepDemo());
	}
	
	/* Apply random pre-scale	*/ 
	if(doPreScale)
	{
		hkVector4	scale;
		g_rndgen.getRandomVector01(scale);
		scale.add4(hkVector4(0.1f,0.1f,0.1f));
		scale.mul4(2.0f);
		projectionPlane.mul4(scale);
		for(int i=0;i<m_points.getSize();++i)
		{
			m_points[i].mul4(scale);
		}
	}
	
	/* Apply random rotation	*/ 
	if(doRotate)
	{
		hkRotation	rotation;
		g_rndgen.getRandomRotation(rotation);
		projectionPlane.setRotatedDir(rotation,projectionPlane);
		for(int i=0;i<m_points.getSize();++i)
		{
			hkVector4	x=m_points[i];
			m_points[i].setRotatedDir(rotation,x);
		}
	}
	
	/* Apply random post-scale	*/ 
	if(doPostScale)
	{
		hkVector4	scale;
		g_rndgen.getRandomVector01(scale);
		scale.add4(hkVector4(0.1f,0.1f,0.1f));
		scale.mul4(2.0f);
		for(int i=0;i<m_points.getSize();++i)
		{
			m_points[i].mul4(scale);
		}
	}
	hkgpConvexHull		ch;
	
	/* Generate convex hull		*/ 
	if(m_points.getSize()>0)
	{
		HK_TIMER_BEGIN("hkConvexHull",this);
		hkgpConvexHull::BuildConfig	config;
		const hkReal				time=m_stopWatch.getElapsedSeconds();
		config.m_allowLowerDimensions	=	true;
		config.m_buildIndices			=	false;
		config.m_sortInputs				=	sortInputs;
		projectionPlane.normalize3IfNotZero();
		if(fromPlanes)
		{
			if(projectionPlane.lengthSquared3().isGreaterZero())
				ch.buildFromPlanes(&m_points[0],m_points.getSize(),projectionPlane,config);
			else
				ch.buildFromPlanes(&m_points[0],m_points.getSize(),config);
		}
		else if(forcePlanar)
		{
			ch.buildPlanar(m_points,projectionPlane,config);
		}
		else
		{
			ch.build(m_points,config);
		}
		m_numHulls++;
		m_timeAccumulator+=m_stopWatch.getElapsedSeconds()-time;
		HK_TIMER_END();
	}
	
	/* Enumerate planes			*/ 
	{
		HK_TIMER_BEGIN("Indexing",this);
		ch.buildIndices();
		HK_TIMER_END();
	}
	
	/* Generate faces			*/ 
	hkArray<int>	verticesPerFace;
	hkArray<int>	vertexIndices;
	{
		HK_TIMER_BEGIN("Generate faces",this);
		ch.generateIndexedFaces(hkgpConvexHull::INTERNAL_VERTICES,verticesPerFace,vertexIndices,true);
		HK_TIMER_END();
	}
	
	/* Mass properties			*/ 
	{
		HK_TIMER_BEGIN("Mass properties",this);
		if(ch.getDimensions()>=2)
		{
			ch.buildMassProperties();
		}
		HK_TIMER_END();
	}
	
	/* Draw points				*/ 
	if(doDrawPoints)
	{
		for(int i=0;i<m_points.getSize();++i)
		{
			HK_DISPLAY_POINT(m_points[i],hkColor::WHITE);
			HK_DISPLAY_STAR(m_points[i],0.01f,hkColor::WHITE);
		}
	}
	
	/* Draw						*/ 
	switch(operation)
	{
	case	eOperation::NONE:
		draw(ch,verticesPerFace,vertexIndices,hkVector4(0,0,0),hkVector4(1,1,1,1));
		break;
	
	case	eOperation::INTERSECTION:
		{
			hkReal		delta = forcePlanar?0.5f:1.0f;
			hkVector4	offset(0,delta,0);
			
			/* Create operand A	*/ 
			for(int i=0;i<m_points.getSize();++i) m_points[i].add4(offset);
			hkgpConvexHull	operandA;
			if(forcePlanar)
				operandA.buildPlanar(m_points,projectionPlane);
			else
				operandA.build(m_points);
			
			/* Create operand B	*/ 
			offset.mul4(-delta*2);
			for(int i=0;i<m_points.getSize();++i) m_points[i].add4(offset);
			hkgpConvexHull	operandB;
			if(forcePlanar)
				operandB.buildPlanar(m_points,projectionPlane);
			else
				operandB.build(m_points);
			
			/* Create A & B		*/ 
			hkgpConvexHull*	result=0;
			HK_TIMER_BEGIN("Boolean intersection",this);
			hkgpConvexHull::booleanIntersection(&operandA,&operandB,result);
			HK_TIMER_END();
			
			/* Draw				*/ 
			if(result) draw(*result,hkVector4(1,1,1,1),true);
			draw(operandA,hkVector4(1,0,0,0.25f),false);
			draw(operandB,hkVector4(0,1,0,0.25f),false);
			
			/* Cleanup			*/ 
			delete result;
		}
		break;
	
	case	eOperation::SUBTRACTION:
		{
			hkReal		delta = forcePlanar?0.5f:1.0f;
			hkVector4	offset(0,delta,0);
			
			/* Create operand A	*/ 
			for(int i=0;i<m_points.getSize();++i) m_points[i].add4(offset);
			hkgpConvexHull	operandA;
			if(forcePlanar)
				operandA.buildPlanar(m_points,projectionPlane);
			else
				operandA.build(m_points);
			
			/* Create operand B	*/ 
			offset.mul4(-delta*2);
			for(int i=0;i<m_points.getSize();++i) m_points[i].add4(offset);
			hkgpConvexHull	operandB;
			if(forcePlanar)
				operandB.buildPlanar(m_points,projectionPlane);
			else
				operandB.build(m_points);
			
			/* Create A - B		*/ 
			HK_TIMER_BEGIN("Boolean subtraction",this);
			hkArray<hkgpConvexHull*>	results;
			hkgpConvexHull::booleanSubtraction(&operandA,&operandB,results);
			HK_TIMER_END();
			
			/* Draw and cleanup	*/ 
			for(int i=0;i<results.getSize();++i)
			{
				results[i]->buildIndices();
				draw(*results[i],hkVector4(1,1,1,1),true);
				delete results[i];
			}			
			draw(operandB,hkVector4(0,1,0,0.25f),false);
		}
		break;
	
	case	eOperation::VORONOI_CELLS:
		{
			const hkReal				margin=0.02f;
			hkArray<hkVector4>			sites;
			hkArray<hkgpConvexHull*>	cells;
			hkgpConvexHull::createRandomSamples(&ch,g_rndgen,ch.getDimensions()==2?64:32,sites);
			HK_TIMER_BEGIN("Voronoi cells",this);
			hkgpConvexHull::createVoronoiCells(&ch,sites,cells,margin);
			HK_TIMER_END();
			for(int i=0;i<cells.getSize();++i)
			{
				hkVector4	rndColor(1,1,1,1);
				cells[i]->buildIndices();
				if(ch.getDimensions()!=2) { g_rndgen.getRandomVector01(rndColor);rndColor(3)=1; }
				draw(*cells[i],rndColor,false);
				delete cells[i];
			}
		}
		break;

	case	eOperation::VORONOI_CVT:
		{
			hkArray<hkVector4>			domain;
			domain.pushBack(hkVector4(0,-cvtRANGE,-cvtRANGE));
			domain.pushBack(hkVector4(0,+cvtRANGE,-cvtRANGE));
			domain.pushBack(hkVector4(0,+cvtRANGE,+cvtRANGE));
			domain.pushBack(hkVector4(0,-cvtRANGE,+cvtRANGE));
			hkArray<hkgpConvexHull*>	cells;
			ch.buildPlanar(domain,hkVector4(1,0,0));
			HK_TIMER_BEGIN("Voronoi cells",this);
			hkgpConvexHull::createVoronoiCells(&ch,m_points,cells);
			HK_TIMER_END();
			for(int i=0;i<cells.getSize();++i)
			{
				hkVector4	rndColor(1,1,1,1);
				cells[i]->buildIndices();
				const int siteIndex = (int)(hkUlong)cells[i]->userData();
				hkVector4 com; com = computeNonUniformCOM(cells[i]);
				hkVector4 delta; delta.setSub4(com,m_points[siteIndex]);
				delta.mul4(cvtSOR); m_points[siteIndex].add4(delta);
				m_points[siteIndex](0) = 0;
				m_points[siteIndex](1) = hkMath::clamp(m_points[siteIndex](1),-cvtRANGE,+cvtRANGE);
				m_points[siteIndex](2) = hkMath::clamp(m_points[siteIndex](2),-cvtRANGE,+cvtRANGE);
				
				if(ch.getDimensions()!=2) { g_rndgen.getRandomVector01(rndColor);rndColor(3)=1; }
				draw(*cells[i],rndColor,false);
				delete cells[i];
			}
		}
		break;
	
	case	eOperation::VORONOI_EDGES:
		{
			const hkReal				margin=0.02f;
			hkArray<hkVector4>			sites;
			hkArray<hkgpConvexHull*>	edges;
			hkgpConvexHull::createRandomSamples(&ch,g_rndgen,ch.getDimensions()==2?32:8,sites);
			HK_TIMER_BEGIN("Voronoi edges",this);
			hkgpConvexHull::createVoronoiEdges(&ch,sites,edges,margin);
			HK_TIMER_END();
			for(int i=0;i<edges.getSize();++i)
			{								
				draw(*edges[i],hkVector4(1,1,1,1),false);
				delete edges[i];
			}			
		}
		break;
	case	eOperation::ABS_SCALE:
		{
			const hkReal	w = ch.getWidth(hkVector4(0,1,0)).getReal()*2;
			const hkReal	h = ch.getWidth(hkVector4(0,0,1)).getReal();
			for(int i=0;i<3;++i)
			{
				hkVector4							off = hkVector4::getZero();
				off(1) = i*w - w;
				hkgpConvexHull*						sch = ch.clone();
				hkgpConvexHull::AbsoluteScaleConfig	asc;
				asc.m_method = (hkgpConvexHull::AbsoluteScaleConfig::Method)i;
				asc.m_featurePreservationFactor = 0.5f;
				sch->absoluteScale(-shrinkLength,asc);
				draw(*sch,hkVector4(1,1,1,1),true,off);
				delete sch;
				
				const char* m[]={"SKM_VERTICES","SKM_PLANES","SKM_CLAMPED_PLANES"};
				off(2) += h/2;
				HK_DISPLAY_3D_TEXT(m[i],off,hkColor::WHITE);
			}
			for(int i=0;i<3;++i)
			{
				hkVector4	off = hkVector4::getZero(); off(1) = i*w - w;
				draw(ch,hkVector4(1,0.5f,0.8f,0.5f),false,off);
			}
			hkStringBuf s; s.printf("offset = %.2fm",-shrinkLength);
			HK_DISPLAY_3D_TEXT(s.cString(),hkVector4(0,0,-h*1.5f),hkColor::LIGHTBLUE);
		}
		break;
	case	eOperation::SLICE:
		{
			hkVector4		com = ch.getCenterOfMass();
			hkVector4		plane(1,0,0,0); plane.setComponent<3>(-plane.dot<3>(com));
			hkgpConvexHull*	slice; hkgpConvexHull::createPlanarSlice(&ch, plane, slice);
			if(slice)
			{
				draw(*slice,hkVector4(1,1,1,1),true);
				delete slice;
			}
			draw(ch,hkVector4(1,1,1,0.25f),false);
		}
		break;
	default: break;
	}

	/* Simplify					*/ 
	if(doSimplify)
	{
		if(ch.getDimensions() == 3)
		{
			const int		maxVertices=ch.getNumVertices();
			const hkReal	baseVolume=ch.getVolume().getReal();
			const int		numVertices=16;
			hkgpConvexHull*	sch=HK_NULL;
			{
				HK_TIMER_BEGIN("Volume simplify",this);
				sch=hkgpConvexHull::volumeSimplify(&ch,0.1f,doConfine);
				HK_TIMER_END();
			}
			if(sch)
			{
				hkVector4	offset;
				hkVector4	supportN,supportP;
				sch->getSupportingVertex(hkVector4(0,-1,0),supportN);
				sch->getSupportingVertex(hkVector4(0,+1,0),supportP);
				offset.setSub4(supportN,supportP);
				offset.mul4(hkVector4(0,2.5f,0,0));
				sch->generateIndexedFaces(hkgpConvexHull::INTERNAL_VERTICES,verticesPerFace,vertexIndices, true);
				draw(*sch,verticesPerFace,vertexIndices,offset,hkVector4(1,0.5,1,1));
				hkStringBuf text;
				text.printf("%u/%u Vertices\n(%.2f%% Vol)",sch->getNumVertices(),maxVertices,100*(sch->getVolume().getReal() / baseVolume));
				HK_DISPLAY_3D_TEXT(text,offset,hkColor::WHITE);
				delete sch;
			}

			{
				HK_TIMER_BEGIN("Simplify",this);
				ch.decimateVertices(numVertices,doConfine);
				HK_TIMER_END();
			}
			{
				hkVector4	offset;
				hkVector4	supportN,supportP;
				ch.getSupportingVertex(hkVector4(0,-1,0),supportN);
				ch.getSupportingVertex(hkVector4(0,+1,0),supportP);
				offset.setSub4(supportN,supportP);
				offset.mul4(hkVector4(0,1.25f,0,0));
				ch.generateIndexedFaces(hkgpConvexHull::INTERNAL_VERTICES,verticesPerFace,vertexIndices, true);
				draw(ch,verticesPerFace,vertexIndices,offset,hkVector4(1,0.5,1,1));
				hkStringBuf	text;
				text.printf("%u/%u Vertices\n(%.2f%% Vol)",numVertices,maxVertices,100*(ch.getVolume().getReal() / baseVolume));
				HK_DISPLAY_3D_TEXT(text,offset,hkColor::WHITE);
			}
		}
		else
		{	
			hkVector4		offset; offset.setZero();
			hkgpConvexHull*	sch = ch.clone();			
			sch->decimateVertices(sch->getNumVertices() / 2, doConfine);			
			draw(*sch, hkVector4(0,1,1,1), true, offset);
			delete sch;
		}
	}
	
	/* Next						*/ 
	m_numSteps++;
	if(m_numSteps>maxStepPerTest)
	{
		m_hullPerSeconds.insert(m_currentTest,(int)(m_numHulls/m_timeAccumulator+0.5f));		
		HK_REPORT(""<<m_variant->m_name<<" : "<<m_hullPerSeconds.getWithDefault(m_currentTest,0));
		if(m_loopDemos) m_currentTest++; else m_numLoops++;
		m_numSteps			=	0;
		m_numHulls			=	0;
		m_accuracy			=	1;
		m_timeAccumulator	=	0;
		
	}
	
	/* Inputs					*/ 
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_ADD))		m_numSamples<<=1;
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_SUBTRACT)) m_numSamples>>=1;
	if(m_env->m_window->getKeyboard().wasKeyPressed('N'))				{ m_currentTest++;m_numSteps=0;m_loopDemos=false; }
	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_LEFT))		m_accuracy	-=	0.005f;
	if(m_env->m_window->getKeyboard().getKeyState(HKG_VKEY_RIGHT))		m_accuracy	+=	0.005f;
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_NUMPAD5))	g_seed-=10;
	if(m_env->m_window->getKeyboard().wasKeyPressed(HKG_VKEY_NUMPAD6))	g_seed+=10;
	if(m_accuracy<0)		m_accuracy=0;
	if(m_accuracy>1)		m_accuracy=1;
	if(m_numSamples<4)		m_numSamples=4;
	if(m_numSamples>8192)	m_numSamples=8192;	
	
	/* Text						*/ 
	hkStringBuf text;
	const int	hps=m_hullPerSeconds.getWithDefault(m_currentTest,0);
	text.printf("[+/-]Samples: %u",m_numSamples);
	m_env->m_textDisplay->outputText(text,8,(int)getWindowHeight()-24);
	if(hps>0)
	{
		text.printf("%u Convex hull per seconds",hps);
		m_env->m_textDisplay->outputText(text,8,(int)getWindowHeight()-40);
	}	
	
	int vidx=0;
	for(int i=0;i<(int)((sizeof(g_variants)/sizeof(g_variants[0])));++i)
	{
		if(g_variants[i].m_test==m_currentTest) { vidx=i;break; }
	}
	
	if(m_loopDemos)
		text.printf("<Loop all: %s>",g_variants[vidx].m_name);
	else
		text.printf("<%s>",g_variants[vidx].m_name);
	m_env->m_textDisplay->outputText(text,8,(int)getWindowHeight()-56);
	return(DEMO_OK);
}


HK_DECLARE_DEMO_VARIANT_USING_STRUCT( ConvexHullDemo, HK_DEMO_TYPE_PHYSICS_2012 | HK_DEMO_TYPE_DONT_BOOTSTRAP, ConvexHullDemo::Variant, g_variants, HK_NULL );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
