/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Common/Api/Serialize/SimpleLoad/SimpleLoadDemo.h>


// Serialize includes
#include <Common/Base/System/Io/IStream/hkIStream.h>
#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/Reflection/Registry/hkTypeInfoRegistry.h>
#include <Common/Serialize/Util/hkStructureLayout.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Physics2012/Utilities/Serialize/hkpPhysicsData.h>

// We have a different binary file depending on the compiler and platform
static inline void SimpleLoadDemo_getBinaryFileName(hkStringBuf& e)
{
	e.printf("/simple_L%d%d%d%d.hkx", 
		hkStructureLayout::HostLayoutRules.m_bytesInPointer,
		hkStructureLayout::HostLayoutRules.m_littleEndian? 1 : 0,
		hkStructureLayout::HostLayoutRules.m_reusePaddingOptimization? 1 : 0,
		hkStructureLayout::HostLayoutRules.m_emptyBaseClassOptimization? 1 : 0);
}

namespace
{
	void verifyPredicates(const char* fileName)
	{
		hkIstream istr(fileName);
		hkSerializeUtil::FormatDetails f;
		hkSerializeUtil::detectFormat(istr.getStreamReader(), f);
		if(HK_TEST(f.m_formatType != hkSerializeUtil::FORMAT_ERROR) && HK_TEST(f.m_formatType != hkSerializeUtil::FORMAT_UNKNOWN))
		{
			HK_TEST(f.supportsPredicates());
			HK_TEST(f.m_truePredicates.getSize() >= 1);
			hkSerializeUtil::FormatDetails::PredicateResult res = f.verifiesPredicate(hkContentPredicate::HAS_ROOT_LEVEL_CONTAINER);
			HK_TEST(res.isValid());
			HK_TEST(res.value());
		}
	}
}

SimpleLoadDemo::SimpleLoadDemo( hkDemoEnvironment* env) 
:	hkDefaultPhysics2012Demo(env),
	m_loadedData(HK_NULL)
{
	// Disable warnings
	hkError::getInstance().setEnabled(0xf03243ed, false); // 'm_contactRestingVelocity not set, setting it to REAL_MAX, so that the new collision restitution code will be disabled'
	hkError::getInstance().setEnabled(0x9fe65234, false);  // 'Unsupported simulation on type, setting to SIMULATION_TYPE_CONTINUOUS. See documentation on world stepping and time management'

	// Build the path
	hkStringBuf path("Resources/Common/Api/Serialize/SimpleLoad");
	{
		hkStringBuf fileName;

		switch (m_variantId)
		{
			case 0:
			{
				SimpleLoadDemo_getBinaryFileName(fileName);
				break;
			}
		
			case 1:
			{
				fileName = "/simple.xml";
				break;
			}
			case 2:
			{
				fileName = HK_ASSET_NAME("/simple.hkt");
				break;
			}
			case 3:
			{
				fileName = "/simpleTagfile.xml";
				break;
			}
			default:
			{
				HK_ASSERT2(0xdbc0f1bd, 0, "Bad demo variant");
			}
		}

		path += fileName;
	}
	
	
	// Load the file
	hkSerializeUtil::ErrorDetails loadError;
	m_loadedData = hkSerializeUtil::load(HK_GET_DEMOS_ASSET_FILENAME(path.cString()), &loadError);
	if ( !m_loadedData )
	{
		{
			HK_ASSERT3(0xa6451543, m_loadedData != HK_NULL, "Could not load file (" << path << "). The error is:\n" << loadError.defaultMessage.cString() );
		}
	}

	// Get the top level object in the file, which we know is a hkRootLevelContainer
	hkRootLevelContainer* container = m_loadedData->getContents<hkRootLevelContainer>();
	HK_ASSERT2(0xa6451543, container != HK_NULL, "Could not load root level object" );

	// Get the physics data
	hkpPhysicsData* physicsData = static_cast<hkpPhysicsData*>( container->findObjectByType( hkpPhysicsDataClass.getName() ) );
	HK_ASSERT2(0xa6451544, physicsData != HK_NULL, "Could not find physics data in root level object" );
	HK_ASSERT2(0xa6451535, physicsData->getWorldCinfo() != HK_NULL, "No physics cinfo in loaded file - cannot create a hkpWorld" );
	
	// Create a world and add the physics systems to it
	{
		m_world = new hkpWorld( *physicsData->getWorldCinfo() );
		m_world->lock();

		// Register all collision agents
		hkpAgentRegisterUtil::registerAllAgents( m_world->getCollisionDispatcher() );

		// Add all the physics systems to the world
		for ( int i = 0; i < physicsData->getPhysicsSystems().getSize(); ++i )
		{
			m_world->addPhysicsSystem( physicsData->getPhysicsSystems()[i] );
		}
		m_world->unlock();
	}
	

	

	switch (m_variantId)
	{
		case 0:
		case 1:
			{
				hkStringBuf fileName;
				SimpleLoadDemo_getBinaryFileName(fileName);
				fileName.chompStart(1); // Remove the leading '/'
				hkSerializeUtil::savePackfile( container, hkRootLevelContainerClass, hkOstream(fileName).getStreamWriter(), hkPackfileWriter::Options());
				verifyPredicates(fileName);
				break;
			}

		case 2:
			{
				hkSerializeUtil::saveTagfile( container, hkRootLevelContainerClass, hkOstream("simple.hkt").getStreamWriter() );
				verifyPredicates("simple.hkt");
				break;
			}
		case 3:
			{
				hkSerializeUtil::saveTagfile( container, hkRootLevelContainerClass, hkOstream("simpleTagfile.xml").getStreamWriter(), HK_NULL, hkSerializeUtil::SAVE_TEXT_FORMAT);
				verifyPredicates("simpleTagfile.xml");
				break;
			}
		default:
			{
				HK_ASSERT2(0xdbc0f1bd, 0, "Bad demo variant");
			}
	}


	// Set up graphics
	{
		hkVector4 from(31, 31, 14);
		hkVector4 to(2.5f, -2, -.3f);
		hkVector4 up(0, 0, 1);
		setupDefaultCameras( env, from, to, up );

		// this creates graphics objects for all rigid bodies and phantoms in the world
		setupGraphics();
	}
}

SimpleLoadDemo::~SimpleLoadDemo()
{
	// Re-enable warnings
	hkError::getInstance().setEnabled(0xf03243ed, true);
	hkError::getInstance().setEnabled(0x9fe65234, true);

	if ( m_world )
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}

	if ( m_loadedData )
	{
		m_loadedData->removeReference();
		m_loadedData = HK_NULL;
	}
}

hkDemo::Result SimpleLoadDemo::stepDemo()
{
	return hkDefaultPhysics2012Demo::stepDemo();
}



static const char helpString[] = \
"This demo demonstrates how to load a simple physics file.";

// loading float32 assets in double64 builds is not supported
#if defined(HK_REAL_IS_DOUBLE)
#	define THIS_DEMO_TYPE (HK_DEMO_TYPE_PHYSICS_2012 | HK_DEMO_TYPE_SERIALIZE)
#else
#	define THIS_DEMO_TYPE (HK_DEMO_TYPE_PHYSICS_2012 | HK_DEMO_TYPE_SERIALIZE | HK_DEMO_TYPE_CRITICAL)

	HK_DECLARE_DEMO_VARIANT(SimpleLoadDemo, THIS_DEMO_TYPE, "Binary",	0,	"Shows how to load a simple binary physics file", helpString);
		HK_DECLARE_DEMO_VARIANT(SimpleLoadDemo, THIS_DEMO_TYPE, "XML",		1,	"Shows how to load a simple xml physics file", helpString);
#endif

HK_DECLARE_DEMO_VARIANT(SimpleLoadDemo, THIS_DEMO_TYPE, "Tagfile",	2,	"Shows how to load a simple physics tag file", helpString);

#if !defined(HK_REAL_IS_DOUBLE)
	HK_DECLARE_DEMO_VARIANT(SimpleLoadDemo, THIS_DEMO_TYPE, "XML Tagfile",	3,	"Shows how to load a simple physics Xml tag file", helpString);
#endif

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
