/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Graphics/Common/Window/hkgWindow.h>
#include <Demos/Common/Api/MemoryWalk/MemoryWalk.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Common/Base/Memory/Allocator/FreeList/hkFreeListAllocator.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/DemoFramework/hkTextDisplay.h>
#include <Demos/DemoCommon/Utilities/GameUtils/GameUtils.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>

static const char helpString[] = \
"Demo showing memory allocation layout, through memory walk."
"\nRed - large blocks  Green - Small Blocks";

static const char noCollecthelpString[] = \
"Demo showing memory allocation layout, through memory walk."
"\nRed - large blocks  Green - Small Blocks"
"\nNotice memory fills with freelists (dark green), and therefore large blocks (red) get smaller over time.";

static const MemoryWalkDemo::DemoVariant s_variants[] =
{
	// name             filename                 details
	{ "Normal memory usage, with full garbage collection",  helpString, MemoryWalkDemo::DemoVariant::NORMAL_COLLECT, 0},
	{ "Incremental garbage collection", helpString, MemoryWalkDemo::DemoVariant::INCREMENTAL_COLLECT, 5},
	{ "No garbage collection", noCollecthelpString, MemoryWalkDemo::DemoVariant::INCREMENTAL_COLLECT, 0},
};

MemoryWalkDemo::MemoryWalkDemo(hkDemoEnvironment* env): hkDefaultPhysics2012Demo( env ),
	m_variant(s_variants[env->m_variantId]),
	m_rand(1000)
{
    m_numRows = 23;

    m_memorySize = 1024*1024;
    m_memoryStart = hkAllocateChunk<char>(m_memorySize, HK_MEMORY_CLASS_DEMO);

	// Set up the large block allocator
	m_largeBlockAllocator = hkMemHeapBlockAlloc<hkLargeBlockAllocator>(1);
	new (m_largeBlockAllocator) hkLargeBlockAllocator(m_memoryStart, m_memorySize);

	// Set up the freelist allocator
	m_memory = hkMemHeapBlockAlloc<hkFreeListAllocator>(1);
	new (m_memory) hkFreeListAllocator(m_largeBlockAllocator, m_largeBlockAllocator);

	m_memory->setMemorySoftLimit(m_memorySize);

	m_allocating = true;

	//
	// Setup the camera
	//
	{
		hkVector4 from(0.0f, 10.0f, 0.0f);
		hkVector4 to  (0.0f, 0.0f, 0.0f);
		hkVector4 up(0.0f, 0.0f, 1.0f);
		setupDefaultCameras( env, from, to, up, 1.f, 1000.0f );
	}

	//	Create the world
	{
		hkpWorldCinfo info;
		info.m_gravity.set( 0.0f, 0.0f, -10.0f );
		info.m_simulationType = hkpWorldCinfo::SIMULATION_TYPE_CONTINUOUS;

		m_world = new hkpWorld( info );
		m_world->lock();

		// Register ALL agents (though some may not be necessary)
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
		setupGraphics();
	}

	//
	// Create objects
	//

	m_world->unlock();

	m_width = m_env->m_window->getWidth();
	m_height = getWindowHeight();
	m_rowSize = m_memorySize / m_numRows;

	if (m_variant.m_operation == DemoVariant::INCREMENTAL_COLLECT)
	{
		// With the incremental, start with half of the memory filled
		_addAllocations(m_memorySize / 2, hkThreadMemory::MEMORY_MAX_SIZE_SMALL_BLOCK + 1);
	}
}


MemoryWalkDemo::~MemoryWalkDemo()
{
    m_memory->~hkFreeListAllocator();
	hkMemHeapBlockFree(m_memory, 1);

	m_largeBlockAllocator->~hkLargeBlockAllocator();
	hkMemHeapBlockFree(m_largeBlockAllocator, 1);

    hkDeallocateChunk(m_memoryStart, m_memorySize, HK_MEMORY_CLASS_DEMO);

	m_world->markForWrite();
	delete m_world;
	m_world = HK_NULL;
}

void MemoryWalkDemo::_alloc(hk_size_t size)
{
    void* alloc = m_memory->blockAlloc(int(size));
    if (alloc)
    {
        Block block;
        block.m_start = alloc;
        block.m_size = size;

        m_blocks.pushBack(block);

        /// zero the alloc
        hkString::memSet(alloc,0,(int)size);
    }
    else
    {
        /// Failed to alloc
    }
}

void MemoryWalkDemo::_free(int index)
{
    // We need to remove an allocation
    Block& block = m_blocks[index];
    m_memory->blockFree(block.m_start,int(block.m_size));
    m_blocks.removeAt(index);
}

void MemoryWalkDemo::_drawRowBlock(int row,int startX,int endX)
{
	hkgDisplayContext* context = m_env->m_displayHandler->getContext();

	float v0[4];
	float v1[4];
	float v2[4];
	float v3[4];

	float width = m_width-10.0f;

	//if (startX == endX) endX++;

	v0[2] = v1[2] = v2[2] = v3[2] = -0.01f; //magic z value

	float height = (m_height-10.0f)/m_numRows;

	float y0 = (row*height) + 5.0f;
	float y1 = y0 + (height*0.6f);

	float x0 = (((float)startX)*width/m_rowSize) + 5.0f;
	float x1 = (((float)endX)*width/m_rowSize) + 5.0f;

	v0[0] = x0;
	v0[1] = y0;

	v1[0] = x1;
	v1[1] = y0;

	v2[0] = x1;
	v2[1] = y1;

	v3[0] = x0;
	v3[1] = y1;

	context->setCurrentPosition( v0);
	context->setCurrentPosition( v1);
	context->setCurrentPosition( v2);

	context->setCurrentPosition( v0);
	context->setCurrentPosition( v2);
	context->setCurrentPosition( v3);
}

void MemoryWalkDemo::_drawAllocations()
{
	hkgDisplayContext* context = m_env->m_displayHandler->getContext();

	context->lock();
	context->beginGroup( HKG_IMM_TRIANGLE_LIST );

	for (int i=0;i<m_allocs.getSize();i++)
	{
		Allocation& alloc = m_allocs[i];

		hkUint32 separateBlocks = (alloc.m_start>>4)&0x7f;
		if (!alloc.m_allocated)
		{
			separateBlocks/=4;
		}

        if (alloc.m_pool ==0)
        {
			if (alloc.m_allocated)
				context->setCurrentColorPacked( 0xffff0000 | separateBlocks|(separateBlocks<<8));
			else
				context->setCurrentColorPacked( 0xff400000 | separateBlocks|(separateBlocks<<8));
        }
        else
        {
			if (alloc.m_pool <0)
			{
				if (alloc.m_allocated)
					context->setCurrentColorPacked( 0xff0000ff | (separateBlocks<<8)|(separateBlocks<<16));
				else
					context->setCurrentColorPacked( 0xff000040 | (separateBlocks<<8)|(separateBlocks<<16));
			}
			else
			{
				if (alloc.m_allocated)
					context->setCurrentColorPacked( 0xff00ff00 | separateBlocks|(separateBlocks<<16));
				else
					context->setCurrentColorPacked( 0xff004000 | separateBlocks|(separateBlocks<<16));
			}
			
        }

		int rowStart = alloc.m_start/m_rowSize;
		int startOffset = alloc.m_start%m_rowSize;
		int rowEnd = (alloc.m_start+alloc.m_size)/m_rowSize;
		int endOffset = (alloc.m_start+alloc.m_size)%m_rowSize;

		if (rowStart == rowEnd)
		{
			_drawRowBlock(rowStart,startOffset,endOffset);
		}
		else
		{
			_drawRowBlock(rowStart,startOffset,m_rowSize);
			// Middle bits
			for (int j=rowStart+1;j<=rowEnd-1;j++)
			{
				_drawRowBlock(j,0,m_rowSize);
			}
			_drawRowBlock(rowEnd,0,endOffset);
		}
	}
	context->endGroup();
	context->unlock();

}

void MemoryWalkDemo::_updateAllocations()
{
    int num = m_rand.getRand32();

    if ((num&15)>4)
    {
        // Add an allocation
        hk_size_t size = m_rand.getRand32()%50000;
        // do the allocation
        _alloc(size);
    }

    if (((num>>8)&15)>8)
    {
        // Add an allocation
		hk_size_t size = m_rand.getRand32()%( hkThreadMemory::MEMORY_MAX_SIZE_SMALL_BLOCK + 1 );
        // do the allocation
        _alloc(size);
    }


    if ((num&0x100)&&m_blocks.getSize()>0)
    {
        _free(m_rand.getRand32()%m_blocks.getSize());
    }

    num = (m_rand.getRand32())&0xffff;
    if (num>4000 && num<4002)
    {
        /// Free all
        while (m_blocks.getSize()) { _free(0); }
    }

    if (num>405 && num<500)
    {
        m_memory->garbageCollect();
    }

    /*if (num>20 && num<125)
    {
        m_memory->optimize();
    } */
}

void MemoryWalkDemo::_addAllocations(hk_size_t maxSize, hk_size_t maxAllocation)
{
	while (true)
	{
		hk_size_t size = m_rand.getRand32() % maxAllocation; 

		if (size > maxSize)
		{
			break;
		}

		_alloc(size);

		maxSize -= size;
	}
}

void MemoryWalkDemo::_normalUpdateAllocations()
{
	for (int i=0;i<100;i++)
	{
		_updateAllocations();
	}
}

void MemoryWalkDemo::_incrementalUpdateAllocations()
{
	const int numOps = 10;
	const int numAlloc = numOps + ((m_allocating) ? 1 : -1);
	const int numFree = numOps;

	for (int j = 0; j < 10; j++)
	{
		// Allocate
		{
			hk_size_t size = m_rand.getRand32()%( hkThreadMemory::MEMORY_MAX_SIZE_SMALL_BLOCK + 1 );
			// do the allocation
			for (int i = 0; i < numAlloc; i++)
			{
				if ((m_rand.getRand32() & 0xff) == 0x7f)
				{
					// do a large ish alloc 
					_alloc(m_rand.getRand32() % 50000);
				}
				else
				{
					_alloc(size);
				}
			}
		}

		// Free
		{
			for (int i = 0; i < numFree; i++)
			{
				if (m_blocks.getSize()>0)
				{
					_free(m_rand.getRand32()%m_blocks.getSize());
				}
			}
		}
	}

	if (m_allocating)
	{
		if (!m_memory->canAllocTotal(m_memorySize / 4))
		{
			m_allocating = false;
		}
	}
	else
	{
		if( m_memory->canAllocTotal(m_memorySize - (m_memorySize / 16)) )
		{
			m_allocating = true;
		}
	}
	
	// Do the collection
	if (m_variant.m_numIncrementalBlocks > 0)
	{
		m_memory->incrementalGarbageCollect(m_variant.m_numIncrementalBlocks);
	}
}


hkDemo::Result MemoryWalkDemo::stepDemo()
{
	switch (m_variant.m_operation)
	{
		case DemoVariant::NORMAL_COLLECT:
		{
			_normalUpdateAllocations();
			break;
		}
		case DemoVariant::INCREMENTAL_COLLECT:
		{
			_incrementalUpdateAllocations();
			break;
		}
	}

    m_allocs.clear();
    m_memory->walkMemory(_addBlock, this);

	_drawAllocations();

	hkDefaultPhysics2012Demo::stepDemo();

	m_world->lock();
	m_world->unlock();

	return DEMO_OK;
}


HK_DECLARE_DEMO_VARIANT_USING_STRUCT( MemoryWalkDemo, HK_DEMO_TYPE_PHYSICS_2012, MemoryWalkDemo::DemoVariant, s_variants, HK_NULL );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
