/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/DemoCommon/DemoFramework/hkDemo.h>
#include <Common/Base/Container/Tree/hkTree.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Algorithm/PseudoRandom/hkPseudoRandomGenerator.h>
#include <Graphics/Common/Font/hkgFont.h>

#include <Graphics/Bridge/ImGui/hkgImGui.h>

class ImGuiDemo : public hkDemo
{
	public:

		struct TreeNode
		{
			TreeNode() : open(0), expandable(false) {}
			hkStringPtr name;
			hkBool open;
			hkBool expandable;
		};
		typedef hkTree<TreeNode> Tree;

		ImGuiDemo(hkDemoEnvironment* env)
			: hkDemo(env)
		{
		}

		~ImGuiDemo()
		{
		}

		static void fillTree(Tree& tree, Tree::Iter iter, const char* path)
		{
			hkFileSystem& fs = hkFileSystem::getInstance();
			hkFileSystem::DirectoryListing listing( &hkMemoryRouter::getInstance().temp() );
			fs.listDirectory(path, listing);
			for( int i = 0; i < listing.getEntries().getSize(); ++i )
			{
				const hkFileSystem::Entry& e = listing.getEntries()[i];
				TreeNode n; n.name = e.getName(); n.expandable = e.isDir()!=hkFalse32; n.open = iter == HK_NULL;
				Tree::Iter it = tree.appendChild(iter, n);
				if( e.isDir() )
				{
					hkStringBuf sb; sb.printf("%s/%s", path, n.name.cString());
					fillTree(tree, it, sb);
				}
			}
		}

		Result stepDemo()
		{
			m_gui.step(m_env);
			return DEMO_OK;
		}

		struct ThemePage
		{
			ImGui::FlatPainter::Theme m_theme;
			int m_tableColWidths[5];

			ThemePage()
			{
				hkString::memSet(m_tableColWidths, 0, sizeof(m_tableColWidths));
			}

			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::VerticalBox vbox(m_context);
				// Draw the tweaking interface
				{
					ImTk::PaddedArea padded(m_context);
					padded.setPad(20,20,20,20);
					ImTk::Table table(m_context, m_tableColWidths, HK_COUNT_OF(m_tableColWidths) );

					ImTk::Label(m_context, "Example Widget");
					ImTk::Spacer(m_context, 0,2.5);
					ImTk::Label(m_context, "Text color (ARGB)");
					ImTk::Spacer(m_context, 0,2.5);
					ImTk::Label(m_context, "Background (ARGB)");

					struct Item { const char* label; ImGui::WidgetFlags state; ImGui::Argb* fgcolor; ImGui::Argb* bgcolor; };
					#define ITEM(WHAT,FG,BG) {(#WHAT " Widget"), ImGui::STATE_ ## WHAT, &m_theme.FG, &m_theme.BG }
					Item items[] =
					{
						ITEM(FILL, fg_fill, bg_fill),
						ITEM(NORMAL, fg_widget, bg_widget),
						ITEM(MOUSE_HOVER, fg_hover, bg_hover),
						ITEM(ACTIVE, fg_active, bg_active),
						ITEM(PRESSED, fg_pressed, bg_pressed),
					};
					#undef ITEM

					for( int i = 0; i < (int)HK_COUNT_OF(items); ++i )
					{
						Item& item = items[i];
						{	// preview of button in the given state to 
							ImGui::Layout& layout = m_context.layout();
							ImGui::Painter& painter = m_context.painter();
							ImGui::Pair textSize = painter.measureText(item.label);
							ImGui::Pair buttonSize = painter.measureButton(textSize);
							ImGui::Rectangle rbutton = layout.alloc(buttonSize);
							ImGui::Rectangle rtext = painter.paintButton(rbutton, item.state);
							painter.paintText(rtext, item.label, item.state);
						}
						ImTk::Spacer(m_context, 2, 2);
						ImTk::ColorSpinner(m_context,*item.fgcolor);
						ImTk::Spacer(m_context, 1, 2);
						ImTk::ColorSpinner(m_context,*item.bgcolor);
					}
				}
				{
					static ImGui::Argb color = 0x80808080;

					ImTk::HorizontalBox hbox(m_context);
					{
						ImTk::VerticalBox vboxl(m_context);
						for( int comp = 0 ; comp < 4; ++comp )
						{
							unsigned shift = 8*comp;
							ImGui::Painter& painter = m_context.painter();
					
                            const char* cr = "R\0G\0B\0A";
							ImTk::Label(m_context, cr+(comp*2) );
					
							hkArray<unsigned>::Temp argb; argb.setSize(256*20);
							for( int j = 0; j < 20; ++j )
							{
								for( int i = 0; i < 256; ++i )
								{
									unsigned mask = 0xff << shift;
									unsigned c = color & ~mask;
									argb[j*256 + i] = c + (i << shift);
								}
							}
							ImGui::Pair size = painter.measureImage(256,20);
							ImGui::Rectangle colorRect = m_context.layout().alloc( size );
							const ImGui::Painter::Texture* tex = painter.createTexture(argb.begin(), 256,20);
							painter.paintImage(colorRect, tex, 0xffffffff);

							ImGui::Rectangle grabRect = colorRect;
							grabRect.pos[0] += -1 + ((color >> shift) & 0xff);
							grabRect.size[0] = 2;
							painter.fillRectangle(grabRect, 0xffffffff);
												
							ImGui::WidgetFlags flags = ImGui::STATE_NORMAL;
							hkReal r = 0;
							m_context.doSpinner(0, colorRect, r, 1, flags);
							if( int(r) )
							{
								unsigned mask = 0xff << shift;
								unsigned val = hkMath::clamp( int( ((color >> shift)&0xff) - int(r) ), 0, 255);
								color = (color & ~mask) + (val << shift);
							}
						}
					}
					{
						ImGui::Painter& painter = m_context.painter();
					
						ImGui::Rectangle r = m_context.layout().alloc2(100,100);
						hkArray<unsigned>::Temp argb; argb.setSize(100*100);
						for( int j = 0; j < 100; ++j )
						{
							for( int i = 0; i < 100; ++i )
							{
								argb[j*100 + i] = color;
							}
						}
						const ImGui::Painter::Texture* tex = painter.createTexture(argb.begin(), 256,20);
						painter.paintImage(r, tex, 0xffffffff);
					}
				}
			}
		};

		struct ButtonsPage
		{
			hkBool m_toggle;
			ButtonsPage() : m_toggle(false) {  }
			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::PaddedArea padding( m_context );
				padding.setPad( 4,4,4,4 );
				ImTk::VerticalBox vertical(m_context);
				if( ImTk::Button(m_context, "toggle button (ctrl+x)").accelerator(HKG_VKEY_CONTROL, 'X').isActive(m_toggle) )
				{
					ImTk::Label(m_context, "hey very very long which only appears when parent is toggled");
				}
				if( ImTk::Button(m_context, "normal button (p)").accelerator('P').wasClicked() ) {}
				if( ImTk::Button(m_context, "normal button 2").wasClicked() ) {}
				{
				}

				{
					ImTk::Button special(m_context);
					{
						ImTk::HorizontalBox horizontal(m_context);  // make a button with image and text
						horizontal.setCentered(true);
						const int N = 44;
						static hkUint32 texture[N*N];
						if( texture[0] == 0 )
						{
							for(int j = 0; j < N; ++j )
							{
								for(int i = 0; i < N; ++i )
								{
									hkReal x= hkReal(i-N/2), y=hkReal(j-N/2);
									hkUint32 r = hkUint32( 255 * ((x*x + y*y) / (N*N)));
									texture[j*N+i] = (0xff<<24) | (r<<16) | (r<<8) | r;
								}
							}
						}
						ImTk::Image(m_context, texture, N,N, 0xffffffff);
						ImTk::Image(m_context, texture, N,N, 0xffffffff);
						ImTk::Label(m_context, "Button with an image and centered label");
					}
					if( special.wasClicked() )
					{
					}
				}
			}
		};

		struct ScrollPage
		{
			ImGui::ScrollState m_scrollState1[2];
			int m_repeats;
			hkBool m_toggle;

			ScrollPage() : m_repeats(10), m_toggle(false) {}

			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::VerticalBox vbox(m_context);
				ImTk::Label(m_context, "Scroll by dragging the scrollbar or the grabbing the panel directly");
				ImTk::ScrolledArea scrollArea(m_context, m_scrollState1);
				ImTk::VerticalBox vertical(m_context);
				for( int i = 0; i < m_repeats; ++i )
				{
					if( ImTk::Button(m_context, "toggle very wide").accelerator(HKG_VKEY_SHIFT, 'X').isActive(m_toggle) )
					{
						ImTk::Label(m_context, "hey very very long which label which should cause the horizontal scrollbar to kick in");
					}
					if( ImTk::Button(m_context, "click me to add items").wasClicked() )
					{
						m_repeats += 1;
					}
					if( ImTk::Button(m_context, "click to remove items").wasClicked() )
					{
						m_repeats -= 1;
					}
					m_repeats = hkMath::max2(2,m_repeats);
				}
			}
		};

		struct SpinnerPage
		{
			hkReal m_spin;
			hkReal m_spin2;
			ImGui::TextEditState m_textState;
			hkStringPtr m_text;

			SpinnerPage() : m_spin(0), m_spin2(0)
			{
				m_text = "text edit box";
			}

			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::VerticalBox mainPanel(m_context);
				static int count = 250; ++count;
				ImTk::Progress( m_context, hkMath::fabs( (count%500)/250.0f - 1.0f) );
				ImTk::Label(m_context, "");
				ImTk::Progress( m_context, hkMath::fabs( (count%500)/250.0f - 1.0f) ).text("with label");

				ImTk::Label(m_context, "");
				ImTk::Spinner(m_context, m_spin, 10);
				ImTk::Label(m_context, "");
				ImTk::Spinner(m_context, m_spin2, 1);

				ImTk::Label(m_context, ">");
				ImTk::TextEdit(m_context, m_text, m_textState); //todo size not correctly allocated
				ImTk::Label(m_context, "<");
			}
		};

		struct ExpanderPage
		{
			hkBool m_expanderOpen;
			int m_radio[2];

			ExpanderPage() : m_expanderOpen(false)
			{
				m_radio[0] = m_radio[1] = 0;
			}
			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::HorizontalBox vlayout(m_context);
				{
					ImTk::Expander expander(m_context, "expand", m_expanderOpen);
					//expander.accelerator(char('A'+i));
					if( expander.isOpen() )
					{
						ImTk::VerticalBox layout(m_context);
						ImTk::Label(m_context, "hello");
						ImTk::Label(m_context, "hello lonlaskdfalskdfasdf");
						if( ImTk::Button(m_context, "hello asdf").wasClicked() )
						{
						}
						ImTk::Label(m_context, "hello");
					}
				}

				for( int i = 0; i < (int)HK_COUNT_OF(m_radio); ++i )
				{
					ImTk::VerticalBox hpanel(m_context);
					for( int r = 0; r < 4; ++r )
					{
						hkStringBuf b; b.printf("option %i", r);
						ImTk::RadioButton(m_context, r, b.cString(), &m_radio[i]);
						if( i == 0 && m_radio[i] == r )
						{
							ImTk::Button(m_context, "Select").wasClicked();
						}
					}
					ImTk::Label(m_context, "");
				}
			}
		};


		struct TreeviewPage
		{
			hkStringPtr m_treeSelection;
			ImGui::ScrollState m_scrollState[2];
			Tree m_tree;

			TreeviewPage()
			{
				fillTree(m_tree, HK_NULL, "Resources");
			}

			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::VerticalBox vertical(m_context);
				if( m_treeSelection )
				{
					ImTk::Label(m_context, m_treeSelection);
				}
				ImTk::ScrolledArea scrollArea(m_context, m_scrollState);
				{
					ImTk::VerticalBox hpanel(m_context);

					Tree::Iter cur = m_tree.iterGetFirst();
					while( cur )
					{
						TreeNode& n = m_tree.getValue(cur);
						if( ImTk::Tree(m_context, n.name, m_tree.getDepth(cur), n.expandable, n.open).isOpen() )
						{
							if( n.expandable == false )
							{
								hkStringBuf sb;
								const char* sep = "";
								for( Tree::Iter i = cur; i != 0; i = m_tree.iterParent(i) )
								{
									sb.prepend( sep ); sep = "/";
									sb.prepend( m_tree.getValue(i).name );
								}
								m_treeSelection = sb;
							}
							cur = m_tree.iterNextPreOrder(cur);
						}
						else
						{
							cur = m_tree.iterNextPreOrderSkippingChildren(cur);
						}
					}
				}
			}
		};

		struct TablePage
		{
			hkBool m_expanded;
			int m_tableColWidths[2];
			hkReal m_spinners[4];
			TablePage() : m_expanded(true)
			{
				hkMemUtil::memSet(m_tableColWidths, 0, sizeof(m_tableColWidths));
				hkMemUtil::memSet(m_spinners, 0, sizeof(m_spinners));
			}

			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::Expander expander(m_context, "expand", m_expanded);
				if( expander.isOpen() )
				{
					ImTk::Table table(m_context, m_tableColWidths, HK_COUNT_OF(m_tableColWidths) );
					for( int i = 1; i < (int)HK_COUNT_OF(m_spinners); ++i )
					{
						hkStringBuf sb; sb.printf("item %i", i);
						for( int j = 0; j < i; ++j ) sb.append(".",1);
						ImTk::Label( m_context, sb);
						ImTk::Spinner(m_context, m_spinners[i], 1);
					}
					ImTk::Spinner(m_context, m_spinners[0], 1);
					hkStringBuf sb = "spin me ";
					for( int i = 0; i < (m_spinners[0]/10); ++i )
					{
						sb.appendPrintf("%c", 'a'+i);
					}
					ImTk::Label( m_context, sb.cString());
				}					
			}
		};

		struct FlowPage
		{
			hkReal m_padWidth;
			ImGui::FlowLayoutState m_flowState;

			FlowPage() : m_padWidth(10) {}

			void step(ImGui::Context& m_context)
			{
				namespace ImTk = ImGui::Widget;
				ImGui::FlowLayoutState state = m_flowState;
				ImTk::HorizontalBox hbox(m_context);
				// uncomment to see the update process
				//hkBool32 advanceWasClicked = ImTk::Button(m_context, "Advance").wasClicked();
				ImTk::Spinner(m_context, m_padWidth, 1); m_padWidth = hkMath::clamp(m_padWidth,hkReal(0),hkReal(300));
				ImTk::Spacer(m_context, m_padWidth, 0, 0x88888888);
				ImTk::FlowBox flow(m_context, &state);
				hkPseudoRandomGenerator prng(0);
				for( int i = 0; i < 64; ++i )
				{
					int n = prng.getRandChar(5);
					hkArray<char> dashes(n, '-'); dashes.pushBack(0);
					hkStringBuf sb; sb.printf("%s %i %s", dashes.begin(), i, dashes.begin());
					if( ImTk::Button(m_context, sb).wasClicked() )
					{
					}
				}
// 				if( advanceWasClicked )
// 				{
					m_flowState = state;
// 				}
			}
		};
		struct Gui
		{
			ImGui::Context m_context;
			ImGui::NotebookState m_notebookState;
			ThemePage m_themePage;
			ButtonsPage m_buttonsPage;
			ScrollPage m_scrollPage;
			SpinnerPage m_spinnerPage;
			ExpanderPage m_expanderPage;
			TreeviewPage m_treeviewPage;
			TablePage m_tablePage;
			FlowPage m_flowPage;

			Gui()
			{
				m_notebookState.activePage = 0;
			}

			void step(hkDemoEnvironment* m_env)
			{
				hkgImGui::Renderer renderer(m_env->m_window->getContext());
				ImGui::RenderBuffer buffer;
				ImGui::FlatPainter m_painter(&renderer, buffer);
				m_painter.m_theme = m_themePage.m_theme;
				m_context.setPainter(&m_painter);
				hkgImGui::sendInput(m_context, m_env->m_window);
				{
					ImGui::Rectangle wholeScreen(200,100, 900,500);
					//ImGui::Rectangle wholeScreen(0,0, m_env->m_window->getWidth(),getWindowHeight());
					doNoteBook(wholeScreen);
				}
				m_context.tick();
				renderer.render( buffer );
			}

			void doNoteBook(const ImGui::Rectangle& wholeScreen)
			{
				namespace ImTk = ImGui::Widget;
				ImTk::Notebook notebook(m_context, wholeScreen, m_notebookState);
				//ImTk::Notebook notebook(m_context, m_notebookState);

				if(notebook.isActivePage("Theme"))
				{
					m_themePage.step(m_context);
				}
				if(notebook.isActivePage("Buttons"))
				{
					m_buttonsPage.step(m_context);
				}
				if(notebook.isActivePage("Scrollbar"))
				{
					m_scrollPage.step(m_context);
				}
				if(notebook.isActivePage("Progress, Spinner"))
				{
					m_spinnerPage.step(m_context);
				}
				if(notebook.isActivePage("Expander, Radio"))
				{
					m_expanderPage.step(m_context);
				}
				if(notebook.isActivePage("Treeview"))
				{
					m_treeviewPage.step(m_context);
				}
				if(notebook.isActivePage("Table"))
				{
					m_tablePage.step(m_context);
				}
				if(notebook.isActivePage("Flow"))
				{
					m_flowPage.step(m_context);
				}
			}
		};

		virtual Result stepVisualDebugger() { return DEMO_OK; }
		virtual bool visualDebuggerEnabled() { return false; }
		virtual void makeFakeInput() { }

		Gui m_gui;
};


HK_DECLARE_DEMO(ImGuiDemo, HK_DEMO_TYPE_PHYSICS_2012, "", "");

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
