/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Test/Feature/Compression/Timings/SampleOnly/SampleOnlyTimingsDemo.h>
#include <Animation/Animation/Animation/Mirrored/hkaMirroredAnimation.h>
#include <Animation/Animation/Animation/Mirrored/hkaMirroredSkeleton.h>
#include <Animation/Animation/Animation/SplineCompressed/hkaSplineCompressedAnimation.h>
#include <Animation/Animation/Animation/Quantized/hkaQuantizedAnimation.h>
#include <Animation/Animation/Animation/PredictiveCompressed/hkaPredictiveCompressedAnimation.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/SampleAndBlend/hkaSampleBlendJob.h>
#include <Animation/Animation/Playback/SampleAndBlend/hkaSampleBlendJobQueueUtils.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

enum
{
	ANIMATION_TYPE_UNCOMPRESSED,
	ANIMATION_TYPE_SPLINE,
	ANIMATION_TYPE_QUANTIZED,
	ANIMATION_TYPE_PREDICTIVE,
	ANIMATION_TYPE_MIRRORED_SPLINE,
	NUM_ANIMATION_TYPES,

	FIRST_MT_VARIANT = NUM_ANIMATION_TYPES,

	MT_UNCOMPRESSED = FIRST_MT_VARIANT,
	MT_SPLINE,
	MT_QUANTIZED,
	MT_PREDICTIVE,
	MT_MIRRORED_SPLINE,
};

const hkReal TRANSLATION_TOLERANCE = 0.001f;
const hkReal ROTATION_TOLERANCE = 0.001f;
const hkReal SCALE_TOLERANCE = 0.001f;
const hkReal FLOAT_TOLERANCE = 0.001f;

const int NUM_CHARACTERS = 100;

const int CHAR_COLS = 10;
const int CHAR_ROWS = 10;
const hkReal CHAR_GAP = 2.0f;

static const char* ANIMATION_FILE_NAMES[] =
{
#if defined(HK_REAL_IS_DOUBLE)
	"cin_groggygetup_DP.hkt",
	"hkDive_DP.hkt",			// missing 1 bone
	"hkGetupBack1_DP.hkt",		// missing 1 bone
	"hkGetupFront1_DP.hkt",	// missing 1 bone
	"hkHardLand_DP.hkt",		// missing 1 bone
	"hkIdle1_DP.hkt",			// missing 1 bone
	"hkInAir_DP.hkt",			// missing 1 bone
	"hkRun_DP.hkt",
	"hkRunJump_DP.hkt",		// missing 1 bone
	"hkWalk_DP.hkt",
	"idl_cough1_DP.hkt",
	"idl_crouchidle_DP.hkt",
	"idl_plantC4_DP.hkt",
	"idl_snippet01_DP.hkt",
	"mov_crouchwalk_DP.hkt",
#else
	"cin_groggygetup.hkt",
	"hkDive.hkt",			// missing 1 bone
	"hkGetupBack1.hkt",		// missing 1 bone
	"hkGetupFront1.hkt",	// missing 1 bone
	"hkHardLand.hkt",		// missing 1 bone
	"hkIdle1.hkt",			// missing 1 bone
	"hkInAir.hkt",			// missing 1 bone
	"hkRun.hkt",
	"hkRunJump.hkt",		// missing 1 bone
	"hkWalk.hkt",
	"idl_cough1.hkt",
	"idl_crouchidle.hkt",
	"idl_plantC4.hkt",
	"idl_snippet01.hkt",
	"mov_crouchwalk.hkt",
#endif
};

const int NUM_ANIMS = sizeof(ANIMATION_FILE_NAMES) / sizeof(char*);
#if defined(HK_REAL_IS_DOUBLE)
static const char* RIG_FILE_NAME = "Resources/Animation/ShowCase/Model/Firefighter_Rig_No_Physics_DP.hkt";
#else
static const char* RIG_FILE_NAME = "Resources/Animation/ShowCase/Model/Firefighter_Rig_No_Physics.hkt";
#endif
static const char* ANIM_PATH = "Resources/Animation/FireFighter/Animations/";

struct MirrorPair
{
	const char* left;
	const char* right;
};

static MirrorPair MIRROR_PAIRS[] =
{
	{ "Left", "Right" },
	{ "L", "R" },
};

const int NUM_MIRROR_PAIRS = sizeof(MIRROR_PAIRS) / sizeof(MirrorPair);

SampleOnlyTimingsDemo::SampleOnlyTimingsDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env),
	m_totalSampleTimeSeconds(0.0f),
	m_numFrames(0)
{
	// Disable some compression report spew
	setErrorEnabled(0x432434a4, false);
	setErrorEnabled(0x36118e94, false);
	setErrorEnabled(0x3214badc, false);
	setErrorEnabled(0x9832bf32, false);
	setErrorEnabled(0x1e663ab1, false);

	//
	// Setup the camera
	//
	{
		hkVector4 from(  5.0f,  10.0f,  20.0f);
		hkVector4 to  (  0.0f,  -3.0f,   0.0f);
		hkVector4 up  (  0.0f,  1.0f,   0.0f);
		setupDefaultCameras( env, from, to, up );
	}

	// Get the rig
	{
		hkRootLevelContainer* container = hkSerializeUtil::loadObject<hkRootLevelContainer>( HK_GET_DEMOS_ASSET_FILENAME(RIG_FILE_NAME) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		m_skeleton = ac->m_skeletons[0];

		delete container;
	}

	m_numBytesUncompressed = 0;
	m_numBytesCompressed = 0;

	// Load the animations
	for( int i = 0; i < NUM_ANIMS; i++ )
	{
		hkStringBuf assetFile( ANIMATION_FILE_NAMES[i] );

		if ( ANIM_PATH != HK_NULL )
		{
			assetFile.prepend( ANIM_PATH );
		}

		hkRootLevelContainer* container = hkSerializeUtil::loadObject<hkRootLevelContainer>( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "No animation loaded");
		hkaInterleavedUncompressedAnimation* uncompressedAnimation = static_cast<hkaInterleavedUncompressedAnimation*> (ac->m_animations[0].val());

		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
		hkaAnimationBinding* binding = ac->m_bindings[0];

		int numTracks = binding->m_transformTrackToBoneIndices.getSize();
		numTracks = 0;

		int originalSize = uncompressedAnimation->getSizeInBytes();
		m_numBytesUncompressed += originalSize;

		switch(m_variantId % NUM_ANIMATION_TYPES)
		{
			case ANIMATION_TYPE_UNCOMPRESSED:
			{
				m_bindings.pushBack( binding );

				m_numBytesCompressed += originalSize;

				break;
			}
			case ANIMATION_TYPE_SPLINE:
			{
				hkaSplineCompressedAnimation::TrackCompressionParams p;
				hkaSplineCompressedAnimation::AnimationCompressionParams a;

				p.m_rotationTolerance = ROTATION_TOLERANCE;
				p.m_translationTolerance = TRANSLATION_TOLERANCE;
				p.m_scaleTolerance = SCALE_TOLERANCE;
				p.m_floatingTolerance = FLOAT_TOLERANCE;

				hkaSplineCompressedAnimation* sa = new hkaSplineCompressedAnimation( *uncompressedAnimation, p, a );
				binding->m_animation = sa;
				m_bindings.pushBack( binding );
				sa->removeReference();

				m_numBytesCompressed += sa->getSizeInBytes();

				break;
			}
			case ANIMATION_TYPE_MIRRORED_SPLINE:
			{
				hkaSplineCompressedAnimation* sa = HK_NULL;
				{
					hkaSplineCompressedAnimation::TrackCompressionParams p;
					hkaSplineCompressedAnimation::AnimationCompressionParams a;

					p.m_rotationTolerance = ROTATION_TOLERANCE;
					p.m_translationTolerance = TRANSLATION_TOLERANCE;
					p.m_scaleTolerance = SCALE_TOLERANCE;
					p.m_floatingTolerance = FLOAT_TOLERANCE;

					sa = new hkaSplineCompressedAnimation( *uncompressedAnimation, p, a );

					m_numBytesCompressed += sa->getSizeInBytes();
				}
				hkaAnimationBinding* mirroredBinding = makeMirroredAnimation( binding, sa );
				m_bindings.pushBack( mirroredBinding );
				mirroredBinding->removeReference();
				sa->removeReference();
				break;
			}
			case ANIMATION_TYPE_QUANTIZED:
			{
				hkaQuantizedAnimation::TrackCompressionParams p;

				p.m_rotationTolerance = ROTATION_TOLERANCE;
				p.m_translationTolerance = TRANSLATION_TOLERANCE;
				p.m_scaleTolerance = SCALE_TOLERANCE;
				p.m_floatingTolerance = FLOAT_TOLERANCE;

				hkaQuantizedAnimation* qa = new hkaQuantizedAnimation( *binding, *m_skeleton );
				binding->m_animation = qa;
				m_bindings.pushBack( binding );
				qa->removeReference();

				m_numBytesCompressed += qa->getSizeInBytes();

				break;
			}
			case ANIMATION_TYPE_PREDICTIVE:
			{
				hkaPredictiveCompressedAnimation::CompressionParams p(
					TRANSLATION_TOLERANCE,
					ROTATION_TOLERANCE,
					SCALE_TOLERANCE,
					FLOAT_TOLERANCE,
					TRANSLATION_TOLERANCE,
					0.0f, // a special case that tends to compress better
					SCALE_TOLERANCE,
					FLOAT_TOLERANCE );

				hkaPredictiveCompressedAnimation* pa = new hkaPredictiveCompressedAnimation( *binding, *m_skeleton, p );
				binding->m_animation = pa;
				m_bindings.pushBack( binding );
				pa->removeReference();

				m_numBytesCompressed += pa->getSizeInBytes();

				break;
			}
		}

		delete container;
	}

	// Create controls
	for (int i=0; i< NUM_CHARACTERS; i++)
	{
		hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl(m_bindings[i % NUM_ANIMS]);
		control->setLocalTime(i * 0.05f);
		m_controls.pushBack( control );
		control->removeReference();
	}

	hkaSampleBlendJobQueueUtils::registerWithJobQueue(m_jobQueue);

	setupGraphics();
}

hkaAnimationBinding* SampleOnlyTimingsDemo::makeMirroredAnimation(hkaAnimationBinding* binding, hkaAnimation* animation )
{
	hkArray<hkStringPtr> ltag;
	hkArray<hkStringPtr> rtag;

	for( int i = 0; i < NUM_MIRROR_PAIRS; i++ )
	{
		ltag.pushBack( MIRROR_PAIRS[i].left );
		rtag.pushBack( MIRROR_PAIRS[i].right );
	}

	hkaMirroredSkeleton *mirroredSkeleton = new hkaMirroredSkeleton( m_skeleton );

	mirroredSkeleton->computeBonePairingFromNames( ltag, rtag );

	hkQuaternion v_mir( 1.0f, 0.0f, 0.0f, 0.0f );
	mirroredSkeleton->setAllBoneInvariantsFromReferencePose( v_mir, 0.0f );

	hkaMirroredAnimation *mirroredAnimation = new hkaMirroredAnimation( animation, binding, mirroredSkeleton );

	mirroredSkeleton->removeReference();

	hkaAnimationBinding* mirroredBinding = mirroredAnimation->createMirroredBinding();

	mirroredAnimation->removeReference();

	return mirroredBinding;
}

void SampleOnlyTimingsDemo::sampleSt( hkaPose** poses )
{
	// This demo is intended to measure sampling only, where all the bones are present in the animation (identity binding).
	// But some of the Firefighter animations have the last bone missing, so we initialize the pose
	// to the reference pose to avoid a garbage bone.
	{
		hkQsTransform* refPose = m_skeleton->m_referencePose.begin();
		int numBones = m_skeleton->m_referencePose.getSize();

		for (int i = 0; i < NUM_CHARACTERS; ++i )
		{
			hkString::memCpy16( poses[i]->accessUnsyncedPoseLocalSpace().begin(), refPose, numBones * (sizeof(hkQsTransform) / 16) );
		}
	}

	HK_TIMER_BEGIN( "sampleSt", HK_NULL );

	// Sample animation on the PPU is single threaded mode
	for (int i = 0; i < NUM_CHARACTERS; ++i )
	{
		hkaAnimation* animation = m_controls[i]->getAnimationBinding()->m_animation;

		// N.B. We are putting track data directly into a pose, which necessarily assumes that the binding is the identity.
		// In general of course this will not be true, but for the purpose of this demo it is simplest to do this rather
		// than sample into a temporary array and track data and copy over into the bone (pose) data.
		animation->sampleTracks( m_controls[i]->getLocalTime(), poses[i]->accessUnsyncedPoseLocalSpace().begin(), poses[i]->getFloatSlotValues().begin() );
	}

	HK_TIMER_END();
}

void SampleOnlyTimingsDemo::sampleMt( hkaPose** poses )
{
	HK_TIMER_BEGIN_LIST( "sampleMt", "job setup" );

	int numThreads = m_threadPool->getNumThreads();

	int maxNumAnimsPerJob = (NUM_CHARACTERS + numThreads - 1) / numThreads;

	// We'll distribute the work evenly over the threads
	hkArray<hkJobQueue::JobQueueEntry> jobs( numThreads );
	jobs.setSize( numThreads );
	for( int i = 0; i < numThreads; i++ )
	{
		// We allocated JobQueEntrys to get the size right for the job queue (otherwise we might get a page fault)
		// so here we call the constructor for hkaSampleBlendJob.
		hkaSampleBlendJob* sbj = reinterpret_cast<hkaSampleBlendJob*>( &jobs[i] );
		new (sbj) hkaSampleBlendJob();
		sbj->initSampleOnly( maxNumAnimsPerJob, *m_skeleton );
	}

	for (int i=0 ; i < NUM_CHARACTERS; i++)
	{
		hkReal localTime = m_controls[i]->getLocalTime();
		const hkaAnimationBinding* binding = m_controls[i]->getAnimationBinding();
		hkaAnimation* animation = binding->m_animation;
		hkaSampleBlendJob* sbj = reinterpret_cast<hkaSampleBlendJob*>( &jobs[i%numThreads] );
		sbj->addAnimation(	*m_skeleton,
			*animation,
			localTime,
			poses[i]->accessUnsyncedPoseLocalSpace().begin(),
			poses[i]->getFloatSlotValues().begin(), binding );
	}

	HK_TIMER_SPLIT_LIST( "add jobs" );

	// for testing with fewer characters than threads
	int numJobs = hkMath::min2( numThreads, NUM_CHARACTERS );

	for (int s=0; s < numJobs; s++)
	{
		m_jobQueue->addJob( jobs[s], hkJobQueue::JOB_HIGH_PRIORITY );
	}

	HK_TIMER_SPLIT_LIST( "do jobs" );

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for the jobs to finish
	m_threadPool->waitForCompletion();

	// call destructors on jobs
	for( int i = 0; i < numThreads; i++ )
	{
		hkaSampleBlendJob* sbj = reinterpret_cast<hkaSampleBlendJob*>( &jobs[i] );
		sbj->~hkaSampleBlendJob();
	}

	HK_TIMER_END_LIST();
}

hkDemo::Result SampleOnlyTimingsDemo::stepDemo()
{
	for (int i=0; i< NUM_CHARACTERS; i++)
	{
		// Advance the animation - this happens in single threaded mode
		m_controls[i]->update( m_timestep );
	}

	// create output buffers
	hkLocalArray<hkaPose*> poses( NUM_CHARACTERS );
	poses.setSize( NUM_CHARACTERS );
	for (int p=0; p < NUM_CHARACTERS; p++)
	{
		poses[p] = new hkaPose(m_skeleton);
	}

	// Do the sampling and blending
	hkStopwatch timer;
	timer.start();

	#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
		// don't do MT if there are no threads in the pool
		bool mt = (m_variantId >= FIRST_MT_VARIANT) && (m_threadPool != HK_NULL) && (m_threadPool->getNumThreads() > 0);
	#else
		bool mt = false;
	#endif

	if ( mt )
	{
		sampleMt( poses.begin() );
	}
	else
	{
		sampleSt( poses.begin() );
	}

	timer.stop();
	hkReal seconds = timer.getSplitSeconds();
	m_totalSampleTimeSeconds += seconds;
	m_numFrames++;

	hkReal compressionRatio = hkReal(m_numBytesUncompressed) / hkReal(m_numBytesCompressed);
	int uSecAverage = (int)( (m_totalSampleTimeSeconds / m_numFrames) * 1000000 );

	// Render the pose buffers
	if( hkString::strNcmp( m_env->m_options->m_renderer, "n", 1) != 0 )
	{
		hkQsTransform worldFromModel (hkQsTransform::IDENTITY);
		for (int i = 0; i < NUM_CHARACTERS; ++i )
		{
			int r = i / CHAR_COLS;
			int c = i % CHAR_COLS;
			hkReal w = (CHAR_COLS - 1) * CHAR_GAP;
			hkReal h = (CHAR_ROWS - 1) * CHAR_GAP;
			hkReal x0 = -0.5f * w;
			hkReal y0 = -0.5f * h;
			hkReal x = x0 + CHAR_GAP * c;
			hkReal y = y0 + CHAR_GAP * r;
			worldFromModel.m_translation.set( x, 0, y );
			AnimationUtils::drawPose( *poses[i], worldFromModel );
		}

		hkStringBuf buf;
		int x = 20;
		int y = getWindowHeight() - 200;

		buf.printf( "%d characters", NUM_CHARACTERS );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "%d unique animations", NUM_ANIMS );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "Rig: %s (%d bones)", RIG_FILE_NAME, m_skeleton->m_bones.getSize() );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "Original animation memory for %d animations: %d KB", NUM_ANIMS, m_numBytesUncompressed / 1024 );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "Compressed animation memory for %d animations: %d KB", NUM_ANIMS, m_numBytesCompressed / 1024 );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "Compression ratio: %.1f", compressionRatio );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "Sample time for %d animations (uSecs): %d", NUM_CHARACTERS, (int)( seconds * 1000000 ) );
		m_env->m_textDisplay->outputText( buf, x, y );

		y += 20;
		buf.printf( "Average sample time for %d animations (uSecs): %d", NUM_CHARACTERS, uSecAverage );
		m_env->m_textDisplay->outputText( buf, x, y );

		hkReal uSecPerBone = hkReal(uSecAverage) / (NUM_CHARACTERS * m_skeleton->m_bones.getSize());
		y += 20;
		buf.printf( "Average sample time per bone (uSecs): %0.2f", uSecPerBone );
		m_env->m_textDisplay->outputText( buf, x, y );

		// just use first skeleton to display float data if exists
		/*for(int j = 0; j < m_skeleton->m_floatSlots.getSize(); j++)
		{
			hkStringBuf s;
			int boneIndex = m_bindings[0]->m_floatTrackToFloatSlotIndices[j];
			s.printf("%s : \t %f", m_skeleton->m_floatSlots[boneIndex].cString(),  poses[0]->getFloatSlotValue(j)  );
			m_env->m_textDisplay->outputText(s, 10, 50 + j * 15);
		}*/
	}

	// delete the poses
	for( int i = 0; i < NUM_CHARACTERS; i++ )
	{
		delete poses[i];
	}

	// print regression reports
	#if !defined(HK_DEBUG)

		if (m_numFrames == m_bootstrapIterations)
		{
			int animationType = (m_variantId % NUM_ANIMATION_TYPES);

			if ( ( animationType > ANIMATION_TYPE_UNCOMPRESSED ) && ( animationType <= ANIMATION_TYPE_PREDICTIVE ) )
			{
				const char* names[] =
				{
					"Spline",
					"Quantized",
					"Predictive"
				};

				const char* stOrMt = ( m_variantId < FIRST_MT_VARIANT ) ? "ST" : "MT";

				if ( m_variantId >= FIRST_MT_VARIANT )
				{
					HK_TRACE( "#threads is " << m_threadPool->getNumThreads() );
				}

				HK_REGRESSION_REPORT( "SampleOnlyTimingsDemo_" << stOrMt << "_" << names[animationType-ANIMATION_TYPE_SPLINE] << "_CompressionRatio", "ratio", compressionRatio );
				HK_REGRESSION_REPORT( "SampleOnlyTimingsDemo_" << stOrMt << "_" << names[animationType-ANIMATION_TYPE_SPLINE] << "_TimePerFrame", "us", uSecAverage );
			}
		}
	#endif

	return DEMO_OK;
}


static const char helpString[] = \
"Gathers timings for animation decompression." ;

#	define ST_STRING "Single-threaded"
#	define MT_STRING "Multi-threaded"

HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION, ST_STRING" - Uncompressed",		ANIMATION_TYPE_UNCOMPRESSED, ST_STRING" - Uncompressed", ST_STRING" - Uncompressed");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_STATS, ST_STRING" - Spline",			ANIMATION_TYPE_SPLINE, ST_STRING" - Spline", ST_STRING" - Spline");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_STATS, ST_STRING" - Quantized",			ANIMATION_TYPE_QUANTIZED, ST_STRING" - Quantized", ST_STRING" - Quantized");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_STATS, ST_STRING" - Predictive",		ANIMATION_TYPE_PREDICTIVE, ST_STRING" - Predictive", ST_STRING" - Predictive");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION, ST_STRING" - Mirrored Spline",	ANIMATION_TYPE_MIRRORED_SPLINE, ST_STRING" - Mirrored Spline", ST_STRING" - Mirrored Spline");

#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION, MT_STRING" - Uncompressed",		MT_UNCOMPRESSED, MT_STRING" - Uncompressed", MT_STRING" - Uncompressed");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_STATS, MT_STRING" - Spline",			MT_SPLINE, MT_STRING" - Spline", MT_STRING" - Spline");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_STATS, MT_STRING" - Quantized",			MT_QUANTIZED, MT_STRING" - Quantized", MT_STRING" - Quantized");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL | HK_DEMO_TYPE_STATS, MT_STRING" - Predictive",		MT_PREDICTIVE, MT_STRING" - Predictive", MT_STRING" - Predictive");
HK_DECLARE_DEMO_VARIANT(SampleOnlyTimingsDemo, HK_DEMO_TYPE_ANIMATION, MT_STRING" - Mirrored Spline",	MT_MIRRORED_SPLINE, MT_STRING" - Mirrored Spline", MT_STRING" - Mirrored Spline");
#endif

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
