/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/Showcase/RetargetingShowcaseDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Ik/TwoJoints/hkaTwoJointsIkSolver.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/Drawing/DrawUtils.h>
#include <Demos/DemoCommon/Utilities/SplashScreen/SplashScreen.h>
#include <Demos/DemoCommon/Utilities/WindowedAverage/WindowedAverage.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* ASSET_PATH = "Resources/Animation/Retargeting";
static const char* SCENE_NAME = "retargeting_scene";
static const int WINDOWED_AVERAGE_SIZE = 30;


struct AnimationMappingShowcaseDemoVariant
{
	const char* m_name;
	const char* m_details;
	int m_numCharacters;
	const char* m_characterName[ 10 ];
	const char* m_characterType[ 10 ];
	const char* m_animationType[ 10 ];
	hkReal m_motionScale[ 10 ];
	hkReal m_spacing;
	bool m_startWithExtractedMotion;
	bool m_useHandIK;
	const char* m_mapperName; // Optional
	hkReal cameraFrom[ 3 ];
	hkReal cameraTo[ 3 ];
};

#define HELP_STRING \
	"The driving character is indicated with a green diamond\n" \
	"Press \x16 or \x17 to switch the driving character\n" \
	"Press \x10 or H for further options"

static const AnimationMappingShowcaseDemoVariant g_variants[] =
{
	{
		"Characters of different shapes and sizes",
		"This demo shows retargeting between several unique characters.\n\n"
		HELP_STRING, 5,
		{ "male", "ninja", "girl", "boy", "female" },
		{ "nocape", "default", "default", "default", "ponytail" },
		{ "walk", "walk", "dive", "runjump", "walkbackwards" },
		{ 1.0f, 1.0f, 1.0f, 1.0f, 1.0f },
		1.0f, false, false, HK_NULL,
		{ 0.0f, -4.5f, 1.35f }, // Camera from
		{ 0.0f, 0.0f, 1.0f }   // Camera to
	},
	{
		"Dissimilar characters",
		"This demo shows retargeting between two very dissimilar characters.  "
		"Note the marked assymetry in the shape and bone structure of the monster.\n\n"
		HELP_STRING
		"\nShulgoth character copyright 2007 Flagship Studios", 2,
		{ "shulgoth", "male" },
		{ "default", "nocape" },
		{ "walk", "walk_slow" },
		{ 0.55f, 1.0f },
		2.5f, false, false, HK_NULL,
		{ 0.0f, -6.0f, 2.0f }, // Camera from
		{ 0.0f, 0.0f, 1.5f }  // Camera to
	},
	{
		"Attachments",
		"This demo shows retargeting characters with attached objects.  "
		"Both hands of the driving character lock onto its respective attached object.  "
		"Left hand contact with the attached object of the retargeted character is only approximate.  "
		"Model space hand IK can be used to ensure both hands firmly contact the attachment.  "
		"Press I To toggle Model Space Hand IK.\n\n"
		HELP_STRING, 2,
		{ "ninja", "singer" },
		{ "staff", "default" },
		{ "swing", "sing" },
		{ 1.0f, 1.0f },
		1.5f, true, true, HK_NULL,
		{ 0.0f, -3.5f, 1.3f }, // Camera from
		{ 0.0f, 0.0f, 0.8f }  // Camera to
	},
	{
		"Superset and Subset of Bones",
		"The character with a cape has a superset of bones of the other character.  "
		"When the without cape character is driving, the caped character's cape is filled in with the reference pose.  "
		"When the caped character is driving the without caped character is fully driven.\n\n"
		HELP_STRING, 2,
		{ "male", "male" },
		{ "nocape", "cape" },
		{ "walk", "walk" },
		{ 1.0f, 1.0f },
		2.0f, false, false, HK_NULL,
		{ 0.0f, -3.5f, 1.0f }, // Camera from
		{ 0.0f, 0.0f, 1.0f }  // Camera to
	},
	{
		"Similar Trunk Bones, Different Leaf Bones",
		"Both characters share a common trunk of bones.  "
		"Each character has additional leaf bones which are not shared.  "
		"Bones not shared are filled with the reference pose.  "
		"Switch the driving character to see the affect on the ponytail and dress bones.\n\n"
		HELP_STRING, 2,
		{ "female", "female" },
		{ "ponytail", "dress" },
		{ "walkbackwards", "walkbackwards" },
		{ 1.0f, 1.0f },
		2.0f, false, false, HK_NULL,
		{ 0.0f, -3.5f, 1.0f }, // Camera from
		{ 0.0f, 0.0f, 1.0f }  // Camera to
	},
	{
		"Extracted motion",
		"This demo shows the effect of character size on retargeted extraced motion.  "
		"The larger character moves faster and jumps higher than the smaller character.  "
		"Switch the driving character to 'steer' the characters on screen.\n\n"
		HELP_STRING, 2,
		{ "female", "girl" },
		{ "ponytailbig", "default" },
		{ "walkbig", "jumpturn" },
		{ 1.0f, 1.0f },
		2.0f, true, false, HK_NULL,
		{ 0.0f, -3.5f, 5.0f }, // Camera from
		{ 0.0f, 0.0f, 1.0f }  // Camera to
	},
	{
		"Different Resolutions (without chains)",
		"This demo shows retargeting between characters with different skeleton resolutions using simple retargeting mappings.  "
		"The singing character has upperarm, forearm, hand and finger bones.  "
		"The gingerbread man has only one bone in each arm and leg.  "
		"For each 'chain' of bones (upperarm..hand, spine bones, etc) only the base bone is retargeted.  "
		"Notice that the gingerbread man's hands only grossly approximate the motion of the singer's.\n\n"
		HELP_STRING, 2,
		{ "singer", "gingerbread" },
		{ "default", "default" },
		{ "sing", "walk" },
		{ 1.0f, 1.0f },
		2.0f, false, false, 0,
		{ 0.0f, -3.5f, 1.0f }, // Camera from
		{ 0.0f, 0.0f, 1.0f }  // Camera to
	},
	{
		"Different Resolutions (with chains)",
		"This demo shows retargeting between characters with different skeleton resolutions using chain maps.  "
		"The singing character has upperarm, forearm, hand and finger bones.  "
		"The gingerbread man has only one bone in each arm and leg.  "
		"Each chain of bones (upperarm..hand, thigh..foot, spine bones) is fully retargeted.  "
		"Compared to demo without chains, "
		"the gingerbread man's hands more closely approximate those of the singer.\n\n"
		HELP_STRING, 2,
		{ "singer", "gingerbread" },
		{ "default", "default" },
		{ "sing", "walk" },
		{ 1.0f, 1.0f },
		2.0f, false, false, "chain_mapping.hkt",
		{ 0.0f, -3.5f, 1.0f }, // Camera from
		{ 0.0f, 0.0f, 1.0f }  // Camera to
	},
	{
		"Talking",
		"This demo shows the ability to retarget characters' facial animations.\n\n"
		HELP_STRING, 2,
		{ "havokgirl", "bigbad" },
		{ "default", "default" },
		{ "talk", "Yell" },
		{ 1.0f, 1.0f },
		1.0f, false, false, HK_NULL,
		{ 0.0f, -1.75f, 1.6f }, // Camera from
		{ 0.0f, 0.0f, 1.6f }  // Camera to
	}
};

AnimationMappingShowcaseDemo::AnimationMappingShowcaseDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Access the variant settings
	const AnimationMappingShowcaseDemoVariant& variant = g_variants[ m_env->m_variantId ];

	// Set defaults
	m_speed = 1.0f;
	m_drawSkins = true;
	m_drawBoneNames = false;
	m_useReferencePose = false;
	m_useHandIK = variant.m_useHandIK;
	m_activeCharacter = 0;
	m_useExtractedMotion = variant.m_startWithExtractedMotion;
	m_paused = false;
	m_useWireFrame = false;
	m_showTimings = false;

	// Used for smoothing out timings
	m_windowedAverageRetargeting = new WindowedAverage( WINDOWED_AVERAGE_SIZE );
	m_windowedAverageSampling = new WindowedAverage( WINDOWED_AVERAGE_SIZE );

	m_splashScreen = new SplashScreen( this, SplashScreen::ANIMATION_SPLASH );

	// Enable the shader library
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );

	m_loader = new hkLoader();

	// Disable detailed reporting of matched bones 
	setErrorEnabled( 0x54E4323E, false );

	// Setup the camera
	{
		hkVector4 from( variant.cameraFrom[ 0 ], variant.cameraFrom[ 1 ], variant.cameraFrom[ 2 ] );
		hkVector4 to  ( variant.cameraTo[ 0 ], variant.cameraTo[ 1 ], variant.cameraTo[ 2 ] );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}

	m_splashScreen->setMessage( "Loading assets" );

	// For each character
	for ( int i = 0; i < variant.m_numCharacters; i++ )
	{
		// Create a new mapped character data structure
		m_characters.pushBack( new MappedCharacter );

		hkStringBuf rigName( ASSET_PATH, "/", variant.m_characterName[ i ], "_", variant.m_characterType[ i ], ".hkt" );
		hkStringBuf animationName( ASSET_PATH, "/", variant.m_characterName[ i ], "_", variant.m_characterType[ i ], "_" );
		animationName += variant.m_animationType[ i ];
		animationName += ".hkt";

		// Initialize the world from model for each character
		initWorldFromModel( i );
		
		// Get the rig
		{
			hkStringBuf assetFile = rigName;
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT3( 0x27343437, container != HK_NULL , "Could not load asset: " << assetFile );
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );

			m_characters[ i ]->m_animatedSkeleton = new hkaAnimatedSkeleton( ac->m_skeletons[0] );

			// this will speed things up since we don't need to blend in the reference pose
			m_characters[ i ]->m_animatedSkeleton->setReferencePoseWeightThreshold( 0.0f );
		}

		// Get the animations and the bindings
		{
			hkStringBuf assetFile = animationName;
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
			HK_ASSERT2( 0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No binding loaded");

			// Grab the binding
			hkaAnimationBinding* binding = ac->m_bindings[0];

			// Check that this animation contains an appropriate number of tracks
			HK_ON_DEBUG( const hkaSkeleton* skeleton = m_characters[ i ]->m_animatedSkeleton->getSkeleton() );
			HK_ASSERT3( 0x160c45fe, binding->m_transformTrackToBoneIndices.getSize() == skeleton->m_bones.getSize(), "\n\n**********\nNumber of bones in animation does not match the number in the rig.\nLoading: " << assetFile.cString() << "\n**********\n\n" );

			// Create a new animation control
			hkaAnimationControl* control = new hkaDefaultAnimationControl( binding );
			m_characters[ i ]->m_animatedSkeleton->addAnimationControl( control );
			control->removeReference();

			// Determine the character height in the reference pose
			{
				hkaPose refPose( m_characters[ i ]->m_animatedSkeleton->getSkeleton() );

				refPose.setToReferencePose();

				hkVector4 refMin;
				hkVector4 refMax;

				hkaSkeletonMapperUtils::ExtractedMotionParams::getPoseMinMax( refPose.getSyncedPoseModelSpace().begin(), refPose.getSkeleton()->m_bones.getSize(), refMin, refMax );

				m_characters[ i ]->m_characterHeight = refMax( 2 );
			}
		}

		// Convert the skin
		{
			hkStringBuf assetFile = rigName;
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );
			HK_ASSERT2( 0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );
			HK_ASSERT2( 0x27343435, ac && (ac->m_skins.getSize() > 0 ), "No skins loaded");

			m_characters[ i ]->m_skinBindings = ac->m_skins;
			m_characters[ i ]->m_attachments = ac->m_attachments;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );

			// Create meshes for attachment objects
			AnimationUtils::createGraphicsMeshesForAttachments( m_env, m_characters[ i ]->m_attachments, m_characters[ i ]->m_attachmentObjects );
		}

		m_splashScreen->updateProgress( hkReal( i + 1 ) / hkReal( variant.m_numCharacters ) * 90.0f );
	}

	// Load the floor
	{
		hkStringBuf assetFile( ASSET_PATH, "/", SCENE_NAME, ".hkt" );
		hkAssetManagementUtil::getFilePath( assetFile );
		
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2( 0x0729b5e1, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		HK_ASSERT2( 0x1c0160cc, scene, "No scene loaded" );
		removeLights( m_env );
		env->m_sceneConverter->convert( scene );
	}

	m_splashScreen->setMessage( "Creating Retargeting Mappers" );

	// Now set up the maps
	{
		// Create a mapper at runtime

		hkaSkeletonMapperUtils::Params params;

		params.m_autodetectChains = false;
		params.m_autodetectSimple = true;
		params.m_compareNames = HK_NULL;
		params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
		params.m_positionMatchTolerance = 1.0e-4f;

		// For every character...
		for ( int i = 0; i < variant.m_numCharacters; i++ )
		{
			// For all other characters...
			for ( int j = 0; j < variant.m_numCharacters; j++ )
			{
				if ( variant.m_mapperName == HK_NULL || i == j )
				{
					// Create a new hkaSkeletonMapperData
					m_characters[ i ]->m_map.pushBack( new hkaSkeletonMapperData );

					params.m_skeletonA = m_characters[ i ]->m_animatedSkeleton->getSkeleton();
					params.m_skeletonB = m_characters[ j ]->m_animatedSkeleton->getSkeleton();

					// Use custom extracted motion scaling
					params.m_motionParams.m_extractedMotionFactorA = variant.m_motionScale[ i ];
					params.m_motionParams.m_extractedMotionFactorB = variant.m_motionScale[ j ];

					hkaSkeletonMapperData* mapperData = m_characters[ i ]->m_map[ j ];

					// For the purposes of this demo, only the aToB mapping is used
					hkaSkeletonMapperData unusedData;

					hkaSkeletonMapperUtils::createMapping( params, *mapperData, unusedData );
				}
				else
				{
					// Load a mapper from the specified asset
					HK_ASSERT2( 0x0175c92e, variant.m_numCharacters == 2, "This demo can only load a mapper for two characters" );

					hkStringBuf assetFile( ASSET_PATH, "/", variant.m_mapperName );
					hkAssetManagementUtil::getFilePath( assetFile );

					hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
					HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

					hkaSkeletonMapper* mapper = reinterpret_cast< hkaSkeletonMapper* >( container->findObjectByType( hkaSkeletonMapperClass.getName() ) );
					HK_ASSERT2( 0x1d332f6d, mapper != HK_NULL, "No mapper found" );

					if ( i )
					{
						mapper = reinterpret_cast< hkaSkeletonMapper* >( container->findObjectByType( hkaSkeletonMapperClass.getName(), mapper ) );
					}

					m_characters[ i ]->m_map.pushBack( &mapper->m_mapping );
				}
			}

			m_splashScreen->updateProgress( hkReal( i + 1 ) / hkReal( variant.m_numCharacters ) * 100.0f );
		}
	}

	// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// We want to see shadows in this demo
	forceShadowState( true );

	// Setup the graphics
	setupGraphics();

	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();

		m_env->m_window->setShadowMapSize( 1024 ); // Use high quality shadows

		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		hkDefaultDemo::setupFixedShadowFrustum( m_env, *l, shadowAabb );
	}
}

AnimationMappingShowcaseDemo::~AnimationMappingShowcaseDemo()
{
	const AnimationMappingShowcaseDemoVariant& variant = g_variants[ m_env->m_variantId ];

	// Delete all allocated memory
	for ( int i = 0; i < variant.m_numCharacters; i++ )
	{
		for ( int j = 0; j < variant.m_numCharacters; j++ )
		{
			if ( variant.m_mapperName == HK_NULL || i == j )
			{
				delete m_characters[ i ]->m_map[ j ];
			}
			else
			{
				
			}
		}

		m_characters[ i ]->m_animatedSkeleton->removeReference();

		delete m_characters[ i ];
	}

	delete m_splashScreen;

	delete m_loader;

	delete m_windowedAverageRetargeting;
	delete m_windowedAverageSampling;
}

hkDemo::Result AnimationMappingShowcaseDemo::stepDemo()
{
	const AnimationMappingShowcaseDemoVariant& variant = g_variants[ m_env->m_variantId ];

	// Check user input
	{
		const hkgKeyboard& keyboard = m_env->m_window->getKeyboard();

		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
		{
			// Increment or decrement the active character
			m_activeCharacter += m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) ? -1 : 1;
			m_activeCharacter += variant.m_numCharacters;
			m_activeCharacter %= variant.m_numCharacters;

			// Restart all animations at the beginning
			if ( m_useExtractedMotion )
			{
				for ( int i = 0; i < variant.m_numCharacters; i++ )
				{
					hkaAnimationControl* control = m_characters[ m_activeCharacter ]->m_animatedSkeleton->getAnimationControl( 0 );
				
					control->setLocalTime( 0.0f );
				}
			}
		}

		// Force the reference pose
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP ) || keyboard.wasKeyPressed( 'R' ) )
		{
			m_useReferencePose = !m_useReferencePose;

//			m_speed = m_useReferencePose ? 0.0f : 1.0f;
		}

		// Draw/undraw the character bones
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) || keyboard.wasKeyPressed( 'B' ) )
		{
			m_drawSkins = !m_drawSkins;
		}

		// Display help
		if ( m_env->m_gamePad->isButtonPressed( HKG_PAD_BUTTON_0 ) || keyboard.getKeyState( 'H' ) )
		{
			showHelp();
		}

		// Toggle extracted motion
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_1 ) || keyboard.wasKeyPressed( 'E' ) )
		{
			m_useExtractedMotion = !m_useExtractedMotion;
		}

		// Reset world from model
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_2 ) || keyboard.wasKeyPressed( 'O' ) )
		{
			for ( int i = 0; i < variant.m_numCharacters; i++ )
			{
				initWorldFromModel( i );
			}
		}

		// Pause/Unpause
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_3 ) || keyboard.wasKeyPressed( 'P' ))
		{
			m_speed = m_speed > 0.0f ? 0.0f : 1.0f;
		}

		// Show timings
		if ( keyboard.wasKeyPressed( 'T' ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_L1 ) )
		{
			m_showTimings = !m_showTimings;
		}

		// Draw/undraw bone names
		if ( keyboard.wasKeyPressed( 'N' ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_R1 ) )
		{
			m_drawBoneNames = !m_drawBoneNames;
		}

		// Slow down
		if ( keyboard.wasKeyPressed( 'K' ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_L2 ) )
		{
			m_speed /= 1.5f;
		}

		// Speed up
		if ( keyboard.wasKeyPressed( 'L' ) || m_env->m_gamePad->wasButtonPressed( HKG_PAD_BUTTON_R2 ) )
		{
			m_speed *= 1.5f;
		}

		// Wireframe display
		if ( keyboard.wasKeyPressed( 'F' ) )
		{
			m_useWireFrame = !m_useWireFrame;
			setGraphicsState( HKG_ENABLED_WIREFRAME, m_useWireFrame );
		}

		if ( keyboard.wasKeyPressed( 'I' ) && variant.m_useHandIK )
		{
			m_useHandIK = !m_useHandIK;
		}
	}

	// Label the active character
	{
		hkVector4 activePos = m_characters[ m_activeCharacter ]->m_worldFromModel.m_translation;

		const hkReal diamondSize = 0.15f * m_characters[ m_activeCharacter ]->m_characterHeight;

		activePos( 2 ) += m_characters[ m_activeCharacter ]->m_characterHeight + 1.5f * diamondSize;

		hkVector4 p[ 4 ] = { activePos, activePos, activePos, activePos };

		p[ 0 ]( 0 ) += diamondSize;
		p[ 1 ]( 0 ) -= diamondSize;
		p[ 2 ]( 2 ) += diamondSize;
		p[ 3 ]( 2 ) -= diamondSize;

		hkColor::Argb color = hkColor::rgbFromChars( 64, 255, 64, 200 );
		
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 2 ], color );
		HK_DISPLAY_TRIANGLE( p[ 0 ], p[ 1 ], p[ 3 ], color );
	}

	// Advance time
	for ( int i = 0; i < variant.m_numCharacters; i++ )
	{
		m_characters[ i ]->m_animatedSkeleton->stepDeltaTime( m_timestep * m_speed );
	}

	hkaPose srcPose( m_characters[ m_activeCharacter ]->m_animatedSkeleton->getSkeleton() );

	// Timing data
	hkReal retargetingTime = 0.0f;
	hkReal samplingTime = 0.0f;

	// Sample the pose of the master character
	if ( m_useReferencePose )
	{
		srcPose.setToReferencePose();
	}
	else
	{
		hkStopwatch stopwatch;
		stopwatch.start();
		
		m_characters[ m_activeCharacter ]->m_animatedSkeleton->sampleAndCombineAnimations( srcPose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

		stopwatch.stop();
		samplingTime += stopwatch.getSplitSeconds() * 1000000;
	}

	// Handle extracted motion
	hkQsTransform srcExtractedMotion( hkQsTransform::IDENTITY );
	if ( m_useExtractedMotion )
	{
		m_characters[ m_activeCharacter ]->m_animatedSkeleton->getDeltaReferenceFrame( m_timestep * m_speed, srcExtractedMotion );
	}

	// Find the pose for each character
	for ( int i = 0; i < variant.m_numCharacters; i++ )
	{
		const hkaSkeletonMapperData* mapping = m_characters[ m_activeCharacter ]->m_map[ i ];

		hkaSkeletonMapper mapper( *mapping );

		hkaPose dstPose( m_characters[ i ]->m_animatedSkeleton->getSkeleton() );

		// Map the pose from the master character to the current character
		if ( i != m_activeCharacter )
		{
			// Initialize the dst character to the reference pose
			dstPose.setToReferencePose();

			hkStopwatch stopwatch;
			stopwatch.start();

			// Map the character pose
			mapper.mapPose( srcPose, dstPose );

			stopwatch.stop();

			retargetingTime += stopwatch.getSplitSeconds() * 1000000;
		}
		else
		{
			// No need to map the src character to itself
			dstPose.setPoseLocalSpace( srcPose.getSyncedPoseLocalSpace() );
		}

		HK_ASSERT2( 0x3f124035, srcPose.checkPoseTransformsValidity(), "Source pose is invalid." );
		HK_ASSERT2( 0xab3ff41c, dstPose.checkPoseTransformsValidity(), "Destination pose is invalid." );

		// Handle Hand IK
		if ( variant.m_useHandIK )
		{
			if ( m_useHandIK )
			{
				const hkaSkeleton* srcSkeleton = srcPose.getSkeleton();
				const hkaSkeleton* dstSkeleton = dstPose.getSkeleton();

				srcPose.syncModelSpace();
				dstPose.syncModelSpace();

				for ( int hand = 0; hand < 2; hand++ )
				{
					const hkInt16 dstTargetBone = hkaSkeletonUtils::findBoneWithName( *dstSkeleton, hand == 0 ? "StaffGrip" : "Staff" );
					const hkInt16 srcTargetBone = hkaSkeletonUtils::findBoneWithName( *srcSkeleton, hand == 0 ? "StaffGrip" : "Staff" );

					const hkInt16 dstHandBone = hkaSkeletonUtils::findBoneWithName( *dstSkeleton, hand == 0 ? "Bip01 L Hand" : "Bip01 R Hand" );
//				const hkInt16 srcHandBone = hkaSkeletonUtils::findBoneWithName( *srcSkeleton, hand == 0 ? "Bip01 L Hand" : "Bip01 R Hand" );

					const hkInt16 dstElbowBone = hkaSkeletonUtils::findBoneWithName( *dstSkeleton, hand == 0 ? "Bip01 L Forearm" : "Bip01 R Forearm" );
					const hkInt16 dstShoulderBone = hkaSkeletonUtils::findBoneWithName( *dstSkeleton, hand == 0 ? "Bip01 L UpperArm" : "Bip01 R UpperArm" );
					const hkInt16 dstUpperTwist = hkaSkeletonUtils::findBoneWithName( *dstSkeleton, hand == 0 ? "Bip01 LUpArmTwist" : "Bip01 RUpArmTwist" );
					const hkInt16 dstForeTwist = hkaSkeletonUtils::findBoneWithName( *dstSkeleton, hand == 0 ? "Bip01 L ForeTwist" : "Bip01 R ForeTwist" );

					hkaTwoJointsIkSolver::Setup setup;
					setup.m_endTargetMS = srcPose.getBoneModelSpace( srcTargetBone ).m_translation;
					setup.m_endTargetRotationMS = srcPose.getBoneModelSpace( srcTargetBone ).m_rotation;
					setup.m_firstJointIdx = dstShoulderBone;
					setup.m_secondJointIdx = dstElbowBone;
					setup.m_endBoneIdx = dstHandBone;
					setup.m_hingeAxisLS.set( 0.0f, 0.0f, 1.0f );
					setup.m_enforceEndRotation = true;
					setup.m_firstJointTwistIdx = dstUpperTwist;
					setup.m_secondJointTwistIdx = dstForeTwist;
					hkQsTransform offset = dstPose.getBoneLocalSpace( dstTargetBone );
					setup.m_endBoneOffsetLS = offset.m_translation;
					setup.m_endBoneRotationOffsetLS = offset.m_rotation;

					hkQsTransform mytmp = srcPose.getBoneModelSpace( srcTargetBone );

					if ( i != m_activeCharacter )
					{
						setup.m_endTargetMS.mul4( mapping->m_extractedMotionMapping.m_scale );
					}

					hkaTwoJointsIkSolver::solve( setup, dstPose );
				}

				m_env->m_textDisplay->outputText( "Hand IK ON", 50u, getWindowHeight() - 40 );
			}
			else
			{
				m_env->m_textDisplay->outputText( "Hand IK OFF", 50u, getWindowHeight() - 40 );
			}
		}

		// Handle worldFromModel
		{
			hkQsTransform root = m_characters[ i ]->m_worldFromModel;
			root.setMulEq( dstPose.getBoneLocalSpace( 0 ) );
			dstPose.setBoneLocalSpace( 0, root );
		}

		// Handle extracted motion
		{
			hkQsTransform dstExtractedMotion;

			mapper.mapExtractedMotion( srcExtractedMotion, dstExtractedMotion );

			m_characters[ i ]->m_worldFromModel.setMulEq( dstExtractedMotion );
			m_characters[ i ]->m_worldFromModel.m_rotation.normalize();
		}

		// Draw bone names
		if ( m_drawBoneNames )
		{
			const hkaSkeleton* skeleton = m_characters[ i ]->m_animatedSkeleton->getSkeleton();

			dstPose.syncAll();

			hkInt32 numBones = skeleton->m_bones.getSize();

			for ( hkInt32 bone = 0; bone < numBones; bone++ )
			{
				hkVector4 vec = dstPose.getBoneModelSpace( bone ).m_translation;

				m_env->m_textDisplay->outputText3D( skeleton->m_bones[ bone ].m_name, float(vec( 0 )), float(vec( 1 )), float(vec( 2 )) );
			}
		}

		// Handle the drawing of skins or skeletons
		if ( !m_drawSkins )
		{
			hkColor::Argb skeletonColor = hkColor::rgbFromChars( 0, 0, 0, 255 );
			
			// Draw the skeleton
			AnimationUtils::drawPose( dstPose, hkQsTransform::IDENTITY, skeletonColor );

			if ( !m_useWireFrame )
			{
				// Move the skins off-camera
				hkQsTransform mover = hkQsTransform::IDENTITY;
				mover.m_translation( 2 ) -= 5.0f;
				mover.setMulEq( dstPose.getBoneLocalSpace( 0 ) );
				dstPose.setBoneLocalSpace( 0, mover );
			}

			// Label the number of bones
			{
						
				hkStringBuf message;
				message.printf( "%d Bones", m_characters[ i ]->m_animatedSkeleton->getSkeleton()->m_bones.getSize() );

				hkVector4 activePos = m_characters[ i ]->m_worldFromModel.m_translation;
				activePos( 2 ) += m_characters[ i ]->m_characterHeight + 0.1f;

				hkColor::Argb color = hkColor::rgbFromChars( 0, 0, 0, 255 );

				m_env->m_textDisplay->outputText3D( message.cString(), float(activePos( 0 )), float(activePos( 1 )) - 0.01f, float(activePos( 2 )), color );
			}
		}

		// Draw the skins and attachments
		AnimationUtils::skinMesh( dstPose, m_characters[ i ]->m_skinBindings, hkTransform::getIdentity(), *m_env->m_sceneConverter );
		AnimationUtils::drawAttachments( m_env, dstPose.getSyncedPoseModelSpace().begin(), m_characters[ i ]->m_attachments, m_characters[ i ]->m_attachmentObjects );
	}

	// Draw the mapping timings
	if ( m_showTimings )
	{
		m_windowedAverageRetargeting->pushNewValue( retargetingTime );
		const hkReal meanRetargeting = m_windowedAverageRetargeting->getWindowedMean();

		m_windowedAverageSampling->pushNewValue( samplingTime );
		const hkReal meanSampling = m_windowedAverageSampling->getWindowedMean();

		hkStringBuf str;
		str.printf( "Avg. Retargeting Time: %7.3fus per char  (%d Chars, %7.3fus total)\n"
					"Sample & Combine Time: %7.3fus  (1 Character)",
					meanRetargeting / hkReal( variant.m_numCharacters - 1 ),
					variant.m_numCharacters - 1, meanRetargeting,
					meanSampling );

		const int h = getWindowHeight();

		m_env->m_textDisplay->outputText( str, 20, h - 40 );
	}	

	return hkDefaultAnimationDemo::stepDemo();
}


hkInt32 AnimationMappingShowcaseDemo::findBoneWithName( const hkaSkeleton* skeleton, const char* name )
{
	const int n = skeleton->m_bones.getSize();

	for ( int i = 0; i < n; i++ )
	{
		const char* boneName = skeleton->m_bones[ i ].m_name;

		if ( hkString::strCmp( name, boneName ) == 0 )
		{
			return i;
		}
	}

	return -1;
}

void AnimationMappingShowcaseDemo::initWorldFromModel( int i )
{
	// Access the variant settings
	const AnimationMappingShowcaseDemoVariant& variant = g_variants[ m_env->m_variantId ]; 

	m_characters[ i ]->m_worldFromModel.setIdentity();
	
	const hkReal pos = hkReal( variant.m_numCharacters - 1 ) / 2.0f - hkReal( i );

	m_characters[ i ]->m_worldFromModel.m_translation.set( -pos * variant.m_spacing, 0, 0, 0 );
}

void AnimationMappingShowcaseDemo::showHelp()
{
	const char* helpstr =
		"\x16   - Select left animation    \x14/R - Toggle reference pose\n"
		"\x17   - Select right animation   \x15/B - Toggle skins\n"
		"\x10/H - Show help                \x11/E - Toggle extracted motion\n"
		"O   - Reset to Origin          \x13/P - Pause\n"
		"\x1B/T - Show Timing Info         \x1C/N   - Draw Bone Names\n"
		"\x1D/K - Slow down animations     \x1D/L - Speed up animations\n"
		"F   - Draw Wireframe           I   - Use Hand IK\n";
	
	m_env->m_textDisplay->outputText( helpstr, 10, 100 );
}


// runs out of memory on ps3 so only bootstrap on win32
HK_DECLARE_DEMO_VARIANT_USING_STRUCT( AnimationMappingShowcaseDemo, HK_DEMO_TYPE_ANIMATION, AnimationMappingShowcaseDemoVariant, g_variants, "Retargeting Character Examples and Use Cases" );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
