/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

// Demo Includes
#include <Demos/demos.h>
#include <Demos/Animation/Api/Retargeting/Blend/BlendedRetargetingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Playback/SampleAndBlend/hkaSampleBlendJob.h>
#include <Animation/Animation/Playback/SampleAndBlend/hkaSampleBlendJobQueueUtils.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Graphics/Common/DisplayWorld/hkgDisplayWorld.h>
#include <Graphics/Common/Light/hkgLightManager.h>

static const char* HELP_STRING =
	"This demo shows the ability to blend retargeted and standard animations.  "
	"Intially each character plays its own animation.  "
	"Change the blend weights to blend in more of the other character's animation.  "
	"At 100% retargeted the animations will be completely swapped.\n\n"
	"Press \x16/\x17 to change blend weights\n"
	"Press \x14 Toggle extracted motion\n"
	"Press\x15 Toggle skins";

static const char* ANIMATION_PATH[ 2 ] = { "Resources/Animation/Retargeting/shulgoth_default_walk.hkt", "Resources/Animation/Retargeting/male_nocape_walk_slow.hkt" };
static const char* RIG_PATH[ 2 ] = { "Resources/Animation/Retargeting/shulgoth_default.hkt", "Resources/Animation/Retargeting/male_nocape.hkt" };
static const char* SCENE_PATH = "Resources/Animation/Retargeting/retargeting_scene.hkt";
static const hkReal EXTRACTED_MOTION_SCALE[] = { 0.55f, 1.0f };


BlendedRetargetingDemo::BlendedRetargetingDemo( hkDemoEnvironment* env )
:	hkDefaultAnimationDemo(env)
{
	// Set defaults
	m_drawSkins = true;
	m_useExtractedMotion = false;
	m_blendFraction = 0.0f;

	// Setup the camera
	{
		hkVector4 from( 0.0f, -5.5f, 2.0f );
		hkVector4 to  ( 0.0f, 0.0f, 1.75f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );

		setupDefaultCameras( env, from, to, up, 0.1f, 100.0f );
	}

	// Enable the shader library
	m_env->m_sceneConverter->setShaderLibraryEnabled( true );

	m_loader = new hkLoader();

	// For each character
	for ( int i = 0; i < 2; i++ )
	{
		// Initialize the world from model
		{
			m_worldFromModel[ i ].setIdentity();

			m_worldFromModel[ i ].m_translation( 0 ) += ( i ? 1.0f : -1.0f ) * 1.25f;
		}

		// Get the rig
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2( 0x27343437, container != HK_NULL , hkStringBuf( "Could not load asset: ", assetFile ).cString() );
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_skeletons.getSize() > 0 ), "No skeleton loaded" );

			m_animatedSkeleton[i] = new hkaAnimatedSkeleton( ac->m_skeletons[0] );

			// this will speed things up since we don't need to blend in the reference pose
			m_animatedSkeleton[i]->setReferencePoseWeightThreshold( 0.0f );
		}

		// Get the animations and bindings
		{
			hkStringBuf assetFile = ANIMATION_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );

			HK_ASSERT2( 0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
			HK_ASSERT2( 0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No binding loaded");

			// Grab the binding
			m_binding[ i ] = ac->m_bindings[0];
		}

		// Convert the skin
		{
			hkStringBuf assetFile = RIG_PATH[ i ];
			hkAssetManagementUtil::getFilePath( assetFile );

			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
			HK_ASSERT2( 0x27343437, container != HK_NULL , "Could not load asset");

			hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );
			HK_ASSERT2( 0x27343435, scene , "No scene loaded");

			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ) );
			HK_ASSERT2( 0x27343435, ac && (ac->m_skins.getSize() > 0 ), "No skins loaded");

			m_skinBindings[ i ] = ac->m_skins;

			// Make graphics output buffers for the skins
			env->m_sceneConverter->convert( scene );
		}
	}

	// Now set up the maps
	{
		// Create a mapper at runtime

		hkaSkeletonMapperUtils::Params params;

		// Set up the mapping parameters
		{
			params.m_autodetectChains = false;
			params.m_autodetectSimple = true;
			params.m_compareNames = HK_NULL;
			params.m_mappingType = hkaSkeletonMapperData::HK_RETARGETING_MAPPING;
			params.m_positionMatchTolerance = 1.0e-4f;

			// Set the skeletons appropriately
			params.m_skeletonA = m_animatedSkeleton[ 0 ]->getSkeleton();
			params.m_skeletonB = m_animatedSkeleton[ 1 ]->getSkeleton();

			// Set up the extracted motion scaling
			// As the monster is tall, but has stumpy legs
			params.m_motionParams.m_extractedMotionFactorA = EXTRACTED_MOTION_SCALE[ 0 ];
			params.m_motionParams.m_extractedMotionFactorB = EXTRACTED_MOTION_SCALE[ 1 ];
		}

		hkaSkeletonMapperData mapperData[ 2 ];

		hkaSkeletonMapperUtils::createMapping( params, mapperData[ 1 ], mapperData[ 0 ] );

		// Create the new skeleton mappers
		for ( int i = 0; i < 2; i++ )
		{
			m_mapper[ i ] = new hkaSkeletonMapper( mapperData[ i ] );
		}
	}

	// Create the animation controls for each character
	for ( int i = 0; i < 2; i++ )
	{
		hkaAnimatedSkeleton* character = m_animatedSkeleton[ i ];

		const hkaAnimationBinding* normalBinding = m_binding[ i ];
		const hkaAnimationBinding* mappedBinding = m_binding[ 1 - i ];

		// Create a new animation control using the current animation
		hkaDefaultAnimationControl* normalControl = new hkaDefaultAnimationControl( normalBinding );
		normalControl->setMasterWeight( 1.0f );
		character->addAnimationControl( normalControl );
		normalControl->removeReference();

		// Create a new animation control using the mapped animation
		hkaDefaultAnimationControl* mappedControl = new hkaDefaultAnimationControl( mappedBinding );
		mappedControl->setMasterWeight( 0.0f );

		// Add the mapper to the mapped control
		HK_ASSERT2( 0x12448fbb, m_animatedSkeleton[ i ]->getSkeleton() == m_mapper[ i ]->m_mapping.m_skeletonB, "Output skeleton of mapper does not match current skeleton" );
		mappedControl->setSkeletonMapper( m_mapper[ i ] );

		// Let the control own the mapper
		m_mapper[ i ]->removeReference();

		// Add the control to the animated skeleton
		character->addAnimationControl( mappedControl );
		mappedControl->removeReference();
	}

	// Load the scene
	{
		hkStringBuf assetFile = SCENE_PATH;
		hkAssetManagementUtil::getFilePath( assetFile );

		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2( 0x05dedbdc, container != HK_NULL , "Could not load asset" );
		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ) );

		removeLights( m_env );

		HK_ASSERT2( 0x1ab686c6, scene, "No scene loaded" );
		env->m_sceneConverter->convert( scene );
	}

	// We want to see backfacing polygons (esp for characters with loose 2-sided clothing)
	setGraphicsState( HKG_ENABLED_CULLFACE, false );

	// Enable shadows
	forceShadowState( true );

	// register our animation job
	hkaSampleBlendJobQueueUtils::registerWithJobQueue(m_jobQueue);

	// Setup the graphics
	setupGraphics();

	// Set up the shadow bounds
	{
		hkgAabb shadowAabb;
		const hkgLight* l=  m_env->m_displayWorld->getLightManager()->getBestShadowCaster();

		m_env->m_window->setShadowMapSize( 1024 ); // Use high quality shadows

		m_env->m_displayWorld->getShadowWorldBounds( shadowAabb, *m_env->m_window->getCurrentViewport()->getCamera(), l->getDirectionPtr() );

		hkDefaultDemo::setupFixedShadowFrustum( m_env, *l, shadowAabb );
	}
}

BlendedRetargetingDemo::~BlendedRetargetingDemo()
{
	// Release the animated skeletons
	m_animatedSkeleton[ 0 ]->removeReference();
	m_animatedSkeleton[ 1 ]->removeReference();

	for( unsigned int i = 0; i < HK_COUNT_OF(m_skinBindings); ++i )
	{
		m_skinBindings[i].clear();
	}

	// Release loaded assets
	m_loader->removeReference();
}

// Sample the pose
// The animated skeleton class retargets the blended animations

void BlendedRetargetingDemo::sampleSt( hkaPose** poses )
{
	HK_TIMER_BEGIN( "sampleSt", HK_NULL );

	for (int i = 0; i < 2; ++i )
	{
		m_animatedSkeleton[ i ]->sampleAndCombineAnimations( poses[i]->accessUnsyncedPoseLocalSpace().begin(), HK_NULL );
	}

	HK_TIMER_END();
}

void BlendedRetargetingDemo::sampleMt( hkaPose** poses )
{
	HK_TIMER_BEGIN_LIST( "sampleMt", "job setup" );

	// We'll distribute the work evenly over the threads
	hkArray<hkaSampleBlendJob> job( 2 );
	job.setSize( 2 );
	for( int i = 0; i < 2; i++ )
	{
		job[i].build( m_animatedSkeleton[i], poses[i] );
	}

	HK_TIMER_SPLIT_LIST( "add jobs" );

	for (int s=0; s < 2; s++)
	{
		m_jobQueue->addJob( job[s], hkJobQueue::JOB_HIGH_PRIORITY );
	}

	HK_TIMER_SPLIT_LIST( "do jobs" );

	m_threadPool->processJobQueue( m_jobQueue );
	m_jobQueue->processAllJobs();

	// Wait for the jobs to finish
	m_threadPool->waitForCompletion();

	HK_TIMER_END_LIST();
}

hkDemo::Result BlendedRetargetingDemo::stepDemo()
{
	// Check user input
	{
		// Decrease the blend fraction
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) )
		{
			m_blendFraction -= 0.1f;
		}

		// Increase the blend fraction
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
		{
			m_blendFraction += 0.1f;
		}

		// Bound the blend fraction
		m_blendFraction = hkMath::clamp( m_blendFraction, hkReal(0.0f), hkReal(1.0f) );

		// Enable extracted motion
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_UP ) )
		{
			m_useExtractedMotion = !m_useExtractedMotion;
		}

		// Draw/undraw the character bones
		if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_DOWN ) )
		{
			m_drawSkins = !m_drawSkins;
		}
	}

	// Update the control weights
	for ( int i = 0; i < 2; i++ )
	{
		static_cast< hkaDefaultAnimationControl* >( m_animatedSkeleton[ i ]->getAnimationControl( 0 ) )->setMasterWeight( 1.0f - m_blendFraction );
		static_cast< hkaDefaultAnimationControl* >( m_animatedSkeleton[ i ]->getAnimationControl( 1 ) )->setMasterWeight( m_blendFraction );
	}

	hkaPose* poses[2];

	// Draw and update each character
	for ( int i = 0; i < 2; i++ )
	{
		// Advance time
		m_animatedSkeleton[ i ]->stepDeltaTime( m_timestep );

		poses[i] = new hkaPose( m_animatedSkeleton[ i ]->getSkeleton() );
	}

	#if defined(HK_PLATFORM_MULTI_THREAD) && (HK_CONFIG_THREAD == HK_CONFIG_MULTI_THREADED)
		if ( ( m_threadPool == HK_NULL ) || ( 0 == m_threadPool->getNumThreads() ) )
		{
			sampleSt( poses );
		}
		else
		{
			sampleMt( poses );
		}
	#else
		sampleSt( poses );
	#endif

	for ( int i = 0; i < 2; i++ )
	{
		// Handle worldFromModel
		{
			// Handle extracted motion
			if ( m_useExtractedMotion )
			{
				hkQsTransform deltaMotion;

				m_animatedSkeleton[ i ]->getDeltaReferenceFrame( m_timestep, deltaMotion );

				m_worldFromModel[ i ].setMulEq( deltaMotion );
			}

			// Move the character's root by the world from model transform
			hkQsTransform root = m_worldFromModel[ i ];
			root.setMulEq( poses[i]->getBoneLocalSpace( 0 ) );
			poses[i]->setBoneLocalSpace( 0, root );
		}

		// Handle the drawing of skins or skeletons
		if ( !m_drawSkins )
		{
			hkColor::Argb skeletonColor = hkColor::rgbFromChars( 0, 0, 0, 255 );

			// Draw the skeleton
			AnimationUtils::drawPose( *poses[i], hkQsTransform::IDENTITY, skeletonColor );

			// Move the skins off-camera
			hkQsTransform mover = hkQsTransform::IDENTITY;
			mover.m_translation( 2 ) -= 5.0f;
			mover.setMulEq( poses[i]->getBoneLocalSpace( 0 ) );
			poses[i]->setBoneLocalSpace( 0, mover );
		}

		// Draw the skins
		AnimationUtils::skinMesh( *poses[i], m_skinBindings[ i ], hkTransform::getIdentity(), *m_env->m_sceneConverter );

		delete poses[i];
	}

	{
		hkStringBuf str;
		str.printf( "Original:%3.0f%%,  Retargeted:%5.1f%%", ( 1.0f - m_blendFraction ) * 100.0f, m_blendFraction * 100.0f, 0x0, 0 );
		const int w = m_env->m_window->getWidth();
		const int h = getWindowHeight();
		m_env->m_textDisplay->outputText( str, w - 340, h - 40 );
	}

	return hkDefaultAnimationDemo::stepDemo();
}


// runs out of memory on ps3 so only bootstrap on win32
HK_DECLARE_DEMO( BlendedRetargetingDemo, HK_DEMO_TYPE_ANIMATION, "Blends original and retargeted animations", HELP_STRING );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
