/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */


#include <Demos/demos.h>
#include <Graphics/Common/Window/hkgWindow.h>

#include <Demos/Animation/Api/Ragdoll/RagdollScaling/RagdollScalingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Playback/Utilities/hkaSampleAndCombineUtils.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Physics2012Bridge/Controller/RigidBody/hkaRagdollRigidBodyController.h>
#include <Animation/Physics2012Bridge/Utils/hkaRagdollUtils.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Visualize/Shape/hkDisplayConvex.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>
#include <Physics2012/Utilities/Dynamics/ScaleSystem/hkpSystemScalingUtility.h>

// For HKA_DEMO defines
#include <Demos/DemoCommon/DemoFramework/hkDefaultAnimationDemo.h>

static const char helpString[] =
"Hold  \x14/\x15 to scale the character continuously\n"
"Press \x16/\x17 to scale the character in steps";

static const char* ASSET_ANIMATION = HK_ASSET_NAME("Resources/Animation/HavokGirl/hkRunLoop.hkt");
static const char* ASSET_RAGDOLL = HK_ASSET_NAME("Resources/Animation/Ragdoll/dismemberment_ragdoll.hkt");
static const char* ASSET_SKIN = HK_ASSET_NAME("Resources/Animation/Ragdoll/dismemberment_skin.hkt");

RagdollScalingDemo::RagdollScalingDemo(hkDemoEnvironment* env)
:	hkDefaultPhysics2012Demo(env) 
{
	// Disable warning about reposing the ragdoll.
	setErrorEnabled(0x71C72FE7, false);

	// Setup the camera
	{
		hkVector4 from( 0.0f, 5.0f, 1.0f );
		hkVector4 to  ( 0.0f, 0.0f, 1.0f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.1f, 100 );
	}

	// Create physical world
	{
		hkpWorldCinfo info;
		info.setBroadPhaseWorldSize( 100.0f );
		info.m_gravity.set( 0.0f, 0.0f, -0.0f, 0.0f );
		info.m_collisionTolerance = 0.05f;
		info.m_broadPhaseBorderBehaviour = info.BROADPHASE_BORDER_DO_NOTHING;

		m_world = new hkpWorld( info );
		m_world->lock();

		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
	}

	// Load the data
	m_loader = new hkLoader();

	// setup layer collision
	{
		// Replace filter
		hkpGroupFilter* groupFilter = new hkpGroupFilter();

		// We disable collisions between different layers to determine what behaviour we want
		groupFilter->disableCollisionsBetween( 2, 1 );

		m_world->setCollisionFilter( groupFilter, true);
		groupFilter->removeReference();
	}

	// load ragdoll rig, create ragdoll instance
	{
		// penetration_rig.hkt - highRes and lowRes  skeletons, ragdoll, mappers
		hkStringBuf assetFile(ASSET_RAGDOLL); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* dataContainerRig = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, dataContainerRig != HK_NULL , "Could not load data asset");

		m_ragdoll = reinterpret_cast<hkaRagdollInstance*>( dataContainerRig->findObjectByType( hkaRagdollInstanceClass.getName() ) );
		HK_ASSERT2(0, m_ragdoll, "Couldn't load ragdoll setup");

		// Assign the ragdoll skeleton
		m_ragdollSkeleton = const_cast<hkaSkeleton*>(m_ragdoll->getSkeleton());

		// This routine iterates through the bodies pointed to by the constraints and stabilizes their inertias.
		// This makes both ragdoll controllers lees sensitive to angular effects and hence more effective
		const hkArray<hkpConstraintInstance*>& constraints = m_ragdoll->getConstraintArray();
		hkpInertiaTensorComputer::optimizeInertiasOfConstraintTree( constraints.begin(), constraints.getSize(), m_ragdoll->getRigidBodyOfBone(0) );

		// Find the two mappings
		{
			// Iterate over all hkaSkeletonMapper objects
			void* objectFound = dataContainerRig->findObjectByType( hkaSkeletonMapperClass.getName() );
			while ( objectFound )
			{
				hkaSkeletonMapper* mapper = reinterpret_cast<hkaSkeletonMapper*>( objectFound );

				// Use the skeleton to determine which mapper is which
				// Assign the animation skeleton

				if ( mapper->m_mapping.m_skeletonA == m_ragdoll->getSkeleton() )
				{
					m_animationFromRagdoll = mapper;
					m_animationSkeleton = const_cast<hkaSkeleton*>(mapper->m_mapping.m_skeletonB.val());
				}

				if ( mapper->m_mapping.m_skeletonB == m_ragdoll->getSkeleton() )
				{
					m_ragdollFromAnimation = mapper;
					m_animationSkeleton = const_cast<hkaSkeleton*>(mapper->m_mapping.m_skeletonA.val());
				}

				// Find the next object of this type
				objectFound = dataContainerRig->findObjectByType( hkaSkeletonMapperClass.getName(), objectFound );
			}

			HK_ASSERT2( 0, m_animationFromRagdoll, "Couldn't load high-to-ragdoll mapping" );
			HK_ASSERT2( 0, m_ragdollFromAnimation, "Couldn't load ragdoll-to-high mapping" );
		}

		// Set all scale factors initially to 1.0f
		m_scale = 1.0f;

		m_worldFromModel.setIdentity();
		m_worldFromModel.m_rotation.set( 0,0,1,0 );
	}

	// Get the animation and the binding
	{
		hkStringBuf assetFile(ASSET_ANIMATION); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0 ), "No binding loaded");

		m_binding = ac->m_bindings[0];
	}

	// Create the animated character
	{
		m_character = new hkaAnimatedSkeleton( m_animationSkeleton );
		m_control = new hkaDefaultAnimationControl( m_binding, true );
	
		m_character->addAnimationControl( m_control );
	}

	// Set the initial pose of the character
	{
		hkaPose animationPose( m_animationSkeleton );
		hkaPose ragdollPose( m_ragdollSkeleton );

		// Sample the character's current animated pose
		m_character->sampleAndCombineAnimations( animationPose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

		// Convert the animation pose to a ragdoll pose
		m_ragdollFromAnimation->mapPose( animationPose.getSyncedPoseModelSpace().begin(), m_ragdollSkeleton->m_referencePose.begin(), ragdollPose.accessUnsyncedPoseModelSpace().begin(), hkaSkeletonMapper::CURRENT_POSE );

		// Set the pose of the ragdoll
		m_ragdoll->setPoseModelSpace( ragdollPose.getSyncedPoseModelSpace().begin(), m_worldFromModel );

		m_ragdoll->addToWorld( m_world, true );
	}

	// Convert the skins
	{
		hkStringBuf assetFile(ASSET_SKIN); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
		HK_ASSERT2(0x27343435, scene , "No scene loaded");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skins.getSize() > 0), "No skins loaded");

		m_skinBindings = ac->m_skins;

		// Make graphics output buffers for the skins
		env->m_sceneConverter->convert( scene );
	}

	// setup the graphics
	setupGraphics();

	// Create some geometry for display purposes
	{
		m_geometry = new hkGeometry;

		// landscape parameters
		int side = 3;
		int numVertices = side * side;
		m_geometry->m_vertices.setSize( numVertices );

		int numTriangles = 2 * ( side - 1 ) * ( side - 1 );
		m_geometry->m_triangles.setSize( numTriangles );

		// build the landscape given this above parameters
		buildLandscape( side, 1.5f );

		m_displayConvex = new hkDisplayConvex( m_geometry ); // owns the geom given.
		m_geometryArray.pushBack( m_displayConvex );

		// add the geometry to the display manager
		hkDebugDisplay::getInstance().addGeometry( m_geometryArray, hkTransform::getIdentity(), 0x1001, 0, 0);

		// set the landscape colour to green
		hkDebugDisplay::getInstance().setGeometryColor( 0xff00a030, 0x1001, 0 );
	}

	// Create the new controllers
	m_ragdollRigidBodyController = new hkaRagdollRigidBodyController( m_ragdoll );
	
	m_world->unlock();
}

RagdollScalingDemo::~RagdollScalingDemo()
{
	m_world->lock();

	// remove ragdoll from world
	if ( m_ragdoll->getWorld() )
	{
		m_ragdoll->removeFromWorld();
	}
	
	// The ragdoll bodies are loaded from a packfile will not have their destructors called (the memory is just dumped)
	// Because we change from keyframed to dynamic we need to make sure the dtor is called to destroy the motion
	for ( int i = 0; i < m_ragdoll->m_rigidBodies.getSize(); i++ )
	{
		hkpRigidBody* rb = m_ragdoll->m_rigidBodies[ i ];

		// Explictly call to the destructor
		if ( rb != HK_NULL )
		{
			rb->~hkpRigidBody();
		}

	}

	// before we wipe the loader
	m_world->unlock();
	{
		m_world->markForWrite();
		m_world->removeReference();
		m_world = HK_NULL;
	}

 	delete m_ragdollRigidBodyController;
	if( m_character ) m_character->removeReference();
	if( m_control ) m_control->removeReference();

	m_skinBindings.clear();
	delete m_loader;

	// Delete the floor plane
	{
		// remove our landscape from the display manager
		hkDebugDisplay::getInstance().removeGeometry( 0x1001, 0, 0 );

		// delete our geometry
		m_geometryArray.clear();
		delete m_displayConvex;
	}
}

hkDemo::Result RagdollScalingDemo::stepDemo()
{
	// Handle button presses...
	hkReal& oldScale = m_scale;
	hkReal newScale = oldScale;
	static const hkReal slowFactor = hkMath::pow( 2.0f, 1.0f / 100.0f );
	static const hkReal fastFactor = hkMath::pow( 2.0f, 1.0f / 10.0f );

	if ( m_env->m_gamePad->isButtonPressed( HKG_PAD_DPAD_UP ) )
	{
		newScale *= slowFactor;
	}

	if ( m_env->m_gamePad->isButtonPressed( HKG_PAD_DPAD_DOWN ) )
	{
		newScale /= slowFactor;
	}

	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_RIGHT ) )
	{
		newScale *= fastFactor;
	}

	if ( m_env->m_gamePad->wasButtonPressed( HKG_PAD_DPAD_LEFT ) )
	{
		newScale /= fastFactor;
	}

	// Move the characters vertically (Pelvis of the character is originally at the origin)
	m_worldFromModel.m_translation.set( 0, 0, 0.82f * newScale, 0 );

	// Test if the user has requested scaling
	if ( newScale != oldScale )
	{
		const hkReal scale = newScale / oldScale;

		// Scale the physics
		{
			m_world->lock();

			m_ragdoll->addReference();
			m_ragdoll->removeFromWorld();

			hkpPhysicsSystem system;
			for( int i=0; i<m_ragdoll->getRigidBodyArray().getSize(); ++i )
			{
				system.addRigidBody( m_ragdoll->getRigidBodyArray()[i] );
			}
			for( int i=0; i<m_ragdoll->getConstraintArray().getSize(); ++i )
			{
				system.addConstraint( m_ragdoll->getConstraintArray()[i] );
			}
			hkpSystemScalingUtility::scaleSystem( &system, scale );

			m_ragdoll->addToWorld( m_world, false );
			m_ragdoll->removeReference();

			m_world->unlock();
		}

		// Scale the skeleton mapper
		hkaSkeletonMapperUtils::scaleMapping( *m_ragdollFromAnimation, scale );
		hkaSkeletonMapperUtils::scaleMapping( *m_animationFromRagdoll, scale );

		// Scale the reference poses of the animated and ragdoll characters
		hkaSampleAndCombineUtils::scaleTranslations( scale, m_ragdollSkeleton->m_referencePose.begin(), m_ragdollSkeleton->m_bones.getSize() );
		hkaSampleAndCombineUtils::scaleTranslations( scale, m_animationSkeleton->m_referencePose.begin(), m_animationSkeleton->m_bones.getSize() );

		// Update the scale value
		oldScale = newScale;
	}


	{
		// Sample the animation and drive the ragdolls to the desired pose
		hkaPose animationPose( m_animationSkeleton );
		hkaPose ragdollPose( m_ragdollSkeleton );
		hkaPose upsampledPose( m_animationSkeleton );
		
		// Sample the character's current animated pose
		m_character->sampleAndCombineAnimations( animationPose.accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

		// Scale the bone lengths of the poses
		hkaSampleAndCombineUtils::scaleTranslations( m_scale, ragdollPose.accessUnsyncedPoseLocalSpace().begin(), m_ragdollSkeleton->m_bones.getSize() );
		hkaSampleAndCombineUtils::scaleTranslations( m_scale, animationPose.accessUnsyncedPoseLocalSpace().begin(), m_animationSkeleton->m_bones.getSize() );

		// Scale the root (necessary for skinning)
		animationPose.accessUnsyncedPoseLocalSpace()[ 0 ].m_scale.mul4( m_scale );

		// Convert the animation pose to a ragdoll pose
		m_ragdollFromAnimation->mapPose( animationPose.getSyncedPoseModelSpace().begin(), m_ragdollSkeleton->m_referencePose.begin(), ragdollPose.accessUnsyncedPoseModelSpace().begin(), hkaSkeletonMapper::CURRENT_POSE );

		// Map the ragdoll pose back to an animation pose
		m_animationFromRagdoll->mapPose( ragdollPose.getSyncedPoseModelSpace().begin(), m_animationSkeleton->m_referencePose.begin(), upsampledPose.accessUnsyncedPoseModelSpace().begin(), hkaSkeletonMapper::CURRENT_POSE );

		// Copy the scales back into the upsampled pose
		hkaRagdollUtils::copyScales( upsampledPose.accessUnsyncedPoseModelSpace().begin(), animationPose.getSyncedPoseModelSpace().begin(), m_animationSkeleton->m_bones.getSize() );

		// Draw the upsampled Pose
		{
			hkQsTransform animWorldFromModel( m_worldFromModel );
			animWorldFromModel.m_translation( 0 ) = -1.0f;
			
			AnimationUtils::drawPose( upsampledPose, animWorldFromModel );
		}

		// Skin the animation
		hkTransform worldFromModel( m_worldFromModel.m_rotation, m_worldFromModel.m_translation );
		worldFromModel.getTranslation()( 0 ) = 1.0f;
		AnimationUtils::skinMesh( upsampledPose, m_skinBindings, worldFromModel, *m_env->m_sceneConverter );

		m_world->lock();

		m_ragdollRigidBodyController->driveToPose( m_timestep, ragdollPose.accessSyncedPoseLocalSpace().begin(), m_worldFromModel, HK_NULL );

		m_world->unlock();

		// Step the animation
		m_character->stepDeltaTime( m_timestep );
	}


	// Print the current scale factor
	{
		char buf[ 255 ];
		hkString::sprintf(buf, "Scale: %.3f", m_scale );
		const int h = getWindowHeight();
		m_env->m_textDisplay->outputText( buf, 20, h-40, 0xffffffff, 1);
	}



	// Step the physics
	hkDefaultPhysics2012Demo::stepDemo();

	return DEMO_OK;	// original animation demo return
}


// Creates a ground plane from triangles
void RagdollScalingDemo::buildLandscape( int side, hkReal size )
{
	// create a rolling landscape for our character to walk across

	// create the vertices
	{
		{
			for( int i = 0; i < side; i++ )
			{
				for( int j = 0; j < side; j++ )
				{
					float h = 0.0f;

					m_geometry->m_vertices[ ( i * side + j ) ](0) = ( i * 1.0f - ( side - 1.0f ) * 0.5f ) * size;
					m_geometry->m_vertices[ ( i * side + j ) ](2) = h;
					m_geometry->m_vertices[ ( i * side + j ) ](1) = ( j * 1.0f - ( side - 1.0f ) * 0.5f ) * size;
					m_geometry->m_vertices[ ( i * side + j ) ](3) = 0.0f;
				}
			}
		}
	}

	// Create the triangles
	{
		int index = 0;
		hkUint16 corner = 0;

		for( int i = 0; i < side - 1; i++ )
		{
			for( int j = 0; j < side - 1; j++ )
			{
				m_geometry->m_triangles[index].set( hkUint16(corner), hkUint16(corner + side), hkUint16(corner + 1));
				m_geometry->m_triangles[ index + 1 ].set( hkUint16(corner + 1), hkUint16(corner + side), hkUint16(corner + side + 1) );

				index += 2;
				corner++;
			}

			corner++;
		}
	}
}


static const char description[] = "ROUS: Ragdolls of unusual size.";

HKA_DECLARE_PHYSICS_2012_DEMO( RagdollScalingDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, description, helpString );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
