/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/Ragdoll/RagdollController/RagdollControllerDemo.h>
#include <Demos/Animation/Api/Ragdoll/RagdollController/RagdollControllerDemoTweakOptions.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Demos/DemoCommon/Utilities/GameUtils/TweakerUtils.h>
#include <Demos/DemoCommon/DemoFramework/hkDefaultAnimationDemo.h>

#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Base/Reflection/hkClass.h>
#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/Reflection/Registry/hkDefaultClassNameRegistry.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Serialize/Util/hkSerializeUtil.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/SceneData/Mesh/hkxMesh.h>

#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>
#include <Physics/Constraint/Motor/Velocity/hkpVelocityConstraintMotor.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics/Constraint/Data/hkpConstraintDataUtils.h>

#include <Physics2012/Utilities/Dynamics/KeyFrame/hkpKeyFrameUtility.h>
#include <Physics2012/Utilities/Weapons/hkpBallGun.h>
#include <Physics2012/Dynamics/Collide/Filter/Constraint/hkpConstraintCollisionFilter.h>

#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControlListener.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/Animation/Deform/Skinning/hkaMeshBinding.h>
#include <Animation/Animation/Rig/hkaBoneAttachment.h>

#include <Animation/Physics2012Bridge/Controller/PoweredChain/hkaRagdollPoweredChainController.h>
#include <Animation/Physics2012Bridge/Controller/PoweredConstraint/hkaRagdollPoweredConstraintController.h>
#include <Animation/Physics2012Bridge/Instance/hkaRagdollInstance.h>
#include <Animation/Physics2012Bridge/Utils/hkaRagdollUtils.h>
#include <Animation/Physics2012Bridge/Controller/RigidBody/hkaRagdollRigidBodyController.h>

#include <Graphics/Common/Window/hkgWindow.h>
#include <Graphics/Common/DisplayObject/hkgDisplayObject.h>
#include <Graphics/Bridge/DisplayHandler/hkgDisplayHandler.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

#define Z_AXIS_IS_UP_DIRECTION  (0)

#if Z_AXIS_IS_UP_DIRECTION
static const hkVector4 g_worldUpVector(0.0f, 0.0f, 1.0f);
#else
static const hkVector4 g_worldUpVector(0.0f, 1.0f, 0.0f);
#endif

static const char g_tweakOptionFileName[] = "RagdollControllerDemo_tweak_options.xml";
static const char g_skinResourcePath[]    = "Resources/Animation/FireFighter/Internal/CharacterAssets/Firefighter_Skin.hkt";
static const char g_ragdollResourcePath[] = "Resources/Animation/FireFighter/Internal/CharacterAssets/Firefighter_Rig.hkt";
static const char g_animationFolderPath[] = "Resources/Animation/FireFighter/Animations";

static void buildTransform(hkTransform& transformOut, const hkVector4& position, const hkVector4& direction)
{
	// Set translational component
	transformOut.getTranslation() = position;

	// Set rotational component
	hkVector4 up(g_worldUpVector);
	hkVector4 right;
	right.setCross(direction, up);
	right.normalize3();
	up.setCross(right, direction);
	up.normalize3();
	transformOut.getRotation().setCols(direction, up, right);
}

static void buildViewTransformFromCamera(hkTransform& transformOut, const hkgCamera& cam)
{
	hkVector4 dir, eye;
	cam.getDir(dir);
	cam.getFrom(eye);
	buildTransform(transformOut, eye, dir);
}

static void buildWorldSpaceRayFromScreenCoordinate(hkgRay& output, int scrx, int scry, const hkgViewport& vp)
{
	const hkgCamera& cam = *vp.getCamera();
	float from[3];
	float to[3];
	float dir[3];
	{
		cam.unProject(scrx, scry, 0.0f, vp.getWidth(), vp.getHeight(), from);
		cam.unProject(scrx, scry, 1.0f, vp.getWidth(), vp.getHeight(), to);
		hkgVec3Sub(dir, to, from);
		hkgVec3Normalize(dir);
	}
	new (&output) hkgRay(from, dir);
}

static void setPositionConstraintMotor(const RagdollControllerDemoPositionMotorConstraintOptions& from, hkpPositionConstraintMotor* to)
{
	to->m_maxForce                     = from.m_maxForceByMotor;
	to->m_tau                          = from.m_tau;
	to->m_damping                      = from.m_damping;
	to->m_constantRecoveryVelocity     = from.m_constantRecoveryVelocity;
	to->m_proportionalRecoveryVelocity = from.m_proportionalRecoveryVelocity;
}

static void startRagdollMotors( hkpConstraintMotor* motor, hkaRagdollInstance* ragdoll )
{
	const int cc = ragdoll->getConstraintArray().getSize();
	for (int ci=0; ci<cc; ++ci)
	{
		hkpConstraintInstance* constraintInst = ragdoll->getConstraintArray()[ci];
		hkpConstraintData* constraintData = constraintInst->getDataRw();
		hkpConstraintDataUtils::convertToPowered(constraintData, motor, true);
	}
}

static void stopRagdollMotors( hkaRagdollInstance* ragdoll )
{
	hkaRagdollPoweredConstraintController::stopMotors( ragdoll );
}

class RagdollControllerDemo::AnimationControlListener : public hkaDefaultAnimationControlListener
{
public:

	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_UTILITIES, RagdollControllerDemo::AnimationControlListener );

	// Called whenever the hkaDefaultAnimationControl overflows (hkaDefaultAnimationControl::update passes the end of the animation)
	void loopOverflowCallback(hkaDefaultAnimationControl* control, hkReal deltaTime, hkUint32 overflows)
	{
		m_atLoopOverflow = true;
	}

	hkBool detectBeginningOfAnimationLoop()
	{
		const hkBool detected = m_atLoopOverflow;
		m_atLoopOverflow = false;
		return detected;
	}

	AnimationControlListener() : m_atLoopOverflow(false) {}

private:
	hkBool m_atLoopOverflow;
};

// Aggregation of some minor variables not to modify the header file.
struct RagdollControllerDemo::Extension
{
	HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_UTILITIES, RagdollControllerDemo::Extension );

	hkRefPtr<RagdollControllerDemoTweakOptions> m_options;
	hkStringPtr m_selected;
	hkInt32 m_prevDemoMode;
	hkBool m_tweaking;

	Extension()
		:	m_selected("/")
		,	m_tweaking(false)
	{
		m_prevDemoMode = RagdollControllerDemoTweakOptions::KEYFRAMED;

		m_options = RagdollControllerDemoTweakOptions::load(g_tweakOptionFileName);
		if( m_options == HK_NULL )
		{
			m_options = new RagdollControllerDemoTweakOptions();
		}

		// Remove extra ref count
		m_options->removeReference();
	}

	~Extension()
	{
		RagdollControllerDemoTweakOptions::save(m_options, g_tweakOptionFileName);
	}
};

RagdollControllerDemo::RagdollControllerDemo(hkDemoEnvironment* env)
	:	hkDefaultPhysics2012Demo(env)
	,	m_loader(HK_NULL)
	,	m_highToLowMapper(HK_NULL)
	,	m_lowToHighMapper(HK_NULL)
	,	m_animatedSkeleton(HK_NULL)
	,	m_animationControlListener(HK_NULL)
	,	m_ragdollInstance(HK_NULL)
	,	m_ragdollSkeleton(HK_NULL)
	,	m_ragdollRigidBodyController(HK_NULL)
	,	m_posConstraintMotor(HK_NULL)
	,	m_velConstraintMotor(HK_NULL)
	,	m_playing(true)
	,	m_ballGun(HK_NULL)
	,	m_extension(new Extension())
{
	// Set up the camera.
	{
#if Z_AXIS_IS_UP_DIRECTION
		hkVector4 from( -4, 4, 3 );
#else
		hkVector4 from( -4, 3, 4 );
#endif
		hkVector4 to  ( 0,0,0 );
		this->setupDefaultCameras( env, from, to, g_worldUpVector, 0.1f, 100 );
	}

	// Load character related assets.
	{
		m_loader = new hkLoader();
		
		{
			hkRootLevelContainer* container = HK_NULL;
			hkaAnimationContainer* ac = HK_NULL;

			{
				hkStringBuf assetFile(g_skinResourcePath);
				hkAssetManagementUtil::getFilePath(assetFile);
				container = m_loader->load( assetFile.cString() );
				HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
				ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
				HK_ASSERT2(0x27343435, ac, "No animation data");
			}

			// Load the skins.
			{
				m_skinBindings = ac->m_skins;
				m_boneAttachments = ac->m_attachments;

				hkxScene* scene = reinterpret_cast<hkxScene*>( container->findObjectByType( hkxSceneClass.getName() ));
				if (scene)
				{
					m_env->m_sceneConverter->convert( scene );
					AnimationUtils::createGraphicsMeshesForAttachments(m_env, ac->m_attachments, m_boneAttachmentObjects);
				}
			}

			// Get the rig.
			{
				HK_ASSERT2(0x27343435, ac->m_skeletons.getSize() > 0, "No skeleton loaded");
				m_animatedSkeleton = new hkaAnimatedSkeleton( ac->m_skeletons[0] );
			}
		}

		// Load animation.
		if ( m_animatedSkeleton )
		{
			hkStringBuf assetFile(m_resourcePath);
			hkAssetManagementUtil::getFilePath(assetFile);
			hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
			hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

			HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "No animation loaded");
			HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
			hkaAnimationBinding* runBinding = ac->m_bindings[0];

			hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl (runBinding);
			control->setMasterWeight(1.0f);
			control->setPlaybackSpeed(1.0f);
			control->setEaseInCurve(0, 0.3f, 0.6f, 1);	// Smooth
			control->setEaseOutCurve(1, .6f, .3f, 0);	// Smooth
			m_animationControlListener = new AnimationControlListener();
			control->addDefaultControlListener(m_animationControlListener);
			m_animatedSkeleton->addAnimationControl( control );
			control->removeReference();
		}
	}

	// The following filter allows to disable collisions between two bodies if they are connected through a constraint.
	// Generally, more sophisticated collision filter than this will be needed for production quality code.
	hkpConstraintCollisionFilter* filter = new hkpConstraintCollisionFilter();

	// Create the World.
	{
		hkpWorldCinfo info;
		const hkReal gravityScale = 10;
		info.m_gravity.setMul(g_worldUpVector, -gravityScale);
		info.setupSolverInfo( info.SOLVER_TYPE_4ITERS_MEDIUM );
		info.m_solverIterations = 4;
		info.m_enableDeactivation = false;
		info.m_collisionFilter = filter;

		m_world = new hkpWorld( info );
		filter->removeReference();

		m_world->lock();
		hkpAgentRegisterUtil::registerAllAgents(m_world->getCollisionDispatcher());
		setupGraphics();
	}

	// Create the floor
	{
		// Data specific to this shape.
#if Z_AXIS_IS_UP_DIRECTION
		hkVector4 halfExtents(5.0f, 5.0f, 1.0f);
#else
		hkVector4 halfExtents(5.0f, 1.0f, 5.0f);
#endif

		hkpBoxShape* shape = new hkpBoxShape( halfExtents, 0 );

		// To illustrate using the shape, create a rigid body by first defining a template.
		hkpRigidBodyCinfo rigidBodyInfo;

		rigidBodyInfo.m_shape = shape;
		rigidBodyInfo.m_position.setMul(g_worldUpVector, -1.1f);
		rigidBodyInfo.m_motionType = hkpMotion::MOTION_FIXED;
		
		// Create a rigid body (using the template above).
		hkpRigidBody* rigidBody = new hkpRigidBody(rigidBodyInfo);
		shape->removeReference();
		m_world->addEntity(rigidBody);
		rigidBody->removeReference();
	}

	// Load the ragdoll
	this->createRagdollInstance();

	// Create a ball gun.
	m_ballGun = new hkpBallGun();

	m_world->unlock();
}

void RagdollControllerDemo::createRagdollInstance()
{
	hkStringBuf assetFile(g_ragdollResourcePath);
	hkAssetManagementUtil::getFilePath(assetFile);
	hkRootLevelContainer* container = m_loader->load( assetFile.cString() );
	HK_ASSERT3(0, container != HK_NULL, "failed to load a file " << assetFile.cString());
	m_ragdollInstance = static_cast<hkaRagdollInstance*>( container->findObjectByType( hkaRagdollInstanceClass.getName()) );
	HK_ASSERT2(0, m_ragdollInstance != HK_NULL, "Failed to load a hkaRagdollInstance\n");

	hkArray<hkpRigidBody*>& rigidBodies = const_cast<hkArray<hkpRigidBody*>&>(m_ragdollInstance->getRigidBodyArray());
	hkArray<hkpConstraintInstance*>& constraints = const_cast<hkArray<hkpConstraintInstance*>&>(m_ragdollInstance->getConstraintArray());

	// This routine iterates through the bodies pointed to by the constraints and stabilizes their inertias.
	// This makes ragdoll controllers less sensitive to angular effects and hence more effective
	hkpInertiaTensorComputer::optimizeInertiasOfConstraintTree( constraints.begin(), constraints.getSize(), m_ragdollInstance->getRigidBodyOfBone(0) );

	m_ragdollSkeleton = const_cast<hkaSkeleton*>(m_ragdollInstance->getSkeleton());

	m_ragdollRigidBodyController = new hkaRagdollRigidBodyController(m_ragdollInstance);

	// Add ragdoll to the world.
	m_ragdollInstance->addToWorld(m_world, true);

	// Let each hkpRigidBody access to its hkgDisplayObject at ease.
	{
		const int rc = rigidBodies.getSize();
		for (int i=0; i<rc; ++i)
		{
			hkpRigidBody& rb = *rigidBodies[i];
			const hkUlong id = (hkUlong)(rb.getCollidable());
			hkgDisplayObject* dispObj = m_env->m_displayHandler->findDisplayObject(id);
			rb.setUserData((hkUlong)dispObj);
		}
	}

	// Create a position constraint motor.
	{
		hkpPositionConstraintMotor* posMotor = new hkpPositionConstraintMotor;
		setPositionConstraintMotor(m_extension->m_options->m_positionMotorConstraintOptions, posMotor);
		m_posConstraintMotor = posMotor;
	}

	// Find the two mappings
	{
		// Iterate over all hkaSkeletonMapper objects
		void* objectFound = container->findObjectByType( hkaSkeletonMapperClass.getName() );
		while ( objectFound )
		{
			hkaSkeletonMapper* mapper = reinterpret_cast<hkaSkeletonMapper*>( objectFound );

			// Use the skeleton to determine which mapper is which
			// Assign the animation skeleton

			if ( mapper->m_mapping.m_skeletonA == m_ragdollSkeleton )
			{
				m_lowToHighMapper = mapper;
			}

			if ( mapper->m_mapping.m_skeletonB == m_ragdollSkeleton )
			{
				m_highToLowMapper = mapper;
			}


			// Find the next object of this type
			objectFound = container->findObjectByType( hkaSkeletonMapperClass.getName(), objectFound );
		}

		HK_ASSERT2( 0, m_highToLowMapper, "Couldn't load high-to-ragdoll mapping" );
		HK_ASSERT2( 0, m_lowToHighMapper, "Couldn't load ragdoll-to-high mapping" );
	}
}

#define SAFE_RELEASE(obj)\
{\
	if ( (obj) )\
	{\
		(obj)->removeReference();\
		(obj) = HK_NULL;\
	}\
}

#define SAFE_DELETE(obj)\
{\
	if ( (obj) )\
	{\
		delete (obj);\
		(obj) = HK_NULL;\
	}\
}

RagdollControllerDemo::~RagdollControllerDemo()
{
	SAFE_DELETE(m_animationControlListener);

	m_world->lock();
	SAFE_RELEASE(m_ballGun);
	SAFE_RELEASE(m_lowToHighMapper);
	SAFE_RELEASE(m_highToLowMapper);
	SAFE_RELEASE(m_animatedSkeleton);
	SAFE_RELEASE(m_posConstraintMotor);
	SAFE_RELEASE(m_velConstraintMotor);
	SAFE_DELETE(m_ragdollRigidBodyController);
	SAFE_RELEASE(m_ragdollInstance);
	SAFE_RELEASE(m_ragdollSkeleton);
	m_world->unlock();

	m_world->markForWrite();
	m_world->removeReference();
	m_world = HK_NULL;

	SAFE_DELETE(m_loader);
	SAFE_DELETE(m_extension);
}

hkDemo::Result RagdollControllerDemo::stepDemo()
{
	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);

	m_world->lock();
	
	if (m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1))
	{
		m_playing = !m_playing;
	}

	// Advance the animation.
	if ( m_playing )
	{
		m_animatedSkeleton->stepDeltaTime( m_timestep );
	}

	// Create a runtime pose for the animation.
	hkaPose animPose(m_animatedSkeleton->getSkeleton());

	// Sample animation.
	m_animatedSkeleton->sampleAndCombineAnimations( animPose.accessUnsyncedPoseLocalSpace().begin(), animPose.getFloatSlotValues().begin() );

	// Draw the original(target) animation pose.
	if (options.m_displayOptions.m_showTargetAnimationPose)
	{
		const hkColor::Argb color = hkColor::WHITE;
		const hkQsTransform worldFromModel = hkQsTransform::getIdentity();
		AnimationUtils::drawPose( animPose, worldFromModel, color );
	}

	if (options.m_useKeyframeForRootBone)
	{
		// Lock the root bone (mostly pelvis).
		m_ragdollInstance->getRigidBodyOfBone(0)->setMotionType(hkpMotion::MOTION_KEYFRAMED);
	}
	else
	{
		// Unlock the root bone.
		m_ragdollInstance->getRigidBodyOfBone(0)->setMotionType(hkpMotion::MOTION_DYNAMIC);
	}

	// Create a runtime pose for the ragdoll.
	hkaPose ragdollPose(m_ragdollInstance->getSkeleton());
	ragdollPose.setToReferencePose();

	// Map from animation to physics. This pose will be a target ragdoll pose.
	m_highToLowMapper->mapPose( animPose, ragdollPose, hkaSkeletonMapper::CURRENT_POSE );

	if (options.m_displayOptions.m_showTargetRagdollPose)
	{
		const hkColor::Argb color = hkColor::BLUE;
		const hkQsTransform worldFromModel = hkQsTransform::getIdentity();
		AnimationUtils::drawPose( ragdollPose, worldFromModel, color );
	}
	
	const int rbCount = m_ragdollInstance->m_rigidBodies.getSize();

	hkLocalArray<hkColor::Argb> rbColors(rbCount);
	rbColors.setSize(rbCount, hkColor::GRAY);

	if (m_extension->m_prevDemoMode != options.m_demoMode)
	{
		// Call a handler function only when the demo mode has changed.
		onModeChanged(m_extension->m_prevDemoMode, options.m_demoMode);
	}

	// Drive the ragdoll to that pose.
	if (RagdollControllerDemoTweakOptions::RIGID_BODY_CONTROLLER == options.m_demoMode)
	{
		// Using rigid body controller.
		onRigidBodyContollerMode(ragdollPose, rbColors);
	}
	else if (RagdollControllerDemoTweakOptions::POSITION_MOTOR_POWERED_CONTROLLER == options.m_demoMode)
	{
		// Using powered controller with position constraint motor.
		onPoweredContollerMode(ragdollPose);
	}
	else if (RagdollControllerDemoTweakOptions::KEYFRAMED == options.m_demoMode)
	{
		// Using no controller. Just applying animation to the ragdoll.
		onKeyframeMode(ragdollPose);
	}

	m_extension->m_prevDemoMode = options.m_demoMode;

	// Update colors of ragdoll rigid bodies.
	{
		const hkBool visible = options.m_displayOptions.m_showRagdollRigidbodies;
		const hkUint8 alpha = hkUint8(options.m_displayOptions.m_dynamicRagdollAlpha * 0xff * visible);
		for (int i=0; i < rbCount; ++i)
		{
			hkColor::Argb& color = rbColors[i];
			color = hkColor::replaceAlpha(alpha, color);
			HK_SET_OBJECT_COLOR( hkUlong( m_ragdollInstance->m_rigidBodies[i]->getCollidable()), color );
		}
	}

	const hkgKeyboard& keys = m_env->m_window->getKeyboard();
	if ( keys.wasKeyPressed('B') )
	{
		applyBallGun();
	}
	else if ( keys.wasKeyPressed('P') )
	{
		applyRaycastGun();
	}

	m_world->unlock();

	hkDefaultPhysics2012Demo::stepDemo();

	drawResultPoses(ragdollPose, animPose);

	handleTweakingInput();

	return hkDemo::DEMO_OK;
}

void RagdollControllerDemo::drawResultPoses( hkaPose& ragdollPose, hkaPose& animPose )
{
	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);
	const int boneCount = m_ragdollInstance->getNumBones();
	const int rbCount = m_ragdollInstance->m_rigidBodies.getSize();
	hkLocalBuffer<hkQsTransform> ragdollModelSpace( boneCount );
	m_ragdollInstance->getPoseModelSpace( ragdollModelSpace.begin(), hkQsTransform::getIdentity() );

	// Blend the result ragdoll pose.
	{
		const hkReal t = options.m_blendWeightOfTargetPose;
		for (int i=0; i < rbCount; ++i)
		{
			const hkQsTransform tr0 = ragdollModelSpace[i]; // Ragdoll pose driven by the controller.
			const hkQsTransform& tr1 = ragdollPose.getSyncedPoseModelSpace()[i]; // Target ragdoll pose.
			// tr1 = tr0 + t * (tr1 - tr0)
			ragdollModelSpace[i].setInterpolate4(tr0, tr1, t);
		}
	}

	// Map back from ragdoll to animation.
	{
		const hkQsTransform* localBIn  = animPose.getSyncedPoseLocalSpace().begin();
		hkQsTransform* modelBOut = animPose.accessSyncedPoseModelSpace().begin();

		m_lowToHighMapper->mapPose( ragdollModelSpace.begin(), localBIn, modelBOut, hkaSkeletonMapper::CURRENT_POSE );
	}

	// Draw the result animation pose.
	if (options.m_displayOptions.m_showResultAnimationPose)
	{
		const hkColor::Argb color = hkColor::RED;
		const hkQsTransform worldFromModel = hkQsTransform::getIdentity();
		AnimationUtils::drawPose( animPose, worldFromModel, color );
	}

	// Draw the result ragdoll pose.
	if (options.m_displayOptions.m_showDynamicRagdollPose)
	{
		hkArray<hkQsTransform> ragdollModelSpaceArray;
		ragdollModelSpaceArray.setDataUserFree(ragdollModelSpace.begin(), ragdollModelSpace.getSize(), ragdollModelSpace.getSize());

		ragdollPose.setPoseModelSpace(ragdollModelSpaceArray);

		const hkColor::Argb color = hkColor::GREEN;
		const hkQsTransform worldFromModel = hkQsTransform::getIdentity();
		AnimationUtils::drawPose( ragdollPose, worldFromModel, color );
	}

	// Draw the skinned character if available.
	{
		hkTransform worldTransform = hkTransform::getIdentity();
		if (!options.m_displayOptions.m_showSkinnedCharacter)
		{
			// Make the skinned character invisible anyway.
			worldTransform.setTranslation(hkVector4(.0f, .0f, 100000.0f, 1.0f));
		}
		doSkinning(animPose, worldTransform);
	}
}

void RagdollControllerDemo::onRigidBodyContollerMode( hkaPose& ragdollPose, hkLocalArray<hkColor::Argb>& rbColors )
{
	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);
	const int rbCount = m_ragdollInstance->m_rigidBodies.getSize();

	// We are at the first frame of each animation cycle?
	const hkBool atBeginningOfAnimLoop = m_animationControlListener->detectBeginningOfAnimationLoop();

	if (options.m_rigidbodyControllerOptions.m_lockFirstFrameOfAnimationLoop && atBeginningOfAnimLoop)
	{
		// Some animations (like get-up or fall-down) has discontinuity between each cycle.
		// Of course, this motion discontinuity causes simulation discontinuity which is a lot noticeable in most cases.
		// We can make this simulation discontinuity less noticeable
		// by resetting the motion of the 1st frame of each cycle as that of the target motion.

		for (int i=0; i < rbCount; ++i)
		{
			hkpRigidBody* rb = (hkpRigidBody*)m_ragdollInstance->m_rigidBodies[i];

			// Clear dynamic motion properties.
			rb->setLinearVelocity(hkVector4::getZero());
			rb->setAngularVelocity(hkVector4::getZero());

			// Synchronize the position & orientation with the target animation explicitly.
			const hkVector4& pos = ragdollPose.getSyncedPoseModelSpace()[i].m_translation;
			const hkQuaternion rot = ragdollPose.getSyncedPoseModelSpace()[i].m_rotation;
			rb->setPosition(pos);
			rb->setRotation(rot);
		}

		// Update the internal data of the controller.
		m_ragdollRigidBodyController->reinitialize();
	}

	hkLocalArray< hkaKeyFrameHierarchyUtility::Output > output( rbCount );
	output.setSize( rbCount );

	hkaKeyFrameHierarchyUtility::ControlData& controlData = m_ragdollRigidBodyController->m_controlDataPalette[0];
	{
		const RagdollControllerDemoRigidbodyRagdollControllerOptions& op = options.m_rigidbodyControllerOptions;
		controlData.m_hierarchyGain              = op.m_hierarchyGain;
		controlData.m_velocityDamping            = op.m_velocityDamping;
		controlData.m_accelerationGain           = op.m_accelerationGain;
		controlData.m_velocityGain               = op.m_velocityGain;
		controlData.m_positionGain               = op.m_positionGain;
		controlData.m_positionMaxLinearVelocity  = op.m_positionMaxLinearVelocity;
		controlData.m_positionMaxAngularVelocity = op.m_positionMaxAngularVelocity;
		controlData.m_snapGain                   = op.m_snapGain;
		controlData.m_snapMaxLinearVelocity      = op.m_snapMaxLinearVelocity;
		controlData.m_snapMaxAngularVelocity     = op.m_snapMaxAngularVelocity;
		controlData.m_snapMaxLinearDistance      = op.m_snapMaxLinearDistance;
		controlData.m_snapMaxAngularDistance     = op.m_snapMaxAngularDistance;
	}

	m_ragdollRigidBodyController->driveToPose
	(
		m_timestep,
		ragdollPose.getSyncedPoseLocalSpace().begin(),
		hkQsTransform::getIdentity(),
		output.begin()
	);

	// Show stress. 
	for (int i=0; i < rbCount; ++i)
	{
		hkColor::Argb& color = rbColors[i];
		hkReal stress = hkMath::sqrt( output[i].m_stressSquared );
		stress = hkMath::max2( stress - 0.5f,  hkReal(0.0f) );
		stress = hkMath::min2( stress * 0.05f, hkReal(1.0f) );
		int colorOffset = int( stress * 127 );
		color += colorOffset << 16;
		color -= colorOffset << 8;
		color -= colorOffset << 0;
	}
}

void RagdollControllerDemo::onPoweredContollerMode( hkaPose& ragdollPose )
{
	// NOTE :
	//    The following demonstrated code is for real-time tuning purpose, NOT optimization purpose.
	//    Basically, you don't have to do this kind of set-up every frame.
	//    Once you get your parameters fine-tuned, you do this at each initialization stage only in most cases.
	//    See 'Animation > Api > Ragdoll > BlendTest > BlendTestDemo.cpp' for a demonstration.

	hkaRagdollPoweredConstraintController::driveToPose( m_ragdollInstance, ragdollPose.getSyncedPoseLocalSpace().begin() );

	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);
	const int boneCount = m_ragdollInstance->getNumBones();

	for (int b=0; b<boneCount; ++b)
	{
		hkpConstraintInstance* constraint = m_ragdollInstance->getConstraintOfBone(b);
		if (!constraint)
		{
			continue;
		}
		hkpConstraintData* constraintData = constraint->getDataRw();
		if (!constraintData)
		{
			continue;
		}
		const int constraintType = constraintData->getType();

		if (hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL == constraintType)
		{
			hkpRagdollConstraintData* rc = static_cast<hkpRagdollConstraintData*> (constraintData);

			// Apply motor parameters to each constraint.
			for (int m=0; m<3; ++m)
			{
				hkpConstraintMotor* motor = rc->m_atoms.m_ragdollMotors.m_motors[m];
				if (!motor)
				{
					continue;
				}

				if (hkpConstraintMotor::TYPE_POSITION == motor->getType())
				{
					hkpPositionConstraintMotor* posMotor = static_cast<hkpPositionConstraintMotor*>(motor);
					setPositionConstraintMotor(options.m_positionMotorConstraintOptions, posMotor);
				}
			}
		}
		else if (hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE == constraintType)
		{
			hkpLimitedHingeConstraintData* hd = static_cast<hkpLimitedHingeConstraintData*>(constraintData);
			hkpConstraintMotor* motor = hd->m_atoms.m_angMotor.m_motor;
			if (motor)
			{
				if (hkpConstraintMotor::TYPE_POSITION == motor->getType())
				{
					hkpPositionConstraintMotor* posMotor = static_cast<hkpPositionConstraintMotor*>(motor);
					setPositionConstraintMotor(options.m_positionMotorConstraintOptions, posMotor);
				}
			}
		}
	}

	// Keyframe the root bone.
	if ( m_ragdollInstance->getRigidBodyOfBone(0)->getMotionType() == hkpMotion::MOTION_KEYFRAMED )
	{
		const hkVector4& nextPos = ragdollPose.getSyncedPoseModelSpace()[0].m_translation;
		const hkQuaternion nextOrient = ragdollPose.getSyncedPoseModelSpace()[0].m_rotation;
		hkpKeyFrameUtility::applyHardKeyFrame( nextPos, nextOrient, 1.0f /m_timestep, m_ragdollInstance->getRigidBodyOfBone(0) );
	}
}

void RagdollControllerDemo::onKeyframeMode( hkaPose& ragdollPose )
{
	const int rbCount = m_ragdollInstance->m_rigidBodies.getSize();

	for (int i=0; i < rbCount; ++i)
	{
		const hkVector4& nextPos = ragdollPose.getSyncedPoseModelSpace()[i].m_translation;
		const hkQuaternion nextOrient = ragdollPose.getSyncedPoseModelSpace()[i].m_rotation;
		hkpKeyFrameUtility::applyHardKeyFrame( nextPos, nextOrient, 1.0f /m_timestep, m_ragdollInstance->getRigidBodyOfBone(i) );
	}
}

void RagdollControllerDemo::onModeChanged( hkInt32 oldMode, hkInt32 newMode )
{
	typedef RagdollControllerDemoTweakOptions options_t;
	if (0) {}
	else if (options_t::RIGID_BODY_CONTROLLER==oldMode && options_t::POSITION_MOTOR_POWERED_CONTROLLER==newMode)
	{
		startRagdollMotors(m_posConstraintMotor, m_ragdollInstance);
	}
	else if (options_t::POSITION_MOTOR_POWERED_CONTROLLER==oldMode && options_t::RIGID_BODY_CONTROLLER==newMode)
	{
		stopRagdollMotors(m_ragdollInstance);
	}
	else if (options_t::RIGID_BODY_CONTROLLER==oldMode && options_t::KEYFRAMED==newMode) {}
	else if (options_t::KEYFRAMED==oldMode && options_t::RIGID_BODY_CONTROLLER==newMode) {}
	else if (options_t::POSITION_MOTOR_POWERED_CONTROLLER==oldMode && options_t::KEYFRAMED==newMode)
	{
		stopRagdollMotors(m_ragdollInstance);
	}
	else if (options_t::KEYFRAMED==oldMode && options_t::POSITION_MOTOR_POWERED_CONTROLLER==newMode)
	{
		startRagdollMotors(m_posConstraintMotor, m_ragdollInstance);
	}
	else
	{
		HK_ASSERT(0xffff, false);
	}
}

void RagdollControllerDemo::applyBallGun()
{
	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);
	hkgViewport& vp = *m_env->m_window->getCurrentViewport();
	hkgCamera& cam = *vp.getCamera();
	
	hkTransform vt;
	const int mouseX = m_env->m_window->getMouse().getPosX();
	const int mouseY = m_env->m_window->getMouse().getPosY();
	hkgViewportPickData vpd;
	vp.pick(mouseX, mouseY, m_env->m_displayWorld, vpd);
	if (vpd.isValid())
	{
		const float* wp = vpd.getClosestObjectWorldPos();
		hkVector4 pos;
		pos.set(wp[0], wp[1], wp[2]);
		hkVector4 dir, eye;
		cam.getFrom(eye);
		dir.setSub(pos, eye);
		dir.normalize3();
		buildTransform(vt, eye, dir);
	}
	m_ballGun->m_bulletRadius	= options.m_ballGunOptions.m_bulletRadius;
	m_ballGun->m_bulletVelocity	= options.m_ballGunOptions.m_bulletVelocity;
	m_ballGun->m_bulletMass		= options.m_ballGunOptions.m_bulletMass;
	m_ballGun->fireGun(m_world, vt);
}

void RagdollControllerDemo::applyRaycastGun()
{
	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);
	hkgViewport& vp = *m_env->m_window->getCurrentViewport();
	hkgCamera& cam = *vp.getCamera();
	const int mouseX = m_env->m_window->getMouse().getPosX();
	const int mouseY = m_env->m_window->getMouse().getPosY();

	hkgRay ray;
	buildWorldSpaceRayFromScreenCoordinate(ray, mouseX, mouseY, vp);

	hkArray<hkpRigidBody*>& rigidBodies = const_cast<hkArray<hkpRigidBody*>&>(m_ragdollInstance->getRigidBodyArray());
	const int rc = rigidBodies.getSize();
	float hitDistance = cam.getFar() - cam.getNear();
	int hitBodyIndex = -1;

	// Test hit for each rigid body of the ragdoll.
	for (int i=0; i<rc; ++i)
	{
		hkgDisplayObjectPickData data;
		const hkgDisplayObject* dispObj = (const hkgDisplayObject*)rigidBodies[i]->getUserData();
		dispObj->intersect(ray, true, data);
		if (!data.isValid())
		{
			continue;
		}
		const float dist = data.getClosestDistance();
		if (dist > hitDistance)
		{
			continue;
		}
		hitDistance = dist;
		hitBodyIndex = i;
	}

	if (hitBodyIndex > -1)
	{
		// A valid hit by the pick.
		const float* dp = ray.getDirPtr();
		hkVector4 rayDir(dp[0], dp[1], dp[2]);

		const float* op = ray.getFromPtr();
		hkVector4 rayOrigin(op[0], op[1], op[2]);
		hkVector4 hitPos;
		hitPos.setAddMul(rayOrigin, rayDir, hitDistance);

		hkVector4 imp;
		imp.setMul(rayDir, options.m_raycastGunImpulse);
		rigidBodies[hitBodyIndex]->applyPointImpulse(imp, hitPos);
	}
}

void RagdollControllerDemo::doSkinning( const hkaPose& pose, const hkTransform& worldTransform )
{
	AnimationUtils::skinMesh(pose, m_skinBindings, worldTransform, *m_env->m_sceneConverter);

	const hkArray<hkQsTransform>& poseLS = pose.getSyncedPoseModelSpace();
	const int c = poseLS.getSize();
	hkArray<hkQsTransform> poseMS(c);
	poseMS.copy(poseMS.begin(), poseLS.begin(), c);
	hkQsTransform wt;
	wt.setRotation(worldTransform.getRotation());
	wt.setTranslation(worldTransform.getTranslation());
	for (int i=0; i<c; ++i)
	{
		poseMS[i].setMul(poseMS[i], wt);
	}
	AnimationUtils::drawAttachments(m_env, poseMS.begin(), m_boneAttachments, m_boneAttachmentObjects);
}

void RagdollControllerDemo::handleTweakingInput()
{
	RagdollControllerDemoTweakOptions& options = *(m_extension->m_options);
	hkBool& tweaking = m_extension->m_tweaking;
	hkStringPtr& selected = m_extension->m_selected;
	hkgKeyboard& keys = const_cast<hkgKeyboard&>(m_env->m_window->getKeyboard());
	if (!keys.isConnected())
	{
		return;
	}

	if ( keys.wasKeyPressed('T') || (!tweaking && keys.wasKeyPressed(HKG_VKEY_RIGHT)) )
	{
		tweaking = !tweaking;
	}

	if ( keys.wasKeyPressed('R') )
	{
		new (&options) RagdollControllerDemoTweakOptions;
	}

	showHelp();

	if (!tweaking)
	{
		return;
	}

	if (!selected.getLength() || selected[0]!='/')
	{
		selected = "/";
	}

	const hkClass& klass = RagdollControllerDemoTweakOptionsClass;

	if ( keys.wasKeyPressed('0') || keys.wasKeyPressed(HKG_VKEY_NUMPAD0) )
	{
		TweakerUtils::clearData(selected, &options, klass);
	}

	if (keys.wasKeyPressed(HKG_VKEY_DOWN))
	{
		selected = TweakerUtils::getNextSiblingPath(selected, &options, klass);
	}

	if (keys.wasKeyPressed(HKG_VKEY_UP))
	{
		selected = TweakerUtils::getPrevSiblingPath(selected, &options, klass);
	}

	if (keys.wasKeyPressed(HKG_VKEY_LEFT))
	{
		selected = TweakerUtils::getParentPath(selected, &options, klass);
	}

	if (keys.wasKeyPressed(HKG_VKEY_RIGHT))
	{
		selected = TweakerUtils::getChildPath(selected, &options, klass);
	}

	// Apply the tweaked result to the data.
	{
		hkReal delta = .0f;

		if (keys.wasKeyPressed(HKG_VKEY_SUBTRACT))
		{
			delta = -0.1f;
		}
		else if (keys.wasKeyPressed(HKG_VKEY_OEM_MINUS))
		{
			delta = -0.25f;
		}
		else if (keys.wasKeyPressed(HKG_VKEY_OEM_COMMA))
		{
			delta = -0.5f;
		}
		else if (keys.wasKeyPressed(HKG_VKEY_ADD))
		{
			delta = 0.1f;
		}
		else if (keys.wasKeyPressed(HKG_VKEY_OEM_PLUS))
		{
			delta = 0.25f;
		}
		else if (keys.wasKeyPressed(HKG_VKEY_OEM_PERIOD))
		{
			delta = 0.5f;
		}

		if (delta != .0f)
		{
			const hkReal threshold = delta > .0f ? HK_FLOAT_MAX : .0f;
			TweakerUtils::tweakData(selected, &options, klass, float(delta), threshold, TweakerUtils::ADDITIVE);
		}
	}

	// Make sure the tweaked data is valid
	options.validate();

	const float x = 20;
	const float y = 80;
	TweakerUtils::DisplayOptions dispOpt;
	TweakerUtils::displayData(selected, &options, klass, *m_env->m_textDisplay, x, y, &dispOpt);
}

void RagdollControllerDemo::showHelp()
{
	const int x = m_env->m_window->getWidth() - 400;
	const int y = 80;

	hkStringBuf s;

	if (m_extension->m_tweaking)
	{
		s.append(
			"[T] - hide the tweak window\n"
			"Arrow keys - move across menus\n"
			"[+] or [.] - tweak up values\n"
			"[-] or [,] - tweak down values\n"
		);
	}
	else
	{
		s.append("[T] or [>] - show the tweak window\n");
	}

	s.append(
		"[R] - Reset all the tweak options to defaults\n"
		"[B] - Ball Gun\n"
		"      - (hovering the cursor over the target position)\n"
		"[P] - Raycast Gun\n"
		"      - (hovering the cursor over the target position)\n"
		"      - (works only when you target the ragdoll bodies)\n"
	);

	m_env->m_textDisplay->outputText( s, x, y, 0xffffffff);
}


static void HK_CALL scanDirectoryAndCreateDemosForHavokFiles(hkDemoEntry& thisEntry, extArray<hkDemoEntry*>& entriesOut )
{
	hkDemo::scanDirectoryAndCreateDemosForHavokFiles( g_animationFolderPath, ".hkt", thisEntry, false, entriesOut );
}

HKA_DECLARE_PHYSICS_2012_DEMO_FACTORY(
	RagdollControllerDemo, /*DEMO_CLASS*/
	HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_DONT_BOOTSTRAP, /*DEMO_TYPE*/
	scanDirectoryAndCreateDemosForHavokFiles, /*CREATE_FUNC*/
	"A ragdoll controller demo", /*HELP*/
	"..." /*DETAILS*/
);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
