/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/Ragdoll/PenetrationDetection/NpRagdollPenetrationDetectionDemo.h>

#include <Graphics/Common/Window/hkgWindow.h>

#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapper.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/PhysicsBridge/Ragdoll/hknpRagdoll.h>

#include <Common/SceneData/Scene/hkxScene.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Graphics/Bridge/SceneData/hkgSceneDataConverter.h>

#include <Physics/Physics/Collide/Query/Collector/hknpClosestHitCollector.h>
#include <Physics/Physics/Collide/Filter/Group/hknpGroupCollisionFilter.h>
#include <Physics/Physics/Extensions/PhysicsSystem/hknpPhysicsSceneData.h>
#include <Physics/Physics/Extensions/Viewers/Shape/hknpShapeViewer.h>


namespace
{
	static const hkVector4 UP(0,0,1);

	// hkaNpRagdollPenetrationDetector requires an interface to perform raycast.
	class NpPenetrationDemoRagdollRaycastInterface : public hkReferencedObject, public hkaNpRagdollRaycastInterface
	{
		public:

			HK_DECLARE_CLASS_ALLOCATOR( HK_MEMORY_CLASS_DEMO );

			NpPenetrationDemoRagdollRaycastInterface( hknpWorld* world ) : m_world(world) {}
			hkBool castRay( const hkVector4& fromWS, const hkVector4& toWS, hkVector4& hitPointWS, hkVector4& hitNormalWS );

		protected:

			hknpWorld* m_world;
	};

	hkBool NpPenetrationDemoRagdollRaycastInterface::castRay(
		const hkVector4& fromWS, const hkVector4& toWS, hkVector4& hitPointWS, hkVector4& hitNormalWS )
	{
		hknpRayCastQuery query;
		query.m_ray.setEndPoints( fromWS, toWS );
		query.m_filter = m_world->getCollisionFilter();
		query.m_filterData.m_collisionFilterInfo = hknpGroupCollisionFilter::calcFilterInfo( NpRagdollPenetrationDetectionDemo::LAYER_RAYCAST, 0 );

		hknpClosestHitCollector rayCollector;
		rayCollector.reset();

		m_world->castRay( query, &rayCollector );
		if( rayCollector.hasHit() )
		{
			const hknpRayCastQueryResult& raycastOut = rayCollector.getHits()[0].asRayCast();
			hitPointWS = raycastOut.getPosition();
			hitNormalWS = raycastOut.getSurfaceNormal();
			return true;
		}
		else
		{
			return false;
		}
	}
}



NpRagdollPenetrationDetectionDemo::NpRagdollPenetrationDetectionDemo(hkDemoEnvironment* env)
	:	hkDefaultPhysicsDemo(env),
	m_time(0.0f),
	m_statusFlag(false),
	m_poseId(0),
	m_correctionFlag(true),
	m_turn(0.25f*HK_REAL_PI),
	m_pos(0.0f,-0.5f,0.9f)
{

	// graphic setup
	{
		forceShadowState(false);

		hkgViewport* viewport = m_env->m_window->getViewport(0);
		viewport->setDesiredState(viewport->getDesiredState() & ~HKG_ENABLED_CULLFACE);
	}

	// Setup the camera
	{
		hkVector4 from( -1.5f, 1.5f, 1.5f );
		hkVector4 to  ( 1.0f,-2.0f,0.5f );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.1f, 100 );
	}

	// Create physical world
	{
		hknpWorldCinfo info;
		info.setBroadPhaseSize( 100.0f );  // our world is not very big
		info.m_gravity.setMul4(-10, UP);
		info.m_collisionTolerance = 0.05f;
		info.m_leavingBroadPhaseBehavior = hknpWorldCinfo::ON_LEAVING_BROAD_PHASE_DO_NOTHING;
		info.m_persistentStreamAllocator = getPersistentAllocator();

		// We disable collisions between different layers to determine what behavior we want
		hknpGroupCollisionFilter* groupFilter = new hknpGroupCollisionFilter();
		groupFilter->disableCollisionsBetween(LAYER_RAGDOLL_KEYFRAMED, LAYER_LANDSCAPE);
		groupFilter->disableCollisionsBetween(LAYER_RAGDOLL_PENETRATING, LAYER_LANDSCAPE);
		groupFilter->disableCollisionsBetween(LAYER_RAYCAST, LAYER_RAGDOLL_DYNAMIC);
		groupFilter->disableCollisionsBetween(LAYER_RAYCAST, LAYER_RAGDOLL_KEYFRAMED);
		groupFilter->disableCollisionsBetween(LAYER_RAYCAST, LAYER_RAGDOLL_PENETRATING);

		info.m_collisionFilter = groupFilter;

		m_npWorld = new hknpWorld( info );
		groupFilter->removeReference();
	}

	// Load the data
	m_loader = new hkLoader();

	// Create the landscape 
	{
		hkStringBuf assetFile("Resources/Animation/Penetration/penetration_landscape_np.hkt"); hkAssetManagementUtil::getFilePath(assetFile);

		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");

		hknpPhysicsSceneData* scene = reinterpret_cast<hknpPhysicsSceneData*> ( container->findObjectByType( hknpPhysicsSceneDataClass.getName()));

		if (scene)
		{
			hknpPhysicsSystemData* psys = scene->m_systemDatas[0];

			// Change the layer of the rigid bodies
			for (int rb = 0; rb < psys->m_bodyCinfos.getSize(); rb++)
			{
				psys->m_bodyCinfos[rb].m_collisionFilterInfo = hknpGroupCollisionFilter::calcFilterInfo( LAYER_LANDSCAPE,0 );
			}
			m_landscape = new hknpPhysicsSystem(psys, m_npWorld, hkTransform::getIdentity());
		}
	}

	// Two mappers (anim->ragdoll  and  ragdoll->anim)
	hkaSkeletonMapper* ragdollToHighResMapper = HK_NULL;
	hkaSkeletonMapper* highResToRagdollMapper = HK_NULL;

	// High resolution skeleton (poses)
	hkaSkeleton* skeletonHighRes = HK_NULL;

	//load rig, create ragdoll instance and extract poses
	{
		// three files exported from 3DS Max are needed in the demo:
		// penetration_landscape1.hkt or penetrationdemo2.hkt - small and large landscapes
		// penetration_rig.hkt - highRes and lowRes  skeletons, ragdoll, mappers
		// penetration_poses.hkt - animation with available posses
		// data from second file are stored in dataContainerRig
		// data from third file are stored in dataContainerPoses

		hkStringBuf assetFile("Resources/Animation/Penetration/penetration_rig_np.hkt"); 
		hkRootLevelContainer* dataContainerRig = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, dataContainerRig != HK_NULL , "Could not load data asset");

		m_ragdollData = reinterpret_cast<hknpRagdollData*>( dataContainerRig->findObjectByType( hknpRagdollDataClass.getName() ));
		HK_ASSERT2(0x27343438, m_ragdollData, "Couldn't load ragdoll data");

		m_ragdollInstance = new hknpRagdoll(m_ragdollData, m_npWorld, hkTransform::getIdentity());
		m_masses.setSize(m_ragdollInstance->getNumBones());
		for ( int i = 0; i < m_ragdollInstance->getNumBones(); i++)
		{
			setBodyKeyframed( i );
		}

		// Get HighRes skeleton
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( dataContainerRig->findObjectByType( hkaAnimationContainerClass.getName() ));
		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
		// Pick the skeleton with the highest number of bones
		skeletonHighRes = ac->m_skeletons[0];
		for (int skelIdx = 1; skelIdx < ac->m_skeletons.getSize(); skelIdx++)
		{
			if (ac->m_skeletons[skelIdx]->m_bones.getSize() > skeletonHighRes->m_bones.getSize() )
			{
				skeletonHighRes = ac->m_skeletons[skelIdx];
			}
		}

		// Find the two mappings
		{
			const hkaSkeleton* ragdollSkeleton = m_ragdollInstance->getSkeleton();
			void *objectFound = dataContainerRig->findObjectByType(hkaSkeletonMapperClass.getName());
			while (objectFound)
			{
				hkaSkeletonMapper* mapperFound = reinterpret_cast<hkaSkeletonMapper*> (objectFound);

				// Use the skeleton to determine which mapper is which
				if (mapperFound->m_mapping.m_skeletonA == ragdollSkeleton)
				{
					ragdollToHighResMapper = mapperFound;
				}
				else
				{
					HK_ASSERT(0,mapperFound->m_mapping.m_skeletonB == ragdollSkeleton);
					highResToRagdollMapper = mapperFound;
				}

				objectFound = dataContainerRig->findObjectByType(hkaSkeletonMapperClass.getName(), objectFound);
			}
			HK_ASSERT2(0, highResToRagdollMapper, "Couldn't load high-to-ragdoll mapping");
			HK_ASSERT2(0, ragdollToHighResMapper, "Couldn't load ragdoll-to-high mapping");
		}

	}

	// load poses from animation in HighRes and map them to LowRes
	{
		hkStringBuf assetFile("Resources/Animation/Penetration/penetration_poses.hkt"); hkAssetManagementUtil::getFilePath(assetFile);
		hkRootLevelContainer* dataContainerPoses = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
		HK_ASSERT2(0x27343437, dataContainerPoses != HK_NULL , "Could not load rig asset");

		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( dataContainerPoses->findObjectByType( hkaAnimationContainerClass.getName() ));

		hkaAnimation* animation =  ac->m_animations[0];
		HK_ASSERT2(0x25586454, animation != HK_NULL , "Could not load animation");
		
		hkaPose*	poseHighRes = new hkaPose( skeletonHighRes );

		// fullfill array of posses
		for (int p=0; p < NUM_POSES; p++)
		{
			m_posesLowRes[p] = new hkaPose( m_ragdollInstance->getSkeleton() );

			// Sample HighRes Pose
			hkReal sampleTime = p * (animation->m_duration/(NUM_POSES-1));
			animation->sampleTracks( sampleTime , poseHighRes->accessUnsyncedPoseLocalSpace().begin(), HK_NULL );

			// We use this pose as the in+out pose to the mapper below
			hkArray<hkQsTransform> ragdollArrayModelSpace( m_ragdollInstance->getNumBones() );

			// Map the pose from the animation (highres) to ragdoll (lowres)
			highResToRagdollMapper->mapPose( poseHighRes->getSyncedPoseModelSpace().begin(), m_ragdollInstance->getSkeleton()->m_referencePose.begin(), ragdollArrayModelSpace.begin(), hkaSkeletonMapper::CURRENT_POSE );

			m_posesLowRes[p]->setPoseModelSpace(ragdollArrayModelSpace);

		}

		delete poseHighRes;

	}
	// setup the graphics
	setupGraphics();

	// init pose as necessary
	{
		// set rotation
		hkQuaternion q0;
		q0.setAxisAngle( UP , m_turn );

		m_currentTransform.setIdentity();
		m_currentTransform.m_rotation = q0;
		m_currentTransform.m_translation = m_pos;
	}

	// create detection object and raycast interface
	{
		// create raycast object
		m_ragdollRaycastInterface = new NpPenetrationDemoRagdollRaycastInterface(m_npWorld);

		hkaNpRagdollPenetrationDetector::Setup setup;
		{
			setup.m_ragdollSkeleton = m_ragdollInstance->getSkeleton();
			setup.m_raycastInterface = m_ragdollRaycastInterface;
		}
		m_detectRagdollPenetration = new hkaNpRagdollPenetrationDetector(setup);
	}

	//// display ragdoll transparent
	//{
	//	for ( int i = 0; i < m_ragdollInstance->getNumBones(); i++)
	//	{
	//		hkpRigidBody* rb = m_ragdollInstance->getRigidBodyOfBone(i);

	//		if ( rb != HK_NULL )
	//		{
	//			HK_SET_OBJECT_COLOR((hkUlong)rb->getCollidable(), hkColor::rgbFromChars( 0, 100, 0, 120 ));
	//		}

	//	}
	//}
}

NpRagdollPenetrationDetectionDemo::~NpRagdollPenetrationDetectionDemo()
{
	if (m_detectRagdollPenetration)
	{
		m_detectRagdollPenetration->removeReference();
	}

	if (m_ragdollRaycastInterface)
	{
		static_cast<NpPenetrationDemoRagdollRaycastInterface*>(m_ragdollRaycastInterface)->removeReference();
	}

	// delete low res animation poses
	for (int i=0; i < NUM_POSES; i++ )
	{
		if (m_posesLowRes[i])
		{
			delete m_posesLowRes[i];
		}
	}

	// remove ragdoll from world
	if (m_ragdollInstance->getWorld())
	{
		m_ragdollInstance->removeReference();
	}

	// The ragdoll bodies are loaded from a packfile will not have their destructors called (the memory is just dumped)
	// Because we change from keyframed to dynamic we need to make sure the dtor is called to destroy the motion
	if (m_landscape->getWorld()) 
	{
		m_landscape->removeReference();
	}

	m_npWorld->removeReference();

	delete m_loader;
}

hkDemo::Result NpRagdollPenetrationDetectionDemo::stepDemo()
{
	// update local time
	m_time += m_timestep;

	// process control input
	bool statusChanged = false;
	{
		const hkReal incTurn = 0.015f;
		const hkReal incMove = 0.015f;

		const hkgPad* pad = m_env->m_gamePad;

		if ( pad->wasButtonPressed(HKG_PAD_BUTTON_1) )
		{
			m_statusFlag = !m_statusFlag;
			statusChanged = true;
		}

		if ( pad->wasButtonPressed(HKG_PAD_BUTTON_2) )
		{
			m_poseId++;
			m_poseId = (m_poseId > (NUM_POSES-1)) ? 0 : m_poseId;
		}

		if ( pad->wasButtonPressed(HKG_PAD_BUTTON_3) )
		{
			m_correctionFlag = !m_correctionFlag;
		}

		if ( pad->isButtonPressed(HKG_PAD_DPAD_RIGHT) )
		{
			m_turn -= incTurn;
		}

		if ( pad->isButtonPressed(HKG_PAD_DPAD_LEFT) )
		{
			m_turn += incTurn;
		}

		if ( pad->isButtonPressed(HKG_PAD_DPAD_DOWN)  && !(pad->isButtonPressed(HKG_PAD_BUTTON_0)) )
		{
			m_pos(0) += cos(m_turn + 0.5f*HK_REAL_PI)*incMove;
			m_pos(1) += sin(m_turn + 0.5f*HK_REAL_PI)*incMove;
		}

		if ( pad->isButtonPressed(HKG_PAD_DPAD_UP) && !(pad->isButtonPressed(HKG_PAD_BUTTON_0)) )
		{
			m_pos(0) -= cos(m_turn + 0.5f*HK_REAL_PI)*incMove;
			m_pos(1) -= sin(m_turn + 0.5f*HK_REAL_PI)*incMove;
		}

		if (pad->isButtonPressed(HKG_PAD_DPAD_DOWN) && pad->isButtonPressed(HKG_PAD_BUTTON_0) )
		{
			m_pos(2) -= incMove;
		}

		if (pad->isButtonPressed(HKG_PAD_DPAD_UP) && pad->isButtonPressed(HKG_PAD_BUTTON_0) )
		{
			m_pos(2) += incMove;
		}
	}

	// move and rotate pose as necessary
	{
		// set rotation
		hkQuaternion q0;
		q0.setAxisAngle( UP , m_turn );

		m_currentTransform.m_rotation = q0;
		m_currentTransform.m_translation = m_pos;
	}

	HK_TIMER_BEGIN("WholePenetration", HK_NULL);

	// drive ragdoll, detect penetration and smoothly
	{
		if (m_statusFlag)
		{

			// detect penetration
			{
				HK_TIMER_BEGIN("DetectPenetration", HK_NULL);

				// ModelPose is set directly to WorldPose used for penetration detection
				hkaPose ragdollPose( m_ragdollInstance->getSkeleton() );
				m_ragdollInstance->getPoseWorldSpace( ragdollPose.accessUnsyncedPoseModelSpace().begin() );

				// Method detectPenetration returns true if there is a penetration, or false if there isn't
				hkBool penetrationStatus; penetrationStatus = m_detectRagdollPenetration->detectPenetration( ragdollPose, m_penetratedBones );

				HK_TIMER_END();
			}

			// fully dynamics ragdoll
			stopRagdollFeedback();
		}
		else
		{
			// ragdoll driven by keyframed pose
			doRagdollFeedback();

			// keyframed state, reset bone State
			m_detectRagdollPenetration->resetBoneStates();
		}
	}

	HK_TIMER_END();

	if( statusChanged )
	{
		// Refresh display of dynamic bodies, so they change color
		m_npShapeViewer->refreshAllBodies( m_npWorld, hknpCollisionFlags::IS_DYNAMIC );
	}

	hkDefaultPhysicsDemo::stepDemo();
	// display detection pose
	displayDetectionPose();

	// display application status
	{
		char strOutStats[4096];
		int xPos = m_env->m_window->getTVDeadZoneH();
		int yPos = m_env->m_window->getTVDeadZoneV();

		char strStatus[16];
		{
			if (m_statusFlag)
				hkString::sprintf(strStatus,"%s","dynamic");
			else
				hkString::sprintf(strStatus,"%s","keyframed");
		}

		char strCorrection[4];
		{
			if (m_correctionFlag)
				hkString::sprintf(strCorrection,"%s","yes");
			else
				hkString::sprintf(strCorrection,"%s","no");
		}

		// display status info text after help
		if ( m_time > 15.0f )
		{
			hkString::sprintf( strOutStats, " Ragdoll: %s   Pose: %d/%d   Correction: %s ", strStatus ,m_poseId+1, NUM_POSES, strCorrection );
			m_env->m_textDisplay->outputText(strOutStats, xPos+20, yPos+20);
		}

	}

	return DEMO_OK;	// original animation demo return
}

// doRagdollFeedback() drives the rigid bodies that make up the ragdoll keyframed.
void NpRagdollPenetrationDetectionDemo::doRagdollFeedback()
{
	// set ragdoll to requested pose
	m_ragdollInstance->setPoseModelSpace(m_posesLowRes[m_poseId]->getSyncedPoseModelSpace().begin(), m_currentTransform);

	// set all bodies to keyframed motion
	for ( int i = 0; i < m_ragdollInstance->getNumBones(); i++)
	{
		// Initialize with quality type and collision filter
		if (m_ragdollInstance->getBodyOfBone(i).isValid())
		{
			setBodyKeyframed( i );
		}
	}
}

// stopRagdollFeedback() turns the non-penetrating rigid bodies to fully dynamics, with collisions between them, except parent vs. child.
// The penetrated rigid bodies and their descendants are fully dynamics too, but with disable collision with landscape and correction impulse
// is applied to them!
void NpRagdollPenetrationDetectionDemo::stopRagdollFeedback()
{
	// get boneStatusArray from hkaNpRagdollPenetrationDetector object
	const hkArray<hkaNpRagdollPenetrationDetector::BonePenetrationStatus>& boneStatusArray = m_detectRagdollPenetration->getBoneStatusArray();

	for ( int b = 0; b < m_ragdollInstance->getNumBones(); b++)
	{
		hknpBodyId rb = m_ragdollInstance->getBodyOfBone(b);

		if ( rb.isValid() )
		{
			const int parentId = m_ragdollInstance->getParentOfBone( b );

			if ( ( boneStatusArray[b] != hkaNpRagdollPenetrationDetector::HK_NOP )  &&  m_correctionFlag )
			{
				if ( boneStatusArray[b] == hkaNpRagdollPenetrationDetector::HK_YESP )
				{
					// apply correction force
					setBodyDynamicWithCorrection( rb, b, parentId, true );
				}
				else
				{
					// don't apply correction force
					setBodyDynamicWithCorrection( rb, b, parentId, false );
				}

			}
			else
			{
				setBodyDynamic( b );
			}
		}
	}
}

// Given a rigid body (a ragdoll bone), make it dynamic, enable collisions with all bones except its parent,
// and set the layer and quality properties accordingly. If body is penetrated, correction impulse is applied in
// penetration point in direction similar to normal of raycast triangle to move body out of landscape!
void NpRagdollPenetrationDetectionDemo::setBodyDynamicWithCorrection( hknpBodyId rb, int boneId, int parentId, bool applyForce )
{
	// don't allow  collision with landscape;
	const hkUint32 newFi = hknpGroupCollisionFilter::calcFilterInfo( NpRagdollPenetrationDetectionDemo::LAYER_RAGDOLL_PENETRATING,1, boneId+1, parentId+1);
	
	if (rb.isValid()) 
	{	
		const hknpBody& body = m_npWorld->getBody(rb);
		const hknpMotion& motion = m_npWorld->getMotion(body.m_motionId);
		m_npWorld->setBodyCollisionFilterInfo(rb, newFi );

		// Only dynamic keyframed bodies can be set to normal 
		if ( body.isDynamic() && motion.hasInfiniteMass() )
		{
			hknpMotionCinfo motionCinfo;
			motionCinfo.initializeFromBody(body, m_masses[boneId] );

			m_npWorld->setBodyMotion(rb, m_npWorld->createMotion(motionCinfo));
		}
		
		const hknpMotion& newMotion = m_npWorld->getMotion(body.m_motionId);

		// Apply small impulse in collision point in landscape normal collision direction
		{

			hkVector4 hitPoint;  hitPoint.setZero4();
			hkVector4 hitNormal; hitNormal.setZero4();

			if ( applyForce )
			{

				// get hitPoint and hitNormal
				{
					for(int i = 0; i< m_penetratedBones.m_penetrations.getSize(); i++)
					{
						if( boneId ==  m_penetratedBones.m_penetrations[i].m_boneBeginIndex )
						{
							hitPoint = m_penetratedBones.m_penetrations[i].m_penetrationPoint;
							hitNormal = m_penetratedBones.m_penetrations[i].m_penetrationNormal;

						}
					}
				}

				// Impulse = limbmass*(vdesires - vactual*cos(alpha))
				const hkReal desiredVel = 1.5f;
				const hkReal impulseLimit = 15.0f;

				// get point velocity in hit point
				hkVector4 pointVel;
				newMotion._getPointVelocity(hitPoint, pointVel);
				hkReal velMag = pointVel.length3();

				hkReal cosAlpha;

				// correct problem - unable normalize very small vector
				if ( velMag > HK_REAL_EPSILON )
				{
					pointVel.normalize3();
					cosAlpha = hitNormal.dot3(pointVel);
				}
				else
				{
					cosAlpha = 0.0f;
				}

				hkReal impulseMag = newMotion.getMass().getReal() * ( desiredVel-cosAlpha*velMag );

				impulseMag = (impulseMag > impulseLimit) ? impulseLimit : impulseMag;
				impulseMag = (impulseMag < -impulseLimit) ? -impulseLimit : impulseMag;

				hkVector4 correctionImpulse;
				correctionImpulse.setMul4( impulseMag,hitNormal );

				m_npWorld->applyBodyImpulseAt(rb, correctionImpulse, hitPoint);

				// hit point and correctImpulse visualization
				correctionImpulse.mul4(0.1f);
				HK_DISPLAY_STAR( hitPoint, 0.4f, hkColor::CYAN );
				HK_DISPLAY_ARROW( hitPoint,correctionImpulse, hkColor::ORANGE );
			}
		} // end apply small impulse
	}
}

// Given a rigid body (a ragdoll bone), make it dynamic, enable collisions with all bones except its parent,
// and set the layer and quality properties accordingly. No correction impulse!
void NpRagdollPenetrationDetectionDemo::setBodyDynamic(int boneId )
{
	int parentId = m_ragdollInstance->getParentOfBone(boneId);
	const hkUint32 newFi = hknpGroupCollisionFilter::calcFilterInfo(NpRagdollPenetrationDetectionDemo::LAYER_RAGDOLL_DYNAMIC, 1, boneId+1, parentId+1);

	hknpBodyId rb = m_ragdollInstance->getBodyOfBone(boneId);

	if (rb.isValid()) 
	{	
		const hknpBody& body = m_npWorld->getBody(rb);
		const hknpMotion& motion = m_npWorld->getMotion(body.m_motionId);
		
		m_npWorld->setBodyCollisionFilterInfo(rb, newFi );

		// Only dynamic keyframed bodies can be set to normal 
		if (body.isDynamic() && motion.hasInfiniteMass() )
		{
			hknpMotionCinfo motionCinfo;
			motionCinfo.initializeFromBody(body, m_masses[boneId] );

			m_npWorld->setBodyMotion(rb, m_npWorld->createMotion(motionCinfo));
		}
	}
}

// Given a rigid body (a ragdoll bone), makes it keyframed
void NpRagdollPenetrationDetectionDemo::setBodyKeyframed(int boneId )
{
	hknpBodyId rb = m_ragdollInstance->getBodyOfBone(boneId);
	
	if (rb.isValid())
	{
		const hkUint32 fi = hknpGroupCollisionFilter::calcFilterInfo(NpRagdollPenetrationDetectionDemo::LAYER_RAGDOLL_KEYFRAMED, 1, 0, 0);

		const hknpBody& body = m_npWorld->getBody(rb);
		const hknpMotion& motion = m_npWorld->getMotion(body.m_motionId);
	
		// Only dynamic non-keyframed bodies can be set to keyframed
		if (body.isDynamic() && !motion.hasInfiniteMass() )
		{
			// Save the mass for body rebuilding
			m_masses[boneId] = motion.getMass();	
			m_npWorld->setBodyKeyframed(rb);
			m_npWorld->setBodyCollisionFilterInfo(rb, fi);
		}
	}
}

void NpRagdollPenetrationDetectionDemo::displayDetectionPose()
{
	const hkInt32 numBones = m_ragdollInstance->getNumBones();

	// get pose in world space
	hkArray<hkQsTransform> ragdollWorldSpaceArray( numBones );
	m_ragdollInstance->getPoseWorldSpace( ragdollWorldSpaceArray.begin() );

	hkaPose thePoseWS ( m_ragdollInstance->getSkeleton() );
	thePoseWS.setPoseModelSpace(ragdollWorldSpaceArray); // ModelPose is set directly to WorldPose

	AnimationUtils::drawPose( thePoseWS, hkQsTransform::getIdentity(), hkColor::GREENYELLOW );

	// Display end-up limbs and show bones indexes
	{
		hkVector4 endPointMS;
		hkVector4 bonePointFrom;
		hkVector4 bonePointTo;

		for (int b = 0; b < numBones; b++)
		{
			// Show bones ids in world space 3D position
			const hkVector4& pos = thePoseWS.getBoneModelSpace(b).getTranslation();
			hkStringBuf boneId; boneId.printf("%d", b);
			m_env->m_textDisplay->outputText3D( boneId , pos(0), pos(1), pos(2), 0xffffffff, 1);
		}
	}
}

#if defined(HK_COMPILER_MWERKS)
#	pragma force_active on
#	pragma fullpath_file on
#endif

static const char helpString[] = \
	"This demo demonstrates the usage of the penetration detector and technique how to smoothly change \
	animation driven ragdoll to fully dynamic ragdoll during penetration with landscape! Root bone (zero) has to be out of landscape on the 'correct' side!"\
	" Usage: Select active pose (1/6) and correction mode (yes/no), move ragdoll to landscape and switch to dynamic state by pressing \x11!\n"\
	" Use \x16/\x17 to turn and \x14/\x15 to move the ragdoll\n" \
	" Press \x10 and \x14/\x15 to change the altitude\n" \
	" Press \x11 to change the ragdoll state\n" \
	" Press \x12 to change the active pose\n" \
	" Press \x13 to change the correction mode";

HK_DECLARE_DEMO(NpRagdollPenetrationDetectionDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_SERIALIZE, "Penetration Detection Demo - Detection and correction of penetrating ragdoll", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
