/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>
#include <Demos/Animation/Api/Ragdoll/ConstraintMatching/NpConstraintMatchingDemo.h>
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>
#include <Animation/PhysicsBridge/Ragdoll/Controller/KeyFrameHierarchy/hknpRagdollKeyFrameHierarchyController.h>
#include <Animation/PhysicsBridge/Ragdoll/hknpRagdoll.h>
#include <Animation/PhysicsBridge/RagdollUtils/hknpRagdollUtils.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>
#include <Common/Visualize/hkProcessFactory.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics/Physics/Extensions/Viewers/Shape/hknpShapeViewer.h>

// For HKA_DEMO defines
#include <Demos/DemoCommon/DemoFramework/hkDefaultAnimationDemo.h>

static const char Model[] = "Resources/Animation/NpRagdoll/Firefighter_Rig.hkt";
static const char Animation[] = "Resources/Animation/ShowCase/Animations/hkRun.hkt";

NpConstraintMatchingDemo::NpConstraintMatchingDemo( hkDemoEnvironment* env )
:	hkDefaultPhysicsDemo(env)
{
	m_loader = new hkLoader();

	// Setup the camera
	{
		hkVector4 from(0,1,5);
		hkVector4 to(0.0f, 1.0f, 0.0f);;
		hkVector4 up = to;

		setupDefaultCameras( env, from, to, up );
	}

	// set up the world
	{
		hknpWorldCinfo info;
		info.setBroadPhaseSize( 500.0f );
		info.m_gravity.setAll(0);
		info.m_gravity.set(0,-9.8f,0);
		info.m_persistentStreamAllocator = getPersistentAllocator();

		hknpGroupCollisionFilter* filter  = new hknpGroupCollisionFilter();
		info.m_collisionFilter = filter;

		m_npWorld.setAndDontIncrementRefCount( new hknpWorld( info ) );
		filter->removeReference();
	}

	// Create a floor for our ragdoll to stand on.
	{
		hkVector4 halfExtents(3,.1f,3);

		hknpShape* boxShape = hknpConvexShape::createFromHalfExtents(halfExtents);

		hknpBodyCinfo info;
		info.m_shape = boxShape;
		info.m_position.set(0,-1,0);

		m_npWorld->createStaticBody( info );
		boxShape->removeReference();
	}

	// Load the following from m_skeletonFilename:
	//	- rig skeleton
	//  - ragdoll instance
	//  - skeleton mappers
	{
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(Model) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		// Grab the skeleton
		HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "Couldn't load high-res skeleton");
		m_highSkeleton = ac->m_skeletons[0];

		// Grab the ragdoll data and optimize it
		hknpRagdollData* ragdollData = reinterpret_cast<hknpRagdollData*>( container->findObjectByType( hknpRagdollDataClass.getName() ));
		HK_ASSERT2(0, ragdollData, "Couldn't load ragdoll setup");
		hknpRagdollUtils::optimizeInertias( ragdollData );

		// Create our instance
		m_ragdoll = new hknpRagdoll( ragdollData, m_npWorld, hkTransform::getIdentity() );

		// Grab the skeleton mappers
		// We iterate through all the skeleton mappers contained in the file,
		// and compare them against the ragdoll's skeleton to find out which one is the
		// low-resolution skeleton
		const hkaSkeleton* tempLowSkeleton = ragdollData->m_skeleton;
		void *objectFound = container->findObjectByType(hkaSkeletonMapperClass.getName());
		m_highToLowMapper = HK_NULL;
		while (objectFound)
		{
			hkaSkeletonMapper* mapperFound = reinterpret_cast<hkaSkeletonMapper*> (objectFound);

			if (mapperFound->m_mapping.m_skeletonB == tempLowSkeleton)
			{
				m_highToLowMapper = mapperFound;
			}

			objectFound = container->findObjectByType(hkaSkeletonMapperClass.getName(), objectFound);
		}
		HK_ASSERT2(0x22440117, m_highToLowMapper, "Couldn't load high-to-ragdoll mapping");
	}

	// Load the following from m_animationFilename:
	//	- animation
	//  - animation binding
	//  - skin
	{
		hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(Animation) );
		HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
		hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

		// Grab the animation
		HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0), "Couldn't load high-res animation");
		m_animation = ac->m_animations[0];

		// Grab the animation binding
		HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "Couldn't load high-res binding");
		m_binding = ac->m_bindings[0];

		// Create an animation control
		hkaDefaultAnimationControl* control = new hkaDefaultAnimationControl(m_binding);

		// Create a new animated skeleton
		m_animatedSkeleton = new hkaAnimatedSkeleton( m_highSkeleton );

		// Bind the control to the skeleton
		m_animatedSkeleton->addAnimationControl( control );

		// The animated skeleton now owns the control
		control->removeReference();
	}

	// Lock translations on all the bones, then unlock them only on the pelvis
	{
		hkaSkeleton& skeletonA = *const_cast<hkaSkeleton*>(m_highToLowMapper->m_mapping.m_skeletonA.val());
		hkaSkeletonUtils::lockTranslations(skeletonA);

		const hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName(skeletonA, "Root");
		if (pelvis >= 0)
		{
			skeletonA.m_bones[pelvis].m_lockTranslation = false;
		}
	}

	// Lock translations on all the bones, then unlock them only on the pelvis
	{
		hkaSkeleton& skeletonB = *const_cast<hkaSkeleton*>(m_highToLowMapper->m_mapping.m_skeletonB.val());
		hkaSkeletonUtils::lockTranslations(skeletonB);

		hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName(skeletonB, "Biped_Root");
		if (pelvis >=0 )
		{
			skeletonB.m_bones[pelvis].m_lockTranslation = false;
		}
	}

	setupGraphics();

	setupRagdoll();
}

void NpConstraintMatchingDemo::setupRagdoll()
{
	// Initialize collision/motion settings for the rigidbody ragdoll parts
	for ( int b =0; b < m_ragdoll->getNumBones(); b++)
	{
		hknpBodyId bodyId = m_ragdoll->getBodyOfBone(b);
		if ( !bodyId.isValid() )
		{
			continue;
		}

		m_npWorld->setBodyCollisionFilterInfo( bodyId, hknpGroupCollisionFilter::calcFilterInfo(0, 1 ) );
		HK_ASSERT2(0x22440123, !m_npWorld->getBody( bodyId ).isStaticOrKeyframed(), "Non-dynamic body in ragdoll" );
	}

	// set up the rigid body controllers
	m_keyFrameController = new hknpRagdollKeyFrameHierarchyController( m_ragdoll, m_npWorld );
	m_keyFrameController->update( hkQsTransform::getIdentity(), m_timestep );
}

NpConstraintMatchingDemo::~NpConstraintMatchingDemo()
{
	m_keyFrameController->removeReference();
	m_ragdoll->removeReference();
	m_animatedSkeleton->removeReference();
	m_highToLowMapper->removeReference();
	delete m_loader;
}

hkDemo::Result NpConstraintMatchingDemo::stepDemo()
{
	// Advance the animated skeleton
	m_animatedSkeleton->stepDeltaTime( m_timestep );

	// Transform used for display
	hkQsTransform worldFromModel (hkQsTransform::IDENTITY );

	// Get a high-res pose from the high resolution animated skeleton
	hkaPose poseHigh (m_highSkeleton);
	m_animatedSkeleton->sampleAndCombineAnimations( poseHigh.accessUnsyncedPoseLocalSpace().begin(), poseHigh.getFloatSlotValues().begin());

	// Construct a low-res pose in the reference pose
	const hkaSkeleton* lowSkeleton = m_ragdoll->getSkeleton();
	hkaPose poseLow (lowSkeleton);
	poseLow.setToReferencePose();

	// Map the high-res pos to the low-res pose
	m_highToLowMapper->mapPose(poseHigh, poseLow, hkaSkeletonMapper::CURRENT_POSE);

	// Drive the ragdoll's rigid bodies to the proper position
	hkLocalBuffer<hkQsTransform> lowResLocalSpace( lowSkeleton->m_bones.getSize() );
	hkaSkeletonUtils::transformModelPoseToLocalPose( lowSkeleton->m_bones.getSize(), lowSkeleton->m_parentIndices.begin(), poseLow.getSyncedPoseModelSpace().begin(), lowResLocalSpace.begin() );
	m_keyFrameController->setDesiredPose( lowResLocalSpace.begin() );
	m_keyFrameController->driveToPose();
	
	// Since the rigid body controller works by setting velocities,
	// step the physics world AFTER we drive to the pose.
	hkDefaultPhysicsDemo::stepDemo();

	showConstraintStress();

	return DEMO_OK;
}

// Go through each constraint and run the appropriate relaxation function on it.
void NpConstraintMatchingDemo::showConstraintStress()
{
	for (int i=0; i<m_ragdoll->getConstraintArray().getSize(); i++)
	{
		hknpConstraintId constraintId = m_ragdoll->getConstraintArray()[i];
		const hknpConstraint* constraint = &m_npWorld->getConstraint( constraintId );
		int type = constraint->m_data->getType();

		char buffer[128];
		hkReal stress = 0;
		if (type == hkpConstraintData::CONSTRAINT_TYPE_LIMITEDHINGE)
		{
			stress = checkLimitedHingeConstraint(constraint, i);
			hkString::sprintf(buffer, "Hinge  ");
		}
		else if (type == hkpConstraintData::CONSTRAINT_TYPE_RAGDOLL)
		{
			stress = checkRagdollConstraint(constraint, i);
			hkString::sprintf(buffer, "Ragdoll  ");
		}

		m_env->m_textDisplay->outputText( buffer, 20, 100+20*i, hkColor::WHITE );

		// blend the color between red and white
		int offset = (int) ( 255 * (1.0f - hkMath::min2(stress, hkReal(1.0f))));
		hkColor::Argb color = hkColor::RED;
		color |= (offset << 8);
		color |= (offset << 0);
		hkString::sprintf(buffer, "%2.3f", stress);
		m_env->m_textDisplay->outputText( buffer, 100, 100+20*i, color );
	}
}

hkReal NpConstraintMatchingDemo::checkLimitedHingeConstraint(const hknpConstraint* constraint, int i)
{
	// Get the solver results so that we can check the impulse that gets applied
	const hkpLimitedHingeConstraintData* data = static_cast<const hkpLimitedHingeConstraintData*>(constraint->m_data.val());
	data = data; // to avoid compiler warning
	hkpSolverResults* solverResults = &data->getRuntime( constraint->m_runtime )->m_solverResults[0];

	hkReal impulse = solverResults[hkpLimitedHingeConstraintData::SOLVER_RESULT_LIMIT].getImpulseApplied();

	// change the color of the attached rigid bodies for visualization purposes
	colorConstraintByStress(constraint, hkMath::fabs(impulse));

	return hkMath::fabs(impulse);
}

hkReal NpConstraintMatchingDemo::checkRagdollConstraint(const hknpConstraint* constraint, int i)
{
	const hkpRagdollConstraintData* data = static_cast<const hkpRagdollConstraintData*>(constraint->m_data.val());
	data = data; // to avoid compiler warning
	hkpSolverResults* solverResults = &data->getRuntime( constraint->m_runtime )->m_solverResults[0];

	hkReal impulseCone = solverResults[hkpRagdollConstraintData::SOLVER_RESULT_CONE].getImpulseApplied();
	hkReal impulsePlane = solverResults[hkpRagdollConstraintData::SOLVER_RESULT_PLANE].getImpulseApplied();
	hkReal impulseTwist = solverResults[hkpRagdollConstraintData::SOLVER_RESULT_TWIST].getImpulseApplied();

	// change the color of the attached rigid bodies for visualization purposes
	hkReal stress = hkMath::sqrt(impulseCone*impulseCone + impulsePlane*impulsePlane + impulseTwist*impulseTwist);
	colorConstraintByStress(constraint, stress);

	return stress;
}

void NpConstraintMatchingDemo::colorConstraintByStress(const hknpConstraint* constraint, hkReal stress)
{
	// blend between default body color and red
	hkColor::Argb color;
	{
		stress = hkMath::max2( .5f*(stress - 0.5f), hkReal(0.0f) );
		stress = hkMath::min2( stress * 0.2f, hkReal(1.0f) );

		hkReal r = hkColor::getRedAsFloat(hknpDefaultViewerColorScheme::DYNAMIC_BODY_COLOR);
		hkReal g = hkColor::getGreenAsFloat(hknpDefaultViewerColorScheme::DYNAMIC_BODY_COLOR);
		hkReal b = hkColor::getBlueAsFloat(hknpDefaultViewerColorScheme::DYNAMIC_BODY_COLOR);

		r = (1.f - r) * stress +r;
		g = (1.f - stress) * g;
		b = (1.f - stress) * b;

		color = hkColor::rgbFromFloats(r,g,b);
	}

	// Set the property for the body
	m_npWorld->setBodyProperty( constraint->m_bodyIdA, hknpBodyPropertyKeys::DEBUG_DISPLAY_COLOR, color );

	// Update the shape viewer in case the body has already been added
	{
		for ( int i = 0; i < m_debugProcesses.getSize(); i++ )
		{
			const char* processName = hkProcessFactory::getInstance().getProcessName( m_debugProcesses[i]->getProcessTag() );
			if ( hkString::strCmp( processName, hknpShapeViewer::getName() ) == 0 )
			{
				hknpShapeViewer* shapeViewer = static_cast<hknpShapeViewer*>( m_debugProcesses[i] );
				shapeViewer->setBodyColor( m_npWorld, constraint->m_bodyIdA, color );
			}
		}
	}
}


#if defined(HK_COMPILER_MWERKS)
#	pragma force_active on
#	pragma fullpath_file on
#endif

static const char helpString[] = \
"This demo maps an animation to a ragdoll and shows when the range of motion allowed by the constraints is violated by the animation.";
HKA_DECLARE_PHYSICS_DEMO(NpConstraintMatchingDemo, HK_DEMO_TYPE_ANIMATION | HK_DEMO_TYPE_CRITICAL, "Demo showing how well an animation matches a ragdolls the constraint limits", helpString);

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
