/*
 *
 * Confidential Information of Telekinesys Research Limited (t/a Havok). Not for disclosure or distribution without Havok's
 * prior written consent. This software contains code, techniques and know-how which is confidential and proprietary to Havok.
 * Product and Trade Secret source code contains trade secrets of Havok. Havok Software (C) Copyright 1999-2014 Telekinesys Research Limited t/a Havok. All Rights Reserved. Use of this software is subject to the terms of an end user license agreement.
 *
 */

#include <Demos/demos.h>

#include <Demos/Animation/Api/Ragdoll/BlendTest/NpRagdollBlendTestDemo.h> 

// Common
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>
#include <Common/Serialize/Util/hkLoader.h>
#include <Common/Serialize/Util/hkRootLevelContainer.h>
#include <Common/Visualize/hkDebugDisplay.h>

// Physics
#include <Physics/Physics/Collide/Filter/Constraint/hknpConstraintCollisionFilter.h>
#include <Physics/Physics/Extensions/PhysicsSystem/hknpPhysicsSystem.h>
#include <Physics/Physics/Extensions/PhysicsSystem/hknpPhysicsSceneData.h>
#include <Physics/Physics/Extensions/Viewers/Shape/hknpShapeViewer.h>

// Constraints
#include <Physics/Constraint/Data/hkpConstraintDataUtils.h>
#include <Physics/Constraint/Motor/Position/hkpPositionConstraintMotor.h>

// Animation
#include <Animation/Animation/hkaAnimationContainer.h>
#include <Animation/Animation/Mapper/hkaSkeletonMapperUtils.h>
#include <Animation/Animation/Playback/Control/Default/hkaDefaultAnimationControl.h>
#include <Animation/Animation/Playback/hkaAnimatedSkeleton.h>
#include <Animation/Animation/Rig/hkaPose.h>

// Ragdolls
#include <Animation/PhysicsBridge/Ragdoll/hknpRagdoll.h>
#include <Animation/PhysicsBridge/RagdollUtils/hknpRagdollUtils.h>
#include <Animation/PhysicsBridge/Ragdoll/Controller/Motor/hknpRagdollMotorController.h>
#include <Animation/PhysicsBridge/Ragdoll/Controller/KeyFrameHierarchy/hknpRagdollKeyFrameHierarchyUtility.h>
#include <Animation/PhysicsBridge/Ragdoll/Controller/KeyFrameHierarchy/hknpRagdollKeyFrameHierarchyController.h>
#include <Physics/Constraint/Data/Ragdoll/hkpRagdollConstraintData.h>
#include <Physics/Constraint/Data/LimitedHinge/hkpLimitedHingeConstraintData.h>

// Demos utilities
#include <Demos/DemoCommon/Utilities/Animation/AnimationUtils.h>
#include <Demos/DemoCommon/Utilities/Asset/hkAssetManagementUtil.h>

extern const hkClass hknpRagdollDataClass;

inline void stripPrefix( hkStringBuf& strInOut )
{
	// Replace spaces by underscores.
	strInOut.replace(' ','_');

	// Remove skeleton prefix.
	// This the text before the first underscore.
	const int endPrefix = strInOut.indexOf('_',0);

	if (endPrefix!=-1)
	{
		strInOut.chompStart(endPrefix+1);
	}
}

// This is the string comparison we use to match names from the animation skeleton
// to names in the ragdoll skeleton. The filter just extracts the bone name,
// from either, so we will use the same filter for both skeletons.
// For example, it will match "Rig Pelvis" to "Ragdoll_Pelvis".
static int HK_CALL hkCompareWithoutPrefix( const char* str1, const char* str2 )
{
	hkStringBuf string1(str1);
	stripPrefix(string1);

	hkStringBuf string2(str2);
	stripPrefix(string2);
	hkStringBuf* smallest;
	hkStringBuf* largest;

	if ( string1.getLength() > string2.getLength() )
	{
		smallest = &string2;
		largest = & string1;
	}
	else
	{
		smallest = &string1;
		largest = & string2;
	}
	if ( smallest->compareTo("HavokBipedRig") == 0)
	{
		return 1;
	}
	return (largest->indexOf(smallest->cString()) == -1 ) 
	?	1
	:	0;
}




struct BlendTestVariant
{
	const char*	 m_name;
	int    m_iterations;
	bool   m_fixRoot;
	bool   m_useKeyFrameUtility;

	hkReal	m_motorTau;
	hkReal	m_motorMaxForce;
	const char* m_details;
};

static const char helpString[] = "";

static const BlendTestVariant g_variants[] =
{
	//					name							solver      Fix		use    motor parameters
	//													iterations	root	KF
	{ "A0 Free root, soft motor controller, 4 iterations",	4,		false, false, 0.2f, 100.0f, helpString },
	{ "A1 Free root, soft motor controller, 8 iterations",	8,		false, false, 0.15f,100.0f, helpString },
	{ "A2 Free root, soft motor controller, 16 iterations",16,		false, false, 0.1f, 100.0f, helpString },
	{ "A3 Free root, hard motor controller, 4 iterations",	4,		false, false, 0.8f, 1000.0f, helpString },
	{ "A4 Free root, hard motor controller, 8 iterations",	8,		false, false, 0.8f, 1000.0f, helpString },
	{ "A5 Free root, hard motor controller, 16 iterations",16,		false, false, 0.8f, 1000.0f, helpString },
	{ "A6  Keyframe Hierarchy controller ",				  	4,		false, true, 0.0f,  00.0f, helpString },
	{ "B0 Fixed root, soft motor controller, 4 iterations",	4,		true, false, 0.2f, 100.0f, helpString },
	{ "B1 Fixed root, soft motor controller, 8 iterations",	8,		true, false, 0.15f,100.0f, helpString },
	{ "B2 Fixed root, soft motor controller, 16 iterations",16,		true, false, 0.2f, 100.0f, helpString },
	{ "B3 Fixed root, hard motor controller, 4 iterations",	4,		true, false, 0.8f, 500.0f, helpString },
	{ "B4 Fixed root, hard motor controller, 8 iterations",	8,		true, false, 0.8f, 500.0f, helpString },
	{ "B5 Fixed root, hard motor controller, 16 iterations",16,		true, false, 0.8f, 500.0f, helpString },
	{ "B6 Fixed root, Keyframe Hierarchy controller ",	  	4,		true, true, 0.0f,  00.0f, helpString },
};
#define USE_HIGHRES_RAGDOLL 1

NpRagdollBlendTestDemo::NpRagdollBlendTestDemo(hkDemoEnvironment* env)
:	hkDefaultPhysicsDemo(env)
,	m_loader(HK_NULL)
,	m_highToLowMapper(HK_NULL)
,	m_lowToHighMapper(HK_NULL)
,	m_animatedSkeleton(HK_NULL)
,	m_ragdollData(HK_NULL)
,	m_ragdoll(HK_NULL)
,	m_motor(HK_NULL)
{

	//
	// Disable warnings:
	//
	setErrorEnabled(0xad67888e, false); //'Function not implemented.'
	setErrorEnabled(0x9FE65234, false); // wrong simulation type in old asset

	for (int i = 0; i < NUM_ANIMS; i++)
	{
		m_control[i] = HK_NULL;
	}

	const BlendTestVariant& variant =  g_variants[ this->getDemoVariant() ];
	// Set up the camera.
	{
		hkVector4 from( -3,-3,3 );
		hkVector4 to  ( 0,0,0 );
		hkVector4 up  ( 0.0f, 0.0f, 1.0f );
		setupDefaultCameras( env, from, to, up, 0.1f, 100 );
	}

	//
	// Create physics world and objects.
	// 
	{
		hknpConstraintCollisionFilter * filter = new hknpConstraintCollisionFilter(NULL);

		// Create the World.
		{
			hknpWorldCinfo info;
			info.m_persistentStreamAllocator = getPersistentAllocator();
			info.m_gravity.set(0,0,-10);
			info.setSolverType( hknpWorldCinfo::SOLVER_TYPE_4ITERS_MEDIUM );
			info.m_solverIterations = variant.m_iterations;
			info.m_enableDeactivation = false;
			info.m_collisionFilter.setAndDontIncrementRefCount( filter );

			m_npWorld = new hknpWorld( info );

			setupGraphics();
		}

		// Create the floor
		{
			hkVector4 halfExtents(5.0f, 5.0f, 1.0f);
			hknpConvexShape* shape = hknpConvexShape::createFromHalfExtents(halfExtents); 
			hknpBodyCinfo bodyInfo;

			bodyInfo.m_shape = shape;
			#if USE_HIGHRES_RAGDOLL
			bodyInfo.m_position.set(0.0f, 0.0f, -2.f);
			#else 
			bodyInfo.m_position.set(0.0f, 0.0f, -1.f);
			#endif
			m_npWorld->createStaticBody(bodyInfo);

			shape->removeReference();
		}

		// Create a moving box
		{
			hkVector4 halfExtents(0.5f, 0.5f, 0.3f);
			hknpConvexShape* shape = hknpConvexShape::createFromHalfExtents(halfExtents); 
			hknpBodyCinfo bodyInfo;

			bodyInfo.m_shape = shape;
			bodyInfo.m_position.set(0.0f, 3.0f, 0.4f);

			hknpMotionCinfo motionInfo;
			motionInfo.initializeWithMass(&bodyInfo, 1, 20.0f);
			m_npWorld->createDynamicBody(bodyInfo, motionInfo);

			shape->removeReference();
		}

		// Load the ragdoll
		{
			m_loader = new hkLoader();
			#if USE_HIGHRES_RAGDOLL
			hkStringBuf filename("Resources/Animation/NpRagdoll/highres_ragdoll.hkt");
			# else
			hkStringBuf filename("Resources/Animation/NpRagdoll/havokgirl_ragdoll.hkt");
			#endif
			hkAssetManagementUtil::getFilePath(filename);

			
			hkRootLevelContainer* dataContainer = m_loader->load(HK_GET_DEMOS_ASSET_FILENAME(filename.cString()));
			HK_ASSERT( 0x0, dataContainer );
			m_ragdollData = reinterpret_cast<hknpRagdollData*>(dataContainer->findObjectByType(hknpRagdollDataClass.getName()));
			HK_ASSERT( 0x0, m_ragdollData );

			// Add the motors to the constraints.
			if (!variant.m_useKeyFrameUtility)
			{
				m_motor = new hkpPositionConstraintMotor();
				{
					m_motor->m_tau = variant.m_motorTau;
					m_motor->m_maxForce = variant.m_motorMaxForce;
					m_motor->m_proportionalRecoveryVelocity = 5.0f;
					m_motor->m_constantRecoveryVelocity = 0.2f;
				}
				for (int ci=0; ci< m_ragdollData->m_constraintCinfos.getSize(); ci++)
				{
					hkpConstraintDataUtils::convertToPowered(m_ragdollData->m_constraintCinfos[ci].m_constraintData, m_motor, false);	
				}

				hknpRagdollUtils::optimizeInertias( m_ragdollData, hkSimdReal::fromFloat(1.5f) );
			}

			hkTransform t; t.setIdentity();
			m_rootOffset = m_ragdollData->m_bodyCinfos[0].m_position;

			// Create the ragdoll.
			m_ragdoll = new hknpRagdoll(m_ragdollData, m_npWorld,t);

			// Lock the pelvis.
			if ( variant.m_fixRoot )
			{
				hknpBodyId rootId = m_ragdoll->getBodyOfBone(0);
				m_npWorld->setBodyKeyframed(rootId);
			}
		}

		filter->updateFromWorld(m_npWorld);
	}

	//
	// Load Animations.
	//
	{
		hkaSkeleton* animationSkeleton = HK_NULL;

		// Get the rig.
		{
			hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRig.hkt");
			hkAssetManagementUtil::getFilePath(assetFile);
			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
#ifndef HK_PLATFORM_RVL
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
#endif

			if ( container )
			{
				hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));
				HK_ASSERT2(0x27343435, ac && (ac->m_skeletons.getSize() > 0), "No skeleton loaded");
				animationSkeleton = ac->m_skeletons[0];
				animationSkeleton->m_referencePose[0].m_translation.add(m_rootOffset);
			}
		}

		// Create the animated skeleton for the animation.
		m_animatedSkeleton = HK_NULL;
		if ( animationSkeleton )
		{
			m_animatedSkeleton = new hkaAnimatedSkeleton( animationSkeleton );
		}
		
		// Run Animation.
		if ( m_animatedSkeleton )
		{
			hkStringBuf assetFile("Resources/Animation/HavokGirl/hkRunLoop.hkt");
			hkAssetManagementUtil::getFilePath(assetFile);
			hkRootLevelContainer* container = m_loader->load( HK_GET_DEMOS_ASSET_FILENAME(assetFile.cString()) );
#ifndef HK_PLATFORM_RVL
			HK_ASSERT2(0x27343437, container != HK_NULL , "Could not load asset");
#endif
			if ( container )
			{
				hkaAnimationContainer* ac = reinterpret_cast<hkaAnimationContainer*>( container->findObjectByType( hkaAnimationContainerClass.getName() ));

				HK_ASSERT2(0x27343435, ac && (ac->m_animations.getSize() > 0 ), "No animation loaded");
				HK_ASSERT2(0x27343435, ac && (ac->m_bindings.getSize() > 0), "No binding loaded");
				hkaAnimationBinding* runBinding = ac->m_bindings[0];

				m_control[0] = new hkaDefaultAnimationControl (runBinding);
				m_control[0]->setMasterWeight(1.0f);
				m_control[0]->setPlaybackSpeed(1.0f);
				m_control[0]->setEaseInCurve(0, 0.3f, 0.6f, 1);	// Smooth
				m_control[0]->setEaseOutCurve(1, .6f, .3f, 0);	// Smooth
				m_animatedSkeleton->addAnimationControl( m_control[0] );
				m_control[0]->removeReference();
			}
		}
	}

	// 
	// Mapping between the animated skeleton and the ragdoll skeleton.
	// 
	{
		if ( m_animatedSkeleton )
		{
			// Lock the translation of most bones of the animations, except some.
			{
				hkaSkeleton* animationSkeleton = const_cast<hkaSkeleton*>(m_animatedSkeleton->getSkeleton());

				// Locks all translations, except root.
				hkaSkeletonUtils::lockTranslations(*animationSkeleton, true);

				// Unlocks translation for pelvis and both thighs (triangle pelvis).
				const hkInt16 pelvis = hkaSkeletonUtils::findBoneWithName( *animationSkeleton, "Ragdoll Pelvis", hkCompareWithoutPrefix );
				animationSkeleton->m_bones[pelvis].m_lockTranslation = false;

				const hkInt16 lthigh = hkaSkeletonUtils::findBoneWithName( *animationSkeleton, "Ragdoll L_Thigh", hkCompareWithoutPrefix );
				animationSkeleton->m_bones[lthigh].m_lockTranslation = false;

				const hkInt16 rthigh = hkaSkeletonUtils::findBoneWithName( *animationSkeleton, "Ragdoll R_Thigh", hkCompareWithoutPrefix );
				animationSkeleton->m_bones[rthigh].m_lockTranslation = false;
			}
			// Create mappers.
			{
				hkaSkeletonMapperData high_low_data;
				hkaSkeletonMapperData low_high_data;
				{
					hkaSkeletonMapperUtils::Params params;

					params.m_skeletonA = m_animatedSkeleton->getSkeleton();
					params.m_skeletonB = m_ragdoll->getSkeleton();
					params.m_compareNames = hkCompareWithoutPrefix;
					// Explicit mappings.
					// Re-parent the forearms twists to the fore arm.
					#if ! USE_HIGHRES_RAGDOLL
					{
						hkaSkeletonMapperUtils::UserMapping mapping;
						mapping.m_boneIn = "HavokBipedRig L ForeTwist";
						mapping.m_boneOut = "HavokBipedRig L ForeArm";
						params.m_userMappingsBtoA.pushBack(mapping);
						mapping.m_boneIn = "HavokBipedRig R ForeTwist";
						mapping.m_boneOut = "HavokBipedRig R ForeArm";
						params.m_userMappingsBtoA.pushBack(mapping);
					}
					#else
					{
						hkaSkeletonMapperUtils::UserMapping mapping;
						mapping.m_boneIn = "HavokBipedRig L_ForeArm";
						mapping.m_boneOut = "Ragdoll L_ForeTwist";
						params.m_userMappingsBtoA.pushBack(mapping);
						mapping.m_boneIn = "HavokBipedRig R_ForeArm";
						mapping.m_boneOut = "Ragdoll R_ForeTwist";
						params.m_userMappingsBtoA.pushBack(mapping);
					}
					#endif
					params.m_autodetectChains = false;

					hkaSkeletonMapperUtils::createMapping( params, high_low_data, low_high_data );
				}

				m_highToLowMapper = new hkaSkeletonMapper(high_low_data);
				m_lowToHighMapper = new hkaSkeletonMapper(low_high_data);
			}
		}
	}
	// Set up the controller.
	if (variant.m_useKeyFrameUtility)
	{
		hkTransform worldToModel;
		worldToModel.setIdentity();
		worldToModel.getTranslation().add(m_rootOffset);
		hknpRagdollKeyFrameHierarchyUtility::ControlData controlData;
		{
			if ( variant.m_fixRoot )
			{
				controlData.m_hierarchyGain = 0.6f;
				controlData.m_velocityDamping = 0.0f;
				controlData.m_accelerationGain = 1.0f;
				controlData.m_velocityGain = 0.3f;

				controlData.m_positionGain = 0.08f;
				controlData.m_positionMaxLinearVelocity = 0.2f;
				controlData.m_positionMaxAngularVelocity = 0.5f;

				controlData.m_snapGain = 0.2f;
				controlData.m_snapMaxLinearVelocity  = 0.3f;
				controlData.m_snapMaxAngularVelocity = 0.3f;
				controlData.m_snapMaxLinearDistance  = 0.03f;
				controlData.m_snapMaxAngularDistance = 0.1f;
			}
			else
			{
				controlData.m_hierarchyGain = 0.6f;
				controlData.m_velocityDamping = 0.0f;
				controlData.m_accelerationGain = 1.0f;
				controlData.m_velocityGain = 0.2f;

				controlData.m_positionGain = 0.1f;
				controlData.m_positionMaxLinearVelocity = 0.2f;
				controlData.m_positionMaxAngularVelocity = 0.3f;

				controlData.m_snapGain = 0.1f;
				controlData.m_snapMaxLinearVelocity  = 0.3f;
				controlData.m_snapMaxAngularVelocity = 0.3f;
				controlData.m_snapMaxLinearDistance  = 0.03f;
				controlData.m_snapMaxAngularDistance = 0.1f;
			}
		}
		m_controller = new hknpRagdollKeyFrameHierarchyController( m_ragdoll, m_npWorld, controlData );

		// the ragdoll doesn't move every step, and the delta time never changes, so we can just do this once.
		hkQsTransform modelToWorld;
		modelToWorld.setFromTransformNoScale( worldToModel );
		static_cast<hknpRagdollKeyFrameHierarchyController*>( m_controller )->update( modelToWorld, m_timestep );

	}
	else
	{
		m_controller = new hknpRagdollMotorController(m_ragdoll);
	}

	m_playing = true;
}


#define SAFE_RELEASE(obj)\
{\
	if ( (obj) )\
	{\
		(obj)->removeReference();\
		(obj) = HK_NULL;\
	}\
}

NpRagdollBlendTestDemo::~NpRagdollBlendTestDemo()
{

	SAFE_RELEASE(m_motor);
	SAFE_RELEASE(m_lowToHighMapper);
	SAFE_RELEASE(m_highToLowMapper);
	SAFE_RELEASE(m_animatedSkeleton);
	SAFE_RELEASE(m_ragdoll);
	SAFE_RELEASE(m_ragdollData);
	SAFE_RELEASE(m_controller);
	
	m_npWorld->removeReference();
	delete m_loader;
}


hkDemo::Result NpRagdollBlendTestDemo::stepDemo()
{
	hkQsTransform rootTransform;
	hknpBodyId rootId = m_ragdoll->getBodyOfBone(0);
	rootTransform.setIdentity();
	rootTransform.setTranslation(m_rootOffset);

	if (m_motor)
	{
		if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_UP) || m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_DOWN) )
		{
			if( m_env->m_gamePad->isButtonPressed(HKG_PAD_DPAD_UP) )
			{
				m_motor->m_maxForce *= 1.1f;
				m_motor->m_maxForce = m_motor->m_maxForce;
			}
			else
			{
				m_motor->m_maxForce /= 1.1f;
				m_motor->m_maxForce = m_motor->m_maxForce;
			}
		}

		hkStringBuf str; str.printf("Current torque limit: %f   Use up/down arrows to modify.", m_motor ? m_motor->m_maxForce : 0.0f );
		m_env->m_textDisplay->outputText(str, 10, 10);
	}


	const BlendTestVariant& variant =  g_variants[ this->getDemoVariant() ];

	// Ease in or out the ragdoll controller.
	{
		if (m_env->m_gamePad->wasButtonPressed(HKG_PAD_BUTTON_1))
		{
			m_playing = !m_playing;
		}
	}

	if ( !m_animatedSkeleton || !m_highToLowMapper || !m_ragdoll)
	{
		hkDefaultPhysicsDemo::stepDemo();
		return hkDemo::DEMO_OK;
	}

	// Advance the animation.
	if ( m_playing )
	{
		m_animatedSkeleton->stepDeltaTime( m_timestep );
	}

	//// Create a runtime pose for the animation.
	hkaPose animPose(m_animatedSkeleton->getSkeleton());

	//// Sample animation.
	m_animatedSkeleton->sampleAndCombineAnimations( animPose.accessUnsyncedPoseLocalSpace().begin(), animPose.getFloatSlotValues().begin() );

	//// Draw the original animation.
	AnimationUtils::drawPose( animPose, rootTransform);

	//// Create a runtime pose for the ragdoll.
	hkaPose ragdollPose(m_ragdoll->getSkeleton());
	ragdollPose.setToReferencePose();

	//// Map from animation to physics.
	m_highToLowMapper->mapPose( animPose, ragdollPose, hkaSkeletonMapper::CURRENT_POSE );
	//// Set the weight.
	const hkaSkeleton* rSkel = m_ragdoll->getSkeleton();
	hkLocalBuffer<hkQsTransform> ragdollModelSpace( rSkel->m_bones.getSize() );

	//// Drive the ragdoll to that pose.
	{
		hkaSkeletonUtils::transformLocalPoseToModelPose( rSkel->m_bones.getSize(), rSkel->m_parentIndices.begin(), rSkel->m_referencePose.begin(), ragdollModelSpace.begin() );

		// Map the pose from the animation (high) to ragdoll (low).
		m_highToLowMapper->mapPose( animPose.getSyncedPoseModelSpace().begin(), rSkel->m_referencePose.begin(), ragdollModelSpace.begin(), hkaSkeletonMapper::CURRENT_POSE );

		// Drive the powered constraints to this ragdoll pose using a controller.
		{
			hkLocalBuffer<hkQsTransform> ragdollLocalSpace( rSkel->m_bones.getSize() );
			hkaSkeletonUtils::transformModelPoseToLocalPose( rSkel->m_bones.getSize(), rSkel->m_parentIndices.begin(), ragdollModelSpace.begin(), ragdollLocalSpace.begin() );
			m_controller->setDesiredPose(ragdollLocalSpace.begin());
		}

		m_controller->driveToPose();
	}

	if (variant.m_useKeyFrameUtility)
	{
		// Show stress.
		{
			for (int i =0; i < m_ragdoll->getBodyArray().getSize();i++ )
			{
				hknpBodyId bodyId = m_ragdoll->getBodyIds()[i]; 

				int color = m_npWorld->getBody(bodyId).isKeyframed() ?
							hknpDefaultViewerColorScheme::KEYFRAMED_BODY_COLOR:	
							hknpDefaultViewerColorScheme::DYNAMIC_BODY_COLOR; 


				// Linearly interpolate the color between the body color and full red.
				hkReal r,g,b;
				r = hkColor::getRedAsFloat(color);
				g = hkColor::getGreenAsFloat(color);
				b = hkColor::getBlueAsFloat(color);


				hkReal stress = hkMath::sqrt( static_cast<hknpRagdollKeyFrameHierarchyController*>(m_controller)->getOutput(bodyId).m_stressSquared );
				stress = hkMath::max2( stress - 0.5f,  hkReal(0.0f) );
				stress = hkMath::min2( stress * 0.05f, hkReal(1.0f) );

				r = (1.f - r) * stress +r;
				g = (1.f - stress) * g;
				b = (1.f - stress) * b;

				color = hkColor::rgbFromFloats(r,g,b);

				m_npShapeViewer->setBodyColor(m_npWorld,bodyId,color);
			}
		}
	}

	//// Keyframe the root bone.
	if ( m_npWorld->getBody(rootId).isKeyframed() )
	{
		hkVector4 nextPos = ragdollPose.getSyncedPoseModelSpace()[0].m_translation;
		nextPos.add(m_rootOffset);
		hkQuaternion nextOrient = ragdollPose.getSyncedPoseModelSpace()[0].m_rotation;
		m_npWorld->applyHardKeyFrame( rootId, nextPos, nextOrient, m_timestep );
	}
	
	//// Step the simulation
	hkDefaultPhysicsDemo::stepDemo();

	//// Do the reverse mapping to display the poses.
	{
		// Get the pose from the ragdoll after stepping the physics.
		m_ragdoll->getPoseModelSpace( ragdollModelSpace.begin(), hkQsTransform::getIdentity() );

		// Map back from ragdoll to animation.
		{
			const hkQsTransform* localBIn  = animPose.getSyncedPoseLocalSpace().begin();
			hkQsTransform* modelBOut = animPose.accessSyncedPoseModelSpace().begin();
			m_lowToHighMapper->mapPose( ragdollModelSpace.begin(), localBIn, modelBOut, hkaSkeletonMapper::CURRENT_POSE );
		}
	}
	//// Draw the poses.
	AnimationUtils::drawPose( ragdollPose, rootTransform, hkColor::CYAN);
	AnimationUtils::drawPose( animPose, hkQsTransform::getIdentity(), hkColor::BLUE );

	return hkDemo::DEMO_OK;
}


#if defined(HK_COMPILER_MWERKS)
#	pragma force_active on
#	pragma fullpath_file on
#endif

static const char helpString2[] = "Press 1 to pause animation.\n" ;

HK_DECLARE_DEMO_VARIANT_USING_STRUCT( NpRagdollBlendTestDemo, HK_DEMO_TYPE_ANIMATION, BlendTestVariant, g_variants, helpString2 );

/*
 * Havok SDK - NO SOURCE PC DOWNLOAD, BUILD(#20140907)
 * 
 * Confidential Information of Havok.  (C) Copyright 1999-2014
 * Telekinesys Research Limited t/a Havok. All Rights Reserved. The Havok
 * Logo, and the Havok buzzsaw logo are trademarks of Havok.  Title, ownership
 * rights, and intellectual property rights in the Havok software remain in
 * Havok and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and indicates
 * acceptance of the End User licence Agreement for this product. A copy of
 * the license is included with this software and is also available at www.havok.com/tryhavok.
 * 
 */
